/*========================================================================*\

Copyright (c) 1990-2003  Paul Vojta

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL PAUL VOJTA OR ANY OTHER AUTHOR OF OR CONTRIBUTOR TO
THIS SOFTWARE BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
IN THE SOFTWARE.

NOTE:
	xdvi is based on prior work, as noted in the modification history
	in xdvi.c.

\*========================================================================*/

#include "filf-app.h"	/* application-related defs, etc. */
#include "filefind.h"

#include <errno.h>
#include <signal.h>

#ifdef	X_NOT_STDC_ENV
extern	int	errno;
extern	char	*getenv ARGS((_Xconst char *));
#endif

/*
 *	If you think you have to change DEFAULT_TAIL, then you haven't read the
 *	documentation closely enough.
 */

#ifndef	VMS
#define	PATH_SEP	':'
#define	DEFAULT_TAIL	"/%f.%d%p"
#define	DEFAULT_VF_TAIL	"/%f.vf"
#else	/* VMS */
#define	PATH_SEP	'/'
#define	DEFAULT_TAIL	":%f.%d%p"
#define	DEFAULT_VF_TAIL	":%f.vf"
#endif	/* VMS */

#ifndef DEFAULT_FONT_SIZES
	/* default sizes for current dpi */
#define	DEFAULT_FONT_SIZES	"m0:m0.5:m1:m2:m3:m4:m5"
#endif

#ifdef MKTEXPK
static	_Xconst	char	*makepkcmd		= NULL;
#endif

/*
 *	Information on how to search for pk and gf files.
 */

static	_Xconst	char	no_f_str_pkgf[]	= DEFAULT_TAIL;

static	struct findrec			search_pkgf	= {
	/* path1	*/	NULL,
#if	CFGFILE
	/* envptr	*/	NULL,
#endif
	/* path2	*/	DEFAULT_FONT_PATH,
	/* type		*/	"font",
	/* fF_etc	*/	"fFdbpm",
	/* x_var_char	*/	'd',
	/* n_var_opts	*/	3,
	/* no_f_str	*/	no_f_str_pkgf,
	/* no_f_str_end	*/	no_f_str_pkgf + sizeof(no_f_str_pkgf) - 1,
	/* abs_str	*/	"%f.%d%p",
#ifdef	USE_GF
	/* no_f_str_flags */	F_FILE_USED | F_PK_USED,
	/* abs_str_flags */	F_FILE_USED | F_PK_USED,
	/* pk_opt_char	*/	'p',
	/* pk_gf_addr	*/	&fF_values[4],
#endif
	/* pct_s_str	*/	"%qfonts/%p/%m//:%qfonts/%p/modeless//",
	{
	  /* v.stephead		*/	NULL,
	  /* v.pct_s_head	*/	NULL,
	  /* v.pct_s_count	*/	0,
	  /* v.pct_s_atom	*/	NULL,
	  /* v.rootp		*/	NULL,
	}
};

#ifdef	DOSNAMES

static	_Xconst	char	no_f_str_dos[]	= "/dpi%d/%f.%p";

static	struct findrec			search_pkgf_dos	= {
	/* path1	*/	NULL,
#if	CFGFILE
	/* envptr	*/	NULL,
#endif
	/* path2	*/	DEFAULT_FONT_PATH,
	/* type		*/	"font",
	/* fF_etc	*/	"fFdbpm",
	/* x_var_char	*/	'd',
	/* n_var_opts	*/	3,
	/* no_f_str	*/	no_f_str_dos,
	/* no_f_str_end	*/	no_f_str_dos + sizeof(no_f_str_dos) - 1,
	/* abs_str	*/	"%f.%d%p",
#ifdef	USE_GF
	/* no_f_str_flags */	F_FILE_USED | F_PK_USED,
	/* abs_str_flags */	F_FILE_USED | F_PK_USED,
	/* pk_opt_char	*/	'p',
	/* pk_gf_addr	*/	&fF_values[4],
#endif
	/* pct_s_str	*/	"%qfonts/%p/%m//:%qfonts/%p/modeless//",
	{
	  /* v.stephead		*/	NULL,
	  /* v.pct_s_head	*/	NULL,
	  /* v.pct_s_count	*/	0,
	  /* v.pct_s_atom	*/	NULL,
	  /* v.rootp		*/	NULL,
	}
};
#endif	/* DOSNAMES */

/*
 *	Information on how to search for vf files.
 */

static	_Xconst	char	no_f_str_vf[]	= DEFAULT_VF_TAIL;

static	struct findrec			search_vf	= {
	/* path1	*/	NULL,
#if	CFGFILE
	/* envptr	*/	NULL,
#endif
	/* path2	*/	DEFAULT_VF_PATH,
	/* type		*/	"vf",
	/* fF_etc	*/	"fF",
	/* x_var_char	*/	'f',		/* i.e., none */
	/* n_var_opts	*/	2,
	/* no_f_str	*/	no_f_str_vf,
	/* no_f_str_end	*/	no_f_str_vf + sizeof(no_f_str_vf) - 1,
	/* abs_str	*/	"%f.vf",
#ifdef	USE_GF
	/* no_f_str_flags */	F_FILE_USED,
	/* abs_str_flags */	F_FILE_USED,
	/* pk_opt_char	*/	'f',		/* none */
	/* pk_gf_addr	*/	NULL,
#endif
	/* pct_s_str	*/	"%qfonts/vf//",
	{
	  /* v.stephead		*/	NULL,
	  /* v.pct_s_head	*/	NULL,
	  /* v.pct_s_count	*/	0,
	  /* v.pct_s_atom	*/	NULL,
	  /* v.rootp		*/	NULL,
	}
};

static	int	*sizes, *sizend;
static	char	default_size_list[]	= DEFAULT_FONT_SIZES;

static	char		bdpi_string[10];
static	char		dpi_string[10];

static	double		magsteps[10]	= {1.0, 1.2, 1.44, 1.728, 2.0736,
					   2.48832, 2.985984, 3.5831808,
					   4.29981696, 5.159780352};

static	int
atosize(p)
	_Xconst	char	*p;
{
	_Xconst	char	*q;
	Boolean		minus;
	double		factor;

	if (*p != 'm')
	    return atoi(p);

	for (q = "agstep";; ++q) {
	    ++p;
	    if (*q == '\0' || *p != *q) break;
	}

	minus = False;
	if (*p == '-') {
	    minus = True;
	    ++p;
	}

	if (*p < '0' || *p > '9')
	    return 0;
	factor = magsteps[*p - '0'];
	++p;

	if (*p == '.' && p[1] == '5')
	    factor *= 1.0954451150103321;	/* sqrt(1.2) */

	return (int)
	  ((minus ? pixels_per_inch / factor : pixels_per_inch * factor) + 0.5);
}

static	void
get_sizes(size_list, spp)
	char	*size_list;
	int	**spp;
{
	if (*size_list == PATH_SEP) ++size_list;
	for (;;) {
	    *(*spp)++ = atosize(size_list);
	    size_list = index(size_list, PATH_SEP);
	    if (size_list == NULL) return;
	    ++size_list;
	}
}

void
init_font_open()
{
	char	*size_list;
	int	*sp, *sp1;
	unsigned int n;
	char	*p;

	Sprintf(bdpi_string, "%d", pixels_per_inch);

#if	CFGFILE

#ifndef	XDVIFONTS_ONLY
	if ((search_pkgf.path1 = getenv("XDVIFONTS")) == NULL
	  && (search_pkgf.path1 = getenv("PKFONTS")) == NULL
	  && (search_pkgf.path1 = getenv("TEXPKS")) == NULL)
	    search_pkgf.path1 = getenv("TEXFONTS");
#else
	search_pkgf.path1 = getenv("XDVIFONTS");
#endif
#ifdef	DOSNAMES
	search_pkgf_dos.path1 = search_pkgf.path1;
#endif
	search_pkgf.envptr = ffgetenv("PKFONTS");
	/* clear it if it's a getenv() placeholder */
	if (search_pkgf.envptr != NULL && search_pkgf.envptr->value == NULL)
	    search_pkgf.envptr = NULL;
#ifdef	DOSNAMES
	search_pkgf_dos.envptr = search_pkgf.envptr;
#endif

#else	/* not CFGFILE */

	if ((search_pkgf.path1 = getenv("XDVIFONTS")) == NULL
#ifndef	XDVIFONTS_ONLY
		&& (search_pkgf.path1 = getenv("PKFONTS")) == NULL
		&& (search_pkgf.path1 = getenv("TEXPKS")) == NULL
		&& (search_pkgf.path1 = getenv("TEXFONTS")) == NULL
#endif
		) {
	    search_pkgf.path1 = search_pkgf.path2;
	    search_pkgf.path2 = NULL;
	}
#ifdef	DOSNAMES
	search_pkgf_dos.path1 = search_pkgf.path1;
	search_pkgf_dos.path2 = search_pkgf.path2;
#endif

#endif	/* not CFGFILE */

	/*
	 * pk/gf searching is the only kind that uses more than three
	 * characters in fF_etc, so these can be initialized once and then
	 * forgotten.
	 */

	fF_values[2] = dpi_string;
	fF_values[3] = bdpi_string;
#ifndef	USE_GF
	fF_values[4] = "pk";
#endif
	fF_values[5] = resource.mfmode;

#if	CFGFILE

#ifndef	XDVIFONTS_ONLY
	if ((search_vf.path1 = getenv("XDVIVFS")) == NULL)
	    search_vf.path1 = getenv("VFFONTS");
#else
	search_vf.path1 = getenv("XDVIVFS");
#endif
	search_vf.envptr = ffgetenv("VFFONTS");
	/* clear it if it's a getenv() placeholder */
	if (search_vf.envptr != NULL && search_vf.envptr->value == NULL)
	    search_vf.envptr = NULL;

#else	/* not CFGFILE */

	if ((search_vf.path1 = getenv("XDVIVFS")) == NULL
#ifndef	XDVIFONTS_ONLY
		&& (search_vf.path1 = getenv("VFFONTS")) == NULL
#endif
		) {
	    search_vf.path1 = search_vf.path2;
	    search_vf.path2 = NULL;
	}

#endif	/* not CFGFILE */

	size_list = getenv("XDVISIZES");
	n = 1;	/* count number of sizes */
	if (size_list == NULL || *size_list == '\0' || *size_list == PATH_SEP)
	    for (p = default_size_list; (p = index(p, PATH_SEP)) != NULL; ++p)
		++n;
	if (size_list != NULL)
	    for (p = size_list; (p = index(p, PATH_SEP)) != NULL; ++p) ++n;
	sizes = xmalloc(n * sizeof(int));
	sizend = sizes;	/* get the actual sizes */
	if (size_list == NULL || *size_list == '\0' || *size_list == PATH_SEP)
	    get_sizes(default_size_list, &sizend);
	if (size_list != NULL && *size_list != '\0')
	    get_sizes(size_list, &sizend);

	/* sort the sizes (insertion sort) */
	for (sp = sizes + 1; sp < sizend; ++sp)
	    if (*sp < sp[-1]) {
		int	i	= *sp;

		sp1 = sp;
		do
		    *sp1 = sp1[-1];
		while (--sp1 > sizes && i < sp1[-1]);
		*sp1 = i;
	    }

	/* eliminate duplicates and erroneous values */
	n = 0;	/* previous value */
	for (sp = sp1 = sizes; sp < sizend; ++sp)
	    if (*sp != n)
		n = *sp1++ = *sp;
	sizend = sp1;
}

/*
 *	Try a given size.
 */

static	FILE *
try_size(font, dpi, ret_path)
	_Xconst char	*font;
	int		dpi;
	_Xconst char	**ret_path;
{
#ifdef	DOSNAMES
	FILE		*retval;
#endif

	Sprintf(dpi_string, "%d", dpi);

#ifdef	DOSNAMES
	retval = filefind(font, &search_pkgf, ret_path);
	if (retval != NULL) return retval;
	return filefind(font, &search_pkgf_dos, ret_path);
#else
	return filefind(font, &search_pkgf, ret_path);
#endif
}


#ifdef MKTEXPK

#ifndef	MKTEXPK_PATH
#ifdef MAKETEXPK
#define	MKTEXPK_PATH	"MakeTeXPK"
#else
#define	MKTEXPK_PATH	"mktexpk"
#endif
#endif

#ifdef MAKETEXPK
#define	MKPK_DEFAULT_MODE	"default"
#else
#define	MKPK_DEFAULT_MODE	"/"
#endif

#if HAVE_GOOD_SETSID_VFORK
# if HAVE_VFORK_H
#  include <vfork.h>
# endif
#else
  /* Mac OS X 10.3 (Panther) (11/2003) doesn't allow setsid() within vfork() */
# undef vfork
# define vfork fork
#endif

#ifndef	EXIT_SUCCESS
#ifndef	VMS
#define	EXIT_SUCCESS	0
#else
#define	EXIT_SUCCESS	1
#endif
#endif

#if HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif
#ifndef WEXITSTATUS
# define WEXITSTATUS(stat_val) ((unsigned)(stat_val) >> 8)
#endif
#ifndef WIFEXITED
# define WIFEXITED(stat_val) (((stat_val) & 255) == 0)
#endif

#if HAVE_POLL
# include <poll.h>
# define XIO_IN POLLIN
# define XIO_OUT POLLOUT
#else
# define XIO_IN 1
# define XIO_OUT 2
#endif

#ifdef	EWOULDBLOCK
#ifdef	EAGAIN
#define	AGAIN_CONDITION	(errno == EWOULDBLOCK || errno == EAGAIN)
#else	/* EAGAIN */
#define	AGAIN_CONDITION	(errno == EWOULDBLOCK)
#endif	/* EAGAIN */
#else	/* EWOULDBLOCK */
#ifdef	EAGAIN
#define	AGAIN_CONDITION	(errno == EAGAIN)
#endif	/* EAGAIN */
#endif	/* EWOULDBLOCK */

#define	NOBUILD		29999

/*
 *	ffmemcpy - Copy the given number of bytes to the given position
 */

static	unsigned int
ffmemcpy(pos, src, len)
	unsigned int	pos;
	_Xconst char	*src;
	unsigned int	len;
{
	if (pos + len >= ffline_len)
	    expandline(pos + len);
	memcpy(ffline + pos, src, len);
	return pos + len;
}

/*
 *	ffstrcpy - Copy string to ffline at the given position
 */

static	unsigned int
ffstrcpy(pos, str)
	unsigned int	pos;
	_Xconst char	*str;
{
	return ffmemcpy(pos, str, strlen(str));
}

/*
 *	ff2memcpy - Copy the given number of bytes to the given position
 *	  *within* ffline.
 */

static	unsigned int
ff2memcpy(pos, src, len)
	unsigned int	pos;
	unsigned int	src;
	unsigned int	len;
{
	if (pos + len >= ffline_len)
	    expandline(pos + len);
	memcpy(ffline + pos, ffline + src, len);
	return pos + len;
}

/*
 *	Stuff to handle child process termination
 */

static	Boolean		mkpk_done;
static	int		mkpk_status;

static	void
mkpk_term(status)
	int status;
{
	mkpk_done = True;
	mkpk_status = status;
	ev_flags |= EV_ACK;
}

static	struct xchild	mkpk_child	= {NULL, 0, False, mkpk_term};
static	Boolean		mkpk_timer_set;

static	void
mkpk_alarm(arg)
	struct xtimer	*arg;
{
	mkpk_timer_set = False;
	ev_flags |= EV_ACK;
}

static	struct xtimer	mkpk_timer	= TIMER_INIT(mkpk_alarm);

/*
 *	Stuff to handle child process output
 */

static	void	mkpk_get_result ARGS((void));

static	unsigned int	mkpk_pos;

static	struct xio	mkpk_result	= {NULL, 0, XIO_IN,
#if HAVE_POLL
					   NULL,
#endif
					   mkpk_get_result, NULL};

static	void
mkpk_get_result()
{
	int	bytes;

	for (;;) {
	    bytes = read(mkpk_result.fd, ffline + mkpk_pos,

	      ffline_len - mkpk_pos);
	    if (bytes < 0) {
		if (AGAIN_CONDITION)
		    break;
		perror("xdvi: mkpk_result read");
		break;
	    }
	    if (bytes == 0) break;

	    mkpk_pos += bytes;
	    if (mkpk_pos >= ffline_len)
		expandline(ffline_len);
	}
}


/*
 *	makefont - call system() to make the font.
 */

static	FILE *
makefont(font, dpi, name, magstepval)
	_Xconst char	*font;
	int		dpi;
	_Xconst char	**name;
	int		magstepval;
{
	int		pipefds[2];
	Boolean		used_fontname	= False;
	Boolean		used_mfmode	= False;
	int		redirect_to	= 1;
	_Xconst	char	*p;
	unsigned int	argc;
	unsigned int	pos_end, pos1, destpos;
	_Xconst char	*actual_command;
	char		**argv;
	unsigned int	pos;
	static Boolean	did_putenv	= False;
	FILE		*f;
	char		*q;
	unsigned int	i;

	if (xpipe(pipefds) != 0) {	/* create the pipe */
	    perror("[xdvi] pipe");
	    return NULL;
	}

	/*
	 * Generate the mktexpk command line.
	 */

	if (makepkcmd == NULL) {
	    makepkcmd = getenv("XDVIMAKEPK");
	    if (makepkcmd == NULL) makepkcmd = MKTEXPK_PATH;
	}
	p = makepkcmd;
	pos = 1;
	if (pos >= ffline_len)
	    expandline(ffline_len);
	ffline[0] = '\0';
	argc = 0;

	for (;;) {
	    if (*p == '%') {
		char	buf[32];

		switch (*++p) {
		case 'n':
		    pos = ffstrcpy(pos, font);
		    used_fontname = True;
		    break;
		case 'd':
		    Sprintf(buf, "%d", dpi);
		    pos = ffstrcpy(pos, buf);
		    break;
		case 'b':
		    Sprintf(buf, "%d", pixels_per_inch);
		    pos = ffstrcpy(pos, buf);
		    break;
		case 'm':
		    if (magstepval == NOMAGSTP)
			Sprintf(buf, "%d+%d/%d", dpi / pixels_per_inch,
			    dpi % pixels_per_inch, pixels_per_inch);
		    else if (magstepval >= 0)
			Sprintf(buf, "magstep(%d%s)", magstepval / 2,
			    magstepval % 2 ? ".5" :"");
		    else
			Sprintf(buf, "magstep(-%d%s)", -magstepval / 2,
			    magstepval % 2 ? ".5" :"");
		    pos = ffstrcpy(pos, buf);
		    break;
		case 'o':
		    pos = ffstrcpy(pos, resource.mfmode != NULL
		      ? resource.mfmode : MKPK_DEFAULT_MODE);
		    used_mfmode = True;
		    break;
		case 'r':
		    pos = ffmemcpy(pos, ">&3", 3);
		    redirect_to = 3;
		    break;
		case '%':
		    if (pos + 1 >= ffline_len)
			expandline(pos + 1);
		    ffline[pos++] = '%';
		    break;
		case '\0':
		    --p;
		    break;
		default:
		    if (pos + 2 >= ffline_len)
			expandline(pos + 1);
		    ffline[pos++] = '%';
		    ffline[pos++] = *p;
		    break;
		}
	    }
	    else if (*p == '\0')
		if (used_fontname) break;
		else {
#ifndef MAKETEXPK
		    p = " --mfmode %o --bdpi %b --mag %m --dpi %d %n %r";
#else
#ifdef MKPK_REDIRECT
		    p = " %n %d %b %m %o '' %r";
#else
		    p = " %n %d %b %m";
#endif
#endif
		    continue;
		}
	    else if (*p == ' ' || *p == '\t') {
		if (ffline[pos - 1] != '\0') {
		    ++argc;
		    ffline[pos++] = '\0';
		}
	    }
	    else {
		if (pos + 1 >= ffline_len)
		    expandline(pos + 1);
		ffline[pos++] = *p;
	    }
	    ++p;
	}

	if (resource.mfmode != NULL && !used_mfmode)
	    pos = ffstrcpy(pos, resource.mfmode);

	if (ffline[pos - 1] != '\0') {
	    ++argc;
	    ffline[pos++] = '\0';
	}

	/* Form command line string */
	pos1 = 1;
	destpos = pos;
	if (destpos >= ffline_len)
	    expandline(ffline_len);
	ffline[destpos++] = '-';
	for (;;) {	/* loop over args */
	    if (pos1 >= pos) break;	/* if end of args */
	    ffline[destpos++] = ' ';	/* Convert \0 to space */
	    pos_end = pos1 + strlen(ffline + pos1);
	    for (;;) {
		unsigned int	pos2, pos3;

		p = memchr(ffline + pos1, '\'', pos_end - pos1);
		pos2 = (p == NULL ? pos_end : p - ffline);
		for (pos3 = pos1;; ++pos3)
		    if (pos3 >= pos2) {
			destpos = ff2memcpy(destpos, pos1, pos2 - pos1);
			break;
		    }
		    else if (index(" \t`\"()$&<>~*?\\", ffline[pos3]) != NULL) {
			if (destpos >= ffline_len)
			    expandline(ffline_len);
			ffline[destpos++] = '\'';
			destpos = ff2memcpy(destpos, pos1, pos2 - pos1);
			if (destpos >= ffline_len)
			    expandline(ffline_len);
			ffline[destpos++] = '\'';
			break;
		    }
		pos1 = pos2 + 1;
		if (p == NULL) break;
		destpos = ffmemcpy(destpos, "\\'", 2);
	    }
	}

	ffline[destpos++] = '\0';
	actual_command = xmemdup(ffline + pos, destpos - pos);
	Puts(actual_command);

	argv = xmalloc((argc + 1) * sizeof(*argv));
	q = ffline + 1;
	for (i = 0; i < argc; ++i) {
	    argv[i] = q;
	    q += strlen(q) + 1;
	}
	argv[argc] = NULL;

	/*
	 * Put the metafont mode into the environment, if available.
	 */

	if (!did_putenv) {
	    if (resource.mfmode != NULL)
		xputenv("MAKETEX_MODE", resource.mfmode);
	    did_putenv = True;
	}

	/*
	 * Create a child process.
	 */

	Fflush(stderr);		/* avoid double buffering */
	mkpk_child.pid = vfork();
	if (mkpk_child.pid == 0) {		/* if child */
	    (void) close(pipefds[0]);
	    if (redirect_to != pipefds[1]) {
		(void) dup2(pipefds[1], redirect_to);
		(void) close(pipefds[1]);
	    }
	    if (setsid() == -1) {	/* so we can kill the process group */
		perror("setsid");
		Fflush(stderr);
		_exit(1);
	    }
	    (void) execvp(*argv, argv);
	    Fprintf(stderr, "Execvp of %s failed.\n", *argv);
	    Fflush(stderr);
	    _exit(1);
	}

	(void) close(pipefds[1]);
	++n_files_left;
	free(argv);

	if (mkpk_child.pid == -1) {
	    (void) close(pipefds[0]);
	    ++n_files_left;
	    perror("[xdvi] fork");
	    free((void *) actual_command);
	    return NULL;
	}

	/*
	 * Now wait until the process terminates, reading whatever it writes
	 * to the pipe.  An eof on the pipe assumes that the child terminated.
	 */

	set_chld(&mkpk_child);

	prep_fd(pipefds[0], True);	/* set fd for non-blocking I/O */

	mkpk_pos = 0;
	mkpk_result.fd = pipefds[0];
	set_io(&mkpk_result);

	mkpk_done = False;
	for (;;) {
	    (void) read_events(EV_GE_NEWDOC | EV_ACK);
	    ev_flags &= ~EV_ACK;

	    if (ev_flags & EV_GE_NEWDOC) {	/* if we're aborting */
		kill(-mkpk_child.pid, SIGINT);
		set_timer(&mkpk_timer, 3000);
		mkpk_timer_set = True;
		for (;;) {
		    (void) read_events(EV_ACK);
		    ev_flags &= ~EV_ACK;

		    if (mkpk_done) {
			if (mkpk_timer_set)
			    cancel_timer(&mkpk_timer);
			break;
		    }
		    if (!mkpk_timer_set) {
			kill(-mkpk_child.pid, SIGKILL);
			clear_chld(&mkpk_child);
			break;
		    }
		}
		if (ev_flags & EV_GE_TERM)
		    xdvi_exit(0);
		clear_io(&mkpk_result);
		(void) close(pipefds[0]);
		++n_files_left;
		free((void *) actual_command);
		return NULL;
	    }

	    if (mkpk_done)
		break;
	}

	clear_io(&mkpk_result);
	(void) close(pipefds[0]);
	++n_files_left;

	if (!WIFEXITED(mkpk_status) || WEXITSTATUS(mkpk_status) != EXIT_SUCCESS)
	{
	    WARN1(XmDIALOG_WARNING,
	      "The following command for building a font file has failed:\n%s",
	      actual_command + 2);
	    free((void *) actual_command);
	    return NULL;
	}

	if (mkpk_pos != 0 && ffline[mkpk_pos - 1] == '\n')
	    --mkpk_pos;	/* trim off last \n */

	/* if no response, then it probably failed, but look anyway */
	if (mkpk_pos == 0) {
	    if (debug & DBG_OPEN)
		Printf("No response from %s\n", actual_command + 2);
	    free((void *) actual_command);
	    return try_size(font, dpi, name);
	}

	if (mkpk_pos >= ffline_len)
	    expandline(mkpk_pos);
	ffline[mkpk_pos++] = '\0';

	if (debug & DBG_OPEN)
	    Printf("%s ---> %s\n", actual_command + 2, ffline);
	free((void *) actual_command);

	f = xfopen(ffline, OPEN_MODE);
	if (f == NULL) {
	    perror(ffline);
	    return NULL;
	}

	if (debug & DBG_OPEN)
	    puts("--Success--\n");
	*name = xmemdup(ffline, mkpk_pos);
	return f;
}

#endif	/* MKTEXPK */

/*
 *	Try a given font name
 */

#ifndef MKTEXPK
#define	PRE_FONT_OPEN(font, fdpi, dpi_ret, name, ignore) \
		pre_font_open(font, fdpi, dpi_ret, name)
#else
#define	PRE_FONT_OPEN	pre_font_open
#endif

static	FILE *
PRE_FONT_OPEN(font, fdpi, dpi_ret, name, magstepval)
	_Xconst char	*font;
	double		fdpi;
	int		*dpi_ret;
	_Xconst char	**name;
#ifdef MKTEXPK
	int	magstepval;
#endif
{
	FILE	*f;
	int	*p1, *p2;
	int	dpi	= fdpi + 0.5;
	int	tempdpi;

	/*
	 * Loop over sizes.  Try actual size first, then closest sizes.
	 * If the pathname is absolutely or explicitly relative, don't
	 * use the usual paths to search for it; just look for it in the
	 * directory specified.
	 */

	f = try_size(font, *dpi_ret = dpi, name);
	if (f != NULL)
	    return f;

	/* Try at one away from the size we just tried, to account
	   for rounding error.  */
	tempdpi = dpi + (dpi < fdpi ? 1 : -1);
	f = try_size(font, tempdpi, name);
	if (f != NULL) {
	    *dpi_ret = tempdpi;
	    return f;
	}

	/* Try a virtual font. */
	f = filefind(font, &search_vf, name);
	if (f != NULL)
	    return f;

#ifdef MKTEXPK
	/* Try to create the font. */
	if (magstepval != NOBUILD && resource.makepk) {
	    f = makefont(font, dpi, name, magstepval);
	    if (f != NULL || ev_flags & EV_GE_NEWDOC)
		return f;
	}
#endif

	/* Now try at all the sizes. */
	for (p2 = sizes; p2 < sizend; ++p2) if (*p2 >= dpi) break;
	p1 = p2;
	for (;;) {
		/* find another resolution */
	    if (p1 <= sizes)
		if (p2 >= sizend) return NULL;
		else tempdpi = *p2++;
	    else if (p2 >= sizend || (long) dpi * dpi <= (long) p1[-1] * *p2)
		    tempdpi = *--p1;
		else tempdpi = *p2++;
	    f = try_size(font, *dpi_ret = tempdpi, name);
	    if (f != NULL)
		return f;
	}
}

/* ARGSUSED */
FILE *
font_open(font, font_ret, dpi, dpi_ret, magstepval, name)
	_Xconst char	*font;
	char	**font_ret;
	double	dpi;
	int	*dpi_ret;
	int	magstepval;
	char	**name;
{
	FILE	*f;
	int	actual_pt, low_pt, high_pt, trial_pt;
	char	fn[50], *fnend;

	f = PRE_FONT_OPEN(font, dpi, dpi_ret, name, magstepval);
	if (f != NULL || ev_flags & EV_GE_NEWDOC) {
	    *font_ret = NULL;
	    return f;
	}
	Strcpy(fn, font);
	fnend = fn + strlen(fn);
	while (fnend > fn && fnend[-1] >= '0' && fnend[-1] <= '9') --fnend;
	actual_pt = low_pt = high_pt = atoi(fnend);
	if (actual_pt) {
	    low_pt = actual_pt - 1;
	    high_pt = actual_pt + 1;
	    for (;;) {
		if (2 * low_pt >= actual_pt &&
		    (low_pt * high_pt > actual_pt * actual_pt ||
		    high_pt > actual_pt + 5))
			trial_pt = low_pt--;
		else if (high_pt > actual_pt + 5) break;
		else trial_pt = high_pt++;
		Sprintf(fnend, "%d", trial_pt);
		f = PRE_FONT_OPEN(fn, dpi * actual_pt / trial_pt, dpi_ret,
		    name, NOBUILD);
		if (f != NULL) {
		    *font_ret = xstrdup(fn);
		    return f;
		}
	    }
	}
	if (alt_font != NULL) {
	    f = PRE_FONT_OPEN(alt_font, dpi, dpi_ret, name, magstepval);
	    if (f != NULL)
		*font_ret = xstrdup(alt_font);
	}
	return f;
}
