/*========================================================================*\

Copyright (c) 1990-2003  Paul Vojta

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL PAUL VOJTA OR ANY OTHER AUTHOR OF OR CONTRIBUTOR TO
THIS SOFTWARE BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
IN THE SOFTWARE.

NOTE:
	xdvi is based on prior work, as noted in the modification history
	in xdvi.c.

\*========================================================================*/

#include "xdvi.h"
#include "dvi.h"
#include <ctype.h>

#if HAVE_VFORK_H
# include <vfork.h>
#endif

#if XAW
# include <X11/StringDefs.h>
# if XAW3d
#  include <X11/Xaw3d/Label.h>
# else
#  include <X11/Xaw/Label.h>
# endif
#elif MOTIF
#endif

#if	NeedVarargsPrototypes		/* this is for tell_oops */
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#ifdef	DOPRNT	/* define this if vfprintf gives you trouble */
#define	vfprintf(stream, message, args)	_doprnt(message, args, stream)
#endif

static	struct frame	frame0;		/* dummy head of list */
#ifdef	TEXXET
static	struct frame	*scan_frame;	/* head frame for scanning */
#endif

#ifndef	DVI_BUFFER_LEN
#define	DVI_BUFFER_LEN	512
#endif

static	ubyte	dvi_buffer[DVI_BUFFER_LEN];
static	struct frame	*current_frame;

/* Points to drawinf record containing current dvi file location (for update by
   geom_scan).  */
static	struct drawinf	*dvi_pointer_frame	INIT(NULL);

#ifndef	TEXXET
#define	DIR	1
#else
#define	DIR	currinf.dir
#endif

/*
 *	Explanation of the following constant:
 *	offset_[xy]   << 16:	margin (defaults to one inch)
 *	shrink_factor << 16:	one pixel page border
 *	shrink_factor << 15:	rounding for pixel_conv
 */
#define OFFSET_X	(offset_x << 16) + (shrink_factor * 3 << 15)
#define OFFSET_Y	(offset_y << 16) + (shrink_factor * 3 << 15)

#if (BMBYTES == 1)
BMUNIT	bit_masks[9] = {
	0x0,	0x1,	0x3,	0x7,
	0xf,	0x1f,	0x3f,	0x7f,
	0xff
};
#else
#if (BMBYTES == 2)
BMUNIT	bit_masks[17] = {
	0x0,	0x1,	0x3,	0x7,
	0xf,	0x1f,	0x3f,	0x7f,
	0xff,	0x1ff,	0x3ff,	0x7ff,
	0xfff,	0x1fff,	0x3fff,	0x7fff,
	0xffff
};
#else	/* BMBYTES == 4 */
BMUNIT	bit_masks[33] = {
	0x0,		0x1,		0x3,		0x7,
	0xf,		0x1f,		0x3f,		0x7f,
	0xff,		0x1ff,		0x3ff,		0x7ff,
	0xfff,		0x1fff,		0x3fff,		0x7fff,
	0xffff,		0x1ffff,	0x3ffff,	0x7ffff,
	0xfffff,	0x1fffff,	0x3fffff,	0x7fffff,
	0xffffff,	0x1ffffff,	0x3ffffff,	0x7ffffff,
	0xfffffff,	0x1fffffff,	0x3fffffff,	0x7fffffff,
	0xffffffff
};
#endif
#endif

#ifdef	VMS
#define	off_t	int
#endif
extern	off_t	lseek();

#ifndef	SEEK_SET	/* if <unistd.h> is not provided (or for <X11R5) */
#define	SEEK_SET	0
#define	SEEK_CUR	1
#define	SEEK_END	2
#endif

static	void	draw_part();
static	void	source_fwd_draw_box();

/*
 *	X routines.
 */

/*
 *	Put a rectangle on the screen.
 */

static	void
put_rule(x, y, w, h)
	int		x, y;
	unsigned int	w, h;
{
	if (x < max_x && x + (int) w >= min_x
	  && y < max_y && y + (int) h >= min_y) {
	    if (--event_counter == 0)
		if (read_events(EV_NOWAIT) & EV_GE_MAG_GONE)
		    longjmp(canit_env, 1);
#if COLOR
	    if (fg_active != fg_current) do_color_change();
#endif
	    XFillRectangle(DISP, currwin.win, ruleGC,
		x - currwin.base_x, y - currwin.base_y, w ? w : 1, h ? h : 1);
	}
}

static	void
put_bitmap(bitmap, x, y)
	struct bitmap	*bitmap;
	int		x, y;
{

	if (debug & DBG_BITMAP)
		Printf("X(%d,%d)\n", x - currwin.base_x, y - currwin.base_y);
	if (x < max_x && x + (int) bitmap->w >= min_x
	  && y < max_y && y + (int) bitmap->h >= min_y) {
	    if (--event_counter == 0)
		if (read_events(EV_NOWAIT) & EV_GE_MAG_GONE)
		    longjmp(canit_env, 1);
#if COLOR
	    if (fg_active != fg_current) do_color_change();
#endif
	    image->width = bitmap->w;
	    image->height = bitmap->h;
	    image->data = bitmap->bits;
	    image->bytes_per_line = bitmap->bytes_wide;
	    XPutImage(DISP, currwin.win, foreGC, image,
	      0, 0,
	      x - currwin.base_x, y - currwin.base_y,
	      bitmap->w, bitmap->h);
	    if (foreGC2)
		XPutImage(DISP, currwin.win, foreGC2, image,
		  0, 0,
		  x - currwin.base_x, y - currwin.base_y,
		  bitmap->w, bitmap->h);
	}
}

#if GREY

static	void	shrink_glyph_grey ARGS((struct glyph *));

static	void
put_image(g, x, y)
	struct glyph	*g;
	int		x, y;
{
	XImage *img = g->image2;

	if (x < max_x && x + img->width >= min_x &&
	    y < max_y && y + img->height >= min_y) {

	    if (--event_counter == 0)
		if (read_events(EV_NOWAIT) & EV_GE_MAG_GONE)
		    longjmp(canit_env, 1);

#if COLOR
	    if (g->fg != fg_current)	/* if color change since last use */
		shrink_glyph_grey(g);
	    else if (fg_active != fg_current)	/* if GCs need updating */
		do_color_change();
#endif
	    XPutImage(DISP, currwin.win, foreGC, img,
		    0, 0,
		    x - currwin.base_x, y - currwin.base_y,
		    (unsigned int) img->width, (unsigned int) img->height);
	    if (foreGC2 != NULL) {
		img->data = g->pixmap2_t;
		XPutImage(DISP, currwin.win, foreGC2, img,
			0, 0,
			x - currwin.base_x, y - currwin.base_y,
			(unsigned int) img->width, (unsigned int) img->height);
		img->data = g->pixmap2;
	    }
	}
}
#endif	/* GREY */

/*
 *	Byte reading routines for dvi file.
 */

#define	xtell(pos)	(lseek(fileno(dvi_file), 0L, SEEK_CUR) - \
			    (currinf.end - (pos)))

static	ubyte
xxone()
{
	if (currinf.virtual) {
	    ++currinf.pos;
	    return EOP;
	}
	currinf.end = dvi_buffer +
	    read(fileno(dvi_file), (char *) (currinf.pos = dvi_buffer),
		DVI_BUFFER_LEN);
	return currinf.end > dvi_buffer ? *(currinf.pos)++ : EOF;
}

#define	xone()  (currinf.pos < currinf.end ? *(currinf.pos)++ : xxone())

static	unsigned long
xnum(size)
	ubyte	size;
{
	long x = 0;

	while (size--) x = (x << 8) | xone();
	return x;
}

static	long
xsnum(size)
	ubyte	size;
{
	long	x;

#if	__STDC__
	x = (signed char) xone();
#else
	x = xone();
	if (x & 0x80) x -= 0x100;
#endif
	while (--size) x = (x << 8) | xone();
	return x;
}

#define	xsfour()	xsnum(4)

static	void
xskip(offset)
	long	offset;
{
	currinf.pos += offset;
	if (!currinf.virtual && currinf.pos > currinf.end)
	    (void) lseek(fileno(dvi_file), (long) (currinf.pos - currinf.end),
		SEEK_CUR);
}

#if	NeedVarargsPrototypes
static void tell_oops(_Xconst char *, ...) NORETURN;

static	void
tell_oops(_Xconst char *message, ...)
#else
/* VARARGS */
static	void
tell_oops(va_alist)
	va_dcl
#endif
{
#if	!NeedVarargsPrototypes
	_Xconst char *message;
#endif
	va_list	args;

	Fprintf(stderr, "%s: ", prog);
#if	NeedVarargsPrototypes
	va_start(args, message);
#else
	va_start(args);
	message = va_arg(args, _Xconst char *);
#endif
	(void) vfprintf(stderr, message, args);
	va_end(args);
	if (currinf.virtual)
	    Fprintf(stderr, " in virtual font %s\n", currinf.virtual->fontname);
	else
	    Fprintf(stderr, ", offset %ld\n", xtell(currinf.pos - 1));
	xdvi_exit(1);
}


/*
 *	Code for debugging options.
 */

static	void
print_bitmap(bitmap)
	struct bitmap	*bitmap;
{
	BMUNIT		*ptr	= (BMUNIT *) bitmap->bits;
	int		x, y, i;

	if (ptr == NULL) oops("print_bitmap called with null pointer.");
	Printf("w = %d, h = %d, bytes wide = %d\n",
	    bitmap->w, bitmap->h, bitmap->bytes_wide);
	for (y = 0; y < (int) bitmap->h; ++y) {
	    for (x = bitmap->bytes_wide; x > 0; x -= BMBYTES) {
#ifndef	WORDS_BIGENDIAN
		for (i = 0; i < BMBITS; ++i)
#else
		for (i = BMBITS - 1; i >= 0; --i)
#endif
		    Putchar((*ptr & (1 << i)) ? '@' : ' ');
		++ptr;
	    }
	    Putchar('\n');
	}
}

static	void
print_char(ch, g)
	ubyte ch;
	struct glyph *g;
{
	Printf("char %d", ch);
	if (isprint(ch))
	    Printf(" (%c)", ch);
	Putchar('\n');
	Printf("x = %d, y = %d, dvi = %ld\n", g->x, g->y, g->dvi_adv);
	print_bitmap(&g->bitmap);
}

static	_Xconst	char	*dvi_table1[] = {
	"SET1", NULL, NULL, NULL, "SETRULE", "PUT1", NULL, NULL,
	NULL, "PUTRULE", "NOP", "BOP", "EOP", "PUSH", "POP", "RIGHT1",
	"RIGHT2", "RIGHT3", "RIGHT4", "W0", "W1", "W2", "W3", "W4",
	"X0", "X1", "X2", "X3", "X4", "DOWN1", "DOWN2", "DOWN3",
	"DOWN4", "Y0", "Y1", "Y2", "Y3", "Y4", "Z0", "Z1",
	"Z2", "Z3", "Z4"};

static	_Xconst	char	*dvi_table2[] = {
	"FNT1", "FNT2", "FNT3", "FNT4", "XXX1", "XXX2", "XXX3", "XXX4",
	"FNTDEF1", "FNTDEF2", "FNTDEF3", "FNTDEF4", "PRE", "POST", "POSTPOST",
	"SREFL", "EREFL", NULL, NULL, NULL, NULL};

static	void
print_dvi(ch)
	ubyte ch;
{
	_Xconst	char	*s;

	Printf("%4d %4d ", PXL_H, PXL_V);
	if (ch <= (ubyte) (SETCHAR0 + 127)) {
	    Printf("SETCHAR%-3d", ch - SETCHAR0);
	    if (isprint(ch))
		Printf(" (%c)", ch);
	    Putchar('\n');
	    return;
	}
	else if (ch < FNTNUM0) s = dvi_table1[ch - 128];
	else if (ch <= (ubyte) (FNTNUM0 + 63)) {
	    Printf("FNTNUM%d\n", ch - FNTNUM0);
	    return;
	}
	else s = dvi_table2[ch - (FNTNUM0 + 64)];
	if (s) Puts(s);
	else
	    tell_oops("unknown op-code %d", ch);
}


/*
 *	Count the number of set bits in a given region of the bitmap
 */

static	char	sample_count[]	= {0, 1, 1, 2, 1, 2, 2, 3,
				   1, 2, 2, 3, 2, 3, 3, 4};

static	int
sample(bits, bytes_wide, bit_skip, w, h)
	BMUNIT	*bits;
	int	bytes_wide, bit_skip, w, h;
{
	BMUNIT	*ptr, *endp;
	BMUNIT	*cp;
	int	bits_left;
	int	n, bit_shift, wid;

	ptr = bits + bit_skip / BMBITS;
	endp = ADD(bits, h * bytes_wide);
	bits_left = w;
#ifndef	WORDS_BIGENDIAN
	bit_shift = bit_skip % BMBITS;
#else
	bit_shift = BMBITS - bit_skip % BMBITS;
#endif
	n = 0;
	while (bits_left) {
#ifndef	WORDS_BIGENDIAN
	    wid = BMBITS - bit_shift;
#else
	    wid = bit_shift;
#endif
	    if (wid > bits_left) wid = bits_left;
	    if (wid > 4) wid = 4;
#ifdef	WORDS_BIGENDIAN
	    bit_shift -= wid;
#endif
	    for (cp = ptr; cp < endp; cp = ADD(cp, bytes_wide))
		n += sample_count[(*cp >> bit_shift) & bit_masks[wid]];
#ifndef	WORDS_BIGENDIAN
	    bit_shift += wid;
	    if (bit_shift == BMBITS) {
		bit_shift = 0;
		++ptr;
	    }
#else
	    if (bit_shift == 0) {
		bit_shift = BMBITS;
		++ptr;
	    }
#endif
	    bits_left -= wid;
	}
	return n;
}

static	void
shrink_glyph(g)
	struct glyph	*g;
{
	int	shrunk_bytes_wide, shrunk_height;
	int	rows_left, rows, init_cols;
	int	cols_left;
	int	cols;
	BMUNIT	*old_ptr, *new_ptr;
	BMUNIT	m, *cp;
	int	min_sample = shrink_factor * shrink_factor * density / 100;
	int	rtmp;

	/* These machinations ensure that the character is shrunk according to
	   its hot point, rather than its upper left-hand corner. */
	g->x2 = g->x / shrink_factor;
	init_cols = g->x - g->x2 * shrink_factor;
	if (init_cols <= 0) init_cols += shrink_factor;
	else ++g->x2;
	g->bitmap2.w = g->x2 + ROUNDUP((int) g->bitmap.w - g->x, shrink_factor);
	/* include row zero with the positively numbered rows */
	rtmp = g->y + 1;
	g->y2 = rtmp / shrink_factor;
	rows = rtmp - g->y2 * shrink_factor;
	if (rows <= 0) {
	    rows += shrink_factor;
	    --g->y2;
	}
	g->bitmap2.h = shrunk_height = g->y2 +
	    ROUNDUP((int) g->bitmap.h - rtmp, shrink_factor) + 1;
	alloc_bitmap(&g->bitmap2);
	old_ptr = (BMUNIT *) g->bitmap.bits;
	new_ptr = (BMUNIT *) g->bitmap2.bits;
	shrunk_bytes_wide = g->bitmap2.bytes_wide;
	rows_left = g->bitmap.h;
	bzero((char *) new_ptr, shrunk_bytes_wide * shrunk_height);
	while (rows_left) {
	    if (rows > rows_left) rows = rows_left;
	    cols_left = g->bitmap.w;
#ifndef	WORDS_BIGENDIAN
	    m = (1 << 0);
#else
	    m = ((BMUNIT) 1 << (BMBITS-1));
#endif
	    cp = new_ptr;
	    cols = init_cols;
	    while (cols_left) {
		if (cols > cols_left) cols = cols_left;
		if (sample(old_ptr, g->bitmap.bytes_wide,
			(int) g->bitmap.w - cols_left, cols, rows)
			>= min_sample)
		    *cp |= m;
#ifndef	WORDS_BIGENDIAN
		if (m == ((BMUNIT)1 << (BMBITS-1))) {
		    m = (1 << 0);
		    ++cp;
		}
		else m <<= 1;
#else
		if (m == (1 << 0)) {
		    m = ((BMUNIT) 1 << (BMBITS-1));
		    ++cp;
		}
		else m >>= 1;
#endif
		cols_left -= cols;
		cols = shrink_factor;
	    }
	    *((char **) &new_ptr) += shrunk_bytes_wide;
	    *((char **) &old_ptr) += rows * g->bitmap.bytes_wide;
	    rows_left -= rows;
	    rows = shrink_factor;
	}
	g->y2 = g->y / shrink_factor;
	if (debug & DBG_BITMAP)
	    print_bitmap(&g->bitmap2);
}

#ifdef	GREY
static	void
shrink_glyph_grey(g)
	struct glyph *g;
{
	int		rows_left, rows, init_cols;
	int		cols_left;
	int		cols;
	int		x, y;
	long		thesample;
	BMUNIT		*old_ptr;
	unsigned int	size;
	int		rtmp;

#if COLOR
	if (fg_active != fg_current) do_color_change();
#endif

	/* These machinations ensure that the character is shrunk according to
	   its hot point, rather than its upper left-hand corner. */
	g->x2 = g->x / shrink_factor;
	init_cols = g->x - g->x2 * shrink_factor;
	if (init_cols <= 0) init_cols += shrink_factor;
	else ++g->x2;
	g->bitmap2.w = g->x2 + ROUNDUP((int) g->bitmap.w - g->x, shrink_factor);
	/* include row zero with the positively numbered rows */
	rtmp = g->y + 1;
	g->y2 = rtmp / shrink_factor;
	rows = rtmp - g->y2 * shrink_factor;
	if (rows <= 0) {
	    rows += shrink_factor;
	    --g->y2;
	}
	g->bitmap2.h = g->y2 + ROUNDUP((int) g->bitmap.h - rtmp, shrink_factor)
	    + 1;

	if (g->pixmap2 == NULL) {
	    g->image2 = XCreateImage(DISP, our_visual, our_depth, ZPixmap,
	      0, (char *) NULL, g->bitmap2.w, g->bitmap2.h, BMBITS, 0);
	    size = g->image2->bytes_per_line * g->bitmap2.h;
	    g->pixmap2 = g->image2->data = xmalloc(size != 0 ? size : 1);
	}
	if (foreGC2 != NULL && g->pixmap2_t == NULL) {
	    size = g->image2->bytes_per_line * g->bitmap2.h;
	    g->pixmap2_t = xmalloc(size != 0 ? size : 1);
	}

	old_ptr = (BMUNIT *) g->bitmap.bits;
	rows_left = g->bitmap.h;
	y = 0;
	while (rows_left) {
	    x = 0;
	    if (rows > rows_left) rows = rows_left;
	    cols_left = g->bitmap.w;
	    cols = init_cols;
	    while (cols_left) {
		if (cols > cols_left) cols = cols_left;

		thesample = sample(old_ptr, g->bitmap.bytes_wide,
			(int) g->bitmap.w - cols_left, cols, rows);
		XPutPixel(g->image2, x, y, pixeltbl[thesample]);
		if (foreGC2 != NULL) {
		    g->image2->data = g->pixmap2_t;
		    XPutPixel(g->image2, x, y, pixeltbl_t[thesample]);
		    g->image2->data = g->pixmap2;
		}

		cols_left -= cols;
		cols = shrink_factor;
		x++;
	    }
	    *((char **) &old_ptr) += rows * g->bitmap.bytes_wide;
	    rows_left -= rows;
	    rows = shrink_factor;
	    y++;
	}

	while (y < (int) g->bitmap2.h) {
	    for (x = 0; x < (int) g->bitmap2.w; x++) {
		XPutPixel(g->image2, x, y, *pixeltbl);
		if (foreGC2 != NULL) {
		    g->image2->data = g->pixmap2_t;
		    XPutPixel(g->image2, x, y, *pixeltbl_t);
		    g->image2->data = g->pixmap2;
		}
	    }
	    y++;
	}

	g->y2 = g->y / shrink_factor;
#if COLOR
	g->fg = fg_current;
#endif
}
#endif	/* GREY */

/*
 *	Find font #n.
 */

static	void
change_font(n)
	unsigned long	n;
{
	struct tn	*tnp;

	if (n < currinf.tn_table_len) currinf.fontp = currinf.tn_table[n];
	else {
	    currinf.fontp = NULL;
	    for (tnp = currinf.tn_head; tnp != NULL; tnp = tnp->next)
		if (tnp->TeXnumber == n) {
		    currinf.fontp = tnp->fontp;
		    break;
		}
	}
	if (currinf.fontp == NULL) tell_oops("non-existent font #%d", n);
	maxchar = currinf.fontp->maxchar;
	currinf.set_char_p = currinf.fontp->set_char_p;
}


/*
 *	Open a font file.
 */

static	void
open_font_file(fontp)
	struct font *fontp;
{
	if (fontp->file == NULL) {
	    fontp->file = xfopen(fontp->filename, OPEN_MODE);
	    if (fontp->file == NULL)
		oops("Font file disappeared:  %s", fontp->filename);
	}
}

/*
 *	Read special string.
 */

static	char *
read_special(nbytes)
	long	nbytes;
{
	static	char	*spcl	= NULL;
	static	long	spcl_len = -1;
	char	*p;

	if (spcl_len < nbytes) {
	    if (spcl != NULL) free(spcl);
	    spcl = xmalloc((unsigned) nbytes + 1);
	    spcl_len = nbytes;
	}
	p = spcl;
	for (;;) {
	    int i = currinf.end - currinf.pos;

	    if (i > nbytes) i = nbytes;
	    bcopy((char *) currinf.pos, p, i);
	    currinf.pos += i;
	    p += i;
	    nbytes -= i;
	    if (nbytes == 0) break;
	    (void) xxone();
	    --currinf.pos;
	}
	*p = '\0';
	return spcl;
}


/*
 *	Table used for scanning.  If >= 0, then skip that many bytes.
 *	M1 means end of page, M2 means special, M3 means FNTDEF,
 *	M4 means unrecognizable, and M5 means doesn't belong here.
 */

#define	M1	255
#define	M2	254
#define	M3	253
#define	M4	252
#define	M5	251
#define	MM	251

static	ubyte	scantable[256] = {
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,	/* chars 0 - 127 */
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	1,M4,			/* SET1,- (128,129) */
			/* -,-,SETRULE,PUT1,-,-,-,PUTRULE,NOP,BOP (130-139) */
	M4,M4,8,1,M4,M4,M4,8,0,44,
	M1,0,0,1,2,3,4,0,1,2,	/* EOP,PUSH,POP,RIGHT1-4,W0M2 (140-149) */
	3,4,0,1,2,3,4,1,2,3,	/* W3-4,X0-4,DOWN1-3 (150-159) */
	4,0,1,2,3,4,0,1,2,3,	/* DOWN4,Y0-4,Z0-3 (160-169) */
	4,			/* Z4 (170) */
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,	/* change font 171 - 234 */
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	1,2,3,4,M2,		/* FNT1-4,XXX1 (235-239) */
			/* XXX2-4,FNTDEF1-4,PRE,POST,POSTPOST (240-249) */
	M2,M2,M2,M3,M3,M3,M3,M5,M5,M5,
	0,0,M4,M4,M4,M4};	/* SREFL,EREFL,-,-,-,- (250-255) */

/*
 *	This is the generic scanning routine.  It assumes that currinf, etc.
 *	are ready to go at the start of the page to be scanned.
 */

static	void
spcl_scan(spcl_proc)
	void	(*spcl_proc) ARGS((char *));
{
	ubyte	ch;
	ubyte	n;
	long	a;

	for (;;) {
	    ch = xone();
	    n = scantable[ch];
	    if (n < MM)
		while (n-- != 0)
		    (void) xone();
	    else if (n == M1) break;	/* end of page */
	    else switch (n) {
		case M2:	/* special */
		    a = xnum(ch - XXX1 + 1);
		    if (a > 0)
			spcl_proc(read_special(a));
		    break;
		case M3:	/* FNTDEF */
		    xskip((long) (12 + ch - FNTDEF1 + 1));
		    ch = xone();
		    xskip((long) ch + (long) xone());
		    break;
		case M4:	/* unrecognizable */
		    tell_oops("unknown op-code %d", ch);
		    break;
		case M5:	/* doesn't belong */
		    tell_oops("shouldn't happen: %s encountered",
		      dvi_table2[ch - (FNTNUM0 + 64)]);
		    break;
	    }
	}
}


/*
 *	Size of page interval for "Scanning pages xx-xx" message.
 */

#ifndef	REPORT_INCR
#define	REPORT_INCR	50
#endif

/*
 *	Prescanning routine for dvi file.  This looks for specials like
 *	`header=' and `!'.
 */

#if TOOLKIT

static	void	show_status ARGS((struct xtimer *));

static	struct xtimer	pst = TIMER_INIT(show_status);

struct prescan_rec {
	Boolean			timer_active;
	Boolean			popup_active;
	struct status_popup	*pp;
	char			*scanmsg;
};

static	struct prescan_rec	*psrp	= NULL;

/* Window manager destroy callback */
/* ARGSUSED */
static	void
status_cb_destroy(w, client_data, call_data)
	Widget		w;
	XtPointer	client_data;
	XtPointer	call_data;
{
	psrp->popup_active = False;
	simple_popdown(psrp->pp);
}

static	void
show_status(arg)
	struct xtimer	*arg;
{
	psrp->timer_active = False;
	psrp->popup_active = True;
	simple_popup(psrp->pp, psrp->scanmsg, status_cb_destroy);
}

#endif /* TOOLKIT */

void
prescan()
{
	int	nextreportpage;
	char	scanmsg[40];
#if TOOLKIT
	static	struct status_popup	popup	= {NULL, NULL, 0, False, False};
	struct prescan_rec		pr;
#endif

	nextreportpage = scanned_page;
	(void) lseek(fileno(dvi_file), page_info[scanned_page + 1].offset,
	  SEEK_SET);
	currinf.pos = currinf.end = dvi_buffer;

#if TOOLKIT
	pr.timer_active = True;
	pr.popup_active = False;
	pr.pp = &popup;
	pr.scanmsg = scanmsg;
	psrp = &pr;
	set_timer(&pst, 1000);
#endif
	for (;;) {
	    page_info[scanned_page + 1].pw = page_info[scanned_page].pw;
	    page_info[scanned_page + 1].ph = page_info[scanned_page].ph;
	    page_info[scanned_page + 1].ww = page_info[scanned_page].ww;
	    page_info[scanned_page + 1].wh = page_info[scanned_page].wh;

	    if (debug & DBG_PS)
		Printf("Scanning page %d\n", scanned_page + 1);
	    if (scanned_page >= nextreportpage) {
		nextreportpage += REPORT_INCR;
		if (nextreportpage > current_page)
		    nextreportpage = current_page;
		Sprintf(scanmsg, "Scanning pages %d-%d",
		    scanned_page + 1 + pageno_correct,
		    nextreportpage + pageno_correct);
#if XAW
		if (pr.popup_active)
		    XtVaSetValues(popup.label, XtNlabel, scanmsg, NULL);
#elif MOTIF
		if (pr.popup_active) {
		    XmString	str;

		    str = XmStringCreateLocalized((char *) scanmsg);
		    XtVaSetValues(popup.label, XmNlabelString, str, NULL);
		    XmStringFree(str);
		}
#else /* no toolkit */
		if (top_level != 0) {
		    showmessage(scanmsg);
		    XFlush(DISP);
		}
#endif
	    }

	    if (read_events(EV_NOWAIT) & EV_GE_NEWPAGE) break;
#if PS
	    /* NOTE:  longjmp(canit_env) should not be done within
	       read_events(). */
	    if (!setjmp(canit_env))
		spcl_scan(scan_special);
	    else {		/* if interrupted */
		psp.interrupt();
		break;
	    }
	    if (ev_flags & EV_GE_NEWPAGE) break;
#else
	    /* No calls to read_events() occur in this case.  */
	    spcl_scan(scan_special);
#endif

	    ++scanned_page;
#if COLOR
	    if (scanned_page_color < scanned_page) {
		scan_color_eop();
		scanned_page_color = scanned_page;
	    }
#endif
#if PS
	    if (scanned_page_ps < scanned_page)
		scanned_page_ps = scanned_page;
#endif

	    if (scanned_page >= current_page) break;
	}

#if TOOLKIT
	if (pr.popup_active)
	    simple_popdown(&popup);
	else if (pr.timer_active)
	    cancel_timer(&pst);
#else /* !TOOLKIT */
	if (top_level != 0)
	    XClearWindow(DISP, mane.win);
#endif

#if PS
	if (!(ev_flags & EV_GE_NEWPAGE))
	    psp.endheader();
#endif
}

/*
 *	Routines to print characters.
 */

#ifndef	TEXXET
#define	ERRVAL	0L
#else
#define	ERRVAL
#endif

#ifndef	TEXXET
long
set_char P1C(wide_ubyte, ch)
#else
void
set_char P2C(wide_ubyte, cmd, wide_ubyte, ch)
#endif
{
	struct glyph	*g;
#ifdef	TEXXET
	long		dvi_h_sav;
#endif

	if (ch > maxchar) realloc_font(currinf.fontp, WIDENINT ch);
	if ((g = &currinf.fontp->glyph[ch])->bitmap.bits == NULL) {
	    if (g->addr == 0) {
		if (!hush_chars)
		    Fprintf(stderr, "Character %d not defined in font %s\n", ch,
			currinf.fontp->fontname);
		g->addr = -1;
		return ERRVAL;
	    }
	    if (g->addr == -1)
		return ERRVAL;	/* previously flagged missing char */
	    open_font_file(currinf.fontp);
	    Fseek(currinf.fontp->file, g->addr, 0);
	    (*currinf.fontp->read_char)(currinf.fontp, ch);
	    if (debug & DBG_BITMAP) print_char((ubyte) ch, g);
	    currinf.fontp->timestamp = ++current_timestamp;
	}

#ifdef	TEXXET
	dvi_h_sav = DVI_H;
	if (currinf.dir < 0) DVI_H -= g->dvi_adv;
	if (scan_frame == NULL) {
#endif
	    if (shrink_factor == 1)
		put_bitmap(&g->bitmap, PXL_H - g->x, PXL_V - g->y);
	    else {
#ifdef	GREY
		if (use_grey) {
		    if (g->pixmap2 == NULL) {
			shrink_glyph_grey(g);
		    }
		    put_image(g, PXL_H - g->x2, PXL_V - g->y2);
		} else {
		    if (g->bitmap2.bits == NULL) {
			shrink_glyph(g);
		    }
		    put_bitmap(&g->bitmap2, PXL_H - g->x2, PXL_V - g->y2);
		}
#else
		if (g->bitmap2.bits == NULL) {
		    shrink_glyph(g);
		}
		put_bitmap(&g->bitmap2, PXL_H - g->x2, PXL_V - g->y2);
#endif
	    }
#ifndef	TEXXET
	return g->dvi_adv;
#else
	}
	if (cmd == PUT1)
	    DVI_H = dvi_h_sav;
	else
	    if (currinf.dir > 0) DVI_H += g->dvi_adv;
#endif
}


	/* ARGSUSED */
#ifndef	TEXXET
static	long
set_empty_char P1C(wide_ubyte, ch)
#else
static	void
set_empty_char P2C(wide_ubyte, cmd, wide_ubyte, ch)
#endif
{
#ifndef	TEXXET
	return 0;
#else
	return;
#endif
}


#ifndef	TEXXET
long
load_n_set_char P1C(wide_ubyte, ch)
#else
void
load_n_set_char P2C(wide_ubyte, cmd, wide_ubyte, ch)
#endif
{
	if (!load_font(currinf.fontp)) {	/* if not found */
	    if (ev_flags & EV_GE_NEWDOC)	/* if abort */
		longjmp(canit_env, 1);
	    Fputs("Character(s) will be left blank.\n", stderr);
	    currinf.set_char_p = currinf.fontp->set_char_p = set_empty_char;
#ifndef	TEXXET
	    return 0;
#else
	    return;
#endif
	}
	maxchar = currinf.fontp->maxchar;
	currinf.set_char_p = currinf.fontp->set_char_p;
#ifndef	TEXXET
	return (*currinf.set_char_p)(ch);
#else
	(*currinf.set_char_p)(cmd, ch);
	return;
#endif
}


#ifndef	TEXXET
long
set_vf_char P1C(wide_ubyte, ch)
#else
void
set_vf_char P2C(wide_ubyte, cmd, wide_ubyte, ch)
#endif
{
	struct macro	*m;
	struct drawinf	oldinfo;
	ubyte		oldmaxchar;
	static	ubyte	c;
#ifdef	TEXXET
	long		dvi_h_sav;
#endif

	if (ch > maxchar) realloc_virtual_font(currinf.fontp, ch);
	if ((m = &currinf.fontp->macro[ch])->pos == NULL) {
	    if (!hush_chars)
		Fprintf(stderr, "Character %d not defined in font %s\n", ch,
		    currinf.fontp->fontname);
	    m->pos = m->end = &c;
	    return ERRVAL;
	}
#ifdef	TEXXET
	dvi_h_sav = DVI_H;
	if (currinf.dir < 0) DVI_H -= m->dvi_adv;
	if (scan_frame == NULL) {
#endif
	    oldinfo = currinf;
	    if (!currinf.virtual)
		dvi_pointer_frame = &oldinfo;
	    oldmaxchar = maxchar;
	    WW = XX = YY = ZZ = 0;
	    currinf.tn_table_len = VFTABLELEN;
	    currinf.tn_table = currinf.fontp->vf_table;
	    currinf.tn_head = currinf.fontp->vf_chain;
	    currinf.pos = m->pos;
	    currinf.end = m->end;
	    currinf.virtual = currinf.fontp;
	    draw_part(current_frame, currinf.fontp->dimconv);
	    if (currinf.pos != currinf.end + 1)
		tell_oops("virtual character macro does not end correctly");
	    currinf = oldinfo;
	    if (!currinf.virtual)
		dvi_pointer_frame = &currinf;
	    maxchar = oldmaxchar;
#ifndef	TEXXET
	return m->dvi_adv;
#else
	}
	if (cmd == PUT1)
	    DVI_H = dvi_h_sav;
	else
	    if (currinf.dir > 0) DVI_H += m->dvi_adv;
#endif
}


#ifndef	TEXXET
static	long
set_no_char P1C(wide_ubyte, ch)
#else
static	void
set_no_char P2C(wide_ubyte, cmd, wide_ubyte, ch)
#endif
{
	if (currinf.virtual) {
	    currinf.fontp = currinf.virtual->first_font;
	    if (currinf.fontp != NULL) {
		maxchar = currinf.fontp->maxchar;
		currinf.set_char_p = currinf.fontp->set_char_p;
#ifndef	TEXXET
		return (*currinf.set_char_p)(ch);
#else
		(*currinf.set_char_p)(cmd, ch);
		return;
#endif
	    }
	}
	tell_oops("attempt to set character of unknown font");
	/* NOTREACHED */
}


/*
 *	Set rule.  Arguments are coordinates of lower left corner.
 */

static	void
set_rule(h, w)
	int	h, w;
{
#ifndef	TEXXET
	put_rule(PXL_H, PXL_V - h + 1, (unsigned int) w, (unsigned int) h);
#else
	put_rule(PXL_H - (currinf.dir < 0 ? w - 1 : 0), PXL_V - h + 1,
	    (unsigned int) w, (unsigned int) h);
#endif
}


/*
 *	Interpret a sequence of dvi bytes (either the page from the dvi file,
 *	or a character from a virtual font).
 */

#define	xspell_conv(n)	spell_conv0(n, current_dimconv)

static	void
draw_part(minframe, current_dimconv)
	struct frame	*minframe;
	double		current_dimconv;
{
	ubyte ch;
#ifdef	TEXXET
	struct drawinf	oldinfo;
	ubyte	oldmaxchar;
	off_t	file_pos;
	int	refl_count;
#endif

	currinf.fontp = NULL;
	currinf.set_char_p = set_no_char;
#ifdef	TEXXET
	currinf.dir = 1;
	scan_frame = NULL;	/* indicates we're not scanning */
#endif
	for (;;) {
	    ch = xone();
	    if (debug & DBG_DVI)
		print_dvi(ch);
	    if (ch <= (ubyte) (SETCHAR0 + 127))
#ifndef	TEXXET
		DVI_H += (*currinf.set_char_p)(ch);
#else
		(*currinf.set_char_p)(ch, ch);
#endif
	    else if (FNTNUM0 <= ch && ch <= (ubyte) (FNTNUM0 + 63))
		change_font((unsigned long) (ch - FNTNUM0));
	    else {
		long a, b;

		switch (ch) {
		    case SET1:
		    case PUT1:
#ifndef	TEXXET
			a = (*currinf.set_char_p)(xone());
			if (ch != PUT1) DVI_H += a;
#else
			(*currinf.set_char_p)(ch, xone());
#endif
			break;

		    case SETRULE:
			/* Be careful, dvicopy outputs rules with
			   height = 0x80000000.  We don't want any
			   SIGFPE here. */
			a = xsfour();
			b = xspell_conv(xsfour());
#ifndef	TEXXET
			if (a > 0 && b > 0)
#else
			if (a > 0 && b > 0 && scan_frame == NULL)
#endif
			    set_rule(pixel_round(xspell_conv(a)),
				pixel_round(b));
			DVI_H += DIR * b;
			break;

		    case PUTRULE:
			a = xspell_conv(xsfour());
			b = xspell_conv(xsfour());
#ifndef	TEXXET
			if (a > 0 && b > 0)
#else
			if (a > 0 && b > 0 && scan_frame == NULL)
#endif
			    set_rule(pixel_round(a), pixel_round(b));
			break;

		    case NOP:
			break;

		    case BOP:
			xskip((long) 11 * 4);
			DVI_H = OFFSET_X;
			DVI_V = OFFSET_Y;
			PXL_V = pixel_conv(DVI_V);
			WW = XX = YY = ZZ = 0;
			break;

		    case EOP:
			if (current_frame != minframe)
			    tell_oops("stack not empty at EOP");
			return;

		    case PUSH:
			if (current_frame->next == NULL) {
			    struct frame *newp = xmalloc(sizeof(struct frame));

			    current_frame->next = newp;
			    newp->prev = current_frame;
			    newp->next = NULL;
			}
			current_frame = current_frame->next;
			current_frame->data = currinf.data;
			break;

		    case POP:
			if (current_frame == minframe)
			    tell_oops("more POPs than PUSHes");
			currinf.data = current_frame->data;
			current_frame = current_frame->prev;
			break;

#ifdef	TEXXET
		    case SREFL:
			if (scan_frame == NULL) {
			    /* we're not scanning:  save some info. */
			    oldinfo = currinf;
			    oldmaxchar = maxchar;
			    if (!currinf.virtual)
				file_pos = xtell(currinf.pos);
			    scan_frame = current_frame; /* now we're scanning */
			    refl_count = 0;
			    break;
			}
			/* we are scanning */
			if (current_frame == scan_frame) ++refl_count;
			break;

		    case EREFL:
			if (scan_frame != NULL) {	/* if we're scanning */
			    if (current_frame == scan_frame && --refl_count < 0)
			    {
				/* we've hit the end of our scan */
				scan_frame = NULL;
				/* first:  push */
				if (current_frame->next == NULL) {
				    struct frame *newp =
					xmalloc(sizeof(struct frame));

				    current_frame->next = newp;
				    newp->prev = current_frame;
				    newp->next = NULL;
				}
				current_frame = current_frame->next;
				current_frame->data = currinf.data;
				/* next:  restore old file position, XX, etc. */
				if (!currinf.virtual) {
				    off_t bgn_pos = xtell(dvi_buffer);

				    if (file_pos >= bgn_pos) {
					oldinfo.pos = dvi_buffer
					    + (file_pos - bgn_pos);
					oldinfo.end = currinf.end;
				    }
				    else {
					(void) lseek(fileno(dvi_file), file_pos,
					    SEEK_SET);
					oldinfo.pos = oldinfo.end;
				    }
				}
				currinf = oldinfo;
				maxchar = oldmaxchar;
				/* and then:  recover position info. */
				DVI_H = current_frame->data.dvi_h;
				DVI_V = current_frame->data.dvi_v;
				PXL_V = current_frame->data.pxl_v;
				/* and finally, reverse direction */
				currinf.dir = -currinf.dir;
			    }
			    break;
			}
			/* we're not scanning, */
			/* so just reverse direction and then pop */
			currinf.dir = -currinf.dir;
			currinf.data = current_frame->data;
			current_frame = current_frame->prev;
			break;
#endif	/* TEXXET */

		    case RIGHT1:
		    case RIGHT2:
		    case RIGHT3:
		    case RIGHT4:
			DVI_H += DIR * xspell_conv(xsnum(ch - RIGHT1 + 1));
			break;

		    case W1:
		    case W2:
		    case W3:
		    case W4:
			WW = xspell_conv(xsnum(ch - W0));
		    case W0:
			DVI_H += DIR * WW;
			break;

		    case X1:
		    case X2:
		    case X3:
		    case X4:
			XX = xspell_conv(xsnum(ch - X0));
		    case X0:
			DVI_H += DIR * XX;
			break;

		    case DOWN1:
		    case DOWN2:
		    case DOWN3:
		    case DOWN4:
			DVI_V += xspell_conv(xsnum(ch - DOWN1 + 1));
			PXL_V = pixel_conv(DVI_V);
			break;

		    case Y1:
		    case Y2:
		    case Y3:
		    case Y4:
			YY = xspell_conv(xsnum(ch - Y0));
		    case Y0:
			DVI_V += YY;
			PXL_V = pixel_conv(DVI_V);
			break;

		    case Z1:
		    case Z2:
		    case Z3:
		    case Z4:
			ZZ = xspell_conv(xsnum(ch - Z0));
		    case Z0:
			DVI_V += ZZ;
			PXL_V = pixel_conv(DVI_V);
			break;

		    case FNT1:
		    case FNT2:
		    case FNT3:
		    case FNT4:
			change_font(xnum(ch - FNT1 + 1));
			break;

		    case XXX1:
		    case XXX2:
		    case XXX3:
		    case XXX4:
			a = xnum(ch - XXX1 + 1);
			if (a > 0)
			    applicationDoSpecial(read_special(a));
			break;

		    case FNTDEF1:
		    case FNTDEF2:
		    case FNTDEF3:
		    case FNTDEF4:
			xskip((long) (12 + ch - FNTDEF1 + 1));
			a = (long) xone();
			xskip(a + (long) xone());
			break;

#ifndef	TEXXET
		    case SREFL:
		    case EREFL:
#endif
		    case PRE:
		    case POST:
		    case POSTPOST:
			tell_oops("shouldn't happen: %s encountered",
				dvi_table2[ch - (FNTNUM0 + 64)]);
			break;

		    default:
			tell_oops("unknown op-code %d", ch);
		} /* end switch*/
	    } /* end else (ch not a SETCHAR or FNTNUM) */
	} /* end for */
}

#undef	xspell_conv

void
draw_page()
{
	/* Check for changes in dvi file. */
	if (!check_dvi_file()) return;

#if COLOR
	color_bottom = &fg_initial;
	color_bot_size = 1;
	if (page_colors != NULL && current_page > 0) {
	    color_bottom = page_colors[current_page - 1].colorstack;
	    color_bot_size = page_colors[current_page - 1].stacksize;
	}
	rcs_top = NULL;
	set_fg_color(&color_bottom[color_bot_size - 1]);
#endif

	/* draw border */
	XDrawRectangle(DISP, currwin.win, highGC,
	  -currwin.base_x, -currwin.base_y,
	  ROUNDUP(page_info[current_page].pw, shrink_factor) + 1,
	  ROUNDUP(page_info[current_page].ph, shrink_factor) + 1);

	(void) lseek(fileno(dvi_file), page_info[current_page].offset,
	  SEEK_SET);

	bzero((char *) &currinf.data, sizeof currinf.data);
	currinf.tn_table_len = TNTABLELEN;
	currinf.tn_table = tn_table;
	currinf.tn_head = tn_head;
	currinf.pos = currinf.end = dvi_buffer;
	currinf.virtual = NULL;
	dvi_pointer_frame = &currinf;
	psfig_begun = False;
	drawing_mag = (currwin.win == alt.win);

	/* NOTE:  longjmp(canit_env) should not be done within read_events(). */
	if (!setjmp(canit_env))
	    draw_part(current_frame = &frame0, dimconv);
	else {		/* if interrupted */
	    /* If we were interrupted, put the expose event back, so that the
	     * region gets redrawn.  The if statement is necessary because the
	     * magnifier may have been destroyed as part of the interrupt.  */
	    if (currwin.win == mane.win || currwin.win == alt.win)
		expose(currwin.win == mane.win ? &mane : &alt,
		  min_x - currwin.base_x, min_y - currwin.base_y,
		  max_x - min_x, max_y - min_y);
#if PS
	    psp.interrupt();
#endif
	    ev_flags &= ~EV_MAG_GONE;
	}

	drawing_mag = False;
	dvi_pointer_frame = NULL;
	if (currwin.win == mane.win && source_fwd_box_page >= 0)
	    source_fwd_draw_box();	/* draw box showing found source line */
#if PS
	psp.endpage();
#endif
	currwin.win = (Window) 0;
}


/*
 *	General dvi scanning routines.  These are used for:
 *	    o	source special lookups and
 *	    o	finding the dimensions of links (if compiling with support for
 *		hypertext specials).
 *	This routine can be a bit slower than draw_page()/draw_part(), since
 *	it is not run that often; that is why it is a separate routine in
 *	spite of much duplication.
 *
 *	Note that it does not use a separate copy of define_font().
 */

/*
 *	All calculations are done with shrink factor = 1, so we re-do some
 *	macros accordingly.  Many of these are also defined in special.c.
 */

#define	xspell_conv(n)	spell_conv0(n, current_dimconv)
#define	xpixel_conv(x)	((int) ((x) >> 16))
#define	xpixel_round(x)	((int) ROUNDUP(x, 1 << 16))

#define	G_PXL_H		xpixel_conv(currinf.data.dvi_h)
#define	G_OFFSET_X	(offset_x << 16) + (3 << 15)
#define	G_OFFSET_Y	(offset_y << 16) + (3 << 15)

#if TOOLKIT
# define mane_base_x	0
# define mane_base_y	0
#else
# define mane_base_x	mane.base_x
# define mane_base_y	mane.base_y
#endif

/*
 *	This set of routines can be called while draw_part() is active,
 *	so the global variables must be separate.
 */

static	struct frame	geom_frame0;		/* dummy head of list */
#if TEXXET
static	struct frame	*geom_scan_frame;	/* head frame for scanning */
#endif
static	struct frame	*geom_current_frame;

static	void		geom_scan_part ARGS((struct geom_info *,
			  struct frame *, double));

/*
 *	Handle a character in geometric scanning routine.
 */

static	long
geom_do_char(g_info, ch)
	struct geom_info	*g_info;
	wide_ubyte		ch;
{
	if (currinf.set_char_p == set_no_char) {
	    if (currinf.virtual == NULL
	      || (currinf.fontp = currinf.virtual->first_font) == NULL)
		return 0;	/* error; we'll catch it later */
	    maxchar = currinf.fontp->maxchar;
	    currinf.set_char_p = currinf.fontp->set_char_p;
	}

	if (currinf.set_char_p == set_empty_char)
	    return 0;	/* error; we'll catch it later */

	if (currinf.set_char_p == load_n_set_char) {
	    if (ev_flags & EV_GE_NEWDOC)	/* if abort */
		return 0;
	    if (!load_font(currinf.fontp)) {	/* if not found */
		if (ev_flags & EV_GE_NEWDOC)	/* if abort */
		    return 0;
		Fputs("Character(s) will be left blank.\n", stderr);
		currinf.set_char_p = currinf.fontp->set_char_p = set_empty_char;
		return 0;
	    }
	    maxchar = currinf.fontp->maxchar;
	    currinf.set_char_p = currinf.fontp->set_char_p;
	}

	if (currinf.set_char_p == set_char) {
	    struct glyph *g;
	    long x, y;

	    if (ch > maxchar)
		return 0;	/* catch the error later */
	    if ((g = &currinf.fontp->glyph[ch])->bitmap.bits == NULL) {
		if (g->addr == 0)
		    return 0;	/* catch the error later */
		if (g->addr == -1)
		    return 0;	/* previously flagged missing char */
		open_font_file(currinf.fontp);
		Fseek(currinf.fontp->file, g->addr, 0);
		(*currinf.fontp->read_char)(currinf.fontp, ch);
		if (debug & DBG_BITMAP) print_char((ubyte) ch, g);
		currinf.fontp->timestamp = ++current_timestamp;
	    }
#if TEXXET
	    if (geom_scan_frame == NULL) {
		long dvi_h_sav = DVI_H;
		if (currinf.dir < 0) DVI_H -= g->dvi_adv;
#endif
		x = G_PXL_H - g->x;
		y = PXL_V - g->y;
		g_info->geom_box(g_info, x, y,
		  x + g->bitmap.w - 1, y + g->bitmap.h - 1);

#if TEXXET
		DVI_H = dvi_h_sav;
	    }
#endif
	    return DIR * g->dvi_adv;
	}
	else if (currinf.set_char_p == set_vf_char) {
	    struct macro	*m;
	    struct drawinf	oldinfo;
	    ubyte		oldmaxchar;
#if TEXXET
	    long		dvi_h_sav;
#endif

	    if (ch > maxchar)
		return 0;	/* catch the error later */
	    if ((m = &currinf.fontp->macro[ch])->pos == NULL)
		return 0;	/* catch the error later */
#if TEXXET
	    dvi_h_sav = DVI_H;
	    if (currinf.dir < 0) DVI_H -= m->dvi_adv;
	    if (geom_scan_frame == NULL) {
#endif
		oldinfo = currinf;
		oldmaxchar = maxchar;
		WW = XX = YY = ZZ = 0;
		currinf.tn_table_len = VFTABLELEN;
		currinf.tn_table = currinf.fontp->vf_table;
		currinf.tn_head = currinf.fontp->vf_chain;
		currinf.pos = m->pos;
		currinf.end = m->end;
		currinf.virtual = currinf.fontp;
		geom_scan_part(g_info, geom_current_frame,
		  currinf.fontp->dimconv);
		currinf = oldinfo;
		maxchar = oldmaxchar;
#if TEXXET
		DVI_H = dvi_h_sav;
	    }
#endif
	    return DIR * m->dvi_adv;
	}
	else {
	    oops("internal error -- currinf.set_char_p = 0x%x",
		currinf.set_char_p);
	    /* NOTREACHED */
	}
}

/*
 *	Do a rule in the geometry-scanning routine.
 */

static	void
geom_do_rule(g_info, h, w)
	struct geom_info	*g_info;
	long			h, w;
{
	long			x, y;
#if TEXXET
	long			dvi_h_save = DVI_H;
#endif

#if TEXXET
	if (currinf.dir < 0) DVI_H -= w - 1;
#endif
	x = G_PXL_H;
	y = PXL_V;
	g_info->geom_box(g_info, x, y - xpixel_round(h) + 1,
	  x + xpixel_round(w) - 1, y);
#if TEXXET
	DVI_H = dvi_h_save;
#endif
}


/*
 *	Geometric dvi scanner work routine.  This does most of the work
 *	(a) reading from a page, and (b) executing vf macros.
 */

static	void
geom_scan_part(g_info, minframe, current_dimconv)
	struct geom_info	*g_info;
	struct frame		*minframe;
	double			current_dimconv;
{
	ubyte				ch;
#if TEXXET
	struct drawinf	oldinfo;
	ubyte	oldmaxchar;
	off_t	file_pos;
	int	refl_count;
#endif

	currinf.fontp = NULL;
	currinf.set_char_p = set_no_char;
#if TEXXET
	currinf.dir = 1;
	geom_scan_frame = NULL;	/* indicates we're not scanning */
#endif
	for (;;) {
	    ch = xone();
	    if (ch <= (ubyte) (SETCHAR0 + 127))
		DVI_H += geom_do_char(g_info, ch);
	    else if (FNTNUM0 <= ch && ch <= (ubyte) (FNTNUM0 + 63))
		change_font((unsigned long) (ch - FNTNUM0));
	    else {
		long	a, b;

		switch (ch) {
		    case SET1:
		    case PUT1:
			a = geom_do_char(g_info, xone());
			if (ch != PUT1) DVI_H += a;
			break;

		    case SETRULE:
			/* Be careful, dvicopy outputs rules with
			   height = 0x80000000.  We don't want any
			   SIGFPE here. */
			a = xsfour();
			b = xspell_conv(xsfour());
#if TEXXET
			if (a >= 0 && b >= 0 && geom_scan_frame == NULL)
#else
			if (a >= 0 && b >= 0)
#endif
			    geom_do_rule(g_info, xspell_conv(a), b);
			DVI_H += DIR * b;
			break;

		    case PUTRULE:
			a = xspell_conv(xsfour());
			b = xspell_conv(xsfour());
#if TEXXET
			if (a >= 0 && b >= 0 && geom_scan_frame == NULL)
#else
			if (a >= 0 && b >= 0)
#endif
			    geom_do_rule(g_info, a, b);
			break;

		    case NOP:
			break;

		    case BOP:
			xskip((long) 11 * 4);
			DVI_H = G_OFFSET_X;
			DVI_V = G_OFFSET_Y;
			PXL_V = xpixel_conv(DVI_V);
			WW = XX = YY = ZZ = 0;
			break;

		    case PUSH:
			if (geom_current_frame->next == NULL) {
			    struct frame *newp = xmalloc(sizeof(struct frame));

			    geom_current_frame->next = newp;
			    newp->prev = geom_current_frame;
			    newp->next = NULL;
			}
			geom_current_frame = geom_current_frame->next;
			geom_current_frame->data = currinf.data;
			break;

		    case POP:
			if (geom_current_frame == minframe)
			    tell_oops("more POPs than PUSHes");
			currinf.data = geom_current_frame->data;
			geom_current_frame = geom_current_frame->prev;
			break;

#if TEXXET
		    case SREFL:
			if (geom_scan_frame == NULL) {
			    /* we're not scanning:  save some info. */
			    oldinfo = currinf;
			    oldmaxchar = maxchar;
			    if (!currinf.virtual)
				file_pos = xtell(currinf.pos);
			    geom_scan_frame = geom_current_frame; /* start scanning*/
			    refl_count = 0;
			    break;
			}
			/* we are scanning */
			if (geom_current_frame == geom_scan_frame) ++refl_count;
			break;

		    case EREFL:
			if (geom_scan_frame != NULL) {	/* if we're scanning */
			    if (geom_current_frame == geom_scan_frame
			      && --refl_count < 0)
			    {
				/* we've hit the end of our scan */
				geom_scan_frame = NULL;
				/* first:  push */
				if (geom_current_frame->next == NULL) {
				    struct frame *newp =
					xmalloc(sizeof(struct frame));

				    geom_current_frame->next = newp;
				    newp->prev = geom_current_frame;
				    newp->next = NULL;
				}
				geom_current_frame = geom_current_frame->next;
				geom_current_frame->data = currinf.data;
				/* next:  restore old file position, XX, etc. */
				if (!currinf.virtual) {
				    off_t bgn_pos = xtell(dvi_buffer);

				    if (file_pos >= bgn_pos) {
					oldinfo.pos = dvi_buffer
					    + (file_pos - bgn_pos);
					oldinfo.end = currinf.end;
				    }
				    else {
					(void) lseek(fileno(dvi_file), file_pos,
					    SEEK_SET);
					oldinfo.pos = oldinfo.end;
				    }
				}
				currinf = oldinfo;
				maxchar = oldmaxchar;
				/* and then:  recover position info. */
				DVI_H = geom_current_frame->data.dvi_h;
				DVI_V = geom_current_frame->data.dvi_v;
				PXL_V = geom_current_frame->data.pxl_v;
				/* and finally, reverse direction */
				currinf.dir = -currinf.dir;
			    }
			    break;
			}
			/* we're not scanning, */
			/* so just reverse direction and then pop */
			currinf.dir = -currinf.dir;
			currinf.data = geom_current_frame->data;
			geom_current_frame = geom_current_frame->prev;
			break;
#endif /* TEXXET */

		    case RIGHT1:
		    case RIGHT2:
		    case RIGHT3:
		    case RIGHT4:
			DVI_H += DIR * xspell_conv(xsnum(ch - RIGHT1 + 1));
			break;

		    case W1:
		    case W2:
		    case W3:
		    case W4:
			WW = xspell_conv(xsnum(ch - W0));
		    case W0:
			DVI_H += DIR * WW;
			break;

		    case X1:
		    case X2:
		    case X3:
		    case X4:
			XX = xspell_conv(xsnum(ch - X0));
		    case X0:
			DVI_H += DIR * XX;
			break;

		    case DOWN1:
		    case DOWN2:
		    case DOWN3:
		    case DOWN4:
			DVI_V += xspell_conv(xsnum(ch - DOWN1 + 1));
			PXL_V = xpixel_conv(DVI_V);
			break;

		    case Y1:
		    case Y2:
		    case Y3:
		    case Y4:
			YY = xspell_conv(xsnum(ch - Y0));
		    case Y0:
			DVI_V += YY;
			PXL_V = xpixel_conv(DVI_V);
			break;

		    case Z1:
		    case Z2:
		    case Z3:
		    case Z4:
			ZZ = xspell_conv(xsnum(ch - Z0));
		    case Z0:
			DVI_V += ZZ;
			PXL_V = xpixel_conv(DVI_V);
			break;

		    case FNT1:
		    case FNT2:
		    case FNT3:
		    case FNT4:
			change_font(xnum(ch - FNT1 + 1));
			break;

		    case XXX1:
		    case XXX2:
		    case XXX3:
		    case XXX4:
			a = xnum(ch - XXX1 + 1);
			if (a > 0) {
			    char *str = read_special(a);

			    /* process the bounding box */
			    geom_do_special(g_info, str, current_dimconv);

			    /* process the specials we're looking for */
			    g_info->geom_special(g_info, str);
			}
			break;

		    case FNTDEF1:
		    case FNTDEF2:
		    case FNTDEF3:
		    case FNTDEF4:
			xskip((long) (12 + ch - FNTDEF1 + 1));
			a = (long) xone();
			xskip(a + (long) xone());
			break;

#if !TEXXET
		    case SREFL:
		    case EREFL:
#endif
		    case PRE:
		    case POST:
		    case POSTPOST:
		    case EOP:
		    default:
			return;

		} /* end switch*/
	    } /* end else (ch not a SETCHAR or FNTNUM) */
	} /* end for */
}


/*
 *	Main scanning routine.
 */

static	void
geom_scan(g_info)
	struct geom_info	*g_info;
{
	off_t		pos_save;
	struct drawinf	currinf_save;
	ubyte		maxchar_save;

#if PS
	if (scanned_page < current_page) return;	/* should not happen */
#endif

	if (dvi_pointer_frame != NULL)
	    pos_save = lseek(fileno(dvi_file), 0L, SEEK_CUR)
	      - (dvi_pointer_frame->end - dvi_pointer_frame->pos);
	(void) lseek(fileno(dvi_file), page_info[current_page].offset,
	  SEEK_SET);

	currinf_save = currinf;
	maxchar_save = maxchar;

	bzero((char *) &currinf.data, sizeof currinf.data);
	currinf.tn_table_len = TNTABLELEN;
	currinf.tn_table = tn_table;
	currinf.tn_head = tn_head;
	currinf.pos = currinf.end = dvi_buffer;
	currinf.virtual = NULL;

	if (!setjmp(g_info->done_env))
	    geom_scan_part(g_info, geom_current_frame = &geom_frame0, dimconv);

	maxchar = maxchar_save;
	currinf = currinf_save;

	if (dvi_pointer_frame != NULL) {
	    (void) lseek(fileno(dvi_file), pos_save, SEEK_SET);
	    dvi_pointer_frame->pos = dvi_pointer_frame->end = dvi_buffer;
	}
}


/*
 *	Routines for source special lookup
 */

static	void
src_spec_box ARGS((struct geom_info *, long, long, long, long));

static	void
src_spec_special ARGS((struct geom_info *, _Xconst char *));

struct src_parsed_special {
	int		line;
	int		col;
	char		*filename;
	size_t		filename_len;
};

struct src_spec_data {
	long		x, y;		/* coordinates we're looking for */
	unsigned long	distance;	/* best distance so far */
	Boolean		recent_in_best;	/* most recent string == XXX->best */
	struct src_parsed_special	/* best special so far */
			best;
	struct src_parsed_special	/* most recent special */
			recent;
};

static	void
src_parse(str, parsed)
	_Xconst char			*str;
	struct src_parsed_special	*parsed;
{
	_Xconst char	*p;

	p = str + 4;	/* skip "src:" */

	if (*p >= '0' && *p <= '9') {
	    parsed->line = atoi(p);
	    do ++p;
	    while (*p >= '0' && *p <= '9');
	}

	parsed->col = 0;
	if (*p == ':') {
	    ++p;
	    parsed->col = atoi(p);
	    while (*p >= '0' && *p <= '9') ++p;
	}

	if (*p == ' ') ++p;

	if (*p != '\0') {
	    size_t len = strlen(p) + 1;

	    if (len > parsed->filename_len) {
		if (parsed->filename_len != 0)
		    free(parsed->filename);
		parsed->filename_len = (len & -8) + 64;
		parsed->filename = xmalloc(parsed->filename_len);
	    }
	    memcpy(parsed->filename, p, len);
	}
}

static	void
src_spec_box(g_info, ulx, uly, lrx, lry)
	struct geom_info	*g_info;
	long			ulx, uly, lrx, lry;
{
	struct src_spec_data	*data	= g_info->geom_data;
	unsigned long		distance;

	distance = 0;

	if (data->x < ulx) distance += (ulx - data->x) * (ulx - data->x);
	else if (data->x > lrx) distance += (data->x - lrx) * (data->x - lrx);

	if (data->y < uly) distance += (uly - data->y) * (uly - data->y);
	else if (data->y > lry) distance += (data->y - lry) * (data->y - lry);

	if (distance < data->distance) {
	    data->distance = distance;

	    /* Copy it over */
	    if (!data->recent_in_best) {
		data->best.line = data->recent.line;
		data->best.col = data->recent.col;
		if (data->recent.filename_len != 0) {
		    if (data->best.filename_len < data->recent.filename_len) {
			if (data->best.filename_len != 0)
			    free(data->best.filename);
			data->best.filename_len = data->recent.filename_len;
			data->best.filename = xmalloc(data->best.filename_len);
		    }
		    memcpy(data->best.filename, data->recent.filename,
		      data->recent.filename_len);
		}

		data->recent_in_best = True;
	    }

	    /* Quit early if we've found our glyph.  */
	    if (distance == 0 && data->best.filename_len != 0)
		longjmp(g_info->done_env, 1);
	}
}

static	void
src_spec_special(g_info, str)
	struct geom_info	*g_info;
	_Xconst char		*str;
{
	struct src_spec_data	*data	= g_info->geom_data;

	if (memcmp(str, "src:", 4) != 0)
	    return;

	src_parse(str, &data->recent);

	/*
	 * If this is the first special on the page, we may already have
	 * spotted the nearest box.
	 */

	if (data->best.filename_len == 0) {
	    data->best.line = data->recent.line;
	    data->best.col = data->recent.col;
	    if (data->recent.filename_len != 0) {
		if (data->best.filename_len < data->recent.filename_len) {
		    if (data->best.filename_len != 0)
			free(data->best.filename);
		    data->best.filename_len = data->recent.filename_len;
		    data->best.filename = xmalloc(data->best.filename_len);
		}
		memcpy(data->best.filename, data->recent.filename,
		  data->recent.filename_len);

		data->recent_in_best = True;
	    }

	    if (data->distance == 0)
		longjmp(g_info->done_env, 1);
	}
	else
	    data->recent_in_best = False;
}

/*
 *	Routines for reverse searches on other pages.
 */

static	struct src_parsed_special	found;
static	jmp_buf	scan_env;

static	void
scan_first_src_spcl(str)
	char	*str;
{
	if (memcmp(str, "src:", 4) != 0)
	    return;

	src_parse(str, &found);

	longjmp(scan_env, 1);
}

static	void
scan_last_src_spcl(str)
	char	*str;
{
	if (memcmp(str, "src:", 4) != 0)
	    return;

	src_parse(str, &found);
}

/*
 *	Information on how to search for source files.
 */

#include "filf-app.h"		/* application-related defs, etc. */
#include "filefind.h"

static	_Xconst	char	no_f_str_tex[]	= "/%f";

static	struct findrec			search_tex	= {
	/* path1	*/	no_f_str_tex,	/* flag value:  uninitialized */
#if CFGFILE
	/* envptr	*/	NULL,
#endif
	/* path2	*/	DEFAULT_SOURCE_PATH,
	/* type		*/	"source",
	/* fF_etc	*/	"fF",
	/* x_var_char	*/	'f',
	/* n_var_opts	*/	2,
	/* no_f_str	*/	no_f_str_tex,
	/* no_f_str_end	*/	no_f_str_tex + sizeof(no_f_str_tex) - 1,
	/* abs_str	*/	"%f",
#if USE_GF
	/* no_f_str_flags */	F_FILE_USED,
	/* abs_str_flags */	F_FILE_USED,
	/* pk_opt_char	*/	'f',
	/* pk_gf_addr	*/	NULL,
#endif
	/* pct_s_str	*/	"%qtex//",
	{
	  /* v.stephead		*/	NULL,
	  /* v.pct_s_head	*/	NULL,
	  /* v.pct_s_count	*/	0,
	  /* v.pct_s_atom	*/	NULL,
	  /* v.rootp		*/	NULL,
	}
};


#if !HAVE_ULLTOSTR

#define	ulltostr xulltostr	/* prevents collision with header file def. */

static	char *
ulltostr(value, p)
	unsigned long	value;
	char		*p;
{
	do *--p = value % 10 + '0';
	while ((value /= 10) != 0);

	return p;
}

#endif


static	void
src_spawn_editor(parsed)
	_Xconst struct src_parsed_special	*parsed;
{
#if HAVE_ULLTOSTR
	char		scr_str[5 * sizeof(unsigned long long) / 2];
#else
	char		scr_str[5 * sizeof(unsigned long) / 2];
#endif
	_Xconst char	*filename2;
	Boolean		found_filename	= False;
	size_t		buffer_pos;
	int		argc;
	char		**argv;
	_Xconst	char	*p, *p1;
	char		*q;
	FILE		*f;
	int		i;

	/* Used to store argv[] text.  */
	static	char	*buffer;
	static	size_t	buffer_len	= 0;

	/* first, determine the editor if necessary */
	if (resource.editor == NULL || *resource.editor == '\0') {
	    p = getenv("XEDITOR");
	    if (p != NULL)
		resource.editor = xstrdup(p);
	    else {

		p = getenv("VISUAL");
		if (p == NULL) {
		    p = getenv("EDITOR");
		    if (p == NULL)
			p = "vi";
		}
		q = xmalloc(strlen(p) + 10);
		memcpy(q, "xterm -e ", 9);
		strcpy(q + 9, p);
		resource.editor = q;
	    }
	}

	/* Now search for the file (same algorithm as PS files).  */
	f = NULL;
	    /* first try the same path as the dvi file */
	if (parsed->filename[0] != '/') {
	    p = rindex(dvi_name, '/');
	    if (p == NULL) filename2 = parsed->filename;
	    else {
		unsigned len1, len2;

		len1 = ++p - dvi_name;
		len2 = strlen(parsed->filename) + 1;
		if (len1 + len2 > ffline_len)
		    expandline(len1 + len2);
		bcopy(dvi_name, ffline, len1);
		bcopy(parsed->filename, ffline + len1, len2);
		filename2 = ffline;
	    }
	    if (debug & DBG_OPEN) Printf("Trying source special file %s\n",
	      filename2);
	    f = xfopen(filename2, OPEN_MODE);
	}

	if (f == NULL) {

	    /*
	     * Set up the paths, if necessary.
	     */

	    if (search_tex.path1 == no_f_str_tex) {	/* if uninitialized */
#if CFGFILE
		if ((search_tex.path1 = getenv("XDVISOURCES")) == NULL)
		    search_tex.path1 = getenv("TEXINPUTS");
		search_tex.envptr = ffgetenv("TEXINPUTS");
		/* clear it if it's a getenv() placeholder */
		if (search_tex.envptr != NULL
		  && search_tex.envptr->value == NULL)
		    search_tex.envptr = NULL;
#else /* not CFGFILE */
		if ((search_tex.path1 = getenv("XDVISOURCES")) == NULL
		  && (search_tex.path1 = getenv("TEXINPUTS")) == NULL) {
		    search_tex.path1 = search_tex.path2;
		    search_tex.path2 = NULL;
		}
#endif /* not CFGFILE */
	    }

	    f = filefind(parsed->filename, &search_tex, (_Xconst char **) NULL);
	    filename2 = ffline;
	}

	/* if still no luck, complain */
	if (f == NULL)
	    WARN1(XmDIALOG_ERROR, "Cannot find source special file\n`%s'",
		parsed->filename);
	else {
	    struct stat	buf;

	    if (fstat(fileno(f), &buf) == -1) {
		perror(parsed->filename);
	    }
	    else {
		if (buf.st_mtime > dvi_time)
		    WARN1(XmDIALOG_ERROR,
		      "Source file\n%s\nis newer than dvi file",
		      parsed->filename);
	    }

	    Fclose(f);
	    ++n_files_left;

	    if (buffer_len == 0)
		buffer = xmalloc(buffer_len = 128);
	    buffer_pos = 0;
	    argc = 0;

	    p = resource.editor;
	    while (*p == ' ' || *p == '\t') ++p;

	    for (;;) {
		size_t	len;

		if (*p == '%') {
		    p1 = p;
		    switch (p[1]) {
			case '\0':
			    --p;	/* partially undo p += 2 later */
			    /* control falls through */
			case '%':
			    len = 1;
			    break;
			case 'l':
			    p1 = ulltostr(parsed->line,
			      scr_str + sizeof scr_str);
			    len = scr_str + sizeof scr_str - p1;
			    break;
			case 'c':
			    if (parsed->col == 0) {
				p += 2;
				continue;
			    }
			    p1 = ulltostr(parsed->col,
			      scr_str + sizeof scr_str);
			    len = scr_str + sizeof scr_str - p1;
			    break;
			case 'f':
			    p1 = filename2;
			    len = strlen(filename2);
			    found_filename = True;
			    break;
			default:
			    len = 2;
		    }
		    p += 2;
		}
		else if (*p == '\0' || *p == ' ' || *p == '\t') {
		    buffer[buffer_pos++] = '\0';
		    ++argc;
		    while (*p == ' ' || *p == '\t') ++p;
		    if (*p == '\0') {
			if (found_filename)
			    break;	/* done */
			p = "+%l %f";
		    }
		    continue;	/* don't copy anything over */
		}
		else {
		    p1 = p;
		    len = strcspn(p, "% \t");
		    p += len;
		}

		/* copy over len bytes starting at p1 into the buffer, */
		/* leaving at least one byte free */
		if (buffer_pos + len >= buffer_len) {
		    buffer_len = ((buffer_pos + len) / 128 + 1) * 128;
		    buffer = xrealloc(buffer, buffer_len);
		}
		memcpy(buffer + buffer_pos, p1, len);
		buffer_pos += len;
	    }

	    argv = xmalloc((argc + 1) * sizeof(char *));
	    q = buffer;
	    for (i = 0; i < argc; ++i) {
		argv[i] = q;
		q += strlen(q) + 1;
	    }

	    /* NULL-terminate argument list */
	    argv[argc] = NULL;

	    Fflush(stdout);	/* to avoid double buffering */
	    Fflush(stderr);

	    switch (vfork()) {
		case -1:
		    perror("[xdvi] vfork");
		    break;
		case 0: /* child */
		    execvp(argv[0], argv);

		    Fprintf(stderr, "%s: Execvp of %s failed.\n", prog,
			argv[0]);
		    Fflush(stderr);
		    _exit(1);
	    }

	    free(argv);
	}
}


/*
 *	The main routine for source specials (reverse search).
 */

void
source_reverse_search(x, y)
	int	x, y;
{
	struct geom_info	g_info;
	struct src_spec_data	data;
	struct src_parsed_special *foundp;

	g_info.geom_box = src_spec_box;
	g_info.geom_special = src_spec_special;
	g_info.geom_data = &data;

	data.x = x;
	data.y = y;
	data.distance = 0xffffffff;
	data.recent_in_best = True;
	data.best.filename_len = data.recent.filename_len = 0;
	foundp = &data.best;

	geom_scan(&g_info);

	if (data.best.filename_len == 0) {
	    /*
	     * nothing found on current page;
	     * scan next and previous pages with increasing offset
	     */
	    int			upper, lower;
	    off_t		pos_save;
	    struct drawinf	currinf_save;
	    ubyte		maxchar_save;

	    /* Save file position */

	    if (dvi_pointer_frame != NULL)
		pos_save = lseek(fileno(dvi_file), 0L, SEEK_CUR)
		  - (dvi_pointer_frame->end - dvi_pointer_frame->pos);

	    currinf_save = currinf;
	    maxchar_save = maxchar;

	    upper = lower = current_page;
	    found.filename_len = 0;	/* mark it as empty */
	    for (;;) {

		if (++upper < total_pages) {
		    (void) lseek(fileno(dvi_file), page_info[upper].offset,
		      SEEK_SET);
		    bzero((char *) &currinf.data, sizeof currinf.data);
		    currinf.tn_table_len = TNTABLELEN;
		    currinf.tn_table = tn_table;
		    currinf.tn_head = tn_head;
		    currinf.pos = currinf.end = dvi_buffer;
		    currinf.virtual = NULL;

		    if (setjmp(scan_env) == 0) {
			/* find first src special */
			spcl_scan(scan_first_src_spcl);
		    }
		    else {	/* found it */
			lower = upper;
			break;
		    }
		}
		else if (lower < 0)
		    break;

		if (--lower >= 0) {
		    (void) lseek(fileno(dvi_file), page_info[lower].offset,
		      SEEK_SET);
		    bzero((char *) &currinf.data, sizeof currinf.data);
		    currinf.tn_table_len = TNTABLELEN;
		    currinf.tn_table = tn_table;
		    currinf.tn_head = tn_head;
		    currinf.pos = currinf.end = dvi_buffer;
		    currinf.virtual = NULL;

		    spcl_scan(scan_last_src_spcl);
		    if (found.filename_len != 0)
			break;
		}
	    }

	    if (found.filename_len != 0)
		WARN1(XmDIALOG_WARNING,
		  "No source specials on this page - nearest on page %d",
		  lower + pageno_correct);
	    else {
		/* nothing found at all; complain */
		XBell(DISP, 0);
		WARN(XmDIALOG_ERROR, "No source specials in this .dvi file");
	    }

	    /* Restore file status.  */

	    maxchar = maxchar_save;
	    currinf = currinf_save;

	    if (dvi_pointer_frame != NULL) {
		(void) lseek(fileno(dvi_file), pos_save, SEEK_SET);
		dvi_pointer_frame->pos = dvi_pointer_frame->end = dvi_buffer;
	    }

	    foundp = &found;
	}

	if (data.recent.filename_len != 0)
	    free(data.recent.filename);

	if (foundp->filename_len != 0) {
	    src_spawn_editor(foundp);
	    free(foundp->filename);
	}
}


/*
 *	Debug routines for source special display (highlight the first box
 *	after each source special, or highlight each box).
 */

static	void
src_spec_show_box ARGS((struct geom_info *, long, long, long, long));

static	void
src_spec_show_special ARGS((struct geom_info *, _Xconst char *));

struct src_spec_show_data {
	Boolean		do_this_one;	/* flag set after source special */
	Boolean		do_them_all;	/* flag to reset the above to */
};

static	void
src_spec_show_box(g_info, ulx, uly, lrx, lry)
	struct geom_info	*g_info;
	long			ulx, uly, lrx, lry;
{
	struct src_spec_show_data	*data = g_info->geom_data;

	if (data->do_this_one) {
	    long x = ulx / mane.shrinkfactor;
	    long y = uly / mane.shrinkfactor;

	    XDrawRectangle(DISP, mane.win, highGC,
	      x - mane_base_x, y - mane_base_y,
	      lrx / mane.shrinkfactor - x, lry / mane.shrinkfactor - y);

	    data->do_this_one = data->do_them_all;
	}
}

	/* ARGSUSED */
static	void
src_spec_show_special(g_info, str)
	struct geom_info	*g_info;
	_Xconst char		*str;
{
	if (memcmp(str, "src:", 4) != 0)
	    return;

	((struct src_spec_show_data *)g_info->geom_data)->do_this_one = True;
}

void
source_special_show(do_them_all)
	wide_bool	do_them_all;
{
	struct geom_info		g_info;
	struct src_spec_show_data	data;

	g_info.geom_box = src_spec_show_box;
	g_info.geom_special = src_spec_show_special;
	g_info.geom_data = &data;

	data.do_this_one = data.do_them_all = do_them_all;

	geom_scan(&g_info);
}


/*
 *	Routines for forward search (look up a source line).
 *
 *	The general procedure is:
 *	   (1)	Use spcl_scan() to find the page containing the line (or at
 *		least the closest line).  This could be optimized further.
 *	   (2)	Use geom_scan_part() to find the exact location of the source
 *		special, and the box to highlight.
 */

/* These variables are referenced by src_scan_special().  */

static	int		src_this_line;
static	size_t		src_this_file_strcmp;
static	int		src_line;
static	int		src_col;
static	_Xconst char	*src_file;
static	int		src_page;
static	jmp_buf		src_env;
static	Boolean		found_src;
static	unsigned long	best_distance;
static	unsigned long	best_col_dist;
static	int		best_line;
static	int		best_page;
static	off_t		best_offset;
static	off_t		max_offset;

/* Some of the above, plus these below, are used by geom_scan_part().  */

static	Boolean		src_fwd_active;
static	long		src_fwd_min_x, src_fwd_max_x;
static	long		src_fwd_min_y, src_fwd_max_y;
static	long		src_fwd_min_x2, src_fwd_max_x2;	/* hot point for spcl */
static	long		src_fwd_min_y2, src_fwd_max_y2;

static	void
src_scan_special(str)
	char	*str;
{
	char		*p;
	int		col	= 0;
	unsigned long	distance;
	unsigned long	col_dist;

	if (memcmp(str, "src:", 4) != 0)
	    return;

	found_src = True;

	p = str + 4;
	if (*p >= '0' && *p <= '9') {
	    src_this_line = atoi(p);
	    do ++p;
	    while (*p >= '0' && *p <= '9');
	}

	if (*p == ':') {
	    ++p;
	    col = atoi(p);
	    while (*p >= '0' && *p <= '9') ++p;
	}

	if (*p == ' ') ++p;

	if (*p == '.' && p[1] == '/')	/* ignore leading ./ path component */
	    p += 2;

	if (*p != '\0')
	    src_this_file_strcmp = strcmp(p, src_file);

	if (src_this_file_strcmp != 0)
	    return;

	distance = (src_line > src_this_line
	  ? src_line - src_this_line
	  : 2 * (src_this_line - src_line));	/* favor earlier lines */

	if (distance < best_distance) {	/* found a better line */
	    best_distance = distance;
	    best_line = src_this_line;
	    best_page = src_page;
	    max_offset = best_offset = xtell(currinf.pos);
	}
	else if (distance == best_distance)	/* if still on a good line */
	    max_offset = xtell(currinf.pos);

	if (distance == 0 && best_distance == 0) {	/* found a better col */
	    col_dist = (src_col > col
		? src_col - col
		: 2 * (col - src_col));

	    if (col_dist < best_col_dist) {
		best_col_dist = col_dist;
		best_page = src_page;
		max_offset = best_offset = xtell(currinf.pos);
	    }
	    else if (col_dist == best_col_dist)
		max_offset = xtell(currinf.pos);
	}
}

static	void
src_spec_fwd_box ARGS((struct geom_info *, long, long, long, long));

static	void
src_spec_fwd_special ARGS((struct geom_info *, _Xconst char *));

	/* ARGSUSED */
static	void
src_spec_fwd_box(g_info, ulx, uly, lrx, lry)
	struct geom_info	*g_info;
	long			ulx, uly, lrx, lry;
{
	if (src_fwd_active) {
	    if (ulx < src_fwd_min_x) src_fwd_min_x = ulx;
	    if (uly < src_fwd_min_y) src_fwd_min_y = uly;
	    if (lrx > src_fwd_max_x) src_fwd_max_x = lrx;
	    if (lry > src_fwd_max_y) src_fwd_max_y = lry;
	}
}

static	void
src_spec_fwd_special(g_info, str)
	struct geom_info	*g_info;
	_Xconst char		*str;
{
	long	pos;

	if (memcmp(str, "src:", 4) != 0)	/* if not "src:" */
	    return;

	pos = xtell(currinf.pos);
	if (pos >= best_offset)
	    src_fwd_active = True;

	if (src_fwd_active) {
	    if (G_PXL_H < src_fwd_min_x2) src_fwd_min_x2 = G_PXL_H;
	    if (G_PXL_H > src_fwd_max_x2) src_fwd_max_x2 = G_PXL_H;
	    if (PXL_V < src_fwd_min_y2) src_fwd_min_y2 = PXL_V;
	    if (PXL_V > src_fwd_max_y2) src_fwd_max_y2 = PXL_V;

	    if (pos > max_offset)
		longjmp(g_info->done_env, 1);
	}
}

/*
 *	Routine to actually draw the box.
 */

static	void
source_fwd_draw_box()
{
	long	x, y;

	if (source_fwd_box_page != current_page)
	    source_fwd_box_page = -1;	/* different page---clear it */
	else {
	    if (src_fwd_min_x == 0x7fffffff) {
		/* If no glyphs or rules, use hot point of special instead.  */
		src_fwd_min_x = src_fwd_min_x2;
		src_fwd_min_y = src_fwd_min_y2;
		src_fwd_max_x = src_fwd_max_x2;
		src_fwd_max_y = src_fwd_max_y2;
	    }
#define	PAD	10
	    x = (src_fwd_min_x - PAD) / mane.shrinkfactor;
	    y = (src_fwd_min_y - PAD) / mane.shrinkfactor;

	    XDrawRectangle(DISP, mane.win, highGC,
	      x - mane_base_x, y - mane_base_y,
	      (src_fwd_max_x + PAD) / mane.shrinkfactor - x,
	      (src_fwd_max_y + PAD) / mane.shrinkfactor - y);
	}
}

void
source_forward_search(str)
	_Xconst char	*str;
{
	off_t		pos_save;
	struct drawinf	currinf_save;
	ubyte		maxchar_save;
	struct geom_info g_info;

	if (debug & DBG_CLIENT)
	    printf("Entering source_forward_search(%s)\n", str);

	src_file = str;
	while (*src_file == '0') ++src_file;
	if (*src_file < '1' || *src_file > '9') {
	    fprintf(stderr,
	      "Badly formatted source special; ignoring:  \"%s\"\n", str);
	    return;
	}
	src_line = atoi(src_file);
	while (*src_file >= '0' && *src_file <= '9') ++src_file;

	src_col = 0;
	if (*src_file == ':') {
	    ++src_file;
	    src_col = atoi(src_file);
	    while (*src_file >= '0' && *src_file <= '9') ++src_file;
	}

	if (*src_file == ' ') ++src_file;

	if (debug & DBG_CLIENT)
	    printf("File = \"%s\", line = %d, col = %d\n", src_file, src_line,
	      src_col);

	/* Save status of dvi_file reading (in case we hit an error and resume
	   drawing).  */

	if (dvi_pointer_frame != NULL)
	    pos_save = lseek(fileno(dvi_file), 0L, SEEK_CUR)
	      - (dvi_pointer_frame->end - dvi_pointer_frame->pos);
	(void) lseek(fileno(dvi_file), page_info[0].offset, SEEK_SET);

	currinf_save = currinf;
	maxchar_save = maxchar;

	bzero((char *) &currinf.data, sizeof currinf.data);
	currinf.tn_table_len = TNTABLELEN;
	currinf.tn_table = tn_table;
	currinf.tn_head = tn_head;
	currinf.pos = currinf.end = dvi_buffer;
	currinf.virtual = NULL;

	/* Start search over pages */

	found_src = False;
	best_distance = best_col_dist = 0xffffffff;
	src_this_line = 0;	/* initialize things that are kept as defaults*/
	src_this_file_strcmp = 1;
	if (setjmp(src_env) == 0) {

	    /* These two lines do the actual scanning (first pass).  */
	    for (src_page = 0; src_page < total_pages; ++src_page)
		spcl_scan(src_scan_special);

	    if (best_distance == 0xffffffff) {
		if (!found_src)
		    WARN(XmDIALOG_ERROR, "No source specials in the dvi file");
		else
		    WARN1(XmDIALOG_ERROR,
		      "No references to source file \"%s\" in dvi file",
		      src_file);

		/* Restore file position.  */
		maxchar = maxchar_save;
		currinf = currinf_save;

		if (dvi_pointer_frame != NULL) {
		    (void) lseek(fileno(dvi_file), pos_save, SEEK_SET);
		    dvi_pointer_frame->pos = dvi_pointer_frame->end
		      = dvi_buffer;
		}

		return;
	    }
	    if (debug & DBG_CLIENT)
		printf("Found close match:  line %d on page %d, offset %lu\n",
		  best_line, best_page + pageno_correct, best_offset);
	}
	else {
	    if (debug & DBG_CLIENT)
		printf("Found exact match on page %d, offset %lu\n",
		  best_page + pageno_correct, best_offset);
	}

	/*
	 * In this case we don't need to restore maxchar and currinf, since
	 * we won't resume drawing -- we'll jump to a new page instead.
	 */

	/* Move to that page.  */

	if (current_page != best_page)
	    goto_page(best_page, home);
	ev_flags |= EV_NEWPAGE;
	XMapRaised(DISP, XtWindow(top_level));

	/* Now search that particular page.  */

	g_info.geom_box = src_spec_fwd_box;
	g_info.geom_special = src_spec_fwd_special;
	g_info.geom_data = NULL;

	src_fwd_active = False;
	src_fwd_min_x = src_fwd_min_x2 = src_fwd_min_y = src_fwd_min_y2 =
	  0x7fffffff;
	src_fwd_max_x = src_fwd_max_x2 = src_fwd_max_y = src_fwd_max_y2 =
	  -0x7fffffff;
	source_fwd_box_page = -1;	/* in case of error, suppress box */

	(void) lseek(fileno(dvi_file), page_info[best_page].offset, SEEK_SET);
	currinf.tn_table_len = TNTABLELEN;
	currinf.tn_table = tn_table;
	currinf.tn_head = tn_head;
	currinf.pos = currinf.end = dvi_buffer;
	currinf.virtual = NULL;

	if (!setjmp(g_info.done_env))
	    geom_scan_part(&g_info, geom_current_frame = &geom_frame0, dimconv);

	if (!src_fwd_active) {
	    if (debug & DBG_CLIENT)
		fprintf(stderr,
		  "geom_scan_part() failed to re-find the special!\n");
	}
	else
	    source_fwd_box_page = current_page;
}
