/*========================================================================*\

Copyright (c) 1990-2013  Paul Vojta

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
PAUL VOJTA BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

NOTE:
	xdvi is based on prior work, as noted in the modification history
	in xdvi.c.

\*========================================================================*/

#include "xdvi.h"

#include <errno.h>
#include <ctype.h>	/* needed for memicmp() */
#include <pwd.h>
#include <sys/socket.h>
#include <sys/file.h>	/* this defines FASYNC */
#include <sys/ioctl.h>	/* this defines SIOCSPGRP and FIOASYNC */

#include <X11/Xmd.h>	/* get WORD64 and LONG64 */

#if PS
# include <sys/stat.h>
#endif

#ifdef VMS
#include <rmsdef.h>
#endif /* VMS */

#ifdef	X_NOT_STDC_ENV
extern	int	errno;
extern	void	*malloc();
extern	void	*realloc();
#endif

#if defined(macII) && !__STDC__		/* stdlib.h doesn't define these */
extern	void	*malloc();
extern	void	*realloc();
#endif

#if	NeedVarargsPrototypes		/* this is for oops */
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#ifdef	DOPRNT	/* define this if vfprintf gives you trouble */
#define	vfprintf(stream, message, args)	_doprnt(message, args, stream)
#endif

/* if POSIX O_NONBLOCK is not available, use O_NDELAY */
#if !defined O_NONBLOCK && defined O_NDELAY
# define O_NONBLOCK O_NDELAY
#endif

/* Linux prefers O_ASYNC over FASYNC; SGI IRIX does the opposite.  */
#if !defined(FASYNC) && defined(O_ASYNC)
# define FASYNC O_ASYNC
#endif

#if !defined(FLAKY_SIGPOLL) && !HAVE_STREAMS && !defined(FASYNC)
# if !defined(SIOCSPGRP) || !defined(FIOASYNC)
#  define FLAKY_SIGPOLL	1
# endif
#endif

#if !FLAKY_SIGPOLL && HAVE_STREAMS
# include <stropts.h>

# ifndef S_RDNORM
#  define S_RDNORM S_INPUT
# endif

# ifndef S_RDBAND
#  define S_RDBAND 0
# endif

# ifndef S_HANGUP
#  define S_HANGUP 0
# endif

# ifndef S_WRNORM
#  define S_WRNORM S_OUTPUT
# endif

#endif /* not FLAKY_SIGPOLL && HAVE_STREAMS */


/*
 *	General utility routines.
 */

static	void
exit_clean_windows()
{
	long	*window_list;
	size_t	window_list_len;
	long	*window_list_end;
	long	*wp;

	window_list_len = property_get_data(DefaultRootWindow(DISP),
	  ATOM_XDVI_WINDOWS, (unsigned char **) &window_list,
	  XGetWindowProperty);

	if (window_list_len == 0)
	    return;

	if (window_list_len % sizeof(long) != 0) {
	    XDeleteProperty(DISP, DefaultRootWindow(DISP), ATOM_XDVI_WINDOWS);
	    return;
	}

	window_list_len /= sizeof(long);

	/* Loop over list of windows.  */

	window_list_end = window_list + window_list_len;

	for (wp = window_list; wp < window_list_end; ++wp)
	    if (*wp == XtWindow(top_level)) {

		--window_list_len;
		--window_list_end;
		memcpy(wp, wp + 1, (window_list_end - wp) * sizeof(long));

		if (window_list_len == 0)
		    XDeleteProperty(DISP, DefaultRootWindow(DISP),
		      ATOM_XDVI_WINDOWS);
		else
		    XChangeProperty(DISP, DefaultRootWindow(DISP),
		      ATOM_XDVI_WINDOWS, ATOM_XDVI_WINDOWS, 32, PropModeReplace,
		      (unsigned char *) window_list, window_list_len);

		XFlush(DISP);
		return;
	    }

	if (debug & DBG_CLIENT)
	    printf("Cannot find window %08lx in list when exiting.\n",
	      XtWindow(top_level));

	return;
}


/*
 *	This routine should be used for all exits, except for really early ones.
 */

void
xdvi_exit(status)
	int	status;
{
	/* Clean up the "xdvi windows" property in the root window.  */
	if (top_level)
	    exit_clean_windows();
#if PS
	ps_destroy();
#endif
	exit(status);
}

/*
 *	Print error message and quit.
 */

#if	NeedVarargsPrototypes
void
oops(_Xconst char *message, ...)
#else
/* VARARGS */
void
oops(va_alist)
	va_dcl
#endif
{
#if	!NeedVarargsPrototypes
	_Xconst char *message;
#endif
	va_list	args;

	Fprintf(stderr, "%s: ", prog);
#if	NeedVarargsPrototypes
	va_start(args, message);
#else
	va_start(args);
	message = va_arg(args, _Xconst char *);
#endif
	(void) vfprintf(stderr, message, args);
	va_end(args);
	Putc('\n', stderr);
	xdvi_exit(1);
}


#ifndef KPATHSEA

/*
 *	Either (re)allocate storage or fail with explanation.
 */

void *
xmalloc(size)
	unsigned	size;
{
	void	*mem	= malloc(size);

	if (mem == NULL)
	    oops("! Out of memory (allocating %u bytes).\n", size);
	return mem;
}


void *
xrealloc(where, size)
	void		*where;
	unsigned	size;
{
	void	*mem	= realloc(where, size);

	if (mem == NULL)
	    oops("! Out of memory (reallocating %u bytes).\n", size);
	return mem;
}


/*
 *	Allocate a new string.
 */

char	*
xstrdup(str)
	_Xconst char	*str;
{
	size_t	len;
	char	*new;

	len = strlen(str) + 1;
	new = xmalloc(len);
	bcopy(str, new, len);
	return new;
}


/*
 *	Allocate a new string.  The second argument is the length.
 */

char	*
xmemdup(str, len)
	_Xconst char	*str;
	size_t		len;
{
	char	*new;

	new = xmalloc(len);
	bcopy(str, new, len);
	return new;
}

#endif /* not KPATHSEA */


#if TOOLKIT && !HAVE_STRERROR && !defined strerror

/*
 *	Replacement of system routine.
 */

extern	char	*sys_errlist[];
extern	int	sys_nerr;

char *
strerror(n)
	int	n;
{
	return (unsigned) n <= (unsigned) sys_nerr
	  ? sys_errlist[n]
	  : "Unknown error";
}

#endif


/*
 *	Expand the matrix *ffline to at least the given size.
 */

void
expandline(n)
	size_t	n;
{
	size_t	newlen	= n + 128;

	ffline = (ffline == NULL) ? xmalloc(newlen) : xrealloc(ffline, newlen);
	ffline_len = newlen;
}


/*
 *	Allocate bitmap for given font and character
 */

void
alloc_bitmap(bitmap)
	struct bitmap	*bitmap;
{
	unsigned int	size;

	/* width must be multiple of 16 bits for raster_op */
	bitmap->bytes_wide = ROUNDUP((int) bitmap->w, BMBITS) * BMBYTES;
	size = bitmap->bytes_wide * bitmap->h;
	bitmap->bits = xmalloc(size != 0 ? size : 1);
}


#ifndef KPATHSEA

/*
 *	Put a variable in the environment or abort on error.
 */

extern	char	**environ;

void
xputenv(var, value)
	_Xconst char	*var;
	_Xconst char	*value;
{

#if HAVE_PUTENV

	char	*buf;
	int	len1, len2;

	len1 = strlen(var);
	len2 = strlen(value) + 1;
	buf = xmalloc((unsigned int) len1 + len2 + 1);
	bcopy(var, buf, len1);
	buf[len1++] = '=';
	bcopy(value, buf + len1, len2);
	if (putenv(buf) != 0)
	    oops("! Failure in setting environment variable.");
	return;

#elif HAVE_SETENV

	if (setenv(var, value, True) != 0)
	    oops("! Failure in setting environment variable.");
	return;

#else /* not HAVE_{PUTENV,SETENV} */

	int		len1;
	int		len2;
	char		*buf;
	char		**linep;
	static	Boolean	did_malloc = False;

	len1 = strlen(var);
	len2 = strlen(value) + 1;
	buf = xmalloc((unsigned int) len1 + len2 + 1);
	bcopy(var, buf, len1);
	buf[len1++] = '=';
	bcopy(value, buf + len1, len2);
	for (linep = environ; *linep != NULL; ++linep)
	    if (memcmp(*linep, buf, len1) == 0) {
		*linep = buf;
		return;
	    }
	len1 = linep - environ;
	if (did_malloc)
	    environ = xrealloc(environ,
		(unsigned int) (len1 + 2) * sizeof(char *));
	else {
	    linep = xmalloc((unsigned int)(len1 + 2) * sizeof(char *));
	    bcopy((char *) environ, (char *) linep, len1 * sizeof(char *));
	    environ = linep;
	    did_malloc = True;
	}
	environ[len1++] = buf;
	environ[len1] = NULL;

#endif /* not HAVE_{PUTENV,SETENV} */

}

#endif /* not KPATHSEA */


/*
 *	Hopefully a self-explanatory name.  This code assumes the second
 *	argument is lower case.
 */

int
memicmp(s1, s2, n)
	_Xconst char	*s1;
	_Xconst char	*s2;
	size_t		n;
{
	while (n > 0) {
	    int i = tolower(*s1) - *s2;
	    if (i != 0) return i;
	    ++s1;
	    ++s2;
	    --n;
	}
	return 0;
}


/*
 *	Close the pixel file for the least recently used font.
 */

static	void
close_a_file()
{
	struct font *fontp;
	unsigned short oldest = ~0;
	struct font *f = NULL;

	if (debug & DBG_OPEN)
	    Puts("Calling close_a_file()");

	for (fontp = font_head; fontp != NULL; fontp = fontp->next)
	    if (fontp->file != NULL && fontp->timestamp <= oldest) {
		f = fontp;
		oldest = fontp->timestamp;
	    }
	if (f == NULL)
	    oops("Can't find an open pixel file to close");
	Fclose(f->file);
	f->file = NULL;
	++n_files_left;
}

/*
 *	This is necessary on some systems to work around a bug.
 */

#if SUNOS4
static	void
close_small_file()
{
	struct font *fontp;
	unsigned short oldest = ~0;
	struct font *f = NULL;

	if (debug & DBG_OPEN)
	    Puts("Calling close_small_file()");

	for (fontp = font_head; fontp != NULL; fontp = fontp->next)
	    if (fontp->file != NULL && fontp->timestamp <= oldest
	      && (unsigned char) fileno(fontp->file) < 128) {
		f = fontp;
		oldest = fontp->timestamp;
	    }
	if (f == NULL)
	    oops("Can't find an open pixel file to close");
	Fclose(f->file);
	f->file = NULL;
	++n_files_left;
}
#else
#define	close_small_file	close_a_file
#endif

/*
 *	Open a file in the given mode.
 */

FILE *
#ifndef	VMS
xfopen(filename, type)
	_Xconst char	*filename;
	_Xconst char	*type;
#define	TYPE	type
#else
xfopen(filename, type, type2)
	_Xconst char	*filename;
	_Xconst char	*type;
	_Xconst char	*type2;
#define	TYPE	type, type2
#endif	/* VMS */
{
	FILE	*f;

	if (n_files_left == 0) close_a_file();
	f = fopen(filename, TYPE);
#ifndef	VMS
	if (f == NULL && (errno == EMFILE || errno == ENFILE))
#else	/* VMS */
	if (f == NULL && errno == EVMSERR && vaxc$errno == RMS$_ACC)
#endif	/* VMS */
	{
	    n_files_left = 0;
	    close_a_file();
	    f = fopen(filename, TYPE);
	}
	if (f != NULL) {
	    --n_files_left;
#ifdef F_SETFD
	    (void) fcntl(fileno(f), F_SETFD, 1);
#endif
	}
	return f;
}
#undef	TYPE


#if XAW

/*
 *	Plain ordinary open() system call.  Don't adjust n_files_left.
 */

int
xopen(path, flags)
	_Xconst char	*path;
	int		flags;
{
	int	fd;

	if (n_files_left == 0)
	    close_a_file();

	fd = open(path, flags);
	if (fd == -1 && (errno == EMFILE || errno == ENFILE)) {
	    n_files_left = 0;
	    close_a_file();
	    fd = open(path, flags);
	}

	return fd;
}

#endif /* XAW */


/*
 *	Create a pipe, closing a file if necessary.
 *	We use socketpair() instead of pipe() because several operating
 *	systems don't support SIGPOLL/SIGIO on pipes:
 *		SGI IRIX 6.5	F_SETOWN not implemented
 *		Linux 2.4.2	Not supported
 */

int
xpipe(fd)
	int	*fd;
{
	int	retval;

	while (n_files_left < 2) close_a_file();
	for (;;) {
	    retval = socketpair(AF_UNIX, SOCK_STREAM, 0, fd);
	    if (retval == 0) {
		n_files_left -= 2;
		break;
	    }
	    if ((errno != EMFILE && errno != ENFILE))
		break;
	    n_files_left = 0;
	    close_a_file();
	}
	return retval;
}


/*
 *	Open a directory for reading, opening a file if necessary.
 */

DIR *
xopendir(name)
	_Xconst char	*name;
{
	DIR	*retval;

	if (n_files_left == 0) close_a_file();
	for (;;) {
	    retval = opendir(name);
	    if (retval == NULL || (errno != EMFILE && errno != ENFILE)) break;
	    n_files_left = 0;
	    close_a_file();
	}
	if (retval != NULL) --n_files_left;
	return retval;
}


/*
 *	Perform tilde expansion, updating the character pointer unless the
 *	user was not found.
 */

_Xconst	struct passwd *
ff_getpw(pp, p_end)
	_Xconst	char	**pp;
	_Xconst	char	*p_end;
{
	_Xconst	char		*p	= *pp;
	_Xconst	char		*p1;
	unsigned		len;
	_Xconst	struct passwd	*pw;
	int			count;

	++p;	/* skip the tilde */
	p1 = p;
	while (p1 < p_end && *p1 != '/') ++p1;
	len = p1 - p;

	if (len != 0) {
	    if (len >= ffline_len)
		expandline(len);
	    bcopy(p, ffline, len);
	    ffline[len] = '\0';
	}

	for (count = 0;; ++count) {
	    if (len == 0)	/* if no user name */
		pw = getpwuid(getuid());
	    else
		pw = getpwnam(ffline);

	    if (pw != NULL) {
		*pp = p1;
		return pw;
	    }

	    /* On some systems, getpw{uid,nam} return without setting errno,
	     * even if the call failed because of too many open files.
	     * Therefore, we play it safe here.
	     */
	    if (count >= 2 && len != 0 && getpwuid(getuid()) != NULL)
		return NULL;

	    close_small_file();
	}
}


/*
 *
 *      Read size bytes from the FILE fp, constructing them into a
 *      signed/unsigned integer.
 *
 */

unsigned long
num(fp, size)
	FILE	*fp;
	int	size;
{
	long	x	= 0;

	while (size--) x = (x << 8) | one(fp);
	return x;
}

long
snum(fp, size)
	FILE	*fp;
	int	size;
{
	long	x;

#if	__STDC__
	x = (signed char) getc(fp);
#else
	x = (unsigned char) getc(fp);
	if (x & 0x80) x -= 0x100;
#endif
	while (--size) x = (x << 8) | one(fp);
	return x;
}


#if FREETYPE || PS

/*
 *	General AVL tree mechanism.  Search for a node, and return it if found.
 *	Otherwise insert a node.
 *	This uses the AVL algorithm from Knuth Vol. 3.
 */

struct avl *
avladd(key, key_len, headp, size)
	_Xconst char	*key;
	size_t		key_len;
	struct avl	**headp;
	size_t		size;
{
	struct avl	*ap;
	struct avl	**app;
	struct avl	*sp;	/* place where rebalancing may be necessary */
	struct avl	**spp;	/* points to sp */
	int		i;

	/* Search */
	spp = app = headp;
	for (;;) {
	    ap = *app;
	    if (ap == NULL)	/* bottom of tree */
		break;
	    if (ap->bal != 0)
		spp = app;
	    i = key_len - ap->key_len;
	    if (i == 0)
		i = memcmp(key, ap->key, key_len);
	    if (i == 0)		/* found record already */
		return ap;
	    if (i < 0)		/* move left */
		app = &ap->left;
	    else
		app = &ap->right;
	}

	/* Insert */
	ap = xmalloc(size);
	ap->key = key;
	ap->key_len = key_len;
	ap->bal = 0;
	ap->left = ap->right = NULL;
	*app = ap;

	/* Adjust balance factors */
	sp = *spp;
	if (sp == ap)
	    return ap;
	i = key_len - sp->key_len;
	if (i == 0)
	    i = memcmp(key, sp->key, key_len);
	sp = (i < 0 ? sp->left : sp->right);
	while (sp != ap) {
	    i = key_len - sp->key_len;
	    if (i == 0)
		i = memcmp(key, sp->key, key_len);
	    if (i < 0) {
		sp->bal = -1;
		sp = sp->left;
	    }
	    else {
		sp->bal = 1;
		sp = sp->right;
	    }
	}

	/* Balancing act */
	sp = *spp;
	i = key_len - sp->key_len;
	if (i == 0)
	    i = memcmp(key, sp->key, key_len);
	if (i < 0) {
	    if (sp->bal >= 0)
		--sp->bal;
	    else {	/* need to rebalance */
		struct avl *left;

		left = sp->left;
		if (left->bal < 0) {	/* single rotation */
		    sp->left = left->right;
		    left->right = sp;
		    sp->bal = left->bal = 0;
		    *spp = left;
		}
		else {			/* double rotation */
		    struct avl	*newtop;

		    newtop = left->right;
		    sp->left = newtop->right;
		    newtop->right = sp;
		    left->right = newtop->left;
		    newtop->left = left;
		    sp->bal = left->bal = 0;
		    if (newtop->bal < 0) ++sp->bal;
		    else if (newtop->bal > 0) --left->bal;
		    newtop->bal = 0;
		    *spp = newtop;
		}
	    }
	}
	else {
	    if (sp->bal <= 0)
		++sp->bal;
	    else {	/* need to rebalance */
		struct avl *right;

		right = sp->right;
		if (right->bal > 0) {	/* single rotation */
		    sp->right = right->left;
		    right->left = sp;
		    sp->bal = right->bal = 0;
		    *spp = right;
		}
		else {			/* double rotation */
		    struct avl	*newtop;

		    newtop = right->left;
		    sp->right = newtop->left;
		    newtop->left = sp;
		    right->left = newtop->right;
		    newtop->right = right;
		    sp->bal = right->bal = 0;
		    if (newtop->bal > 0) --sp->bal;
		    else if (newtop->bal < 0) ++right->bal;
		    newtop->bal = 0;
		    *spp = newtop;
		}
	    }
	}

	return ap;
}

#endif /* FREETYPE || PS */


/*
 *	On 64-bit platforms, XGetWindowProperty and related functions convert
 *	properties with format=32 to arrays of longs.  This function keeps that
 *	convention.
 *	The return value is the total number of bytes in the buffer.
 */

#if defined(WORD64) || defined(LONG64)
# define LONG_CONV_64(bytes, format)	((bytes) << ((format) >> 5))
#else
# define LONG_CONV_64(bytes, format)	(bytes)
#endif

size_t
property_get_data(w, a, ret_buf, x_get_property)
	Window		w;
	Atom		a;
	unsigned char	**ret_buf;
	int		(*x_get_property) ARGS((Display *, Window, Atom, long,
			  long, Bool, Atom, Atom *, int *, unsigned long *,
			  unsigned long *, unsigned char **));
{
		/* all of these are in 8-bit units */
	unsigned long	byte_offset	= 0;
	Atom		type_ret;
	int		format_ret;
	unsigned long	nitems_ret;
	unsigned long	bytes_after_ret	= 0;
	unsigned char	*prop_ret	= NULL;

	/*
	 * buffer for collecting returned data; this is static to
	 * avoid expensive malloc()s at every call (which is often!)
	 */
	static unsigned char	*buffer		= NULL;
	static size_t		buffer_len	= 0;

	while (x_get_property(DISP, w,
	  a, byte_offset / 4, (bytes_after_ret + 3) / 4, False, a,
	  &type_ret, &format_ret, &nitems_ret, &bytes_after_ret, &prop_ret)
	  == Success) {

	    if (type_ret != a || format_ret == 0) break;

	    nitems_ret *= (format_ret / 8);	/* convert to bytes */

	    if (LONG_CONV_64(byte_offset + nitems_ret, format_ret)
	      >= buffer_len) {
		buffer_len += 256
		  * ((LONG_CONV_64(byte_offset + nitems_ret, format_ret)
		  - buffer_len) / 256 + 1);
		buffer = (buffer == NULL ? xmalloc(buffer_len)
		  : xrealloc(buffer, buffer_len));
	    }

	    /* the +1 captures the extra '\0' that Xlib puts after the end.  */
	    memcpy(buffer + LONG_CONV_64(byte_offset, format_ret), prop_ret,
	      LONG_CONV_64(nitems_ret, format_ret) + 1);
	    byte_offset += nitems_ret;

	    XFree(prop_ret);
	    prop_ret = NULL;

	    if (bytes_after_ret == 0)	/* got all data */
		break;
	}

	if (prop_ret != NULL)
	    XFree(prop_ret);

	*ret_buf = buffer;
	return LONG_CONV_64(byte_offset, format_ret);
}


#if PS

/*
 *	Create a temporary file and return its fd.  Also put its filename
 *	in str.  Create str if it's NULL.
 */

#ifndef P_tmpdir
#define	P_tmpdir	"/tmp"
#endif

static	_Xconst char	tmp_suffix[]	= "/xdvi-XXXXXX";

int
xdvi_temp_fd(str)
	char	**str;
{
	int			fd;
	char			*p;
	size_t			len;
	static	_Xconst char	*template	= NULL;
#if !HAVE_MKSTEMP
	static	unsigned long	seed;
	static	char		letters[]	=
	  "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789._";
	char			*p1;
#endif

	if (*str != NULL) {
	    p = *str;
	    if (n_files_left == 0) close_a_file();
	    /* O_EXCL is there for security (if root runs xdvi) */
	    fd = open(p, O_RDWR | O_CREAT | O_EXCL, S_IRUSR | S_IWUSR);
	    if (fd == -1 && (errno == EMFILE || errno == ENFILE)) {
		n_files_left = 0;
		close_a_file();
		fd = open(p, O_RDWR | O_CREAT | O_EXCL, S_IRUSR | S_IWUSR);
	    }
	    if (!(fd == -1 && errno == EEXIST))
		return fd;
#if HAVE_MKSTEMP
	    memcpy(p + strlen(p) - 6, "XXXXXX", 6);
#endif
	}
	else {
	    if (template == NULL) {
		_Xconst char	*ourdir;

		ourdir = getenv("TMPDIR");
		if (ourdir == NULL || access(ourdir, W_OK) < 0) {
		    ourdir = P_tmpdir;
		    if (access(ourdir, W_OK) < 0)
			ourdir = ".";
		}
		len = strlen(ourdir);
		if (len > 0 && ourdir[len - 1] == '/') --len;
		template = p = xmalloc(len + sizeof tmp_suffix);
		memcpy(p, ourdir, len);
		memcpy(p + len, tmp_suffix, sizeof tmp_suffix);
#if !HAVE_MKSTEMP
		seed = 123456789 * time(NULL) + 987654321 * getpid();
#endif
	    }
	    *str = p = xstrdup(template);
	}

	if (n_files_left == 0) close_a_file();

#if HAVE_MKSTEMP
	fd = mkstemp(p);
	if (fd == -1 && (errno == EMFILE || errno == ENFILE)) {
	    n_files_left = 0;
	    close_a_file();
	    memcpy(p + strlen(p) - 6, "XXXXXX", 6);
	    fd = mkstemp(p);
	}
#else
	p1 = p + strlen(p) - 6;
	for (;;) {
	    unsigned long s = ++seed;
	    char *p2;

	    for (p2 = p1 + 5; p2 >= p1; --p2) {
		*p2 = letters[s & 63];
		s >>= 6;
	    }
	    fd = open(p, O_RDWR | O_CREAT | O_EXCL, S_IRUSR | S_IWUSR);
	    if (fd == -1 && (errno == EMFILE || errno == ENFILE)) {
		n_files_left = 0;
		close_a_file();
		fd = open(p, O_RDWR | O_CREAT | O_EXCL, S_IRUSR | S_IWUSR);
	    }
	    if (!(fd == -1 && errno == EEXIST))
		break;
	}
#endif

	return fd;
}

#endif /* PS */


/*
 *	Prepare the file descriptor to generate SIGPOLL/SIGIO events.
 *	If called with a True argument, set it up for non-blocking I/O.
 */

void
prep_fd(fd, noblock)
	int		fd;
	wide_bool	noblock;
{
	/* Set file descriptor for non-blocking I/O */
	if (noblock)
	    (void) fcntl(fd, F_SETFL, fcntl(fd, F_GETFL, 0) | O_NONBLOCK);

#if !FLAKY_SIGPOLL
# if HAVE_STREAMS
	if (isastream(fd) > 0) {
	    if (ioctl(fd, I_SETSIG,
	      S_RDNORM | S_RDBAND | S_HANGUP | S_WRNORM) == -1)
		perror("xdvi: ioctl I_SETSIG");
	}
	else
# endif
	{
# ifdef FASYNC
	    if (fcntl(fd, F_SETOWN, getpid()) == -1)
		perror("xdvi: fcntl F_SETOWN");
	    if (fcntl(fd, F_SETFL, fcntl(fd, F_GETFL, 0) | FASYNC) == -1)
		perror("xdvi: fcntl F_SETFL");
# elif defined SIOCSPGRP && defined FIOASYNC
	    /* For HP-UX B.10.10 and maybe others.  See "man 7 socket".  */
	    int arg;

	    arg = getpid();
	    if (ioctl(fd, SIOCSPGRP, &arg) == -1)
		perror("xdvi: ioctl SIOCSPGRP");
	    arg = 1;
	    if (ioctl(fd, FIOASYNC, &arg) == -1)
		perror("xdvi: ioctl FIOASYNC");
# endif
	}
#endif /* not FLAKY_SIGPOLL */
}
