/*========================================================================*\

Copyright (c) 1990-2002  Paul Vojta

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
PAUL VOJTA BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

NOTE:
	This module is based on prior work as noted below.

\*========================================================================*/

/*
 * Support drawing routines for TeXsun and TeX
 *
 *      Copyright, (C) 1987, 1988 Tim Morgan, UC Irvine
 *	Adapted for xdvi by Jeffrey Lee, U. of Toronto
 *
 * At the time these routines are called, the values of hh and vv should
 * have been updated to the upper left corner of the graph (the position
 * the \special appears at in the dvi file).  Then the coordinates in the
 * graphics commands are in terms of a virtual page with axes oriented the
 * same as the Imagen and the SUN normally have:
 *
 *                      0,0
 *                       +-----------> +x
 *                       |
 *                       |
 *                       |
 *                      \ /
 *                       +y
 *
 * Angles are measured in the conventional way, from +x towards +y.
 * Unfortunately, that reverses the meaning of "counterclockwise"
 * from what it's normally thought of.
 *
 * A lot of floating point arithmetic has been converted to integer
 * arithmetic for speed.  In some places, this is kind-of kludgy, but
 * it's worth it.
 */

#include "xdvi.h"

#include <math.h>
#include <ctype.h>

extern	char	*strtok ARGS((char *, _Xconst char *));
extern	double	floor ARGS((double));
#define	rint(x)	floor((x) + 0.5)

#ifdef	X_NOT_STDC_ENV
#ifndef	atof
extern	double	atof ARGS((_Xconst char *));
#endif

extern	char	*getenv ARGS((_Xconst char *));
#endif	/* X_NOT_STDC_ENV */

#if PS
# if HAVE_VFORK_H
#  include <vfork.h>
# endif

# include <errno.h>

# if HAVE_SYS_WAIT_H
#  include <sys/wait.h>
# endif
# ifndef WEXITSTATUS
#  define WEXITSTATUS(stat_val) ((unsigned)(stat_val) >> 8)
# endif
# ifndef WIFEXITED
#  define WIFEXITED(stat_val) (((stat_val) & 255) == 0)
# endif

# ifdef	X_NOT_STDC_ENV
extern	int	errno;
# endif
#endif	/* PS */

#define	MAXPOINTS	300	/* Max points in a path */
#define	MAX_PEN_SIZE	7	/* Max pixels of pen width */
#define	TWOPI		(3.14159265359 * 2.0)


static	int	xx[MAXPOINTS], yy[MAXPOINTS];	/* Path in milli-inches */
static	int	path_len = 0;	/* # points in current path */
static	int	pen_size = 1;	/* Pixel width of lines drawn */

static	Boolean	whiten = False;
static	Boolean	shade = False;
static	Boolean	blacken = False;

/* Unfortunately, these values also appear in dvisun.c */
#define	xRESOLUTION	(pixels_per_inch/shrink_factor)
#define	yRESOLUTION	(pixels_per_inch/shrink_factor)


/*
 *	Issue warning messages
 */

static	void
Warning(fmt, msg)
	char	*fmt, *msg;
{
	Fprintf(stderr, "%s: ", prog);
	Fprintf(stderr, fmt, msg);
	(void) fputc('\n', stderr);
}


/*
 *	X drawing routines
 */

#define	toint(x)	((int) ((x) + 0.5))
#define	xconv(x)	(toint(tpic_conv*(x))/shrink_factor + PXL_H)
#define	yconv(y)	(toint(tpic_conv*(y))/shrink_factor + PXL_V)

/*
 *	Draw a line from (fx,fy) to (tx,ty).
 *	Right now, we ignore pen_size.
 */
static	void
line_btw(fx, fy, tx, ty)
int fx, fy, tx, ty;
{
	int	fcx = xconv(fx);
	int	tcx = xconv(tx);
	int	fcy = yconv(fy);
	int	tcy = yconv(ty);

	if ((fcx < max_x || tcx < max_x) && (fcx >= min_x || tcx >= min_x) &&
	  (fcy < max_y || tcy < max_y) && (fcy >= min_y || tcy >= min_y)) {
#if COLOR
	    if (fg_active != fg_current) do_color_change();
#endif
	    XDrawLine(DISP, currwin.win, ruleGC,
	      fcx - currwin.base_x, fcy - currwin.base_y,
	      tcx - currwin.base_x, tcy - currwin.base_y);
	}
}

/*
 *	Draw a dot at (x,y)
 */
static	void
dot_at(x, y)
	int	x, y;
{
	int	cx = xconv(x);
	int	cy = yconv(y);

	if (cx < max_x && cx >= min_x && cy < max_y && cy >= min_y) {
#if COLOR
	    if (fg_active != fg_current) do_color_change();
#endif
	    XDrawPoint(DISP, currwin.win, ruleGC,
	      cx - currwin.base_x, cy - currwin.base_y);
	}
}

/*
 *	Apply the requested attributes to the last path (box) drawn.
 *	Attributes are reset.
 *	(Not currently implemented.)
 */
	/* ARGSUSED */
static	void
do_attribute_path(last_min_x, last_max_x, last_min_y, last_max_y)
int last_min_x, last_max_x, last_min_y, last_max_y;
{
}

/*
 *	Set the size of the virtual pen used to draw in milli-inches
 */

/* ARGSUSED */
static	void
set_pen_size(cp)
	char	*cp;
{
	int	ps;

	if (sscanf(cp, " %d ", &ps) != 1) {
	    Warning("invalid .ps command format: %s", cp);
	    return;
	}
	pen_size = (ps * (xRESOLUTION + yRESOLUTION) + 1000) / 2000;
	if (pen_size < 1) pen_size = 1;
	else if (pen_size > MAX_PEN_SIZE) pen_size = MAX_PEN_SIZE;
}


/*
 *	Print the line defined by previous path commands
 */

static	void
flush_path()
{
	int	i;
	int	last_min_x, last_max_x, last_min_y, last_max_y;

	last_min_x = 30000;
	last_min_y = 30000;
	last_max_x = -30000;
	last_max_y = -30000;
	for (i = 1; i < path_len; i++) {
	    if (xx[i] > last_max_x) last_max_x = xx[i];
	    if (xx[i] < last_min_x) last_min_x = xx[i];
	    if (yy[i] > last_max_y) last_max_y = yy[i];
	    if (yy[i] < last_min_y) last_min_y = yy[i];
	    line_btw(xx[i], yy[i], xx[i+1], yy[i+1]);
	}
	if (xx[path_len] > last_max_x) last_max_x = xx[path_len];
	if (xx[path_len] < last_min_x) last_min_x = xx[path_len];
	if (yy[path_len] > last_max_y) last_max_y = yy[path_len];
	if (yy[path_len] < last_min_y) last_min_y = yy[path_len];
	path_len = 0;
	do_attribute_path(last_min_x, last_max_x, last_min_y, last_max_y);
}


/*
 *	Print a dashed line along the previously defined path, with
 *	the dashes/inch defined.
 */

static	void
flush_dashed(cp, dotted)
	char	*cp;
	Boolean	dotted;
{
	int	i;
	int	numdots;
	int	lx0, ly0, lx1, ly1;
	int	cx0, cy0, cx1, cy1;
	float	inchesperdash;
	double	d, spacesize, a, b, dx, dy, milliperdash;

	if (sscanf(cp, " %f ", &inchesperdash) != 1) {
	    Warning("invalid format for dotted/dashed line: %s", cp);
	    return;
	}
	if (path_len <= 1 || inchesperdash <= 0.0) {
	    Warning("invalid conditions for dotted/dashed line", "");
	    return;
	}
	milliperdash = inchesperdash * 1000.0;
	lx0 = xx[1];	ly0 = yy[1];
	lx1 = xx[2];	ly1 = yy[2];
	dx = lx1 - lx0;
	dy = ly1 - ly0;
	if (dotted) {
	    numdots = sqrt(dx*dx + dy*dy) / milliperdash + 0.5;
	    if (numdots == 0) numdots = 1;
	    for (i = 0; i <= numdots; i++) {
		a = (float) i / (float) numdots;
		cx0 = lx0 + a * dx + 0.5;
		cy0 = ly0 + a * dy + 0.5;
		dot_at(cx0, cy0);
	    }
	}
	else {
	    d = sqrt(dx*dx + dy*dy);
	    numdots = d / (2.0 * milliperdash) + 1.0;
	    if (numdots <= 1)
		line_btw(lx0, ly0, lx1, ly1);
	    else {
		spacesize = (d - numdots * milliperdash) / (numdots - 1);
		for (i = 0; i < numdots - 1; i++) {
		    a = i * (milliperdash + spacesize) / d;
		    b = a + milliperdash / d;
		    cx0 = lx0 + a * dx + 0.5;
		    cy0 = ly0 + a * dy + 0.5;
		    cx1 = lx0 + b * dx + 0.5;
		    cy1 = ly0 + b * dy + 0.5;
		    line_btw(cx0, cy0, cx1, cy1);
		    b += spacesize / d;
		}
		cx0 = lx0 + b * dx + 0.5;
		cy0 = ly0 + b * dy + 0.5;
		line_btw(cx0, cy0, lx1, ly1);
	    }
	}
	path_len = 0;
}


/*
 *	Add a point to the current path
 */

static	void
add_path(cp)
	char	*cp;
{
	int	pathx, pathy;

	if (++path_len >= MAXPOINTS) oops("Too many points");
	if (sscanf(cp, " %d %d ", &pathx, &pathy) != 2)
	    oops("Malformed path command");
	xx[path_len] = pathx;
	yy[path_len] = pathy;
}


/*
 *	Draw to a floating point position
 */

static void
im_fdraw(x, y)
	double	x,y;
{
	if (++path_len >= MAXPOINTS) oops("Too many arc points");
	xx[path_len] = x + 0.5;
	yy[path_len] = y + 0.5;
}


/*
 *	Draw an ellipse with the indicated center and radices.
 */

static	void
draw_ellipse(xc, yc, xr, yr)
	int	xc, yc, xr, yr;
{
	double	angle, theta;
	int	n;
	int	px0, py0, px1, py1;

	angle = (xr + yr) / 2.0;
	theta = sqrt(1.0 / angle);
	n = TWOPI / theta + 0.5;
	if (n < 12) n = 12;
	else if (n > 80) n = 80;
	n /= 2;
	theta = TWOPI / n;

	angle = 0.0;
	px0 = xc + xr;		/* cos(0) = 1 */
	py0 = yc;		/* sin(0) = 0 */
	while ((angle += theta) <= TWOPI) {
	    px1 = xc + xr*cos(angle) + 0.5;
	    py1 = yc + yr*sin(angle) + 0.5;
	    line_btw(px0, py0, px1, py1);
	    px0 = px1;
	    py0 = py1;
	}
	line_btw(px0, py0, xc + xr, yc);
}

/*
 *	Draw an arc
 */

static	void
arc(cp, invis)
	char	*cp;
	Boolean	invis;
{
	int	xc, yc, xrad, yrad, n;
	float	start_angle, end_angle, angle, theta, r;
	double	xradius, yradius, xcenter, ycenter;

	if (sscanf(cp, " %d %d %d %d %f %f ", &xc, &yc, &xrad, &yrad,
		&start_angle, &end_angle) != 6) {
	    Warning("invalid arc specification: %s", cp);
	    return;
	}

	if (invis) return;

	/* We have a specialized fast way to draw closed circles/ellipses */
	if (start_angle <= 0.0 && end_angle >= 6.282) {
	    draw_ellipse(xc, yc, xrad, yrad);
	    return;
	}
	xcenter = xc;
	ycenter = yc;
	xradius = xrad;
	yradius = yrad;
	r = (xradius + yradius) / 2.0;
	theta = sqrt(1.0 / r);
	n = 0.3 * TWOPI / theta + 0.5;
	if (n < 12) n = 12;
	else if (n > 80) n = 80;
	n /= 2;
	theta = TWOPI / n;
	flush_path();
	im_fdraw(xcenter + xradius * cos(start_angle),
	    ycenter + yradius * sin(start_angle));
	angle = start_angle + theta;
	if (end_angle < start_angle) end_angle += TWOPI;
	while (angle < end_angle) {
	    im_fdraw(xcenter + xradius * cos(angle),
		ycenter + yradius * sin(angle));
	    angle += theta;
	}
	im_fdraw(xcenter + xradius * cos(end_angle),
	    ycenter + yradius * sin(end_angle));
	flush_path();
}


/*
 *	APPROXIMATE integer distance between two points
 */

#define	dist(x0, y0, x1, y1)	(abs(x0 - x1) + abs(y0 - y1))


/*
 *	Draw a spline along the previously defined path
 */

static	void
flush_spline()
{
	int	xp, yp;
	int	N;
	int	lastx, lasty;
	Boolean	lastvalid = False;
	int	t1, t2, t3;
	int	steps;
	int	j;
	int	i, w;

#ifdef	lint
	lastx = lasty = -1;
#endif
	N = path_len + 1;
	xx[0] = xx[1];
	yy[0] = yy[1];
	xx[N] = xx[N-1];
	yy[N] = yy[N-1];
	for (i = 0; i < N - 1; i++) {	/* interval */
	    steps = (dist(xx[i], yy[i], xx[i+1], yy[i+1]) +
		dist(xx[i+1], yy[i+1], xx[i+2], yy[i+2])) / 80;
	    for (j = 0; j < steps; j++) {	/* points within */
		w = (j * 1000 + 500) / steps;
		t1 = w * w / 20;
		w -= 500;
		t2 = (750000 - w * w) / 10;
		w -= 500;
		t3 = w * w / 20;
		xp = (t1*xx[i+2] + t2*xx[i+1] + t3*xx[i] + 50000) / 100000;
		yp = (t1*yy[i+2] + t2*yy[i+1] + t3*yy[i] + 50000) / 100000;
		if (lastvalid) line_btw(lastx, lasty, xp, yp);
		lastx = xp;
		lasty = yp;
		lastvalid = True;
	    }
	}
	path_len = 0;
}


/*
 *	Shade the last box, circle, or ellipse
 */

static	void
shade_last()
{
	blacken = whiten = False;
	shade = True;
}


/*
 *	Make the last box, circle, or ellipse, white inside (shade with white)
 */

static	void
whiten_last()
{
	whiten = True;
	blacken = shade = False;
}


/*
 *	Make last box, etc, black inside
 */

static	void
blacken_last()
{
	blacken = True;
	whiten = shade = False;
}


/*
 *	Code for PostScript<tm> specials begins here.
 */

#if	PS

/*
 *	Information on how to search for PS header and figure files.
 */

#include "filf-app.h"		/* application-related defs, etc. */
#include "filefind.h"

static	_Xconst	char	no_f_str_ps[]	= "/%f";

static	struct findrec			search_header	= {
	/* path1	*/	NULL,
#if	CFGFILE
	/* envptr	*/	NULL,
#endif
	/* path2	*/	DEFAULT_HEADER_PATH,
	/* type		*/	"PS header",
	/* fF_etc	*/	"fF",
	/* x_var_char	*/	'f',
	/* n_var_opts	*/	2,
	/* no_f_str	*/	no_f_str_ps,
	/* no_f_str_end	*/	no_f_str_ps + sizeof(no_f_str_ps) - 1,
	/* abs_str	*/	"%f",
#if USE_GF
	/* no_f_str_flags */	F_FILE_USED,
	/* abs_str_flags */	F_FILE_USED,
	/* pk_opt_char	*/	'f',
	/* pk_gf_addr	*/	NULL,
#endif
	/* pct_s_str	*/	"%qdvips//",
	{
	  /* v.stephead		*/	NULL,
	  /* v.pct_s_head	*/	NULL,
	  /* v.pct_s_count	*/	0,
	  /* v.pct_s_atom	*/	NULL,
	  /* v.rootp		*/	NULL,
	}
};

static	struct findrec			search_fig	= {
	/* path1	*/	NULL,
#if	CFGFILE
	/* envptr	*/	NULL,
#endif
	/* path2	*/	DEFAULT_FIG_PATH,
	/* type		*/	"PS",
	/* fF_etc	*/	"fF",
	/* x_var_char	*/	'f',
	/* n_var_opts	*/	2,
	/* no_f_str	*/	no_f_str_ps,
	/* no_f_str_end	*/	no_f_str_ps + sizeof(no_f_str_ps) - 1,
	/* abs_str	*/	"%f",
#if USE_GF
	/* no_f_str_flags */	F_FILE_USED,
	/* abs_str_flags */	F_FILE_USED,
	/* pk_opt_char	*/	'f',
	/* pk_gf_addr	*/	NULL,
#endif
	/* pct_s_str	*/	"%qdvips//:%qtex//",
	{
	  /* v.stephead		*/	NULL,
	  /* v.pct_s_head	*/	NULL,
	  /* v.pct_s_count	*/	0,
	  /* v.pct_s_atom	*/	NULL,
	  /* v.rootp		*/	NULL,
	}
};

static	void	ps_startup ARGS((int, int, _Xconst char *));
static	void	ps_startup2 ARGS((void));
void	NullProc ARGS((void)) {}
static	void	NullProc2 ARGS((_Xconst char *));

struct psprocs	psp = {		/* used for lazy startup of the ps machinery */
	/* toggle */		NullProc,
	/* destroy */		NullProc,
	/* interrupt */		NullProc,
	/* endpage */		NullProc,
	/* drawbegin */		ps_startup,
	/* drawraw */		NullProc2,
	/* drawfile */		NULL,
	/* drawend */		NullProc2,
	/* beginheader */	ps_startup2,
	/* endheader */		NullProc,
	/* newdoc */		NullProc};

struct psprocs	no_ps_procs = {		/* used if postscript is unavailable */
	/* toggle */		NullProc,
	/* destroy */		NullProc,
	/* interrupt */		NullProc,
	/* endpage */		NullProc,
	/* drawbegin */		drawbegin_none,
	/* drawraw */		NullProc2,
	/* drawfile */		NULL,
	/* drawend */		NullProc2,
	/* beginheader */	NullProc,
	/* endheader */		NullProc,
	/* newdoc */		NullProc};

#endif	/* PS */

static	Boolean		bbox_valid;
static	unsigned int	bbox_width;
static	unsigned int	bbox_height;
static	int		bbox_angle;
static	int		bbox_voffset;

void
draw_bbox()
{
	int	xcorner, ycorner;

	if (bbox_valid) {

#if COLOR
	    if (fg_active != fg_current) do_color_change();
#endif

	    xcorner = PXL_H - currwin.base_x;
	    ycorner = PXL_V - currwin.base_y;

	    if (bbox_angle == 0) {
		ycorner -= bbox_voffset;
		XDrawLine(DISP, currwin.win, ruleGC,
		  xcorner, ycorner,
		  xcorner + bbox_width, ycorner);
		XDrawLine(DISP, currwin.win, ruleGC,
		  xcorner + bbox_width, ycorner,
		  xcorner + bbox_width, ycorner + bbox_height);
		XDrawLine(DISP, currwin.win, ruleGC,
		  xcorner + bbox_width, ycorner + bbox_height,
		  xcorner, ycorner + bbox_height);
		XDrawLine(DISP, currwin.win, ruleGC,
		  xcorner, ycorner + bbox_height,
		  xcorner, ycorner);
	    }
	    else {
		float	sin_a	= sin(bbox_angle * (TWOPI / 360));
		float	cos_a	= cos(bbox_angle * (TWOPI / 360));
		float	a, b, c, d;

		a = cos_a * bbox_width;
		b = -sin_a * bbox_width;
		c = -sin_a * bbox_height;
		d = -cos_a * bbox_height;

		XDrawLine(DISP, currwin.win, ruleGC,
		  xcorner, ycorner,
		  xcorner + (int) rint(a), ycorner + (int) rint(b));
		XDrawLine(DISP, currwin.win, ruleGC,
		  xcorner + (int) rint(a), ycorner + (int) rint(b),
		  xcorner + (int) rint(a + c), ycorner + (int) rint(b + d));
		XDrawLine(DISP, currwin.win, ruleGC,
		  xcorner + (int) rint(a + c), ycorner + (int) rint(b + d),
		  xcorner + (int) rint(c), ycorner + (int) rint(d));
		XDrawLine(DISP, currwin.win, ruleGC,
		  xcorner + (int) rint(c), ycorner + (int) rint(d),
		  xcorner, ycorner);
	    }
	    bbox_valid = False;
	}
}

#if	PS

static	XIOErrorHandler	oldhandler;

static	int		XDviIOErrorHandler P1C(Display *, disp)
{
	ps_destroy();
	return oldhandler(disp);
}

/*
 *	Initialize paths for file searching
 */

void
ps_init_paths()
{
#if TOOLKIT
	static	Boolean	done_already	= False;

	if (done_already)
	    return;
	done_already = True;
#endif

#if	CFGFILE

	if ((search_header.path1 = getenv("XDVIHEADERS")) == NULL
	  && (search_header.path1 = getenv("TEXPSHEADERS")) == NULL)
	    search_header.path1 = getenv("PSHEADERS");
	search_header.envptr = ffgetenv("PSHEADERS");
	/* clear it if it's a getenv() placeholder */
	if (search_header.envptr != NULL && search_header.envptr->value == NULL)
	    search_header.envptr = NULL;

	if ((search_fig.path1 = getenv("XDVIPICTS")) == NULL
	  && (search_fig.path1 = getenv("TEXPICTS")) == NULL)
	    search_fig.path1 = getenv("TEXINPUTS");
	search_fig.envptr = ffgetenv("TEXPICTS");
	/* clear it if it's a getenv() placeholder */
	if (search_fig.envptr != NULL && search_fig.envptr->value == NULL)
	    search_fig.envptr = NULL;

#else	/* not CFGFILE */

	if ((search_header.path1 = getenv("XDVIHEADERS")) == NULL
		&& (search_header.path1 = getenv("TEXPSHEADERS")) == NULL
		&& (search_header.path1 = getenv("PSHEADERS")) == NULL
		) {
	    search_header.path1 = search_header.path2;
	    search_header.path2 = NULL;
	}

	if ((search_fig.path1 = getenv("XDVIPICTS")) == NULL
		&& (search_fig.path1 = getenv("TEXPICTS")) == NULL
		&& (search_fig.path1 = getenv("TEXINPUTS")) == NULL
		) {
	    search_fig.path1 = search_fig.path2;
	    search_fig.path2 = NULL;
	}

#endif	/* not CFGFILE */

}


static	void
actual_startup()
{
	ps_init_paths();

	/* This allows us to clean up after server shutdowns, etc.  */
	oldhandler = XSetIOErrorHandler(XDviIOErrorHandler);

	/*
	 * Figure out what we want to use to display postscript figures
	 * and set at most one of the following to True:
	 * resource.useGS, resource.useDPS, resource.useNeWS
	 *
	 * Choose DPS then NEWS then Ghostscript if they are available
	 */
	if (!(
#ifdef	PS_DPS
	    (resource.useDPS && initDPS())
#if	defined(PS_NEWS) || defined(PS_GS)
	    ||
#endif
#endif	/* PS_DPS */

#ifdef	PS_NEWS
	    (resource.useNeWS && initNeWS())
#ifdef	PS_GS
	    ||
#endif
#endif	/* PS_NEWS */

#ifdef	PS_GS
	    (resource.useGS && initGS())
#endif

	    ))
	    psp = no_ps_procs;
}

static	void
ps_startup(xul, yul, cp)
	int		xul, yul;
	_Xconst	char	*cp;
{
	if (!resource._postscript) {
	    draw_bbox();
	    return;
	}
	actual_startup();
	psp.drawbegin(xul, yul, cp);
}

static	void
ps_startup2()
{
	actual_startup();
	psp.beginheader();
}

/* ARGSUSED */
static	void
NullProc2(cp)
	_Xconst	char	*cp;
{}

static	void
ps_parseraw(PostScript_cmd)
	_Xconst	char	*PostScript_cmd;
{
	_Xconst	char	*p;

	/* dumb parsing of PostScript - search for rotation H. Zeller 1/97 */
	bbox_angle = 0;
	p = strstr(PostScript_cmd, "rotate");
	if (p != NULL) {
	    while (*p != '\0' && !isdigit(*p)) --p;
	    while (*p != '\0' && isdigit(*p)) --p;
	    if (*p != '+' && *p != '-') ++p;
	    sscanf(p, "%d neg rotate", &bbox_angle);
	}
}

/* ARGSUSED */
void
#if	NeedFunctionPrototypes
drawbegin_none(int xul, int yul, _Xconst char *cp)
#else	/* !NeedFunctionPrototypes */
drawbegin_none(xul, yul, cp)
	int		xul, yul;
	_Xconst	char	*cp;
#endif	/* NeedFunctionPrototypes */
{
	draw_bbox();
}


/*
 *	Mechanism to cache results of ``tick'' specials.
 */

struct tickrec {
	struct tickrec	*next;
	int		pageno;
	char		*command;
	char		*tempname;
};

static	struct tickrec	*tickhead	= NULL;	/* head of linked list of */
						/* cached information */
static	int		nticks		= 0;	/* number of records total */

#ifndef	TICKCACHESIZE
#define	TICKCACHESIZE	3
#endif

/*
 *	cachetick() - returns:
 *		NULL		error;
 *		fp == NULL	string was not in cache, fd = file
 *		fp != NULL	string was in cache, fp = cached file
 */

static	struct tickrec *
cachetick(filename, pathinfo, fp, fdp)
	_Xconst char	*filename;
	struct findrec	*pathinfo;
	FILE		**fp;
	int		*fdp;
{
	struct tickrec	**linkp;
	struct tickrec	*tikp;
	struct tickrec	**freerecp;

	linkp = &tickhead;
	freerecp = NULL;
	for (;;) {		/* see if we have it already */
	    tikp = *linkp;
	    if (tikp == NULL) {	/* if end of list */
		int	fd;

		if (nticks >= TICKCACHESIZE && freerecp != NULL) {
		    tikp = *freerecp;
		    *freerecp = tikp->next;
		    free(tikp->command);
		    unlink(tikp->tempname);
		    /* reuse tikp and tikp->tempname */
		}
		else {
		    tikp = xmalloc(sizeof(struct tickrec));
		    tikp->tempname = NULL;
		    ++nticks;
		}
		fd = xdvi_temp_fd(&tikp->tempname);
		if (fd == -1) {
		    perror("Cannot create temporary file");
		    free(tikp->tempname);
		    free(tikp);
		    return NULL;
		}
		tikp->command = xstrdup(filename);
		*fp = NULL;
		*fdp = fd;
		break;
	    }
	    if (strcmp(filename, tikp->command) == 0) {	/* found it */
		*linkp = tikp->next;	/* unlink it */
		*fp = xfopen(tikp->tempname, OPEN_MODE);
		if (*fp == NULL) {
		    perror(tikp->tempname);
		    free(tikp->tempname);
		    free(tikp->command);
		    free(tikp);
		    return NULL;
		}
		break;
	    }
	    if (tikp->pageno != current_page) freerecp = linkp;
	    linkp = &tikp->next;
	}
	tikp->next = tickhead;		/* link it in */
	tickhead = tikp;
	tikp->pageno = pathinfo != &search_header ? current_page : -1;
	return tikp;
}

void
ps_clear_cache()
{
	struct tickrec	*tikp;

	while (tickhead != NULL) {
	    tikp = tickhead;
	    tickhead = tickhead->next;
	    free(tikp->command);
	    unlink(tikp->tempname);
	    free((char *) tikp->tempname);
	    free(tikp);
	}
	nticks = 0;
}

#ifndef	UNCOMPRESS
#define	UNCOMPRESS	"uncompress"
#endif

#ifndef	GUNZIP
#define	GUNZIP		"gunzip"
#endif

#ifndef	BUNZIP2
#define	BUNZIP2		"bunzip2"
#endif

static	void
send_ps_file(filename, pathinfo)
	_Xconst char	*filename;
	struct findrec	*pathinfo;
{
	FILE		*f;
	int		fd;
	static _Xconst char *argv[]	= {NULL, "-c", NULL, NULL};
	_Xconst char	*bufp;
	struct tickrec	*tikp;
	char		*p;
	unsigned	len;
	char		magic1, magic2, magic3;

	if (psp.drawfile == NULL || !resource._postscript) return;

	if (filename[0] == '`') {
	    if (!resource.allow_shell) {
		if (warn_spec_now)
		    WARN1(XmDIALOG_ERROR,
		      "Shell escape disallowed for special\n\"%s\"", filename);
		return;
	    }

	    tikp = cachetick(filename, pathinfo, &f, &fd);
	    if (tikp == NULL)	/* if error */
		return;
	    if (f == NULL) {	/* if not cached, need to create */
		close(fd);
		++n_files_left;
		len = strlen(filename) + strlen(tikp->tempname) + (4 - 1);
		if (len > ffline_len)
		    expandline(len);
		Sprintf(ffline, "%s > %s", filename + 1, tikp->tempname);
		(void) system(ffline);
		f = xfopen(tikp->tempname, OPEN_MODE);
		if (f == NULL) {
		    perror(tikp->tempname);
		    return;
		}
	    }
	    bufp = tikp->tempname;
	}
	else {
	    f = NULL;
		/* first try the same path as the dvi file */
	    if (filename[0] != '/') {
		p = rindex(dvi_name, '/');
		if (p == NULL) bufp = filename;
		else {
		    unsigned len1, len2;

		    len1 = ++p - dvi_name;
		    len2 = strlen(filename) + 1;
		    if (len1 + len2 > ffline_len)
			expandline(len1 + len2);
		    bcopy(dvi_name, ffline, len1);
		    bcopy(filename, ffline + len1, len2);
		    bufp = ffline;
		}
		if (debug & DBG_OPEN) Printf("Trying PS file %s\n", bufp);
		f = xfopen(bufp, OPEN_MODE);
	    }
	    if (f == NULL) {
		f = filefind(filename, pathinfo, (_Xconst char **) NULL);
		bufp = ffline;
	    }

	    /* if still no luck, complain */
	    if (f == NULL) {
		WARN1(XmDIALOG_ERROR,
		  "Cannot find PostScript file for inclusion in document:\n%s",
		  filename);
		draw_bbox();
		return;
	    }

	    /* check for compressed files */
	    len = strlen(filename);
	    magic1 = '\037';
	    magic3 = '\0';
	    if ((len > 2 && strcmp(filename + len - 2, ".Z") == 0
		      && (argv[0] = UNCOMPRESS, magic2 = '\235', True))
		    || (len > 3 && strcmp(filename + len - 3, ".gz") == 0
		      && (argv[0] = GUNZIP, magic2 = '\213', True))
		    || (len > 4 && strcmp(filename + len - 4, ".bz2") == 0
		      && (argv[0] = BUNZIP2, magic1 = 'B', magic2 = 'Z',
			magic3 = 'h', True))) {
		if (getc(f) != magic1 || (char) getc(f) != magic2
		  || (magic3 != '\0' && getc(f) != magic3))
		    rewind(f);
		else {
		    Fclose(f);
		    ++n_files_left;
		    tikp = cachetick(filename, pathinfo, &f, &fd);
		    if (tikp == NULL)	/* if error */
			return;
		    if (f == NULL) {	/* if not cached, need to create */
			pid_t	pid;
			int	status;

			argv[2] = bufp;
			Fflush(stderr);	/* avoid double flushing */
			pid = vfork();
			if (pid == 0) {	/* if child */
			    (void) dup2(fd, 1);
			    (void) execvp(argv[0], (char **) argv);
			    Fprintf(stderr, "Execvp of %s failed.\n", argv[0]);
			    Fflush(stderr);
			    _exit(1);
			}
			(void) close(fd);
			++n_files_left;
			for (;;) {
#if HAVE_WAITPID
			    if (waitpid(pid, &status, 0) != -1) break;
#else
# if HAVE_WAIT4
			    if (wait4(pid, &status, 0, (struct rusage *) NULL)
				    != -1)
				break;
# else
			    int retval;

			    retval = wait(&status);
			    if (retval == pid) break;
			    if (retval != -1) continue;
# endif
#endif
			    if (errno == EINTR) continue;
			    perror("[xdvi] waitpid");
			    return;
			}
			f = xfopen(tikp->tempname, OPEN_MODE);
			if (f == NULL) {
			    perror(tikp->tempname);
			    return;
			}
		    }
		    bufp = tikp->tempname;
		}
	    }
	}

	/* Success! */
	psp.drawfile(bufp, f);	/* this is supposed to close the file */
}


void
ps_destroy()
{
	struct tickrec	*tikp;

	/* Note:  old NeXT systems (at least) lack atexit/on_exit.  */
	psp.destroy();
	for (tikp = tickhead; tikp != NULL; tikp = tikp->next)
	    if (unlink(tikp->tempname) < 0)
		perror(tikp->tempname);
}

#endif	/* PS */


void
init_prescan()
{
#if PS
	struct tickrec	*tikp;
#endif

	scanned_page =
#if PS
	  scanned_page_ps = scanned_page_ps_bak =
#endif
#if COLOR
	  scanned_page_color =
#endif
	  scanned_page_reset = resource.prescan ? -1 : total_pages + 1;

#if PS
	if (!resource._postscript)
	    scanned_page_ps = total_pages + 1;

	for (tikp = tickhead; tikp != NULL; tikp = tikp->next)
	    tikp->pageno = -1;
	psp.newdoc();
#endif

#if COLOR
	if (!use_color)
	    scanned_page_color = total_pages + 1;
#endif

	if (ignore_papersize_specials)
	    scanned_page =
#if PS && COLOR
	      scanned_page_ps < scanned_page_color
	      ? scanned_page_ps : scanned_page_color;
#elif PS
	      scanned_page_ps;
#elif COLOR
	      scanned_page_color;
#else
	      total_pages + 1;
#endif
}


static	void
psfig_special(cp)
	char	*cp;
{
	char	*filename;
	int	raww, rawh;

	if (strncmp(cp, ":[begin]", 8) == 0) {
	    cp += 8;
	    bbox_valid = False;
	    bbox_angle = 0;
	    if (sscanf(cp, "%d %d\n", &raww, &rawh) >= 2) {
		bbox_valid = True;
		bbox_width = pixel_conv(spell_conv(raww));
		bbox_height = pixel_conv(spell_conv(rawh));
		bbox_voffset = 0;
	    }
	    if (currwin.win == mane.win)
#if	PS
		psp.drawbegin(PXL_H - currwin.base_x, PXL_V - currwin.base_y,
		    cp);
#else
		draw_bbox();
#endif
	    psfig_begun = True;
	} else if (strncmp(cp, " plotfile ", 10) == 0) {
	    cp += 10;
	    while (isspace(*cp)) cp++;
	    for (filename = cp; !isspace(*cp); ++cp);
	    *cp = '\0';
#if	PS
	    if (currwin.win == mane.win) send_ps_file(filename, &search_fig);
#endif
	} else if (strncmp(cp, ":[end]", 6) == 0) {
	    cp += 6;
#if	PS
	    if (currwin.win == mane.win) psp.drawend(cp);
#endif
	    bbox_valid = False;
	    psfig_begun = False;
	} else if (*cp == ':') {
	    /* I am going to send some raw postscript stuff */
	    ++cp;	/* skip the colon */
#if	PS
	    ps_parseraw(cp);
	    if (currwin.win == mane.win) psp.drawraw(cp);
#endif
	} else {
	    /* also raw PostScript, but no extra colon to skip */
#if	PS
	    if (currwin.win == mane.win) {
		ps_parseraw(cp);
		if (psfig_begun) psp.drawraw(cp);
		else {
		    psp.drawbegin(PXL_H - currwin.base_x,
			PXL_V - currwin.base_y, cp);
		    psp.drawend("");
		}
	    }
#endif
	}
}


/*	Keys for epsf specials */

static	_Xconst	char	*keytab[]	= {"clip",
					   "llx",
					   "lly",
					   "urx",
					   "ury",
					   "rwi",
					   "rhi",
					   "hsize",
					   "vsize",
					   "hoffset",
					   "voffset",
					   "hscale",
					   "vscale",
					   "angle"};

#define	KEY_LLX	keyval[0]
#define	KEY_LLY	keyval[1]
#define	KEY_URX	keyval[2]
#define	KEY_URY	keyval[3]
#define	KEY_RWI	keyval[4]
#define	KEY_RHI	keyval[5]

#define	NKEYS	(sizeof(keytab)/sizeof(*keytab))
#define	N_ARGLESS_KEYS 1

static	void
epsf_special(cp)
	char	*cp;
{
	char	*filename;
	static	char		*buffer;
	static	unsigned int	buflen	= 0;
	unsigned int		len;
	char	*q;
	int	flags	= 0;
	double	keyval[6];

	filename = cp;
	if (*cp == '\'' || *cp == '"') {
	    do ++cp;
	    while (*cp != '\0' && *cp != *filename);
	    ++filename;
	}
	else
	    while (*cp != '\0' && !isspace(*cp)) ++cp;
	if (*cp != '\0') *cp++ = '\0';
	while (isspace(*cp)) ++cp;
	len = strlen(cp) + NKEYS + 30;
	if (buflen < len) {
	    if (buflen != 0) free(buffer);
	    buflen = len;
	    buffer = xmalloc(buflen);
	}
	Strcpy(buffer, "@beginspecial");
	q = buffer + strlen(buffer);
	while (*cp != '\0') {
	    char *p1 = cp;
	    int keyno;

	    while (*p1 != '=' && !isspace(*p1) && *p1 != '\0') ++p1;
	    for (keyno = 0;; ++keyno) {
		if (keyno >= NKEYS) {
		    if (warn_spec_now)
			Fprintf(stderr,
			    "%s: unknown keyword (%.*s) in \\special will be ignored\n",
			    prog, (int) (p1 - cp), cp);
		    break;
		}
		if (memcmp(cp, keytab[keyno], p1 - cp) == 0) {
		    if (keyno >= N_ARGLESS_KEYS) {
			while (isspace(*p1)) ++p1;
			if (*p1 == '=') {
			    ++p1;
			    while (isspace(*p1)) ++p1;
			}
			if (keyno < N_ARGLESS_KEYS + 6) {
			    keyval[keyno - N_ARGLESS_KEYS] = atof(p1);
			    flags |= (1 << (keyno - N_ARGLESS_KEYS));
			}
			*q++ = ' ';
			while (!isspace(*p1) && *p1 != '\0') *q++ = *p1++;
		    }
		    *q++ = ' ';
		    *q++ = '@';
		    Strcpy(q, keytab[keyno]);
		    q += strlen(q);
		    break;
		}
	    }
	    cp = p1;
	    while (!isspace(*cp) && *cp != '\0') ++cp;
	    while (isspace(*cp)) ++cp;
	}
	Strcpy(q, " @setspecial\n");

	bbox_valid = False;
	if ((flags & 0x30) == 0x30 || ((flags & 0x30) && (flags & 0xf) == 0xf)){
	    bbox_valid = True;
	    bbox_width = 0.1 * ((flags & 0x10) ? KEY_RWI
		: KEY_RHI * (KEY_URX - KEY_LLX) / (KEY_URY - KEY_LLY))
		* dimconv / shrink_factor + 0.5;
	    bbox_voffset = bbox_height = 0.1 * ((flags & 0x20) ? KEY_RHI
		: KEY_RWI * (KEY_URY - KEY_LLY) / (KEY_URX - KEY_LLX))
		* dimconv / shrink_factor + 0.5;
	}

	if (currwin.win == mane.win) {
#if	PS
	    psp.drawbegin(PXL_H - currwin.base_x, PXL_V - currwin.base_y,
		buffer);
	    /* talk directly with the DPSHandler here */
	    send_ps_file(filename, &search_fig);
	    psp.drawend(" @endspecial");
#else
	    draw_bbox();
#endif
	}
	bbox_valid = False;
}


static	void
quote_special(cp)
	char	*cp;
{
	bbox_valid = False;

#if	PS
	if (currwin.win == mane.win) {
	    psp.drawbegin(PXL_H - currwin.base_x, PXL_V - currwin.base_y,
		"@beginspecial @setspecial ");
	    /* talk directly with the DPSHandler here */
	    psp.drawraw(cp + 1);
	    psp.drawend(" @endspecial");
	}
#endif

	/* nothing else to do--there's no bbox here */
}

#if	PS

static	void
scan_header(cp)
	char	*cp;
{
	char	*filename;

#if PS_GS
	if (gs_postpone_prescan)
	    return;
#endif
	filename = cp;
	if (*cp == '\'' || *cp == '"') {
	    do ++cp;
	    while (*cp != '\0' && *cp != *filename);
	    *cp = '\0';
	    ++filename;
	}

	psp.beginheader();
	send_ps_file(filename, &search_header);
}

static	void
scan_bang(cp)
	char	*cp;
{
	psp.beginheader();
	psp.drawraw(cp + 1);
}

#endif	/* PS */

#if COLOR

/*
 *	Table of dvips color names.  Produced by passing the table in
 *	dvipsk/color.lpro through the following perl script, and then
 *	through sort.
 *
 *	#! /usr/bin/perl -w
 *	
 *	sub cvpart {
 *	    return $_[0] < 1.0 ? 1.0 - $_[0] : 0.0;
 *	}
 *	
 *	$pat = "^\\/(\\w+)\\s*\\{\\s*([0-9.]+)\\s*([0-9.]+)\\s*([0-9.]+)"
 *	  . "\\s*([0-9.]+)\\s*setcmykcolor\\s*\\}\\s*DC\\s*\$";
 *	while (<STDIN>) {
 *	    chop;
 *	    if (/^%%/) {next;}
 *	    if (/$pat/o) {
 *		printf "\t\tDVIPSCOLORDESC(%2d, \"%s\", %g, %g, %g),\n",
 *		  length($1), $1,
 *		  cvpart($2 + $5), cvpart($3 + $5), cvpart($4 + $5);
 *	    }
 *	    else {
 *		print "Bad line: ", $_, "\n";
 *	    }
 *	}
 */

struct dvipscolor {
	int		len;
	_Xconst char	*name;
	struct rgb	color;
};

#define	CVPART(x)	((int)((x) * 65535 + 0.5))
#define	DVIPSCOLORDESC(n, name, r, g, b) \
		{n, name, {CVPART(r), CVPART(g), CVPART(b)}}

static	struct dvipscolor	colornames[] = {
		DVIPSCOLORDESC( 3, "Red", 1, 0, 0),
		DVIPSCOLORDESC( 3, "Tan", 0.86, 0.58, 0.44),
		DVIPSCOLORDESC( 4, "Blue", 0, 0, 1),
		DVIPSCOLORDESC( 4, "Cyan", 0, 1, 1),
		DVIPSCOLORDESC( 4, "Gray", 0.5, 0.5, 0.5),
		DVIPSCOLORDESC( 4, "Plum", 0.5, 0, 1),
		DVIPSCOLORDESC( 5, "Black", 0, 0, 0),
		DVIPSCOLORDESC( 5, "Brown", 0.4, 0, 0),
		DVIPSCOLORDESC( 5, "Green", 0, 1, 0),
		DVIPSCOLORDESC( 5, "Melon", 1, 0.54, 0.5),
		DVIPSCOLORDESC( 5, "Peach", 1, 0.5, 0.3),
		DVIPSCOLORDESC( 5, "Sepia", 0.3, 0, 0),
		DVIPSCOLORDESC( 5, "White", 1, 1, 1),
		DVIPSCOLORDESC( 6, "Maroon", 0.68, 0, 0),
		DVIPSCOLORDESC( 6, "Orange", 1, 0.39, 0.13),
		DVIPSCOLORDESC( 6, "Orchid", 0.68, 0.36, 1),
		DVIPSCOLORDESC( 6, "Purple", 0.55, 0.14, 1),
		DVIPSCOLORDESC( 6, "Salmon", 1, 0.47, 0.62),
		DVIPSCOLORDESC( 6, "Violet", 0.21, 0.12, 1),
		DVIPSCOLORDESC( 6, "Yellow", 1, 1, 0),
		DVIPSCOLORDESC( 7, "Apricot", 1, 0.68, 0.48),
		DVIPSCOLORDESC( 7, "Emerald", 0, 1, 0.5),
		DVIPSCOLORDESC( 7, "Fuchsia", 0.45, 0.01, 0.92),
		DVIPSCOLORDESC( 7, "Magenta", 1, 0, 1),
		DVIPSCOLORDESC( 7, "SkyBlue", 0.38, 1, 0.88),
		DVIPSCOLORDESC( 7, "Thistle", 0.88, 0.41, 1),
		DVIPSCOLORDESC( 8, "BrickRed", 0.72, 0, 0),
		DVIPSCOLORDESC( 8, "Cerulean", 0.06, 0.89, 1),
		DVIPSCOLORDESC( 8, "Lavender", 1, 0.52, 1),
		DVIPSCOLORDESC( 8, "Mahogany", 0.65, 0, 0),
		DVIPSCOLORDESC( 8, "Mulberry", 0.64, 0.08, 0.98),
		DVIPSCOLORDESC( 8, "NavyBlue", 0.06, 0.46, 1),
		DVIPSCOLORDESC( 8, "SeaGreen", 0.31, 1, 0.5),
		DVIPSCOLORDESC( 8, "TealBlue", 0.12, 0.98, 0.64),
		DVIPSCOLORDESC( 9, "BlueGreen", 0.15, 1, 0.67),
		DVIPSCOLORDESC( 9, "CadetBlue", 0.38, 0.43, 0.77),
		DVIPSCOLORDESC( 9, "Dandelion", 1, 0.71, 0.16),
		DVIPSCOLORDESC( 9, "Goldenrod", 1, 0.9, 0.16),
		DVIPSCOLORDESC( 9, "LimeGreen", 0.5, 1, 0),
		DVIPSCOLORDESC( 9, "OrangeRed", 1, 0, 0.5),
		DVIPSCOLORDESC( 9, "PineGreen", 0, 0.75, 0.16),
		DVIPSCOLORDESC( 9, "RawSienna", 0.55, 0, 0),
		DVIPSCOLORDESC( 9, "RedOrange", 1, 0.23, 0.13),
		DVIPSCOLORDESC( 9, "RedViolet", 0.59, 0, 0.66),
		DVIPSCOLORDESC( 9, "Rhodamine", 1, 0.18, 1),
		DVIPSCOLORDESC( 9, "RoyalBlue", 0, 0.5, 1),
		DVIPSCOLORDESC( 9, "RubineRed", 1, 0, 0.87),
		DVIPSCOLORDESC( 9, "Turquoise", 0.15, 1, 0.8),
		DVIPSCOLORDESC( 9, "VioletRed", 1, 0.19, 1),
		DVIPSCOLORDESC(10, "Aquamarine", 0.18, 1, 0.7),
		DVIPSCOLORDESC(10, "BlueViolet", 0.1, 0.05, 0.96),
		DVIPSCOLORDESC(10, "DarkOrchid", 0.6, 0.2, 0.8),
		DVIPSCOLORDESC(10, "OliveGreen", 0, 0.6, 0),
		DVIPSCOLORDESC(10, "Periwinkle", 0.43, 0.45, 1),
		DVIPSCOLORDESC(11, "Bittersweet", 0.76, 0.01, 0),
		DVIPSCOLORDESC(11, "BurntOrange", 1, 0.49, 0),
		DVIPSCOLORDESC(11, "ForestGreen", 0, 0.88, 0),
		DVIPSCOLORDESC(11, "GreenYellow", 0.85, 1, 0.31),
		DVIPSCOLORDESC(11, "JungleGreen", 0.01, 1, 0.48),
		DVIPSCOLORDESC(11, "ProcessBlue", 0.04, 1, 1),
		DVIPSCOLORDESC(11, "RoyalPurple", 0.25, 0.1, 1),
		DVIPSCOLORDESC(11, "SpringGreen", 0.74, 1, 0.24),
		DVIPSCOLORDESC(11, "YellowGreen", 0.56, 1, 0.26),
		DVIPSCOLORDESC(12, "MidnightBlue", 0, 0.44, 0.57),
		DVIPSCOLORDESC(12, "YellowOrange", 1, 0.58, 0),
		DVIPSCOLORDESC(13, "CarnationPink", 1, 0.37, 1),
		DVIPSCOLORDESC(14, "CornflowerBlue", 0.35, 0.87, 1),
		DVIPSCOLORDESC(14, "WildStrawberry", 1, 0.04, 0.61),
};

#undef CVPART
#undef DVIPSCOLORDESC

static	Boolean
parse_color(cp0, cp, rgbp)
	_Xconst char	*cp0, *cp;
	struct rgb	*rgbp;
{
	double		r, g, b;

	while (*cp == ' ') ++cp;

	if (memicmp(cp, "rgb ", 4) == 0) {
	    if (sscanf(cp + 3, "%lf %lf %lf", &r, &g, &b) == 3
	      && r >= 0 && r <= 1 && g >= 0 && g <= 1 && b >= 0 && b <= 1) {
		rgbp->r = r * 65535 + 0.5;
		rgbp->g = g * 65535 + 0.5;
		rgbp->b = b * 65535 + 0.5;
		return True;
	    }
	}
	else if (memicmp(cp, "gray ", 5) == 0) {
	    if (sscanf(cp + 4, "%lf", &r) == 1 && r >= 0 && r <= 1) {
		rgbp->r = rgbp->g = rgbp->b = r * 65535 + 0.5;
		return True;
	    }
	}
	else if (memicmp(cp, "cmyk ", 5) == 0) {
	    double k;

	    if (sscanf(cp + 4, "%lf %lf %lf %lf", &r, &g, &b, &k) == 4
	      && r >= 0 && r <= 1 && g >= 0 && g <= 1 && b >= 0 && b <= 1
	      && k >= 0 && k <= 1) {
		r = 1.0 - r - k;	/* cyan --> red */
		rgbp->r = (r < 0 ? 0 : r * 65535 + 0.5);
		g = 1.0 - g - k;	/* magenta --> green */
		rgbp->g = (g < 0 ? 0 : g * 65535 + 0.5);
		b = 1.0 - b - k;	/* yellow --> blue */
		rgbp->b = (b < 0 ? 0 : b * 65535 + 0.5);
		return True;
	    }
	}
	else if (memicmp(cp, "hsb ", 4) == 0) {
	    double hue, sat, bri;

	    if (sscanf(cp + 3, "%lf %lf %lf", &hue, &sat, &bri) == 3 && hue >= 0
	      && hue <= 1 && sat >= 0 && sat <= 1 && bri >= 0 && bri <= 1) {
		int	h	= (int) (6 * hue);
		double	p	= bri * (1 - sat);
		double	q	= bri * (1 - sat * (6 * hue - h));
		double	t	= p - q + bri;

		switch (h) {
		    case 0:  r = bri; g = t; b = p; break; /* Red - Yel */
		    case 1:  r = q; g = bri; b = p; break; /* Yel - Grn */
		    case 2:  r = p; g = bri; b = t; break; /* Grn - Cyn */
		    case 3:  r = p; g = q; b = bri; break; /* Cyn - Blu */
		    case 4:  r = t; g = p; b = bri; break; /* Blu - Mag */
		    case 5:  r = bri; g = p; b = q; break; /* Mag - Red */
		    case 6:  r = bri; g = b = p;    break; /* Red */
		}
		rgbp->r = r * 65535 + 0.5;
		rgbp->g = g * 65535 + 0.5;
		rgbp->b = b * 65535 + 0.5;
		return True;
	    }
	}
	else {		/* check table of dvips color names */
	    int i1 = -1;	/* ends of range for bisection search */
	    int i2 = XtNumber(colornames);
	    int i;
	    int len;

	    len = 0;
	    while (isalpha(cp[len]))
		++len;	/* get length of color name */
	    do {
		struct dvipscolor *dp;
		int		j;

		i = (i1 + i2) / 2;
		dp = &colornames[i];
		j = len - dp->len;
		if (j == 0) {
		    j = memcmp(cp, dp->name, len);
		    if (j == 0) {
			*rgbp = dp->color;
			return True;
		    }
		}
		if (j > 0) i1 = i;
		else i2 = i;
	    }
	    while (i2 - i1 > 1);
	}

	if (cp0 != NULL)
	    Fprintf(stderr, "Invalid color name in special `%s'\n", cp0);

	return False;
}


/*
 *	Color stack used when scanning.  These records stay around, to ease
 *	the burden on xmalloc().  The first entry is a dummy entry, giving
 *	the default foreground color (as modified).
 */

struct colorframe {
	struct colorframe	*next, *prev;
	struct rgb		color;
};

static	struct colorframe	scanstack_head;
static	int			scanstack_len;
static	struct colorframe	*scanstack_current;

static	void
init_page_colors()
{
	int	i;

	page_colors = xmalloc(total_pages * sizeof *page_colors);

	for (i = 0; i <= scanned_page + 1; ++i) {
	    page_colors[i].bg = bg_initial;	/* from command line */
	    page_colors[i].colorstack = &fg_initial;
	    page_colors[i].stacksize = 1;
	}
	while (i < total_pages)
	    page_colors[i++].colorstack = NULL;

	scanstack_head.color = fg_initial;
	scanstack_len = 1;			/* nothing yet */
	scanstack_current = &scanstack_head;	/* stack position */
}


static	void
scan_bg_color(cp)
	_Xconst char	*cp;
{
	if (!use_color)
	    return;

	if (page_colors == NULL)
	    init_page_colors();

	(void) parse_color(cp, cp + 11, &page_colors[scanned_page + 1].bg);
}

static	void
scan_color(cp)
	_Xconst char	*cp;
{
	_Xconst char	*cp1 = cp + 6;

	if (!use_color)
	    return;

	while (*cp1 == ' ') ++cp1;

	if (page_colors == NULL)
	    init_page_colors();

	if (memicmp(cp1, "push ", 5) == 0) {
	    if (scanstack_current->next == NULL) {	/* if at end */
		scanstack_current->next = xmalloc(sizeof *scanstack_current);
		scanstack_current->next->prev = scanstack_current;
		scanstack_current->next->next = NULL;
	    }
	    scanstack_current = scanstack_current->next;
	    ++scanstack_len;
	    if (!parse_color(cp, cp1 + 5, &scanstack_current->color))
		scanstack_current->color = scanstack_current->prev->color;
	}
	else if (memicmp(cp1, "pop", 3) == 0) {
	    if (scanstack_len <= 1) {
		fprintf(stderr,
		  "Warning:  Color pop occurred with empty color stack.\n");
	    }
	    else {
		scanstack_current = scanstack_current->prev;
		--scanstack_len;
	    }
	}
	else {
	    (void) parse_color(cp, cp1, &scanstack_head.color);
	    if (scanstack_len > 1) {
		struct colorframe	*cfp;

		Fprintf(stderr, "Warning:  Global color change occurred with non-empty color stack;\n\
coping by setting all stack entries to that color.\n");
		for (cfp = scanstack_head.next;; cfp = cfp->next) {
		    cfp->color = scanstack_head.color;
		    if (cfp == scanstack_current) break;
		}
	    }
	}
}

void
scan_color_eop()
{
	int			i;
	_Xconst struct rgb	*prev;
	struct colorframe	*cf;
	_Xconst struct rgb	*p1;
	struct rgb		*rgbp;

	if (page_colors == NULL)
	    return;

	/* set background color for next page */
	if (scanned_page + 1 < total_pages)
	    page_colors[scanned_page + 1].bg = page_colors[scanned_page].bg;

	/* save the stack contents */
	page_colors[scanned_page].stacksize = scanstack_len;

	prev = &fg_initial;
	i = 1;
	if (scanned_page > 0) {
	    prev = page_colors[scanned_page - 1].colorstack;
	    i = page_colors[scanned_page - 1].stacksize;
	}
	if (scanstack_len <= i) {
	    /* try to reuse the previous array */
	    p1 = prev;
	    cf = &scanstack_head;
	    for (i = 0;;) {
		if (p1->r != cf->color.r || p1->g != cf->color.g
		  || p1->b != cf->color.b)
		    break;
		if (++i >= scanstack_len) {	/* end loop; reuse memory */
		    page_colors[scanned_page].colorstack = prev;
		    return;	/* done */
		}
		++p1;
		cf = cf->next;
	    }
	}
	page_colors[scanned_page].colorstack = rgbp
	  = xmalloc(scanstack_len * sizeof *rgbp);
	cf = &scanstack_head;
	for (i = 0; i < scanstack_len; ++i) {
	    *rgbp++ = cf->color;
	    cf = cf->next;
	}
}


/*
 *	Page stack when displaying.  Again, the records stay around once
 *	allocated.  Bottom entries in the stack (inherited from previous
 *	pages) are stored in an array instead (see comments in xdvi.h).
 */

static	struct colorframe	*rcs_head;


/*
 *	We don't actually do any X calls yet to change colors; that can wait
 *	until a character or rule needs to be drawn.
 */

void
set_fg_color(_Xconst struct rgb *color)
{
	struct fgrec	**fgpp;

	if (fg_current != NULL
	  && color->r == fg_current->color.r
	  && color->g == fg_current->color.g
	  && color->b == fg_current->color.b)
	    return;

	for (fgpp = &bg_current->fg_head;;) {
	    fg_current = *fgpp;
	    if (fg_current == NULL) {	/* if color is not in list */
		fg_current = *fgpp = xmalloc(sizeof *fg_current);
		fg_current->next = NULL;
		fg_current->color = *color;
		fg_current->pixel_good = False;
#if GREY
		fg_current->palette_good = False;
#endif
		break;
	    }
	    if (fg_current->color.r == color->r
	      && fg_current->color.g == color->g
	      && fg_current->color.b == color->b)
		break;
	    fgpp = &fg_current->next;
	}
	if (debug & DBG_DVI)
	    printf("Changing fg color to %5d %5d %5d\n",
	      color->r, color->g, color->b);
}

static	void
color_special(cp)
	_Xconst char	*cp;
{
	if (!use_color)
	    return;

	while (*cp == ' ') ++cp;

	if (memicmp(cp, "push ", 5) == 0) {
	    if (rcs_top == NULL) {
		if (rcs_head == NULL) {
		    rcs_head = xmalloc(sizeof *rcs_head);
		    rcs_head->prev = rcs_head->next = NULL;
		}
		rcs_top = rcs_head;
	    }
	    else {
		struct colorframe *rcs_next;

		rcs_next = rcs_top->next;
		if (rcs_next == NULL) {
		    rcs_next = rcs_top->next = xmalloc(sizeof *rcs_next);
		    rcs_next->prev = rcs_top;
		    rcs_next->next = NULL;
		}
		rcs_top = rcs_next;
	    }
	    if (!parse_color(NULL, cp + 5, &rcs_top->color)) {
		if (rcs_top->prev != NULL)
		    rcs_top->color = rcs_top->prev->color;
		else
		    rcs_top->color = color_bottom[color_bot_size - 1];
	    }
	    set_fg_color(&rcs_top->color);
	}
	else if (memicmp(cp, "pop", 3) == 0) {
	    /* Pop stack */
	    if (rcs_top != NULL) {
		if (color_bot_size > 0)
		    rcs_top = rcs_top->prev;
		else return;
	    }
	    else if (color_bot_size > 1)
		--color_bot_size;
	    else {
		if (scanned_page_reset >= 0) {
		    /* turn on scanning and redraw the page */
		    scanned_page =
#if PS
		      scanned_page_ps =
#endif
		      scanned_page_color = scanned_page_reset = -1;
		    ev_flags |= EV_NEWPAGE;		/* force a redraw */
		    longjmp(canit_env, 1);
		}
		return;
	    }

	    set_fg_color(rcs_top != NULL ? &rcs_top->color
		: &color_bottom[color_bot_size - 1]);
	}
	else {
	    struct rgb color;

	    if (!parse_color(NULL, cp, &color)) return;

	    /* clear stack */
	    if (rcs_head == NULL) {
		rcs_head = xmalloc(sizeof *rcs_head);
		rcs_head->prev = rcs_head->next = NULL;
	    }
	    rcs_top = rcs_head;
	    color_bot_size = 0;

	    /* Change top of stack */
	    rcs_top->color = color;

	    set_fg_color(&color);
	}
}

#endif /* COLOR */


static	unsigned int
myatopix(pp)
	_Xconst char	**pp;
{
	unsigned int	value;
	_Xconst char	*cp = *pp;
	char		scr[16];
	_Xconst char	*p0, *p1;

	p0 = cp;
	while ((*cp >= '0' && *cp <= '9') || *cp == '.') ++cp;
	p1 = cp;
	while (isspace(*cp)) ++cp;
	if (*cp >= 'a' && *cp <= 'z' && cp[1] >= 'a' && cp[1] <= 'z') {
	    /* if units are present */
	    if (p1 - p0 <= XtNumber(scr) - 3) {
		Sprintf(scr, "%.*s%c%c", (int) (p1 - p0), p0, *cp, cp[1]);
		value = atopix(scr, False);
	    }
	    else value = 0;
	    cp += 2;
	}
	else
	    value = atopix(p0, False);

	*pp = cp;
	return value;
}

static	void
scan_papersize(cp0)
	_Xconst char	*cp0;
{
	_Xconst char	*cp	= cp0;
	unsigned int	w, h;
	double		mag	= 1.;

	if (ignore_papersize_specials)
	    return;

	if (*cp == '*') {
	    do ++cp; while (isspace(*cp));
	    mag = magnification * .001;
	}
	    
	w = myatopix(&cp) * mag + 0.5;

	while (isspace(*cp)) ++cp;
	if (*cp == ',')
	    do ++cp; while (isspace(*cp));

	h = myatopix(&cp) * mag + 0.5;

	if (w == 0 || h == 0)
	    Fprintf(stderr, "%s:  invalid papersize special \"%s\"\n", prog,
	      cp0);
	else {
	    page_info[scanned_page + 1].pw = page_info[scanned_page + 1].ww = w;
	    page_info[scanned_page + 1].ph = page_info[scanned_page + 1].wh = h;
	}
}


/*
 *	The following copyright message applies to the rest of this file.  --PV
 */

/*
 *	This program is Copyright (C) 1987 by the Board of Trustees of the
 *	University of Illinois, and by the author Dirk Grunwald.
 *
 *	This program may be freely copied, as long as this copyright
 *	message remaines affixed. It may not be sold, although it may
 *	be distributed with other software which is sold. If the
 *	software is distributed, the source code must be made available.
 *
 *	No warranty, expressed or implied, is given with this software.
 *	It is presented in the hope that it will prove useful.
 *
 *	Hacked in ignorance and desperation by jonah@db.toronto.edu
 */

/*
 *      The code to handle the \specials generated by tpic was modified
 *      by Dirk Grunwald using the code Tim Morgan at Univ. of Calif, Irvine
 *      wrote for TeXsun.
 */

static	char *
endofcommand(cp)
	char	*cp;
{
	while (isspace(*cp)) ++cp;
	if (*cp != '=') return NULL;
	do ++cp; while (isspace(*cp));
	return cp;
}

#define	CMD(x, y)	((x) << 8 | (y))

void
applicationDoSpecial(cp)
	char	*cp;
{
	char	*p;

	if (debug & DBG_DVI)
	    Printf("          `%s'\n", cp);

	/* Skip white space */
	while (isspace(*cp)) ++cp;

	/* Ignore initial "xdvi:" */
	if (memcmp(cp, "xdvi:", 5) == 0) {
	    cp += 5;
	    while (isspace(*cp)) ++cp;
	}

	/* PostScript specials */

	if (*cp == '"') {
	    quote_special(cp);
	    return;
	}
	if (memicmp(cp, "ps:", 3) == 0) {
	    psfig_special(cp + 3);
	    return;
	}
	if (memicmp(cp, "psfile", 6) == 0
		&& (p = endofcommand(cp + 6)) != NULL) {
	    epsf_special(p);
	    return;
	}
#if COLOR
	if (memicmp(cp, "color ", 6) == 0) {
	    color_special(cp + 6);
	    return;
	}
#endif

	/* these should have been scanned */

	if (*cp == '!'
	  || (memicmp(cp, "header", 6) == 0 && endofcommand(cp + 6) != NULL)) {
#if PS
	    if (resource._postscript && scanned_page_reset >= 0) {
		/* turn on scanning and redraw the page */
		scanned_page =
# if COLOR
		  scanned_page_color =
# endif
		  scanned_page_ps = scanned_page_reset = -1;
		ev_flags |= EV_NEWPAGE;		/* force a redraw */
		longjmp(canit_env, 1);
	    }
#endif /* PS */
	    return;
	}

	if (memicmp(cp, "background ", 11) == 0) {
#if COLOR
	    if (use_color && scanned_page_reset >= 0) {
		/* turn on scanning and redraw the page */
		scanned_page =
# if PS
		  scanned_page_ps =
# endif
		  scanned_page_color = scanned_page_reset = -1;
		ev_flags |= EV_NEWPAGE;		/* force a redraw */
		longjmp(canit_env, 1);
	    }
#endif /* COLOR */
	    return;
	}

	if (memcmp(cp, "papersize", 9) == 0 && endofcommand(cp + 9) != NULL) {
	    if (scanned_page_reset >= 0) {
		/* turn on scanning and redraw the page */
		scanned_page = 
#if PS
		  scanned_page_ps =
#endif
#if COLOR
		  scanned_page_color =
#endif
		  scanned_page_reset = -1;
		ev_flags |= EV_NEWPAGE;		/* force a redraw */
		longjmp(canit_env, 1);
	    }
	    return;
	}

	/* tpic specials */

	if (*cp >= 'a' && *cp <= 'z' && cp[1] >= 'a' && cp[1] <= 'z' &&
		(isspace(cp[2]) || cp[2] == '\0')) {
	    switch (CMD(*cp, cp[1])) {
		case CMD('p','n'): set_pen_size(cp + 2); return;
		case CMD('f','p'): flush_path(); return;
		case CMD('d','a'): flush_dashed(cp + 2, False); return;
		case CMD('d','t'): flush_dashed(cp + 2, True); return;
		case CMD('p','a'): add_path(cp + 2); return;
		case CMD('a','r'): arc(cp + 2, False); return;
		case CMD('i','a'): arc(cp + 2, True); return;
		case CMD('s','p'): flush_spline(); return;
		case CMD('s','h'): shade_last(); return;
		case CMD('w','h'): whiten_last(); return;
		case CMD('b','k'): blacken_last(); return;
		case CMD('i','p'): /* throw away the path -- jansteen */
		    path_len = 0; return;
	    }
	}

	if (warn_spec_now && memcmp(cp, "src:", 4) != 0)
	    Fprintf(stderr, "%s:  special \"%s\" not implemented\n", prog, cp);
}

#undef	CMD


void
scan_special(cp)
	char	*cp;
{
	char	*p;

	if (debug & DBG_PS)
	    Printf("Scanning special `%s'.\n", cp);

	/* Skip white space */
	while (isspace(*cp)) ++cp;

	/* Ignore initial "xdvi:" */
	if (memcmp(cp, "xdvi:", 5) == 0) {
	    cp += 5;
	    while (isspace(*cp)) ++cp;
	}

#if PS
# if COLOR
	if (scanned_page_ps <= scanned_page)
# endif
	{
	    if (*cp == '!') {
		scan_bang(cp);
		return;
	    }
	    else if (memicmp(cp, "header", 6) == 0
	      && (p = endofcommand(cp + 6)) != NULL) {
		scan_header(p);
		return;
	    }
	}
#endif /* PS */

#if COLOR
# if PS
	if (scanned_page_color <= scanned_page)
# endif
	{
	    if (memicmp(cp, "background ", 11) == 0) {
		scan_bg_color(cp);
		return;
	    }
	    else if (memicmp(cp, "color ", 6) == 0) {
		scan_color(cp);
		return;
	    }
	}
#endif /* COLOR */

	if (memcmp(cp, "papersize", 9) == 0
	  && (p = endofcommand(cp + 9)) != NULL) {
	    scan_papersize(p);
	    return;
	}
}


#define	xspell_conv(n)	spell_conv0(n, current_dimconv)
#define	xpixel_conv(x)	((int) ((x) >> 16))
#define	G_PXL_H		xpixel_conv(currinf.data.dvi_h)

/* ARGSUSED */
void
geom_do_special(g_info, cp, current_dimconv)
	struct geom_info	*g_info;
		/* cp is not _Xconst, because of endofcommand(). */
	char			*cp;
	double			current_dimconv;
{
	_Xconst char	*p;

	/* Skip white space */
	while (isspace(*cp)) ++cp;

	/* Ignore initial "xdvi:" */
	if (memcmp(cp, "xdvi:", 5) == 0) {
	    cp += 5;
	    while (isspace(*cp)) ++cp;
	}

	if (memicmp(cp, "psfile", 6) == 0
	  && (p = endofcommand(cp + 6)) != NULL) {
	    /* compute epsf bounding box */
	    char c;
	    int flags = 0;
	    double keyval[6];

	    c = *p;
	    if (c == '\'' || c == '"') {
		do ++p;
		while (*p != '\0' && *p != c);
	    }
	    else
		while (*p != '\0' && !isspace(*p)) ++p;
	    while (isspace(*p)) ++p;
	    while (*p != '\0') {
		_Xconst char *p1 = p;
		int keyno;

		while (*p1 != '=' && !isspace(*p1) && *p1 != '\0') ++p1;
		for (keyno = 0; keyno < NKEYS; ++keyno) {
		    if (memcmp(p, keytab[keyno], p1 - p) == 0) {
			if (keyno >= N_ARGLESS_KEYS) {
			    while (isspace(*p1)) ++p1;
			    if (*p1 == '=') {
				++p1;
				while (isspace(*p1)) ++p1;
			    }
			    if (keyno < N_ARGLESS_KEYS + 6) {
				keyval[keyno - N_ARGLESS_KEYS] = atof(p1);
				flags |= (1 << (keyno - N_ARGLESS_KEYS));
			    }
			    while (!isspace(*p1) && *p1 != '\0') ++p1;
			}
			break;
		    }
		}
		p = p1;
		while (!isspace(*p) && *p != '\0') ++p;
		while (isspace(*p)) ++p;
	    }

	    if ((flags & 0x30) == 0x30
	      || ((flags & 0x30) && (flags & 0xf) == 0xf)){
		long	x	= G_PXL_H;
		long	y	= PXL_V;
		long	bbox_w;
		long	bbox_h;

		bbox_w = 0.1 * ((flags & 0x10) ? KEY_RWI
		  : KEY_RHI * (KEY_URX - KEY_LLX) / (KEY_URY - KEY_LLY))
		  * dimconv + 0.5;
		bbox_h = 0.1 * ((flags & 0x20) ? KEY_RHI
		  : KEY_RWI * (KEY_URY - KEY_LLY) / (KEY_URX - KEY_LLX))
		  * dimconv + 0.5;

		g_info->geom_box(g_info, x, y - bbox_h, x + bbox_w, y);
	    }
	}
	else if (memicmp(cp, "ps::[begin]", 11) == 0) {
	    /* compute psfig bounding box */
	    long bbox_w, bbox_h;

	    if (sscanf(cp + 11, "%ld %ld\n", &bbox_w, &bbox_h) >= 2) {
		long	x	= G_PXL_H;
		long	y	= PXL_V;

		bbox_w = xpixel_conv(spell_conv(bbox_w));
		bbox_h = xpixel_conv(spell_conv(bbox_h));

		g_info->geom_box(g_info, x, y, x + bbox_w, y + bbox_h);
	    }
	}
}
