/*========================================================================*\

Copyright (c) 2001-2004  Paul Vojta

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL PAUL VOJTA OR ANY OTHER AUTHOR OF OR CONTRIBUTOR TO
THIS SOFTWARE BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
IN THE SOFTWARE.

\*========================================================================*/

/*
 *	Code for popup windows.
 */

#include "xdvi.h"
#include <ctype.h>
#include <errno.h>
#include <signal.h>
#include <sys/stat.h>

/* Xlib and Xutil are already included */

#include <X11/Xatom.h>
#include <X11/StringDefs.h>

#if XAW
# include <X11/Shell.h>
# if XAW3D
#  include <X11/Xaw3d/Form.h>
#  include <X11/Xaw3d/Label.h>
#  include <X11/Xaw3d/Command.h>
#  include <X11/Xaw3d/Toggle.h>
#  include <X11/Xaw3d/AsciiText.h>
#  include <X11/Xaw3d/Viewport.h>
# else
#  include <X11/Xaw/Form.h>
#  include <X11/Xaw/Label.h>
#  include <X11/Xaw/Command.h>
#  include <X11/Xaw/Toggle.h>
#  include <X11/Xaw/AsciiText.h>
#  include <X11/Xaw/Viewport.h>
# endif
# if XtSpecificationRelease < 5
#  define XawChainLeft	XtChainLeft
#  define XawChainRight	XtChainRight
#  define XawChainTop	XtChainTop
#  define XawChainBottom XtChainBottom
# endif
# if XtSpecificationRelease < 6
#  define XawFmt8Bit	FMT8BIT
# endif
#else /* MOTIF */
# include <Xm/BulletinB.h>
# include <Xm/DialogS.h>
# include <Xm/MessageB.h>
# include <Xm/FileSB.h>
# include <Xm/LabelG.h>
# include <Xm/Form.h>
# include <Xm/Frame.h>
# include <Xm/ToggleBG.h>
# include <Xm/Text.h>
# include <Xm/TextF.h>
# include <Xm/PushB.h>
# include <Xm/Protocols.h>
#endif /* MOTIF */

#if !defined(S_ISDIR) && defined(S_IFDIR)
#define	S_ISDIR(m)	(((m) & S_IFMT) == S_IFDIR)
#endif

#if !defined(S_ISLNK) && defined(S_IFLNK)
#define	S_ISLNK(m)	(((m) & S_IFMT) == S_IFLNK)
#endif

#ifdef X_NOT_STDC_ENV
extern	int	errno;
#endif

#if NeedVarargsPrototypes
# include <stdarg.h>
#else
# include <varargs.h>
#endif

#ifndef va_copy
# ifdef __va_copy
#  define va_copy __va_copy
# else
#  define va_copy(aq, ap) aq = ap
# endif
#endif

#if HAVE_GOOD_SETSID_VFORK
# if HAVE_VFORK_H
#  include <vfork.h>
# endif
#else
  /* Mac OS X 10.3 (Panther) (11/2003) doesn't allow setsid() within vfork() */
# undef vfork
# define vfork fork
#endif

#if HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif
#ifndef WIFEXITED
# define WIFEXITED(status)	(((status) & 255) == 0)
#endif
#ifndef WEXITSTATUS
# define WEXITSTATUS(status)	((unsigned)(status) >> 8)
#endif
#ifndef WIFSIGNALED
# ifndef WIFSTOPPED
#  define WIFSTOPPED(status)	(((status) & 0xff) == 0x7f)
# endif
# define WIFSIGNALED(status)	(!WIFSTOPPED(status) && !WIFEXITED(status))
#endif
#ifndef WTERMSIG
# define WTERMSIG(status)	((status) & 0x7f)
#endif

/* if POSIX O_NONBLOCK is not available, use O_NDELAY */
#if !defined(O_NONBLOCK) && defined(O_NDELAY)
# define O_NONBLOCK O_NDELAY
#endif

#ifdef EWOULDBLOCK
# ifdef EAGAIN
#  define AGAIN_CONDITION	(errno == EWOULDBLOCK || errno == EAGAIN)
# else
#  define AGAIN_CONDITION	(errno == EWOULDBLOCK)
# endif
#else /* EWOULDBLOCK */
# ifdef EAGAIN
#  define AGAIN_CONDITION	(errno == EAGAIN)
# endif
#endif /* EWOULDBLOCK */

#if HAVE_POLL
# include <poll.h>
# define XIO_IN POLLIN
# define XIO_OUT POLLOUT
#else
# define XIO_IN 1
# define XIO_OUT 2
#endif


/*
 *	New variants on printf.
 */

static int uintlen(i)
	unsigned int i;
{
	int	len = 0;

	do {
	    ++len;
	    i /= 10;
	}
	while (i != 0);

	return len;
}

/*
 *	(v)nprintf(fmt, ...) - return the length of the string produced.
 */

#if HAVE_GOOD_VSNPRINTF

# define vnprintf(fmt, ap)	vsnprintf(NULL, 0, fmt, ap)

#else /* !HAVE_GOOD_VSNPRINTF */

static int
vnprintf(format, args)
	_Xconst char	*format;
	va_list		args;
{
	_Xconst char	*p, *p_end, *q;
	int		len;

	p = format;
	p_end = p + strlen(p);
	len = 0;
	for (;;) {
	    q = memchr(p, '%', p_end - p);
	    if (q == NULL) {
		len += p_end - p;
		break;
	    }
	    len += q - p;
	    p = q + 1;
	    switch (*p++) {
		case '%':
		    ++len;
		    break;
		case 'd':
		    {
			int i = va_arg(args, int);

			len += (i < 0 ? uintlen(-i) + 1 : uintlen(i));
		    }
		    break;
		case 's':
		    len += strlen(va_arg(args, _Xconst char *));
		    break;
		default:
		    oops("Invalid designator `%c' in format string \"%s\"\n",
			p[-1], format);
	    }
	}

	return len;
}


#if 0	/* not needed (yet) */

static int
#if NeedVarargsPrototypes
nprintf(_Xconst char *format, ...)
#else
/* VARARGS */
nprintf(va_alist)
	va_dcl
#endif
{
#if !NeedVarargsPrototypes
	_Xconst char	*format;
#endif
	va_list		args;
	int		len;

#if NeedVarargsPrototypes
	va_start(args, format);
#else
	va_start(args);
	format = va_arg(args, _Xconst char *);
#endif

	len = vnprintf(format, args);
	va_end(args);
	return len;
}

#endif /* if 0 */

#endif /* !HAVE_GOOD_VSNPRINTF */

/*
 *	(v)mprintf(fmt, ...) - printf to allocated string.
 */

static char *
vmprintf(format, args)
	_Xconst char	*format;
	va_list		args;
{
#if !HAVE_VASPRINTF
	va_list		args2;
	int		len;
#endif
	char		*result;

#if HAVE_VASPRINTF
	if (vasprintf(&result, format, args) < 0)
	    oops("! Out of memory (call to vasprintf() with format \"%s\").\n",
	      format);
#else
	va_copy(args2, args);
	len = vnprintf(format, args2) + 1;
	va_end(args2);

	result = xmalloc(len);
# if HAVE_VSNPRINTF
	(void) vsnprintf(result, len, format, args);
# else
	(void) vsprintf(result, format, args);
# endif
#endif /* ! HAVE_VASPRINTF */

	return result;
}

static char *
#if NeedVarargsPrototypes
mprintf(_Xconst char *format, ...)
#else
/* VARARGS */
mprintf(va_alist)
	va_dcl
#endif
{
#if !NeedVarargsPrototypes
	_Xconst char	*format;
#endif
	va_list		args;
	char		*result;

#if NeedVarargsPrototypes
	va_start(args, format);
#else
	va_start(args);
	format = va_arg(args, _Xconst char *);
#endif

	result = vmprintf(format, args);
	va_end(args);
	return result;
}


#if XAW

/*
 *	Realize the widget and set the callback for the WM_DESTROY protocol.
 */

static void
XdviXawRealizePopup(shell, callback)
	Widget		shell;
	XtCallbackProc	callback;
{
	XtRealizeWidget(shell);

	XSetWMProtocols(DISP, XtWindow(shell), &XA_WM_DELETE_WINDOW, 1);
	XtAddEventHandler(shell, NoEventMask, True, handle_messages, callback);
}

#endif /* XAW */


/*
 *	Simple popup - Just a text message.  May not be done before realizing
 *	the main window.
 */

/*
 *	In reality, the "simple" popup is not so simple, because certain
 *	window managers (e.g., twm, fvwm1) leave the window showing if you
 *	pop it down too soon after popping it up.  To remedy this, we delay
 *	calling XtPopdown() until the previous call to XtPopup() has generated
 *	a MapNotify event.  Likewise, we delay XtPopup() until the previous
 *	XtPopdown() call has produced an UnmapNotify event.
 *
 *	To make it just a little more complicated, for some reason the popup
 *	may get mapped after being unmapped.  If so, we call XtPopup() to
 *	inform the toolkit of its mistake, and arrange to expect some spurious
 *	events.  To reproduce this bug, remove the "spurious" code, run xdvi,
 *	move the mouse a little, then hit "n" a few times.  This happens only
 *	with twm.
 */

/* ARGSUSED */
static	void
handle_popup_events(widget, rec, ev, cont)
	Widget	widget;
	XtPointer rec;
	XEvent	*ev;
	Boolean	*cont;		/* unused */
{
#define	prec	((struct status_popup *) rec)

	if (ev->type == MapNotify) {
	    if (prec->expected_type == MapNotify) {
		prec->expected_type = 0;
		if (!prec->popped) {
		    prec->expected_type = UnmapNotify;
		    XtPopdown(prec->shell);
		}
	    }
	    else {
		if (debug & DBG_EVENT)
		    fprintf(stderr,
		      "handle_popup_events: got spurious MapNotify\n");
		prec->expected_type = UnmapNotify;
		prec->spurious = True;
		XtPopup(prec->shell, XtGrabNone);
	    }
	}
	else if (ev->type == UnmapNotify) {
	    if (prec->expected_type == UnmapNotify) {
		prec->expected_type = 0;
		if (prec->spurious) {
		    prec->expected_type = MapNotify;
		    prec->spurious = False;
		}
		else if (prec->popped) {
		    prec->expected_type = MapNotify;
		    XtPopup(prec->shell, XtGrabNone);
		}
	    }
	    else
		if (debug & DBG_EVENT)
		    fprintf(stderr,
		      "handle_popup_events: got spurious UnmapNotify\n");
	}

#undef	prec
}


#if XAW

void
simple_popup(rec, message, callback)
	struct status_popup	*rec;
	_Xconst char		*message;
	XtCallbackProc		callback;
{
	Position	x, y;
	Dimension	w1, h1, w2, h2;
	Boolean		old_rec_popped	= rec->popped;

	rec->popped = True;
	if (rec->shell == NULL) {
	    rec->shell = XtVaCreatePopupShell("status",
	      transientShellWidgetClass, top_level,
	      XtNtitle, "Xdvi status",
	      XtNmappedWhenManaged, False,
	      XtNallowShellResize, True,
	      XtNtransientFor, top_level,
	      NULL);
	    XtAddEventHandler(rec->shell, StructureNotifyMask, False,
	      handle_popup_events, (XtPointer) rec);

	    rec->label = XtVaCreateManagedWidget("label", labelWidgetClass,
	      rec->shell,
	      XtNlabel, message,
	      NULL);

	    XdviXawRealizePopup(rec->shell, callback);
	}
	else
	    XtVaSetValues(rec->label, XtNlabel, message, NULL);

	if (mane.win != (Window) 0)
	    XtVaGetValues(top_level, XtNx, &x, XtNy, &y,
	      XtNwidth, &w1, XtNheight, &h1, NULL);
	else {
	    x = y = 0;
	    w1 = WidthOfScreen(SCRN);
	    h1 = HeightOfScreen(SCRN);
	}

	/* Get the size of the popup window */
	XtVaGetValues(rec->label, XtNwidth, &w2, XtNheight, &h2, NULL);

	/* Center the popup over the main window */
	XtVaSetValues(rec->shell, XtNx, x + (w1 - w2) / 2,
	  XtNy, y + (h1 - h2) / 2, NULL);

	if (!rec->expected_type && !old_rec_popped) {
	    rec->expected_type = MapNotify;
	    XtPopup(rec->shell, XtGrabNone);
	}
}

#else /* MOTIF */

void
simple_popup(rec, message, callback)
	struct status_popup	*rec;
	_Xconst char		*message;
	XtCallbackProc		callback;
{
	Widget		bb;
	XmString	str;
	Boolean		old_rec_popped	= rec->popped;

	rec->popped = True;
	str = XmStringCreateLtoR((char *) message, XmFONTLIST_DEFAULT_TAG);
	if (rec->shell == NULL) {
	    rec->shell = XtVaCreatePopupShell("status",
	      xmDialogShellWidgetClass, top_level,
	      XmNallowShellResize, True,
	      XmNdeleteResponse, XmDO_NOTHING,
	      XmNtitle, "Xdvi status",
	      NULL);
	    XtAddEventHandler(rec->shell, StructureNotifyMask, False,
	      handle_popup_events, (XtPointer) rec);
	    XmAddWMProtocolCallback(rec->shell, XA_WM_DELETE_WINDOW, callback,
	      NULL);

	    bb = XtCreateWidget("bulletin", xmBulletinBoardWidgetClass,
	      rec->shell, NULL, 0);

	    rec->label = XtVaCreateManagedWidget("label", xmLabelGadgetClass,
	      bb,
	      XmNlabelString, str,
	      NULL);

	    XtManageChild(bb);
	}
	else
	    XtVaSetValues(rec->label, XmNlabelString, str, NULL);

	XmStringFree(str);

	if (!rec->expected_type && !old_rec_popped) {
	    rec->expected_type = MapNotify;
	    XtPopup(rec->shell, XtGrabNone);
	}
}

#endif

void
simple_popdown(rec)
	struct status_popup	*rec;
{
	rec->popped = False;
	if (!rec->expected_type) {
	    rec->expected_type = UnmapNotify;
	    XtPopdown(rec->shell);
	}
}


/*
 *	Warning popup - used for most error and notice conditions.
 *	It includes a message and a button (and, in the case of the Motif
 *	toolkit, a bitmap).
 */

/* ARGSUSED */
static	void
warn_callback(w, client_data, call_data)
	Widget		w;
	XtPointer	client_data;
	XtPointer	call_data;
{
	/* XtIsShell(w) holds if this is caused by the WM_DESTROY protocol */
	XtDestroyWidget(XtIsShell(w) ? w : (Widget) client_data);
}

#if XAW

void
do_popup(shell)
	Widget		shell;
{
	Position	x, y;
	Dimension	w1, h1, w2, h2;

	/* Get the size and location of the main window */
	x = y = 0;
	w1 = WidthOfScreen(SCRN);
	h1 = HeightOfScreen(SCRN);
	if (mane.win != (Window) 0)
	    XtVaGetValues(top_level, XtNx, &x, XtNy, &y,
	      XtNwidth, &w1, XtNheight, &h1, NULL);

	/* Get the size of the popup window */
	XtVaGetValues(shell, XtNwidth, &w2, XtNheight, &h2, NULL);

	/* Center the popup over the main window */
	XtVaSetValues(shell, XtNx, x + (w1 - w2) / 2,
	  XtNy, y + (h1 - h2) / 2, NULL);

	XtPopup(shell, XtGrabNone);
}

Widget
warning_popup(message, button_name, callback)
	_Xconst char	*message;
	_Xconst char	*button_name;
	XtCallbackProc	callback;
{
	Widget		shell, form, label, button;
	Dimension	w1, w2, bw;
	int		dist;

	shell = XtVaCreatePopupShell("warn", transientShellWidgetClass,
	  top_level,
	  XtNtitle, "Xdvi warning",
	  XtNmappedWhenManaged, False,
	  XtNtransientFor, top_level,
	  NULL);
	form = XtCreateManagedWidget("form", formWidgetClass, shell,
	  NULL, 0);
	label = XtVaCreateManagedWidget("label", labelWidgetClass, form,
	  XtNlabel, message,
	  XtNborderWidth, 0,
	  XtNleft, XtChainLeft,
	  XtNright, XtChainLeft,
	  NULL);
	button = XtVaCreateManagedWidget(button_name, commandWidgetClass, form,
	  XtNaccelerators, accels_cr_click,
	  XtNfromVert, label,
	  NULL);
	XtAddCallback(button, XtNcallback,
	  callback != NULL ? callback : warn_callback, shell);
	XtInstallAccelerators(form, button);

	/* Center button under message */
	XtVaGetValues(label, XtNwidth, &w1, NULL);
	XtVaGetValues(button, XtNwidth, &w2, XtNborderWidth, &bw, NULL);
	w2 += 2 * bw;
	if (w1 > w2) {
	    XtVaGetValues(button, XtNhorizDistance, &dist, NULL);
	    XtVaSetValues(button, XtNhorizDistance, dist + (w1 - w2) / 2, NULL);
	}
	else if (w1 < w2) {
	    XtVaGetValues(label, XtNhorizDistance, &dist, NULL);
	    XtVaSetValues(label, XtNhorizDistance, dist + (w2 - w1) / 2, NULL);
	}

	XdviXawRealizePopup(shell, callback != NULL ? callback : warn_callback);

	if (postpone_popups) {
	    /* save it for later */
	    if (n_init_popups >= alloc_init_popups) {
		if (alloc_init_popups != 0)
		    init_popups = xrealloc(init_popups,
		      (alloc_init_popups += 2) * sizeof (*init_popups));
		else
		    init_popups = xmalloc((alloc_init_popups = 2)
		      * sizeof (*init_popups));
	    }
	    init_popups[n_init_popups++] = shell;
	}
	else
	    do_popup(shell);

	return shell;
}

#else /* MOTIF */

void
do_popup(dialog)
	Widget		dialog;
{
	XtManageChild(dialog);
	XtPopup(XtParent(dialog), XtGrabNone);
}

Widget
warning_popup(message, mtype, button_name, callback)
	_Xconst char	*message;
	int		mtype;
	_Xconst char	*button_name;
	XtCallbackProc	callback;
{
	Widget		shell;
	Widget		dialog;
	XmString	str;

	shell = XtVaCreatePopupShell("warn", xmDialogShellWidgetClass,
	  top_level,
	  XmNdeleteResponse, XmDO_NOTHING,
	  XmNtitle, "Xdvi warning",
	  NULL);
	XmAddWMProtocolCallback(shell, XA_WM_DELETE_WINDOW,
	  callback != NULL ? callback : warn_callback, NULL);

	dialog = XtVaCreateWidget("dialog", xmMessageBoxWidgetClass, shell,
	  XmNdialogType, mtype,
	  NULL);
	XtUnmanageChild(XmMessageBoxGetChild(dialog, XmDIALOG_CANCEL_BUTTON));
	XtUnmanageChild(XmMessageBoxGetChild(dialog, XmDIALOG_HELP_BUTTON));

	XtAddCallback(dialog, XmNokCallback,
	  callback != NULL ? callback : warn_callback, shell);

	str = XmStringCreateLtoR((char *) message, XmFONTLIST_DEFAULT_TAG);
	XtVaSetValues(dialog, XmNmessageString, str, NULL);
	XmStringFree(str);

	if (button_name != NULL) {
	    str = XmStringCreateLocalized((char *) button_name);
	    XtVaSetValues(dialog, XmNokLabelString, str, NULL);
	    XmStringFree(str);
	}

	XtInstallAllAccelerators(dialog,
	  XmMessageBoxGetChild(dialog, XmDIALOG_OK_BUTTON));

	if (postpone_popups) {
	    /* save it for later */
	    if (n_init_popups >= alloc_init_popups) {
		if (alloc_init_popups != 0)
		    init_popups = xrealloc(init_popups,
		      (alloc_init_popups += 2) * sizeof (*init_popups));
		else
		    init_popups = xmalloc((alloc_init_popups = 2)
		      * sizeof (*init_popups));
	    }
	    init_popups[n_init_popups++] = dialog;
	}
	else
	    do_popup(dialog);

	return shell;
}

#endif /* MOTIF */

Widget
#if NeedVarargsPrototypes
warning_popup_long(_Xconst char *format,
# if MOTIF
  int mtype,
# endif
  _Xconst char *button_name, XtCallbackProc callback, ...)
#else /* ! NeedVarargsPrototypes */
# if XAW
warning_popup_long(format, button_name, callback, va_alist)
	_Xconst char	*format;
	_Xconst char	*button_name;
	XtCallbackProc	callback;
# else /* MOTIF */
warning_popup_long(format, mtype, button_name, callback, va_alist)
	_Xconst char	*format;
	int		mtype;
	_Xconst char	*button_name;
	XtCallbackProc	callback;
# endif
	va_dcl
#endif /* ! NeedVarargsPrototypes */
{
	va_list		args;
	char		*str;
	Widget		w;

#if NeedVarargsPrototypes
	va_start(args, callback);
#else
	va_start(args);
#endif
	str = vmprintf(format, args);
	va_end(args);

#if XAW
	w = warning_popup(str, button_name, callback);
#else /* MOTIF */
	w = warning_popup(str, mtype, button_name, callback);
#endif

	free(str);

	return w;
}


#if XAW

/*
 *	Confirm popup - Pop up a window and ask a yes/no question.
 */

Widget
confirm_popup(message, button_name, callback, yes_answer, no_answer)
	_Xconst char	*message;
	_Xconst char	*button_name;
	XtCallbackProc	callback;
	XtPointer	yes_answer, no_answer;
{
	Widget		shell, form, label, button1, button2;
	int		ddist;
	Dimension	w0, w1, w2, bw1, bw2, tw;

	shell = XtVaCreatePopupShell("confirm", transientShellWidgetClass,
	  top_level,
	  XtNtitle, "Xdvi question",
	  XtNmappedWhenManaged, False,
	  XtNtransientFor, top_level,
	  NULL);
	form = XtCreateManagedWidget("form", formWidgetClass, shell,
	  NULL, 0);
	label = XtVaCreateManagedWidget("label", labelWidgetClass, form,
	  XtNlabel, message,
	  XtNborderWidth, 0,
	  XtNleft, XtChainLeft,
	  XtNright, XtChainLeft,
	  NULL);

	button1 = XtVaCreateManagedWidget(button_name, commandWidgetClass, form,
	  XtNaccelerators, accels_cr,
	  XtNfromVert, label,
	  NULL);
	XtAddCallback(button1, XtNcallback, callback, yes_answer);
	XtInstallAccelerators(form, button1);

	button2 = XtVaCreateManagedWidget("Cancel", commandWidgetClass, form,
	  XtNfromVert, label,
	  XtNfromHoriz, button1,
	  NULL);
	XtAddCallback(button2, XtNcallback, callback, no_answer);

	/* Move cancel button to the right */
	XtVaGetValues(label, XtNwidth, &w0, NULL);
	XtVaGetValues(button1, XtNwidth, &w1, XtNborderWidth, &bw1, NULL);
	XtVaGetValues(button2, XtNwidth, &w2, XtNborderWidth, &bw2, NULL);
	XtVaGetValues(form, XtNdefaultDistance, &ddist, NULL);
	tw = w1 + w2 + 2 * (bw1 + bw2);
	if (tw < w0 + ddist)
	    XtVaSetValues(button2, XtNhorizDistance, w0 - tw, NULL);

	XdviXawRealizePopup(shell, callback);
	do_popup(shell);

	return shell;
}

#else /* MOTIF */

Widget
confirm_popup(message, button_name, callback, yes_answer, no_answer)
	char		*message;
	char		*button_name;
	XtCallbackProc	callback;
	XtPointer	yes_answer, no_answer;
{
	Widget		dialog;
	XmString	msg, yes, no;

	{
	    static Arg	cq_args[] = {
		{XmNdeleteResponse, XmDO_NOTHING},
	    };

	    dialog = XmCreateQuestionDialog(top_level, "confirm",
	      cq_args, XtNumber(cq_args));
	    XmAddWMProtocolCallback(XtParent(dialog), XA_WM_DELETE_WINDOW,
	      callback, NULL);
	}
	msg = XmStringCreateLocalized(message);
	yes = XmStringCreateLocalized(button_name);
	no = XmStringCreateLocalized("Cancel");
	XtVaSetValues(dialog,
	  XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL,
	  XmNmessageString, msg,
	  XmNokLabelString, yes,
	  XmNcancelLabelString, no,
	  NULL);
	XmStringFree(msg);
	XmStringFree(yes);
	XmStringFree(no);

	XtAddCallback(dialog, XmNokCallback, callback, yes_answer);
	XtAddCallback(dialog, XmNcancelCallback, callback, no_answer);
	XtUnmanageChild(XmMessageBoxGetChild(dialog, XmDIALOG_HELP_BUTTON));

	XtManageChild(dialog);
	dialog = XtParent(dialog);
	XtPopup(dialog, XtGrabNone);

	return dialog;
}

#endif /* MOTIF */


/*
 *	File popup - pop up a window to allow the user to switch dvi files.
 */

/* Common data for XAW and MOTIF */

static	Boolean	file_active	= False;	/* if file window is showing */
static	Widget	file_shell	= NULL;		/* shell widget of popup */
static	Boolean	reposition	= False;	/* we want to reposition it */

/* Common routines */

/* ARGSUSED */
static	void
cb_file_cancel(w, client_data, call_data)
	Widget		w;
	XtPointer	client_data;
	XtPointer	call_data;
{
	XtPopdown(file_shell);
	file_active = False;

	if (mane.win == (Window) 0)
	    exit(1);
}

#if XAW

static	Widget	f1text, f2vport, f2clip, f2form, f3text, f4text;
static	Widget	*widget_list;			/* list of file entries */
static	size_t	wl_num		= 0;		/* actual number of widgets */
static	Widget	file_highlit	= NULL;		/* highlighted widget */
static	int	file_hl_seq	= -1;		/* sequence # of the above */
static	char	*dirbuf		= NULL;		/* buffer containing dir name */
static	size_t	dirlen		= 0;		/* length of above buffer */
static	XtTranslations	file_xlats;		/* for use in file entries */
static	int	file_entry_height = 0;		/* line height in viewport */
static	Position f2ddist;			/* size of list border */

struct suffix {
	_Xconst char	*str;
	size_t		len;
};
static	struct suffix	*suffixes	= NULL;
static	struct suffix	*suff_end	= NULL;
static	size_t		suff_max	= 0;	/* size of array allocated */

/* Icon bitmap of directory */

#define filedir_width 16
#define filedir_height 13
static unsigned char filedir_bits[] = {
   0xf0, 0x00, 0x08, 0x01, 0x06, 0x7e, 0x01, 0x80, 0x01, 0x80, 0x01, 0x80,
   0x01, 0x80, 0x01, 0x80, 0x01, 0x80, 0x01, 0x80, 0xfe, 0x7f, 0x00, 0x00,
   0x00, 0x00};

static	char	fileblank_bits[sizeof filedir_bits];
static	Pixmap	filedir_pm	= (Pixmap) 0;
static	Pixmap	fileblank_pm	= (Pixmap) 0;

/* Bitmap of check mark */
#define filechk_width 13
#define filechk_height 13
static unsigned char filechk_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x0c, 0x00, 0x06, 0x00, 0x03,
   0x86, 0x01, 0xcc, 0x00, 0x78, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00};

static	char	fileunchk_bits[sizeof filechk_bits];
static	Pixmap	filechk_pm	= (Pixmap) 0;
static	Pixmap	fileunchk_pm	= (Pixmap) 0;
static	char	filehide	= '.';

/* If the file widget is used before the dvi window goes up, then we destroy
 * it because it might not have the right colormap and visual.  We want to save
 * the settings between these two incarnations, though.  For the directory and
 * the hidden-file checkbox, these are OK, but we need to do the same for the
 * suffixes.  */

static	_Xconst char	*file_save_suff		= ".dvi";

static void act_file_go ARGS((Widget, XEvent *, String *, Cardinal *));
static void act_file_click ARGS((Widget, XEvent *, String *, Cardinal *));
static void act_file_suffixes ARGS((Widget, XEvent *, String *, Cardinal *));
static void act_file_scroll ARGS((Widget, XEvent *, String *, Cardinal *));
static void act_file_arrow ARGS((Widget, XEvent *, String *, Cardinal *));
static void act_file_home ARGS((Widget, XEvent *, String *, Cardinal *));
static void act_file_end ARGS((Widget, XEvent *, String *, Cardinal *));

static	XtActionsRec	file_actions[]	= {
	{"file-go",		act_file_go},
	{"file-click",		act_file_click},
	{"file-suff",		act_file_suffixes},
	{"file-scroll",		act_file_scroll},
	{"file-arrow",		act_file_arrow},
	{"file-home",		act_file_home},
	{"file-end",		act_file_end},
};

#ifndef FILEINDENT
# define FILEINDENT	25
#endif

#ifndef DIR_SIZE_INCR
# define DIR_SIZE_INCR	64
#endif


/*
 *	Refresh list of files in the chosen directory.
 */

struct dirnode {
	struct dirnode	*next;
	_Xconst char	*name;
};

static	struct dirnode *
ll_sort(np)
	struct dirnode	*np;
{
	struct dirnode	*np1, *np2;
	struct dirnode	**npp;
	struct dirnode	*np0;

	if (np == NULL || (np2 = np->next) == NULL)
	    return np;

	np1 = np;
	for (;;) {
	    np2 = np2->next;
	    if (np2 == NULL) break;
	    np2 = np2->next;
	    if (np2 == NULL) break;
	    np1 = np1->next;
	}

	np2 = ll_sort(np1->next);
	np1->next = NULL;
	np1 = ll_sort(np);

	npp = &np0;
	for (;;)
	    if (strcmp(np1->name, np2->name) < 0) {
		*npp = np1;
		npp = &np1->next;
		np1 = np1->next;
		if (np1 == NULL) {
		    *npp = np2;
		    break;
		}
	    }
	    else {
		*npp = np2;
		npp = &np2->next;
		np2 = np2->next;
		if (np2 == NULL) {
		    *npp = np1;
		    break;
		}
	    }

	return np0;
}

static	Boolean
suffix_check(s, len)
	_Xconst char	*s;
	size_t		len;
{
	struct suffix *sp;

	if (suffixes == suff_end)
	    return True;

	for (sp = suffixes; sp < suff_end; ++sp)
	    if (len >= sp->len
	      && memcmp(s + len - sp->len, sp->str, sp->len) == 0)
		return True;

	return False;
}

static	void
file_refresh()
{
	struct dirnode	*head;
	struct dirnode	*np;
	struct dirnode	**npp;
	DIR		*dirp;
	size_t		count;
	Widget		prev;
	int		i;
	static Widget	*icon_list;
	static unsigned	wl_max		= 0;	/* allocated space in list */

	if (f3text != NULL)
	    XtVaSetValues(f3text, XtNstring, "", NULL);
	else
	    wl_num = 0;

	dirp = xopendir(dirbuf);
	npp = &head;
	count = 0;
	if (dirp != NULL) {
	    char	*dirbuf2;
	    size_t	dirlen2;
	    size_t	len;

	    dirbuf2 = xmemdup(dirbuf, dirlen);
	    dirlen2 = dirlen;
	    len = strlen(dirbuf2);
	    dirbuf2[len++] = '/';

	    for (;;) {
		struct dirent	*dp;
		struct stat	buf;
		size_t		len2;
		char		dirchar;
		char		*s;

		dp = readdir(dirp);
		if (dp == NULL) break;
		len2 = NAMLEN(dp);
		if (dp->d_name[0] == '.' && (len2 == 1
		  || (len2 == 2 && dp->d_name[1] == '.')))	/* if . or .. */
		    continue;
		if (dp->d_name[0] == filehide)	/* if hidden */
		    continue;

		if (dirlen2 <= len + len2) {
		    dirlen2 = ((len + len2) / DIR_SIZE_INCR + 1)
		      * DIR_SIZE_INCR;
		    dirbuf2 = xrealloc(dirbuf2, dirlen2);
		}
		s = dirbuf2 + len;
		memcpy(s, dp->d_name, len2);
		s[len2] = '\0';
		dirchar = '0';
		if (stat(dirbuf2, &buf) != 0 || !S_ISDIR(buf.st_mode)) {
		    dirchar = '1';
		    if (!suffix_check(s, len2))
			continue;
		}
		++len2;
		s = xmalloc(len2 + 1);
		*s = dirchar;
		memcpy(s + 1, dirbuf2 + len, len2);

		np = xmalloc(sizeof *np);
		np->name = s;
		*npp = np;
		npp = &np->next;
		++count;
	    }
	    closedir(dirp);
	}
	*npp = NULL;

	head = ll_sort(head);

	if (count > wl_max) {
	    if (wl_max == 0) {
		widget_list = xmalloc(count * sizeof(Widget));
		icon_list = xmalloc(count * sizeof(Widget));
	    }
	    else {
		widget_list = xrealloc(widget_list, count * sizeof(Widget));
		icon_list = xrealloc(icon_list, count * sizeof(Widget));
	    }
	    wl_max = count;
	}

	if (XtIsManaged(f2form)) {
	    XtUnmanageChild(f2form);
	    if (file_highlit != NULL) {
		XtCallActionProc(file_highlit, "reset", NULL, NULL, 0);
		file_highlit = NULL;
		file_hl_seq = -1;
	    }
	}

	if (head != NULL) {
	    if (filedir_pm == (Pixmap) 0)
		filedir_pm = XCreateBitmapFromData(DISP,
		  mane.win != (Window) 0 ? mane.win
		    : RootWindowOfScreen(SCRN),
		  (_Xconst char *) filedir_bits,
		  filedir_width, filedir_height);
	    if (*head->name != '0' && fileblank_pm == (Pixmap) 0)
		fileblank_pm = XCreateBitmapFromData(DISP,
		  mane.win != (Window) 0 ? mane.win
		    : RootWindowOfScreen(SCRN),
		  fileblank_bits, filedir_width, filedir_height);
	}

	i = 0;
	if (count > 0 && wl_num > 0) {
	    XtVaSetValues(*icon_list,
	      XtNbitmap, *head->name == '0' ? filedir_pm : fileblank_pm,
	      NULL);
	    for (;;) {
		XtVaSetValues(widget_list[i], XtNlabel, head->name + 1, NULL);
		np = head->next;
		free((char *) head->name);
		free(head);
		head = np;
		++i;
		if (i >= count || i >= wl_num)
		    break;
		if (*head->name == '0') {
		    if (icon_list[i] == NULL) {
			char	s[3 * sizeof(int) + 1];

#if HAVE_VSNPRINTF
			(void) snprintf(s, sizeof s, "i%u", i);
#else
			(void) sprintf(s2, "i%u", i);
#endif
			icon_list[i] = XtVaCreateManagedWidget(s,
			  labelWidgetClass, f2form,
			  XtNbitmap, filedir_pm,
			  XtNborderWidth, 0,
			  XtNfromVert, widget_list[i - 1],
			  XtNvertDistance, 0,
			  XtNleft, XawChainLeft,
			  XtNright, XawChainLeft,
			  XtNtop, XawChainTop,
			  XtNbottom, XawChainTop,
			  NULL);
		    }
		}
		else if (icon_list[i] != NULL) {
		    XtDestroyWidget(icon_list[i]);
		    icon_list[i] = NULL;
		}
	    }
	}
	if (head != NULL) {
	    if (i == 0) {
		Dimension	h, bw;

		prev = *icon_list = XtVaCreateManagedWidget("i0",
		  labelWidgetClass, f2form,
		  XtNbitmap, *head->name == '0' ? filedir_pm : fileblank_pm,
		  XtNborderWidth, 0,
		  XtNleft, XawChainLeft,
		  XtNright, XawChainLeft,
		  XtNtop, XawChainTop,
		  XtNbottom, XawChainTop,
		  NULL);
		prev = *widget_list = XtVaCreateManagedWidget("0",
		  commandWidgetClass, f2form,
		  XtNlabel, head->name + 1,
		  XtNtranslations, file_xlats,
		  XtNborderWidth, 0,
		  XtNresizable, True,
		  XtNfromHoriz, prev,
		  XtNleft, XawChainLeft,
		  XtNright, XawChainLeft,
		  XtNtop, XawChainTop,
		  XtNbottom, XawChainTop,
		  NULL);
		np = head->next;
		free((char *) head->name);
		free(head);
		head = np;
		++i;
		XtVaGetValues(prev,
		  XtNy, &f2ddist,
		  XtNheight, &h,
		  XtNborderWidth, &bw,
		  NULL);
		file_entry_height = h + 2 * bw;
	    }
	    else prev = widget_list[i - 1];
	    while (head != NULL) {
		char	s[3 * sizeof(int) + 1];

#if HAVE_VSNPRINTF
		(void) snprintf(s, sizeof s, "i%u", i);
#else
		(void) sprintf(s2, "i%u", i);
#endif
		icon_list[i] = *head->name != '0' ? NULL
		  : XtVaCreateManagedWidget(s, labelWidgetClass, f2form,
		  XtNbitmap, filedir_pm,
		  XtNborderWidth, 0,
		  XtNfromVert, prev,
		  XtNvertDistance, 0,
		  XtNleft, XawChainLeft,
		  XtNright, XawChainLeft,
		  XtNtop, XawChainTop,
		  XtNbottom, XawChainTop,
		  NULL);
		prev = widget_list[i] = XtVaCreateManagedWidget(s + 1,
		  commandWidgetClass, f2form,
		  XtNlabel, head->name + 1,
		  XtNtranslations, file_xlats,
		  XtNborderWidth, 0,
		  XtNresizable, True,
		  XtNfromHoriz, *icon_list,
		  XtNfromVert, prev,
		  XtNvertDistance, 0,
		  XtNleft, XawChainLeft,
		  XtNright, XawChainLeft,
		  XtNtop, XawChainTop,
		  XtNbottom, XawChainTop,
		  NULL);
		np = head->next;
		free((char *) head->name);
		free(head);
		head = np;
		++i;
	    }
	}

	while (wl_num > count) {
	    XtDestroyWidget(widget_list[--wl_num]);
	    if (icon_list[wl_num] != NULL)
		XtDestroyWidget(icon_list[wl_num]);
	}

	wl_num = count;

	XtManageChild(f2form);
}

/* ARGSUSED */
static	void
cb_file_up(w, client_data, call_data)
	Widget		w;
	XtPointer	client_data;
	XtPointer	call_data;
{
	char	*p;

	p = rindex(dirbuf, '/');
	if (p == NULL)
	    return;

	if (p == dirbuf) {
	    if (dirbuf[1] == '\0')
		return;
	    ++p;
	}

	*p = '\0';
	XtVaSetValues(f1text, XtNstring, dirbuf, NULL);
	file_refresh();
}

/* ARGSUSED */
static	void
cb_file_home(w, client_data, call_data)
	Widget		w;
	XtPointer	client_data;
	XtPointer	call_data;
{
	_Xconst char	*d;
	size_t		l;

	d = getenv("HOME");
	if (d == NULL)
	    return;

	l = strlen(d) + 1;
	if (l > dirlen) {
	    dirlen = ((l - 1) / DIR_SIZE_INCR + 1) * DIR_SIZE_INCR;
	    free(dirbuf);
	    dirbuf = xmalloc(dirlen);
	}

	memcpy(dirbuf, d, l);
	XtVaSetValues(f1text, XtNstring, dirbuf, NULL);
	file_refresh();
}

/* ARGSUSED */
static	void
cb_file_hidechk(w, client_data, call_data)
	Widget		w;
	XtPointer	client_data;
	XtPointer	call_data;
{
	if (filehide == '.') {
	    filehide = '\0';
	    XtVaSetValues(w, XtNbitmap, filechk_pm, NULL);
	}
	else {
	    filehide = '.';
	    XtVaSetValues(w, XtNbitmap, fileunchk_pm, NULL);
	}

	file_refresh();
}

static	Boolean
xdvi_get_cwd()
{
	if (dirbuf == NULL)
	    dirbuf = xmalloc(dirlen = DIR_SIZE_INCR);

	for (;;) {
	    if (getcwd(dirbuf, dirlen) != NULL)
		return True;
	    if (errno != ERANGE)
		break;
	    free(dirbuf);
	    dirbuf = xmalloc(dirlen += DIR_SIZE_INCR);
	}
	dirbuf[0] = '.';
	dirbuf[1] = '\0';
	return False;
}

static	Boolean
get_dir_name(str, len)
	_Xconst char	*str;
	size_t		len;
{
	int	fd;
	Boolean	val;

	if (dirbuf == NULL)
	    dirbuf = xmalloc(dirlen = DIR_SIZE_INCR);

	/* Normalize directory name */
	while (len > 0 && str[len - 1] == '/') --len;
	if (len == 0) {
	    dirbuf[0] = '/';
	    dirbuf[1] = '\0';
	    return True;
	}

	/* Save current directory */
	fd = xopen(".", O_RDONLY);
	if (fd == -1)
	    return False;

	/* Change to new directory */
	if (str[len] == '\0') {
	    if (chdir(str) != 0) {
		close(fd);
		return False;
	    }
	}
	else {
	    char *str2;

	    str2 = xmemdup(str, len + 1);
	    str2[len] = '\0';
	    if (chdir(str2) != 0) {
		free(str2);
		fchdir(fd);
		close(fd);
		return False;
	    }
	    free(str2);
	}

	val = xdvi_get_cwd();
	fchdir(fd);
	close(fd);
	return val;
}

static	Boolean
path_is_clean(str, p1)
	char	*str;
	char	*p1;
{
	char		*p_end	= p1 + strlen(p1);
	char		*p;
	char		c;
	struct stat	buf;
	int		val;

	for (;;) {
	    ++p1;
	    p = memchr(p1, '/', p_end - p1);
	    if (p == NULL)
		p = p_end;

	    /* check for double slash or slash at end or "." or ".." */
	    if (p == p1 || (*p1 == '.' && (p == p1 + 1 ||
	      (p1[1] == '.' && p == p1 + 2))))
		break;

	    c = *p;
	    *p = '\0';
	    val = lstat(str, &buf);
	    *p = c;

	    if (val != 0 || S_ISLNK(buf.st_mode))
		break;

	    p1 = p;
	    if (p1 == p_end)
		return True;
	}

	return False;
}

/* ARGSUSED */
static	void
cb_file_go(w, client_data, call_data)
	Widget		w;
	XtPointer	client_data;
	XtPointer	call_data;
{
	char		*s, *str;
	char		*p1;
	struct stat	buf;

	XtVaGetValues(f3text, XtNstring, &s, NULL);

	if (*s == '\0') {	/* if no string */
	    file_refresh();
	    return;
	}

	if (*s == '/')	/* if ignoring the chosen directory */
	    p1 = str = s;
	else {
	    size_t l1, l2;

	    l1 = strlen(dirbuf);
	    l2 = strlen(s) + 1;
	    str = xmalloc(l1 + l2 + 1);
	    memcpy(str, dirbuf, l1);
	    p1 = str + l1;
	    *p1 = '/';
	    memcpy(p1 + 1, s, l2);
	}

	if (stat(str, &buf) != 0) {	/* if file does not exist (or error) */
	    warning_popup_long("Error opening file: %s\n%s.", "OK", NULL, str,
	      strerror(errno));
	}
	else if (S_ISDIR(buf.st_mode)) {	/* if directory */
	    if (path_is_clean(str, p1) || !get_dir_name(str, strlen(str))) {
		size_t l;

		l = strlen(str) + 1;
		if (l > dirlen) {
		    dirlen = ((l - 1) / DIR_SIZE_INCR + 1) * DIR_SIZE_INCR;
		    free(dirbuf);
		    dirbuf = xmalloc(dirlen);
		}

		memcpy(dirbuf, str, l);
	    }
	    XtVaSetValues(f1text, XtNstring, dirbuf, NULL);
	    file_refresh();
	}
	else {		/* if not directory (file, we hope) */
	    FILE *f;

	    /* Normally, dvi files are are not part of the fd-counting
	       provided by xfopen(), since there is only one of them.
	       However, in this case we're opening one before closing the
	       old one, so we have to use xfopen() in this case.  */
	    f = xfopen(str, OPEN_MODE);
	    if (f == NULL)
		warning_popup_long("Error opening file: %s\n%s.", "OK", NULL,
		  str, strerror(errno));
	    else {
		ino_t	old_inode;

		++n_files_left;
		if (dvi_file != NULL)
		    fclose(dvi_file);
		dvi_file = f;

		if (dvi_name != NULL)
		    free(dvi_name);

		if (str != s) {
		    dvi_name = str;
		    str = s;
		}
		else
		    dvi_name = xstrdup(str);

		dvi_time = buf.st_mtime;

		old_inode = dvi_inode;
		form_dvi_property(buf.st_ino);
		titles_are_stale = True;

		if (mane.win != (Window) 0) {	/* if not in startup code */
		    if (dvi_inode != old_inode)
			set_dvi_property();

		    goto_page(0, home);
		}
		else
		    reposition = True;

		XtPopdown(file_shell);
		file_active = False;

		ev_flags |= EV_NEWDOC;
	    }
	}

	if (str != s)
	    free(str);
}

/* ARGSUSED */
static	void
act_file_go P4C(Widget, w, XEvent *, event, String *, params,
  Cardinal *, num_params)
{
	cb_file_go(NULL, NULL, NULL);
}

static	void
select_file_entry(w)
	Widget	w;
{
	char		*s;
	Position	fy;		/* (minus) position of list in vport */
	Dimension	ch;		/* height of clip widget */
	Position	ey;		/* position of entry */
	Dimension	eh, eb;		/* dimensions of entry */
	ptrdiff_t	dist;
	Widget		sbar;

	if (file_highlit != NULL && file_highlit != w) {
	    XtCallActionProc(file_highlit, "reset", NULL, NULL, 0);
	    file_hl_seq = -1;
	}
	file_highlit = w;

	XtVaGetValues(w, XtNlabel, &s, NULL);
	XtVaSetValues(f3text, XtNstring, s, NULL);
	XawTextSetInsertionPoint(f3text, strlen(s));

	/* Make sure the entry is fully viewable */
	XtVaGetValues(f2form, XtNy, &fy, NULL);
	XtVaGetValues(f2clip, XtNheight, &ch, NULL);
	XtVaGetValues(file_highlit,
	  XtNy, &ey,
	  XtNheight, &eh,
	  XtNborderWidth, &eb,
	  NULL);
	eh += 2 * eb;	/* real height of entry */
	dist = ey + fy;	/* position relative to viewable area */
	if (dist < 0) {
	    sbar = XtNameToWidget(f2vport, "vertical");
	    if (sbar != NULL)
		XtCallCallbacks(sbar, XtNscrollProc, (XtPointer) dist);
	}
	else {
	    dist += eh - ch;
	    if (dist > 0) {
		sbar = XtNameToWidget(f2vport, "vertical");
		if (sbar != NULL)
		    XtCallCallbacks(sbar, XtNscrollProc, (XtPointer) dist);
	    }
	}
}

/* ARGSUSED */
static	void
act_file_click P4C(Widget, w, XEvent *, event, String *, params,
  Cardinal *, num_params)
{
	select_file_entry(w);
}

static	void
select_by_seq(seq)
	int	seq;
{
	Widget	w	= widget_list[seq];

	select_file_entry(w);
	file_hl_seq = seq;
	XtCallActionProc(w, "highlight", NULL, NULL, 0);
}

/* ARGSUSED */
static	void
act_file_arrow P4C(Widget, w, XEvent *, event, String *, params,
  Cardinal *, num_params)
{
	Dimension	ch;		/* height of clip widget */
	Position	fy;		/* (minus) position of list in vport */
	Position	ey;		/* position of highlit entry */
	Dimension	eh, eb;
	int		hl_seq;

	if (wl_num == 0)
	    return;

	/* determine whether currently highlighted widget is visible */
	hl_seq = -1;
	XtVaGetValues(f2form, XtNy, &fy, NULL);
	if (file_highlit != NULL) {
	    XtVaGetValues(f2clip, XtNheight, &ch, NULL);
	    XtVaGetValues(f2form, XtNy, &fy, NULL);
	    XtVaGetValues(file_highlit,
	      XtNy, &ey,
	      XtNheight, &eh,
	      XtNborderWidth, &eb,
	      NULL);
	    eh += eb;	/* only one border is counted here */
	    ey += fy;	/* position relative to viewable area */
	    if (ey + (Position) eh > 0 && ey < (Position) ch) {
		/* Widget is visible.  Find its sequence number.  */
		hl_seq = file_hl_seq;
		if (hl_seq < 0) {
		    for (hl_seq = 0;; ++hl_seq) {
			if (hl_seq >= wl_num) {
			    /* this shouldn't happen */
			    file_highlit = NULL;
			    file_hl_seq = -1;
			    return;
			}
			if (widget_list[hl_seq] == file_highlit)
			    break;
		    }
		}
		if (*num_params > 0 && **params == '-') {	/* if up */
		    if (hl_seq == 0) hl_seq = wl_num;
		    --hl_seq;
		}
		else {	/* down */
		    ++hl_seq;
		    if (hl_seq >= wl_num) hl_seq = 0;
		}
	    }
	}

	if (hl_seq < 0) {	/* if not visible */
	    if (*num_params > 0 && **params == '-') {	/* if up */
		/* select the lowest visible widget */
		XtVaGetValues(f2clip, XtNheight, &ch, NULL);
		hl_seq = ((Position) ch - fy - f2ddist - 1) / file_entry_height;
	    }
	    else {	/* down */
		/* select the highest visible widget */
		hl_seq = (-fy - f2ddist) / file_entry_height;
	    }
	    if (hl_seq < 0) hl_seq = 0;
	    if (hl_seq >= wl_num) hl_seq = wl_num - 1;
	}

	/* Select the new widget.  */
	select_by_seq(hl_seq);
}

/* ARGSUSED */
static	void
act_file_home P4C(Widget, w, XEvent *, event, String *, params,
  Cardinal *, num_params)
{
	if (wl_num != 0)
	    select_by_seq(0);
}

/* ARGSUSED */
static	void
act_file_end P4C(Widget, w, XEvent *, event, String *, params,
  Cardinal *, num_params)
{
	if (wl_num != 0)
	    select_by_seq(wl_num - 1);
}

/*
 *	Page-Up, Page-Down, and wheel actions are pure scrolling actions:
 *	they do not change the highlighted entry, even if it scrolls off
 *	the window.
 */

/* ARGSUSED */
static	void
act_file_scroll P4C(Widget, w, XEvent *, event, String *, params,
  Cardinal *, num_params)
{
	Dimension	ch, fh;		/* clip and form (child) heights */
	Position	y;
	Widget		sbar;
	_Xconst char	*arg;
	ptrdiff_t	sign;
	ptrdiff_t	dist;

	if (wl_num == 0)
	    return;

	XtVaGetValues(f2clip, XtNheight, &ch, NULL);
	XtVaGetValues(f2form, XtNheight, &fh, XtNy, &y, NULL);
	if (fh <= ch)
	    return;	/* no scrolling possible */

	sbar = XtNameToWidget(f2vport, "vertical");
	if (sbar == NULL)
	    return;

	arg = (*num_params != 0 ? *params : "");

	sign = 1;
	if (*arg == '-') {
	    sign = -1;
	    ++arg;
	}

	if (*arg == 'w') {	/* if wheel action */
	    if (ch <= 16 * file_entry_height)
		dist = (ch / (4 * file_entry_height) + 1) * file_entry_height;
	    else
		dist = 5 * file_entry_height;
	}
	else {
	    if (ch <= 15 * file_entry_height)
		dist = ch / 5;
	    else
		dist = 3 * file_entry_height;
	    dist = ch - dist;
	    if (dist < file_entry_height) dist = file_entry_height;
	}

	XtCallCallbacks(sbar, XtNscrollProc, (XtPointer) (sign * dist));
}

static	void
rebuild_suffixes(s)
	_Xconst char	*s;
{
	struct suffix	*sp;
	size_t		n;
	_Xconst char	*p, *p1;
	char		*q;

	/* First free the old suffixes */
	for (sp = suffixes; sp < suff_end; ++sp)
	    free((char *) sp->str);

	/* Next count the suffixes */
	n = 0;
	p = s;
	for (;;) {
	    while (*p == ' ' || *p == '\t' || *p == '\n' || *p == ',') ++p;
	    if (*p == '\0')
		break;
	    ++n;
	    for (;;) {
		if (*p == '\\' && p[1] != '\0') ++p;
		else if (*p == ' ' || *p == '\t' || *p == '\n' || *p == ',')
		    break;
		++p;
		if (*p == '\0') break;
	    }
	}

	/* Allocate space */
	if (n > suff_max) {
	    if (suff_max != 0) free(suffixes);
	    suff_max = n;
	    suffixes = xmalloc(n * sizeof (*suffixes));
	}

	/* Fill the space */
	suff_end = suffixes;
	p = s;
	for (;;) {
	    while (*p == ' ' || *p == '\t' || *p == '\n' || *p == ',') ++p;
	    if (*p == '\0')
		break;
	    n = 0;
	    p1 = p;
	    for (;;) {
		if (*p1 == '\\' && p1[1] != '\0') ++p1;
		else if (*p1 == ' ' || *p1 == '\t' || *p1 == '\n' || *p1 == ',')
		    break;
		++n;
		++p1;
		if (*p1 == '\0') break;
	    }
	    suff_end->len = n;
	    suff_end->str = q = xmalloc(n);
	    ++suff_end;

	    while (p < p1) {
		if (*p == '\\') ++p;
		*q++ = *p++;
	    }
	}
}

/* ARGSUSED */
static	void
act_file_suffixes P4C(Widget, w, XEvent *, event, String *, params,
  Cardinal *, num_params)
{
	char	*s;

	XtVaGetValues(f4text, XtNstring, &s, NULL);
	rebuild_suffixes(s);
	file_refresh();
}

static	void
right_justify_widget(w, form_width)
	Widget		w;
	Dimension	form_width;
{
	Dimension	w1, w2, bw1;
	Position	x;

	XtVaGetValues(w,
	  XtNx, &x,
	  XtNwidth, &w1,
	  XtNborderWidth, &bw1,
	  NULL);
	w2 = form_width - x - 2 * bw1;
	if (w2 > w1)
	    XtVaSetValues(w, XtNwidth, w2, NULL);
}

/* ARGSUSED */
void
Act_open_dvi_file P4C(Widget, w, XEvent *, event, String *, params,
  Cardinal *, num_params)
{
	if (file_active) {
	    XRaiseWindow(DISP, XtWindow(file_shell));
	    return;
	}

	if (file_shell == NULL) {
	    Widget		form;
	    Widget		f1label, f1cmd1, f1cmd2;
	    Widget		f3label;
	    Widget		f4label;
	    Widget		f5chk, f5label;
	    Widget		f6ok, f6cancel;
	    static XtTranslations form_xlats, form_xlats2;
	    XtTranslations	xlats;
	    int			ddist;
	    char		*p;

	    XtAddActions(file_actions, XtNumber(file_actions));
	    form_xlats = XtParseTranslationTable(
#ifdef XK_KP_Left
	      "#override \
<Key>Up:file-arrow(-)\n\
<Key>Down:file-arrow()\n\
<Key>Prior:file-scroll(-)\n\
<Key>Next:file-scroll()\n\
<Key>Home:file-home()\n\
<Key>End:file-end()\n\
<Key>KP_Up:file-arrow(-)\n\
<Key>KP_Down:file-arrow()\n\
<Key>KP_Prior:file-scroll(-)\n\
<Key>KP_Next:file-scroll()\n\
<Key>KP_Home:file-home()\n\
<Key>KP_End:file-end()\n\
<Btn4Down>:file-scroll(-w)\n\
<Btn5Down>:file-scroll(w)"
#else /* no XK_KP_Left */
	      "#override \
<Key>Up:file-arrow(-)\n\
<Key>Down:file-arrow()\n\
<Key>Prior:file-scroll(-)\n\
<Key>Next:file-scroll()\n\
<Key>Home:file-home()\n\
<Key>End:file-end()\n\
<Btn4Down>:file-scroll(-w)\n\
<Btn5Down>:file-scroll(w)"
#endif /* no XK_KP_Left */
	      );
	    form_xlats2 = XtParseTranslationTable(
	      "#override \
<Btn4Down>:file-scroll(-w)\n\
<Btn5Down>:file-scroll(w)");
	    file_xlats = XtParseTranslationTable(
	      "#replace \
<Btn1Down>:highlight()file-click()\n\
<Btn1Down>(2):file-go()\n\
<Btn4Down>:file-scroll(-w)\n\
<Btn5Down>:file-scroll(w)");

	    /* Set initial directory name.  */
	    if (dvi_name != NULL) {
		p = rindex(dvi_name, '/');
		if (p == NULL || !get_dir_name(dvi_name, p - dvi_name))
		    (void) xdvi_get_cwd();
	    }
	    else
		(void) xdvi_get_cwd();

	    /* Create shell widget and its child form widget.  */

	    file_shell = XtVaCreatePopupShell("file", transientShellWidgetClass,
	      top_level,
	      XtNtitle, "Open file",
	      XtNmappedWhenManaged, False,
	      XtNtransientFor, top_level,
	      XtNallowShellResize, True,
	      NULL);

	    form = XtVaCreateManagedWidget("form", formWidgetClass, file_shell,
	      XtNtranslations, form_xlats,
	      NULL);
	    XtVaGetValues(form, XtNdefaultDistance, &ddist, NULL);

	    /* Row 1:  Directory stuff.  */

	    f1label = XtVaCreateManagedWidget("lookin", labelWidgetClass, form,
	      XtNlabel, "Look in:",
	      XtNborderWidth, 0,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainLeft,
	      NULL);

	    f1text = XtVaCreateManagedWidget("dir", asciiTextWidgetClass,
	      form,
	      XtNdataCompression, False,
	      XtNuseStringInPlace, True,
	      XtNstring, dirbuf,
	      XtNlength, dirlen,
	      XtNwidth, 250,
	      XtNtranslations, form_xlats,
	      XtNresizable, True,
	      XtNfromHoriz, f1label,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainRight,
	      NULL);

	    f1cmd1 = XtVaCreateManagedWidget("up", commandWidgetClass,
	      form,
	      XtNlabel, "Up directory",
	      XtNtranslations, form_xlats2,
	      XtNfromVert, f1text,
	      XtNhorizDistance, ddist + FILEINDENT,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainLeft,
	      NULL);
	    XtAddCallback(f1cmd1, XtNcallback, cb_file_up, NULL);

	    f1cmd2 = XtVaCreateManagedWidget("home", commandWidgetClass,
	      form,
	      XtNlabel, "Home directory",
	      XtNtranslations, form_xlats2,
	      XtNfromHoriz, f1cmd1,
	      XtNfromVert, f1text,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainLeft,
	      NULL);
	    XtAddCallback(f1cmd2, XtNcallback, cb_file_home, NULL);

	    /* Row 2:  The viewport.  */

	    f2vport = XtVaCreateManagedWidget("vport", viewportWidgetClass,
	      form,
	      XtNwidth, 300,
	      XtNheight, 200,
	      XtNallowHoriz, True,
	      XtNallowVert, True,
	      XtNuseRight, True,
	      XtNuseBottom, True,
	      XtNfromVert, f1cmd1,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainRight,
	      NULL);

	    f2clip = XtNameToWidget(f2vport, "clip");

	    f2form = XtCreateWidget("vform", formWidgetClass, f2vport,
	      NULL, 0);

	    rebuild_suffixes(file_save_suff);
	    file_refresh();

	    /* Rows 3 and 4:  File name and suffixes.  */

	    f3label = XtVaCreateManagedWidget("filename", labelWidgetClass,
	      form,
	      XtNlabel, "File name:",
	      XtNborderWidth, 0,
	      XtNfromVert, f2vport,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainLeft,
	      NULL);

	    f3text = XtVaCreateManagedWidget("filetext", asciiTextWidgetClass,
	      form,
	      XtNdataCompression, False,
	      XtNeditType, XawtextEdit,
	      XtNtranslations, form_xlats,
	      XtNresizable, True,
	      XtNfromHoriz, f3label,
	      XtNfromVert, f2vport,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainRight,
	      NULL);
	    xlats = XtParseTranslationTable("<Key>Return:file-go()");
	    XtOverrideTranslations(f3text, xlats);

	    f4label = XtVaCreateManagedWidget("sufflabel", labelWidgetClass,
	      form,
	      XtNlabel, "Suffixes:",
	      XtNborderWidth, 0,
	      XtNfromVert, f3text,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainLeft,
	      NULL);

	    f4text = XtVaCreateManagedWidget("sufftext", asciiTextWidgetClass,
	      form,
	      XtNdataCompression, False,
	      XtNeditType, XawtextEdit,
	      XtNstring, file_save_suff,
	      XtNtranslations, form_xlats,
	      XtNresizable, True,
	      XtNfromHoriz, f4label,
	      XtNfromVert, f3text,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainRight,
	      NULL);
	    XawTextSetInsertionPoint(f4text, strlen(file_save_suff));
	    xlats = XtParseTranslationTable("<Key>Return:file-suff()");
	    XtOverrideTranslations(f4text, xlats);

	    {	/* align left edges of asciitext widgets */
		Dimension w3, w4;

		XtVaGetValues(f3label, XtNwidth, &w3, NULL);
		XtVaGetValues(f4label, XtNwidth, &w4, NULL);
		if (w4 > w3)
		    XtVaSetValues(f3text, XtNhorizDistance, ddist + (w4 - w3),
		      NULL);
		else if (w3 > w4)
		    XtVaSetValues(f4text, XtNhorizDistance, ddist + (w3 - w4),
		      NULL);
	    }

	    /* Row 5:  Checkbox for showing hidden files.  */

	    filechk_pm = XCreateBitmapFromData(DISP,
	      mane.win != (Window) 0 ? mane.win : RootWindowOfScreen(SCRN),
	      (_Xconst char *) filechk_bits, filechk_width, filechk_height);

	    fileunchk_pm = XCreateBitmapFromData(DISP,
	      mane.win != (Window) 0 ? mane.win : RootWindowOfScreen(SCRN),
	      fileunchk_bits, filechk_width, filechk_height);

	    f5chk = XtVaCreateManagedWidget("hidchk", commandWidgetClass, form,
	      XtNbitmap, filehide == '.' ? fileunchk_pm : filechk_pm,
	      XtNtranslations, form_xlats2,
	      XtNfromVert, f4text,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainLeft,
	      NULL);
	    XtAddCallback(f5chk, XtNcallback, cb_file_hidechk, NULL);

	    f5label = XtVaCreateManagedWidget("hidlabel", labelWidgetClass,
	      form,
	      XtNlabel, "Show hidden files and directories",
	      XtNborderWidth, 0,
	      XtNfromHoriz, f5chk,
	      XtNfromVert, f4text,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainLeft,
	      NULL);

	    /* Row 6:  Action buttons.  */

	    f6ok = XtVaCreateManagedWidget("ok", commandWidgetClass, form,
	      XtNlabel, "Open",
	      XtNtranslations, form_xlats2,
	      XtNaccelerators, accels_cr,
	      XtNfromVert, f5chk,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainLeft,
	      NULL);
	    XtAddCallback(f6ok, XtNcallback, cb_file_go, NULL);
	    XtInstallAccelerators(form, f6ok);
	    XtInstallAccelerators(f3text, f6ok);

	    f6cancel = XtVaCreateManagedWidget("cancel", commandWidgetClass,
	      form,
	      XtNlabel, "Cancel",
	      XtNtranslations, form_xlats2,
	      XtNfromHoriz, f6ok,
	      XtNfromVert, f5chk,
	      XtNleft, XawChainRight,
	      XtNright, XawChainRight,
	      NULL);
	    XtAddCallback(f6cancel, XtNcallback, cb_file_cancel, NULL);

	    /* Realize and set destroy callback */
	    XdviXawRealizePopup(file_shell, cb_file_cancel);

	    {    /* Move various things to the right margin */
		Dimension	fw, w1, w2, bw1, bw2, tw;

		/* get the form width */
		XtVaGetValues(form, XtNwidth, &fw, NULL);
		fw -= ddist;

		/* stretch various widgets to the right */
		right_justify_widget(f1text, fw);
		XtVaSetValues(f2vport, XtNresizable, True, NULL);
		right_justify_widget(f2vport, fw);
		XtVaSetValues(f2vport, XtNresizable, False, NULL);
		right_justify_widget(f3text, fw);
		right_justify_widget(f4text, fw);

		/* move the cancel button over */
		XtVaGetValues(f6ok, XtNwidth, &w1, XtNborderWidth, &bw1, NULL);
		XtVaGetValues(f6cancel, XtNwidth, &w2, XtNborderWidth, &bw2,
		  NULL);
		tw = w1 + w2 + 2 * (bw1 + bw2 + ddist);
		if (tw < fw)
		    XtVaSetValues(f6cancel, XtNhorizDistance, fw - tw + ddist,
		      NULL);
	    }

	    reposition = True;
	}
	else {	/* if widget already exists */
	    file_refresh();
	}

	if (reposition) {
	    /* Center the popup over the main window */
	    Position	x, y;
	    Dimension	w1, h1, w2, h2;

	    /* Get the size and location of the main window */
	    x = y = 0;
	    w1 = WidthOfScreen(SCRN);
	    h1 = HeightOfScreen(SCRN);
	    if (mane.win != (Window) 0)
		XtVaGetValues(top_level, XtNx, &x, XtNy, &y,
		  XtNwidth, &w1, XtNheight, &h1, NULL);

	    /* Get the size of the popup window */
	    XtVaGetValues(file_shell, XtNwidth, &w2, XtNheight, &h2, NULL);

	    /* Move the popup */
	    XtVaSetValues(file_shell, XtNx, x + (w1 - w2) / 2,
	      XtNy, y + (h1 - h2) / 2, NULL);

	    reposition = False;
	}

	XtPopup(file_shell, XtGrabNone);
	file_active = True;
}

#else /* MOTIF */

/* ARGSUSED */
static	void
cb_file_go(w, client_data, call_data)
	Widget		w;
	XtPointer	client_data;
	XtPointer	call_data;
{
	char		*str;
	XmFileSelectionBoxCallbackStruct *cbs;
	struct stat	buf;

	cbs = (XmFileSelectionBoxCallbackStruct *) call_data;
	if (!XmStringGetLtoR(cbs->value, XmFONTLIST_DEFAULT_TAG, &str))
	    return;	/* shouldn't happen */

	if (stat(str, &buf) != 0) {	/* if file does not exist (or error) */
	    warning_popup_long("Error opening file: %s\n%s.", XmDIALOG_ERROR,
	      "OK", NULL, str, strerror(errno));
	}
	else {
	    FILE *f;

	    /* Normally, dvi files are are not part of the fd-counting
	       provided by xfopen(), since there is only one of them.
	       However, in this case we're opening one before closing the
	       old one, so we have to use xfopen() in this case.  */
	    f = xfopen(str, OPEN_MODE);
	    if (f == NULL)
		warning_popup_long("Error opening file: %s\n%s.",
		  XmDIALOG_ERROR, "OK", NULL, str, strerror(errno));
	    else {
		ino_t	old_inode;

		++n_files_left;
		if (dvi_file != NULL)
		    fclose(dvi_file);
		dvi_file = f;

		if (dvi_name != NULL)
		    free(dvi_name);

		dvi_name = xstrdup(str);
		dvi_time = buf.st_mtime;

		old_inode = dvi_inode;
		form_dvi_property(buf.st_ino);
		titles_are_stale = True;

		if (mane.win != (Window) 0) {	/* if not in startup code */
		    if (dvi_inode != old_inode)
			set_dvi_property();

		    goto_page(0, home);
		}
		else
		    reposition = True;

		XtPopdown(file_shell);
		file_active = False;

		ev_flags |= EV_NEWDOC;
	    }
	}

	XtFree(str);
}

/* ARGSUSED */
void
Act_open_dvi_file P4C(Widget, w, XEvent *, event, String *, params,
  Cardinal *, num_params)
{
	Widget		dialog;

	if (file_active) {
	    XRaiseWindow(DISP, XtWindow(file_shell));
	    return;
	}

	if (file_shell == NULL) {
	    static Arg	cf_args[] = {
		{XmNdirectory,	(XtArgVal) NULL},
		{XmNpattern,	(XtArgVal) NULL},
	    };

	    /* Set initial directory name.  */
	    if (dvi_name != NULL) {
		char	*p;

		p = rindex(dvi_name, '/');
		if (p != NULL) {
		    char c;

		    for (;; --p) {
			if (p <= dvi_name) {
			    ++p;
			    break;
			}
			if (p[-1] != '/')
			    break;
		    }
		    c = *p;
		    *p = '\0';
		    cf_args[0].value =
		      (XtArgVal) XmStringCreateLocalized(dvi_name);
		    *p = c;
		}
	    }

	    cf_args[1].value = (XtArgVal) XmStringCreateLocalized("*.dvi");

	    dialog = XmCreateFileSelectionDialog(top_level, "file",
	      cf_args, XtNumber(cf_args));

	    if (cf_args[0].value != (XtArgVal) NULL) {
		XmStringFree((XmString) cf_args[0].value);
		cf_args[0].value = (XtArgVal) NULL;
	    }
	    XmStringFree((XmString) cf_args[1].value);

	    XtAddCallback(dialog, XmNokCallback, cb_file_go, NULL);
	    XtAddCallback(dialog, XmNcancelCallback, cb_file_cancel, NULL);
	    XtUnmanageChild(
	      XmFileSelectionBoxGetChild(dialog, XmDIALOG_HELP_BUTTON));

	    file_shell = XtParent(dialog);
	    XmAddWMProtocolCallback(file_shell, XA_WM_DELETE_WINDOW,
	      cb_file_cancel, NULL);
	    XtVaSetValues(file_shell, XmNtitle, "Xdvi: Open Dvi File", NULL);

	    XtManageChild(dialog);
	}
	else if (reposition) {
	    /* Center the popup over the main window */
	    Position	x, y;
	    Dimension	w1, h1, w2, h2;

	    /* Get the size and location of the main window */
	    x = y = 0;
	    XtVaGetValues(top_level, XtNx, &x, XtNy, &y,
	      XtNwidth, &w1, XtNheight, &h1, NULL);

	    /* Get the size of the popup window */
	    XtVaGetValues(file_shell, XtNwidth, &w2, XtNheight, &h2, NULL);

	    /* Move the popup */
	    XtVaSetValues(file_shell, XtNx, x + (w1 - w2) / 2,
	      XtNy, y + (h1 - h2) / 2, NULL);

	    reposition = False;
	}

	XtPopup(file_shell, XtGrabNone);
}

#endif


/*
 *	Print popup - pop up a window to allow the user to print page(s)
 *	from the dvi file.
 */

static	Boolean	print_active	= False;	/* if print window is showing */
static	Widget	print_shell	= NULL;		/* shell widget of popup */
static	Widget	r1radio1, r1radio2;
static	Widget	r2label, r2text;
static	Widget	r3label, r3text;
static	Widget	r4text;
static	Widget	r6radio, r7radio;
static	Widget	r7text1, r7text2;
static	Widget	r8text;
static	Widget	r9prev;

/* &r1radio1 = printer, &r1radio2 = to file */
static	Widget	*pr_current_fileradio	= &r1radio1;

/* &r6radio = whole file, &r7radio = page range */
static	Widget	*pr_current_pageradio	= &r6radio;

/* Stuff to save between popups */

static	Widget		*pr_save_fileradio = &r1radio1;
static	Widget		*pr_save_pageradio = &r6radio;
static	_Xconst char	*pr_save_cmd	= NULL;
static	_Xconst char	*pr_save_file	= NULL;
static	_Xconst char	*pr_save_printer= NULL;
static	_Xconst char	*pr_save_xargs	= NULL;

/* Things to do with running the dvips process */

static	Boolean	printlog_active	= False;	/* if the print log is active */
static	Widget	printlog_shell	= NULL;
static	Widget	printlog_keep;

/* Forward reference */

static	void	print_precheck();


static void
set_sensitivity(fileradio)
	Widget	*fileradio;
{
	Boolean	sensitivity;

	if (fileradio == pr_current_fileradio) /* if no change */
	    return;

	pr_current_fileradio = fileradio;

	sensitivity = (fileradio == &r1radio1);
	XtSetSensitive(r2label, sensitivity);
	XtSetSensitive(r2text, sensitivity);

	sensitivity ^= (True ^ False);
	XtSetSensitive(r3label, sensitivity);
	XtSetSensitive(r3text, sensitivity);
}

/* ARGSUSED */
static	void
cb_print_cancel(w, client_data, call_data)
	Widget		w;
	XtPointer	client_data;
	XtPointer	call_data;
{
	XtPopdown(print_shell);
	print_active = False;

	if (w == r9prev && !printlog_active) {
	    XtSetSensitive(printlog_keep, False);
	    XtPopup(printlog_shell, XtGrabNone);
	    printlog_active = True;
	}
}

/* ARGSUSED */
static	void
cb_print(w, client_data, call_data)
	Widget		w;
	XtPointer	client_data;
	XtPointer	call_data;
{
	print_precheck();
}

#if XAW
static void print_act_go ARGS((Widget, XEvent *, String *, Cardinal *));
#endif
static void printlog_act_close ARGS((Widget, XEvent *, String *, Cardinal *));
static void printlog_act_keep ARGS((Widget, XEvent *, String *, Cardinal *));
static void printlog_act_unkeep ARGS((Widget, XEvent *, String *, Cardinal *));
static void printlog_act_cancel ARGS((Widget, XEvent *, String *, Cardinal *));

static	XtActionsRec	print_actions[]	= {
#if XAW
	{"printInternal",	print_act_go},
#endif
	{"printlogIntClose",	printlog_act_close},
	{"printlogIntKeep",	printlog_act_keep},
	{"printlogIntUnkeep",	printlog_act_unkeep},
	{"printlogIntCancel",	printlog_act_cancel},
};

#if XAW

#ifndef PRINTINDENT
# define PRINTINDENT	25
#endif

/* ARGSUSED */
static	void
cb_print_vs_file(w, client_data, call_data)
	Widget		w;
	XtPointer	client_data;
	XtPointer	call_data;
{
	Widget	*fileradio	= XawToggleGetCurrent(r1radio1);

	if (fileradio != NULL)
	    set_sensitivity(fileradio);
}

/* ARGSUSED */
static	void
cb_range(w, client_data, call_data)
	Widget		w;
	XtPointer	client_data;
	XtPointer	call_data;
{
	Widget	*pageradio	= XawToggleGetCurrent(r6radio);

	if (pageradio != NULL)
	    pr_current_pageradio = pageradio;
}

/* ARGSUSED */
void
range_handle_key(widget, closure, ev, cont)
	Widget		widget;
	XtPointer	closure;
	XEvent		*ev;
	Boolean		*cont;
{
	if (pr_current_pageradio != &r7radio)
	    XawToggleSetCurrent(r6radio, &r7radio);
}

/* ARGSUSED */
static	void
print_act_go P4C(Widget, w, XEvent *, event, String *, params,
  Cardinal *, num_params)
{
	print_precheck();
}

/* ARGSUSED */
void
Act_print P4C(Widget, w, XEvent *, event, String *, params,
  Cardinal *, num_params)
{
	static	Widget	r7label3;
	static	Widget	r9cancel;
	static	Dimension sw;	/* shell width */
	static	int	canceladjust;
	char		*ofstring;

	if (print_active) {
	    XRaiseWindow(DISP, XtWindow(print_shell));
	    return;
	}

	if (printlog_active) {
	    XRaiseWindow(DISP, XtWindow(printlog_shell));
	    return;
	}

	if (dvi_file == NULL) {
	    WARN(XmDIALOG_ERROR, "No active dvi file");
	    return;
	}

	ofstring = (pageno_correct == 1 ? mprintf("of %d", total_pages)
	  : mprintf("of %d to %d", pageno_correct,
	    total_pages + pageno_correct - 1));

	if (print_shell == NULL) {
	    Widget		form;
	    Widget		r1label;
	    Widget		r4label;
	    Widget		r5label;
	    Widget		r6form;
	    Widget		r7form, r7label1, r7label2;
	    Widget		r8label;
	    Widget		r9ok;
	    XtTranslations	xlats, xlats2;
	    XtAccelerators	accels2;
	    int			ddist;

	    XtAddActions(print_actions, XtNumber(print_actions));
	    print_shell = XtVaCreatePopupShell("print",
	      transientShellWidgetClass,
	      top_level,
	      XtNtitle, "Print dvi file",
	      XtNmappedWhenManaged, False,
	      XtNtransientFor, top_level,
	      XtNallowShellResize, True,
	      NULL);

	    form = XtCreateManagedWidget("form", formWidgetClass, print_shell,
	      NULL, 0);
	    XtVaGetValues(form, XtNdefaultDistance, &ddist, NULL);

	    r1label = XtVaCreateManagedWidget("printto", labelWidgetClass, form,
	      XtNlabel, "Print to:",
	      XtNborderWidth, 0,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainLeft,
	      NULL);

	    xlats = XtParseTranslationTable("<EnterWindow>:highlight(Always)\n\
<LeaveWindow>:unhighlight()\n\
<Btn1Down>,<Btn1Up>:set()notify()");
	    r1radio1 = XtVaCreateManagedWidget("toprinter", toggleWidgetClass,
	      form,
	      XtNlabel, "Printer",
	      XtNradioData, &r1radio1,
	      XtNstate, True,
	      XtNtranslations, xlats,
	      XtNfromHoriz, r1label,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainLeft,
	      NULL);
	    XtAddCallback(r1radio1, XtNcallback, cb_print_vs_file, NULL);

	    r1radio2 = XtVaCreateManagedWidget("tofile", toggleWidgetClass,
	      form,
	      XtNlabel, "File",
	      XtNradioGroup, r1radio1,
	      XtNradioData, &r1radio2,
	      XtNtranslations, xlats,
	      XtNfromHoriz, r1radio1,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainLeft,
	      NULL);
	    XtAddCallback(r1radio2, XtNcallback, cb_print_vs_file, NULL);

	    /* Rows 2, 3:  Print command or file name */

	    r2label = XtVaCreateManagedWidget("prncommand", labelWidgetClass,
	      form,
	      XtNlabel, "Print command:",
	      XtNborderWidth, 0,
	      XtNfromVert, r1radio1,
	      XtNhorizDistance, ddist + PRINTINDENT,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainLeft,
	      NULL);

	    r2text = XtVaCreateManagedWidget("prntext", asciiTextWidgetClass,
	      form,
	      XtNdataCompression, False,
	      XtNeditType, XawtextEdit,
	      XtNfromHoriz, r2label,
	      XtNfromVert, r1radio1,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainLeft,
	      NULL);
	    xlats2 = XtParseTranslationTable("<Key>Return:printInternal()");
	    XtOverrideTranslations(r2text, xlats2);

	    r3label = XtVaCreateManagedWidget("filename", labelWidgetClass,
	      form,
	      XtNlabel, "File name:",
	      XtNsensitive, False,
	      XtNborderWidth, 0,
	      XtNfromVert, r2text,
	      XtNhorizDistance, ddist + PRINTINDENT,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainLeft,
	      NULL);

	    r3text = XtVaCreateManagedWidget("filetext", asciiTextWidgetClass,
	      form,
	      XtNdataCompression, False,
	      XtNeditType, XawtextEdit,
	      XtNsensitive, False,
	      XtNfromHoriz, r3label,
	      XtNfromVert, r2text,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainLeft,
	      NULL);
	    XtOverrideTranslations(r3text, xlats2);

	    {	/* align left edges of asciitext widgets */
		Dimension w2, w3;

		XtVaGetValues(r2label, XtNwidth, &w2, NULL);
		XtVaGetValues(r3label, XtNwidth, &w3, NULL);
		if (w3 > w2)
		    XtVaSetValues(r2text, XtNhorizDistance, ddist + (w3 - w2),
		      NULL);
		else if (w2 > w3)
		    XtVaSetValues(r3text, XtNhorizDistance, ddist + (w2 - w3),
		      NULL);
	    }

	    /* Row 4:  printer name (for dvips) */

	    r4label = XtVaCreateManagedWidget("prname", labelWidgetClass,
	      form,
	      XtNlabel, "Printer name (used by dvips):",
	      XtNborderWidth, 0,
	      XtNfromVert, r3text,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainLeft,
	      NULL);

	    r4text = XtVaCreateManagedWidget("prtext", asciiTextWidgetClass,
	      form,
	      XtNdataCompression, False,
	      XtNeditType, XawtextEdit,
	      XtNwidth, 50,
	      XtNfromHoriz, r4label,
	      XtNfromVert, r3text,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainLeft,
	      NULL);
	    XtOverrideTranslations(r4text, xlats2);

	    /* Rows 5-7:  page selection */

	    r5label = XtVaCreateManagedWidget("rangelab", labelWidgetClass,
	      form,
	      XtNlabel, "Print range:",
	      XtNborderWidth, 0,
	      XtNfromVert, r4text,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainLeft,
	      NULL);

	    r6form = XtVaCreateManagedWidget("rangeallform", formWidgetClass,
	      form,
	      XtNdefaultDistance, 0,
	      XtNborderWidth, 0,
	      XtNfromVert, r5label,
	      XtNhorizDistance, ddist + PRINTINDENT,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainLeft,
	      NULL);

	    accels2 = XtParseAcceleratorTable(
	      "<Btn1Down>,<Btn1Up>:set()notify()");
	    r6radio = XtVaCreateManagedWidget("rangeall", toggleWidgetClass,
	      r6form,
	      XtNlabel, " ",
	      XtNradioData, &r6radio,
	      XtNstate, True,
	      XtNtranslations, xlats,
	      XtNaccelerators, accels2,
	      NULL);
	    XtAddCallback(r6radio, XtNcallback, cb_range, NULL);
	    XtInstallAccelerators(r6form, r6radio);

	    (void) XtVaCreateManagedWidget("rangealllab", labelWidgetClass,
	      r6form,
	      XtNlabel, "All",
	      XtNborderWidth, 0,
	      XtNfromHoriz, r6radio,
	      XtNhorizDistance, ddist,
	      NULL);

	    r7form = XtVaCreateManagedWidget("rangefromtoform", formWidgetClass,
	      form,
	      XtNdefaultDistance, 0,
	      XtNborderWidth, 0,
	      XtNresizable, True,
	      XtNfromVert, r6form,
	      XtNhorizDistance, ddist + PRINTINDENT,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainLeft,
	      NULL);

	    r7radio = XtVaCreateManagedWidget("rangefromto", toggleWidgetClass,
	      r7form,
	      XtNlabel, " ",
	      XtNradioGroup, r6radio,
	      XtNradioData, &r7radio,
	      XtNtranslations, xlats,
	      XtNaccelerators, accels2,
	      NULL);
	    XtAddCallback(r7radio, XtNcallback, cb_range, NULL);
	    XtInstallAccelerators(r7form, r7radio);

	    r7label1 = XtVaCreateManagedWidget("rangefromlab", labelWidgetClass,
	      r7form,
	      XtNlabel, "From",
	      XtNborderWidth, 0,
	      XtNfromHoriz, r7radio,
	      XtNhorizDistance, ddist,
	      NULL);

	    r7text1 = XtVaCreateManagedWidget("rangefrom", asciiTextWidgetClass,
	      r7form,
	      XtNdataCompression, False,
	      XtNeditType, XawtextEdit,
	      XtNwidth, 50,
	      XtNfromHoriz, r7label1,
	      XtNhorizDistance, ddist,
	      NULL);
	    XtOverrideTranslations(r7text1, xlats2);
	    XtAddEventHandler(r7text1, KeyPressMask, False,
	      range_handle_key, (XtPointer) NULL);

	    r7label2 = XtVaCreateManagedWidget("rangetolab", labelWidgetClass,
	      r7form,
	      XtNlabel, "to",
	      XtNborderWidth, 0,
	      XtNfromHoriz, r7text1,
	      XtNhorizDistance, ddist,
	      NULL);

	    r7text2 = XtVaCreateManagedWidget("rangeto", asciiTextWidgetClass,
	      r7form,
	      XtNdataCompression, False,
	      XtNeditType, XawtextEdit,
	      XtNwidth, 50,
	      XtNfromHoriz, r7label2,
	      XtNhorizDistance, ddist,
	      NULL);
	    XtOverrideTranslations(r7text2, xlats2);
	    XtAddEventHandler(r7text2, KeyPressMask, False,
	      range_handle_key, (XtPointer) NULL);

	    r7label3 = XtVaCreateManagedWidget("rangeof", labelWidgetClass,
	      r7form,
	      XtNlabel, ofstring,
	      XtNborderWidth, 0,
	      XtNresizable, True,
	      XtNfromHoriz, r7text2,
	      XtNhorizDistance, ddist,
	      NULL);

	    /* Row 8:  additional dvips args */

	    r8label = XtVaCreateManagedWidget("xargsname", labelWidgetClass,
	      form,
	      XtNlabel, "Additional dvips arguments (optional):",
	      XtNborderWidth, 0,
	      XtNfromVert, r7form,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainLeft,
	      NULL);

	    r8text = XtVaCreateManagedWidget("xargstext", asciiTextWidgetClass,
	      form,
	      XtNdataCompression, False,
	      XtNeditType, XawtextEdit,
	      XtNwidth, 50,
	      XtNresizable, True,
	      XtNfromVert, r8label,
	      XtNhorizDistance, ddist + PRINTINDENT,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainRight,
	      NULL);
	    XtOverrideTranslations(r8text, xlats2);

	    /* Row 9:  action buttons */

	    r9ok = XtVaCreateManagedWidget("ok", commandWidgetClass, form,
	      XtNlabel, "OK",
	      XtNaccelerators, accels_cr,
	      XtNfromVert, r8text,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainLeft,
	      NULL);
	    XtAddCallback(r9ok, XtNcallback, cb_print, NULL);
	    XtInstallAccelerators(form, r9ok);
	    XtInstallAccelerators(r2text, r9ok);
	    XtInstallAccelerators(r3text, r9ok);

	    r9prev = XtVaCreateManagedWidget("prev", commandWidgetClass, form,
	      XtNlabel, "Show previous log",
	      XtNsensitive, False,
	      XtNfromHoriz, r9ok,
	      XtNfromVert, r8text,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainLeft,
	      NULL);
	    XtAddCallback(r9prev, XtNcallback, cb_print_cancel, NULL);

	    r9cancel = XtVaCreateManagedWidget("cancel", commandWidgetClass,
	      form,
	      XtNlabel, "Cancel",
	      XtNfromHoriz, r9prev,
	      XtNfromVert, r8text,
	      XtNleft, XawChainRight,
	      XtNright, XawChainRight,
	      NULL);
	    XtAddCallback(r9cancel, XtNcallback, cb_print_cancel, NULL);

	    /* Realize and set destroy callback */
	    XdviXawRealizePopup(print_shell, cb_print_cancel);

	    {	/* Center the popup over the main window */
		Dimension	sh;	/* sw was defined earlier */
		Position	tx, ty;
		Dimension	tw, th;

		/* Get the size of the popup window */
		XtVaGetValues(print_shell, XtNwidth, &sw, XtNheight, &sh, NULL);

		/* Center the popup over the main window */
		XtVaGetValues(top_level, XtNx, &tx, XtNy, &ty,
		  XtNwidth, &tw, XtNheight, &th, NULL);
		XtVaSetValues(print_shell, XtNx, tx + (tw - sw) / 2,
		  XtNy, ty + (th - sh) / 2, NULL);
	    }

	    {
		/* Set width of dvips-xargs text widget */
		Dimension	b;

		XtVaGetValues(r8text, XtNborderWidth, &b, NULL);
		XtVaSetValues(r8text,
		  XtNwidth, sw - 2 * (b + ddist) - PRINTINDENT,
		  NULL);
	    }

	    {	/* Set canceladjust */
		Dimension	cw, cb;
		Position	cx;

		XtVaGetValues(r9cancel, XtNwidth, &cw, XtNborderWidth, &cb,
		  XtNx, &cx, NULL);
		canceladjust = cx + cw + 2 * cb - ddist;
	    }
	}
	else {	/* if the window was already created */
	    XawToggleSetCurrent(r1radio1, pr_save_fileradio);
	    set_sensitivity(pr_save_fileradio);
	    XawToggleSetCurrent(r6radio, pr_save_pageradio);

	    XtVaSetValues(r7label3, XtNlabel, ofstring, NULL);

	    if (pr_save_printer != NULL) {
		XtVaSetValues(r4text, XtNstring, pr_save_printer, NULL);
		XawTextSetInsertionPoint(r4text, strlen(pr_save_printer));
	    }
	    else
		XtVaSetValues(r4text, XtNstring, "", NULL);

	    if (pr_save_xargs != NULL) {
		XtVaSetValues(r8text, XtNstring, pr_save_xargs, NULL);
		XawTextSetInsertionPoint(r8text, strlen(pr_save_xargs));
	    }
	    else
		XtVaSetValues(r8text, XtNstring, "", NULL);

	    XtVaGetValues(print_shell, XtNwidth, &sw, NULL);
	}

	free(ofstring);
	XtVaSetValues(r9prev, XtNhorizDistance, (sw - canceladjust) / 2, NULL);
	XtVaSetValues(r9cancel, XtNhorizDistance, (sw - canceladjust) / 2,
	  NULL);

	{	/* set initial values of print command and file name fields */
	    _Xconst char *s;
	    char s2[3 * sizeof(int) + 1];

	    s = pr_save_cmd;
	    if (s == NULL) s = "lpr";
	    /* for some reason two separate XtVaSetValues calls are necessary */
	    XtVaSetValues(r2text, XtNstring, s, NULL);
	    XawTextSetInsertionPoint(r2text, strlen(s));

	    if (pr_save_file == NULL) {
		char		*s2;
		int		n;

		s = rindex(dvi_name, '/');
		if (s == NULL) s = dvi_name;
		else ++s;
		n = strlen(s);
		if (n >= 4 && memicmp(s + n - 4, ".dvi", 4) == 0) n -= 4;
		s2 = xmalloc(n + 4);
		memcpy(s2, s, n);
		memcpy(s2 + n, ".ps", 4);
		pr_save_file = s2;
	    }
	    s = pr_save_file;
	    XtVaSetValues(r3text, XtNstring, s, NULL);
	    XawTextSetInsertionPoint(r3text, strlen(s));

#if HAVE_VSNPRINTF
	    (void) snprintf(s2, sizeof s2, "%d", current_page + pageno_correct);
#else
	    (void) sprintf(s2, "%d", current_page + pageno_correct);
#endif
	    XtVaSetValues(r7text1, XtNstring, s2, NULL);
	    XawTextSetInsertionPoint(r7text1, strlen(s2));
	    XtVaSetValues(r7text2, XtNstring, s2, NULL);
	    XawTextSetInsertionPoint(r7text2, strlen(s2));
	}

	XtPopup(print_shell, XtGrabNone);
	print_active = True;
}

#else /* MOTIF */

#ifndef DDIST
# define DDIST		4
#endif

#ifndef DDIST_MAJOR
# define DDIST_MAJOR	10
#endif

static	Widget	f1radiowidget;
static	Widget	f2radiowidget;
static	Widget	r7label1, r7label2, r7label3;

/* These are needed so that *pr_save_fileradio will work.  */
#define	r2radio	r1radio1
#define	r3radio	r1radio2

/* ARGSUSED */
static	void
cb_print_vs_file(w, client_data, call_data)
	Widget		w;
	XtPointer	client_data;
	XtPointer	call_data;
{
	if (((XmToggleButtonCallbackStruct *) call_data)->set) {
	    if (w != f1radiowidget) {
		XmToggleButtonGadgetSetState(f1radiowidget, False, False);
		f1radiowidget = w;
		set_sensitivity(client_data);
		XmProcessTraversal(
		  pr_current_fileradio == &r1radio1 ? r2text : r3text,
		  XmTRAVERSE_CURRENT);
	    }
	}
	else if (w == f1radiowidget)
	    XmToggleButtonGadgetSetState(w, True, False);
}


static void
range_set_sensitivity(pageradio)
	Widget	*pageradio;
{
	Boolean	sensitivity;

	if (pageradio == pr_current_pageradio) /* if no change */
	    return;

	pr_current_pageradio = pageradio;
	sensitivity = (pageradio == &r7radio);

	XtSetSensitive(r7label1, sensitivity);
	XtSetSensitive(r7text1, sensitivity);
	XtSetSensitive(r7label2, sensitivity);
	XtSetSensitive(r7text2, sensitivity);
	XtSetSensitive(r7label3, sensitivity);
}

/* ARGSUSED */
static	void
cb_range(w, client_data, call_data)
	Widget		w;
	XtPointer	client_data;
	XtPointer	call_data;
{
	if (((XmToggleButtonCallbackStruct *) call_data)->set) {
	    if (w != f2radiowidget) {
		XmToggleButtonGadgetSetState(f2radiowidget, False, False);
		f2radiowidget = w;
		range_set_sensitivity((Widget *) client_data);
		if (pr_current_pageradio == &r7radio)
		    XmProcessTraversal(r7text1, XmTRAVERSE_CURRENT);
	    }
	}
	else if (w == f2radiowidget)
	    XmToggleButtonGadgetSetState(w, True, False);
}

/* ARGSUSED */
void
Act_print P4C(Widget, w, XEvent *, event, String *, params,
  Cardinal *, num_params)
{
	char		*ofstring;

	if (print_active) {
	    XRaiseWindow(DISP, XtWindow(print_shell));
	    return;
	}

	if (printlog_active) {
	    XRaiseWindow(DISP, XtWindow(printlog_shell));
	    return;
	}

	if (dvi_file == NULL) {
	    WARN(XmDIALOG_ERROR, "No active dvi file");
	    return;
	}

	ofstring = (pageno_correct == 1 ? mprintf("of %d", total_pages)
	  : mprintf("of %d to %d", pageno_correct,
	    total_pages + pageno_correct - 1));

	if (print_shell == NULL) {
	    Widget	form;
	    Widget	frame1, f1label, f1child;
	    Widget	r4label;
	    Widget	r8label;
	    Widget	frame2, f2label, f2child;
	    Widget	r9ok, r9cancel;
	    XmString	str;

	    XtAddActions(print_actions, XtNumber(print_actions));
	    print_shell = XtVaCreatePopupShell("print",
	      xmDialogShellWidgetClass, top_level,
	      XmNtitle, "Print dvi file",
	      XmNallowShellResize, True,
	      XmNdeleteResponse, XmDO_NOTHING,
	      NULL);
	    XmAddWMProtocolCallback(print_shell, XA_WM_DELETE_WINDOW,
	      cb_print_cancel, NULL);

	    form = XtVaCreateWidget("form", xmFormWidgetClass, print_shell,
	      XmNhorizontalSpacing, DDIST_MAJOR,
	      XmNverticalSpacing, DDIST_MAJOR,
	      XmNautoUnmanage, False,
	      NULL);

	    /*  First frame (rows 1-3):  print to printer or file */

	    frame1 = XtVaCreateWidget("printtoframe", xmFrameWidgetClass, form,
	      XmNmarginWidth, DDIST,
	      XmNmarginHeight, DDIST,
	      XmNtopAttachment, XmATTACH_FORM,
	      XmNleftAttachment, XmATTACH_FORM,
	      XmNrightAttachment, XmATTACH_FORM,
	      NULL);

	    str = XmStringCreateLocalized("Print To");
	    f1label = XtVaCreateManagedWidget("title", xmLabelGadgetClass,
	      frame1,
	      XmNchildType, XmFRAME_TITLE_CHILD,
	      XmNlabelString, str,
	      NULL);
	    XmStringFree(str);

	    f1child = XtVaCreateWidget("form", xmFormWidgetClass, frame1,
	      XmNhorizontalSpacing, DDIST,
	      XmNverticalSpacing, DDIST,
	      NULL);

	    str = XmStringCreateLocalized("Printer");
	    f1radiowidget = r2radio = XtVaCreateManagedWidget("toprinter",
	      xmToggleButtonGadgetClass, f1child,
	      XmNlabelString, str,
	      XmNindicatorType, XmONE_OF_MANY,
	      XmNset, True,
	      XmNtopAttachment, XmATTACH_FORM,
	      XmNtopOffset, 0,
	      XmNleftAttachment, XmATTACH_FORM,
	      XmNleftOffset, 0,
	      NULL);
	    XmStringFree(str);
	    XtAddCallback(r2radio, XmNvalueChangedCallback, cb_print_vs_file,
	      &r1radio1);

	    str = XmStringCreateLocalized("Print command:");
	    r2label = XtVaCreateManagedWidget("prncommand", xmLabelGadgetClass,
	      f1child,
	      XmNlabelString, str,
	      XmNtopAttachment, XmATTACH_FORM,
	      XmNtopOffset, 0,
	      XmNleftAttachment, XmATTACH_WIDGET,
	      XmNleftWidget, r2radio,
	      NULL);
	    XmStringFree(str);

	    r2text = XtVaCreateManagedWidget("prntext", xmTextFieldWidgetClass,
	      f1child,
	      XmNtopAttachment, XmATTACH_FORM,
	      XmNtopOffset, 0,
	      XmNleftAttachment, XmATTACH_WIDGET,
	      XmNleftWidget, r2label,
	      NULL);
	    XtAddCallback(r2text, XmNactivateCallback, cb_print, NULL);

	    {	/* straighten the row of widgets */
		Dimension h1, h2;

		XtVaGetValues(r2text, XmNheight, &h2, NULL);
		XtVaGetValues(r2radio, XmNheight, &h1, NULL);
		XtVaSetValues(r2radio, XmNtopOffset, (h2 - h1) / 2, NULL);
		XtVaGetValues(r2label, XmNheight, &h1, NULL);
		XtVaSetValues(r2label, XmNtopOffset, (h2 - h1) / 2, NULL);
	    }

	    str = XmStringCreateLocalized("File");
	    r3radio = XtVaCreateManagedWidget("tofile",
	      xmToggleButtonGadgetClass, f1child,
	      XmNlabelString, str,
	      XmNindicatorType, XmONE_OF_MANY,
	      XmNtopAttachment, XmATTACH_WIDGET,
	      XmNtopWidget, r2text,
	      XmNleftAttachment, XmATTACH_FORM,
	      XmNleftOffset, 0,
	      NULL);
	    XmStringFree(str);
	    XtAddCallback(r3radio, XmNvalueChangedCallback, cb_print_vs_file,
	      &r1radio2);

	    str = XmStringCreateLocalized("File name:");
	    r3label = XtVaCreateManagedWidget("filename", xmLabelGadgetClass,
	      f1child,
	      XmNlabelString, str,
	      XmNsensitive, False,
	      XmNtopAttachment, XmATTACH_WIDGET,
	      XmNtopWidget, r2text,
	      XmNleftAttachment, XmATTACH_WIDGET,
	      XmNleftWidget, r3radio,
	      NULL);
	    XmStringFree(str);

	    r3text = XtVaCreateManagedWidget("prntext", xmTextFieldWidgetClass,
	      f1child,
	      XmNsensitive, False,
	      XmNtopAttachment, XmATTACH_WIDGET,
	      XmNtopWidget, r2text,
	      XmNleftAttachment, XmATTACH_WIDGET,
	      XmNleftWidget, r3label,
	      NULL);
	    XtAddCallback(r3text, XmNactivateCallback, cb_print, NULL);

	    {	/* straighten the row of widgets */
		Dimension h1, h2;

		XtVaGetValues(r3text, XmNheight, &h2, NULL);
		XtVaGetValues(r3radio, XmNheight, &h1, NULL);
		XtVaSetValues(r3radio, XmNtopOffset, DDIST + (h2 - h1) / 2,
		  NULL);
		XtVaGetValues(r3label, XmNheight, &h1, NULL);
		XtVaSetValues(r3label, XmNtopOffset, DDIST + (h2 - h1) / 2,
		  NULL);
	    }

	    {	/* align left edges of text widgets */
		Dimension w2r, w3r, w2l, w3l;

		XtVaGetValues(r2radio, XmNwidth, &w2r, NULL);
		XtVaGetValues(r3radio, XmNwidth, &w3r, NULL);
		XtVaGetValues(r2label, XmNwidth, &w2l, NULL);
		XtVaGetValues(r3label, XmNwidth, &w3l, NULL);
		if (w2l < w3l || w2r < w3r) {
		    Dimension offset;

		    offset = DDIST;
		    if (w2l < w3l) offset += w3l - w2l;
		    if (w2r < w3r) offset += w3r - w2r;
		    XtVaSetValues(r2label, XmNleftOffset, offset, NULL);
		}
		if (w2l > w3l || w2r > w3r) {
		    Dimension offset;

		    offset = DDIST;
		    if (w2l > w3l) offset += w2l - w3l;
		    if (w2r > w3r) offset += w2r - w3r;
		    XtVaSetValues(r3label, XmNleftOffset, offset, NULL);
		}
	    }

	    XtManageChild(f1child);
	    XtManageChild(frame1);

	    /* Row 4:  printer name (for dvips) */

	    str = XmStringCreateLocalized("Printer name (used by dvips):");
	    r4label = XtVaCreateManagedWidget("prname", xmLabelGadgetClass,
	      form,
	      XmNlabelString, str,
	      XmNtopAttachment, XmATTACH_WIDGET,
	      XmNtopWidget, frame1,
	      XmNleftAttachment, XmATTACH_FORM,
	      NULL);
	    XmStringFree(str);

	    r4text = XtVaCreateManagedWidget("prtext", xmTextFieldWidgetClass,
	      form,
	      XmNtopAttachment, XmATTACH_WIDGET,
	      XmNtopWidget, frame1,
	      XmNleftAttachment, XmATTACH_WIDGET,
	      XmNleftWidget, r4label,
	      XmNcolumns, 5,
	      NULL);
	    XtAddCallback(r4text, XmNactivateCallback, cb_print, NULL);

	    {	/* straighten the row of widgets */
		Dimension h1, h2;

		XtVaGetValues(r4text, XmNheight, &h2, NULL);
		XtVaGetValues(r4label, XmNheight, &h1, NULL);
		XtVaSetValues(r4label,
		  XmNtopOffset, DDIST_MAJOR + (h2 - h1) / 2,
		  NULL);
	    }

	    /* Row 8 (row 4.5 for Motif):  additional dvips args */

	    str = XmStringCreateLocalized(
	      "Additional dvips arguments (optional):");
	    r8label = XtVaCreateManagedWidget("xargsname", xmLabelGadgetClass,
	      form,
	      XmNlabelString, str,
	      XmNtopAttachment, XmATTACH_WIDGET,
	      XmNtopWidget, r4text,
	      XmNleftAttachment, XmATTACH_FORM,
	      NULL);
	    XmStringFree(str);

	    r8text = XtVaCreateManagedWidget("xargstext",
	      xmTextFieldWidgetClass, form,
	      XmNtopAttachment, XmATTACH_WIDGET,
	      XmNtopWidget, r8label,
	      XmNleftAttachment, XmATTACH_FORM,
	      XmNleftOffset, DDIST + DDIST_MAJOR,
	      XmNrightAttachment, XmATTACH_FORM,
	      XmNrightOffset, DDIST + DDIST_MAJOR,
	      NULL);
	    XtAddCallback(r8text, XmNactivateCallback, cb_print, NULL);

	    /* Second frame (rows 5-7):  page selection */

	    frame2 = XtVaCreateWidget("printtoframe", xmFrameWidgetClass, form,
	      XmNmarginWidth, DDIST,
	      XmNmarginHeight, DDIST,
	      XmNresizable, True,
	      XmNtopAttachment, XmATTACH_WIDGET,
	      XmNtopWidget, r8text,
	      XmNleftAttachment, XmATTACH_FORM,
	      XmNrightAttachment, XmATTACH_FORM,
	      NULL);

	    str = XmStringCreateLocalized("Page Selection");
	    f2label = XtVaCreateManagedWidget("title", xmLabelGadgetClass,
	      frame2,
	      XmNchildType, XmFRAME_TITLE_CHILD,
	      XmNlabelString, str,
	      NULL);
	    XmStringFree(str);

	    f2child = XtVaCreateWidget("form", xmFormWidgetClass, frame2,
	      XmNhorizontalSpacing, DDIST,
	      XmNverticalSpacing, DDIST,
	      XmNresizable, True,
	      NULL);

	    str = XmStringCreateLocalized("All");
	    f2radiowidget = r6radio = XtVaCreateManagedWidget("all",
	      xmToggleButtonGadgetClass, f2child,
	      XmNlabelString, str,
	      XmNindicatorType, XmONE_OF_MANY,
	      XmNset, True,
	      XmNtopAttachment, XmATTACH_FORM,
	      XmNtopOffset, 0,
	      XmNleftAttachment, XmATTACH_FORM,
	      XmNleftOffset, 0,
	      NULL);
	    XmStringFree(str);
	    XtAddCallback(r6radio, XmNvalueChangedCallback, cb_range,
	      &r6radio);

	    str = XmStringCreateLocalized("Range:");
	    r7radio = XtVaCreateManagedWidget("range",
	      xmToggleButtonGadgetClass, f2child,
	      XmNlabelString, str,
	      XmNindicatorType, XmONE_OF_MANY,
	      XmNtopAttachment, XmATTACH_WIDGET,
	      XmNtopWidget, r6radio,
	      XmNleftAttachment, XmATTACH_FORM,
	      XmNleftOffset, 0,
	      NULL);
	    XmStringFree(str);
	    XtAddCallback(r7radio, XmNvalueChangedCallback, cb_range,
	      &r7radio);

	    str = XmStringCreateLocalized("From");
	    r7label1 = XtVaCreateManagedWidget("from", xmLabelGadgetClass,
	      f2child,
	      XmNlabelString, str,
	      XmNsensitive, False,
	      XmNtopAttachment, XmATTACH_WIDGET,
	      XmNtopWidget, r6radio,
	      XmNleftAttachment, XmATTACH_WIDGET,
	      XmNleftWidget, r7radio,
	      NULL);
	    XmStringFree(str);

	    r7text1 = XtVaCreateManagedWidget("frompage",
	      xmTextFieldWidgetClass, f2child,
	      XmNcolumns, 5,
	      XmNsensitive, False,
	      XmNtopAttachment, XmATTACH_WIDGET,
	      XmNtopWidget, r6radio,
	      XmNleftAttachment, XmATTACH_WIDGET,
	      XmNleftWidget, r7label1,
	      NULL);
	    XtAddCallback(r7text1, XmNactivateCallback, cb_print, NULL);

	    str = XmStringCreateLocalized("to");
	    r7label2 = XtVaCreateManagedWidget("to", xmLabelGadgetClass,
	      f2child,
	      XmNlabelString, str,
	      XmNsensitive, False,
	      XmNtopAttachment, XmATTACH_WIDGET,
	      XmNtopWidget, r6radio,
	      XmNleftAttachment, XmATTACH_WIDGET,
	      XmNleftWidget, r7text1,
	      NULL);
	    XmStringFree(str);

	    r7text2 = XtVaCreateManagedWidget("topage", xmTextFieldWidgetClass,
	      f2child,
	      XmNcolumns, 5,
	      XmNsensitive, False,
	      XmNtopAttachment, XmATTACH_WIDGET,
	      XmNtopWidget, r6radio,
	      XmNleftAttachment, XmATTACH_WIDGET,
	      XmNleftWidget, r7label2,
	      NULL);
	    XtAddCallback(r7text2, XmNactivateCallback, cb_print, NULL);

	    str = XmStringCreateLocalized(ofstring);
	    r7label3 = XtVaCreateManagedWidget("of", xmLabelGadgetClass,
	      f2child,
	      XmNlabelString, str,
	      XmNsensitive, False,
	      XmNresizable, True,
	      XmNtopAttachment, XmATTACH_WIDGET,
	      XmNtopWidget, r6radio,
	      XmNleftAttachment, XmATTACH_WIDGET,
	      XmNleftWidget, r7text2,
	      NULL);
	    XmStringFree(str);

	    {	/* straighten the row of widgets */
		Dimension h1, h2;

		XtVaGetValues(r7text1, XmNheight, &h2, NULL);
		XtVaGetValues(r7radio, XmNheight, &h1, NULL);
		XtVaSetValues(r7radio, XmNtopOffset, DDIST + (h2 - h1) / 2,
		  NULL);
		XtVaGetValues(r7label1, XmNheight, &h1, NULL);
		XtVaSetValues(r7label1, XmNtopOffset, DDIST + (h2 - h1) / 2,
		  NULL);
		XtVaSetValues(r7label2, XmNtopOffset, DDIST + (h2 - h1) / 2,
		  NULL);
		XtVaSetValues(r7label3, XmNtopOffset, DDIST + (h2 - h1) / 2,
		  NULL);
	    }

	    XtManageChild(f2child);
	    XtManageChild(frame2);

	    str = XmStringCreateLocalized("OK");
	    r9ok = XtVaCreateManagedWidget("ok", xmPushButtonWidgetClass,
	      form,
	      XmNlabelString, str,
	      XmNshowAsDefault, True,
	      XmNdefaultButtonShadowThickness, 1,
	      XmNtopAttachment, XmATTACH_WIDGET,
	      XmNtopWidget, frame2,
	      XmNtopOffset, DDIST_MAJOR + DDIST,
	      XmNleftAttachment, XmATTACH_FORM,
	      XmNbottomAttachment, XmATTACH_FORM,
	      NULL);
	    XmStringFree(str);
	    XtAddCallback(r9ok, XmNactivateCallback, cb_print, NULL);
	    XtOverrideTranslations(r9ok, XtParseTranslationTable(
	      "<Key>Return:ArmAndActivate()"));
	    XmProcessTraversal(r9ok, XmTRAVERSE_CURRENT);

	    str = XmStringCreateLocalized("Show previous log");
	    r9prev = XtVaCreateManagedWidget("prev",
	      xmPushButtonWidgetClass, form,
	      XmNlabelString, str,
	      XmNsensitive, False,
	      XmNtopAttachment, XmATTACH_WIDGET,
	      XmNtopWidget, frame2,
	      XmNtopOffset, DDIST_MAJOR + DDIST,
	      XmNleftAttachment, XmATTACH_WIDGET,
	      XmNleftWidget, r9ok,
	      NULL);
	    XmStringFree(str);
	    XtAddCallback(r9prev, XmNactivateCallback, cb_print_cancel, NULL);

	    str = XmStringCreateLocalized("Cancel");
	    r9cancel = XtVaCreateManagedWidget("cancel",
	      xmPushButtonWidgetClass, form,
	      XmNlabelString, str,
	      XmNtopAttachment, XmATTACH_WIDGET,
	      XmNtopWidget, frame2,
	      XmNtopOffset, DDIST_MAJOR + DDIST,
	      XmNrightAttachment, XmATTACH_FORM,
	      NULL);
	    XmStringFree(str);
	    XtAddCallback(r9cancel, XmNactivateCallback, cb_print_cancel, NULL);

	    {	/* straighten the row of widgets */
		Dimension h1, h2;

		XtVaGetValues(r9ok, XmNheight, &h1, NULL);
		XtVaGetValues(r9prev, XmNheight, &h2, NULL);
		XtVaSetValues(r9prev,
		  XmNtopOffset, DDIST_MAJOR + DDIST + (h1 - h2) / 2, NULL);
		XtVaGetValues(r9cancel, XmNheight, &h2, NULL);
		XtVaSetValues(r9cancel,
		  XmNtopOffset, DDIST_MAJOR + DDIST + (h1 - h2) / 2, NULL);
	    }

	    XtManageChild(form);

	    {	/* center middle button in bottom row */
		Dimension	w, b;
		Position	x1, x3;

		XtVaGetValues(r9ok, XmNx, &x1, XmNwidth, &w, NULL);
		x1 += w;
		XtVaGetValues(r9prev, XmNwidth, &w, NULL);
		XtVaGetValues(r9cancel, XmNx, &x3, NULL);
		XtVaSetValues(r9prev, XmNleftOffset, (x3 - x1 - w) / 2, NULL);
	    }
	}
	else {	/* if the window was already created */
	    XmString	str;

	    if (pr_save_fileradio != pr_current_fileradio)
		XmToggleButtonGadgetSetState(*pr_save_fileradio, True, True);
	    if (pr_save_pageradio != pr_current_pageradio)
		XmToggleButtonGadgetSetState(*pr_save_pageradio, True, True);

	    str = XmStringCreateLocalized(ofstring);
	    XtVaSetValues(r7label3, XmNlabelString, str, NULL);
	    XmStringFree(str);

	    if (pr_save_printer != NULL) {
		XtVaSetValues(r4text, XmNvalue, pr_save_printer,
		  XmNcursorPosition, strlen(pr_save_printer), NULL);
	    }
	    else
		XtVaSetValues(r4text, XmNvalue, "", NULL);

	    if (pr_save_xargs != NULL) {
		XtVaSetValues(r8text, XmNvalue, pr_save_xargs,
		  XmNcursorPosition, strlen(pr_save_xargs), NULL);
	    }
	    else
		XtVaSetValues(r8text, XmNvalue, "", NULL);
	}

	free(ofstring);

	{	/* set initial values of print command and file name fields */
	    _Xconst char *s;
	    char s2[3 * sizeof(int) + 1];

	    s = pr_save_cmd;
	    if (s == NULL) s = "lpr";
	    XtVaSetValues(r2text, XmNvalue, s, XmNcursorPosition, strlen(s),
	      NULL);

	    if (pr_save_file == NULL) {
		char		*s2;
		int		n;

		s = rindex(dvi_name, '/');
		if (s == NULL) s = dvi_name;
		else ++s;
		n = strlen(s);
		if (n >= 4 && memicmp(s + n - 4, ".dvi", 4) == 0) n -= 4;
		s2 = xmalloc(n + 4);
		memcpy(s2, s, n);
		memcpy(s2 + n, ".ps", 4);
		pr_save_file = s2;
	    }
	    s = pr_save_file;
	    XtVaSetValues(r3text, XmNvalue, s, XmNcursorPosition, strlen(s),
	      NULL);

#if HAVE_VSNPRINTF
	    (void) snprintf(s2, sizeof s2, "%d", current_page + pageno_correct);
#else
	    (void) sprintf(s2, "%d", current_page + pageno_correct);
#endif
	    XtVaSetValues(r7text1, XmNvalue, s2, XmNcursorPosition, strlen(s2),
	      NULL);
	    XtVaSetValues(r7text2, XmNvalue, s2, XmNcursorPosition, strlen(s2),
	      NULL);
	}

	XtPopup(print_shell, XtGrabNone);
	print_active = True;
}

#endif /* MOTIF */


/*
 *	The actual printing
 */

static	int	pageno1, pageno2;		/* page range */
static	Widget	print_confirm = NULL;
static	Widget	printlog_text;
static	Widget	printlog_close;
static	Widget	printlog_cancel;

#if XAW
static	XawTextPosition	printlog_length;
#else /* MOTIF */
static	XmTextPosition	printlog_length;
#endif

static	void		dvips_ended ARGS((int));
static	struct xchild	print_child	= {NULL, 0, True, dvips_ended};

static	void		read_from_dvips ARGS((void));
static	struct xio	print_xio	= {NULL, 0, XIO_IN,
#if HAVE_POLL
					   NULL,
#endif
					   read_from_dvips, NULL};

static	void		dvips_alarm ARGS((struct xtimer *));
static	struct xtimer	dvips_timer	= TIMER_INIT(dvips_alarm);
static	int		dvips_status;
#define	DVIPS_STAT_NONE	0
#define	DVIPS_STAT_RUN	1
#define	DVIPS_STAT_WAIT	2
static	int		dvips_sig;	/* SIGINT or SIGKILL */

/* Forward reference */

static void print_do_it();

static int
getpageno(w)
	Widget	w;
{
	char	*s, *p;

#if XAW
	XtVaGetValues(w, XtNstring, &s, NULL);
#else /* MOTIF */
	s = XmTextFieldGetString(w);
#endif
	p = s;
	if (*p == '-') ++p;
	if (!isdigit(*p)) return 0;
	do ++p; while (isdigit(*p));
	if (*p != '\0') return 0;

#if XAW
	return atoi(s) - pageno_correct + 1;
#else /* MOTIF */
	{
	    int i = atoi(s) - pageno_correct + 1;

	    XtFree(s);
	    return i;
	}
#endif
}

/* ARGSUSED */
static	void
cb_print_confirm(w, client_data, call_data)
	Widget		w;
	XtPointer	client_data;
	XtPointer	call_data;
{
	XtDestroyWidget(print_confirm);
	print_confirm = NULL;

	if (client_data != NULL)
	    print_do_it();
}

static	void
print_precheck()
{
	char		*dest;
	struct stat	statbuf;

	/* Check for active confirm box */
	if (print_confirm != NULL) {
	    XRaiseWindow(DISP, XtWindow(print_confirm));
	    return;
	}

	/* Validate page range (if given) */
	if (pr_current_pageradio == &r7radio) {	/* if page range selected */
	    pageno1 = getpageno(r7text1);
	    pageno2 = getpageno(r7text2);
	    if (pageno1 <= 0 || pageno1 > pageno2 || pageno2 > total_pages) {
		WARN(XmDIALOG_ERROR, "Invalid page range");
		return;
	    }
	}

	/* Pop down window */
	cb_print_cancel(NULL, NULL, NULL);

	/* Check for whether to clobber existing file */
	if (pr_current_fileradio == &r1radio2) {	/* if print to file */
#if XAW
	    XtVaGetValues(r3text, XtNstring, &dest, NULL);
#else /* MOTIF */
	    dest = XmTextFieldGetString(r3text);
#endif
	    if (stat(dest, &statbuf) == 0 && S_ISREG(statbuf.st_mode)) {
		char	*msg;

		msg = mprintf("Overwrite existing file %s?", dest);
		print_confirm = confirm_popup(msg, "OK", cb_print_confirm,
		  (XtPointer) print_shell, NULL);
		free(msg);
#if MOTIF
		XtFree(dest);
#endif
		return;
	    }
#if MOTIF
	    XtFree(dest);
#endif
	}

	print_do_it();
}


static	void	cb_dvips_close ARGS((Widget, XtPointer, XtPointer));
static	void	cb_dvips_keep ARGS((Widget, XtPointer, XtPointer));
static	void	cb_dvips_cancel ARGS((Widget, XtPointer, XtPointer));
static	void	cb_dvips_delete ARGS((Widget, XtPointer, XtPointer));
static	void	printlog_append ARGS((_Xconst char *, int));

struct dvips_env {
	_Xconst char	*envname;
	_Xconst char	*en1, *en2;
	_Xconst char	*envval;
};

static	struct dvips_env	dvips_envs[] = {
	{NULL, "TEXPICTS", "TEXINPUTS", NULL},
	{NULL, "TEXPSHEADERS", "PSHEADERS", NULL}};

static	void
print_do_it()
{
	_Xconst char	*dest;
	_Xconst char	*prn;
	_Xconst char	*xargs;
	FILE		*f;
	unsigned int	len;	/* length of command line */
	unsigned int	argc;
	char		**argv;
	char		**argnext;
	int		print_io[2];
	char		*p, **pp;
	_Xconst char	*q;

	if (dvi_file == NULL) {
	    WARN(XmDIALOG_ERROR, "No active dvi file");
	    return;
	}

	/* Save state of window for next time */
	pr_save_fileradio = pr_current_fileradio;
	pr_save_pageradio = pr_current_pageradio;

#if XAW

	if (pr_save_fileradio == &r1radio1) {	/* if print to printer */
	    if (pr_save_cmd != NULL)
		free((char *) pr_save_cmd);
	    XtVaGetValues(r2text, XtNstring, &dest, NULL);
	    dest = pr_save_cmd = xstrdup(dest);
	}
	else {		/* if print to file */
	    if (pr_save_file != NULL)
		free((char *) pr_save_file);
	    XtVaGetValues(r3text, XtNstring, &dest, NULL);
	    dest = pr_save_file = xstrdup(dest);
	}

	XtVaGetValues(r4text, XtNstring, &prn, NULL);
	if (pr_save_printer != NULL)
	    free((char *) pr_save_printer);
	if (*prn != '\0')
	    prn = pr_save_printer = xstrdup(prn);
	else
	    prn = pr_save_printer = NULL;

	XtVaGetValues(r8text, XtNstring, &xargs, NULL);
	if (pr_save_xargs != NULL)
	    free((char *) pr_save_xargs);
	if (*xargs != '\0')
	    xargs = pr_save_xargs = xstrdup(xargs);
	else
	    xargs = pr_save_xargs = NULL;

#else /* MOTIF */

	if (pr_save_fileradio == &r1radio1) {	/* if print to printer */
	    if (pr_save_cmd != NULL)
		XtFree((char *) pr_save_cmd);
	    dest = pr_save_cmd = XmTextFieldGetString(r2text);
	}
	else {		/* if print to file */
	    if (pr_save_file != NULL)
		XtFree((char *) pr_save_file);
	    dest = pr_save_file = XmTextFieldGetString(r3text);
	}

	if (pr_save_printer != NULL)
	    XtFree((char *) pr_save_printer);
	prn = XmTextFieldGetString(r4text);
	if (*prn == '\0') {
	    XtFree((char *) prn);
	    prn = NULL;
	}
	pr_save_printer = prn;

	if (pr_save_xargs != NULL)
	    XtFree((char *) pr_save_xargs);
	xargs = XmTextFieldGetString(r8text);
	if (*xargs == '\0') {
	    XtFree((char *) xargs);
	    xargs = NULL;
	}
	pr_save_xargs = xargs;

#endif

	/* Open file for writing (if necessary) */
	if (pr_save_fileradio == &r1radio2) {	/* if print to file */
	    f = xfopen(dest, "w");
	    if (f == NULL) {
		WARN2(XmDIALOG_ERROR, "Error opening file: %s\n%s.", dest,
		  strerror(errno));
		return;
	    }
	}

#if XAW
	/* Create popup window */
	if (printlog_shell == NULL) {
	    Widget		form;
	    int			ddist;
	    Dimension		w0, w1, w2, w3, b;

	    XtSetSensitive(r9prev, True);
	    printlog_shell = XtVaCreatePopupShell("printlog",
	      transientShellWidgetClass, top_level,
	      XtNtitle, "Xdvi print process",
	      XtNmappedWhenManaged, False,
	      XtNtransientFor, top_level,
	      NULL);
	    form = XtCreateManagedWidget("form", formWidgetClass,
	      printlog_shell, NULL, 0);
	    printlog_text = XtVaCreateManagedWidget("text",
	      asciiTextWidgetClass, form,
	      XtNstring, "",
	      XtNdataCompression, False,
	      XtNeditType, XawtextAppend,
	      XtNscrollHorizontal, XawtextScrollAlways,
	      XtNscrollVertical, XawtextScrollAlways,
	      XtNwidth, 600,
	      XtNheight, 400,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainRight,
	      XtNtop, XawChainTop,
	      XtNbottom, XawChainBottom,
	      NULL);
	    XtOverrideTranslations(printlog_text, XtParseTranslationTable(
	      "<Key>Return:printlogIntClose()\n\
^<Key>c:printlogIntCancel()\n\
^<Key>s:printlogIntKeep()\n\
^<Key>q:printlogIntUnkeep()"));

	    printlog_close = XtVaCreateManagedWidget("close",
	      commandWidgetClass, form,
	      XtNlabel, "Close window",
	      XtNsensitive, False,
	      XtNfromVert, printlog_text,
	      XtNleft, XawChainLeft,
	      XtNright, XawChainLeft,
	      XtNtop, XawChainBottom,
	      XtNbottom, XawChainBottom,
	      NULL);
	    XtAddCallback(printlog_close, XtNcallback, cb_dvips_close, NULL);

	    printlog_keep = XtVaCreateManagedWidget("keep", toggleWidgetClass,
	      form,
	      XtNlabel, "Keep window open",
	      XtNfromVert, printlog_text,
	      XtNfromHoriz, printlog_close,
	      XtNtop, XawChainBottom,
	      XtNbottom, XawChainBottom,
	      NULL);
	    XtAddCallback(printlog_keep, XtNcallback, cb_dvips_keep, NULL);

	    printlog_cancel = XtVaCreateManagedWidget("cancel",
	      commandWidgetClass, form,
	      XtNlabel, "Cancel",
	      XtNfromVert, printlog_text,
	      XtNfromHoriz, printlog_keep,
	      XtNleft, XawChainRight,
	      XtNright, XawChainRight,
	      XtNtop, XawChainBottom,
	      XtNbottom, XawChainBottom,
	      NULL);
	    XtAddCallback(printlog_cancel, XtNcallback, cb_dvips_cancel, NULL);

	    /* Move the buttons to the right spots */
	    XtVaGetValues(printlog_text, XtNwidth, &w0, NULL);
	    XtVaGetValues(form, XtNdefaultDistance, &ddist, NULL);
	    XtVaGetValues(printlog_close, XtNwidth, &w1, XtNborderWidth, &b,
	      NULL);
	    w1 += 2 * b + ddist;
	    XtVaGetValues(printlog_keep, XtNwidth, &w2, XtNborderWidth, &b,
	      NULL);
	    w2 += 2 * b;
	    if (w0 > w2 + 2 * w1) {
		XtVaSetValues(printlog_keep,
		  XtNhorizDistance, (w0 - w2) / 2 - w1, NULL);
		w1 += (w0 - w2) / 2 - w1 - ddist;
	    }
	    w1 += w2;
	    XtVaGetValues(printlog_cancel, XtNwidth, &w3, XtNborderWidth, &b,
	      NULL);
	    w3 += 2 * b;
	    if (w1 + ddist + w3 < w0)
		XtVaSetValues(printlog_cancel, XtNhorizDistance, w0 - w1 - w3,
		  NULL);

	    XdviXawRealizePopup(printlog_shell, cb_dvips_delete);
	}
	else {
	    XtVaSetValues(printlog_text, XtNstring, "", NULL);
	    XtSetSensitive(printlog_close, False);
	    XtSetSensitive(printlog_keep, True);
	    XtSetSensitive(printlog_cancel, True);
	}

#else /* MOTIF */

	/* Create popup window */
	if (printlog_shell == NULL) {
	    Widget	form;
	    XmString	str;

	    /* Under Motif 2.x (2.1.0 and 2.2.2 at least), XmNeditMode must be
	       set early in order to have an effect.  */
	    static Arg	args[]	= {
		{XmNeditable,	False},
		{XmNrows,	24},
		{XmNcolumns,	80},
		{XmNeditMode,	(XtArgVal) XmMULTI_LINE_EDIT},
	    };

	    XtSetSensitive(r9prev, True);
	    printlog_shell = XtVaCreatePopupShell("printlog",
	      xmDialogShellWidgetClass, top_level,
	      XmNtitle, "Xdvi print process",
	      XmNdeleteResponse, XmDO_NOTHING,
	      NULL);
	    XmAddWMProtocolCallback(printlog_shell, XA_WM_DELETE_WINDOW,
	      cb_dvips_delete, NULL);

	    form = XtVaCreateWidget("form", xmFormWidgetClass, printlog_shell,
	      XmNhorizontalSpacing, DDIST,
	      XmNverticalSpacing, DDIST,
	      XmNautoUnmanage, False,
	      NULL);

	    printlog_text = XmCreateScrolledText(form, "text",
	      args, XtNumber(args));
	    XtVaSetValues(XtParent(printlog_text),
	      XmNtopAttachment, XmATTACH_FORM,
	      XmNleftAttachment, XmATTACH_FORM,
	      XmNrightAttachment, XmATTACH_FORM,
	      NULL);
	    XtOverrideTranslations(printlog_text, XtParseTranslationTable(
	      "<Key>Return:printlogIntClose()\n\
^<Key>c:printlogIntCancel()\n\
^<Key>s:printlogIntKeep()\n\
^<Key>q:printlogIntUnkeep()"));
	    XtManageChild(printlog_text);

	    str = XmStringCreateLocalized(
	      "Keep window open after dvips finishes");
	    printlog_keep = XtVaCreateManagedWidget("keep",
	      xmToggleButtonGadgetClass, form,
	      XmNlabelString, str,
	      XmNnavigationType, XmTAB_GROUP,
	      XmNtopAttachment, XmATTACH_WIDGET,
	      XmNtopWidget, XtParent(printlog_text),
	      XmNleftAttachment, XmATTACH_FORM,
	      NULL);
	    XmStringFree(str);
	    XtAddCallback(printlog_keep, XmNvalueChangedCallback, cb_dvips_keep,
	      NULL);

	    str = XmStringCreateLocalized("Close window");
	    printlog_close = XtVaCreateManagedWidget("close",
	      xmPushButtonWidgetClass, form,
	      XmNlabelString, str,
	      XmNshowAsDefault, True,
	      XmNsensitive, False,
	      XmNdefaultButtonShadowThickness, 1,
	      XmNtopAttachment, XmATTACH_WIDGET,
	      XmNtopWidget, printlog_keep,
	      XmNtopOffset, DDIST,
	      XmNleftAttachment, XmATTACH_FORM,
	      XmNbottomAttachment, XmATTACH_FORM,
	      NULL);
	    XmStringFree(str);
	    XtAddCallback(printlog_close, XmNactivateCallback, cb_dvips_close,
	      NULL);

	    str = XmStringCreateLocalized("Cancel");
	    printlog_cancel = XtVaCreateManagedWidget("cancel",
	      xmPushButtonWidgetClass, form,
	      XmNlabelString, str,
	      XmNtopAttachment, XmATTACH_WIDGET,
	      XmNtopWidget, printlog_keep,
	      XmNtopOffset, DDIST,
	      XmNrightAttachment, XmATTACH_FORM,
	      NULL);
	    XmStringFree(str);
	    XtAddCallback(printlog_cancel, XmNactivateCallback, cb_dvips_cancel,
	      NULL);

	    {	/* straighten the row of widgets */
		Dimension h1, h2;

		XtVaGetValues(printlog_close, XmNheight, &h1, NULL);
		XtVaGetValues(printlog_cancel, XmNheight, &h2, NULL);
		XtVaSetValues(printlog_cancel,
		  XmNtopOffset, DDIST + (h1 - h2) / 2, NULL);
		XtVaSetValues(printlog_cancel,
		  XmNrightOffset, DDIST + (h1 - h2) / 2, NULL);
	    }

	    XtManageChild(form);
	    XmProcessTraversal(printlog_text, XmTRAVERSE_CURRENT);
	}
	else {
	    XmTextSetString(printlog_text, "");
	    XtSetSensitive(printlog_close, False);
	    XtSetSensitive(printlog_keep, True);
	    XtSetSensitive(printlog_cancel, True);
	}

#endif

	/* Compute length of dvips command line */
	/*   dvips [-Pps] -f [-o!command] [-p=n -lm] [xargs] */
	argc = 3;
	len = strlen(resource.dvips_path) + 4;
	  /* "dvips -f\0" */
	if (prn != NULL) argc = 4, len += strlen(prn) + 3;
	if (pr_save_fileradio == &r1radio1) ++argc, len += strlen(dest) + 4;
	if (pr_save_pageradio == &r7radio) {
	    argc += 2;
	    len += uintlen(pageno1) + uintlen(pageno2) + 7;
	}
	if (pr_save_xargs != NULL) {
	    _Xconst char *q = pr_save_xargs;

	    len += strlen(pr_save_xargs);
	    for (;;) {
		while (*q == ' ' || *q == '\t') ++q, --len;
		if (*q == '\0') break;
		++argc;
		++len;
		while (*q != '\0' && *q != ' ' && *q != '\t') ++q;
	    }
	}

	/* Compute actual dvips command line */
	argv = xmalloc(argc * sizeof(char *));
	p = argv[0] = xmalloc(len + 1);
	argnext = argv + 1;
	p += strlen(strcpy(p, resource.dvips_path));
	*p++ = ' ';
	*argnext++ = p;

	if (prn != NULL) {
	    sprintf(p, "-P%s ", prn);
	    p += strlen(p);
	    *argnext++ = p;
	}

	*p++ = '-';
	*p++ = 'f';
	*p++ = ' ';
	*argnext++ = p;

	if (pr_save_fileradio == &r1radio1) {
	    sprintf(p, "-o!%s ", dest);
	    p += strlen(p);
	    *argnext++ = p;
	}

	if (pr_save_pageradio == &r7radio) {
	    sprintf(p, "-p=%u ", pageno1);
	    p += strlen(p);
	    *argnext++ = p;
	    sprintf(p, "-l%u ", pageno2);
	    p += strlen(p);
	    *argnext++ = p;
	}

	if (pr_save_xargs != NULL) {
	    _Xconst char *q = pr_save_xargs;

	    for (;;) {
		while (*q == ' ' || *q == '\t') ++q;
		if (*q == '\0') break;
		while (*q != '\0' && *q != ' ' && *q != '\t') *p++ = *q++;
		*p++ = ' ';
		*argnext++ = p;
	    }
	}

#if 0
	if (argnext - argv != argc)
	    oops("dvips command count mismatch:  %u != %u", argc,
	      argnext - argv);
	if (p - argv[0] != len)
	    oops("dvips command length mismatch:  %u != %u", len, p - argv[0]);
#endif

	p[-1] = '\n';
#if MOTIF
	*p = '\0';
#endif

	printlog_length = 0;

	q = rindex(dvi_name, '/');
	if (q != NULL) {
	    struct dvips_env *dep;

#if PS
	    /* grab copies of environment variables before we change them */
	    ps_init_paths();
#endif

	    if (dvips_envs[0].envname == NULL) {
		for (dep = dvips_envs; dep < dvips_envs + XtNumber(dvips_envs);
		  ++dep) {
		    _Xconst char *envname;
		    _Xconst char *envval;
		    unsigned int len;

		    envval = getenv(envname = dep->en1);
		    if (envval == NULL) {
			envval = getenv(dep->en2);
			if (envval != NULL) envname = dep->en2;
		    }
		    dep->envname = envname;
		    len = q - dvi_name + 2;
		    if (envval != NULL)
			len += strlen(envval);
		    dep->envval = p = xmalloc(len);
		    bcopy(dvi_name, p, q - dvi_name);
		    p += q - dvi_name;
		    *p++ = ':';
		    *p = '\0';
		    if (envval != NULL)
			Strcpy(p, envval);
		    xputenv(dep->envname, dep->envval);
		}
	    }
	    for (dep = dvips_envs; dep < dvips_envs + XtNumber(dvips_envs);
	      ++dep) {
		printlog_append(dep->envname, strlen(dep->envname));
		printlog_append("=", 1);
		printlog_append(dep->envval, strlen(dep->envval));
		printlog_append(" ", 1);
	    }
	}

	printlog_append(argv[0], len);
#if XAW
	do_popup(printlog_shell);
#else
	XtPopup(printlog_shell, XtGrabNone);
#endif
	printlog_active = True;

	--argnext;
	for (pp = argv + 1; pp <= argnext; ++pp) (*pp)[-1] = '\0';
	*argnext = NULL;

	if (xpipe(print_io) != 0) {
	    perror("[xdvi] pipe");
	    return;
	}

	/* Fork process */
	Fflush(stderr);		/* avoid double buffering */
	print_child.pid = vfork();
	if (print_child.pid == 0) {		/* if child */
	    int fd;

	    (void) close(0);
	    (void) close(fileno(dvi_file));
	    fd = open(dvi_name, O_RDONLY);
	    if (fd < 0) {
		perror(dvi_name);
		Fflush(stderr);
		_exit(1);
	    }
	    if (fd > 0) {
		dup2(fd, 0);
		(void) close(fd);
	    }

	    (void) close(1);
	    if (pr_save_fileradio == &r1radio1) {  /* if printing to printer */
		(void) dup2(print_io[1], 1);
	    }
	    else {	/* printing to file */
		(void) dup2(fileno(f), 1);
		(void) close(fileno(f));
	    }
	    (void) close(2);
	    (void) dup2(print_io[1], 2);
	    (void) close(print_io[1]);
	    (void) close(print_io[0]);

	    if (setsid() == -1) {	/* so we can kill the process group */
		perror("setsid");
		Fflush(stderr);
		_exit(1);
	    }
	    (void) execvp(*argv, argv);
	    Fprintf(stderr, "Execvp of %s failed.\n", *argv);
	    Fflush(stderr);
	    _exit(1);
	}

	free(argv[0]);
	free(argv);
	if (pr_save_fileradio == &r1radio2) {
	    fclose(f);
	    ++n_files_left;
	}

	if (print_child.pid == -1) {	/* error */
	    perror("[xdvi] vfork");
	    return;
	}

	set_chld(&print_child);
	dvips_sig = SIGINT;

	(void) close(print_io[1]);
	++n_files_left;

	/* Set up file descriptor for non-blocking I/O */
	prep_fd(print_io[0], True);
	print_xio.fd = print_io[0];
	set_io(&print_xio);

	dvips_status = DVIPS_STAT_RUN;	/* running */
}


#if XAW

static void
printlog_append(str, len)
	_Xconst char	*str;
	int		len;
{
	static	XawTextBlock	block	= {0, 0, NULL, 0};

	block.ptr = (char *) str;
	block.length = len;
	block.format = XawFmt8Bit;
	while (XawTextReplace(printlog_text, printlog_length, printlog_length,
	  &block) != XawEditDone) {
	    int length;

	    XtVaGetValues(printlog_text, XtNlength, &length, NULL);
	    printlog_length = length;
	}
	printlog_length += len;
	XawTextSetInsertionPoint(printlog_text, printlog_length);
}

#else /* MOTIF */

static void
printlog_append(str, len)
	_Xconst char	*str;
	int		len;
{
	XmTextInsert(printlog_text, printlog_length, (char *) str);
	printlog_length += len;
	XtVaSetValues(printlog_text, XmNcursorPosition, printlog_length, NULL);
	XmTextShowPosition(printlog_text, printlog_length);
}

#endif

static void
read_from_dvips()
{
	int	bytes;
	char	line[80];

	for (;;) {
#if XAW
	    bytes = read(print_xio.fd, line, sizeof line);
#else
	    bytes = read(print_xio.fd, line, sizeof line - 1);
#endif
	    if (bytes < 0) {
		if (AGAIN_CONDITION)
		    break;
		perror("xdvi: read_from_dvips");
		break;
	    }

	    if (bytes == 0)
		break;
	    else {
#if MOTIF
		line[bytes] = '\0';
#endif
		printlog_append(line, bytes);
	    }
	}
}

/* ARGSUSED */
static	void
cb_dvips_cancel(w, client_data, call_data)
	Widget		w;
	XtPointer	client_data;
	XtPointer	call_data;
{
	if (dvips_status != DVIPS_STAT_RUN)
	    return;	/* How did we get here? */

	kill(print_child.pid, dvips_sig);
	dvips_sig = SIGKILL;
	printlog_append("^C", 2);
}

static void
dvips_ended(status)
{
	char	*str;
	int	ms;

	read_from_dvips();
	clear_io(&print_xio);
	(void) close(print_xio.fd);
	++n_files_left;

	if (WIFEXITED(status)) {
	    if (WEXITSTATUS(status) == 0) {
		printlog_append("Done.\n", 6);
		str = NULL;
	    }
	    else
		str = mprintf("\nPrint process returned exit code %d.\n",
		  WEXITSTATUS(status));
	}
	else if (WIFSIGNALED(status))
	    str = mprintf("\nPrint process terminated by signal %d.\n",
	      WTERMSIG(status));
	else
	    str = mprintf("\nPrint process returned unknown status 0x%x.\n",
	      status);

	ms = resource.dvips_hang;
	if (str != NULL) {
	    ms = resource.dvips_fail_hang;
	    printlog_append(str, strlen(str));
	    free(str);
	}

	if (ms > 0) {
	    set_timer(&dvips_timer, ms);
	    dvips_status = DVIPS_STAT_WAIT;
	}
	else {
	    printlog_act_keep(NULL, NULL, NULL, NULL);
	    dvips_status = DVIPS_STAT_NONE;
	}

	XtSetSensitive(printlog_close, True);
	XtSetSensitive(printlog_cancel, False);
}

static void
dvips_pop_down()
{
	XtPopdown(printlog_shell);
	printlog_active = False;
}

static void
dvips_alarm(arg)
	struct xtimer	*arg;
{
#if XAW
	Boolean	state;
#endif

#if XAW
	XtVaGetValues(printlog_keep, XtNstate, &state, NULL);
	if (!state)
	    dvips_pop_down();
#else /* MOTIF */
	if (!XmToggleButtonGadgetGetState(printlog_keep))
	    dvips_pop_down();
#endif
	dvips_status = DVIPS_STAT_NONE;
}

/* ARGSUSED */
static	void
cb_dvips_close(w, client_data, call_data)
	Widget		w;
	XtPointer	client_data;
	XtPointer	call_data;
{
	if (dvips_status == DVIPS_STAT_RUN)
	    return;	/* How did we get here? */

	if (dvips_status == DVIPS_STAT_WAIT) {
	    dvips_status = DVIPS_STAT_NONE;
	    cancel_timer(&dvips_timer);
	}

	dvips_pop_down();
}

/* ARGSUSED */
static	void
cb_dvips_keep(w, client_data, call_data)
	Widget		w;
	XtPointer	client_data;
	XtPointer	call_data;
{
#if XAW
	Boolean	state;
#endif

	if (dvips_status != DVIPS_STAT_NONE)
	    return;	/* Nothing to do */

#if XAW
	XtVaGetValues(printlog_keep, XtNstate, &state, NULL);
	if (!state)
	    dvips_pop_down();
#else
	if (!XmToggleButtonGadgetGetState(printlog_keep))
	    dvips_pop_down();
#endif
}

/* ARGSUSED */
static	void
cb_dvips_delete(w, client_data, call_data)
	Widget		w;
	XtPointer	client_data;
	XtPointer	call_data;
{
	if (dvips_status == DVIPS_STAT_RUN)
	    cb_dvips_cancel(w, client_data, call_data);
	else
	    cb_dvips_close(w, client_data, call_data);
}

/* ARGSUSED */
static	void
printlog_act_close P4C(Widget, w, XEvent *, event, String *, params,
  Cardinal *, num_params)
{
	cb_dvips_close(NULL, NULL, NULL);
}

/* ARGSUSED */
static	void
printlog_act_keep P4C(Widget, w, XEvent *, event, String *, params,
  Cardinal *, num_params)
{
#if XAW
	XtVaSetValues(printlog_keep, XtNstate, True, NULL);
#else /* MOTIF */
	XmToggleButtonGadgetSetState(printlog_keep, True, False);
#endif
}

/* ARGSUSED */
static	void
printlog_act_unkeep P4C(Widget, w, XEvent *, event, String *, params,
  Cardinal *, num_params)
{
#if XAW
	XtVaSetValues(printlog_keep, XtNstate, False, NULL);
#else /* MOTIF */
	XmToggleButtonGadgetSetState(printlog_keep, False, False);
#endif
	if (dvips_status == DVIPS_STAT_NONE)
	    dvips_pop_down();
}

/* ARGSUSED */
static	void
printlog_act_cancel P4C(Widget, w, XEvent *, event, String *, params,
  Cardinal *, num_params)
{
	cb_dvips_cancel(NULL, NULL, NULL);
}
