	/* This is a lex(1) file, see http://dinosaur.compilertools.net/
	* or http://en.wikipedia.org/wiki/Lex_programming_tool .
	*
	* Compilation on UNIX systems is done by
	*     make ansel2uni
	* On other systems one may need to call lex or flex and cc explicitly:
	*     lex -o ansel2uni.c ansel2uni.l
	*     cc [-s] [-O] [-D U_TRANSF_OS_LN] [-D GEDCOM_EXT] [-D GEDCOM_LDS_EXT] [-D MARC_EXT] -o wileyEndNoteFilt ansel2uni.c -ll
	*
	* The executable works as a filter (copies stdin to stdout) and converts
	* a text stream encoded in 8-bit ANSEL (Z39-47-1993, ISO-IR-231) to UTF-8 unicode.
	* If the preprocessor symbol U_TRANSF_OS_LN has been defined, the CR-LF
	* combination of Unixes or DOS are modified to the Unicode
	* line separator symbol on output, else they are copied unchanged.
	*
	* If the preprocessor symbol GEDCOM_EXT is defined, some Gedcom-5.5 extensions
	* of the ANSEL standard are implemented. With the current implementation
	* this only means support of the German sharp s. If the slash
	* modifier that strikes through characters or of graphical symbols
	* of open or filled boxes are also needed, GEDCOM_LDS_EXT should be defined.
	*
	* If the preprocessor symbol MARC_EXT is defined, the MARC-8 extensions
	* of the ANSEL standard are implemented, as described in
	* http://lcweb2.loc.gov/diglib/codetables/45.html .
	*
	* Templates of use:
	*   cat infile.ans | ansel2uni > infile.uni
	* For use in HTML/XML pages, recoding with the &#x00000; style
	* can be done via http://billposer.org/Software/uni2ascii.html :
	*   cat infile.ans | ansel2uni | uni2ascii -q -a Q -a H > infile.xml
	*
	* http://en.wikipedia.org/wiki/GEDCOM
	* http://marcpm.sourceforge.net/MARC/Charset.html
	*
	* @author Richard J. Mathar, 2009-01-13
	* @since 2209-01-18 use yyout instead of stdout. Implemented MARC_EXT. Use graph for alpha char class.
	* @see http://www.uniocode.org/versions/Unicode5.1.0/
	*/
%{
#include <stdio.h>

/* Unicode ranges U+0000 to UFFFF
* @param hi the high (first) byte of the code point
* @param lo the low (2nd byte) of the code point 
* @see D92 in the Unicode Standard, Table 3-6
* @see http://en.wikipedia.org/wiki/UTF-8
*/
void utf3byte(const unsigned char hi, const unsigned char lo)
{
	const unsigned long c = 256*hi+lo ;
	if ( c < 128)
		fprintf(yyout,"%c",lo) ;
	else if ( c < 2048 )
	{
			/* mask lower 6 bits and prefix 10, second byte */
		const unsigned char b2 = 128 + ( c & 63 );
			/* remove lower 6 bits and prefix 110, first byte */
		const unsigned char b1 = 192 + ( c >> 6 ) ;
		fprintf(yyout,"%c%c",b1,b2) ;
	}
	else
	{
			/* mask lower 6 bits and prefix 10, third byte */
		const unsigned char b3 = 128 + ( c & 63 );
			/* remove lower 6 bits and prefix 10 */
		const unsigned char b2 = 128 + ( ( c & 4032 ) >> 6 ) ;
			/* remove lower 12 bits and prefix 1110 */
		const unsigned char b1 = 224 + ( (c & 61440 ) >> 12 ) ;
		fprintf(yyout,"%c%c%c",b1,b2,b3) ;
	}
}

/* Unicode ranges U+0080 to U+00FF.
* @param c the low-byte of the unicode point in the range  0x00 to 0x7f
* @see http://en.wikipedia.org/wiki/UTF-8
*/
void utf2byte(const unsigned char c)
{
#if 0
	/* explicit implementation */
	const unsigned char b2 = 128 + ( c & 63 );
	const unsigned char b1 = 192 + ( ( c & 192 ) >> 6 ) ;
	fprintf(yyout,"%c%c",b1,b2) ;
#else
	/* lazy implementation */
	utf3byte('\0',c) ;
#endif
}

#ifdef U_TRANSF_OS_NL
/* Put the LS, U+2028
* @param crlf the line feed in unix or dos representation
* @todo
*/
void newl(char *crlf)
{
	utf3byte('\x20','\x28') ;
}
#else
/* 
* @param crlf the line feed in unix or dos representation
*/
void newl(char *crlf)
{
	fprintf(yyout,"%s",crlf) ;
}
#endif

%}

ANSEL_VIETO	\xac
ANSEL_VIETU	\xad
ANSEL_VIETo	\xbc
ANSEL_VIETu	\xbd

ANSEL_LOWRISINGTONE	\xe0
ANSEL_GRAVEACCENT	\xe1
ANSEL_ACUTEACCENT	\xe2
ANSEL_CIRCUMFLEXACC	\xe3
ANSEL_TILDE	\xe4
ANSEL_MACRON	\xe5
ANSEL_BREVE	\xe6
ANSEL_DOTABOVE	\xe7
ANSEL_DIARESIS	\xe8
ANSEL_CARON	\xe9
ANSEL_CIRCLEABOVE	\xea
ANSEL_LIGATURELEFT	\xeb
ANSEL_LIGATURERIGHT	\xec
ANSEL_HIGHCOMMAOFFC	\xed
ANSEL_DOUBLEACUTEACC	\xee
ANSEL_CANDRABINDU	\xef
ANSEL_CEDILLA	\xf0
ANSEL_RIGHTHOOK	\xf1
ANSEL_DOTBELOW	\xf2
ANSEL_DOUBLEDOTBELOW	\xf3
ANSEL_CIRCLEBELOW	\xf4
ANSEL_DOUBLEUNDERSC	\xf5
ANSEL_UNDERSCORE	\xf6
ANSEL_LEFTHOOK	\xf7
ANSEL_RIGHTCEDILLA	\xf8
ANSEL_HALFCIRCLBELOW	\xf9
ANSEL_DOUBLETILDLEFT	\xfa
ANSEL_DOUBLETILDRIGH	\xfb
ANSEL_HIGHCOMMACENT	\xfe

	/* reserved for control characters */
ANSEL_RES_C1	[\x80-\x9f]

	/* corners (reserved) */
ANSEL_CORN	\xa0|\xff

	/* Gedcom 5.5 extension: German sharp s */
GEDCOM55_SZ	\xcf

	/* rumors but not standard, see http://homepages.rootsweb.ancestry.com/~pmcbride/gedcom/55gcappd.htm#S2 */
GEDCOM55_EMPTYBOX	\xbe
GEDCOM55_BLACKBOX	\xbf
GEDCOM55_EMIDDLE	\xcd
GEDCOM55_OMIDDLE	\xce
GEDCOM55_SLASH	\xfc

	/* Strict ANSEL: reserved for future standardization, but not GEDCOM */
ANSEL_RES_FUT	\xaf|\xbb|\xbe|\xbf|[\xc7-\xdf]|\xfc|\xfd

	/* not sure whether ANSEL allows a double grave to be represented as two accents */
ANSEL_DGRAVE	{ANSEL_GRAVEACCENT}{ANSEL_GRAVEACCENT}
ANSEL_DTILDE	{ANSEL_TILDE}{ANSEL_TILDE}

%%

\x88	{
#ifdef MARC_EXT
		utf2byte('\x98') ; /* START OF STRING U+0098 */
#else
		ECHO ;
#endif
	}

\x89	{
#ifdef MARC_EXT
		utf2byte('\x9c') ; /* STRING TERMINATOR U+009C */
#else
		ECHO ;
#endif
	}

\x8d	{
#ifdef MARC_EXT
		utf3byte('\x20','\x0d') ; /* ZERO WIDTH JOINER U+200D */
#else
		ECHO ;
#endif
	}

\x8e	{
#ifdef MARC_EXT
		utf3byte('\x20','\x0c') ; /* ZERO WIDTH NON-JOINER U+200C */
#else
		ECHO ;
#endif
	}

	/* C1 controls and latin supplement */

	/* utf2byte('\xa0') ; /* unassigned xxxxxxxxxxxxxxxxxxxxxxx U+00A0 */
\xc6	utf2byte('\xa1') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00A1 */
	/* utf2byte('\xa2') ; /* CENT SIGN                          U+00A2 */
\xb9	utf2byte('\xa3') ; /* POUND SIGN                         U+00A3 */
	/* utf2byte('\xa4') ; /* CURRENCY SIGN                      U+00A4 */
	/* utf2byte('\xa5') ; /* YEN SIGN                           U+00A5 */
	/* utf2byte('\xa6') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00A6 */
	/* utf2byte('\xa7') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00A7 */
	/* utf2byte('\xa8') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00A8 */
\xc3	utf2byte('\xa9') ; /* COPYRIGH SIGN                     U+00A9 */
	/* utf2byte('\xaa') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00AA */
	/* utf2byte('\xab') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00AB */
	/* utf2byte('\xac') ; /* NOT SIGN                           U+00AC */
	/* utf2byte('\xad') ; /* SOPFTHYPHEN                        U+00AD */
\xaa	utf2byte('\xae') ; /* REGISTRED SIGN                     U+00AE */
	/* utf2byte('\xaf') ; /* MACRON                             U+00AF */

\xc0	utf2byte('\xb0') ; /* DEGREE SIGN                        U+00B0 */
\xab	utf2byte('\xb1') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00B1 */
	/* utf2byte('\xb2') ; /* SUPER SCRIPT TWO                   U+00B2 */
	/* utf2byte('\xb3') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00B3 */
	/* utf2byte('\xb4') ; /* ACUTE ACCENT                       U+00B4 */
	/* utf2byte('\xb5') ; /* MICRO SIGN                         U+00B5 */
	/* utf2byte('\xb6') ; /* PILCROW SIGN                       U+00B6 */
\xa8	utf2byte('\xb7') ; /* MIDDLE DOT                         U+00B7 */
	/* utf2byte('\xb8') ; /* CEDILLA                            U+00B8 */
	/* utf2byte('\xb9') ; /* SUPERSCRIPT ONE                    U+00B9 */
	/* utf2byte('\xba') ; /* MASCULINE ORIDNAL INDICATOR        U+00BA */
	/* utf2byte('\xbb') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00BB */
	/* utf2byte('\xbc') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00BC */
	/* utf2byte('\xbd') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00BD */
	/* utf2byte('\xbe') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00BE */
\xc5	utf2byte('\xbf') ; /* INVERTED QUESTION MARK             U+00BF */

{ANSEL_GRAVEACCENT}A	utf2byte('\xc0') ; /* A WITH ACUTE, LATIN CAPITAL LETTER U+00C0 */
{ANSEL_ACUTEACCENT}A	utf2byte('\xc1') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00C1 */
{ANSEL_CIRCUMFLEXACC}A	utf2byte('\xc2') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00C2 */
{ANSEL_TILDE}A	utf2byte('\xc3') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00C3 */
{ANSEL_DIARESIS}A	utf2byte('\xc4') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00C4 */
{ANSEL_CIRCLEABOVE}A	utf2byte('\xc5') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00C5 */
\xa5	utf2byte('\xc6') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00C6 */
{ANSEL_CEDILLA}C	utf2byte('\xc7') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00C7 */
{ANSEL_GRAVEACCENT}E	utf2byte('\xc8') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00C8 */
{ANSEL_ACUTEACCENT}E	utf2byte('\xc9') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00C9 */
{ANSEL_CIRCUMFLEXACC}E	utf2byte('\xca') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00CA */
{ANSEL_DIARESIS}E	utf2byte('\xcb') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00CB */
{ANSEL_GRAVEACCENT}I	utf2byte('\xcc') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00CC */
{ANSEL_ACUTEACCENT}I	utf2byte('\xcd') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00CD */
{ANSEL_CIRCUMFLEXACC}I	utf2byte('\xce') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00CE */
{ANSEL_DIARESIS}I	utf2byte('\xcf') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00CF */

{ANSEL_TILDE}N	utf2byte('\xd1') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00D1 */
{ANSEL_GRAVEACCENT}O	utf2byte('\xd2') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00D2 */
{ANSEL_ACUTEACCENT}O	utf2byte('\xd3') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00D3 */
{ANSEL_CIRCUMFLEXACC}O	utf2byte('\xd4') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00D4 */
{ANSEL_TILDE}O	utf2byte('\xd5') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00D5 */
\xa8O	utf2byte('\xd6') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00D6 */
	/* utf2byte('\xd7') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00D7 */
\xa2	utf2byte('\xd8') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00D8 */
{ANSEL_GRAVEACCENT}U	utf2byte('\xd9') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00D9 */
{ANSEL_ACUTEACCENT}U	utf2byte('\xda') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00DA */
{ANSEL_CIRCUMFLEXACC}U	utf2byte('\xdb') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00DB */
{ANSEL_DIARESIS}U	utf2byte('\xdc') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00DC */
{ANSEL_ACUTEACCENT}Y	utf2byte('\xdd') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00DD */
\xa4	utf2byte('\xde') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00DE */
	/* utf2byte('\xdf') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00DF */

{ANSEL_GRAVEACCENT}a	utf2byte('\xe0') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00E0 */
{ANSEL_ACUTEACCENT}a	utf2byte('\xe1') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00E1 */
{ANSEL_CIRCUMFLEXACC}a	utf2byte('\xe2') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00E2 */
{ANSEL_TILDE}a	utf2byte('\xe3') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00E3 */
{ANSEL_DIARESIS}a	utf2byte('\xe4') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00E4 */
{ANSEL_CIRCLEABOVE}a	utf2byte('\xe5') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00E5 */
\xb5	utf2byte('\xe6') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00E6 */
{ANSEL_CEDILLA}c	utf2byte('\xe7') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00E7 */
{ANSEL_GRAVEACCENT}e	utf2byte('\xe8') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00E8 */
{ANSEL_ACUTEACCENT}e	utf2byte('\xe9') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00E9 */
{ANSEL_CIRCUMFLEXACC}e	utf2byte('\xea') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00EA */
{ANSEL_DIARESIS}e	utf2byte('\xeb') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00EB */
{ANSEL_GRAVEACCENT}i	utf2byte('\xec') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00EC */
{ANSEL_ACUTEACCENT}i	utf2byte('\xed') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00ED */
{ANSEL_CIRCUMFLEXACC}i	utf2byte('\xee') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00EE */
{ANSEL_DIARESIS}i	utf2byte('\xef') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00EF */

\xba	utf2byte('\xf0') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00F0 */
{ANSEL_TILDE}n	utf2byte('\xf1') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00F1 */
{ANSEL_GRAVEACCENT}o	utf2byte('\xf2') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00F2 */
{ANSEL_ACUTEACCENT}o	utf2byte('\xf3') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00F3 */
{ANSEL_CIRCUMFLEXACC}o	utf2byte('\xf4') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00F4 */
{ANSEL_TILDE}o	utf2byte('\xf5') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00F5 */
{ANSEL_DIARESIS}o	utf2byte('\xf6') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00F6 */
	/* utf2byte('\xf7') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00F7 */
\xb2	utf2byte('\xf8') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00F8 */
{ANSEL_GRAVEACCENT}u	utf2byte('\xf9') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00F9 */
{ANSEL_ACUTEACCENT}u	utf2byte('\xfa') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00FA */
{ANSEL_CIRCUMFLEXACC}u	utf2byte('\xfb') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00FB */
{ANSEL_DIARESIS}u	utf2byte('\xfc') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00FC */
{ANSEL_ACUTEACCENT}y	utf2byte('\xfd') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00FD */
\xb4	utf2byte('\xfe') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00FE */
{ANSEL_DIARESIS}y	utf2byte('\xff') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+00FF */

	/* Latin-extended A */

{ANSEL_MACRON}A	utf3byte('\x01','\x00') ; /* LATIN CAPITAL LETTER A WITH MACRON U+0100 */
{ANSEL_MACRON}a	utf3byte('\x01','\x01') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0101 */
{ANSEL_BREVE}A	utf3byte('\x01','\x02') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0102 */
{ANSEL_BREVE}a	utf3byte('\x01','\x03') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0103 */
{ANSEL_RIGHTHOOK}A	utf3byte('\x01','\x04') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0104 */
{ANSEL_RIGHTHOOK}a	utf3byte('\x01','\x05') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0105 */
{ANSEL_ACUTEACCENT}C	utf3byte('\x01','\x06') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0106 */
{ANSEL_ACUTEACCENT}c	utf3byte('\x01','\x07') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0107 */
{ANSEL_CIRCUMFLEXACC}C	utf3byte('\x01','\x08') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0108 */
{ANSEL_CIRCUMFLEXACC}c	utf3byte('\x01','\x09') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0109 */
{ANSEL_DOTABOVE}C	utf3byte('\x01','\x0a') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+010a */
{ANSEL_DOTABOVE}c	utf3byte('\x01','\x0b') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+010b */
{ANSEL_CARON}C	utf3byte('\x01','\x0c') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+010c */
{ANSEL_CARON}c	utf3byte('\x01','\x0d') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+010d */
{ANSEL_CARON}D	utf3byte('\x01','\x0e') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+010e */
{ANSEL_CARON}d	utf3byte('\x01','\x0f') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+010f */

	/* could this rather be U+00D0, LATIN CAPITAL LETTER ETH ? */
\xa3	utf3byte('\x01','\x10') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0110 */
\xb3	utf3byte('\x01','\x11') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0111 */
{ANSEL_MACRON}E	utf3byte('\x01','\x12') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0112 */
{ANSEL_MACRON}e	utf3byte('\x01','\x13') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0113 */
{ANSEL_BREVE}E	utf3byte('\x01','\x14') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0114 */
{ANSEL_BREVE}e	utf3byte('\x01','\x15') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0115 */
{ANSEL_DOTABOVE}E	utf3byte('\x01','\x16') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0116 */
{ANSEL_DOTABOVE}e	utf3byte('\x01','\x17') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0117 */
{ANSEL_RIGHTHOOK}E	utf3byte('\x01','\x18') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0118 */
{ANSEL_RIGHTHOOK}e	utf3byte('\x01','\x19') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0119 */
{ANSEL_CARON}E	utf3byte('\x01','\x1a') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+011a */
{ANSEL_CARON}e	utf3byte('\x01','\x1b') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+011b */
{ANSEL_CIRCUMFLEXACC}G	utf3byte('\x01','\x1c') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+011c */
{ANSEL_CIRCUMFLEXACC}g	utf3byte('\x01','\x1d') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+011d */
{ANSEL_BREVE}G	utf3byte('\x01','\x1e') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+011e */
{ANSEL_BREVE}g	utf3byte('\x01','\x1f') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+011f */

{ANSEL_DOTABOVE}G	utf3byte('\x01','\x20') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0120 */
{ANSEL_DOTABOVE}g	utf3byte('\x01','\x21') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0121 */
{ANSEL_CEDILLA}G	utf3byte('\x01','\x22') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0122 */
{ANSEL_CEDILLA}g	utf3byte('\x01','\x23') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0123 */
{ANSEL_CIRCUMFLEXACC}H	utf3byte('\x01','\x24') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0124 */
{ANSEL_CIRCUMFLEXACC}h	utf3byte('\x01','\x25') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0125 */
	/* utf3byte('\x01','\x26') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0126 */
	/* utf3byte('\x01','\x27') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0127 */
{ANSEL_TILDE}I	utf3byte('\x01','\x28') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0128 */
{ANSEL_TILDE}i	utf3byte('\x01','\x29') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0129 */
{ANSEL_MACRON}I	utf3byte('\x01','\x2a') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+012a */
{ANSEL_MACRON}i	utf3byte('\x01','\x2b') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+012b */
{ANSEL_BREVE}I	utf3byte('\x01','\x2c') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+012c */
{ANSEL_BREVE}i	utf3byte('\x01','\x2d') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+012d */
{ANSEL_RIGHTHOOK}I	utf3byte('\x01','\x2e') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+012e */
{ANSEL_RIGHTHOOK}i	utf3byte('\x01','\x2f') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+012f */

{ANSEL_DOTABOVE}I	utf3byte('\x01','\x30') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0130 */
\xb8	utf3byte('\x01','\x31') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0131 */
IJ	utf3byte('\x01','\x32') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0132 */
ij	utf3byte('\x01','\x33') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0133 */
{ANSEL_CIRCUMFLEXACC}J	utf3byte('\x01','\x34') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0134 */
{ANSEL_CIRCUMFLEXACC}j	utf3byte('\x01','\x35') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0135 */
{ANSEL_CEDILLA}K	|
{ANSEL_LEFTHOOK}K	utf3byte('\x01','\x36') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0136 */
{ANSEL_CEDILLA}k	|
{ANSEL_LEFTHOOK}k	utf3byte('\x01','\x37') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0137 */
	/* utf3byte('\x01','\x38') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0138 */
{ANSEL_ACUTEACCENT}L	utf3byte('\x01','\x39') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0139 */
{ANSEL_ACUTEACCENT}l	utf3byte('\x01','\x3a') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+013a */
{ANSEL_CEDILLA}L	|
{ANSEL_LEFTHOOK}L	utf3byte('\x01','\x3b') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+013b */
{ANSEL_CEDILLA}l	|
{ANSEL_LEFTHOOK}l	utf3byte('\x01','\x3c') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+013c */
{ANSEL_CARON}L	utf3byte('\x01','\x3d') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+013d */
{ANSEL_CARON}l	utf3byte('\x01','\x3e') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+013e */
	/* utf3byte('\x01','\x3f') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+013f */

	/* utf3byte('\x01','\x40') ; /* LATIN SMALL LETTER L WITH MIDDLE DOT+0140 */
\xa1	utf3byte('\x01','\x41') ; /* LATIN CAPITAL LETTER L WITH STROKE U+0141 */
\xb1	utf3byte('\x01','\x42') ; /* LATIN SMALL LETTER L WITH STROKE U+0142 */
{ANSEL_ACUTEACCENT}N	utf3byte('\x01','\x43') ; /* LATIN CAPITAL LETTER N WITH ACUTE  U+0143 */
{ANSEL_ACUTEACCENT}n	utf3byte('\x01','\x44') ; /* LATIN SMALL LETTER N WITH ACUTE    U+0144 */
{ANSEL_CEDILLA}N	|
{ANSEL_LEFTHOOK}N	utf3byte('\x01','\x45') ; /* LATIN CAPITAL LETTER N WITH CEDILLA U+0145 */
{ANSEL_CEDILLA}n	|
{ANSEL_LEFTHOOK}n	utf3byte('\x01','\x46') ; /* LATIN SMALL LETTER N WITH CEDILLA  U+0146 */
{ANSEL_CARON}N	utf3byte('\x01','\x47') ; /* LATIN CAPITAL LETTER N WITH CARON U+0147 */
{ANSEL_CARON}n	utf3byte('\x01','\x48') ; /* LATIN SMALL LETTER N WITH CARON    U+0148 */
	/* utf3byte('\x01','\x49') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0149 */
	/* utf3byte('\x01','\x4a') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+014a */
	/* utf3byte('\x01','\x4b') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+014b */
{ANSEL_MACRON}O	utf3byte('\x01','\x4c') ; /* LATIN CAPITAL LETTER O WITH MACRON U+014c */
{ANSEL_MACRON}o	utf3byte('\x01','\x4d') ; /* LATIN SMALL LETTER O WITH MACRON U+014d */
{ANSEL_BREVE}O	utf3byte('\x01','\x4e') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+014e */
{ANSEL_BREVE}o	utf3byte('\x01','\x4f') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+014f */

{ANSEL_DOUBLEACUTEACC}O	utf3byte('\x01','\x50') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0150 */
{ANSEL_DOUBLEACUTEACC}o	utf3byte('\x01','\x51') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0151 */
\xa6	utf3byte('\x01','\x52') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0152 */
\xb6	utf3byte('\x01','\x53') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0153 */
{ANSEL_ACUTEACCENT}R	utf3byte('\x01','\x54') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0154 */
{ANSEL_ACUTEACCENT}r	utf3byte('\x01','\x55') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0155 */
{ANSEL_CEDILLA}R	|
{ANSEL_LEFTHOOK}R	utf3byte('\x01','\x56') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0156 */
{ANSEL_CEDILLA}r	|
{ANSEL_LEFTHOOK}r	utf3byte('\x01','\x57') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0157 */
{ANSEL_CARON}R	utf3byte('\x01','\x58') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0158 */
{ANSEL_CARON}r	utf3byte('\x01','\x59') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0159 */
{ANSEL_ACUTEACCENT}S	utf3byte('\x01','\x5a') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+015a */
{ANSEL_ACUTEACCENT}s	utf3byte('\x01','\x5b') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+015b */
{ANSEL_CIRCUMFLEXACC}S	utf3byte('\x01','\x5c') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+015c */
{ANSEL_CIRCUMFLEXACC}s	utf3byte('\x01','\x5d') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+015d */
{ANSEL_CEDILLA}S	utf3byte('\x01','\x5e') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+015e */
{ANSEL_CEDILLA}s	utf3byte('\x01','\x5f') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+015f */

{ANSEL_CARON}S	utf3byte('\x01','\x60') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0160 */
{ANSEL_CARON}s	utf3byte('\x01','\x61') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0161 */
{ANSEL_CEDILLA}T	utf3byte('\x01','\x62') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0162 */
{ANSEL_CEDILLA}t	utf3byte('\x01','\x63') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0163 */
{ANSEL_CARON}T	utf3byte('\x01','\x64') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0164 */
{ANSEL_CARON}t	utf3byte('\x01','\x65') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0165 */
	/* utf3byte('\x01','\x66') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0166 */
	/* utf3byte('\x01','\x67') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0167 */
{ANSEL_TILDE}U	utf3byte('\x01','\x68') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0168 */
{ANSEL_TILDE}u	utf3byte('\x01','\x69') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0169 */
{ANSEL_MACRON}U	utf3byte('\x01','\x6a') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+016a */
{ANSEL_MACRON}u	utf3byte('\x01','\x6b') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+016b */
{ANSEL_BREVE}U	utf3byte('\x01','\x6c') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+016c */
{ANSEL_BREVE}u	utf3byte('\x01','\x6d') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+016d */
{ANSEL_CIRCLEABOVE}U	utf3byte('\x01','\x6e') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+016e */
{ANSEL_CIRCLEABOVE}u	utf3byte('\x01','\x6f') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+016f */

{ANSEL_DOUBLEACUTEACC}U	utf3byte('\x01','\x70') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0170 */
{ANSEL_DOUBLEACUTEACC}u	utf3byte('\x01','\x71') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0171 */
{ANSEL_RIGHTHOOK}U	utf3byte('\x01','\x72') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0172 */
{ANSEL_RIGHTHOOK}u	utf3byte('\x01','\x73') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0173 */
{ANSEL_CIRCUMFLEXACC}W	utf3byte('\x01','\x74') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0174 */
{ANSEL_CIRCUMFLEXACC}w	utf3byte('\x01','\x75') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0175 */
{ANSEL_CIRCUMFLEXACC}Y	utf3byte('\x01','\x76') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0176 */
{ANSEL_CIRCUMFLEXACC}y	utf3byte('\x01','\x77') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0177 */
{ANSEL_DIARESIS}Y	utf3byte('\x01','\x78') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0178 */
{ANSEL_ACUTEACCENT}Z	utf3byte('\x01','\x79') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0179 */
{ANSEL_ACUTEACCENT}z	utf3byte('\x01','\x7a') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+017a */
{ANSEL_DOTABOVE}Z	utf3byte('\x01','\x7b') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+017b */
{ANSEL_DOTABOVE}z	utf3byte('\x01','\x7c') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+017c */
{ANSEL_CARON}Z	utf3byte('\x01','\x7d') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+017d */
{ANSEL_CARON}z	utf3byte('\x01','\x7e') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+017e */
	/* utf3byte('\x01','\x7f') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+017f */

	/* end of Latin Extended A */

	/* Latin Extended B */

	/* utf3byte('\x01','\x80') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0170 */
	/* utf3byte('\x01','\x81') ; /* xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx U+0171 */
{ANSEL_MACRON}B	utf3byte('\x01','\x82') ; /* LATIN CAPITAL LETTER B WITH TOPBAR U+0182 */
{ANSEL_MACRON}b	utf3byte('\x01','\x83') ; /* LATIN SMALL LETTER B WITH TOPBAR U+0183 */

{ANSEL_HIGHCOMMAOFFC}O	utf3byte('\x01','\xa0') ; /* LATIN CAPITAL LETTER O WITH HORN U+01A0 */
{ANSEL_HIGHCOMMAOFFC}o	utf3byte('\x01','\xa1') ; /* LATIN SMALL LETTER O WITH HORN U+01A1 */
{ANSEL_HIGHCOMMAOFFC}U	utf3byte('\x01','\xaf') ; /* LATIN CAPITAL LETTER U WITH HORN U+01AF */
{ANSEL_HIGHCOMMAOFFC}u	utf3byte('\x01','\x80') ; /* LATIN SMALL LETTER U WIHT HORN U+0180 */

D{ANSEL_CARON}Z	utf3byte('\x01','\xc4') ; /* LATIN CAPITAL LETTER DZ WITH CARON U+01C4 */
D{ANSEL_CARON}z	utf3byte('\x01','\xc5') ; /* LATIN CAPITAL LETTER D WITH SMALL LETTER Z WITH CARON U+01C5 */
d{ANSEL_CARON}z	utf3byte('\x01','\xc6') ; /* LATIN SMALL LETTER DZ WITH CARON U+01C6 */
LJ	utf3byte('\x01','\xc7') ; /* LATIN CAPITAL LETTER LJ U+01C7 */
Lj	utf3byte('\x01','\xc8') ; /* LATIN CAPITAL LETTER L WITH SMALL LTTER J U+01C8  */
	/* lj	utf3byte('\x01','\xc9') ; LATIN CAPITAL LETTER L WITH SMALL LTTER J U+01C9  */
NJ	utf3byte('\x01','\xca') ; /* LATIN CAPITAL LETTER NJ U+01CA */
Nj	utf3byte('\x01','\xcb') ; /* LATIN CAPITAL LETTER N WITH SMALL LTTER J U+01CB */
	/* nj	utf3byte('\x01','\xcc') ; LATIN SMALL LETTER NJ U+01CC */
{ANSEL_CARON}A	utf3byte('\x01','\xcd') ;	/* U+01CD */
{ANSEL_CARON}a	utf3byte('\x01','\xce') ;	/* U+01CE */
{ANSEL_CARON}I	utf3byte('\x01','\xcf') ;	/* U+01CF */
{ANSEL_CARON}i	utf3byte('\x01','\xd0') ;	/* U+01D0 */
{ANSEL_CARON}O	utf3byte('\x01','\xd1') ;	/* U+01D1 */
{ANSEL_CARON}o	utf3byte('\x01','\xd2') ;	/* U+01D2 */
{ANSEL_CARON}U	utf3byte('\x01','\xd3') ;	/* U+01D3 */
{ANSEL_CARON}u	utf3byte('\x01','\xd4') ;	/* U+01D4 */

	/* see U+1E7A for the opposite order */
{ANSEL_MACRON}{ANSEL_DIARESIS}U	utf3byte('\x01','\xd5') ;	/* U+01D5 */
{ANSEL_MACRON}{ANSEL_DIARESIS}u	utf3byte('\x01','\xd6') ;	/* U+01D6 */

{ANSEL_ACUTEACCENT}{ANSEL_DIARESIS}U	|
{ANSEL_DIARESIS}{ANSEL_ACUTEACCENT}U	utf3byte('\x01','\xd7') ;	/* U+01D7 */

{ANSEL_ACUTEACCENT}{ANSEL_DIARESIS}u	|
{ANSEL_DIARESIS}{ANSEL_ACUTEACCENT}u	utf3byte('\x01','\xd8') ; 	/* U+01D8 */

{ANSEL_CARON}{ANSEL_DIARESIS}U	|
{ANSEL_DIARESIS}{ANSEL_CARON}U	utf3byte('\x01','\xd9') ;	/* U+01D9 */

{ANSEL_CARON}{ANSEL_DIARESIS}u	|
{ANSEL_DIARESIS}{ANSEL_CARON}u	utf3byte('\x01','\xda') ;	/* U+01DA */

{ANSEL_GRAVEACCENT}{ANSEL_DIARESIS}U	|
{ANSEL_DIARESIS}{ANSEL_GRAVEACCENT}U	utf3byte('\x01','\xdb') ;	/* U+01DB */

{ANSEL_GRAVEACCENT}{ANSEL_DIARESIS}u	|
{ANSEL_DIARESIS}{ANSEL_GRAVEACCENT}u	utf3byte('\x01','\xdc') ;	/* U+01DC */

{ANSEL_MACRON}{ANSEL_DIARESIS}A	|
{ANSEL_DIARESIS}{ANSEL_MACRON}A	utf3byte('\x01','\xde') ;	/* U+01DE */

{ANSEL_MACRON}{ANSEL_DIARESIS}a	|
{ANSEL_DIARESIS}{ANSEL_MACRON}a	utf3byte('\x01','\xdf') ;	/* U+01DF */

{ANSEL_MACRON}{ANSEL_DOTABOVE}A	|
{ANSEL_DOTABOVE}{ANSEL_MACRON}A	utf3byte('\x01','\xe0') ;	/* U+01E0 */

{ANSEL_MACRON}{ANSEL_DOTABOVE}a	|
{ANSEL_DOTABOVE}{ANSEL_MACRON}a	utf3byte('\x01','\xe1') ;	/* U+01E1 */

{ANSEL_MACRON}\xa5	utf3byte('\x01','\xe2') ;	/* U+01E2 */
{ANSEL_MACRON}\xb5	utf3byte('\x01','\xe3') ;	/* U+01E3 */
{ANSEL_CARON}G	utf3byte('\x01','\xe6') ;	/* U+01E6 */
{ANSEL_CARON}g	utf3byte('\x01','\xe7') ;	/* U+01E7 */
{ANSEL_CARON}K	utf3byte('\x01','\xe8') ;	/* U+01E8 */
{ANSEL_CARON}k	utf3byte('\x01','\xe9') ;	/* U+01E9 */
{ANSEL_RIGHTHOOK}O	utf3byte('\x01','\xea') ;	/* U+01EA */
{ANSEL_RIGHTHOOK}o	utf3byte('\x01','\xeb') ;	/* U+01EB */

{ANSEL_MACRON}{ANSEL_RIGHTHOOK}O	|
{ANSEL_RIGHTHOOK}{ANSEL_MACRON}O	utf3byte('\x01','\xec') ;	/* U+01EC */

{ANSEL_MACRON}{ANSEL_RIGHTHOOK}o	|
{ANSEL_RIGHTHOOK}{ANSEL_MACRON}o	utf3byte('\x01','\xed') ;	/* U+01ED */

{ANSEL_CARON}j	utf3byte('\x01','\xf0') ;	/* U+01F0 */

DZ	utf3byte('\x01','\xf1') ;	/* U+01F1 */
Dz	utf3byte('\x01','\xf2') ;	/* U+01F2 */
dz	utf3byte('\x01','\xf3') ;	/* U+01F3 */

{ANSEL_ACUTEACCENT}G	utf3byte('\x01','\xf4') ; 	/* U+01F4 */
{ANSEL_ACUTEACCENT}g	utf3byte('\x01','\xf5') ; 	/* U+01F5 */
{ANSEL_GRAVEACCENT}N	utf3byte('\x01','\xf8') ; 	/* U+01F8 */
{ANSEL_GRAVEACCENT}n	utf3byte('\x01','\xf9') ; 	/* U+01F9 */

{ANSEL_CIRCLEABOVE}{ANSEL_ACUTEACCENT}A	|
{ANSEL_ACUTEACCENT}{ANSEL_CIRCLEABOVE}A	utf3byte('\x01','\xfa') ;	/* U+01FA */

{ANSEL_CIRCLEABOVE}{ANSEL_ACUTEACCENT}a	|
{ANSEL_ACUTEACCENT}{ANSEL_CIRCLEABOVE}a	utf3byte('\x01','\xfb') ;	/* U+01FB */

{ANSEL_ACUTEACCENT}\xa5	utf3byte('\x01','\xfc') ;	/* U+01FC */
{ANSEL_ACUTEACCENT}\xb5	utf3byte('\x01','\xfd') ;	/* U+01FD */
{ANSEL_ACUTEACCENT}\xa2	utf3byte('\x01','\xfe') ;	/* U+01FE */
{ANSEL_ACUTEACCENT}\xb2	utf3byte('\x01','\xff') ;	/* U+01FF */
{ANSEL_DGRAVE}A	utf3byte('\x02','\x00') ;	/* U+0200 */
{ANSEL_DGRAVE}a	utf3byte('\x02','\x01') ;	/* U+0201 */
{ANSEL_DGRAVE}E	utf3byte('\x02','\x04') ;	/* U+0204 */
{ANSEL_DGRAVE}e	utf3byte('\x02','\x05') ;	/* U+0205 */
{ANSEL_DGRAVE}I	utf3byte('\x02','\x08') ;	/* U+0208 */
{ANSEL_DGRAVE}i	utf3byte('\x02','\x09') ;	/* U+0209 */
{ANSEL_DGRAVE}O	utf3byte('\x02','\x0c') ;	/* U+020C */
{ANSEL_DGRAVE}o	utf3byte('\x02','\x0d') ;	/* U+020D */
{ANSEL_DGRAVE}R	utf3byte('\x02','\x10') ;	/* U+0210 */
{ANSEL_DGRAVE}r	utf3byte('\x02','\x11') ;	/* U+0211 */
{ANSEL_DGRAVE}U	utf3byte('\x02','\x14') ;	/* U+0214 */
{ANSEL_DGRAVE}u	utf3byte('\x02','\x15') ;	/* U+0215 */
{ANSEL_LEFTHOOK}S	utf3byte('\x02','\x18') ;	/* U+0218 */
{ANSEL_LEFTHOOK}s	utf3byte('\x02','\x19') ;	/* U+0219 */
{ANSEL_LEFTHOOK}T	utf3byte('\x02','\x1a') ;	/* U+021A */
{ANSEL_LEFTHOOK}t	utf3byte('\x02','\x1b') ;	/* U+021B */
{ANSEL_CARON}H	utf3byte('\x02','\x1e') ;	/* U+021E */
{ANSEL_CARON}h	utf3byte('\x02','\x1f') ;	/* U+021F */
{ANSEL_DOTABOVE}A	utf3byte('\x02','\x26') ;	/* U+0226 */
{ANSEL_DOTABOVE}a	utf3byte('\x02','\x27') ;	/* U+0227 */
{ANSEL_CEDILLA}E	utf3byte('\x02','\x28') ;	/* U+0228 */
{ANSEL_CEDILLA}e	utf3byte('\x02','\x29') ;	/* U+0229 */

{ANSEL_MACRON}{ANSEL_DIARESIS}O	|
{ANSEL_DIARESIS}{ANSEL_MACRON}O	utf3byte('\x02','\x2a') ;	/* U+022A */

{ANSEL_MACRON}{ANSEL_DIARESIS}o	|
{ANSEL_DIARESIS}{ANSEL_MACRON}o	utf3byte('\x02','\x2b') ;	/* U+022B */

{ANSEL_MACRON}{ANSEL_TILDE}O	|
{ANSEL_TILDE}{ANSEL_MACRON}O	utf3byte('\x02','\x2c') ;	/* U+022C */

{ANSEL_MACRON}{ANSEL_TILDE}o	|
{ANSEL_TILDE}{ANSEL_MACRON}o	utf3byte('\x02','\x2d') ;	/* U+022D */

{ANSEL_DOTABOVE}O	utf3byte('\x02','\x2e') ;	/* U+022E */
{ANSEL_DOTABOVE}o	utf3byte('\x02','\x2f') ;	/* U+022F */

{ANSEL_MACRON}{ANSEL_DOTABOVE}O	|
{ANSEL_DOTABOVE}{ANSEL_MACRON}O	utf3byte('\x02','\x30') ;	/* U+0230 */

{GEDCOM55_SLASH}A	{
#ifdef GEDCOM_LDSEXT
		utf3byte('\x02','\x3a') ;	/* U+023A */
#else
		fprintf(yyout,"%c",yytext[1]) ;
#endif
	}

{GEDCOM55_SLASH}C	{
#ifdef GEDCOM_LDSEXT
		utf3byte('\x02','\x3b') ;	/* U+023B */
#else
		fprintf(yyout,"%c",yytext[1]) ;
#endif
	}

{GEDCOM55_SLASH}c	{
#ifdef GEDCOM_LDSEXT
		utf3byte('\x02','\x3c') ;	/* U+023C */
#else
		fprintf(yyout,"%c",yytext[1]) ;
#endif
	}

{ANSEL_MACRON}{ANSEL_DOTABOVE}o	|
{ANSEL_DOTABOVE}{ANSEL_MACRON}o	utf3byte('\x02','\x31') ;	/* U+0231 */

{ANSEL_MACRON}Y	utf3byte('\x02','\x32') ;	/* U+0232 */
{ANSEL_MACRON}y	utf3byte('\x02','\x33') ;	/* U+0233 */

{GEDCOM55_SLASH}T	{
#ifdef GEDCOM_LDSEXT
		utf3byte('\x02','\x3e') ;	/* U+023E */
#else
		fprintf(yyout,"%c",yytext[1]) ;
#endif
	}

{GEDCOM55_SLASH}E	{
#ifdef GEDCOM_LDSEXT
		utf3byte('\x02','\x46') ;	/* U+0246 */
#else
		fprintf(yyout,"%c",yytext[1]) ;
#endif
	}

{GEDCOM55_SLASH}e	{
#ifdef GEDCOM_LDSEXT
		utf3byte('\x02','\x47') ;	/* U+0247 */
#else
		fprintf(yyout,"%c",yytext[1]) ;
#endif
	}

	/* end of Latin-extended B */

	/* Latin-extended Additional */
{ANSEL_CIRCLEBELOW}A	utf3byte('\x1e','\x00') ;	/* U+1E00 */
{ANSEL_CIRCLEBELOW}a	utf3byte('\x1e','\x01') ;	/* U+1E01 */
{ANSEL_DOTABOVE}B	utf3byte('\x1e','\x02') ;	/* U+1E02 */
{ANSEL_DOTABOVE}b	utf3byte('\x1e','\x03') ;	/* U+1E03 */
{ANSEL_DOTBELOW}B	utf3byte('\x1e','\x04') ;	/* U+1E04 */
{ANSEL_DOTBELOW}b	utf3byte('\x1e','\x05') ;	/* U+1E05 */
{ANSEL_UNDERSCORE}B	utf3byte('\x1e','\x06') ;	/* U+1E06 */
{ANSEL_UNDERSCORE}b	utf3byte('\x1e','\x07') ;	/* U+1E07 */

{ANSEL_ACUTEACCENT}{ANSEL_CEDILLA}C	|
{ANSEL_CEDILLA}{ANSEL_ACUTEACCENT}C	utf3byte('\x1e','\x08') ;	/* U+1E08 */

{ANSEL_ACUTEACCENT}{ANSEL_CEDILLA}c	|
{ANSEL_CEDILLA}{ANSEL_ACUTEACCENT}c	utf3byte('\x1e','\x09') ;	/* U+1E09 */

{ANSEL_DOTABOVE}D	utf3byte('\x1e','\x0a') ;	/* U+1E0A */
{ANSEL_DOTABOVE}d	utf3byte('\x1e','\x0b') ;	/* U+1E0B */
{ANSEL_DOTBELOW}D	utf3byte('\x1e','\x0c') ;	/* U+1E0C */
{ANSEL_DOTBELOW}d	utf3byte('\x1e','\x0d') ;	/* U+1E0D */
{ANSEL_UNDERSCORE}D	utf3byte('\x1e','\x0e') ;	/* U+1E0E */
{ANSEL_UNDERSCORE}d	utf3byte('\x1e','\x0f') ;	/* U+1E0F */
{ANSEL_CEDILLA}D	|
{ANSEL_LEFTHOOK}D	utf3byte('\x1e','\x10') ;	/* U+1E10 */
{ANSEL_CEDILLA}d	|
{ANSEL_LEFTHOOK}d	utf3byte('\x1e','\x11') ;	/* U+1E11 */

{ANSEL_GRAVEACCENT}{ANSEL_MACRON}E	|
{ANSEL_MACRON}{ANSEL_GRAVEACCENT}E	utf3byte('\x1e','\x14') ;	/* U+1E14 */

{ANSEL_GRAVEACCENT}{ANSEL_MACRON}e	|
{ANSEL_MACRON}{ANSEL_GRAVEACCENT}e	utf3byte('\x1e','\x15') ;	/* U+1E15 */

{ANSEL_ACUTEACCENT}{ANSEL_MACRON}E	|
{ANSEL_MACRON}{ANSEL_ACUTEACCENT}E	utf3byte('\x1e','\x16') ;	/* U+1E16 */

{ANSEL_ACUTEACCENT}{ANSEL_MACRON}e	|
{ANSEL_MACRON}{ANSEL_ACUTEACCENT}e	utf3byte('\x1e','\x17') ;	/* U+1E17 */

{ANSEL_CEDILLA}{ANSEL_BREVE}E	|
{ANSEL_BREVE}{ANSEL_CEDILLA}E	utf3byte('\x1e','\x1c') ;	/* U+1E1C */

{ANSEL_CEDILLA}{ANSEL_BREVE}e	|
{ANSEL_BREVE}{ANSEL_CEDILLA}e	utf3byte('\x1e','\x1d') ;	/* U+1E1D */

{ANSEL_DOTABOVE}F	utf3byte('\x1e','\x1e') ;	/* U+1E1E */
{ANSEL_DOTABOVE}f	utf3byte('\x1e','\x1f') ;	/* U+1E1F */
{ANSEL_MACRON}G	utf3byte('\x1e','\x20') ;	/* U+1E20 */
{ANSEL_MACRON}g	utf3byte('\x1e','\x21') ;	/* U+1E21 */
{ANSEL_DOTABOVE}H	utf3byte('\x1e','\x22') ;	/* U+1E22 */
{ANSEL_DOTABOVE}h	utf3byte('\x1e','\x23') ;	/* U+1E23 */
{ANSEL_DOTBELOW}H	utf3byte('\x1e','\x24') ;	/* U+1E24 */
{ANSEL_DOTBELOW}h	utf3byte('\x1e','\x25') ;	/* U+1E25 */
{ANSEL_DIARESIS}H	utf3byte('\x1e','\x26') ;	/* U+1E26 */
{ANSEL_DIARESIS}h	utf3byte('\x1e','\x27') ;	/* U+1E27 */
{ANSEL_CEDILLA}H	|
{ANSEL_LEFTHOOK}H	utf3byte('\x1e','\x28') ;	/* U+1E28 */
{ANSEL_CEDILLA}h	|
{ANSEL_LEFTHOOK}h	utf3byte('\x1e','\x29') ;	/* U+1E29 */

{ANSEL_ACUTEACCENT}{ANSEL_DIARESIS}I	|
{ANSEL_DIARESIS}{ANSEL_ACUTEACCENT}I	utf3byte('\x1e','\x2e') ;	/* U+1E2E */

{ANSEL_ACUTEACCENT}{ANSEL_DIARESIS}i	|
{ANSEL_DIARESIS}{ANSEL_ACUTEACCENT}i	utf3byte('\x1e','\x2f') ;	/* U+1E2F */

{ANSEL_ACUTEACCENT}K	utf3byte('\x1e','\x30') ;	/* U+1E30 */
{ANSEL_ACUTEACCENT}k	utf3byte('\x1e','\x31') ;	/* U+1E31 */
{ANSEL_DOTBELOW}K	utf3byte('\x1e','\x32') ;	/* U+1E32 */
{ANSEL_DOTBELOW}k	utf3byte('\x1e','\x33') ;	/* U+1E33 */
{ANSEL_UNDERSCORE}K	utf3byte('\x1e','\x34') ;	/* U+1E34 */
{ANSEL_UNDERSCORE}k	utf3byte('\x1e','\x35') ;	/* U+1E35 */
{ANSEL_DOTBELOW}L	utf3byte('\x1e','\x36') ;	/* U+1E36 */
{ANSEL_DOTBELOW}l	utf3byte('\x1e','\x37') ;	/* U+1E37 */

{ANSEL_DOTBELOW}{ANSEL_MACRON}L	|
{ANSEL_MACRON}{ANSEL_DOTBELOW}L	utf3byte('\x1e','\x38') ;	/* U+1E38 */

{ANSEL_DOTBELOW}{ANSEL_MACRON}l	|
{ANSEL_MACRON}{ANSEL_DOTBELOW}l	utf3byte('\x1e','\x39') ;	/* U+1E39 */

{ANSEL_UNDERSCORE}L	utf3byte('\x1e','\x3a') ;	/* U+1E3A */
{ANSEL_UNDERSCORE}l	utf3byte('\x1e','\x3b') ;	/* U+1E3B */
{ANSEL_ACUTEACCENT}M	utf3byte('\x1e','\x3e') ;	/* U+1E3E */
{ANSEL_ACUTEACCENT}m	utf3byte('\x1e','\x3f') ;	/* U+1E3F */
{ANSEL_DOTABOVE}M	utf3byte('\x1e','\x40') ;	/* U+1E40 */
{ANSEL_DOTABOVE}m	utf3byte('\x1e','\x41') ;	/* U+1E41 */
{ANSEL_DOTBELOW}M	utf3byte('\x1e','\x42') ;	/* U+1E42 */
{ANSEL_DOTBELOW}m	utf3byte('\x1e','\x43') ;	/* U+1E43 */
{ANSEL_DOTABOVE}N	utf3byte('\x1e','\x44') ;	/* U+1E44 */
{ANSEL_DOTABOVE}n	utf3byte('\x1e','\x45') ;	/* U+1E45 */
{ANSEL_UNDERSCORE}N	utf3byte('\x1e','\x46') ;	/* U+1E46 */
{ANSEL_UNDERSCORE}n	utf3byte('\x1e','\x47') ;	/* U+1E47 */

{ANSEL_ACUTEACCENT}{ANSEL_TILDE}O	|
{ANSEL_TILDE}{ANSEL_ACUTEACCENT}O	utf3byte('\x1e','\x4c') ;	/* U+1E4C */

{ANSEL_ACUTEACCENT}{ANSEL_TILDE}o	|
{ANSEL_TILDE}{ANSEL_ACUTEACCENT}o	utf3byte('\x1e','\x4d') ;	/* U+1E4D */

{ANSEL_DIARESIS}{ANSEL_TILDE}O	|
{ANSEL_TILDE}{ANSEL_DIARESIS}O	utf3byte('\x1e','\x4e') ;	/* U+1E4E */

{ANSEL_DIARESIS}{ANSEL_TILDE}o	|
{ANSEL_TILDE}{ANSEL_DIARESIS}o	utf3byte('\x1e','\x4f') ;	/* U+1E4F */

{ANSEL_GRAVEACCENT}{ANSEL_MACRON}O	|
{ANSEL_MACRON}{ANSEL_GRAVEACCENT}O	utf3byte('\x1e','\x50') ;	/* U+1E50 */

{ANSEL_GRAVEACCENT}{ANSEL_MACRON}o	|
{ANSEL_MACRON}{ANSEL_GRAVEACCENT}o	utf3byte('\x1e','\x51') ;	/* U+1E51 */

{ANSEL_ACUTEACCENT}{ANSEL_MACRON}O	|
{ANSEL_MACRON}{ANSEL_ACUTEACCENT}O	utf3byte('\x1e','\x52') ;	/* U+1E52 */

{ANSEL_ACUTEACCENT}{ANSEL_MACRON}o	|
{ANSEL_MACRON}{ANSEL_ACUTEACCENT}o	utf3byte('\x1e','\x53') ; /* U+1E53 */

{ANSEL_ACUTEACCENT}P	utf3byte('\x1e','\x54') ; /* U+1E54 */
{ANSEL_ACUTEACCENT}p	utf3byte('\x1e','\x55') ; /* U+1E55 */
{ANSEL_DOTABOVE}P	utf3byte('\x1e','\x56') ; /* U+1E56 */
{ANSEL_DOTABOVE}p	utf3byte('\x1e','\x57') ; /* U+1E57 */
{ANSEL_DOTABOVE}R	utf3byte('\x1e','\x58') ; /* U+1E58 */
{ANSEL_DOTABOVE}r	utf3byte('\x1e','\x59') ; /* U+1E59 */
{ANSEL_DOTBELOW}R	utf3byte('\x1e','\x5a') ; /* U+1E5A */
{ANSEL_DOTBELOW}r	utf3byte('\x1e','\x5b') ; /* U+1E5B */

{ANSEL_DOTBELOW}{ANSEL_MACRON}R	|
{ANSEL_MACRON}{ANSEL_DOTBELOW}R	utf3byte('\x1e','\x5c') ; /* U+1E5c */

{ANSEL_DOTBELOW}{ANSEL_MACRON}r	|
{ANSEL_MACRON}{ANSEL_DOTBELOW}r	utf3byte('\x1e','\x5d') ; /* U+1E5D */

{ANSEL_UNDERSCORE}R	utf3byte('\x1e','\x5e') ; /* U+1E5E */
{ANSEL_UNDERSCORE}r	utf3byte('\x1e','\x5f') ; /* U+1E5F */
{ANSEL_DOTABOVE}S	utf3byte('\x1e','\x60') ; /* U+1E60 */
{ANSEL_DOTABOVE}s	utf3byte('\x1e','\x61') ; /* U+1E61 */
{ANSEL_DOTBELOW}S	utf3byte('\x1e','\x62') ; /* U+1E62 */
{ANSEL_DOTBELOW}s	utf3byte('\x1e','\x63') ; /* U+1E63 */

{ANSEL_ACUTEACCENT}{ANSEL_DOTABOVE}S	|
{ANSEL_DOTABOVE}{ANSEL_ACUTEACCENT}S	utf3byte('\x1e','\x64') ; /* U+1E64 */

{ANSEL_ACUTEACCENT}{ANSEL_DOTABOVE}s	|
{ANSEL_DOTABOVE}{ANSEL_ACUTEACCENT}s	utf3byte('\x1e','\x65') ; /* U+1E65 */

{ANSEL_CARON}{ANSEL_DOTABOVE}S	|
{ANSEL_DOTABOVE}{ANSEL_CARON}S	utf3byte('\x1e','\x66') ; /* U+1E66 */

{ANSEL_CARON}{ANSEL_DOTABOVE}s	|
{ANSEL_DOTABOVE}{ANSEL_CARON}s	utf3byte('\x1e','\x67') ; /* U+1E67 */

{ANSEL_DOTBELOW}{ANSEL_DOTABOVE}S	|
{ANSEL_DOTABOVE}{ANSEL_DOTBELOW}S	utf3byte('\x1e','\x68') ; /* U+1E68 */

{ANSEL_DOTBELOW}{ANSEL_DOTABOVE}s	|
{ANSEL_DOTABOVE}{ANSEL_DOTBELOW}s	utf3byte('\x1e','\x69') ; /* U+1E69 */

{ANSEL_DOTABOVE}T	utf3byte('\x1e','\x6a') ; /* U+1E6A */
{ANSEL_DOTABOVE}t	utf3byte('\x1e','\x6b') ; /* U+1E6B */
{ANSEL_DOTBELOW}T	utf3byte('\x1e','\x6c') ; /* U+1E6C */
{ANSEL_DOTBELOW}t	utf3byte('\x1e','\x6d') ; /* U+1E6d */
{ANSEL_UNDERSCORE}T	utf3byte('\x1e','\x6e') ; /* U+1E6d */
{ANSEL_UNDERSCORE}t	utf3byte('\x1e','\x6f') ; /* U+1E6F */
{ANSEL_DOUBLEDOTBELOW}U	utf3byte('\x1e','\x72') ; /* U+1E72 */
{ANSEL_DOUBLEDOTBELOW}u	utf3byte('\x1e','\x73') ; /* U+1E73 */

{ANSEL_ACUTEACCENT}{ANSEL_TILDE}U	|
{ANSEL_TILDE}{ANSEL_ACUTEACCENT}U	utf3byte('\x1e','\x78') ; /* U+1E78 */

{ANSEL_ACUTEACCENT}{ANSEL_TILDE}u	|
{ANSEL_TILDE}{ANSEL_ACUTEACCENT}u	utf3byte('\x1e','\x79') ; /* U+1E79 */

			/* see U+01D5 for the opposite order */
{ANSEL_DIARESIS}{ANSEL_MACRON}U	utf3byte('\x1e','\x7a') ; /* U+1E7A */
{ANSEL_DIARESIS}{ANSEL_MACRON}u	utf3byte('\x1e','\x7b') ; /* U+1E7B */

{ANSEL_TILDE}V	utf3byte('\x1e','\x7c') ; /* U+1E7C */
{ANSEL_TILDE}v	utf3byte('\x1e','\x7d') ; /* U+1E7D */
{ANSEL_DOTBELOW}V	utf3byte('\x1e','\x7e') ; /* U+1E7E */
{ANSEL_DOTBELOW}v	utf3byte('\x1e','\x7f') ; /* U+1E7F */
{ANSEL_GRAVEACCENT}W	utf3byte('\x1e','\x80') ; /* U+1E80 */
{ANSEL_GRAVEACCENT}w	utf3byte('\x1e','\x81') ; /* U+1E81 */
{ANSEL_ACUTEACCENT}W	utf3byte('\x1e','\x82') ; /* U+1E82 */
{ANSEL_ACUTEACCENT}w	utf3byte('\x1e','\x83') ; /* U+1E83 */
{ANSEL_DIARESIS}W	utf3byte('\x1e','\x84') ; /* U+1E84 */
{ANSEL_DIARESIS}w	utf3byte('\x1e','\x85') ; /* U+1E85 */
{ANSEL_DOTABOVE}W	utf3byte('\x1e','\x86') ; /* U+1E86 */
{ANSEL_DOTABOVE}w	utf3byte('\x1e','\x87') ; /* U+1E87 */
{ANSEL_DOTBELOW}W	utf3byte('\x1e','\x88') ; /* U+1E88 */
{ANSEL_DOTBELOW}w	utf3byte('\x1e','\x89') ; /* U+1E89 */
{ANSEL_DOTABOVE}X	utf3byte('\x1e','\x8a') ; /* U+1E8A */
{ANSEL_DOTABOVE}x	utf3byte('\x1e','\x8b') ; /* U+1E8B */
{ANSEL_DIARESIS}X	utf3byte('\x1e','\x8c') ; /* U+1E8C */
{ANSEL_DIARESIS}x	utf3byte('\x1e','\x8d') ; /* U+1E8D */
{ANSEL_DOTABOVE}Y	utf3byte('\x1e','\x8e') ; /* U+1E8E */
{ANSEL_DOTABOVE}y	utf3byte('\x1e','\x8f') ; /* U+1E8F */
{ANSEL_CIRCUMFLEXACC}Z	utf3byte('\x1e','\x90') ; /* U+1E90 */
{ANSEL_CIRCUMFLEXACC}z	utf3byte('\x1e','\x91') ; /* U+1E91 */
{ANSEL_DOTBELOW}Z	utf3byte('\x1e','\x92') ; /* U+1E92 */
{ANSEL_DOTBELOW}z	utf3byte('\x1e','\x93') ; /* U+1E93 */
{ANSEL_UNDERSCORE}Z	utf3byte('\x1e','\x94') ; /* U+1E94 */
{ANSEL_UNDERSCORE}z	utf3byte('\x1e','\x95') ; /* U+1E95 */
{ANSEL_UNDERSCORE}h	utf3byte('\x1e','\x96') ; /* U+1E96 */
{ANSEL_DIARESIS}t	utf3byte('\x1e','\x97') ; /* U+1E97 */
{ANSEL_CIRCLEABOVE}w	utf3byte('\x1e','\x98') ; /* U+1E98 */
{ANSEL_CIRCLEABOVE}y	utf3byte('\x1e','\x99') ; /* U+1E99 */
{ANSEL_DOTBELOW}A	utf3byte('\x1e','\xa0') ; /* U+1EA0 */
{ANSEL_DOTBELOW}a	utf3byte('\x1e','\xa1') ; /* U+1EA1 */
{ANSEL_LOWRISINGTONE}A	utf3byte('\x1e','\xa2') ; /* U+1EA2 */
{ANSEL_LOWRISINGTONE}a	utf3byte('\x1e','\xa3') ; /* U+1EA3 */

{ANSEL_ACUTEACCENT}{ANSEL_CIRCUMFLEXACC}A	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_ACUTEACCENT}A	utf3byte('\x1e','\xa4') ; /* U+1EA4 */

{ANSEL_ACUTEACCENT}{ANSEL_CIRCUMFLEXACC}a	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_ACUTEACCENT}a	utf3byte('\x1e','\xa5') ; /* U+1EA5 */

{ANSEL_GRAVEACCENT}{ANSEL_CIRCUMFLEXACC}A	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_GRAVEACCENT}A	utf3byte('\x1e','\xa6') ; /* U+1EA6 */

{ANSEL_GRAVEACCENT}{ANSEL_CIRCUMFLEXACC}a	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_GRAVEACCENT}a	utf3byte('\x1e','\xa7') ; /* U+1EA7 */

{ANSEL_LOWRISINGTONE}{ANSEL_CIRCUMFLEXACC}A	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_LOWRISINGTONE}A	utf3byte('\x1e','\xa8') ; /* U+1EA8 */

{ANSEL_LOWRISINGTONE}{ANSEL_CIRCUMFLEXACC}a	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_LOWRISINGTONE}a	utf3byte('\x1e','\xa9') ; /* U+1EA9 */

{ANSEL_TILDE}{ANSEL_CIRCUMFLEXACC}A	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_TILDE}A	utf3byte('\x1e','\xaa') ; /* U+1EAA */

{ANSEL_TILDE}{ANSEL_CIRCUMFLEXACC}a	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_TILDE}a	utf3byte('\x1e','\xab') ; /* U+1EAB */

{ANSEL_DOTBELOW}{ANSEL_CIRCUMFLEXACC}A	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_DOTBELOW}A	utf3byte('\x1e','\xac') ; /* U+1EAC */

{ANSEL_DOTBELOW}{ANSEL_CIRCUMFLEXACC}a	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_DOTBELOW}a	utf3byte('\x1e','\xad') ; /* U+1EAD */

{ANSEL_ACUTEACCENT}{ANSEL_BREVE}A	|
{ANSEL_BREVE}{ANSEL_ACUTEACCENT}A	utf3byte('\x1e','\xae') ; /* U+1EAE */

{ANSEL_ACUTEACCENT}{ANSEL_BREVE}a	|
{ANSEL_BREVE}{ANSEL_ACUTEACCENT}a	utf3byte('\x1e','\xaf') ; /* U+1EAF */

{ANSEL_GRAVEACCENT}{ANSEL_BREVE}A	|
{ANSEL_BREVE}{ANSEL_GRAVEACCENT}A	utf3byte('\x1e','\xb0') ; /* U+1EB0 */

{ANSEL_GRAVEACCENT}{ANSEL_BREVE}a	|
{ANSEL_BREVE}{ANSEL_GRAVEACCENT}a	utf3byte('\x1e','\xb1') ; /* U+1EB1 */

{ANSEL_LOWRISINGTONE}{ANSEL_BREVE}A	|
{ANSEL_BREVE}{ANSEL_LOWRISINGTONE}A	utf3byte('\x1e','\xb2') ; /* U+1EB2 */

{ANSEL_LOWRISINGTONE}{ANSEL_BREVE}a	|
{ANSEL_BREVE}{ANSEL_LOWRISINGTONE}a	utf3byte('\x1e','\xb3') ; /* U+1EB3 */

{ANSEL_TILDE}{ANSEL_BREVE}A	|
{ANSEL_BREVE}{ANSEL_TILDE}A	utf3byte('\x1e','\xb4') ; /* U+1EB4 */

{ANSEL_TILDE}{ANSEL_BREVE}a	|
{ANSEL_BREVE}{ANSEL_TILDE}a	utf3byte('\x1e','\xb5') ; /* U+1EB5 */

{ANSEL_DOTBELOW}{ANSEL_BREVE}A	|
{ANSEL_BREVE}{ANSEL_DOTBELOW}A	utf3byte('\x1e','\xb6') ; /* U+1EB6 */

{ANSEL_DOTBELOW}{ANSEL_BREVE}a	|
{ANSEL_BREVE}{ANSEL_DOTBELOW}a	utf3byte('\x1e','\xb7') ; /* U+1EB7 */

{ANSEL_DOTBELOW}E	utf3byte('\x1e','\xb8') ; /* U+1EB8 */
{ANSEL_DOTBELOW}e	utf3byte('\x1e','\xb9') ; /* U+1EB9 */

{ANSEL_LOWRISINGTONE}E	utf3byte('\x1e','\xba') ; /* U+1EBA */
{ANSEL_LOWRISINGTONE}e	utf3byte('\x1e','\xbb') ; /* U+1EBB */
{ANSEL_TILDE}E	utf3byte('\x1e','\xbc') ; /* U+1EBC */
{ANSEL_TILDE}e	utf3byte('\x1e','\xbd') ; /* U+1EBD */

{ANSEL_ACUTEACCENT}{ANSEL_CIRCUMFLEXACC}E	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_ACUTEACCENT}E	utf3byte('\x1e','\xbe') ; /* U+1EBE */

{ANSEL_ACUTEACCENT}{ANSEL_CIRCUMFLEXACC}e	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_ACUTEACCENT}e	utf3byte('\x1e','\xbf') ; /* U+1EBF */

{ANSEL_GRAVEACCENT}{ANSEL_CIRCUMFLEXACC}E	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_GRAVEACCENT}E	utf3byte('\x1e','\xc0') ; /* U+1EC0*/

{ANSEL_GRAVEACCENT}{ANSEL_CIRCUMFLEXACC}e	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_GRAVEACCENT}e	utf3byte('\x1e','\xc1') ; /* U+1EC1 */

{ANSEL_LOWRISINGTONE}{ANSEL_CIRCUMFLEXACC}E	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_LOWRISINGTONE}E	utf3byte('\x1e','\xc2') ; /* U+1EC2 */

{ANSEL_LOWRISINGTONE}{ANSEL_CIRCUMFLEXACC}e	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_LOWRISINGTONE}e	utf3byte('\x1e','\xc3') ; /* U+1EC3 */

{ANSEL_TILDE}{ANSEL_CIRCUMFLEXACC}E	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_TILDE}E	utf3byte('\x1e','\xc4') ; /* U+1EC4 */

{ANSEL_TILDE}{ANSEL_CIRCUMFLEXACC}e	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_TILDE}e	utf3byte('\x1e','\xc5') ; /* U+1EC5 */

{ANSEL_DOTBELOW}{ANSEL_CIRCUMFLEXACC}E	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_DOTBELOW}E	utf3byte('\x1e','\xc6') ; /* U+1EC6 */

{ANSEL_DOTBELOW}{ANSEL_CIRCUMFLEXACC}e	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_DOTBELOW}e	utf3byte('\x1e','\xc7') ; /* U+1EC7 */

{ANSEL_LOWRISINGTONE}I	utf3byte('\x1e','\xc8') ; /* U+1EC8 */
{ANSEL_LOWRISINGTONE}i	utf3byte('\x1e','\xc9') ; /* U+1EC9 */
{ANSEL_DOTBELOW}I	utf3byte('\x1e','\xca') ; /* U+1ECA */
{ANSEL_DOTBELOW}i	utf3byte('\x1e','\xcb') ; /* U+1ECB */
{ANSEL_DOTBELOW}O	utf3byte('\x1e','\xcc') ; /* U+1ECC */
{ANSEL_DOTBELOW}o	utf3byte('\x1e','\xcd') ; /* U+1ECD */
{ANSEL_LOWRISINGTONE}O	utf3byte('\x1e','\xce') ; /* U+1ECE */
{ANSEL_LOWRISINGTONE}o	utf3byte('\x1e','\xcf') ; /* U+1ECF */

{ANSEL_ACUTEACCENT}{ANSEL_CIRCUMFLEXACC}O	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_ACUTEACCENT}O	utf3byte('\x1e','\xd0') ; /* U+1ED0 */

{ANSEL_ACUTEACCENT}{ANSEL_CIRCUMFLEXACC}o	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_ACUTEACCENT}o	utf3byte('\x1e','\xd1') ; /* U+1Ed1 */

{ANSEL_GRAVEACCENT}{ANSEL_CIRCUMFLEXACC}O	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_GRAVEACCENT}O	utf3byte('\x1e','\xd2') ; /* U+1ED2 */

{ANSEL_GRAVEACCENT}{ANSEL_CIRCUMFLEXACC}o	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_GRAVEACCENT}o	utf3byte('\x1e','\xd3') ; /* U+1ED3 */

{ANSEL_LOWRISINGTONE}{ANSEL_CIRCUMFLEXACC}O	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_LOWRISINGTONE}O	utf3byte('\x1e','\xd4') ; /* U+1ED4 */

{ANSEL_LOWRISINGTONE}{ANSEL_CIRCUMFLEXACC}o	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_LOWRISINGTONE}o	utf3byte('\x1e','\xd5') ; /* U+1Ed5 */

{ANSEL_TILDE}{ANSEL_CIRCUMFLEXACC}O	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_TILDE}O	utf3byte('\x1e','\xd6') ; /* U+1ED6 */

{ANSEL_TILDE}{ANSEL_CIRCUMFLEXACC}o	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_TILDE}o	utf3byte('\x1e','\xd7') ; /* U+1ED7 */

{ANSEL_DOTBELOW}{ANSEL_CIRCUMFLEXACC}O	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_DOTBELOW}O	utf3byte('\x1e','\xd8') ; /* U+1ED8 */

{ANSEL_DOTBELOW}{ANSEL_CIRCUMFLEXACC}o	|
{ANSEL_CIRCUMFLEXACC}{ANSEL_DOTBELOW}o	utf3byte('\x1e','\xd9') ; /* U+1ED9 */

{ANSEL_ACUTEACCENT}{ANSEL_VIETO}	utf3byte('\x1e','\xda') ; /* U+1EDA */
{ANSEL_ACUTEACCENT}{ANSEL_VIETo}	utf3byte('\x1e','\xdb') ; /* U+1EDB */
{ANSEL_GRAVEACCENT}{ANSEL_VIETO}	utf3byte('\x1e','\xdc') ; /* U+1EDC */
{ANSEL_GRAVEACCENT}{ANSEL_VIETo}	utf3byte('\x1e','\xdd') ; /* U+1EDD */
{ANSEL_LOWRISINGTONE}{ANSEL_VIETO}	utf3byte('\x1e','\xde') ; /* U+1EDE */
{ANSEL_LOWRISINGTONE}{ANSEL_VIETo}	utf3byte('\x1e','\xdf') ; /* U+1EDF */
{ANSEL_TILDE}{ANSEL_VIETO}	utf3byte('\x1e','\xe0') ; /* U+1EE0 */
{ANSEL_TILDE}{ANSEL_VIETo}	utf3byte('\x1e','\xe1') ; /* U+1EE1 */
{ANSEL_DOTBELOW}{ANSEL_VIETO}	utf3byte('\x1e','\xe2') ; /* U+1EE2 */
{ANSEL_DOTBELOW}{ANSEL_VIETo}	utf3byte('\x1e','\xe3') ; /* U+1EE3 */
{ANSEL_DOTBELOW}U	utf3byte('\x1e','\xe4') ; /* U+1EE4 */
{ANSEL_DOTBELOW}u	utf3byte('\x1e','\xe5') ; /* U+1EE5 */
{ANSEL_LOWRISINGTONE}U	utf3byte('\x1e','\xe6') ; /* U+1EE6 */
{ANSEL_LOWRISINGTONE}u	utf3byte('\x1e','\xe7') ; /* U+1EE7 */
{ANSEL_ACUTEACCENT}{ANSEL_VIETU}	utf3byte('\x1e','\xe8') ; /* U+1EE8 */
{ANSEL_ACUTEACCENT}{ANSEL_VIETu}	utf3byte('\x1e','\xe9') ; /* U+1EE9 */
{ANSEL_GRAVEACCENT}{ANSEL_VIETU}	utf3byte('\x1e','\xea') ; /* U+1EEA */
{ANSEL_GRAVEACCENT}{ANSEL_VIETu}	utf3byte('\x1e','\xeb') ; /* U+1EEB */
{ANSEL_LOWRISINGTONE}{ANSEL_VIETU}	utf3byte('\x1e','\xec') ; /* U+1EEC */
{ANSEL_LOWRISINGTONE}{ANSEL_VIETu}	utf3byte('\x1e','\xed') ; /* U+1EED */
{ANSEL_TILDE}{ANSEL_VIETU}	utf3byte('\x1e','\xee') ; /* U+1EEE */
{ANSEL_TILDE}{ANSEL_VIETu}	utf3byte('\x1e','\xef') ; /* U+1EEF */
{ANSEL_DOTBELOW}{ANSEL_VIETU}	utf3byte('\x1e','\xf0') ; /* U+1EF0 */
{ANSEL_DOTBELOW}{ANSEL_VIETu}	utf3byte('\x1e','\xf1') ; /* U+1EF1 */
{ANSEL_GRAVEACCENT}Y	utf3byte('\x1e','\xf2') ; /* U+1EF2 */
{ANSEL_GRAVEACCENT}y	utf3byte('\x1e','\xf3') ; /* U+1EF3 */
{ANSEL_DOTBELOW}Y	utf3byte('\x1e','\xf4') ; /* U+1EF4 */
{ANSEL_DOTBELOW}y	utf3byte('\x1e','\xf5') ; /* U+1EF5 */
{ANSEL_LOWRISINGTONE}Y	utf3byte('\x1e','\xf6') ; /* U+1EF6 */
{ANSEL_LOWRISINGTONE}y	utf3byte('\x1e','\xf7') ; /* U+1EF7 */
{ANSEL_TILDE}Y	utf3byte('\x1e','\xf8') ; /* U+1EF8 */
{ANSEL_TILDE}y	utf3byte('\x1e','\xf9') ; /* U+1EF9 */

	/* end of Latin Extended Additional */

\xae	utf3byte('\x02','\xbc') ;	/* alif */ /* U+1EBC */
\xb0	utf3byte('\x02','\xbb') ;	/* ayn */ /* U+1EBB */

	/* MUSIC */
\xa9	utf3byte('\x26','\x6d') ; /* MUSIC FLAT SIGN                     U+266D */
\xc4	utf3byte('\x26','\x6f') ; /* MUSIC SHARP SIGN                    U+266F */

\xc1	utf3byte('\x21','\x13') ; /* SCRIPT SMALL L U+2113 */
\xc2	utf3byte('\x21','\x17') ; /* SOUNC RECORDING COPYRIGHT U+2117 */

	/* hard, soft signs */
\xb7	{
#ifdef MARC_EXT
		utf3byte('\x02','\xba') ; /* MODIFIER LETTER DOUBLE PRIME U+02BA */
#else
		utf3byte('\x04','\x4a') ; /* CYRILLIC SMALL LETTER HARD SIGN U+044A */
#endif
	}

\xa7	{
#ifdef MARC_EXT
		utf3byte('\x02','\xb9') ; /* U+02B9 */
#else
		utf3byte('\x04','\x4c') ; /* CYRILLIC SMALL LETTER SOFT SIGN U+044A */
#endif
	}

	/* Vietnamiese from Latin Extended-B */
ANSEL_VIETO	utf3byte('\x01','\xa0') ; /* LATIN CAPITAL LETTER o WITH HORN U+01A0 */
ANSEL_VIETo	utf3byte('\x01','\xa1') ; /* LATIN SMALL  LETTER o WITH HORN U+01A1 */
ANSEL_VIETU	utf3byte('\x01','\xaf') ; /* LATIN CAPITAL LETTER u WITH HORN U+01AF */
ANSEL_VIETu	utf3byte('\x01','\xb0') ; /* LATIN SMALL LETTER u WITH HORN U+01B0 */

{GEDCOM55_EMPTYBOX} {
#ifdef GEDCOM_LDSEXT
		utf3byte('\x25','\xa1') ;	/*WHITE SQUARE U+25A1  this is a rumor not in the 5.5 or 5.5.1 standard */
		;
#else
		;
#endif
	}

{GEDCOM55_BLACKBOX} {
#ifdef GEDCOM_LDSEXT
		utf3byte('\x25','\xa0') ;	/* BLACK SQUARE U+25A0 a rumor again not in GEDCOM 5.5 or 5.5.1 */
		;
#else
		;
#endif
	}

{GEDCOM55_SZ} {
#if defined GEDCOM_EXT || defined GEDCOM_LDSEXT
		utf2byte('\xdf') ;	/* LATIN SMALL LETTER SHARP S U+00DF */
		;
#else
		;
#endif
	}

\xc7	{
#ifdef MARC_EXT
		utf2byte('\xdf') ;	/* LATIN SMALL LETTER SHARP S U+00DF */
#else
		;
#endif
	}

\xc8	{
#ifdef MARC_EXT
		utf3byte('\x20','\xac') ;	/* EURO SIGN U+20AC */
#else
		;
#endif
	}



	/* generic declarations follow. flex(1) specifies that they are
	* superseded by those above if there are mutliple matchings of the same length */

	/* invalid codes in the standard use : ignore (delete on output) */
{ANSEL_RES_C1}	|
{ANSEL_CORN}	|
{ANSEL_RES_FUT}	;

	/* low rising tone mark */
{ANSEL_LOWRISINGTONE}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x09') ;	 /* COMBINING HOOK ABOVE U+0309 */
	}

	/* grave accent */
{ANSEL_GRAVEACCENT}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x00') ;	 /* COMBINING GRAVE ACCENT U+0300 */
	}

	/* acute accent */
{ANSEL_ACUTEACCENT}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x01') ;	 /* COMBINING  ACUTE ACCENT U+0301 */
	}

	/* circumflex accent */
{ANSEL_CIRCUMFLEXACC}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x02') ;	 /* COMBINING CIRCUMFLEX ACCENT U+0302 */
	}

	/* tilde */
{ANSEL_TILDE}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x03') ;	 /* COMBINING TILDE U+0303 */
	}

	/* Macron */
{ANSEL_MACRON}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x04') ;	 /* COMBINING MACRON U+0304 */
	}

	/* Breve */
{ANSEL_BREVE}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x06') ;	 /* COMBINING BREVE U+0306 */
	}

	/* dot above */
{ANSEL_DOTABOVE}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x07') ;	 /* COMBINING DOT ABOVE U+0307 */
	}

	/* Umlaut, diaresis, double dot above */
{ANSEL_DIARESIS}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x08') ;	 /* COMBINING DIARESIS U+0308 */
	}

	/* Caron, hacek */
{ANSEL_CARON}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x0c') ;	 /* COMBINING CARON U+030C */
	}

	/* circle, ring above */
{ANSEL_CIRCLEABOVE}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x0a') ;	 /* COMBINING RING ABOVE U+030A */
	}

	/* ligature, left half   \xeb, ligature right half \xec.
	* For the purpose of the conversion we assume that \xeb[someletter]\xec[secondletter]
	* is always in use */
{ANSEL_LIGATURELEFT}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x61') ;	 /* COMBINING DOUBLE INVERTED BREVE U+0361 */
	}

{ANSEL_LIGATURERIGHT}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
	}

	/* high comma, off center */
{ANSEL_HIGHCOMMAOFFC}([[:graph:]])	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x15') ;	 /* COMBINING COMMA ABOVE RIGHT U+0315 */
	}

	/* Hungarian double acute  */
{ANSEL_DOUBLEACUTEACC}([[:graph:]])	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x0b') ;	 /* COMBINING DOUBLE ACUTE ACCENT U+030B */
	}

	/* combining  candrabindu */
{ANSEL_CANDRABINDU}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x10') ;	 /* COMBININGCANDRABINDU U+0310 */
	}

	/* cedilla  */
{ANSEL_CEDILLA}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x27') ;	 /* COMBINING CEDILLA U+0327 */
	}


	/* right hook */
{ANSEL_RIGHTHOOK}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x28') ;	 /* COMBINING OGONEK U+0328 */
	}

	/* dot below */
{ANSEL_DOTBELOW}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x23') ;	 /* COMBINING DOT BELOW U+0323 */
	}

	/* double dot below */
{ANSEL_DOUBLEDOTBELOW}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x24') ;	 /* COMBINING DOUBLE DIARESIS BELOW U+0324 */
	}

	/* circle below */
{ANSEL_CIRCLEBELOW}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x25') ;	 /* COMBINING RING BELOW U+0325 */
	}

	/* double underscore */
{ANSEL_DOUBLEUNDERSC}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x33') ;	 /* COMBINING DOUBLE LOW LINE U+0333 */
	}

	/* underscore */
{ANSEL_UNDERSCORE}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x32') ;	 /* COMBINING LOW LINE U+0332 */
	}

	/* left hook */
{ANSEL_LEFTHOOK}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x26') ;	 /* COMBINING COMMA BELOW U+0326 */
	}

	/* right cedilla */
{ANSEL_RIGHTCEDILLA}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x1c') ;	 /* COMBINING LEFT HALF RING BELOW U+031C */
	}

	/* half circle below hook */
{ANSEL_HALFCIRCLBELOW}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x2e') ;	 /* COMBINING BREVE BELOW U+032e */
	}

	/* double tilde, left half and right half.
	* For the purpose of the conversion we assume that \xfa[someletter]\xfb[secondletter]
	* is always in use, without actually checking */
{ANSEL_DOUBLETILDLEFT}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x60') ;	 /* COMBINING DOUBLE TILDE U+0360 */
	}

{ANSEL_DOUBLETILDRIGH}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
	}

	/* high comma */
{ANSEL_HIGHCOMMACENT}[[:graph:]]	{
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x13') ;	 /* COMBINING COMMA ABOVE U+0313 */
	}

{GEDCOM55_SLASH}[[:graph:]] {
#ifdef GEDCOM_LDSEXT
		/* the corresponding multiple diacretics with the slash are not implemented below...*/
		fprintf(yyout,"%c",yytext[1]) ;
		utf3byte('\x03','\x38') ;	 /* COMBINING LONG SOLIDUS OVERLAY U+0337 , perhaps U+0337 */
#else
		fprintf(yyout,"%c",yytext[1]) ;
#endif
	}

	/* three forms of newline in MAC, Unixes and DOS */
\l	newl(yytext) ;

\l\n	newl(yytext) ;

\n	newl(yytext) ;

	/* Generic double accents. The U+1DC4 to U+1DCC are probably not 
	* meant to be used as double combining accents.
	*/
{ANSEL_LOWRISINGTONE}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_LOWRISINGTONE}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x00') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_LOWRISINGTONE}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_LOWRISINGTONE}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_LOWRISINGTONE}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_LOWRISINGTONE}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_LOWRISINGTONE}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_LOWRISINGTONE}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_LOWRISINGTONE}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_LOWRISINGTONE}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_LOWRISINGTONE}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_LOWRISINGTONE}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_LOWRISINGTONE}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_LOWRISINGTONE}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_LOWRISINGTONE}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_LOWRISINGTONE}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_LOWRISINGTONE}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_LOWRISINGTONE}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_LOWRISINGTONE}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_LOWRISINGTONE}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_LOWRISINGTONE}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_LOWRISINGTONE}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_LOWRISINGTONE}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_LOWRISINGTONE}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_LOWRISINGTONE}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x09') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x00') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0f') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x00') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x00') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x00') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x00') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x00') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x00') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x00') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x00') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x00') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x00') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x00') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x00') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x00') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x00') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x00') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x00') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x00') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x00') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x00') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x00') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x00') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x00') ;
	}
{ANSEL_GRAVEACCENT}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x00') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x00') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_ACUTEACCENT}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x01') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x00') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_CIRCUMFLEXACC}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x02') ;
	}
{ANSEL_TILDE}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_TILDE}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x00') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_TILDE}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_TILDE}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_TILDE}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_TILDE}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_TILDE}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_TILDE}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_TILDE}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_TILDE}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_TILDE}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_TILDE}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_TILDE}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_TILDE}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_TILDE}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_TILDE}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_TILDE}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_TILDE}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_TILDE}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_TILDE}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_TILDE}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_TILDE}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_TILDE}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_TILDE}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_TILDE}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x03') ;
	}
{ANSEL_MACRON}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_MACRON}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x00') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_MACRON}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_MACRON}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_MACRON}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_MACRON}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_MACRON}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_MACRON}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_MACRON}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_MACRON}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_MACRON}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_MACRON}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_MACRON}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_MACRON}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_MACRON}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_MACRON}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_MACRON}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_MACRON}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_MACRON}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_MACRON}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_MACRON}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_MACRON}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_MACRON}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_MACRON}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_MACRON}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x04') ;
	}
{ANSEL_BREVE}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_BREVE}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x00') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_BREVE}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_BREVE}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_BREVE}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_BREVE}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_BREVE}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_BREVE}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_BREVE}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_BREVE}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_BREVE}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_BREVE}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_BREVE}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_BREVE}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_BREVE}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_BREVE}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_BREVE}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_BREVE}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_BREVE}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_BREVE}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_BREVE}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_BREVE}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_BREVE}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_BREVE}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_BREVE}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x06') ;
	}
{ANSEL_DOTABOVE}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DOTABOVE}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x00') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DOTABOVE}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DOTABOVE}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DOTABOVE}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DOTABOVE}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DOTABOVE}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DOTABOVE}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DOTABOVE}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DOTABOVE}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DOTABOVE}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DOTABOVE}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DOTABOVE}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DOTABOVE}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DOTABOVE}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DOTABOVE}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DOTABOVE}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DOTABOVE}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DOTABOVE}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DOTABOVE}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DOTABOVE}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DOTABOVE}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DOTABOVE}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DOTABOVE}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DOTABOVE}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x07') ;
	}
{ANSEL_DIARESIS}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_DIARESIS}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x00') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_DIARESIS}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_DIARESIS}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_DIARESIS}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_DIARESIS}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_DIARESIS}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_DIARESIS}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_DIARESIS}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_DIARESIS}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_DIARESIS}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_DIARESIS}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_DIARESIS}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_DIARESIS}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_DIARESIS}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_DIARESIS}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_DIARESIS}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_DIARESIS}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_DIARESIS}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_DIARESIS}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_DIARESIS}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_DIARESIS}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_DIARESIS}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_DIARESIS}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_DIARESIS}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x08') ;
	}
{ANSEL_CARON}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CARON}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x00') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CARON}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CARON}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CARON}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CARON}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CARON}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CARON}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CARON}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CARON}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CARON}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CARON}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CARON}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CARON}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CARON}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CARON}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CARON}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CARON}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CARON}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CARON}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CARON}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CARON}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CARON}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CARON}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CARON}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x0c') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x00') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_CIRCLEABOVE}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x0a') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x00') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_HIGHCOMMAOFFC}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x15') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x00') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_DOUBLEACUTEACC}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x0b') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x00') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CANDRABINDU}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x10') ;
	}
{ANSEL_CEDILLA}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_CEDILLA}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x00') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_CEDILLA}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_CEDILLA}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_CEDILLA}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_CEDILLA}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_CEDILLA}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_CEDILLA}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_CEDILLA}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_CEDILLA}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_CEDILLA}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_CEDILLA}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_CEDILLA}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_CEDILLA}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_CEDILLA}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_CEDILLA}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_CEDILLA}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_CEDILLA}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_CEDILLA}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_CEDILLA}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_CEDILLA}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_CEDILLA}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_CEDILLA}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_CEDILLA}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_CEDILLA}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x27') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x00') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_RIGHTHOOK}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x28') ;
	}
{ANSEL_DOTBELOW}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOTBELOW}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x00') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOTBELOW}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOTBELOW}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOTBELOW}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOTBELOW}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOTBELOW}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOTBELOW}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOTBELOW}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOTBELOW}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOTBELOW}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOTBELOW}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOTBELOW}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOTBELOW}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOTBELOW}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOTBELOW}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOTBELOW}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOTBELOW}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOTBELOW}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOTBELOW}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOTBELOW}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOTBELOW}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOTBELOW}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOTBELOW}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOTBELOW}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x23') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x00') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_DOUBLEDOTBELOW}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x24') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x00') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_CIRCLEBELOW}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x25') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x00') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_DOUBLEUNDERSC}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x33') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x00') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_UNDERSCORE}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x32') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x00') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_LEFTHOOK}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x26') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x00') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_RIGHTCEDILLA}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x1c') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x00') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HALFCIRCLBELOW}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x2e') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_LOWRISINGTONE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x09') ;
		utf3byte('\x03','\x13') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_GRAVEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x00') ;
		utf3byte('\x03','\x13') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_ACUTEACCENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x01') ;
		utf3byte('\x03','\x13') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_CIRCUMFLEXACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x02') ;
		utf3byte('\x03','\x13') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_TILDE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x03') ;
		utf3byte('\x03','\x13') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_MACRON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x04') ;
		utf3byte('\x03','\x13') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_BREVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x06') ;
		utf3byte('\x03','\x13') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_DOTABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x07') ;
		utf3byte('\x03','\x13') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_DIARESIS}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x08') ;
		utf3byte('\x03','\x13') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_CARON}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0c') ;
		utf3byte('\x03','\x13') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_CIRCLEABOVE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0a') ;
		utf3byte('\x03','\x13') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_HIGHCOMMAOFFC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x15') ;
		utf3byte('\x03','\x13') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_DOUBLEACUTEACC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x0b') ;
		utf3byte('\x03','\x13') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_CANDRABINDU}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x10') ;
		utf3byte('\x03','\x13') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_CEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x27') ;
		utf3byte('\x03','\x13') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_RIGHTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x28') ;
		utf3byte('\x03','\x13') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_DOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x23') ;
		utf3byte('\x03','\x13') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_DOUBLEDOTBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x24') ;
		utf3byte('\x03','\x13') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_CIRCLEBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x25') ;
		utf3byte('\x03','\x13') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_DOUBLEUNDERSC}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x33') ;
		utf3byte('\x03','\x13') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_UNDERSCORE}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x32') ;
		utf3byte('\x03','\x13') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_LEFTHOOK}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x26') ;
		utf3byte('\x03','\x13') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_RIGHTCEDILLA}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x1c') ;
		utf3byte('\x03','\x13') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_HALFCIRCLBELOW}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x2e') ;
		utf3byte('\x03','\x13') ;
	}
{ANSEL_HIGHCOMMACENT}{ANSEL_HIGHCOMMACENT}[[:print::]]	{
		fprintf(yyout,"%c",yytext[2]);
		utf3byte('\x03','\x13') ;
		utf3byte('\x03','\x13') ;
	}

%%
