/* GPGME regression tests */

#include <stdio.h>
#include <stdlib.h>
#include <windows.h>

#include "../MyGPGME/gpgme.h"


#define fail_if_err_2(rc, file, line) do { \
    if (rc) { \
	printf ("%s:%d: %s\n", file, line, gpgme_strerror (rc)); \
	abort (); \
    } \
} while (0)

#define fail_if_err(rc) \
    fail_if_err_2(rc, __FILE__, __LINE__)


static void
t_keylist (void)
{
    gpgme_ctx_t c;
    gpgme_key_t k;
    gpgme_error_t rc;

    printf ("keylist ");
    rc = gpgme_new (&c);
    fail_if_err (rc);
    rc = gpgme_op_keylist_start (c, NULL, 0);
    fail_if_err (rc);
    while (!(rc=gpgme_op_keylist_next (c, &k)))
	gpgme_key_release (k);
    if (rc != GPGME_EOF)
	fail_if_err (rc);
    gpgme_release (c);
    printf ("OK\n");
}

static void
t_keylist_ext (const char * patt)
{
    gpgme_ctx_t c;
    gpgme_key_t k = NULL;
    gpgme_error_t rc;
    int i;

    printf ("keylist '%s'\n", patt);
    rc = gpgme_new (&c);
    fail_if_err (rc);
    rc = gpgme_op_keylist_start (c, patt, 0);
    fail_if_err (rc);
    rc = gpgme_op_keylist_next (c, &k);
    fail_if_err (rc);
    printf ("pub %4d/%s\n", gpgme_key_get_ulong_attr (k, GPGME_ATTR_LEN, NULL, 0),
			    gpgme_key_get_string_attr (k, GPGME_ATTR_KEYID, NULL, 0)+8);
    for (i=0; i < gpgme_key_count_items (k, GPGME_ATTR_USERID); i++)
	printf ("\t%d:%s\n", gpgme_key_get_ulong_attr (k, GPGME_ATTR_VALIDITY, NULL, i),
			     gpgme_key_get_string_attr (k, GPGME_ATTR_NAME, NULL, i));    

    gpgme_release (c);
    gpgme_key_release (k);
    printf ("OK\n");
}


static void
t_encrypt_data (void)
{
    gpgme_data_t in, out;
    gpgme_recipients_t rset;
    gpgme_error_t rc;
    gpgme_ctx_t c;

    printf ("encrypt-data ");
    rc = gpgme_new (&c);
    fail_if_err (rc);

    rc = gpgme_data_new (&out);
    fail_if_err (rc);
    rc = gpgme_data_new_from_mem (&in, "hello world!", 12, 1);
    fail_if_err (rc);

    rc = gpgme_recipients_new (&rset);
    fail_if_err (rc);

    rc = gpgme_recipients_add_name (rset, "alice");
    fail_if_err (rc);
    rc = gpgme_recipients_add_name (rset, "charly");
    fail_if_err (rc);

    rc = gpgme_op_encrypt (c, rset, in, out);
    fail_if_err (rc);

    gpgme_recipients_release (rset);
    gpgme_release (c);
    gpgme_data_release (in);
    gpgme_data_release (out);
    printf ("OK\n");
}


static void
t_editkey_addkey (void)
{
    gpgme_ctx_t c;
    gpgme_error_t rc;
    gpgme_editkey_t ek;
    int algos[] = {2, 3, 5, 6, 0};
    int i;

    printf ("editkey-addkey ");
    rc = gpgme_new (&c);
    fail_if_err (rc);

    rc = gpgme_editkey_new (&ek);
    fail_if_err (rc);

    for (i=0; algos[i] != 0; i++) {
	rc = gpgme_editkey_addkey_set (ek, "abc", algos[i], 1024, 0);
	fail_if_err (rc);
	gpgme_set_edit_ctx (c, ek, GPGME_EDITKEY_ADDKEY);
	rc = gpgme_op_editkey (c, "alice");
	fail_if_err (rc);
	printf ("*");
    }

    gpgme_release (c);
    gpgme_editkey_release (ek);
    printf (" OK\n");
}


static void
t_editkey_sign (void)
{
    gpgme_ctx_t c;
    gpgme_error_t rc;
    gpgme_editkey_t ek;    
    gpgme_key_t key;

    printf ("editkey-sign ");
    rc = gpgme_new (&c);
    fail_if_err (rc);

    rc = gpgme_op_keylist_start (c, "bobby", 0);
    fail_if_err (rc);

    rc = gpgme_op_keylist_next (c, &key);
    fail_if_err (rc);

    rc = gpgme_editkey_new (&ek);
    fail_if_err (rc);

    gpgme_release (c);
    rc = gpgme_new (&c);
    fail_if_err (rc);

    rc = gpgme_editkey_sign_set (ek, "abc", 0, GPGME_EDITKEY_LSIGN, "2004-01-30");
    fail_if_err (rc);
    gpgme_set_edit_ctx (c, ek, GPGME_EDITKEY_LSIGN);

    gpgme_set_local_user (c, "bob");

    rc = gpgme_op_editkey (c, "char");
    if (rc != GPGME_Conflict)
	fail_if_err (rc);
    
    Sleep (1000); /* fixme */
    gpgme_cancel (c);
    gpgme_editkey_release (ek);
    gpgme_key_release (key);
    gpgme_release (c);
    printf (" OK\n");
}


static void
t_addrevoker (void)
{
    gpgme_ctx_t c;
    gpgme_editkey_t ek;
    gpgme_error_t rc;

    rc = gpgme_new (&c);
    fail_if_err (rc);
    rc = gpgme_editkey_new (&ek);
    fail_if_err (rc);

    rc = gpgme_editkey_addrev_set (ek, "charly", "abc");
    fail_if_err (rc);

    gpgme_set_edit_ctx (c, ek, GPGME_EDITKEY_ADDREV);
    rc = gpgme_op_editkey (c, "alice");
    fail_if_err (rc);

    gpgme_editkey_release (ek);
    gpgme_release (c);
}


int
main (int argc, char ** argv)
{
        
    char f[] = "foo";
    const char * s = f;
    gpgme_set_debug_mode (5);

#if 0
    t_keylist ();
    t_encrypt_data ();    
    t_addrevoker ();
    t_editkey_addkey ();
    t_editkey_sign ();
#endif
    t_keylist_ext ("insecure");
    return 0;
}
