/* WinPT.cpp - Windows Privacy Tray (WinPT)
 *	Copyright (C) 2000-2005 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <windows.h>

#include "../resource.h"
#include "wptTypes.h"
#include "wptW32API.h"
#include "wptVersion.h"
#include "wptErrors.h"
#include "wptGPG.h"
#include "wptRegistry.h"
#include "wptCommonCtl.h"
#include "wptDlgs.h"
#include "wptNLS.h"
#include "wptKeyserver.h"
#include "wptCard.h"
#include "wptFileManager.h"
#include "wptContext.h"

HINSTANCE glob_hinst; /* global instance for the dialogs */
HWND glob_hwnd; /* global window handle for the dialogs */
HWND activ_hwnd;
LOCK mo_file;
int scard_support = 0;
int debug = 0;
int mobile = 0;
int gpg_read_only = 0;
char gpgver[3];


static void
update_keycache (HWND hwnd)
{
    refresh_cache_s rcs = {0};
    rcs.kr_reload = 0; 
    rcs.kr_update = 1;
    rcs.tr_update = 1;
    DialogBoxParam (glob_hinst, (LPCSTR)IDD_WINPT_KEYCACHE, hwnd,
		    keycache_dlg_proc, (LPARAM)&rcs);
} /* update_keycache */


static char *
get_gettext_lang (void)
{    
    char * fname;
    fname = get_reg_entry_mo ();
    if (!fname)
	return NULL;
    return fname;
} /* get_gettext_lang */


static void
load_gettext (void)
{
    char * nls = NULL;
    char * file = NULL;

    nls = get_gettext_lang ();
    if (nls) {
	set_gettext_file ("winpt", nls);
	file = make_filename (nls, "winpt", "mo");
	if (!file_exist_check (nls) && init_file_lock (&mo_file, file))  {
	    msg_box (NULL, _("Could not initizalize file lock.\n"
			     "Native Language Support"),
		     _("WinPT Error"), MB_ERR);
	}
	free_if_alloc (nls);
	free_if_alloc (file);
    }
} /* load_gettext */


/* check if the default key from the gpg.conf file is available in the
   keyring. if not, bail out because encryption won't work properly then. */
static int
check_default_key (gpgme_keycache_t kc)
{
    gpgme_key_t key;
    gpgme_error_t err = GPGME_No_Error;
    char * defkey;

    defkey = get_gnupg_default_key ();
    if (defkey)
	err = gpgme_keycache_find_key (kc, defkey, 0, &key);
    free_if_alloc (defkey);
    return err? -1 : 0;
} /* check_default_key */


/* Return the WinPT program file name (with full pathname). */
static const char *
get_prog_part (const char * fname, int use_cwd)
{
    static char program[1024];
    char currdir[256], * cmd = NULL;
    int j;
	
    memset (currdir, 0, DIM (currdir));
    memset (program, 0, DIM (program));
	
    if (use_cwd) {
	GetCurrentDirectory (DIM (currdir)-1, currdir);
	_snprintf (program, DIM (program)-1, "%s\\%s", currdir, fname);
    } 
    else {
	cmd = GetCommandLine ();
	if (cmd == NULL)
	    return NULL;
	strncpy (currdir, cmd, 255);
	j = strlen (currdir);
	while (j--) {
	    if (currdir[j] == '\\')
		break;		
	}
	currdir[j] = 0;
	_snprintf (program, DIM (program)-1, "%s\\%s", currdir + 1, fname);
    }
    return program;
} /* get_prog_part */


static int
check_crypto_engine (void)
{
    int ma=1, mi=2, pa=4; /* GPG 1.2.4 */
    int rc;

    rc = check_gnupg_engine (&ma, &mi, &pa);
    if (rc == -1) {
	msg_box (NULL, _("Could not read GnuPG version."), _("WinPT Error"), MB_ERR);
	return rc;
    }
    else if (rc) {
	log_box (_("WinPT Error"), MB_ERR,
		 _("Sorry, you need a newer GPG version.\n"
		   "GPG version %d.%d.%d requred GPG version 1.2.4"),
		   ma, mi, pa);
	return rc;
    }
    /* We enable smartcard support for GPG: 1.9.x or >= 1.4.0 */
    if (ma >= 1 && mi >= 4)
	scard_support = 1;

    gpgver[0] = ma;
    gpgver[1] = mi;
    gpgver[2] = pa;
    return rc;
} /* check_crypto_engine */


static int
load_keyserver_conf (int quiet)
{
    const char * t;
    int rc;

    if (reg_prefs.kserv_conf)
	t = reg_prefs.kserv_conf;
    else if (!file_exist_check (get_prog_part ("keyserver.conf", 0)))
	t = get_prog_part ("keyserver.conf", 0);
    else 
	t = "keyserver.conf";
    rc = kserver_load_conf (t);
    if (rc && !quiet)
	msg_box (NULL, winpt_strerror (rc), _("Keyserver"), MB_ERR);
    return rc;
}


static void
enable_mobile_mode (void)
{
    memset (&reg_prefs, 0, sizeof (reg_prefs));
    reg_prefs.always_trust = 0;
    reg_prefs.auto_backup = 0;
    reg_prefs.cache_time = 0;
    reg_prefs.expert = 0;
    reg_prefs.keylist_mode = 1;
    reg_prefs.kserv_conf = m_strdup ("keyserver.conf");
    reg_prefs.no_zip_mmedia = 1;
    reg_prefs.use_tmpfiles = 1;
    reg_prefs.word_wrap = 80;
    reg_prefs.use_viewer = 0; /* XXX */
}

char* get_subkey_fingerprint (gpgme_ctx_t ctx, const char *keyid);

int WINAPI
WinMain (HINSTANCE hinst, HINSTANCE hprev, LPSTR cmdline, int showcmd)
{
    WNDCLASS wc = {0, winpt_main_proc, 0, 0, hinst, 0, 0, 0, 0, PGM_NAME};
    HACCEL accel_tab;
    int rc, ec, created = 0, use_cwd = 0, nfiles = 0;
    int first_start = 0, start_gpgprefs = 0;
    const char * s;
    MSG msg;
    HWND hwnd;

    glob_hinst = hinst;
    
    gpgme_lib_init ();
    #ifdef _DEBUG
    gpgme_set_debug_mode (1);
    #endif
    gpgme_set_pgm_string ("WinPT "PGM_VERSION);

    s = PTD_get_version ();
    if (strcmp (s, "0.8.0")) {
	log_box (_("Privacy Tray Dynamic (PTD)"), MB_ERR,
		 _("Please update your PTD.dll to the newest version, "
		   "the version (%s) you use is too old."), s);
	return 0;
    }

    if (gpg_md_selftest ()) {
	msg_box (NULL, _("Cryptographic selftest failed."),
		 _("WinPT Error"), MB_ERR);
	return 0;
    }

    if (cmdline && stristr (cmdline, "--mobile"))
	mobile = 1;

    set_default_kserver ();

    if (!mobile) {
	regist_inst_gnupg (1);
	regist_inst_winpt (1, &created);
    }
    else {
	enable_mobile_mode ();
	/* XXX: ask for GPG path */
	created = 1; /* Disable registry writing */
    }

    if (!created) {
	memset (&reg_prefs, 0, sizeof (reg_prefs));
	reg_prefs.use_tmpfiles = 1; /* default */
	reg_prefs.fm.progress = 0; /* XXX: fix the bug and enable it again */
	get_reg_winpt_prefs (&reg_prefs);
	if (!reg_prefs.no_hotkeys)
	    hotkeys_modify ();
    }

    rc = gnupg_check_homedir ();
    if (rc) {
	log_box (_("WinPT Error"), MB_ERR, 
	         _("GPG home directory is not set correctly.\n"
		   "Please check the GPG registry settings:\n%s."),
		 winpt_strerror (rc));
	const char * s = get_filename_dlg (GetActiveWindow (), FILE_OPEN,
					   _("Select GPG Public Keyring"),
					   _("GPG Keyrings (*.gpg)\0*.gpg\0\0"),
					   NULL);
	if (s && !file_exist_check (s))
	{
	    size_t n;
	    char * p = strrchr (s, '\\');
	    if (!p)
		BUG (0);
	    n = p - s;
	    if (n) 
	    {
		char * file = new char[n+1];
		if (!file)
		    BUG (NULL);
		memset (file, 0, n);
		memcpy (file, s, n);
		file[n] = '\0';		
		set_reg_entry_gpg ("HomeDir", file);
		free_if_alloc (file);
		gnupg_check_homedir (); /* change gpgProgram if needed */
	    }
	}
	else {
	    msg_box (NULL, _("GPG home directory could not be determited."),
		     _("WinPT Error"), MB_ERR);
	    goto start;
	}
    }

    rc = check_gnupg_prog ();
    if (rc) {
	if (msg_box (NULL, _("Could not find the GPG binary (gpg.exe).\n"
			     "Do you want to start the GPG preferences to "
			     "correct  this problem?"), _("WinPT Error"),
			     MB_INFO|MB_YESNO) == IDYES)
	    start_gpgprefs = 1;
	else 
	{
	    msg_box (NULL, winpt_strerror (rc), _("WinPT Error"), MB_ERR);
	    return 0;
	}
    }

    rc = gnupg_access_files ();
    if (!start_gpgprefs && rc) 
    {
	if (rc == WPTERR_GPG_KEYRINGS || rc == WPTERR_GPG_OPT_KEYRINGS) 
	{
	    ec = msg_box (NULL,
		_("Could not access and/or find the public and secret keyring.\n"
		  "If this is an accident, quit the program and fix it.\n\n"
		  "Continue if you want that WinPT offers you more choices.\n"),
		  "WinPT", MB_INFO|MB_YESNO);
	    if (ec == IDYES)
		first_start = 1;
	}
	if (!first_start) 
	{
	    msg_box (NULL, winpt_strerror (rc), _("WinPT Error"), MB_ERR);
	    return 0;
	}
    }

    if (!first_start)
    {
	rc = gpg_check_permissions (1);
	if (rc && rc == 2)
	    gpg_read_only = 1;
	else if (rc)
	    return 0;
    }

    load_gettext ();
    init_gnupg_table ();

    nfiles = fm_parse_command_line (cmdline);
    if (nfiles > 0)
	return 0;

    if (cmdline && stristr (cmdline, "--wipe-freespace")) {
	dialog_box_param (glob_hinst, (LPCTSTR)IDD_WINPT_SPACE_SECDEL,
			    GetDesktopWindow(), space_wipefrees_dlg_proc, NULL,
			    _("Wipe Free Space"), IDS_WINPT_SPACE_SECDEL);
	free_gnupg_table ();
	return 0;
    }

    load_keyserver_conf (cmdline? 1 : 0);
    if (cmdline && (stristr (cmdline, "--keymanager")
	        || stristr (cmdline, "--cardmanager"))) {
	update_keycache (GetDesktopWindow ());
	if (stristr (cmdline, "keymanager"))
	    dialog_box_param (glob_hinst, (LPCTSTR)IDD_WINPT_KEYMISC,
			    GetDesktopWindow(), keymanager_dlg_proc, NULL,
			    _("Key Manager"), IDS_WINPT_KEYMISC);
	else {
	    gpgme_card_t crd = smartcard_init ();
	    if (crd)
		dialog_box_param (glob_hinst, (LPCTSTR)IDD_WINPT_CARD_EDIT,
				  GetDesktopWindow(), card_edit_dlg_proc,
				  (LPARAM)crd, _("Card Manager"),
				  IDS_WINPT_CARD_EDIT);
	    gpgme_card_release (crd);
	}
	keycache_release ();
	free_gnupg_table ();
	return 0;
    }

    CreateMutex (NULL, TRUE, PGM_NAME);
    if (GetLastError () == ERROR_ALREADY_EXISTS) {
	free_gnupg_table ();
	return 0;
    }

    if (cmdline) {
	if (stristr (cmdline, "--enable-debug") || stristr (cmdline, "--debug")) {
	    gpgme_set_debug_mode (1);
	    winpt_debug_msg ();
	    debug = 1;
	}
    }

    wc.hIcon = LoadIcon (glob_hinst, MAKEINTRESOURCE (IDI_WINPT));
    rc = RegisterClass (&wc);
    if (rc == FALSE) {
	msg_box (NULL, _("Could not register window class"), _("WinPT Error"), MB_ERR);
	free_gnupg_table ();
	return 0;
    }

    hwnd = CreateWindow (PGM_NAME, 
			 PGM_NAME, 
			 0, 0, 0, 0, 0, 
			 NULL, 
			 NULL, 
			 hinst, 
			 NULL);
    if (hwnd == NULL) {
    	msg_box (NULL, _("Could not create window"), _("WinPT Error"), MB_ERR);
	free_gnupg_table ();
	return 0;
    }
    glob_hwnd = hwnd;
    UpdateWindow (hwnd);

    if (!first_start && !start_gpgprefs) {
	gnupg_backup_options (1);
	gnupg_backup_options (0);
	
	rc = check_crypto_engine ();
	if (rc) {
	    DestroyWindow (hwnd);
	    free_gnupg_table ();
	    return 0;
	}
    }

    if (start_gpgprefs) 
    {
	char *ring;
	size_t size = 0;
	DialogBoxParam (glob_hinst, (LPCTSTR)IDD_WINPT_GPGPREFS, hwnd,
			gpgprefs_dlg_proc, NULL);
	ring = get_gnupg_keyring (0, !NO_STRICT);
	if (gnupg_access_keyring (0) == -1 && get_file_size (ring) == 0)
	    first_start = 1; /* The keyring is empty! */
	free_if_alloc (ring);
    }

    if (first_start) {
	struct key_wizard_s c, dummy;
start:
	DialogBoxParam (glob_hinst, (LPCSTR)IDD_WINPT_FIRST, hwnd,
			first_run_dlg_proc, (LPARAM)&dummy);
	switch (dummy.interactive) 
	{
	case SETUP_KEYGEN:
	    c.interactive = 1;
	    rc = DialogBoxParam (glob_hinst, (LPCSTR)IDD_WINPT_KEYWIZARD,
				 hwnd, keygen_wizard_dlg_proc, (LPARAM)&c);
	    if (!rc)
		goto start;
	    break;

	case SETUP_IMPORT:
	    rc = gnupg_copy_keyrings ();
	    if (rc) {
		msg_box (hwnd, winpt_strerror (rc), _("WinPT Error"), MB_ERR);
		goto start;
	    }
	    break;

	case SETUP_EXISTING:
	    DialogBoxParam (glob_hinst, (LPCTSTR)IDD_WINPT_GPGPREFS, hwnd,
			    gpgprefs_dlg_proc, NULL);
	    break;

	case -1:
	    DestroyWindow (hwnd);
	    free_gnupg_table ();
	    return 0;
	}
	update_keycache (hwnd);
	check_crypto_engine ();
    }
    else {
	gpgme_keycache_t c;
	update_keycache (hwnd);
	c = keycache_get_ctx (1);
	if (!c || !gpgme_keycache_count (c)) {
	    gnupg_display_error ();
	    msg_box (hwnd, _("The keycache was not initialized or is empty.\n"
			     "Please check your GPG config (keyrings, pathes...)"),
			     _("WinPT Error"), MB_ERR);
	    ec = msg_box (NULL, _("It seems that GPG is not set properly.\n"
				  "Do you want to start the GPG preferences dialog?"),
			    "WinPT", MB_INFO|MB_YESNO);
	    if (ec == IDYES) {
		DialogBoxParam (glob_hinst, (LPCTSTR)IDD_WINPT_GPGPREFS, hwnd,
				gpgprefs_dlg_proc, NULL);
		update_keycache (hwnd);
	    }
	    else {
		DestroyWindow (hwnd);
		free_gnupg_table ();
		return 0;
	    }
	}
	if (check_default_key (c)) {
	    char * p = get_gnupg_default_key ();
	    log_box (_("WinPT Error"), MB_ERR, 
		     _("Default key from the GPG options file could not be found.\n"
		       "Please check your gpg.conf (options) to correct this:\n\n"
		       "%s: public key not found."), p? p : "[null]");
	    free_if_alloc (p);
	    DestroyWindow (hwnd);
	    free_gnupg_table ();
	    return 0;
	}
	if (count_insecure_elgkeys ())
	    DialogBoxParam (glob_hinst, (LPCTSTR)IDD_WINPT_ELGWARN, glob_hwnd,
			    elgamal_warn_dlg_proc, NULL);
    }

    accel_tab = LoadAccelerators (glob_hinst, (LPCTSTR)IDR_WINPT_ACCELERATOR);
    keyring_check_last_access (); /* init */
    while (GetMessage (&msg, hwnd, 0, 0)) {
	if (!TranslateAccelerator (msg.hwnd, accel_tab, &msg)) {
	    TranslateMessage (&msg);
	    DispatchMessage (&msg);
	}
    }
	
    return 0;
} /* WinMain */
