/* wptFileMangerDlg.cpp - File Manager
 *	Copyright (C) 2001-2005 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <windows.h>
#include <commctrl.h>

#include "../resource.h"
#include "wptCommonCtl.h"
#include "wptGPG.h"
#include "wptW32API.h"
#include "wptKeylist.h"
#include "wptErrors.h"
#include "wptContext.h" /* for passphrase_s */
#include "wptDlgs.h"
#include "wptTypes.h"
#include "wptNLS.h"
#include "wptVersion.h"
#include "wptFileManager.h"

#define send_cmd_id(hwnd, id) \
    PostMessage( (hwnd), WM_COMMAND, MAKEWPARAM( (id), 0), 0 )

static subclass_s filelist_proc;

struct thread_ctx_s {
    HWND   dlg;
    HANDLE fd;
    char * drive;
};
typedef struct thread_ctx_s * thread_ctx_t;


BOOL CALLBACK
file_encrypt_dlg_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    static listview_ctrl_t lv = NULL;
    static fm_state_t c;
    static int req_signer = 0;
    static keylist_t list = NULL;
    gpgme_keycache_t kc;
    gpgme_ctx_t ctx;
    int force_trust = 0, count = 0;
    
    switch (msg) {
    case WM_INITDIALOG:
	req_signer = 0;
        c = (fm_state_t)lparam;
        if (!c)
            dlg_fatal_error( dlg, "Could not get dialog state!" );
        kc = keycache_get_ctx (KEYCACHE_PUB);
        if (!kc)
            BUG (dlg);
#ifndef LANG_DE
        SetWindowText( dlg, _("File Encrypt") );
        SetDlgItemText( dlg, IDC_ENCRYPT_ARMOR, _("&Text Output") );
	SetDlgItemText( dlg, IDC_ENCRYPT_WIPE, _("&Wipe Original") );
#endif
	if (c->req_signer) {
	    EnableWindow (GetDlgItem (dlg, IDC_ENCRYPT_SIGN), TRUE);
	    seclist_init (dlg, IDC_ENCRYPT_SECLIST, KEYLIST_FLAG_SHORT, &list);
	}
        lv = keylist_load (GetDlgItem (dlg, IDC_ENCRYPT_LIST), kc, NULL,
			    KEYLIST_ENCRYPT|KEYLIST_FLAG_FILE, GPGME_ATTR_USERID);
        SetForegroundWindow (dlg);
        return TRUE;
        
    case WM_DESTROY:
	seclist_destroy (&list);
        if (lv) {
            keylist_delete (lv);
            lv = NULL;
        }
        return FALSE;

    case WM_NOTIFY:
	NMHDR *notify;
        notify = (NMHDR *)lparam;
	if( notify && notify->code == LVN_COLUMNCLICK 
	    && notify->idFrom == IDC_ENCRYPT_LIST ) {
	    NMLISTVIEW *p = (LPNMLISTVIEW) lparam;
	    int sortby = 0;

	    switch( p->iSubItem ) {
	    case 0: sortby = GPGME_ATTR_USERID; break;
	    case 1: sortby = GPGME_ATTR_KEYID; break;
	    case 2: sortby = GPGME_ATTR_LEN; break;
	    case 4: sortby = GPGME_ATTR_VALIDITY; break;
	    default:sortby = GPGME_ATTR_USERID; break;
	    }
	    keylist_sort( lv, sortby );
	}
        
    case WM_SYSCOMMAND:
        if( LOWORD( wparam ) == SC_CLOSE ) {
            c->cancel = 1;
            EndDialog( dlg, TRUE );
        }
        return FALSE;
        
    case WM_COMMAND:
	if (HIWORD (wparam) == BN_CLICKED
	    && LOWORD (wparam) == IDC_ENCRYPT_SIGN) {
	    req_signer ^= 1;
	    EnableWindow (GetDlgItem (dlg, IDC_ENCRYPT_SECLIST),
			  req_signer? TRUE : FALSE);
	}
        switch( LOWORD( wparam ) ) {
        case IDOK:
            ctx = c->ctx;
	    gpgme_recipients_release (c->recp);
            c->recp = keylist_get_recipients (lv, &force_trust, &count);
	    if( !gpgme_recipients_count (c->recp) ) {
		msg_box (dlg, _("Please select at least one recipient."),
			 _("File Encrypt"), MB_ERR);
		gpgme_recipients_release(c->recp);
		c->recp = NULL;
                c->cancel = 1;
                return FALSE;
            }
	    if (IsDlgButtonChecked (dlg, IDC_ENCRYPT_SIGN)) {
		gpgme_key_t key;

		if (seclist_select_key (dlg, IDC_ENCRYPT_SECLIST, &key))
		    return FALSE;
		else
		    gpgme_signers_add (c->ctx, key);
	    }

            gpgme_control( ctx, GPGME_CTRL_FORCETRUST, force_trust );
            if (IsDlgButtonChecked (dlg, IDC_ENCRYPT_ARMOR))
                gpgme_control (ctx, GPGME_CTRL_ARMOR, 1);
	    if (IsDlgButtonChecked (dlg, IDC_ENCRYPT_WIPE))
		c->wipe = 1;
	    if (force_trust)
		gpgme_control (ctx, GPGME_CTRL_FORCETRUST, 1);
	    if (IsDlgButtonChecked (dlg, IDC_ENCRYPT_ANON))
		gpgme_control (ctx, GPGME_CTRL_THROWKEYID, 1);
            c->cancel = 0;
            EndDialog (dlg, TRUE);
            return TRUE;
            
        case IDCANCEL:
            c->cancel = 1;
            EndDialog (dlg, FALSE);
            return FALSE;
        }
        return TRUE;
    }
    
    return FALSE;
} /* file_encrypt_dlg_proc */


BOOL CALLBACK
file_import_dlg_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    static listview_ctrl_t lv = NULL;
    static fm_state_t c;
    static int is_revcert = 0, has_seckeys = 0;
    
    switch( msg ) {
    case WM_INITDIALOG:
        c = (fm_state_s *)lparam;
        if (!c)
            dlg_fatal_error( dlg, "Could not get dialog param!" );
#ifndef LANG_DE
        SetWindowText (dlg, _("File Import"));
#endif
        ShowWindow( GetDlgItem( dlg, IDC_IMPORT_KEYMNGT ), WS_DISABLED );
	/* fixme: use the new code */
        implist_build( &lv, GetDlgItem( dlg, IDC_IMPORT_KEYLIST ) );
        implist_load( lv, c->opaque, &is_revcert, &has_seckeys );
        SetForegroundWindow( dlg );
	if( !listview_count_items( lv, 0 ) ) {
	    msg_box( dlg, _("No valid OpenPGP data found."), _("File Import"), MB_ERR );
	    c->cancel = 1;
	    EndDialog( dlg, FALSE );
	}
        return TRUE;
        
    case WM_DESTROY:
        if( lv ) {
            listview_release( lv );
            lv = NULL;
        }
        return FALSE;
        
    case WM_SYSCOMMAND:
        if( LOWORD( wparam ) == SC_CLOSE ) {
            c->cancel = 1;
            EndDialog( dlg, FALSE );
        }
        return FALSE;
        
    case WM_COMMAND:
        switch( LOWORD( wparam ) ) {
        case IDC_IMPORT_DOIT:
            c->cancel = 0;
	    c->implist_revcert = is_revcert? 1 : 0;
            EndDialog( dlg, TRUE );
            return TRUE;
            
        case IDCANCEL:
            c->cancel = 1;
            EndDialog( dlg, FALSE );
            return FALSE;
        }
        break;
    }
    
    return FALSE;
} /* file_import_dlg_proc */


BOOL CALLBACK
file_sign_dlg_proc( HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam )
{
    static fm_state_t c;
    static listview_ctrl_t lv;
    gpgme_keycache_t kc, sec_kc;    
    gpgme_key_t key;
    int lv_idx;
    char signer[32];
    
    switch( msg ) {
    case WM_INITDIALOG:
        c = (fm_state_s *)lparam;
        if( c == NULL )
            dlg_fatal_error( dlg, "Could not get dialog state!" );
	#ifndef LANG_DE
        SetWindowText (dlg, _("File Sign"));
        SetDlgItemText (dlg, IDC_FILE_SIG_ARMOR, _("&Text Output") );
        SetDlgItemText (dlg, IDC_FILE_SIG_NORMAL, _("&Normal Signature"));
        SetDlgItemText (dlg, IDC_FILE_SIG_DETACH, _("&Detached Signature"));
        SetDlgItemText (dlg, IDC_FILE_SIG_CLEAR, _("&Cleartext Signature"));
	#endif
	CheckDlgButton (dlg, IDC_FILE_SIG_DETACH, BST_CHECKED);
        kc = keycache_get_ctx (KEYCACHE_PUB);
        if (!kc)
            BUG (dlg);
        sec_kc = keycache_get_ctx (KEYCACHE_PRV);
	if (!sec_kc)
	    BUG( dlg );
	lv = keylist_load( GetDlgItem(dlg, IDC_FILE_SIG_LIST), kc, sec_kc,
			    KEYLIST_SIGN, GPGME_ATTR_USERID );
        SetForegroundWindow( dlg );
        return TRUE;
        
    case WM_DESTROY:
        if( lv ) {
            listview_release( lv );
            lv = NULL;
        }
        return FALSE;
        
    case WM_SYSCOMMAND:
        if( LOWORD (wparam) == SC_CLOSE )
            EndDialog( dlg, TRUE );
        return FALSE;
        
    case WM_COMMAND:
        switch( LOWORD( wparam ) ) {
        case IDOK:
            if( IsDlgButtonChecked( dlg, IDC_FILE_SIG_ARMOR ) )
                gpgme_control( c->ctx, GPGME_CTRL_ARMOR, 1 );
            if( IsDlgButtonChecked( dlg, IDC_FILE_SIG_NORMAL ) )
                c->sigmode = GPGME_SIG_MODE_NORMAL;
            else if( IsDlgButtonChecked( dlg, IDC_FILE_SIG_DETACH ) )
                c->sigmode = GPGME_SIG_MODE_DETACH;
            else if( IsDlgButtonChecked( dlg, IDC_FILE_SIG_CLEAR ) )
                c->sigmode = GPGME_SIG_MODE_CLEAR;
            else {
                /* default is to produce detached signatures */
                c->sigmode = GPGME_SIG_MODE_DETACH;
            }
            lv_idx = listview_get_curr_pos( lv );
            if (lv_idx == -1) {
		if (listview_count_items (lv, 0) == 1)
		    lv_idx = 0;
		else {
		    msg_box (dlg, _("Please select a key."), _("Sign"), MB_ERR);
		    return FALSE;
		}
	    }
            listview_get_item_text( lv, lv_idx, 1, signer, sizeof signer-1 );
	    if( get_seckey( signer, &key ) )
		BUG( NULL );
	    gpgme_signers_clear (c->ctx);
	    gpgme_signers_add (c->ctx, key);
	    free_if_alloc (c->opaque);
	    c->opaque = m_strdup (signer); /* store for later use */
            c->cancel = 0;
	    if (!c->opaque)
		BUG (0);
            EndDialog (dlg, TRUE);
            return TRUE;
            
        case IDCANCEL:
            c->cancel = 1;
            EndDialog (dlg, FALSE);
            return FALSE;
        }
        break;
    }
    
    return FALSE;
} /* file_sign_dlg_proc */



BOOL CALLBACK
filelist_subclass_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    switch (msg)
    {
	case WM_KEYUP:
        int virt_key = (int)wparam;
        if (virt_key == VK_DELETE)
	    send_cmd_id (filelist_proc.dlg, ID_FILEMISC_WIPE);
	break;
    }
    return CallWindowProc (filelist_proc.old, dlg, msg, wparam, lparam);
}


BOOL CALLBACK
file_manager_dlg_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    static listview_ctrl_t lv = NULL;
    static HMENU menu = NULL;
    static int always_ontop = 1;
    HWND fl;
    
    switch (msg) 
    {
    case WM_INITDIALOG:
	always_ontop = 1;
	menu = LoadMenu (glob_hinst, (LPCSTR)IDR_WINPT_FILEMISC);
        SetWindowText( dlg, _("File Manager (use drag & drop to add files)") );
        set_menu_text( menu, ID_FILEMISC_OPEN, _("&Open...") );
        set_menu_text( menu, ID_FILEMISC_ENCRYPT, _("&Encrypt") );
        set_menu_text( menu, ID_FILEMISC_DECRYPT, _("&Decrypt") );
        set_menu_text( menu, ID_FILEMISC_SIGN, _("&Sign") );
        set_menu_text( menu, ID_FILEMISC_SIGNENC, _("Sign && Encrypt") );
        set_menu_text( menu, ID_FILEMISC_VERIFY, _("&Verify") );
        set_menu_text( menu, ID_FILEMISC_WIPE, _("&Wipe") );
        set_menu_text( menu, ID_FILEMISC_SYMENC, _("S&ymmetric") );
        set_menu_text( menu, ID_FILEMISC_IMPORT, _("&Import") );
        set_menu_text( menu, ID_FILEMISC_EXPORT, _("E&xport") );
        set_menu_text( menu, ID_FILEMISC_QUIT, _("Exit"));
        set_menu_text( menu, ID_FILEMISC_PREFS, _("Preferences") );
        set_menu_text( menu, ID_FILEMISC_CLEAR, _("Clear List") );
	fl = GetDlgItem (dlg, IDC_FILE_LIST);
	filelist_proc.dlg = dlg;
	filelist_proc.current = (WNDPROC)filelist_subclass_proc;
	filelist_proc.old = (WNDPROC)GetWindowLong (fl, GWL_WNDPROC);
	if (filelist_proc.old) {
	    if (!SetWindowLong (fl, GWL_WNDPROC, (LONG)filelist_proc.current)) {
		msg_box (dlg, _("Could not set filelist window procedure."), _("Key Manager"), MB_ERR);
		BUG (0);
	    }
	}
        SetMenu (dlg, menu);
        fm_build (&lv, GetDlgItem (dlg, IDC_FILE_LIST));
	center_window2 (dlg, HWND_TOPMOST);
	center_window (dlg);
        SetForegroundWindow (dlg);
        return TRUE;

    case WM_DESTROY:
        if (lv) {
            fm_delete (lv);
            lv = NULL;
        }
        if (menu) {
            DestroyMenu (menu);
            menu = NULL;
        }
        return FALSE;
        
    case WM_DROPFILES:
        fm_add_dropped_files (lv, (HDROP)wparam);
        return TRUE;
        
    case WM_NOTIFY:
        NMHDR *notify;
	int pos;
	HMENU hm;

        notify = (NMHDR *)lparam;
        if (notify && notify->code == NM_RCLICK
            && notify->idFrom == IDC_FILE_LIST) {
            POINT p;
            HMENU popup;
            
            GetCursorPos (&p);
            hm = LoadMenu (glob_hinst, MAKEINTRESOURCE (IDR_WINPT_FILEMISC_CTX));
            popup = GetSubMenu (hm, 0);
            set_menu_text (popup, ID_FILECTX_ENCRYPT, _("Encrypt"));
            set_menu_text (popup, ID_FILECTX_DECRYPT, _("Decrypt"));
            set_menu_text (popup, ID_FILECTX_SIGN, _("Sign"));
            set_menu_text (popup, ID_FILECTX_SIGNENC, _("Sign && Encrypt"));
            set_menu_text (popup, ID_FILECTX_VERIFY, _("Verify"));
            set_menu_text (popup, ID_FILECTX_WIPE, _("Wipe"));
	    set_menu_text (popup, ID_FILECTX_SEND, _("Send as Mail"));
	    pos = listview_get_curr_pos (lv);
	    if (fm_check_file_type (lv, pos, FM_DECRYPT))
		set_menu_state (popup, ID_FILECTX_SEND, MF_ENABLED);
	    if (fm_check_file_type (lv, pos, FM_ENCRYPT)) {
		set_menu_state (popup, ID_FILECTX_DECRYPT, MF_GRAYED|MF_DISABLED);
		set_menu_state (popup, ID_FILECTX_VERIFY, MF_GRAYED|MF_DISABLED);
	    }
	    if (fm_check_file_type (lv, pos, FM_DECRYPT)) {
		set_menu_state (popup, ID_FILECTX_ENCRYPT, MF_GRAYED|MF_DISABLED);
		set_menu_state (popup, ID_FILECTX_SIGN, MF_GRAYED|MF_DISABLED);
		set_menu_state (popup, ID_FILECTX_SIGNENC, MF_GRAYED|MF_DISABLED);
	    }
	    if (fm_check_file_type (lv, pos, FM_VERIFY)) {
		set_menu_state (popup, ID_FILECTX_VERIFY, MF_ENABLED);
		set_menu_state (popup, ID_FILECTX_SIGN, MF_GRAYED|MF_DISABLED);
		set_menu_state (popup, ID_FILECTX_SIGNENC, MF_GRAYED|MF_DISABLED);
	    }

            TrackPopupMenu(popup, TPM_RIGHTALIGN, p.x, p.y, 0, dlg, NULL);
            DestroyMenu (popup);
            DestroyMenu (hm);
        }
	if (notify && notify->code == NM_CLICK
            && notify->idFrom == IDC_FILE_LIST) {
	    hm = GetMenu (dlg);
	    pos = listview_get_curr_pos (lv);
	    if (!fm_check_file_type (lv, pos, FM_DECRYPT))
		set_menu_state (hm, ID_FILEMISC_SEND, MF_GRAYED|MF_DISABLED);
	    else
		set_menu_state (hm, ID_FILEMISC_SEND, MF_ENABLED);
	}
        return TRUE;
        
    case WM_SYSCOMMAND:
        if( LOWORD( wparam ) == SC_CLOSE )
            EndDialog( dlg, TRUE );
        return FALSE;
        
    case WM_COMMAND:
        switch( LOWORD( wparam ) ) {        
        case ID_FILEMISC_QUIT:
            EndDialog( dlg, TRUE );
            return TRUE;

	case ID_FILEMISC_WIPE_FREES:
	    DialogBoxParam (glob_hinst, (LPCTSTR)IDD_WINPT_SPACE_SECDEL, dlg,
			    space_wipefrees_dlg_proc, NULL);
	    break;
            
	case ID_FILEMISC_SELALL:
	    listview_select_all (lv);
	    break;

	case ID_FILEMISC_PASTE:
	    fm_verify_pasted_detsig (lv, dlg);
	    break;

	case ID_FILEMISC_ONTOP:
	    always_ontop ^= 1;	    		
	    set_menu_state (menu, ID_FILEMISC_ONTOP, 
			    always_ontop == 0? MFS_UNCHECKED : MFS_CHECKED);
	    center_window2 (dlg, always_ontop == 0? HWND_NOTOPMOST : HWND_TOPMOST);
	    break;

	case ID_FILEMISC_SEND:
	    fm_send_file (lv);
	    break;

        case ID_FILEMISC_LIST:
        case ID_FILEMISC_ENCRYPT:
        case ID_FILEMISC_SYMENC:
        case ID_FILEMISC_DECRYPT:
        case ID_FILEMISC_SIGN:
        case ID_FILEMISC_SIGNENC:
        case ID_FILEMISC_VERIFY:
        case ID_FILEMISC_WIPE:
            fm_parse_files (lv, dlg, LOWORD (wparam));
	    SetForegroundWindow (dlg);
            return TRUE;
	    
	case ID_FILEMISC_MD_MD5:
	case ID_FILEMISC_MD_SHA1:
	case ID_FILEMISC_MD_RMD160:
	    fm_print_md (lv, dlg, LOWORD (wparam)-40136);
	    SetForegroundWindow (dlg);
	    return TRUE;
            
        case ID_FILEMISC_CLEAR:
            listview_del_all( lv );
            return TRUE;
            
        case ID_FILEMISC_OPEN:
            fm_add_opened_files( lv, dlg );
            return TRUE;
            
            /* context menu entries */
        case ID_FILECTX_ENCRYPT: send_cmd_id( dlg, ID_FILEMISC_ENCRYPT); break;
        case ID_FILECTX_DECRYPT: send_cmd_id( dlg, ID_FILEMISC_DECRYPT );break;
        case ID_FILECTX_SIGN:    send_cmd_id( dlg, ID_FILEMISC_SIGN );   break;
        case ID_FILECTX_SIGNENC: send_cmd_id( dlg, ID_FILEMISC_SIGNENC );break;
        case ID_FILECTX_VERIFY:  send_cmd_id( dlg, ID_FILEMISC_VERIFY ); break;
        case ID_FILECTX_WIPE:    send_cmd_id( dlg, ID_FILEMISC_WIPE );   break;
        case ID_FILECTX_LIST:    send_cmd_id( dlg, ID_FILEMISC_LIST );   break;
	case ID_FILECTX_SEND:    send_cmd_id( dlg, ID_FILEMISC_SEND );   break;
        }
        break;
    }
    
    return FALSE;
} /* file_manager_dlg_proc */


BOOL CALLBACK
file_secdel_confirm_dlg_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    static secdel_confirm_s * ctx;
    const char * s;
    void * tmp;

    switch (msg) {
    case WM_INITDIALOG:
	ctx = (struct secdel_confirm_s *)lparam;
	if( !ctx )
	    BUG( NULL );
	gpgme_recipients_enum_open( ctx->rset, &tmp );
	while( (s =gpgme_recipients_enum_read( ctx->rset, &tmp )) )
	    SendDlgItemMessage( dlg, IDC_SECDEL_FILES, LB_ADDSTRING, 
				NULL, (WPARAM)(char *)s );
	break;

    case WM_COMMAND:
	switch( LOWORD( wparam ) ) {
	case IDYES:
	    ctx->yes = 1;
	    EndDialog( dlg, TRUE );
	    break;

	case IDNO:
	    ctx->yes = 0;
	    EndDialog( dlg, FALSE );
	    break;
	}
	break;
    }

    return FALSE;
} /* file_secdel_confirm_dlg_proc */


static void
wipe_cb (void * ctx, ui64 off, ui64 n)
{    
    SendDlgItemMessage ((HWND)ctx, IDC_SPACE_PROGRESS, PBM_SETPOS, 
			(WPARAM) (off*100/n), 0);
    SetDlgItemInt ((HWND)ctx, IDC_SPACE_OFF, (UINT)(off/1024), FALSE);
    SetDlgItemInt ((HWND)ctx, IDC_SPACE_N, (UINT)(n/1024), FALSE);
} /* wipe_cb */


static DWORD WINAPI
wipe_thread (void * ctx)
{     
    thread_ctx_t a = (thread_ctx_t)ctx;
    int rc;

    rc = wipe_freespace (a->drive, &a->fd, wipe_cb, a->dlg);
    SetDlgItemText (a->dlg, IDC_SPACE_INFO, 
		    rc? _("Operation Status: Error") : _("Operation Status: Done."));
    if (rc)
	msg_box (a->dlg, winpt_strerror (rc), _("Wipe Free Space"), MB_ERR);
    else
	MessageBeep (MB_OK);
    ExitThread (0);
    return 0;
} /* wipe_thread */


static void
fill_drive_combobox (HWND dlg)
{
    char buf[384], p[16];
    int n, i, j = 0;

    n = GetLogicalDriveStrings (DIM (buf)-1, buf);
    while (j < n) {
	for (i = 0; buf[j] != '\0'; i++, j++)
	    p[i] = buf[j];
	p[i] = '\0'; 
	j++;
	if (GetDriveType (p) != DRIVE_CDROM)
	    SendDlgItemMessage (dlg, IDC_SPACE_DRIVES, CB_ADDSTRING, 
				0, (LPARAM)(const char *)p);
    }
    SendDlgItemMessage (dlg, IDC_SPACE_DRIVES, CB_SETCURSEL, 0, 0);
} /* fill_drive_combobox */


static void
update_disc_stats (HWND dlg)
{    
    ULARGE_INTEGER caller = {0}, total = {0}, free = {0};
    char buf[128], drive[32], a_total[64] = {0}, a_free[64] = {0};
    int idx;
    
    idx = SendDlgItemMessage (dlg, IDC_SPACE_DRIVES, CB_GETCURSEL, 0, 0);
    GetDlgItemText (dlg, IDC_SPACE_DRIVES, drive, DIM (drive)-1);
    if (GetDiskFreeSpaceEx (drive, &caller, &total, &free)) {
	_ui64toa (total.QuadPart/1024, a_total, 10);
	_ui64toa (free.QuadPart/1024, a_free, 10);
    }
    _snprintf (buf, DIM (buf)-1, _("Total Capacity: %12sk\n"
			           "Free Space    : %12sk"), a_total, a_free);
    SetDlgItemText(dlg, IDC_SPACE_DISC_INFO, buf);
    SetDlgItemInt (dlg, IDC_SPACE_OFF, 0, FALSE);
    SetDlgItemInt (dlg, IDC_SPACE_N, 0, FALSE);
} /* update_disc_stats */


BOOL CALLBACK
space_wipefrees_dlg_proc (HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam)
{
    static HANDLE thread_hd = NULL;
    static thread_ctx_s ctx;
    char drive[32], * p;
    int idx;
    DWORD tid;

    switch (msg) {
    case WM_INITDIALOG:	
	fill_drive_combobox (dlg);
	update_disc_stats (dlg);
	center_window (dlg);
	SetForegroundWindow (dlg);
	break; 
	
    case WM_DESTROY:
	thread_hd = NULL;
	free_if_alloc (ctx.drive);
	memset (&ctx, 0, sizeof ctx);
	break;

    case WM_COMMAND:
	if (HIWORD (wparam) == CBN_SELCHANGE)
	    update_disc_stats (dlg);
	switch (LOWORD (wparam)) 
	{
	case IDOK:
	    idx = SendDlgItemMessage (dlg, IDC_SPACE_DRIVES, CB_GETCURSEL, 0, 0);
	    if (idx != CB_ERR && !thread_hd) {
		GetDlgItemText (dlg, IDC_SPACE_DRIVES, drive, DIM (drive)-1);
		ctx.dlg = dlg;
		ctx.drive = m_strdup (drive);
		ctx.fd = NULL;
		thread_hd = CreateThread (NULL, 0, wipe_thread, &ctx, 0, &tid);
	    }
	    if (WaitForSingleObject (thread_hd, 0) == WAIT_OBJECT_0) {
		CloseHandle (thread_hd);
		thread_hd = NULL;
		EndDialog (dlg, TRUE);
	    }
	    break;

	case IDCANCEL:
	    if (thread_hd) {
		WaitForSingleObject (thread_hd, 2000);
		TerminateThread (thread_hd, 1);
		CloseHandle (thread_hd);
		CloseHandle (ctx.fd);
		p = make_filename (ctx.drive, "temp_winpt.tmp", NULL);
		if (p && !file_exist_check (p))
		    unlink (p);
		free_if_alloc (p);
	    }
	    EndDialog( dlg, FALSE );
	    break;
	}
	break;
    }

    return FALSE;
} /* space_wipefrees_dlg_proc */
