/* wptFileManager.cpp - File Manager routines
 *	Copyright (C) 2001-2005 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation; either version 2 
 * of the License, or (at your option) any later version.
 *  
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */
/* TODO: 
 *    check_armor_type: we should check the whole file and not only the first line! 
 */

#include <sys/types.h>
#include <windows.h>
#include <commdlg.h>
#include <io.h>

#include "../resource.h"
#include "wptTypes.h"
#include "wptGPG.h"
#include "wptAgent.h"
#include "wptCommonCtl.h"
#include "wptContext.h"
#include "wptErrors.h"
#include "wptKeylist.h"
#include "wptFileManager.h"
#include "wptNLS.h"
#include "wptW32API.h"
#include "wptVersion.h"
#include "wptDlgs.h"
#include "wptGPGZIP.h"
#include "wptUTF8.h"
#include "wptRegistry.h"

#include "openpgp.h"

int  algo_from_list (gpgme_recipients_t rset, const char * keyid);
void progress_cleanup (progress_filter_s * pfx);

/*-- wptFileVerifyDlg.cpp --*/
int  file_verify_add_state (siglog_context_t c);
void file_verify_use_event (void);
void file_verify_wait (void);

static const char * mm_files[] = {".mov", ".avi", ".mpg", ".mpeg",
				  ".mp3", ".wav", ".mid", ".wma",
				  ".gif", ".jpg", ".png", ".jpeg", ".dib", 0};

char *
fm_quote_file (const char * name)
{
    char * p;
    size_t len = strlen (name) + 8;

    if (*name == '"')
	return m_strdup (name); /* avoid double quotes */
    p = new char[len + 1];
    if (!p)
	BUG (0);
    _snprintf (p, len, "\"%s\"", name);

    return p;
} /* fm_quote_file */


int
overwrite_file (const char * fname)
{
    int id;

    if (file_exist_check (fname))
	return 1;
    id = log_box (_("File Manager"), MB_YESNO,
		  _("\"%s\" already exists.\n"
		    "Replace existing file?"), fname);
    return id == IDNO ? 0 : 1;
} /* overwrite_file */


static void
remove_crit_file_attrs (const char * fname, int force)
{
    u32 f_attr;
    int id;

    if (file_exist_check (fname))
	return; /* Does not exist */
	
    f_attr = GetFileAttributes (fname);
    if ((f_attr & FILE_ATTRIBUTE_READONLY) && force)
	SetFileAttributes (fname, FILE_ATTRIBUTE_NORMAL);
    else if (f_attr & FILE_ATTRIBUTE_READONLY) {
        id = log_box (_("File Manager"), MB_YESNO, 	    
	       	      _("\"%s\" has read-only attribute.\n"
		        "Set attribute to normal?"), fname);
	if (id == IDYES)
	    SetFileAttributes (fname, FILE_ATTRIBUTE_NORMAL);
    }
} /* remove_crit_file_attrs */


static int inline
is_directory (const char * fname)
{    
    return GetFileAttributes (fname) & FILE_ATTRIBUTE_DIRECTORY? 1 : 0;
} /* is_directory */


static int inline
is_openpgp_ext (const char * name)
{
    if (strstr (name, ".gpg") || strstr (name, ".asc")
	|| strstr (name, ".sig") || strstr (name, ".pgp"))
	return -1;
    return 0;
}


static int
is_multi_media (const char * name)
{    
    const char * val;
    char * p;
    int i;
    int ans=0;

    i = get_reg_winpt_single (CFG_NOZIP_MMEDIA);
    if (i == -1) 
    {
	ans = msg_box (NULL, _("Multi-Media files are already compressed, GPG would compress\n"
	                       "them anyway and this costs a lot of time.\n"
			       "It is possible to disable compression for these files.\n"
			       "Do you want to disable it?"), 
			       _("File Manager"), MB_YESNO|MB_INFO);
	set_reg_winpt_single (CFG_NOZIP_MMEDIA, ans == IDYES? 1 : 0);
	if (ans == IDNO)
	    return 0;
    }
    else if (i == 0)
	return 0;

    p = strrchr (name, '.');
    if (!p)
	return 0;
    for (i=0; (val = mm_files[i]); i++) 
    {
	if (!stricmp (p, val))
	    return -1;
    }
    return 0;
}


const char*
file_get_extension (gpgme_ctx_t ctx, gpgme_sigmode_t sigmode)
{
    int use_armor = (int)gpgme_control (ctx, GPGME_CTRL_ARMOR, -1);

    if (use_armor || sigmode == GPGME_SIG_MODE_CLEAR)
        return ".asc";
    if (!use_armor && sigmode == GPGME_SIG_MODE_DETACH)
        return ".sig";
    return ".gpg";
} /* file_get_extension */


int
fm_build( listview_ctrl_t *lv, HWND ctrl )
{
    int i, rc = 0;
    listview_ctrl_t c;
    struct listview_column_s col[] = 
    {
	{0,  80, (char *)_("Status") },
	{1, 256, (char *)_("Name") },
	{2, 128, (char *)_("Operation") },
	{0,   0, NULL }	
    };
	
    rc = listview_new( &c );
    if( rc )
	BUG( NULL );
    c->ctrl = ctrl;
    for ( i = 0; col[i].width; i++ )
	listview_add_column( c, &col[i] );
    listview_set_ext_style( c );
    if( lv )
	*lv = c;
    return 0;
} /* fm_build */


void
fm_delete( listview_ctrl_t lv )
{
    if( lv ) {
	listview_release( lv );	
    }
} /* fm_delete */


int
fm_state_new (fm_state_t * ctx)
{
    gpgme_error_t rc;
    fm_state_s * c;

    c = new fm_state_s;
    if (!c)
	BUG (0);
    memset (c, 0, sizeof * c);
    rc = gpgme_new (&c->ctx);
    if (!rc)	
	rc = gpgme_recipients_new (&c->recp);
    if (rc)
	BUG (0);
    gpgme_set_comment (c->ctx, "Generated by WinPT "PGM_VERSION);
    *ctx = c;
    return 0;
} /* fm_state_new */


void
fm_state_release (fm_state_t c)
{
    if (c) 
    {
	if (c->recp)
	{
	    gpgme_recipients_release (c->recp);
	    c->recp = NULL;
	}
	if (c->ctx) 
	{
	    gpgme_release (c->ctx);
	    c->ctx = NULL;
	}
	free_if_alloc (c->opaque);
	free_if_alloc (c->output);
	delete c; c = NULL;
    }
} /* fm_state_release */


static int
fm_check_for_entry( listview_ctrl_t lv, const char *file )
{
    char name[512];
    int i;

    memset (name, 0, sizeof (name));
    for( i = 0; i < listview_count_items( lv, 0 ); i++ ) 
    {
	listview_get_item_text( lv, i, 1, name, sizeof (name) - 1 );
	if( !strcmp( name, file ) )
	    return 1; /* found */	
    }

    return 0;
} /* fm_check_for_entry */


static int
fm_set_ftype (listview_ctrl_t lv, const char * name)
{
    const char *type;
    int rc;

    rc = fm_check_for_entry (lv, name);
    if (rc)
	return 0;
    type = fm_get_file_type (name);
    if (!type || !strcmp (type, "UNKNOWN"))
	type = gnupg_check_file_ext (name);	
    rc = listview_add_item (lv, " ");
    if (rc)
	return -1;
    listview_add_sub_item (lv, 0, 0, type);
    listview_add_sub_item (lv, 0, 1, name);
    return 0;
}


static int
fm_add_dir_files (listview_ctrl_t lv, char *path)
{
    struct _finddata_t fd;
    char * p;
    long hd;

    strcat (path, "\\*");
    hd = _findfirst (path, &fd);
    do {
	p = new char [(strlen (path) + strlen (fd.name))+1];
	if (!p)
	    BUG (0);
	memcpy (p, path, strlen (path)-1);
	p[strlen (path)-1] = 0;
	strcat (p, fd.name);
	if (!is_directory (p))
	    fm_set_ftype (lv, p);
	free_if_alloc (p);
	
    } while (_findnext (hd, &fd) == 0);
    _findclose (hd);
    return 0;
}


int
fm_add_dropped_files (listview_ctrl_t lv, HDROP dd_files)
{
    char name[384+4];
    int nfiles, rc, i;
    
    memset( name, 0, sizeof (name) );
    nfiles = DragQueryFile( dd_files, 0xFFFFFFFF, NULL, 0 );
    for (i = 0;  i < nfiles; i++) {
	DragQueryFile (dd_files, i, name, sizeof (name) -1);
	if (is_directory (name))
	    rc = fm_add_dir_files (lv, name);
	else
	    rc = fm_set_ftype (lv, name);
	if (rc == -1)
	    break;
	
    }
    return rc;
} /* fm_add_dropped_files */


int
fm_add_opened_files (listview_ctrl_t lv, HWND dlg)
{
    OPENFILENAME open;
    const char *type;	
    char file[1024] = "";
    int rc;
    
    memset( &open, 0, sizeof (open) );
    open.lStructSize = sizeof (OPENFILENAME);
    open.hInstance = glob_hinst;
    open.lpstrTitle = _("File Open");
    open.lpstrFilter = _("All Files (*.*)\0*.*");
    open.hwndOwner = dlg;
    open.lpstrFile = file;
    open.nMaxFile = sizeof (file) - 1;
    open.Flags = 0;
    
    if (GetOpenFileName (&open)) {
        type = fm_get_file_type (open.lpstrFile);
        if (!type)
	    return WPTERR_FILE_OPEN;
	if (!strcmp (type, "UNKNOWN"))
            type = gnupg_check_file_ext (open.lpstrFile);
        rc = listview_add_item (lv, "");
        if( !rc ) {
            listview_add_sub_item (lv, 0, 0, type);
            listview_add_sub_item (lv, 0, 1, open.lpstrFile);
        }
    }
    
    return rc;
} /* fm_add_opened_files */


static const char *
fm_check_armor_type (const char * fname)
{
    FILE * fp;
    char header[768], * p;
    
    fp = fopen (fname, "rb");
    if (!fp)
	return "UNKNOWN";
    p = fgets (header, sizeof (header) - 1, fp);
    fclose (fp);
    if (!p)
	return "UNKNOWN";

    if( strncmp( header, "-----", 5 ) )
	goto leave;
    if( strstr( header, "BEGIN PGP PUBLIC KEY" ) )
	return "PUBKEY";
    else if( strstr( header, "BEGIN PGP PRIVATE KEY" ) )
	return "SECKEY";
    else if( strstr( header, "BEGIN PGP SECRET KEY" ) )
	return "SECKEY";
    else if( strstr( header, "BEGIN PGP MESSAGE" ) )
	return "ENCRYPTED";
    else if( strstr( header, "BEGIN PGP SIGNED MESSAGE" ) )
	return "SIGNED-CLEAR";
    else if( strstr(header, "BEGIN PGP SIGNATURE" ) )
	return "SIGNED-DETACH";

leave:
    return "UNKNOWN";
} /* fm_check_armor_type */


int
fm_assume_onepass_sig (const char * fname)
{
    gpgme_data_t dat;
    armor_filter_context_t afx;
    gpg_iobuf_t fp;
    PACKET * pkt = (PACKET *)calloc (1, sizeof *pkt);
    int check = 0;

    if (!fname) 
    {
	gpgme_data_new_from_clipboard (&dat);
	gpgme_data_release_and_set_file (dat, "gpgme.tmp");

	fp = gpg_iobuf_open ("gpgme.tmp");
	if (!fp)
	    return 0;
	gpg_iobuf_ioctl (fp, 3, 1, NULL);
	if (gpg_use_armor_filter(fp)) 
	{
	    memset (&afx, 0, sizeof (afx));
	    gpg_iobuf_push_filter (fp, gpg_armor_filter, &afx);
	}
	gpg_init_packet (pkt);
	if (!gpg_parse_packet (fp, pkt)
	    && pkt->pkttype == PKT_COMPRESSED)
	    check = 1;	
	gpg_free_packet (pkt);
	safe_free (pkt);
	gpg_iobuf_close (fp);
	unlink ("gpgme.tmp");
    }
    /* XXX: implement it for real files */
    return check;
}


static int
is_floppy_disc (const char * fname)
{
    char drv[32] = {0};
    int i=0;

    if (!strstr (fname, ":\\"))
	return 0;

    while (fname && *fname && *fname != '\\')
	drv[i++] = *fname++;
    drv[i++] = '\\';
    drv[i++] = '\0';
    i = GetDriveType (drv);
    if (i == DRIVE_REMOVABLE)
	return 1;
    return 0;
}


const char *
fm_get_file_type (const char * fname)
{        
    gpg_iobuf_t inp;
    armor_filter_context_t afx;
    PACKET * pkt = (PACKET *)calloc (1, sizeof *pkt);
    int i = 0, rc = 0;
    const char * s = NULL;

    if (!fname) {
	safe_free (pkt);
	return NULL;
    }

    if (is_floppy_disc (fname))	
	return fm_check_armor_type (fname);

    inp = gpg_iobuf_open (fname);
    if (!inp) {
	const char *s = winpt_strerror (WPTERR_FILE_OPEN);
	log_box( _("File Manager"), MB_ERR, "\"%s\": %s", fname, s );
	safe_free( pkt );
	return NULL;
    }
    gpg_iobuf_ioctl (inp, 3, 1, NULL); /* disable cache */
    if (gpg_iobuf_get_filelength (inp) > 32000000 /* 32MB */
	&& !is_openpgp_ext (fname)) {
	gpg_iobuf_close (inp);
	return "UNKNOWN";
    }

    if (gpg_use_armor_filter(inp)) {
	memset (&afx, 0, sizeof (afx));
	gpg_iobuf_push_filter (inp, gpg_armor_filter, &afx);
    }
    
    gpg_init_packet (pkt);
    while (!(rc = gpg_parse_packet (inp, pkt))) {
	switch (pkt->pkttype) {
	case PKT_PUBKEY_ENC:  s = "ENCRYPTED";rc = -2; break;
	case PKT_SYMKEY_ENC:
	case PKT_ENCRYPTED:   s = "SYMKEYENC";rc = -2; break;
	case PKT_SIGNATURE:
	case PKT_ONEPASS_SIG: s = "SIGNED";   rc = -2; break;
	case PKT_PUBLIC_KEY:  s = "PUBKEY";   rc = -2; break;
	case PKT_SECRET_KEY:  s = "SECKEY";   rc = -2; break;
	}
	gpg_free_packet (pkt);
	gpg_init_packet (pkt);
	if (rc == -2)
	    break; /* found */
    }
    safe_free (pkt);
    gpg_iobuf_close (inp);
    if (!s)
	s = fm_check_armor_type (fname);
    if (!s)
	s = "UNKNOWN";
    if (!strcmp( s, "SIGNED")
	&& strcmp (fm_check_armor_type (fname), "SIGNED-CLEAR "))
	s = "SIGNED-DETACH";
    return s;
} /* fm_get_file_type */


int
fm_get_current_pos (listview_ctrl_t lv)
{
    int i = 0, items;

    items = listview_count_items (lv, 0);
    if (!items)
	return -1;
    else if (items == 1)
    {
	listview_select_one (lv, 0);
	return 0;
    }
    else if (items > 1)
    {
	i = listview_get_curr_pos (lv);
	if (i == -1)
	{
	    msg_box (lv->ctrl, _("Please select a file."), _("File Manager"), MB_ERR);
	    return -1;
	}
	return i;
    }

    return -1;
} /* fm_get_current_pos */


static int
fm_check_detached_sig( listview_ctrl_t lv, int pos )
{
    char type[128];

    listview_get_item_text( lv, pos, 0, type, 127 );
    return !strcmp( type, "SIGNED-DETACH" )? 1 : 0;
} /* fm_check_detached_sig */


int
fm_check_file_type (listview_ctrl_t lv, int pos, int fm_cmd)
{
    char status[128];
    int rc = 0;
    
    listview_get_item_text (lv, pos, 0, status, sizeof (status) - 1);
    
    switch (fm_cmd) {
    case FM_ENCRYPT:
    case FM_ENCRYPT_DIR:
    case FM_SIGNENCRYPT:   
	if (strcmp (status, "ENCRYPTED") 
	    && strcmp (status, "SYMKEYENC"))
	    rc = 1;
	break;
	
    case FM_DECRYPT: 
	if (!strcmp (status, "DATA")
	    || !strcmp (status, "ENCRYPTED")
	    || !strcmp (status, "SYMKEYENC") 
	    || !strcmp (status, "ARMORED"))
	    rc = 1;
	break;
	
    case FM_SIGN: 
	if( strncmp( status, "SIGNED", 6 ) ) 
	    rc = 1;
	break;
	
    case FM_VERIFY: 
	if( !strncmp( status, "SIGNED", 6 ) 
             || !strcmp( status, "COMPRESSED" ) )
	    rc = 1;
	break;
        
    case FM_SYMENC:
	if( strcmp( status, "SYMKEYENC" ) )
	    rc = 1;
	break;
	
    case FM_IMPORT:
	if( !strcmp( status, "PUBKEY" )
	    || !strcmp( status, "SECKEY" ) )
	    rc = 1;
	break;
        
    case FM_WIPE:
    case FM_LIST:
	rc = 1; 
	break;	
    }
    
    return rc;
} /* fm_check_file_type */


static void
fm_set_status (listview_ctrl_t lv, int pos, int fm_cmd, int success,
	       const char * output)
{
    char status[128], operat[128];
    int update = 1;
    const char *s;

    if ( fm_cmd == FM_LIST )
	return;
    success ? s = "SUCCESS" : s = "FAILED";
    strcpy( operat, s );

    switch (fm_cmd) {
    case FM_ENCRYPT: 
    case FM_ENCRYPT_DIR:
    case FM_SIGNENCRYPT: strcpy( status, "ENCRYPTED" ); break;
    case FM_DECRYPT:     strcpy( status, "UNKNOWN" );   break;
    case FM_SIGN:        strcpy( status, "SIGNED" );    break;
    case FM_VERIFY:      update = 0;                    break;
    case FM_SYMENC:      strcpy( status, "SYMKEYENC" ); break;
    case FM_IMPORT:	 update = 0;                    break;
    case FM_WIPE:        strcpy( status, "WIPED" );     break;
    default:             strcpy( status, "UNKNOWN");    break;
    }

    if (success) {
	if (update) {
	    listview_add_sub_item (lv, pos, 0, status);
	    listview_add_sub_item (lv, pos, 1, output);
	}
    }
    listview_add_sub_item( lv, pos, 2, operat );
} /* fm_set_status */


int
fm_clearsign_8bit (listview_ctrl_t lv, fm_state_s *ctx)
{
    FILE *f;
    byte buf[32];
    char name[256];
    int i, n, cnt=0;

    if (ctx->sigmode != GPGME_SIG_MODE_CLEAR)
	return 0;
    listview_get_item_text (lv, -1, 1, name, sizeof (name)-1);
    if (stristr (name, ".TXT"))
	return 0;
    f = fopen (name, "rb");
    if (!f)
	return -1; /* should never happen */
    n = fread (buf, 1, 32, f);
    for (i = 0; i < n; i++) {
	if (buf[i] == 0x00 || buf[i] > 170)
	    cnt++;
    }
    fclose (f);
    if (!cnt)
	return 0;
    n = -1;
    i = log_box (_("File Manager"), MB_WARN|MB_YESNO, 
    		 _("\"%s\" does not seems to be a text file.\n"
		   "Do you really want to clearsign it?"), name);
    if (i == IDYES)
	n = 0;
    return n;
}


int
fm_parse_files (listview_ctrl_t lv, HWND dlg, int cmd)
{
    struct secdel_confirm_s confirm = {0};
    struct progress_filter_s pfx, pfx2;
    fm_state_s * ctx;
    int fm_cmd, sig_detached = 0;
    int rc = 0, i, n, ndel = 0;
    char fname[512], status[128];
    
    switch (cmd) {
    case ID_FILEMISC_ENCRYPT: fm_cmd = FM_ENCRYPT; break;
    case ID_FILEMISC_DECRYPT: fm_cmd = FM_DECRYPT; break;
    case ID_FILEMISC_SYMENC:  fm_cmd = FM_SYMENC;  break;
    case ID_FILEMISC_SIGN:    fm_cmd = FM_SIGN;    break;    
    case ID_FILEMISC_VERIFY:  fm_cmd = FM_VERIFY;  break;
    case ID_FILEMISC_IMPORT:  fm_cmd = FM_IMPORT;  break;
    case ID_FILEMISC_WIPE:    fm_cmd = FM_WIPE;    break;
    case ID_FILEMISC_LIST:    fm_cmd = FM_LIST;    break;
    case ID_FILEMISC_SIGNENC: fm_cmd = FM_SIGNENCRYPT; break;
    default: return 1; /* unknown command */
    }
    
    if (fm_get_current_pos (lv) == -1)
        return WPTERR_GENERAL;    
    rc = fm_state_new (&ctx);
    if (rc)
	BUG (0);
    ctx->dlg = dlg;
    
    memset (&pfx, 0, sizeof (pfx));
    pfx.hwnd = dlg;
    gpgme_set_progress_cb (ctx->ctx, progress_callback, &pfx);
    
    /* Commands we need before we can perform the main command */
    switch (fm_cmd) {
    case FM_ENCRYPT:
    case FM_SIGNENCRYPT:
	if (fm_cmd == FM_SIGNENCRYPT)
	    ctx->req_signer = 1;
        DialogBoxParam (glob_hinst, (LPCTSTR)IDD_WINPT_FILE_ENCRYPT, ctx->dlg,
                        file_encrypt_dlg_proc, (LPARAM)ctx);
        if (ctx->cancel == 1) {
            rc = WPTERR_GENERAL; 
	    goto leave;
        }
        break;
        
    case FM_SIGN:
        DialogBoxParam (glob_hinst, (LPCSTR)IDD_WINPT_FILE_SIGN, dlg,
                        file_sign_dlg_proc, (LPARAM) ctx);
        if (ctx->cancel == 1 || fm_clearsign_8bit (lv, ctx)) {
            rc = WPTERR_GENERAL; 
	    goto leave;
        }
        break;

    case FM_WIPE:
	memset (&pfx2, 0, sizeof (pfx2));
	secure_unlink_set_cb (progress_callback, &pfx2);
	break;
    }
    
    for( i = 0, n = 0;  i < listview_count_items( lv, 0 ); i++ ) {
        if( !listview_get_item_state( lv, i ) )
            continue;
        listview_get_item_text( lv, i, 0, status, sizeof (status) -1 );
        if (!strcmp( status, "ENCRYPTED" ) && fm_cmd == FM_DECRYPT)
            n++;
        if (!strcmp( status, "UNKNOWN" ) && fm_cmd == FM_SIGN)
            n++;
	if (fm_cmd == FM_WIPE) {
	    if (!confirm.rset)
		gpgme_recipients_new (&confirm.rset);
	    listview_get_item_text (lv, i, 1, fname, sizeof (fname)-1);
	    gpgme_recipients_add_name (confirm.rset, fname);
	    ndel++;
	}
    }
    
    if (n > 1 && fm_cmd != FM_SYMENC)
        ctx->cache_cb = 1;

    if (fm_cmd == FM_WIPE && ndel > 0) {
	DialogBoxParam (glob_hinst, (LPCTSTR)IDD_WINPT_FILES_SECDEL, ctx->dlg,
			    file_secdel_confirm_dlg_proc, (LPARAM)&confirm);
	if (!confirm.yes)
	    goto leave;
    }
    
    for( i = 0; i < listview_count_items( lv, 0 ); i++ ) {
        if( !listview_get_item_state( lv, i ) )
            continue;
        listview_get_item_text( lv, i, 1, fname, sizeof (fname) - 1 );
        if( file_exist_check( fname ) && !is_directory( fname ) ) {
	    log_box( _("File Manager"), MB_ERR, _("\"%s\" does not exist"), fname );
            continue;
        }
	if( is_directory( fname ) )
	    fm_cmd = FM_ENCRYPT_DIR;        
        if( !fm_check_file_type( lv, i, fm_cmd ) )
            continue;
        sig_detached = fm_check_detached_sig( lv, i );
        switch( fm_cmd ) {
        case FM_LIST:        rc = fm_list( fname, dlg );       break;
        case FM_WIPE:        rc = fm_wipe( fname );            break;
        case FM_ENCRYPT:     rc = fm_encrypt( ctx, fname, 0 ); break;
	case FM_ENCRYPT_DIR: rc = fm_encrypt_directory( ctx, fname ); break;
        case FM_SIGNENCRYPT: rc = fm_encrypt( ctx, fname, 1 ); break;
        case FM_DECRYPT:     rc = fm_decrypt( ctx, fname );    break;
        case FM_SIGN:        rc = fm_sign( ctx, fname );       break;
	case FM_SYMENC:      rc = fm_sym_encrypt( ctx, fname );break;
        case FM_VERIFY:      rc = fm_verify( ctx, sig_detached, fname );break;
        case FM_IMPORT:
	    free_if_alloc (ctx->opaque);
	    ctx->opaque = m_strdup (fname);
	    if (!ctx->opaque)
		BUG (0);
            DialogBoxParam( glob_hinst, (LPCSTR)IDD_WINPT_IMPORT, dlg,
                           file_import_dlg_proc, (LPARAM)ctx );
            if (ctx->cancel == 1)
                continue;
            rc = fm_import (ctx, fname);
            break;
        }
        fm_set_status (lv, i, fm_cmd, !rc, ctx->output);
	free_if_alloc (ctx->output);
	progress_cleanup (&pfx);
    }
    if (fm_cmd == FM_WIPE) {
	secure_unlink_set_cb (NULL, NULL);
	progress_cleanup (&pfx2);
    }
    if (ctx->cache_cb) {
        memset (ctx->pass_cb.pwd, 0, sizeof (ctx->pass_cb));
        ctx->cache_cb = 0; /* make sure it's only used for this session! */
    }
    
    /* remove wipe files from the list */
    n = listview_count_items (lv, 0);
    while (n--) {
        char status[128];
        listview_get_item_text (lv, n, 0, status, sizeof (status) - 1);
        if( !strcmp (status, "WIPED"))
            listview_del_item (lv, n);
    }
    
leave:
    if (!rc)
        fm_state_release (ctx);
    if (confirm.rset)
	gpgme_recipients_release (confirm.rset);
    progress_cleanup (&pfx);
    return rc;
} /* fm_parse_files */


int
fm_wipe (const char * name)
{
    int rc;

    SetCursor (LoadCursor (NULL, IDC_WAIT));
    remove_crit_file_attrs (name, 1);
    rc = secure_unlink (name, reg_prefs.wipe_mode);
    SetCursor (LoadCursor (NULL, IDC_ARROW));
    return rc;
} /* fm_wipe */


int
fm_list( const char * name, HWND dlg )
{
    dialog_box_param( glob_hinst, (LPCTSTR)IDD_WINPT_FILE_STAT, dlg,
                      file_stat_dlg_proc, (LPARAM)name, _("File Status"),
                      IDS_WINPT_FILE_STAT );
    return 0;
} /* fm_list */


static int
ask_filename (fm_state_t c, const char * msg, char ** dst)
{
    const char * s;

    s = get_filename_dlg (c->dlg, FILE_SAVE, msg, NULL, NULL);
    if (!s)
	return WPTERR_GENERAL;

    free_if_alloc (*dst);
    free_if_alloc (c->output);
    c->output = m_strdup (s);
    if (!c->output)
	BUG (0);
    *dst = fm_quote_file (s);
    return 0;
}


int
fm_encrypt (fm_state_t c, const char * name, int sign)
{
    gpgme_error_t err;
    gpgme_key_t key = NULL;
    gpgme_ctx_t ctx = c->ctx;
    char * src = NULL, * dst = NULL;
    char * keyid = NULL, ext[5];    
    int no_compr = 0;
    int rc = 0;
    
    src = fm_quote_file (name);
    c->output = new char[strlen (name) + 5 + 1];
    if (!c->output)
	BUG (0);
    strcpy (ext, file_get_extension (ctx, c->sigmode));
    strcpy (c->output, name );
    strcat (c->output, ext );    
    dst = fm_quote_file (c->output);
    
    if (!overwrite_file (c->output)) 
    {
	rc = ask_filename (c, _("Enter filename for encrypted file"), &dst);
	if (rc)
	    goto leave;
    }
    
    no_compr = is_multi_media (name);
    gpgme_control (ctx, GPGME_CTRL_NO_COMPR, no_compr);

    if (sign) {
	if (gpgme_signers_enum (ctx, 0) == NULL) {
	    keyid = get_gnupg_default_key ();
	    if (!keyid) {
		msg_box (c->dlg, _("Could not get default secret key."), 
			 _("Signing"), MB_ERR);
		rc = WPTERR_GENERAL;
		goto leave;
	    }
	    if (get_seckey (keyid, &key))
		BUG (0);
	    gpgme_signers_add (ctx, key);
	}
	else {
	    const char * s;
	    s = (char *)gpgme_key_get_string_attr (gpgme_signers_enum (ctx, 0), 
						    GPGME_ATTR_KEYID, NULL, 0);
	    keyid = m_strdup (s);
	}
	if (!c->init_cb || !c->cache_cb) {
	    set_gpg_passphrase_cb (c->ctx, &c->pass_cb, GPG_CMD_SIGN, 
				   c->dlg, _("Signing"));
	    c->init_cb = 1;
	}
	err = gpgme_op_file_sign_encrypt (ctx, c->recp, src, dst);
	if (!c->cache_cb)
	    memset (c->pass_cb.pwd, 0, sizeof (c->pass_cb.pwd));
	if (c->pass_cb.cancel) {
	    rc = WPTERR_GENERAL;
	    goto leave;
	}
	if (err) {
	    msg_box (c->dlg, gpgme_strerror (err), _("Sign"), MB_ERR);
	    if (err == GPGME_Bad_Passphrase)
		agent_del_cache (keyid);
	    rc = WPTERR_GENERAL; 
	    goto leave;
	}
	gpgme_key_release (key);
    }
    else {
	err = gpgme_op_file_encrypt (ctx, c->recp, src, dst);
	if (err) {
	    msg_box (c->dlg, gpgme_strerror (err), _("Encrypt"), MB_ERR);
	    rc = WPTERR_GENERAL; 
	    goto leave;
	}
    }
    if (c->wipe)
	secure_unlink (name, WIPE_MODE_SIMPLE);
    
leave:
    free_if_alloc (keyid);
    free_if_alloc (dst);
    free_if_alloc (src);
    return rc;
} /* fm_encrypt */


int
fm_sym_encrypt (fm_state_t c, const char * name)
{
    int rc = 0, cancel = 0;
    char * src = NULL, * dst = NULL;
    char ext[5], * pass;
    gpgme_ctx_t ctx = c->ctx;
    gpgme_error_t err;    
    
    pass = request_passphrase2 (_("Symmetric"), &cancel);
    if (cancel)
	return 0;
    
    gpgme_control (ctx, GPGME_CTRL_CIPHER, -1);
    src = fm_quote_file (name);
    c->output = new char[strlen (name) + 5 + 1];
    if (!c->output)
	BUG (0);
    strcpy (ext, file_get_extension (ctx, c->sigmode));
    strcpy (c->output, name);
    strcat (c->output, ext);    
    dst = fm_quote_file (c->output);

    if (overwrite_file (c->output) == 0) {
	rc = WPTERR_GENERAL;
	goto leave;	
    }
    gpgme_set_passphrase (ctx, pass);
    err = gpgme_op_file_encrypt (ctx, NULL, src, dst);
    if (err) {
	msg_box (c->dlg, gpgme_strerror (err), _("Symmetric"), MB_ERR);
	rc = WPTERR_GENERAL;
	goto leave;
    }
    if (file_exist_check (c->output)) {
	msg_box (c->dlg, _("Encryption failed."), _("Symmetric"), MB_ERR);
	rc = WPTERR_GENERAL;
    }
    
leave:
    free_if_alloc (src);
    free_if_alloc (dst);
    sfree_if_alloc (pass);
    return rc;
} /* fm_sym_encrypt */


static gpgme_error_t
fm_list_keys( const char * name, gpgme_recipients_t *r_keys )
{
    return gpgme_op_list_keys( NULL, name, r_keys );
} /* fm_list_keys */


int
fm_decrypt (fm_state_t c, const char * name)
{
    gpgme_error_t err;
    gpgme_ctx_t ctx = c->ctx;
    gpgme_recipients_t keys = NULL;
    gpgme_sig_t sig = NULL;
    gpgme_op_flags_t flags;
    char * src = NULL, * dst = NULL, keyid[17];
    int is_signed = 0, sigok = 0;
    int rc = 0;
    
    if (!c->init_cb || !c->cache_cb) {
	set_gpg_passphrase_cb (c->ctx, &c->pass_cb, GPG_CMD_DECRYPT, 
			       c->dlg, _("Decryption"));
	c->init_cb = 1;	
    }    
    
    src = fm_quote_file (name);
    c->output = m_strdup (name);
    if (!c->output)
	BUG (0);
    if (is_openpgp_ext (c->output))
	c->output[strlen (c->output)-4] = '\0';
    else {
	const char *s = get_filename_dlg (c->dlg, FILE_SAVE, _("Choose Filename for Output"), 
					  NULL, NULL);
	if( s ) {
	    free_if_alloc( c->output );
	    c->output = m_strdup( s );
	    if( !c->output )
		BUG( NULL );
	}
    }
    dst = fm_quote_file( c->output );

    err = fm_list_keys( src, &keys );
    if( err )
	goto leave;
    c->pass_cb.enc_to = keys;

    if (overwrite_file (c->output) == 0) {
	rc = ask_filename (c, _("Please enter filename for plaintext file"), &dst);
	if (rc)
	    goto leave;
    }    
    remove_crit_file_attrs( c->output, 0 );
    err = gpgme_op_file_decrypt( ctx, src, dst );
    if( !c->cache_cb )
	memset( c->pass_cb.pwd, 0, sizeof (c->pass_cb.pwd) );
    if( c->pass_cb.cancel ) {
	rc = WPTERR_GENERAL;
	goto leave;
    }
    gpgme_decrypt_get_status( ctx, keyid, &flags );
    if( err == GPGME_No_Seckey && (flags & GPGME_OPFLAG_NOSECKEY) ) {
	char * p = get_key_userid( keyid+8 );
	int pkalgo = algo_from_list( keys, keyid );
	log_box( _("Decryption"), MB_ERR, 
		 _("Encrypted with %s key, ID %s.%s\n"
		   "Decryption failed: secret key not available."), 
		   gpgme_key_expand_attr( GPGME_ATTR_ALGO, pkalgo ),
		   keyid+8, p );
	rc = WPTERR_GENERAL;
	free_if_alloc( p );
	goto leave;
    }
    else if( err ) {
	msg_box( c->dlg, gpgme_strerror( err ), _("Decrypt"), MB_ERR );
	rc = WPTERR_GENERAL;
	goto leave;
    }
    if( file_exist_check( c->output ) ) {
	msg_box( c->dlg, _("Decryption failed"), _("Decrypt"), MB_ERR );
	rc = WPTERR_GENERAL;
    }
    
    gpgme_decrypt_get_sig_ctx( ctx, &sig );
    
    sigok = gpgme_sig_get_ulong_attr( sig, 0, GPGME_ATTR_VALIDITY ) == GPGME_SIG_STAT_GOOD;
    if( sig ) {
	const char *id, *s = sigok? _("Good signature") : _("BAD signature");
	int type = sigok? MB_OK: MB_ICONWARNING|MB_OK;
	gpgme_key_t key;
	const char * keyid = gpgme_sig_get_string_attr( sig, GPGME_ATTR_KEYID );
	if( !keyid )
	    keyid = "DEADBEEFDEADBEEF";
	if( get_pubkey( keyid+8, &key ) )
	    log_box( _("Verify"), type, _("%s using keyID 0x%s"), s, keyid+8 );
	else {
	    id = gpgme_sig_get_string_attr( sig, GPGME_ATTR_USERID );
	    log_box( _("Verify"), type, "%s using keyID 0x%08X from %s",
				s, keyid, id? id : _("Invalid User ID") );
	}
    }
    
leave:
    free_if_alloc( dst );
    free_if_alloc( src );
    gpgme_sig_release( sig );
    gpgme_recipients_release( keys );
    return rc;
} /* fm_decrypt */


int
fm_sign (fm_state_t c, const char * name)
{	
    int rc = 0;
    gpgme_ctx_t ctx = c->ctx;
    gpgme_error_t err;
    char *src = NULL, *dst = NULL;
    char ext[5];

    if( !c->init_cb || !c->cache_cb ) {
	set_gpg_passphrase_cb( c->ctx, &c->pass_cb, GPG_CMD_SIGN, c->dlg, _("Signing") );
	c->init_cb = 1;
    }
    
    src = fm_quote_file( name );
    free_if_alloc( c->output );
    c->output = new char[strlen( name ) + 5 + 1];
    if( !c->output )
	BUG( NULL );
    strcpy( ext, file_get_extension( ctx, c->sigmode ) );
    strcpy( c->output, name );
    strcat( c->output, ext );
    dst = fm_quote_file( c->output );
    
    if (!overwrite_file (c->output)) {
	rc = ask_filename (c, _("Enter filename for signed file"), &dst);
	if (rc)
	    goto leave;
    }
    remove_crit_file_attrs( c->output, 0 );
    err = gpgme_op_file_sign( ctx, c->sigmode, src, dst );
    if( !c->cache_cb )
	memset( c->pass_cb.pwd, 0, sizeof (c->pass_cb.pwd) );
    if( c->pass_cb.cancel ) {
	rc = WPTERR_GENERAL;
	goto leave;
    }
    if( err ) {
	msg_box( c->dlg, gpgme_strerror( err ), _("Sign"), MB_ERR );
	rc = WPTERR_GENERAL;
	goto leave;	
    }

leave:
    free_if_alloc( src );
    free_if_alloc( dst );
    return rc;
} /* fm_sign */


static int
fm_add_sig_stat( siglog_context_t log )
{
    gpgme_key_t key;    
    const char *uid, *keyid, * kid;
    int not_found = 0;

    kid = gpgme_sig_get_string_attr( log->sig, GPGME_ATTR_KEYID );
    if( !kid )
	kid = "DEADBEEFDEADBEEF";
    if( strlen( kid ) == 16 )
	keyid = kid + 8;
    else if( strlen( kid ) == 32 )
	keyid = kid;
    else if( strlen( kid ) == 40 )
	keyid = kid + 24;
    if( get_pubkey( keyid, &key ) )
	not_found = 1;
    log->use_uid = 0;
    if( !not_found ) {
	uid = gpgme_key_get_string_attr( key, GPGME_ATTR_USERID, NULL, 0 );
	log->user_id = uid;
	log->use_uid = 1;	
    }
    file_verify_add_state( log );

    return 0;
} /* fm_add_sig_stat */


static int
verify_pasted (listview_ctrl_t lv, fm_state_t ctx, const char * dat,
	       int i, HWND dlg)
{
    FILE * fp;
    char stat[32];
    char file[256], * fname = NULL;
    int del_end=0;

    listview_get_item_text (lv, i, 0, stat, sizeof (stat)-1);
    listview_get_item_text (lv, i, 1, file, sizeof (file)-1);
    if (strcmp (stat, "UNKNOWN"))
	return 0;
    fname = make_filename (NULL, file, "asc");
    if (file_exist_check (fname) != 0) {
	fp = fopen (fname, "wb");
	if (fp == NULL) {
	    log_box (_("File Manager"), MB_ERR, "could not create '%s'", fname);
	    free_if_alloc (fname);
	    return WPTERR_GENERAL;	
	}	
	fwrite (dat, 1, strlen (dat), fp);
	fclose (fp);
	del_end = 1;
    }
    fm_verify (ctx, 1, fname);
    if (del_end)
	unlink (fname);
    free_if_alloc (fname);
    return 0;
}


int
fm_verify_pasted_detsig (listview_ctrl_t lv, HWND dlg)
{
    fm_state_t ctx = NULL;
    char * dat=NULL;
    int i, fnd = 0;

    dat = get_clip_text (NULL);
    if (!dat || !strstr (dat, "BEGIN PGP SIGNATURE")) {
	msg_box (dlg, _("Could not find detached signature in the clipboard."),
		 _("File Manager"), MB_ERR);
	free_if_alloc (dat);
	return WPTERR_GENERAL;
    }
    /* XXX find a way to filter out bad signatures or just ignore all in 
           this case */
    fm_state_new (&ctx);
    if ((i=listview_get_curr_pos (lv)) != -1) {
	verify_pasted (lv, ctx, dat, i, dlg);
	fnd = 1;
    }
    else {
	for (i=0; i < listview_count_items (lv, 0); i++) {
	    verify_pasted (lv, ctx, dat, i, dlg);
	    fnd = 1;
	}
    }
    if (!fnd)
	msg_box (dlg, _("No files to check."), _("File Manager"), MB_INFO);
    free_if_alloc (dat);
    fm_state_release (ctx);
    return 0;
}


int
fm_verify( fm_state_t c, int detached, const char *name )
{
    gpgme_ctx_t ctx = c->ctx;
    gpgme_error_t err;
    gpgme_sig_t sig;
    struct siglog_context_s log;    
    char * src = NULL;
    int rc = 0;
    size_t i;
	
    if( detached ) {
	const char *file = NULL;
	if( strstr( name, ".sig" ) || strstr( name, ".asc" ) ) {
	    char fname[512];
	    _snprintf( fname, sizeof (fname) - 1, "%s", name );
	    fname[strlen( fname ) - 4] = '\0';
	    if( file_exist_check( fname ) == 0 )
		file = fname;
	}	
	if( !file )
	    file = get_filename_dlg( c->dlg, FILE_OPEN, _("Select Data File"), NULL, NULL );
	if( file ) {
	    free_if_alloc( c->output );
	    c->output = m_strdup( file );
	}
	else {
	    msg_box( c->dlg, _("Invalid file name. Exit"), _("Verify"), MB_ERR );
	    return WPTERR_GENERAL;
	}
	c->sigmode = GPGME_SIG_MODE_DETACH;
    }
    else {
	if( strstr( name, ".asc" ) )    
	    c->sigmode = GPGME_SIG_MODE_CLEAR;
	else
	    c->sigmode = GPGME_SIG_MODE_NORMAL;
    }

    memset( &log, 0, sizeof (log) );
    strcpy( log.file, name );
    file_verify_create_dlg();
    src = fm_quote_file( name );
    
    err = gpgme_op_file_verify( ctx, c->sigmode, &sig, src, c->output );
    if( err == GPGME_Bad_Signature ) {
	log.sig = sig;
	fm_add_sig_stat( &log );
	rc = WPTERR_GENERAL; 
	goto leave;
    }
    if( err ) {
	msg_box( c->dlg, gpgme_strerror( err ), _("Verify"), MB_ERR );
	rc = WPTERR_GENERAL; 
	goto leave;
    }
    for( i = 0; i < gpgme_sig_get_ulong_attr( sig, 0, GPGME_ATTR_LEVEL ); i++ ) {
	gpgme_sig_t _sig;
	_sig = (gpgme_sig_t)gpgme_sig_get_ulong_attr( sig, i, GPGME_ATTR_OPAQUE );
	log.sig = _sig;
	fm_add_sig_stat( &log );
    }
    if( !c->output )
	c->output = m_strdup( name ); /* for later use */

leave:
    free_if_alloc( src );
    return rc;
} /* fm_verify */


int
fm_import( fm_state_t c, const char *name )
{
    gpgme_ctx_t ctx = c->ctx;
    gpgme_error_t err;
    char *src = NULL;
    int import_res[14] = {0};
    int rc = 0;

    free_if_alloc( c->output );
    c->output = m_strdup( name );
    if( !c->output )
	BUG( NULL );
    src = fm_quote_file( name );
    err = gpgme_op_file_import( ctx, NULL, src );
    if( err ) {
	msg_box( c->dlg, gpgme_strerror( err ), _("Import"), MB_ERR );
	rc = WPTERR_GENERAL; 
	goto leave;
    }
    gpgme_get_import_status( ctx, import_res, NULL );
    print_import_status( import_res, c->implist_revcert );
    if( import_res[GPGME_IMPSTAT_NOSELFSIG] > 0  ) {
	msg_box( c->dlg, _("Key without a self signature was dectected!\n"	
			   "(This key is NOT usable for encryption, etc)\n"
			   "\n"	
			   "Cannot import these key(s)!"), _("Import"), MB_INFO );
    }

leave:
    free_if_alloc( src );
    return rc;
} /* fm_import */


int
fm_export( fm_state_t c )
{
    int rc = 0, id = 0;
    gpgme_ctx_t ctx = c->ctx;
    gpgme_error_t err;
    gpgme_recipients_t rset = c->recp;
    const char *name, *s = NULL;
    char *p = NULL, *dst = NULL;
    void *recp;

    if( !gpgme_recipients_count( rset ) ) {
	msg_box( c->dlg, _("No key was selected for export."), _("Export"), MB_ERR );
	rc = WPTERR_GENERAL; 
	goto leave;
    }

    if( gpgme_recipients_count( rset ) == 1 ) {
	err = gpgme_recipients_enum_open( rset, &recp );
	if( err )
	    BUG( NULL );
	s = gpgme_recipients_enum_read( rset, &recp );
	gpgme_recipients_enum_close( rset, &recp );
	p = new char[strlen( s )+1+8];
	if( !p )
	    BUG( NULL );
	strcpy( p, s );
	strcat( p, ".asc" );
    }

    name = get_filename_dlg( c->dlg, FILE_SAVE, _("Choose Name for Key File"), NULL, p? p : NULL );
			     
    if( !name )
	name = "keys.gpg";

    dst = fm_quote_file( name );
    err = gpgme_op_file_export( ctx, rset, dst );
    if( err ) {
	msg_box( c->dlg, gpgme_strerror( err ), _("Export"), MB_ERR );
	rc = WPTERR_GENERAL; 
	goto leave;	
    }
    log_box( _("GnuPG status"), MB_OK, _("Finished (Output: %s)"),  name );

leave:
    free_if_alloc( dst );
    free_if_alloc( p );
	
    return rc;
} /* fm_export */


int
fm_parse_command_line (char *cmdl)
{
    fm_state_t ctx;
    const char *s;
    char *p, *fn = NULL;
    int count = 0, detached = 0;
    
    if( !cmdl || !*cmdl )
	return 0;

    fm_state_new( &ctx );    
    ctx->dlg = GetActiveWindow( );
    ctx->cache_cb = 1;        
    
    p = cmdl;
    if( p && *p > 32 && !memistr( p, strlen( p ), "winpt.exe" ) 
		     && !strstr( p, "--" ) ) {
	count++;
	if( *p == '"' ) { /* need to remove quotes */
	    fn = new char[strlen( p )];
	    if( !fn )
		BUG( NULL );
	    memcpy( fn, p+1, strlen( p ) - 2 );
	    fn[strlen( p ) -2] = '\0';
	}
	else
	    fn = m_strdup (p);
	s = fm_get_file_type (fn);
	if (!s || !strcmp (s, "UNKNOWN"))
	    s = gnupg_check_file_ext (fn);
	if (*s == 'U') {
	    log_box( _("File Manager"), MB_ERR, _("%s: no valid OpenPGP data found."), p );
	    return count;
	}
	switch( *s ) {
	case 'E': fm_decrypt( ctx, fn ); break;
	case 'P': fm_import( ctx, fn ); break;
	case 'S':
	    file_verify_use_event( );
	    if( s[1] == 'I' ) {
		if( strlen( s ) == 13 && s[7] == 'D' )
		    detached = 1;
		fm_verify( ctx, detached, fn );
	    }
	    file_verify_wait( );
	    break; 
	}
    }

    memset( &ctx->pass_cb, 0, sizeof (ctx->pass_cb) );
    safe_free( fn );
    fm_state_release( ctx );
    return count;
} /* fm_parse_command_line */


const char *
default_dirname( const char * name )
{
    char * p = strrchr( name, '\\' );
    if( !p )
	return NULL;
    return p+1;
} /* default_dirname */


int
fm_encrypt_directory( fm_state_t c, const char * name )
{
    PK_FILE_LIST list = NULL;
    WIN32_FIND_DATA findbuf;
    HANDLE hd;
    const char * s;
    char * patt = NULL, * p;
    int rc = 0;
    
    if( !is_directory( name ) )    
	return -1;
    patt = new char[strlen( name ) + 4];
    if( !patt )
	BUG( NULL );
    strcpy( patt, name );
    strcat( patt, "\\*" );
    hd = FindFirstFile( patt, &findbuf );    
    if( !hd ) {
	free_if_alloc( patt );	
	return -1;	
    }
    if( strcmp( findbuf.cFileName, "." ) && strcmp( findbuf.cFileName, ".." ) ) {
	p = make_filename( name, findbuf.cFileName, NULL );
	pk_list_add( &list, p );
	free_if_alloc( p );
    }
    while( FindNextFile( hd, &findbuf ) ) {
	if( strcmp( findbuf.cFileName, "." ) && strcmp( findbuf.cFileName, ".." ) ) {
	    p = make_filename( name, findbuf.cFileName, NULL );
	    pk_list_add( &list, p );
	    free_if_alloc( p );
	}
    }
    s = get_filename_dlg( c->dlg, FILE_SAVE, _("Choose a Name for the Archive"), 
			  NULL, default_dirname( name ) );
    if( !s ) {
	msg_box( c->dlg, _("Invalid archive name. Exit."), _("Encrypt Directory"), MB_ERR );
	rc = -1;
	goto leave;
    }
    if( !overwrite_file( s ) ) {
	rc = -1;
	goto leave;
    }

    rc = pk_archiv_create( list, s );
    if( rc )
	msg_box( c->dlg, _("Could not create zip archive."), _("Encrypt Directory"), MB_ERR );
    else {
	fm_encrypt( c, s, 0 );
	unlink( s );
    }
leave:
    pk_list_free( list );
    free_if_alloc( patt );
    return rc;
} /* fm_encrypt_directory */


static int CALLBACK
fm_cmp_cb( LPARAM first, LPARAM second, LPARAM sortby )
{
    const char * a = 0, * b = 0;

    switch( (int)sortby ) {
    case FM_SORT_STAT:
	break;
    case FM_SORT_NAME:
	break;
    case FM_SORT_OP:
	break;
    }
    return stricmp( a, b );
} /* fm_cmp_cb */
	

int
fm_sort( listview_ctrl_t lv, int sortby )
{
    return listview_sort_items( lv, sortby, fm_cmp_cb );
} /* fm_sort */


void
fm_print_md( listview_ctrl_t lv, HWND dlg, int mdalgo )
{
    struct md_file_s mdctx;

    if( listview_count_items( lv, 0 ) == 0 )
	return;
    memset( &mdctx, 0, sizeof (mdctx) );
    mdctx.lv = lv;
    mdctx.mdalgo = mdalgo;
    DialogBoxParam( glob_hinst, (LPCTSTR)IDD_WINPT_FILE_MDSUM, dlg,
		    mdsum_dlg_proc, (LPARAM)&mdctx );
} /* fm_print_md */


int
fm_send_file (listview_ctrl_t lv)
{
    char buf[128];
    int rc;

    rc = listview_get_item_text (lv, -1, 1, buf, sizeof (buf)-1);
    if (rc == -1)
	return 0;
    mapi_send_ascfile (buf);
    return 0;
}
