/* wptClipImportDlg.cpp - WinPT key import dialog
 *	Copyright (C) 2001-2004 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <windows.h>

#include "../resource.h"

#include "wptGPG.h"
#include "wptCommonCtl.h"
#include "wptKeylist.h"
#include "wptNLS.h"
#include "wptErrors.h"
#include "wptTypes.h"
#include "wptContext.h" /* for passphrase_s */
#include "wptDlgs.h"
#include "wptW32API.h"
#include "wptVersion.h"


static HANDLE thread_hd = NULL;
static int dlg_init = 0;

struct data_callback_s {
    listview_ctrl_t lv;
    int is_revcert;
    int has_seckeys;
    const char * filename;
    HWND root_dlg, dlg;
};
typedef struct data_callback_s *data_callback_t;


BOOL CALLBACK
data_cb_dlg_proc( HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam )
{    
    static data_callback_t ctx;
    int rc;

    switch( msg ) {
    case WM_INITDIALOG:
	ctx = (data_callback_t)lparam;
	if (!ctx)
	    BUG (NULL);
	SetForegroundWindow( dlg );
	center_window( dlg );	
	dlg_init = 1;
	ctx->dlg = dlg;
	ShowWindow (ctx->root_dlg, SW_HIDE);
	SetTimer (dlg, 0x123, 100, NULL);
	break;

    case WM_TIMER:
	/* if the procedure is done, we make the callback window invisible
	   and the root dialog is forced to the forceground again. */
	rc = implist_load (ctx->lv, ctx->filename, &ctx->is_revcert, &ctx->has_seckeys);
	if (rc) {
	    KillTimer (dlg, 0x123);
	    PostQuitMessage (0);
	    EndDialog (dlg, FALSE);
	}
	SetForegroundWindow( ctx->root_dlg );
	ShowWindow( ctx->root_dlg, SW_SHOW );
	ShowWindow( dlg, SW_HIDE );
	KillTimer( dlg, 0x123 );
	break;
    }
    return FALSE;
} /* data_cb_dlg_proc */


DWORD WINAPI
data_cb_thread (void * opaque)
{
    DialogBoxParam (glob_hinst, (LPCTSTR)IDD_WINPT_DATACB, glob_hwnd, 
		    data_cb_dlg_proc, (LPARAM)opaque);
    ExitThread (0);
    return 0;
} /* data_cb_thread */


int
data_cb_dlg_create (data_callback_t ctx)
{
    DWORD tid;
    SECURITY_ATTRIBUTES sec_attr;

    if (dlg_init)
	return 0;

    memset (&sec_attr, 0, sizeof (sec_attr));
    sec_attr.bInheritHandle = FALSE;
    sec_attr.nLength = sizeof (sec_attr);
    thread_hd = CreateThread( &sec_attr, 0, data_cb_thread, (void *)ctx, 0, &tid );
    if( thread_hd == NULL ) {
	msg_box( NULL, "Could not create data callback thread", _("WinPT"), MB_ERR );
	return WPTERR_GENERAL;
    }
    return 0;
} /* data_cb_dlg_create */


void
data_cb_dlg_destroy( void )
{
    if( dlg_init ) {
	TerminateThread( thread_hd, 0 );
	CloseHandle( thread_hd );
	thread_hd = NULL;
	dlg_init = 0;
    }
} /* data_cb_dlg_destroy */


void
print_import_status( int *import_res, int is_revcert )
{
    struct import_status_s res;
    int i, rc = 0;

    memset (&res, 0, sizeof (res));
    for (i = 0; i < 14; i++)
	res.import_res[i] = import_res[i];
    res.rev_cert = is_revcert? 1 : 0;
    dialog_box_param (glob_hinst, (LPCSTR)IDD_WINPT_IMPORT_STAT, glob_hwnd,
		      import_status_dlg_proc, (LPARAM)&res,
		      _("Key Import Statistics"), IDS_WINPT_IMPORT_STAT);
    if (import_res[GPGME_IMPSTAT_IPKEYS] || import_res[GPGME_IMPSTAT_ISKEYS])
        keycache_set_reload (1);
} /* print_import_status */


data_callback_t
data_cb_new( const char *filename, listview_ctrl_t lv, HWND dlg )
{
    data_callback_t ctx;

    ctx = new struct data_callback_s;	
    if( !ctx )
	BUG( NULL );
    ctx->filename = filename;
    ctx->is_revcert = 0;
    ctx->lv = lv;
    ctx->root_dlg = dlg;
    return ctx;
} /* data_cb_new */



BOOL CALLBACK
clip_import_dlg_proc( HWND dlg, UINT msg, WPARAM wparam, LPARAM lparam )
{
    static listview_ctrl_t lv = NULL;
    static data_callback_t ctx;
    int rc = 0, id = 0, ctrl = 0, import_res[14];
    int n;
    char keyid[32];
    gpgme_error_t err;
    gpgme_recipients_t selkeys = NULL;
    
    switch( msg ) {
    case WM_INITDIALOG:
#ifndef LANG_DE
        SetWindowText( dlg, _("Key Import") );
        SetDlgItemText( dlg, IDC_IMPORT_DOIT, _("&Import") );
#endif
	rc = implist_build( &lv, GetDlgItem( dlg, IDC_IMPORT_KEYLIST ) );
	if( rc )
	    BUG( NULL );
	SetForegroundWindow( dlg );
	center_window( dlg );
	//ctx = data_cb_new( NULL, lv, dlg );
	//data_cb_dlg_create( ctx );
	ctx = new data_callback_s;
	memset (ctx, 0, sizeof (data_callback_s));
	rc = implist_load (lv, NULL, &ctx->is_revcert, &ctx->has_seckeys);
        ctrl = 0;
        return TRUE;
        
    case WM_DESTROY:
        if( lv ) {
            implist_delete( lv );
            lv = NULL;
        }
	free_if_alloc (ctx);
        return FALSE;
        
    case WM_SYSCOMMAND:
        if( LOWORD (wparam) == SC_CLOSE )
            EndDialog( dlg, TRUE );
        return FALSE;
        
    case WM_COMMAND:
        switch( LOWORD( wparam ) ) {
        case IDCANCEL:
	    data_cb_dlg_destroy();
            EndDialog( dlg, TRUE );
            return TRUE;
            
        case IDC_IMPORT_DOIT:
	    ShowWindow( ctx->dlg, SW_SHOW );
            if (ctx->has_seckeys) {
		msg_box (dlg, _("Some of the imported keys are secret keys.\n\n"
				"The ownertrust values of these keys must be\n"
				"set manually via the Key Properties dialog."),
			 _("Import"), MB_INFO);
            }
	    SetForegroundWindow( ctx->dlg );
	    center_window( ctx->dlg );
	    n = listview_count_items( lv, 0 );
	    if( n > 1 && listview_count_items( lv, 1 ) > 0 ) {
		gpgme_recipients_new( &selkeys );
		for( id = 0; id < n; id++ ) {
		    if( listview_get_item_state( lv, id ) ) {
			listview_get_item_text( lv, id, 2, keyid, sizeof (keyid)-1 );
			gpgme_recipients_add_name( selkeys, keyid+2 );
		    }
		}
	    }
		
            err = gpgme_op_clip_import( ctrl, selkeys, import_res );
            if( err ) {
                ctrl = 0;
		if( err == GPGME_Interal_GPG_Problem )
		    gnupg_display_error ();
		else
		    msg_box( dlg, gpgme_strerror( err ), _("Import"), MB_ERR );
                return FALSE;
            }
	    print_import_status( import_res, ctx->is_revcert );
            if( import_res[GPGME_IMPSTAT_NOSELFSIG] > 0  ) {
                msg_box( dlg, _("Key without a self signature was dectected!\n"
				"(This key is NOT usable for encryption, etc)\n"
				"\n"
				"Cannot import these key(s)."), _("Import"), MB_INFO );
            }
            EndDialog( dlg, TRUE );
            return TRUE;
        }
        break;
    }
    
    return FALSE;
} /* clip_import_dlg_proc */
