/* key.c - Key and keyList objects
 *	Copyright (C) 2000, 2001 Werner Koch (dd9jn), g10 Code GmbH
 *	Copyright (C) 2001-2004 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include <time.h>
#include <ctype.h>

#include "util.h"
#include "ops.h"
#include "key.h"

static const char *
pkalgo_to_string( int algo )
{
    switch( algo ) {
    case 0:
    case 1: 
    case 2:
    case 3: return "RSA";
    case 16:
    case 20: return "ElG";
    case 17: return "DSA";
    default: return "Unknown";
    }
} /* pkalgo_to_string */

static gpgme_error_t
key_new( gpgme_key_t *r_key, int secret )
{
    gpgme_key_t key;
    
    if( r_key )
	*r_key = NULL;
    key = calloc ( 1, sizeof *key );
    if (!key)
        return mk_error (Out_Of_Core);
    key->ref_count = 1;
    if( r_key )
	*r_key = key;
    if( secret )
        key->secret = 1;

    return 0;
} /* key_new */


gpgme_error_t
_gpgme_key_new( gpgme_key_t *r_key )
{
    return key_new( r_key, 0 );
} /* _gpgme_key_new */


gpgme_error_t
_gpgme_key_new_secret( gpgme_key_t *r_key )
{
    return key_new( r_key, 1 );
} /* _gpgme_key_new_secret */


void
gpgme_key_ref( gpgme_key_t key )
{
    return_if_fail( key );
    key->ref_count++;
} /* gpgme_key_ref */


static struct subkey_s *
add_subkey( gpgme_key_t key, int secret )
{
    struct subkey_s *k, *kk;
    
    k = calloc (1, sizeof *k);
    if( !k )
        return NULL;
    
    kk = key->keys.next;
    if( !kk )
        key->keys.next = k;
    else {
        while ( kk->next )
            kk = kk->next;
        kk->next = k;
    }
    if( secret )
        k->secret = 1;
    return k;
} /* add_subkey */


struct subkey_s *
_gpgme_key_add_subkey( gpgme_key_t key )
{
    return add_subkey( key, 0 );
} /* _gpgme_key_add_subkey */


struct subkey_s *
_gpgme_key_add_secret_subkey( gpgme_key_t key )
{
    return add_subkey( key, 1 );
} /* _gpgme_key_add_secret_subkey */


struct signature_s *
_gpgme_key_add_signature( gpgme_key_t key )
{
    struct signature_s *s;

    s = calloc( 1, sizeof *s );
    if( !s )
	return NULL;
    s->next = key->sigs;
    key->sigs = s;
    return s;
} /* _gpgme_key_add_signature */


void
gpgme_key_release( gpgme_key_t key )
{
    struct user_id_s *u, *u2;
    struct subkey_s *k, *k2;
    struct revoker_key_s *r, *r2;
    struct signature_s *s, *s2;
    
    if( key == NULL )
        return;
    
    assert( key->ref_count );
    if( --key->ref_count )
        return;
    
    safe_free( key->attrib.d );
    safe_free( key->keys.fingerprint );
    for( k = key->keys.next; k; k = k2 ) {
        k2 = k->next;
        safe_free( k->fingerprint );
        safe_free( k );
    }
    for( u = key->uids; u; u = u2 ) {
        u2 = u->next;
	safe_free (u->hash);
        safe_free (u);
    }
    for( r = key->rvks; r; r = r2 ) {
	r2 = r->next;
	safe_free( r );
    }
    for( s = key->sigs; s; s = s2 ) {
	s2 = s->next;
	safe_free( s->issuer );
	safe_free( s->user_id );
	safe_free( s );
    }
    safe_free( key );
} /* gpgme_key_release */

void
gpgme_key_unref( gpgme_key_t key )
{
    gpgme_key_release( key );
}


static char *
set_user_id_part( char *tail, const char *buf, size_t len )
{
    while ( len && (buf[len-1] == ' ' || buf[len-1] == '\t') ) 
        len--;
    for ( ; len; len--)
        *tail++ = *buf++;
    *tail++ = 0;
    return tail;
}


static void
parse_user_id( struct user_id_s *uid, char *tail )
{
    const char *s, *start=NULL;
    int in_name = 0;
    int in_email = 0;
    int in_comment = 0;
    
    for( s=uid->name; *s; s++ ) {
        if ( in_email ) {
            if ( *s == '<' )
                in_email++; /* not legal but anyway */
            else if (*s== '>') {
                if ( !--in_email ) {
                    if (!uid->email_part) {
                        uid->email_part = tail;
                        tail = set_user_id_part ( tail, start, s-start );
                    }
                }
            }
        }
        else if( in_comment ) {
            if( *s == '(' )
                in_comment++;
            else if (*s== ')') {
                if ( !--in_comment ) {
                    if (!uid->comment_part) {
                        uid->comment_part = tail;
                        tail = set_user_id_part ( tail, start, s-start );
                    }
                }
            }
        }
        else if( *s == '<' ) {
            if( in_name ) {
                if( !uid->name_part ) {
                    uid->name_part = tail;
                    tail = set_user_id_part (tail, start, s-start );
                }
                in_name = 0;
            }
            in_email = 1;
            start = s+1;
        }
        else if ( *s == '(' ) {
            if ( in_name ) {
                if ( !uid->name_part ) {
                    uid->name_part = tail;
                    tail = set_user_id_part (tail, start, s-start );
                }
                in_name = 0;
            }
            in_comment = 1;
            start = s+1;
        }
        else if ( !in_name && *s != ' ' && *s != '\t' ) {
            in_name = 1;
            start = s;
        }    
    }
    
    if( in_name ) {
        if ( !uid->name_part ) {
            uid->name_part = tail;
            tail = set_user_id_part (tail, start, s-start );
        }
    }
    
    /* let unused parts point to an EOS */ 
    tail--;
    if (!uid->name_part)
        uid->name_part = tail;
    if (!uid->email_part)
        uid->email_part = tail;
    if (!uid->comment_part)
        uid->comment_part = tail;
    
}


/* 
 * Take a name from the --with-colon listing, remove certain escape sequences
 * sequences and put it into the list of UIDs
 */
gpgme_error_t
_gpgme_key_append_name (gpgme_key_t key, const char * s, struct user_id_s ** u)
{
    struct user_id_s *uid, *u1;
    char *dst;
    
    assert( key );
    /* we can malloc a buffer of the same length, because the
     * converted string will never be larger. Actually we allocate it
     * twice the size, so that we are able to store the parsed stuff
     * there too */
    uid = malloc (sizeof *uid + 2*strlen (s)+3);
    if (!uid)
        return mk_error (Out_Of_Core);
    uid->flags.revoked = 0;
    uid->flags.invalid = 0;
    uid->validity = 0;
    uid->name_part = NULL;
    uid->email_part = NULL;
    uid->comment_part = NULL;    
    uid->hash = NULL;
    uid->next = NULL;
    dst = uid->name;

    _gpgme_decode_c_string( s, &dst, strlen( s )+1 );
    dst += strlen( s ) + 1;
    parse_user_id( uid, dst );
    
    if (!key->uids)
	key->uids = uid;
    else {
	for (u1 = key->uids; u1 && u1->next; u1 = u1->next)
	    ;
	u1->next = uid;
    }
    if (u)
	*u = uid;
    return 0;
}


static const char *
capabilities_to_string (struct subkey_s *k)
{
    static char *strings[8] = {
        "",
        "c",
        "s",
        "sc",
        "e",
        "ec",
        "es",
        "esc"
    };
    return strings[  (!!k->flags.can_encrypt << 2)
                  | (!!k->flags.can_sign    << 1)
                  | (!!k->flags.can_certify     ) ];
}

const char *
gpgme_key_get_string_attr( gpgme_key_t key, gpgme_attr_t what, 
			   void **reserved, int idx )
{
    const char *val = NULL;
    struct subkey_s * k;
    struct user_id_s * u;
    struct revoker_key_s * r;
    struct signature_s * s;
    struct mpi_s * m;
    
    if( !key )
        return NULL;    
    if( idx < 0 )
        return NULL;
    
    switch( what ) {
    case GPGME_ATTR_KEYID:
        for (k=&key->keys; k && idx; k=k->next, idx-- )
            ;
        if (k) 
            val = k->keyid;
        break;
    case GPGME_ATTR_FPR:
        for (k=&key->keys; k && idx; k=k->next, idx-- )
            ;
        if (k) 
            val = k->fingerprint;
        break;
    case GPGME_ATTR_ALGO:    
        for (k=&key->keys; k && idx; k=k->next, idx-- )
            ;
        if (k) 
            val = pkalgo_to_string (k->key_algo);
        break;
    case GPGME_ATTR_LEN:     
    case GPGME_ATTR_CREATED: 
    case GPGME_ATTR_EXPIRE:  
        break; /* use another get function */
    case GPGME_ATTR_OTRUST:  		  
        val = "[fixme]";
        break;
    case GPGME_ATTR_USERID:  
        for (u=key->uids; u && idx; u=u->next, idx-- )
            ;
        val = u? u->name : NULL;
        break;
    case GPGME_ATTR_NAME:   
        for (u=key->uids; u && idx; u=u->next, idx-- )
            ;
        val = u? u->name_part : NULL;
        break;
    case GPGME_ATTR_EMAIL:
        for (u=key->uids; u && idx; u=u->next, idx-- )
            ;
        val = u? u->email_part : NULL;
        break;
    case GPGME_ATTR_COMMENT:
        for( u = key->uids; u && idx; u = u->next, idx-- )
            ;
        val = u? u->comment_part : NULL;
        break;
    case GPGME_ATTR_VALIDITY:
        for( u = key->uids; u && idx; u = u->next, idx-- )
            ;
        if( u ) {
            switch( u->validity ) {
            case GPGME_VALIDITY_UNKNOWN:   val = "?"; break;
            case GPGME_VALIDITY_UNDEFINED: val = "q"; break;
            case GPGME_VALIDITY_NEVER:     val = "n"; break;
            case GPGME_VALIDITY_MARGINAL:  val = "m"; break;
            case GPGME_VALIDITY_FULL:      val = "f"; break;
            case GPGME_VALIDITY_ULTIMATE:  val = "u"; break;
            }
        }
        break;

    case GPGME_ATTR_LEVEL:  /* not used here */
    case GPGME_ATTR_TYPE:
    case GPGME_ATTR_KEY_REVOKED:
    case GPGME_ATTR_KEY_INVALID:
    case GPGME_ATTR_UID_REVOKED:
    case GPGME_ATTR_UID_INVALID:
    case GPGME_ATTR_CAN_ENCRYPT:
    case GPGME_ATTR_CAN_SIGN:
    case GPGME_ATTR_CAN_CERTIFY:
        break;

    case GPGME_ATTR_UID_HASH:
	for (u=key->uids; u && idx; u=u->next, idx--)
	    ;
	if (u)
	    val = u->hash;
	break;

    case GPGME_ATTR_IS_SECRET:
        if( key->secret )
            val = "1";
        break;

    case GPGME_ATTR_IS_ULTIMATE:
        if( key->gloflags.ultimate )
            val = "1";
        break;

    case GPGME_ATTR_KEY_CAPS:    
        for( k = &key->keys; k && idx; k = k->next, idx-- )
            ;
        if( k ) 
            val = capabilities_to_string( k );
        break;

    case GPGME_ATTR_REVKEY_FPR:
	for( r = key->rvks; r && idx; r = r->next, idx-- )
	    ;
	if( r )
	    val = r->fpr;
	break;

    case GPGME_ATTR_SIG_KEYID:
	for( s = key->sigs; s && idx; s = s->next, idx-- )
	    ;
	if( s )
	    val = s->keyid;
	break;

    case GPGME_ATTR_SIG_ISSUER:
	for( s = key->sigs; s && idx; s = s->next, idx-- )
	    ;
	if( s )
	    val = s->issuer;
	break;

    case GPGME_ATTR_SIG_USERID:
	for( s = key->sigs; s && idx; s = s->next, idx-- )
	    ;
	if( s )
	    val = s->user_id;

    case GPGME_ATTR_KEYDAT_VAL:
	for( m=key->pkey; m && idx; m=m->next, idx-- )
	    ;
	if( m )
	    val = m->hexval;
	break;

    case GPGME_ATTR_PHOTO:
	if( key->attrib.used )
	    val = key->attrib.d;
	if( reserved )
	    *reserved = (size_t *)key->attrib.len;
	break;

    case GPGME_ATTR_KEY_SYMPREFS:
	if( key->sym_prefs )
	    val = key->sym_prefs;
	break;

    case GPGME_ATTR_KEY_CARDNO:
	for( k=&key->keys; k && idx; k=k->next, idx-- )
	    ;
	if (k)
	    val = k->cardno;
	break;
    }
    return val;
}


unsigned long
gpgme_key_get_ulong_attr( gpgme_key_t key, gpgme_attr_t what,
                          void ** reserved, int idx )
{
    unsigned long val = 0;
    struct subkey_s * k;
    struct user_id_s * u;
    struct revoker_key_s * r;
    struct signature_s * s;
    struct mpi_s * m;
    int n=0;
    
    if( !key )
        return 0;    
    if( idx < 0 )
        return 0;
    
    switch( what ) {
    case GPGME_ATTR_ALGO:
        for (k=&key->keys; k && idx; k=k->next, idx-- )
            ;
        if (k) 
            val = (unsigned long)k->key_algo;
        break;
    case GPGME_ATTR_LEN:     
        for (k=&key->keys; k && idx; k=k->next, idx-- )
            ;
        if (k) 
            val = (unsigned long)k->key_len;
        break;
    case GPGME_ATTR_CREATED: 
        for (k=&key->keys; k && idx; k=k->next, idx-- )
            ;
        if (k) 
            val = k->timestamp < 0? 0L:(unsigned long)k->timestamp;
        break;
    case GPGME_ATTR_VALIDITY:
        for (u = key->uids; u && idx; u=u->next, idx--)
            ;
        if (u)
            val = u->validity;
        break;

    case GPGME_ATTR_IS_SECRET:
        val = !!key->secret;
        break;

    case GPGME_ATTR_IS_PROTECTED:
	val = key->gloflags.is_protected;
	break;

    case GPGME_ATTR_IS_ULTIMATE:
        val = key->gloflags.ultimate;
        break;

    case GPGME_ATTR_KEY_EXPIRED:
        for( k = &key->keys; k && idx; k=k->next, idx-- )
            ;
        if( k )
            val = k->flags.expired;
        break;

    case GPGME_ATTR_EXPIRES:
        for( k = &key->keys; k && idx; k=k->next, idx-- )
            ;
        if( k )
            val = k->expires;
        break;

    case GPGME_ATTR_KEY_REVOKED:
        for( k = &key->keys; k && idx; k=k->next, idx-- )
            ;
        if( k ) 
            val = k->flags.revoked;
        break;

    case GPGME_ATTR_KEY_INVALID:
        for( k = &key->keys; k && idx; k=k->next, idx-- )
            ;
        if( k ) 
            val = k->flags.invalid;
        break;

    case GPGME_ATTR_UID_CREATED:
	for (u=key->uids; u && idx; u=u->next, idx--)
	    ;
	if (u)
	    val = u->created;
	break;    

    case GPGME_ATTR_UID_REVOKED:
        for( u = key->uids; u && idx; u=u->next, idx-- )
            ;
        if( u )
            val = u->flags.revoked;
        break;

    case GPGME_ATTR_UID_INVALID:
        for( u =key->uids; u && idx; u=u->next, idx-- )
            ;
        if( u )
            val = u->flags.invalid;
        break;
    case GPGME_ATTR_CAN_ENCRYPT:
        val = key->gloflags.can_encrypt;
        break;

    case GPGME_ATTR_CAN_SIGN:
        val = key->gloflags.can_sign;
        break;

    case GPGME_ATTR_CAN_CERTIFY:
        val = key->gloflags.can_certify;
        break;

    case GPGME_ATTR_CAN_AUTH:
	val = key->gloflags.can_auth;
	break;

    case GPGME_ATTR_DIVERT_CARD:
	val = key->gloflags.divert_to_card;
	break;
        
    case GPGME_ATTR_OTRUST:
        val = key->otrust;
        break;

    case GPGME_ATTR_KEY_VALIDITY:
	val = key->validity;
	break;

    case GPGME_ATTR_REVKEY_ALGO:
	for( r = key->rvks; r && idx; r = r->next, idx-- )
	    ;
	if( r )
	    val = r->algo;
	break;

    case GPGME_ATTR_SIG_ALGO:
	for( s = key->sigs; s && idx; s = s->next, idx-- )
	    ;
	if( s )
	    val = s->algo;
	break;

    case GPGME_ATTR_SIG_CREATED:
	for( s = key->sigs; s && idx; s = s->next, idx-- )
	    ;
	if( s )
	    val = s->created;
	break;

    case GPGME_ATTR_SIG_EXPIRES:
	for( s = key->sigs; s && idx; s = s->next, idx-- )
	    ;
	if( s )
	    val = s->expires;
	break;

    case GPGME_ATTR_SIG_CLASS:
	for( s = key->sigs; s && idx; s = s->next, idx-- )
	    ;
	if( s )
	    val = s->sigclass;
	break;

    case GPGME_ATTR_SIG_FLAGS:
	for (s=key->sigs; s && idx; s=s->next, idx--)
	    ;
	if (s) {
	    if (s->is_local)
		val |= GPGME_SIGF_LOCAL;
	    if (s->is_nonrev)
		val |= GPGME_SIGF_NREV;
	}
	break;

    case GPGME_ATTR_SIG_STAT:
	for( s = key->sigs; s && idx; s = s->next, idx-- )
	    ;
	if( s )
	    val = s->stat;
	break;

    case GPGME_ATTR_KEYDAT_NP:
	for( m = key->pkey; m && idx; m=m->next, idx-- )
	    n++;
	if( m )
	    val = n;
	break;

    case GPGME_ATTR_KEYDAT_BITS:
	for( m = key->pkey; m && idx; m=m->next, idx-- )
	    ;
	if( m )
	    val = m->bits;
	break;

    case GPGME_ATTR_KEY_DISABLED:
	val = key->gloflags.disabled;
	break;

    case GPGME_ATTR_KEY_USABLE:
	n = 3;
	for (k=&key->keys; k && idx; k=k->next, idx--)
            ;
	if (k)
	{
	    if (!k->flags.revoked)
		n--;
	    if (!k->flags.expired)
		n--;
	    if (!k->flags.invalid)
		n--;
	}
	if (n == 0)
	    val = 1;
	break;
        
    default:
        break;
    }
    return val;
}


const char*
gpgme_key_expand_attr( int what, unsigned long attr )
{
    static char tmpbuf[16+1];
    struct tm *iso_date;

    switch( what ) {
    case GPGME_ATTR_ALGO:
	switch( attr ) {
	case 0:
	case 1:
	case 2:
	case 3:  return "RSA";
	case 16:
	case 20: return "ELG";
	case 17: return "DSA";
	default: return "???";
	}	
	break;
	
    case GPGME_ATTR_ALGO_SHORT:
	switch( attr ) {
	case 17: return "D";
	case 0:
	case 1:
	case 2:
	case 3: return "R";
	case 20: return "G";
	default: return "?";
	}
	break;
	
    case GPGME_ATTR_VALIDITY:
	switch( attr ) {		
	case '?': 
	case GPGME_VALIDITY_UNKNOWN:   return "Unknown";
	case 'r':
	case GPGME_VALIDITY_REVOKED:   return "Revoked";
	case 'q':
	case '-': 
	case GPGME_VALIDITY_UNDEFINED: return "Undefined";
	case 'n': 
	case GPGME_VALIDITY_NEVER:     return "Never";
	case 'm': 
	case GPGME_VALIDITY_MARGINAL:  return "Marginal";
	case 'f':
	case GPGME_VALIDITY_FULL:      return "Full";
	case 'u':
	case GPGME_VALIDITY_ULTIMATE:  return "Ultimate";
	default:                       return "???";	    
	}
	break;
        
    case GPGME_ATTR_CREATED:	
	iso_date = localtime((long*) &attr);
	_snprintf( tmpbuf, sizeof tmpbuf -1, "%.04d-%.02d-%.02d",
                   iso_date->tm_year+1900, 
                   iso_date->tm_mon+1, 
                   iso_date->tm_mday );	
	return tmpbuf;
	
    case GPGME_ATTR_LEN:       
        sprintf( tmpbuf, "%d", attr );
        return tmpbuf;

    case GPGME_ATTR_KEY_SYMPREFS:
	switch (attr) {
	case 0: return "PLAINTEXT";
	case 1: return "IDEA";
	case 2: return "3DES"; 
	case 3: return "CAST5";
	case 4: return "BLOWFISH";
	case 5: return "RESERVED";
	case 6: return "RESERVED";
	case 7: return "AES";
	case 8: return "AES-192";
	case 9: return "AES-256";
	}
	break;
    }
    return NULL;
} /* gpgme_key_expand_attr */


int
gpgme_key_get_cability( gpgme_key_t key, int attr, int keyidx )
{
    struct subkey_s *s;
    
    if( !key )
	return 0;

    for ( s = &key->keys; s && keyidx; s=s->next, keyidx-- )
        ;	
    if (!s)
	return 0;    
    switch (attr) {
    case GPGME_ATTR_CAN_ENCRYPT:	
	return s->flags.can_encrypt;
        
    case GPGME_ATTR_CAN_SIGN:	
	return s->flags.can_sign;
        
    case GPGME_ATTR_CAN_CERTIFY:
	return s->flags.can_certify;

    case GPGME_ATTR_CAN_AUTH:
	return s->flags.can_auth;
    }    
    return 0;
} /* gpgme_key_get_cability */

int
gpgme_key_count_items( gpgme_key_t key, int what )
{
    union {
	struct user_id_s *u;
	struct subkey_s *k;
	struct revoker_key_s *r;
	struct signature_s *s;
    } dat;
    int count = 0;
    
    if( !key )
	return 0;
    switch( what ) {
    case GPGME_ATTR_USERID:	
	for( dat.u = key->uids; dat.u; dat.u = dat.u->next )			
	    count++;
	return count;
        
    case GPGME_ATTR_KEYID:
	for( dat.k = &key->keys; dat.k; dat.k = dat.k->next )
	    count++;
	return count;

    case GPGME_ATTR_REVKEY_FPR:
	for( dat.r = key->rvks; dat.r; dat.r = dat.r->next )
	    count++;
	return count;

    case GPGME_ATTR_SIG_KEYID:
	for( dat.s = key->sigs; dat.s; dat.s = dat.s->next )
	    count++;
	return count;
    }
    return 0;
} /* gpgme_key_count_items */


gpgme_key_t
_gpgme_key_new_fromkeyid (const char * keyid)
{
    gpgme_key_t key;

    if (!keyid)
	return NULL;
    if (_gpgme_key_new (&key))
	return NULL;
    memset (key->keys.keyid, 0, sizeof key->keys.keyid);
    strncpy (key->keys.keyid, keyid, DIM (key->keys.keyid)-1);
    return key;
} /* _gpgme_key_new_fromkeyid */


int
gpgme_key_cability_from_algo( gpgme_pk_cipher_t algo )
{
    switch( algo ) {
    case GPGME_PK_DSA:
    case GPGME_PK_RSA_S:
	return GPGME_KEY_CANSIGN;

    case GPGME_PK_RSA:
    case GPGME_PK_ELG_ES:
	return GPGME_KEY_CANSIGN|GPGME_KEY_CANENCR;

    case GPGME_PK_RSA_E:
    case GPGME_PK_ELG_E:
	return GPGME_KEY_CANENCR;

    default:
	return 0;
    }
    return 0;
} /* gpgme_key_cability_from_algo */
