/* genkey.c -  key generation
 *	Copyright (C) 2000, 2001 Werner Koch (dd9jn), g10 Code GmbH
 *	Copyright (C) 2001-2004 Timo Schulz
 *
 * This file is part of MyGPGME.
 *
 * MyGPGME is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MyGPGME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "util.h"
#include "context.h"
#include "ops.h"


void
_gpgme_genkey_status_handler( gpgme_ctx_t ctx, gpg_status_code_t code, char *args )
{
    if( code == STATUS_PROGRESS && *args ) {
        if( ctx->cb.progress ) {
            char *p;
            int type=0, current=0, total=0;
            
            if( (p = strchr (args, ' ')) ) {
                *p++ = 0;
                if (*p) {
                    type = *(byte*)p;
                    if( (p = strchr (p+1, ' ')) ) {
                        *p++ = 0;
                        if( *p ) {
                            current = atoi (p);
                            if( (p = strchr (p+1, ' ')) ) {
                                *p++ = 0;
                                total = atoi (p);
                            }
                        }
                    }
                }
            }           
            if( type != 'X' )
                ctx->cb.progress( ctx->cb.progress_value, args, type,
                                  current, total );
        }
        return;
    }
    if (code == STATUS_KEY_CREATED) {
	char * p;
	safe_free (ctx->keygen_fpr);
	p = ctx->keygen_fpr = strdup (args+2);
	if (!p)
	    ctx->out_of_core = 1;
    }
    DEBUG2 ("genkey_status: code=%d args=`%s'\n", code, args );
}


/* 
 * Here is how the parms should be formatted:
<GnupgKeyParms format="internal">
Key-Type: DSA
Key-Length: 1024
Subkey-Type: ELG-E
Subkey-Length: 1024
Name-Real: Joe Tester
Name-Comment: with stupid passphrase
Name-Email: joe@foo.bar
Expire-Date: 0
Passphrase: abc
</GnupgKeyParms>
 * Strings should be given in UTF-8 encoding.  The format we support for now
 * "internal".  The content of the <GnupgKeyParms> container is passed 
 * verbatim to GnuPG.  Control statements (e.g. %pubring) are not allowed.
 */
gpgme_error_t
gpgme_op_genkey_start( gpgme_ctx_t c, const char * parms,
                       gpgme_data_t pubkey, gpgme_data_t seckey )
{
    gpgme_error_t rc = 0;
    const char * s, * s2, * sx;
    
    fail_on_pending_request( c );
    c->pending = 1;
    
    gpgme_data_release( c->help_data_1 ); 
    c->help_data_1 = NULL;
    
    /* create a process object */
    _gpgme_gpg_release( &c->gpg );
    rc = _gpgme_gpg_new( &c->gpg );
    if( rc )
        goto leave;
    
    /* We need a special mechanism to get the fd of a pipe here, so
     * that we can use this for the %pubring and %secring parameters.
     * We don't have this yet, so we implement only the adding to the
     * standard keyrings */
    if( pubkey || seckey ) {
        rc = mk_error( Not_Implemented );
        goto leave;
    }

    if( c->use_logging )
	_gpgme_gpg_set_logging_handler( c->gpg, c );
    _gpgme_gpg_set_status_handler( c->gpg, _gpgme_genkey_status_handler, c );
    
    /* build the commandline */
    _gpgme_gpg_add_arg ( c->gpg, "--gen-key" );
    if( c->use_armor )
        _gpgme_gpg_add_arg ( c->gpg, "--armor" );
    if ( !pubkey && !seckey )
        ; /* okay: Add key to the keyrings */
    else if( !pubkey
              || gpgme_data_get_type( pubkey ) != GPGME_DATA_TYPE_NONE ) {
        rc = mk_error ( Invalid_Value );
        goto leave;
    }
    else if( !seckey
              || gpgme_data_get_type( seckey ) != GPGME_DATA_TYPE_NONE ) {
        rc = mk_error ( Invalid_Value );
        goto leave;
    }
    
    if( pubkey ) {
        _gpgme_data_set_mode( pubkey, GPGME_DATA_MODE_IN );
        _gpgme_data_set_mode( seckey, GPGME_DATA_MODE_IN );
        /* need some more things here */
    }
    
    if( (parms = strstr( parms, "<GnupgKeyParms " ))
         && (s = strchr( parms, '>') )
         && (sx = strstr( parms, "format=\"internal\"" ))
         && sx < s
         && (s2 = strstr( s+1, "</GnupgKeyParms>" )) ) {
        /* fixme: check that there are no control statements inside */
        rc = gpgme_data_new_from_mem( &c->help_data_1, s + 1, s2 - s - 1, 1 );
    }
    else
        rc = mk_error( Invalid_Value );
    if( rc )
        goto leave;
    
    _gpgme_data_set_mode( c->help_data_1, GPGME_DATA_MODE_OUT );
    _gpgme_gpg_add_data( c->gpg, c->help_data_1, 0 );
    rc = _gpgme_gpg_spawn ( c->gpg, c );
    
leave:
    if( rc ) {
        c->pending = 0; 
        _gpgme_gpg_release ( &c->gpg );
    }

    return rc;
}


/**
 * gpgme_op_genkey:
 * @c: the context
 * @parms: XML string with the key parameters
 * @pubkey: Returns the public key
 * @seckey: Returns the secret key
 * 
 * Generate a new key and store the key in the default keyrings if both
 * @pubkey and @seckey are NULL.  If @pubkey and @seckey are given, the newly
 * created key will be returned in these data objects.
 * See gpgme_op_genkey_start() for a description of @parms.
 * 
 * Return value: 0 for success or an error code
 **/
gpgme_error_t
gpgme_op_genkey( gpgme_ctx_t c, const char * parms, 
		 gpgme_data_t pubkey, gpgme_data_t seckey )
                 
{
    gpgme_error_t rc;
    
    rc = gpgme_op_genkey_start ( c, parms, pubkey, seckey );
    if( !rc ) {
        gpgme_wait ( c, 1 );
        c->pending = 0;
    }

    return rc;
} /* gpgme_op_genkey */

static const char key_params[] =
	"<GnupgKeyParms format=\"internal\">\n"
	"Key-Type: %s\n"
	"Key-Length: %d\n"
	"Subkey-Type: %s\n"
	"Subkey-Length: %d\n"
	"Name-Real: %s\n"	
	"Name-Email: %s\n"
	"Expire-Date: %s\n"
	"Passphrase: %s\n"
	"</GnupgKeyParms>\n";

static const char key_params_with_comment[] = 
	"<GnupgKeyParms format=\"internal\">\n"
	"Key-Type: %s\n"
	"Key-Length: %d\n"
	"Subkey-Type: %s\n"
	"Subkey-Length: %d\n"
	"Name-Real: %s\n"
	"Name-Comment: %s\n"
	"Name-Email: %s\n"
	"Expire-Date: %s\n"
	"Passphrase: %s\n"
	"</GnupgKeyParms>\n";

static const char key_params_one[] = 
	"<GnupgKeyParms format=\"internal\">\n"
	"Key-Type: %s\n"
	"Key-Length: %d\n"
	"Key-Usage: %s\n"	
	"Name-Real: %s\n"	
	"Name-Email: %s\n"
	"Expire-Date: %s\n"
	"Passphrase: %s\n"
	"</GnupgKeyParms>\n";

static const char key_params_one_with_comment[] = 
	"<GnupgKeyParms format=\"internal\">\n"
	"Key-Type: %s\n"
	"Key-Length: %d\n"
	"Key-Usage: %s\n"	
	"Name-Real: %s\n"	
	"Name-Email: %s\n"
	"Name-Comment: %s\n"
	"Expire-Date: %s\n"
	"Passphrase: %s\n"
	"</GnupgKeyParms>\n";


char *
gpgme_genkey_params( int keytype, int bits, 
                     const char * user, const char * comment,const char * email,
                     const char * expdate, const char * passphrase )
{
    char *p = NULL;
    int addsize = strlen("sign encrypt");
    int size = 0;
    
    if( keytype == GPGME_KEYGEN_NONE )
        return NULL;
    
    if( comment && *comment ) {
        size = strlen( key_params_with_comment ) 
	    + 16 
	    + strlen( user )
            + addsize
	    + strlen( comment )
	    + strlen( email )
            +  strlen( passphrase ) + 32;
    }
    else {
        size = strlen( key_params ) 
	    + 16 
	    + strlen( user ) 
	    + strlen( email ) 
            + strlen( passphrase ) 
	    + addsize 
	    + 32;
    }
    if( expdate )
        size += strlen( expdate );
    p = malloc( size+1 );
    if( !p )
        return NULL;
    if( comment && *comment ) {
        switch( keytype ) {
        case GPGME_KEYGEN_DSA_ELG:
            sprintf( p, key_params_with_comment,
                     "DSA", 1024, "ELG-E", bits, user, comment, email,
                     expdate ? expdate : "0", passphrase );
            break;
            
        case GPGME_KEYGEN_DSA_RSA:
            sprintf( p, key_params_with_comment, 
                     "DSA", 1024, "RSA", bits, user, comment, email,
                     expdate ? expdate : "0", passphrase );
            break;
            
        case GPGME_KEYGEN_DSA_SIG:
            sprintf( p, key_params_one_with_comment, 
                     "DSA", 1024, "sign",
                     user, comment, email,
                     expdate ? expdate : "0", passphrase );
            break;
            
        case GPGME_KEYGEN_RSA_SIG:
            sprintf( p, key_params_one_with_comment, 
                     "RSA", bits, "sign",
                     user, comment, email,
                     expdate ? expdate : "0", passphrase );
            break;
            
        case GPGME_KEYGEN_RSA:
            sprintf( p, key_params_one_with_comment, 
                    "RSA", bits, "sign encrypt",
                    user, comment, email,
                    expdate ? expdate : "0", passphrase );
            break;

	case GPGME_KEYGEN_RSA_RSA:
	    sprintf( p, key_params_with_comment,
		     "RSA", bits, "RSA", bits, user, comment, email,
		     expdate? expdate : "0", passphrase );
	    break;
            
        default:
            safe_free( p );
	    p = NULL;
            break;
        }
    }
    else {
        switch ( keytype ) {
        case GPGME_KEYGEN_DSA_ELG:
            sprintf( p, key_params,
                    "DSA", 1024, "ELG-E", bits, user, email,
                    expdate ? expdate : "0", passphrase );
            break;
            
        case GPGME_KEYGEN_DSA_RSA:
            sprintf( p, key_params,
                    "DSA", 1024, "RSA", bits, user, email,
                    expdate ? expdate : "0", passphrase );
            break;
            
        case GPGME_KEYGEN_DSA_SIG:
            sprintf( p, key_params_one, 
                    "DSA", 1024, "sign",
                    user, email,
                    expdate ? expdate : "0", passphrase );
            break;
            
        case GPGME_KEYGEN_RSA_SIG:
            sprintf( p, key_params_one, 
                    "RSA", bits, "sign",
                    user, email,
                    expdate ? expdate : "0", passphrase );
            break;
            
        case GPGME_KEYGEN_RSA:
            sprintf( p, key_params_one, 
                    "RSA", bits, "sign encrypt",
                    user, email,
                    expdate ? expdate : "0", passphrase );
            break;

	case GPGME_KEYGEN_RSA_RSA:
	    sprintf( p, key_params,
		     "RSA", bits, "RSA", bits, user, email,
		     expdate? expdate : "0", passphrase );
	    break;
            
        default:
            safe_free( p ); 
	    p = NULL;
            break;
        }
    }
    return p;
} /* gpgme_genkey_params */


gpgme_error_t
gpgme_op_genkey_auto( const char * params, gpgme_progress_cb_t prog_cb, 
		      char ** fpr )
{
    gpgme_error_t err = 0;
    gpgme_ctx_t ctx;
    
    err = gpgme_new( &ctx );
    if( err )
        return err;
    
    if( prog_cb )
        gpgme_set_progress_cb( ctx, prog_cb, NULL );
    err = gpgme_op_genkey( ctx, params, NULL, NULL );
    if( fpr )
	*fpr = ctx->keygen_fpr? strdup( ctx->keygen_fpr ) : NULL;
    gpgme_release( ctx );
    return err;
} /* gpgme_op_genkey_auto */
