/* wptGPG.h - GnuPG interface
 *	Copyright (C) 2000-2005 Timo Schulz
 *
 * This file is part of WinPT.
 *
 * WinPT is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation; either version 2 
 * of the License, or (at your option) any later version.
 *  
 * WinPT is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with WinPT; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */

#ifndef WPT_GPG_H
#define WPT_GPG_H

#include "wptNLS.h"
#include "gpgme.h"

#define NO_STRICT 0

enum {
    SETUP_KEYGEN    = 1,
    SETUP_IMPORT    = 2,
    SETUP_EXISTING  = 3
};

enum {
    GPG_CMD_DECRYPT = 0,
    GPG_CMD_SIGN    = 1
};

enum {
    KEYCACHE_PRV = 0,
    KEYCACHE_PUB = 1
};

enum {
    ENTRY_OPAQUE = 1,
    ENTRY_SINGLE = 2,
    ENTRY_MULTI  = 3,
    ENTRY_GROUP  = 4,
};

struct gpg_filetype {    
    const char *descr;
    const char *ext;
    int nicon;
};

struct refresh_cache_s {
    int kr_reload;
    int kr_update;
    int tr_update;
};

struct gpg_option_s {
    struct gpg_option_s * next;
    char * name;
    char * val;
    unsigned int used:1;
    int type;  
};
typedef struct gpg_option_s * gpg_option_t;

struct gpg_member_s {
    struct gpg_member_s * next;
    char * name;
    unsigned int used:1;  
};
typedef struct gpg_member_s * gpg_member_t;

struct gpg_group_s {
    struct gpg_group_s * next;
    struct gpg_member_s * list;
    char * name;
    unsigned int used:1;  
};
typedef struct gpg_group_s * gpg_group_t;

struct gpg_optfile_s {
    struct gpg_option_s * list;
    struct gpg_group_s * grp;
};
typedef struct gpg_optfile_s * gpg_optfile_t;

static gpg_filetype gpg_filetypes[] = {
    {"GPG Detached Signature", ".sig", 1},
    {"GPG Encrypted Data",     ".gpg", 2},
    {"GPG Armored Data",       ".asc", 2},
    {0}
};

struct passphrase_cb_s {
    int gpg_cmd;
    gpgme_recipients_t enc_to;
    int pwd_init;
    char pwd[256];
    char info[1024]; /* hold up the info message */
    char title[256];
    const char * keyid;
    HWND hwnd;
    int cancel;
    int is_card;
};

static const char * gpg_sigstat[] = {
    _("Error during verification process."),
    _("The signature is good."),
    _("The signature is BAD!"),
    _("The signature could not be checked due to a missing key."),
    _("No valid OpenPGP signature."),
    _("Signature Error"),
    _("Good Signature (Expired Key)"),
    _("Good Signature (Revoked Key)"),
    NULL
};

#define SIGSTAT_MASK 8

extern int idea_available;

/*-- wptGPG.cpp --*/
char* get_gnupg_default_key( void );
int set_gnupg_default_key( const char *key );
char* get_gnupg_config (void);
char* get_gnupg_keyring_from_options( const char *fname, int pub );
int check_gnupg_config (const char *fname, int *secrings, int *pubrings);
char* get_gnupg_path( void );
int check_gnupg_prog( void );
int gnupg_access_files (void);
char* get_gnupg_prog( void );
char* get_gnupg_keyring (int pub, int strict);
char* get_gnupg_cfgfile (void);
int set_gnupg_options( const char *buf, size_t buflen );

int check_gnupg_options (const char *buf);
void init_gnupg_table (void);
void free_gnupg_table (void);
int keyring_check_last_access (void);
const char* gnupg_check_file_ext (const char * fname);
int gpg_check_permissions (int showmsg);
int gnupg_check_homedir (void);
int gnupg_access_keyring (int _pub);
void gnupg_backup_options (int keep);
void gnupg_backup_keyrings (void);
void gnupg_display_error (void);
int gnupg_copy_keyrings (void);
int check_gnupg_engine (int *r_major, int *r_minor, int *r_patch);

gpgme_card_t smartcard_init (void);

/*-- wptGPGME.cpp --*/
const char * get_signature_status( gpgme_sigstat_t sigstat );
void keycache_reload( HWND dlg );

int get_pubkey( const char *keyid, gpgme_key_t *ret_key );
int get_seckey( const char *keyid, gpgme_key_t *ret_skey );

void keycache_reload( HWND dlg );
void keycache_release( void );
gpgme_error_t keycache_init( const char *pubring, const char * secring );
void keycache_set_reload( int yes );
int keycache_get_reload( void );
gpgme_keycache_t keycache_get_ctx( int _pub );

int count_insecure_elgkeys( void );

int gpgme_encrypt_symmetric( void );

#define gpgme_enable_logging( ctx ) \
    gpgme_control( (ctx), GPGME_CTRL_LOGGING, 1 )

#define gpgme_show_error( hwnd, rc, ctx, title, mode )		\
do {								\
    if( (rc) == GPGME_Interal_GPG_Problem && ctx ) {		\
	char * strerr = gpgme_get_logging( (ctx) );		\
	msg_box( (hwnd), (strerr)? (strerr) :			\
	_("No GPG error description available."), title, mode );\
	safe_free( strerr );					\
    }								\
    else {							\
	msg_box( (hwnd), gpgme_strerror( (rc) ),		\
		  title, mode );				\
    }								\
} while ( 0 )

/*-- wptGPGParser.cpp --*/
/* find */
gpg_group_t find_group( gpg_optfile_t opt, const char *str );
gpg_option_t find_option( gpg_optfile_t opt, const char *str );
gpg_member_t find_member( gpg_optfile_t opt, const char *grp, const char *str );
/* delete */
int delete_group( gpg_optfile_t opt, const char *str );
int delete_member( gpg_optfile_t opt, const char *grp, const char *str );
int delete_option( gpg_optfile_t opt, const char *str );
/* add */
int modify_entry( gpg_optfile_t opt, int type, const char *name, const char *val );
int add_entry( gpg_optfile_t opt, int type, const char *name, const char *val );
int add_member( gpg_optfile_t opt, const char *grp, const char *str );
int add_group( gpg_optfile_t opt, const char *str );
/* high-level */
int parse_gpg_options( const char *file, gpg_optfile_t *r_opt );
int commit_gpg_options( const char *file, gpg_optfile_t opt );
void release_gpg_options( gpg_optfile_t opt );
/* memory */
void release_group( gpg_group_t grp );

/*-- wptPassphraseCB.cpp --*/
void set_gpg_passphrase_cb( gpgme_ctx_t c, passphrase_cb_s * ctx, int cmd, 
			    HWND hwnd, const char * title );
char * request_passphrase( const char * title, int init, int * ret_cancel );
char * request_passphrase2 (const char * title, int * ret_cancel);
const char * passphrase_cb( void * opaque, const char * desc, void * r_hd );
char * get_key_userid( const char * keyid );

#endif /* WPT_GPG_H */