/* pggstrvector.c - tools for managing vectors of strings
 *      Copyright (C) 1999 Michael Roth <mroth@gnupg.org>
 *
 * This file is part of PGG (Privacy Guard Glue).
 *
 * PGG is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * PGG is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */


#include <includes.h>
#include <pggstrvector.h>


int pgg_strvector_init(PggStrVector *strvector)
{
    strvector->used = 0;
    strvector->capacity = 32;
    
    if (!( strvector->vector = (char **)malloc(sizeof(char *) * strvector->capacity) ))
        return -1;
    
    return 0;
}


int pgg_strvector_cloneto(PggStrVector *strvector, PggStrVector *to)
{
    int		i;
    
    to->used = strvector->used;
    to->capacity = strvector->capacity;
    
    if (!( to->vector = (char **)malloc(sizeof(char *) * to->capacity) ))
        return -1;
    
    for (i=0; i<to->used; ++i)
        if (!( to->vector[i] = strdup(strvector->vector[i]) )) {
            while (--i >= 0)
                free(to->vector[i]);
            free(to->vector);
            return -1;
        }
    
    return 0;
}


void pgg_strvector_clear(PggStrVector *strvector)
{
    int		i;
    
    for (i=0; i<strvector->used; ++i)
        free(strvector->vector[i]);
    
    strvector->used = 0;
}


void pgg_strvector_release(PggStrVector *strvector)
{
    pgg_strvector_clear(strvector);
    free(strvector->vector);
}


static int strvector_ensure_capacity(PggStrVector *strvector)
{
    if (strvector->used == strvector->capacity) {
        char ** new_vector;
        
        if (!( new_vector = (char **)realloc(strvector->vector, sizeof(char *) * 2 * strvector->capacity) ))
            return -1;
        
        strvector->vector = new_vector;
        strvector->capacity *= 2;
    }
    
    return 0;
}


int pgg_strvector_append(PggStrVector *strvector, const char *str)
{
    char * 	tmp;
    
    if (strvector_ensure_capacity(strvector))
        return -1;
    
    if (!( tmp = strdup(str) ))
        return -1;
    
    strvector->vector[strvector->used++] = tmp;
    
    return 0;
}


int pgg_strvector_insert(PggStrVector *strvector, int index, const char *str)
{
    int		i;
    char *	tmp;
    
    if (strvector_ensure_capacity(strvector))
        return -1;
    
    if (!( tmp = strdup(str) ))
        return -1;
    
    for (i=strvector->used; i>index; --i)
        strvector->vector[i] = strvector->vector[i-1];
    
    strvector->vector[index] = tmp;
    
    return 0;
}


void pgg_strvector_remove(PggStrVector *strvector, int index)
{
    int		i;
    
    free(strvector->vector[index]);
    
    for (i=index; i<strvector->used; ++i)
        strvector->vector[i] = strvector->vector[i+1];
}


