/* symdecrypt.c - show how to decrypt a symmetrically encrypted file
 *      Copyright (c) 1999 by Michael Roth <mroth@gnupg.org>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * MICHAEL ROTH BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Michael Roth shall not be
 * used in advertising or otherwise to promote the sale, use or other dealings
 * in this Software without prior written authorization from Michael Roth.
 */


#include <stdio.h>
#include <pgg.h>


/*
 * You need a symmetrically encrypted file to run this example.
 * You can for example use the supplied programm called `symencrypt'
 * and pipe its output in a file.
 * Please note: It's a high security risc to submit a secret 
 * passphrase as a command line argument. However, because this is 
 * only a simple demonstration to show how to use this library we 
 * don't care about the security risk. Use this programm only for
 * testing purposes!
 *
 */
int main(int argc, char *argv[])
{
    PggSymcrypt		sc = NULL;
    PggStdio		stdio = NULL;
    PggInput		input = NULL;
    PggOutput		output = NULL;
    PggErrenv		errenv;
    
    if (argc != 3) {
        fprintf(stderr, "Usage: symdecrypt FILE PASSPHRASE\n");
        return 1;
    }
    
    /*
     * An error environment tracks and catches all errors. This call
     * resets the error environment and clear any errors set in the
     * environment.
     */
    pgg_errenv_reset(errenv);
    
    
    /*
     * A PggStdio carries the input source and output destination.
     */
    stdio = pgg_stdio_new(errenv);
    
    
    /*
     * We create a new PggInput to determine the source of the
     * data we wish to decrypt. We use the supplied filename
     * as source.
     */
    input = pgg_input_new(errenv);
    pgg_input_set_filename(input, argv[1], errenv);
    pgg_stdio_set_input(stdio, input, errenv);
    
    
    /*
     * The decrypted data should go to stdout. Because we don't
     * would like to handle the output in any other way, we just
     * request to send the decrypted data to stdout which is
     * file descriptor 1 by default.
     */
    output = pgg_output_new(errenv);
    pgg_output_set_fd(output, 1, errenv);
    pgg_stdio_set_output(stdio, output, errenv);
    
    
    /*
     * A PggSymcrypt is the actual workhorse used to decrypt a symmetric
     * encrypted message, too. We set the passphrase used for decryption
     * and the PggStdio object to specify the source and destination 
     * of all data.
     */
    sc = pgg_symcrypt_new(errenv);
    pgg_symcrypt_set_passphrase(sc, argv[2], errenv);
    pgg_symcrypt_set_stdio(sc, stdio, errenv);
    
    
    /*
     * This call actually does the complete work of calling GnuPG and 
     * parsing all GnuPG messages and outputs. When this call is returns
     * the data was decrypted and send to stdout.
     */
    pgg_symcrypt_decrypt(sc, errenv);
    
    
    /*
     * Now we check if their was any error in the calls above. When their
     * was no error the data was send to stdout
     */
    if (pgg_errenv_is_set(errenv)) {
        printf("An error occured\n");
        return 1;
    }
    
    
    /*
     * When the various Pgg Objects are no longer needed, don't forgett to
     * release them to prevent memory leaks. Because we are not interested
     * in errors on releasing objects in we are using an error environment
     * set to NULL.
     */
    pgg_output_release(output, NULL);
    pgg_input_release(input, NULL);
    pgg_stdio_release(stdio, NULL);
    pgg_symcrypt_release(sc, NULL);
    
    
    return 0;
}





