/* decrypt.c - Decrypts a file and print any signatures additionally made
 *      Copyright (c) 1999 by Michael Roth <mroth@gnupg.org>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * MICHAEL ROTH BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Michael Roth shall not be
 * used in advertising or otherwise to promote the sale, use or other dealings
 * in this Software without prior written authorization from Michael Roth.
 */



#include <includes.h>
#include <termios.h>
#include <pgg.h>


int ask_passphrase(PggPasscache passcache, PggPasscache_CbCmd cmd, const char * arg)
{
    char		input[256];
    int			i;
    struct termios	term;
    
    switch (cmd) {
        case PGG_PASSCACHE_CMD_INIT:		/* No initialization neccessary in this example */
            break;
        
        case PGG_PASSCACHE_CMD_RELEASE:		/* Cleanup isn't needed in out example, too */
            break;
        
        case PGG_PASSCACHE_CMD_PASSPHRASE:	/* Passphrase for normal key requiered */
            /*
             * Prints a message for the requested passphrase,
             * turns off the echoing from the terminal,
             * reads the passphrase typed by the user to 'input',
             * turns echo on.
             */
            fprintf(stdout, "Please type in the passphrase for keyid 0x%s > ", arg);
            fflush(stdout);
            
            tcgetattr(STDIN_FILENO, &term);
            term.c_lflag &= ~ECHO;
            tcsetattr(STDIN_FILENO, TCSANOW, &term);
            
            fgets(input, 256, stdin);
            
            term.c_lflag |= ECHO;
            tcsetattr(STDIN_FILENO, TCSANOW, &term);
            
            if (*input)
                input[strlen(input)-1] = 0;	/* Discard newline character */
            
            /* 
             * Store passphrase in the passcache. 
             * When the passphrase is stored, burn the input buffer.
             * Please note: pgg_passcache_release() burns all stored
             * passphrase in any way.
             */
            pgg_passcache_set_passphrase(passcache, arg, input, NULL);
            for (i=0; i<256; ++i)
                input[i] = 0;
            
            break;
            
        default:				/* In this example, we don't handle other events */
            return -1;				/* So we return en error indicator */
    }
    
    return 0;					/* All went ok */
}


int main(int argc, char *argv[])
{
    PggErrenv		errenv;
    PggDecrypt		decrypt;
    PggStdio		stdio;
    PggPasscache	passcache;
    int			i;
    int			error_counter = 0;
    int			success_counter = 0;
    
    pgg_errenv_reset(errenv);
    
    decrypt = pgg_decrypt_new(errenv);
    
    stdio = pgg_stdio_new(errenv);
    pgg_decrypt_set_stdio(decrypt, stdio, errenv);
    
    passcache = pgg_passcache_new(errenv);
    pgg_passcache_set_callback(passcache, ask_passphrase, errenv);
    pgg_passcache_set_policy(passcache, PGG_PASSCACHE_POLICY_REFRESH, errenv);
    pgg_passcache_set_timeout(passcache, 600, errenv);
    pgg_decrypt_set_passcache(decrypt, passcache, errenv);
    
    if (pgg_errenv_is_set(errenv)) {
        fprintf(stderr, "Can't create and/or setup decrypt, stdio and passcache objects.\n");
        return 1;
    }
    
    for (i=1; i<argc; ++i) {
        /*
         * Sets the input and output filenames for decryption.
         * Uses the nice helper function to avoid using temporary
         * PggInput and PggOutput object.
         */
        pgg_stdio_set_input_filename(stdio, argv[i], errenv);
        pgg_stdio_set_output_filename_woext(stdio, argv[i], errenv);
        
        /*
         * Now decrypt the files.
         * After the call returns and there wasn't an error, the whole
         * work was done!
         */
        pgg_decrypt_execute(decrypt, errenv);
        
        /*
         * Check for errors on the actually used file and report
         * them.
         */
        if (pgg_errenv_is_set(errenv)) {
            fprintf(stderr, "Can't decrypt file '%s'.\n", argv[i]);
            pgg_errenv_reset(errenv);
            error_counter++;
        }
        else
            success_counter++;
    }
    
    fprintf(stdout, "Files successfully decrypted: %d\n", success_counter);
    fprintf(stdout, "Failures on decryption: %d\n", error_counter);
    
    pgg_passcache_release(passcache, NULL);
    pgg_stdio_release(stdio, NULL);
    pgg_decrypt_release(decrypt, NULL);
    
    return 0;
}





