/*
 * Copyright (C) 2000 Nikos Mavroyanopoulos
 *
 * This file is part of GNUTLS.
 *
 *  The GNUTLS library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public   
 *  License as published by the Free Software Foundation; either 
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
#include "gnutls_int.h"
#include "gnutls_errors.h"
#include "debug.h"
#include <gnutls_session_pack.h>

#define SESSION_SIZE _gnutls_session_size( session)

/**
  * gnutls_session_get_data - Returns all session parameters.
  * @session: is a &gnutls_session structure.
  * @session_data: is a pointer to space to hold the session.
  * @session_data_size: is the session_data's size, or it will be set by the function.
  *
  * Returns all session parameters - in order to support resuming.
  * The client should call this - and keep the returned session - if he wants to
  * resume that current version later by calling gnutls_session_set_data()
  * This function must be called after a successful handshake.
  *
  * Resuming sessions is really useful and speedups connections after a succesful one.
  **/
int gnutls_session_get_data( gnutls_session session, opaque* session_data, int *session_data_size) {

	gnutls_datum psession;
	int ret;
	
	if (*session_data_size < SESSION_SIZE || session_data==NULL) {
		*session_data_size = SESSION_SIZE;
		session_data = NULL; /* return with the new session_data_size value */
	}

	if (session->internals.resumable==RESUME_FALSE) return GNUTLS_E_INVALID_SESSION;
	/* just return the session size */
	if (session_data==NULL) {
		return 0;
	}
	
	psession.data = session_data;
	
	ret = _gnutls_session_pack( session, &psession);
	if (ret< 0) {
		gnutls_assert();
		return ret;
	}
	*session_data_size = psession.size;

	return 0;
}


/**
  * gnutls_session_get_id - Returns session id.
  * @session: is a &gnutls_session structure.
  * @session_id: is a pointer to space to hold the session id.
  * @session_id_size: is the session id's size, or it will be set by the function.
  *
  * Returns the current session id. This can be used if you want to check if
  * the next session you tried to resume was actually resumed.
  * This is because resumed sessions have the same sessionID with the 
  * original session.
  *
  * Session id is some data set by the server, that identify the current session. 
  * In TLS 1.0 session id should not be more than 32 bytes.
  **/
int gnutls_session_get_id( gnutls_session session, void* session_id, int *session_id_size) {

	*session_id_size = session->security_parameters.session_id_size;
	
	/* just return the session size */
	if (session_id==NULL) {
		return 0;
	}
	memcpy( session_id, &session->security_parameters.session_id, *session_id_size);
	
	return 0;
}

/**
  * gnutls_session_set_data - Sets all session parameters
  * @session: is a &gnutls_session structure.
  * @session_data: is a pointer to space to hold the session.
  * @session_data_size: is the session's size
  *
  * Sets all session parameters - in order to support resuming
  * session must be the one returned by gnutls_session_get_data();
  * This function should be called before gnutls_handshake().
  * Keep in mind that session resuming is advisory. The server may
  * choose not to resume the session, thus a full handshake will be
  * performed.
  **/
int gnutls_session_set_data( gnutls_session session, const opaque* session_data, 
	int session_data_size) 
{
	int ret;
	gnutls_datum psession = { (opaque*)session_data, session_data_size };

	if (session_data==NULL || session_data_size == 0) {
		gnutls_assert();
		return GNUTLS_E_INVALID_REQUEST;
	}
	ret = _gnutls_session_unpack( session, &psession);
	if (ret < 0) {
		gnutls_assert();
		return ret;
	}
	
	return 0;
}
