package gnu.crypto.sasl.srp;

// ----------------------------------------------------------------------------
// $Id: ServerStore.java,v 1.1 2003/05/30 13:05:57 raif Exp $
//
// Copyright (C) 2003 Free Software Foundation, Inc.
//
// This file is part of GNU Crypto.
//
// GNU Crypto is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// GNU Crypto is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; see the file COPYING.  If not, write to the
//
//    Free Software Foundation Inc.,
//    59 Temple Place - Suite 330,
//    Boston, MA 02111-1307
//    USA
//
// Linking this library statically or dynamically with other modules is
// making a combined work based on this library.  Thus, the terms and
// conditions of the GNU General Public License cover the whole
// combination.
//
// As a special exception, the copyright holders of this library give
// you permission to link this library with independent modules to
// produce an executable, regardless of the license terms of these
// independent modules, and to copy and distribute the resulting
// executable under terms of your choice, provided that you also meet,
// for each linked independent module, the terms and conditions of the
// license of that module.  An independent module is a module which is
// not derived from or based on this library.  If you modify this
// library, you may extend this exception to your version of the
// library, but you are not obligated to do so.  If you do not wish to
// do so, delete this exception statement from your version.
// ----------------------------------------------------------------------------

import java.util.HashMap;

/**
 * <p>The server-side implementation of the SRP security context store.</p>
 *
 * @version $Revision: 1.1 $
 */
public class ServerStore {

   // Constants and variables
   // -------------------------------------------------------------------------

   /** The underlying singleton. */
   private static ServerStore singleton = null;

   /** The map of sid --> Security Context record. */
   private static final HashMap sid2ssc = new HashMap();

   /** The map of sid --> Session timing record. */
   private static final HashMap sid2ttl = new HashMap();

   /** A synchronisation lock. */
   private static final Object lock = new Object();

   /** A counter to generate legible SIDs. */
   private static int counter = 0;

   // Constructor(s)
   // -------------------------------------------------------------------------

   /** Private constructor to enforce Singleton pattern. */
   private ServerStore() {
      super();

      // TODO: add a cleaning timer thread
   }

   // Class methods
   // -------------------------------------------------------------------------

   /**
    * <p>Returns the classloader Singleton.</p>
    *
    * @return the classloader Singleton instance.
    */
   static synchronized final ServerStore instance() {
      if (singleton == null) {
         singleton = new ServerStore();
      }
      return singleton;
   }

   /**
    * <p>Returns a legible new session identifier.</p>
    *
    * @return a new session identifier.
    */
   static synchronized final String getNewSessionID() {
      String sid = String.valueOf(++counter);
      return new StringBuffer("SID-")
            .append("0000000000".substring(0, 10 - sid.length())).append(sid)
            .toString();
   }

   // Instance methods
   // -------------------------------------------------------------------------

   /**
    * <p>Returns a boolean flag indicating if the designated session is still
    * alive or not.</p>
    *
    * @param sid the identifier of the session to check.
    * @return <code>true</code> if the designated session is still alive.
    * <code>false</code> otherwise.
    */
   boolean isAlive(String sid) {
      boolean result = false;
      if (sid != null && !"".equals(sid)) {
         synchronized(lock) {
            StoreEntry ctx = (StoreEntry) sid2ttl.get(sid);
            if (ctx != null) {
               result = ctx.isAlive();
               if (!result) { // invalidate it en-passant
                  sid2ssc.remove(sid);
                  sid2ttl.remove(sid);
               }
            }
         }
      }
      return result;
   }

   /**
    * <p>Records a mapping between a session identifier and the Security Context
    * of the designated SRP server mechanism instance.</p>
    *
    * @param ttl the session's Time-To-Live indicator (in seconds).
    * @param ctx the server's security context.
    */
   void cacheSession(int ttl, SecurityContext ctx) {
      synchronized(lock) {
         sid2ssc.put(ctx.getSID(), ctx);
         sid2ttl.put(ctx, new StoreEntry(ttl));
      }
   }

   /**
    * <p>Updates the mapping between the designated session identifier and the
    * designated server's SASL Security Context. In the process, computes
    * and return the underlying mechanism server's evidence that shall be
    * returned to the client in a session re-use exchange.</p>
    *
    * @param sid the identifier of the session to restore.
    * @return an SRP server's security context.
    */
   SecurityContext restoreSession(String sid) {
      SecurityContext result = null;
      synchronized(lock) {
         result = (SecurityContext) sid2ssc.remove(sid);
         sid2ttl.remove(sid);
      }
      return result;
   }

   /**
    * <p>Removes all information related to the designated session ID.</p>
    *
    * @param sid the identifier of the seesion to invalidate.
    */
   void invalidateSession(String sid) {
      synchronized(lock) {
         sid2ssc.remove(sid);
         sid2ttl.remove(sid);
      }
   }
}