package gnu.crypto.sasl.srp;

// ----------------------------------------------------------------------------
// $Id: PasswordFile.java,v 1.2 2003/05/30 13:05:57 raif Exp $
//
// Copyright (C) 2003 Free Software Foundation, Inc.
//
// This file is part of GNU Crypto.
//
// GNU Crypto is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// GNU Crypto is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; see the file COPYING.  If not, write to the
//
//    Free Software Foundation Inc.,
//    59 Temple Place - Suite 330,
//    Boston, MA 02111-1307
//    USA
//
// Linking this library statically or dynamically with other modules is
// making a combined work based on this library.  Thus, the terms and
// conditions of the GNU General Public License cover the whole
// combination.
//
// As a special exception, the copyright holders of this library give
// you permission to link this library with independent modules to
// produce an executable, regardless of the license terms of these
// independent modules, and to copy and distribute the resulting
// executable under terms of your choice, provided that you also meet,
// for each linked independent module, the terms and conditions of the
// license of that module.  An independent module is a module which is
// not derived from or based on this library.  If you modify this
// library, you may extend this exception to your version of the
// library, but you are not obligated to do so.  If you do not wish to
// do so, delete this exception statement from your version.
// ----------------------------------------------------------------------------

import gnu.crypto.Registry;
import gnu.crypto.hash.IMessageDigest;
import gnu.crypto.key.srp6.SRPAlgorithm;
import gnu.crypto.sasl.NoSuchUserException;
import gnu.crypto.sasl.UserAlreadyExistsException;
import gnu.crypto.util.Util;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.math.BigInteger;
import java.util.HashMap;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.StringTokenizer;

/**
 * <p>The implementation of SRP-6 password files.</p>
 *
 * <p>For SRP, there are three (3) files:
 * <ol>
 *    <li>The password configuration file: tpasswd.conf. It contains the pairs
 *    <N,g> indexed by a number for each pair used for a user. By default, this
 *    file's pathname is constructed from the base password file pathname by
 *    prepending it with the ".conf" suffix.</li>
 *    <li>The base password file: tpasswd. It contains the related password
 *    entries for all the users with values computed using SRP's default
 *    message digest algorithm: SHA-1 (with 160-bit output block size).</li>
 *    <li>The extended password file: tpasswd2. Its name is ALWAYS constructed
 *    by adding the suffix "2" to the fully qualified pathname of the base
 *    password file. It contains, in addition to the same fields as the base
 *    password file, albeit with a different verifier value, an extra field
 *    identifying the message digest algorithm used to compute this (verifier)
 *    value.</li>
 * </ol></p>
 *
 * <p>This implementation assumes the following message digest algorithm codes:
 * <ul>
 *    <li>0: the default hash algorithm, which is SHA-1 (or its alias SHA-160).</li>
 *    <li>1: MD5.</li>
 *    <li>2: RIPEMD-128.</li>
 *    <li>3: RIPEMD-160.</li>
<!--
 *    <li>4: SHA-256.</li>
 *    <li>5: SHA-384.</li>
 *    <li>6: SHA-512.</li>
-->
 * </ul></p>
 *
 * <p>Reference:</p>
 * <ol>
 *    <li><a href="http://srp.stanford.edu/design.html">SRP Protocol Design</a><br>
 *    Thomas J. Wu.</li>
 * </ol>
 *
 * @version $Revision: 1.2 $
 */
public class PasswordFile {

   // Constants and variables
   // -------------------------------------------------------------------------

   // names of property keys used in this class
   private static final String USER_FIELD =      "user";
   private static final String VERIFIERS_FIELD = "verifier";
   private static final String SALT_FIELD =      "salt";
   private static final String CONFIG_FIELD =    "config";

   private static String DEFAULT_FILE;
   static {
      DEFAULT_FILE = System.getProperty(
            SRPRegistry.PASSWORD_FILE, SRPRegistry.DEFAULT_PASSWORD_FILE);
   }

//   /** The names of the available SRP mechanisms. */
//   public static String[] SRPMechanismNames = null;
//
   /** The SRP algorithm instances used by this object. */
   private static HashMap srps;
   static {
//      String[] names = new String[] {
//         SRPRegistry.SRP_DEFAULT_DIGEST_NAME,
//         Registry.MD5_HASH, Registry.RIPEMD_128_HASH, Registry.RIPEMD_160_HASH
////         , SRPRegistry.SRP_SHA_256_NAME
////         , SRPRegistry.SRP_SHA_384_NAME
////         , SRPRegistry.SRP_SHA_512_NAME
//      };
      HashMap map = new HashMap(SRPRegistry.SRP_ALGORITHMS.length);
//      ArrayList validnames = new ArrayList(SRPRegistry.SRP_ALGORITHMS.length);
      // The first entry MUST exist. The others are optional.
      map.put("0", SRP.instance(SRPRegistry.SRP_ALGORITHMS[0]));
//      validnames.add("SRP-"+SRPRegistry.SRP_ALGORITHMS[0]);
      for (int i = 1; i < SRPRegistry.SRP_ALGORITHMS.length; i++) {
         try {
            map.put(String.valueOf(i), SRP.instance(SRPRegistry.SRP_ALGORITHMS[i]));
//            validnames.add("SRP-"+SRPRegistry.SRP_ALGORITHMS[i]);
         } catch (Exception ignored) {
         }
      }
      srps = map;
//      SRPMechanismNames = (String[]) validnames.toArray(new String[0]);
   }

   private String confName, pwName, pw2Name;
   private File configFile, passwdFile, passwd2File;
   private long lastmodPasswdFile, lastmodPasswd2File;

   private HashMap entries = new HashMap();
   private HashMap configurations = new HashMap();

   // default N values to use when creating a new password.conf file
   private static final BigInteger[] Nsrp = new BigInteger[] {
      SRPAlgorithm.N_2048, SRPAlgorithm.N_1536, SRPAlgorithm.N_1280,
      SRPAlgorithm.N_1024, SRPAlgorithm.N_768,  SRPAlgorithm.N_640,
      SRPAlgorithm.N_512
   };

   // Constructor(s)
   // -------------------------------------------------------------------------

   public PasswordFile() throws IOException {
      this(DEFAULT_FILE);
   }

   public PasswordFile(File pwFile) throws IOException {
      this(pwFile.getAbsolutePath());
   }

   public PasswordFile(String pwName) throws IOException {
      this(pwName, pwName+"2", pwName+".conf");
   }

   public PasswordFile(String pwName, String confName) throws IOException {
      this(pwName, pwName+"2", confName);
   }

   public PasswordFile(String pwName, String pw2Name, String confName)
   throws IOException {
      super();

      this.pwName =   pwName;
      this.pw2Name =  pw2Name;
      this.confName = confName;

      readOrCreateConf();
      update();
   }

   // Class methods
   // -------------------------------------------------------------------------

   /**
    * <p>Returns a string representing the decimal value of an integer
    * identifying the message digest algorithm to use for the SRP computations.
    * </p>
    *
    * @param mdName the canonical name of a message digest algorithm.
    * @return a string representing the decimal value of an ID for that algorithm.
    */
   private static final String nameToID(String mdName) {
      if (Registry.SHA_HASH.equalsIgnoreCase(mdName)
            || Registry.SHA1_HASH.equalsIgnoreCase(mdName)
            || Registry.SHA160_HASH.equalsIgnoreCase(mdName)) {
         return "0";
      } else if (Registry.MD5_HASH.equalsIgnoreCase(mdName)) {
         return "1";
      } else if (Registry.RIPEMD128_HASH.equalsIgnoreCase(mdName)) {
         return "2";
      } else if (Registry.RIPEMD160_HASH.equalsIgnoreCase(mdName)) {
         return "3";
      }
//      else if (Registry.SHA256_HASH.equalsIgnoreCase(mdName)) {
//         return "4";
//      } else if (Registry.SHA384_HASH.equalsIgnoreCase(mdName)) {
//         return "5";
//      } else if (Registry.SHA512_HASH.equalsIgnoreCase(mdName)) {
//         return "6";
//      }
      return "0";
   }

   // SRP password configuration file methods ---------------------------------

   public synchronized boolean containsConfig(String index) throws IOException {
      checkCurrent();
      boolean result = configurations.containsKey(index);
      return result;
   }

   public synchronized String[] lookupConfig(String index) throws IOException {
      checkCurrent();
      String[] result = null;
      if (configurations.containsKey(index)) {
         result = (String[]) configurations.get(index);
      }
      return result;
   }

   // SRP base and extended password configuration files methods --------------

   public synchronized boolean contains(String user) throws IOException {
      checkCurrent();
      return entries.containsKey(user);
   }

   public synchronized void
   add(String user, String passwd, byte[] salt, String index)
   throws IOException {
      checkCurrent();
      if (entries.containsKey(user)) {
         throw new UserAlreadyExistsException(user);
      }
      HashMap fields = new HashMap(4);
      fields.put(USER_FIELD,      user); // 0
//      fields.put(VERIFIERS_FIELD, newVerifiers(salt, user, passwd, index)); // 1
      fields.put(VERIFIERS_FIELD, newVerifiers(salt, passwd, index)); // 1
      fields.put(SALT_FIELD,      Util.toBase64(salt)); // 2
      fields.put(CONFIG_FIELD,    index); // 3
      entries.put(user, fields);
      savePasswd();
   }

   public synchronized void changePasswd(String user, String passwd)
   throws IOException {
      checkCurrent();
      if (!entries.containsKey(user)) {
         throw new NoSuchUserException(user);
      }
      HashMap fields = (HashMap) entries.get(user);
      byte[] salt;
      try {
         salt = Util.fromBase64((String) fields.get(SALT_FIELD));
      } catch (NumberFormatException x) {
         throw new IOException("Password file corrupt");
      }
      String index = (String) fields.get(CONFIG_FIELD);
//      fields.put(VERIFIERS_FIELD, newVerifiers(salt, user, passwd, index));
      fields.put(VERIFIERS_FIELD, newVerifiers(salt, passwd, index));
      entries.put(user, fields);
      savePasswd();
   }

   public synchronized void savePasswd() throws IOException {
      FileOutputStream f1 = new FileOutputStream(passwdFile);
      FileOutputStream f2 = new FileOutputStream(passwd2File);
      PrintWriter pw1 = null;
      PrintWriter pw2 = null;
      try {
         pw1 = new PrintWriter(f1, true);
         pw2 = new PrintWriter(f2, true);
         this.writePasswd(pw1, pw2);
      } finally {
         if (pw1 != null) {
            try {
               pw1.flush();
            } finally {
               pw1.close();
            }
         }
         if (pw2 != null) {
            try {
               pw2.flush();
            } finally {
               pw2.close();
            }
         }
         if (f1 != null) {
            try {
               f1.close();
            } catch (IOException ignored) {
            }
         }
         if (f2 != null) {
            try {
               f2.close();
            } catch (IOException ignored) {
            }
         }
      }
      lastmodPasswdFile = passwdFile.lastModified();
      lastmodPasswd2File = passwd2File.lastModified();
   }

   /**
    * <p>Returns the triplet: verifier, salt and configuration file index, of a
    * designated user, and a designated message digest algorithm name, as an
    * array of strings.</p>
    *
    * @param user the username.
    * @param mdName the canonical name of the SRP's message digest algorithm.
    * @return a string array containing, in this order, the BASE-64 encodings of
    * the verifier, the salt and the index in the password configuration file of
    * the MPIs N and g of the designated user.
    */
   public synchronized String[] lookup(String user, String mdName)
   throws IOException {
      checkCurrent();
      if (!entries.containsKey(user)) {
         throw new NoSuchUserException(user);
      }
      HashMap fields = (HashMap) entries.get(user);
      HashMap verifiers = (HashMap) fields.get(VERIFIERS_FIELD);
      String salt =       (String)  fields.get(SALT_FIELD);
      String index =      (String)  fields.get(CONFIG_FIELD);
      String verifier =   (String) verifiers.get(nameToID(mdName));
      return new String[] { verifier, salt, index };
   }

   // Other instance methods --------------------------------------------------

   private synchronized void readOrCreateConf() throws IOException {
      configurations.clear();
      FileInputStream fis;
      configFile = new File(confName);
      try {
         fis = new FileInputStream(configFile);
         readConf(fis);
      } catch (FileNotFoundException x) { // create a default one
         String g = Util.toBase64(Util.trim(new BigInteger("2")));
         String index, N;
         for (int i = 0; i < Nsrp.length; i++) {
            index = String.valueOf(i+1);
            N = Util.toBase64(Util.trim(Nsrp[i]));
            configurations.put(index, new String[] {N, g});
         }
         FileOutputStream f0 = null;
         PrintWriter pw0 = null;
         try {
            f0 = new FileOutputStream(configFile);
            pw0 = new PrintWriter(f0, true);
            this.writeConf(pw0);
         } finally {
            if (pw0 != null) {
               pw0.close();
            } else if (f0 != null) {
               f0.close();
            }
         }
      }
   }

   private void readConf(InputStream in) throws IOException {
      BufferedReader din = new BufferedReader(new InputStreamReader(in));
      String line, index, N, g;
      StringTokenizer st;
      while ((line = din.readLine()) != null) {
         st = new StringTokenizer(line, ":");
         try {
            index = st.nextToken();
            N = st.nextToken();
            g = st.nextToken();
         } catch (NoSuchElementException x) {
            throw new IOException("SRP password configuration file corrupt");
         }
         configurations.put(index, new String[] {N, g});
      }
   }

   private void writeConf(PrintWriter pw) {
      String ndx;
      String[] mpi;
      StringBuffer sb;
      for (Iterator it = configurations.keySet().iterator(); it.hasNext(); ) {
         ndx = (String) it.next();
         mpi = (String[]) configurations.get(ndx);
         sb = new StringBuffer(ndx).append(":").append(mpi[0]).append(":").append(mpi[1]);
         pw.println(sb.toString());
      }
   }

   private HashMap newVerifiers(byte[] salt, String password, String index) {
//   newVerifiers(byte[] salt, String username, String password, String index) {
      String[] mpi = (String[]) configurations.get(index);
      BigInteger N = new BigInteger(1, Util.fromBase64(mpi[0]));
      BigInteger g = new BigInteger(1, Util.fromBase64(mpi[1]));

      HashMap result = new HashMap(4);
      IMessageDigest hash;
      BigInteger x, v;
      String verifier, digestID;
      SRP srp;
      for (int i = 0; i < srps.size(); i++) {
         digestID = String.valueOf(i);
         srp = (SRP) srps.get(digestID);
         hash = srp.newDigest();
         hash.update(salt, 0, salt.length);
         // align with SRP-6 password database definition
//         byte[] xBytes = srp.userHash(username, password);
         byte[] xBytes = password.getBytes();
         hash.update(xBytes, 0, xBytes.length);
         x = new BigInteger(1, hash.digest());
         v = g.modPow(x, N);
         verifier = Util.toBase64(v.toByteArray());

         result.put(digestID, verifier);
      }
      return result;
   }

   private synchronized void update() throws IOException {
      entries.clear();

      FileInputStream fis;
      passwdFile = new File(pwName);
      lastmodPasswdFile = passwdFile.lastModified();
      try {
         fis = new FileInputStream(passwdFile);
         readPasswd(fis);
      } catch (FileNotFoundException ignored) {
      }
      passwd2File = new File(pw2Name);
      lastmodPasswd2File = passwd2File.lastModified();
      try {
         fis = new FileInputStream(passwd2File);
         readPasswd2(fis);
      } catch (FileNotFoundException ignored) {
      }
   }

   private void checkCurrent() throws IOException {
      if (passwdFile.lastModified() > lastmodPasswdFile
            || passwd2File.lastModified() > lastmodPasswd2File) {
         update();
      }
   }

   private void readPasswd(InputStream in) throws IOException {
      BufferedReader din = new BufferedReader(new InputStreamReader(in));
      String line, user, verifier, salt, index;
      StringTokenizer st;
      while ((line = din.readLine()) != null) {
         st = new StringTokenizer(line, ":");
         try {
            user     = st.nextToken();
            verifier = st.nextToken();
            salt     = st.nextToken();
            index    = st.nextToken();
         } catch (NoSuchElementException x) {
            throw new IOException("SRP base password file corrupt");
         }

         HashMap verifiers = new HashMap(6);
         verifiers.put("0", verifier);

         HashMap fields = new HashMap(4);
         fields.put(USER_FIELD,      user);
         fields.put(VERIFIERS_FIELD, verifiers);
         fields.put(SALT_FIELD,      salt);
         fields.put(CONFIG_FIELD,    index);

         entries.put(user, fields);
      }
   }

   private void readPasswd2(InputStream in) throws IOException {
      BufferedReader din = new BufferedReader(new InputStreamReader(in));
      String line, digestID, user, verifier;
      StringTokenizer st;
      HashMap fields, verifiers;
      while ((line = din.readLine()) != null) {
         st = new StringTokenizer(line, ":");
         try {
            digestID = st.nextToken();
            user     = st.nextToken();
            verifier = st.nextToken();
         } catch (NoSuchElementException x) {
            throw new IOException("SRP extended password file corrupt");
         }

         fields = (HashMap) entries.get(user);
         if (fields != null) {
            verifiers = (HashMap) fields.get(VERIFIERS_FIELD);
            verifiers.put(digestID, verifier);
         }
      }
   }

   private void writePasswd(PrintWriter pw1, PrintWriter pw2) throws IOException {
      String user, digestID;
      HashMap fields, verifiers;
      StringBuffer sb1, sb2;
      Iterator j, i = entries.keySet().iterator();
      while (i.hasNext()) {
         user = (String) i.next();
         fields = (HashMap) entries.get(user);
         if (!user.equals(fields.get(USER_FIELD))) {
            throw new IOException("Inconsistent SRP password data");
         }
         verifiers = (HashMap) fields.get(VERIFIERS_FIELD);
         sb1 = new StringBuffer()
               .append(user).append(":")
               .append((String) verifiers.get("0")).append(":")
               .append((String) fields.get(SALT_FIELD)).append(":")
               .append((String) fields.get(CONFIG_FIELD));
         pw1.println(sb1.toString());
         // write extended information
         j = verifiers.keySet().iterator();
         while (j.hasNext()) {
            digestID = (String) j.next();
            if (!"0".equals(digestID)) {
               // #0 is the default digest, already present in tpasswd!
               sb2 = new StringBuffer()
                     .append(digestID).append(":")
                     .append(user).append(":")
                     .append((String) verifiers.get(digestID));
               pw2.println(sb2.toString());
            }
         }
      }
   }
}
