/* Seahorse
 *
 * Copyright (C) 1999, 2000 Anthony Mulcahy
 *   
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>

#include <sys/types.h>
#include <sys/wait.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "gpg-interface.h"
#include "seahorsesig.h"
#include "seahorse.h"
#include "support.h"


GtkWidget *decrypt_fileselection;
GtkWidget *encrypt_fileselection;
GtkWidget *verify_decrypt;
GtkWidget *sign_encrypt;
GtkWidget *gen_key;
GtkWidget *recv_key;
GtkWidget *send_key;
GtkWidget *recvkey_fs;
GtkWidget *sendkey_fs;

/********************* Internal State Variables *********************/

static char	*clipboard_string = NULL;
static char	*recvkey_keyfilename;
static char	*recvkey_keyid;
static char	*sendkey_keyfilename;
static char	*signencrypt_filename;
static char	*verifydecrypt_filename;
static int	ascii_arm = 1;
static int	clipboard = FALSE;
static int	decrypt = 1;
static int	detachedsign = 1;
static int	encrypt = 0;
static int	recvkey_fromkeyserver = TRUE;
static int	selection_received = FALSE;
static int	sendkey_tokeyserver = TRUE;
static int	sign = 0;
static int	signencrypt_active_btn = 0;


/********************* Utility Routines *********************/

static int
check_entry		       (GtkWidget	*widget,
				char		**text)
{
        *text = gtk_entry_get_text (GTK_ENTRY (widget));
	if (strlen (*text) == 0) {
		gtk_widget_grab_focus (widget);
		return 1;
	} else return 0;
}


static int
check_free_entry	       (GtkWidget	*widget,
				char		**text)
{
	char	*text_buf;

        text_buf = gtk_entry_get_text (GTK_ENTRY (widget));
	if (*text != NULL)
		free (*text);
        *text = strdup (text_buf);
	if (strlen (*text) == 0) {
		gtk_widget_grab_focus (widget);
		return 1;
	} else return 0;
}


static int
get_selection			(GtkWidget	*widget)
{
	static GdkAtom atom = GDK_NONE;

	selection_received = FALSE;
	atom = gdk_atom_intern ("STRING", FALSE);
	if (atom == GDK_NONE) {
		return 1;
	} else {
		gtk_selection_convert (widget,
				GDK_SELECTION_PRIMARY, atom,
				GDK_CURRENT_TIME);
		return 0;
	}
}


static GList*
make_key_list			()
{
	const char delimiter[] = "\n";
	char	*key_list;
	char	*line;
	GList	*glist = NULL;

	gpg_list_keys (&key_list);
	if (key_list != NULL) {
		line = strtok (key_list, delimiter);
		if (!strncmp (line, "pub", 3))
				glist = g_list_append (glist, line);
		while ((line = strtok (NULL, delimiter)) != NULL) {
				if (!strncmp (line, "pub", 3))
					glist = g_list_append (glist, line+11);
		}
	}

	return glist;
}

static GList*
make_name_list			()
{
	const char delimiter[] = "\n";
	char	*key_list;
	char	*line;
	GList	*glist = NULL;

	gpg_list_keys (&key_list);
	if (key_list != NULL) {
		line = strtok (key_list, delimiter);
		if (!strncmp (line, "pub", 3))
				glist = g_list_append (glist, line);
		while ((line = strtok (NULL, delimiter)) != NULL) {
				if (!strncmp (line, "pub", 3))
					glist = g_list_append (glist, line+31);
		}
	}

	return glist;
}

gboolean
on_main_win_delete_event               (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
	gtk_exit (0);
	return (FALSE); /* NOT REACHED */
}


void
on_exit1_activate                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	gtk_exit (0);
}


void
on_errorbox_clicked                    (GnomeDialog     *gnomedialog,
                                        gint             arg1,
                                        gpointer         user_data)
{
	gtk_exit (1);
}


void
encrypt_fs_button_clicked              (GtkButton       *button,
                                        gpointer         user_data)
{
	char *filename;
  	GtkWidget *sign_encrypt_file;

	filename = gtk_file_selection_get_filename
		(GTK_FILE_SELECTION (encrypt_fileselection));

  	sign_encrypt_file = get_widget (sign_encrypt, "sign_encrypt_file");
        gtk_entry_set_text (GTK_ENTRY (sign_encrypt_file),
			filename);

	gtk_widget_destroy (encrypt_fileselection);
}


void
decrypt_fs_button_clicked              (GtkButton       *button,
                                        gpointer         user_data)
{
	char *filename;
  	GtkWidget *verify_decrypt_file;

	filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION (decrypt_fileselection));

  	verify_decrypt_file = get_widget (verify_decrypt, "verify_decrypt_file");
        gtk_entry_set_text (GTK_ENTRY (verify_decrypt_file), filename);
	
	gtk_widget_destroy (decrypt_fileselection);
}


/********************* Toggle Buttons *********************/

void
on_ascii_arm_btn_toggled               (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	ascii_arm = GTK_TOGGLE_BUTTON (togglebutton)->active;
}


void
on_decrypt_clipboard_btn_toggled       (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  	GtkWidget *widget;

	clipboard = GTK_TOGGLE_BUTTON (togglebutton)->active;
 	if (clipboard) {
		widget = get_widget (verify_decrypt, 
				"verify_decrypt_file");
		gtk_widget_set_sensitive (widget, FALSE);
		widget = get_widget (verify_decrypt, 
				"verify_decrypt_file_btn");
		gtk_widget_set_sensitive (widget, FALSE);
		widget = get_widget (verify_decrypt, 
				"outfile_btn");
		gtk_widget_set_sensitive (widget, FALSE);
	} else {
		widget = get_widget (verify_decrypt, 
				"verify_decrypt_file");
		gtk_widget_set_sensitive (widget, TRUE);
		widget = get_widget (verify_decrypt, 
				"verify_decrypt_file_btn");
		gtk_widget_set_sensitive (widget, TRUE);
		widget = get_widget (verify_decrypt, 
				"outfile_btn");
		gtk_widget_set_sensitive (widget, TRUE);
	}
}


void
on_decrypt_btn_toggled                 (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  	GtkWidget *passphrase_entry;

	decrypt = GTK_TOGGLE_BUTTON (togglebutton)->active;
  	passphrase_entry = get_widget (verify_decrypt, 
				       "verify_decrypt_passphrase");
	gtk_widget_set_sensitive (passphrase_entry, decrypt);
}


void
on_detachedsign_btn_toggled            (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  	GtkWidget *widget;

	signencrypt_active_btn = 0;
 	if (GTK_TOGGLE_BUTTON (togglebutton)->active) {
		detachedsign = 1;
		encrypt = 0;
		sign = 0;
  		widget = get_widget (sign_encrypt, 
				"sign_encrypt_passphrase");
		gtk_widget_set_sensitive (widget, TRUE);
  		widget = get_widget (sign_encrypt, 
				"signencrypt_usernamecombo");
		gtk_widget_set_sensitive (widget, FALSE);
	}

}


void
on_encrypt_btn_toggled                 (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  	GtkWidget *widget;

	signencrypt_active_btn = 2;
	if (GTK_TOGGLE_BUTTON (togglebutton)->active) {
		detachedsign = 0;
		encrypt = 1;
		sign = 0;
  		widget = get_widget (sign_encrypt, 
				"sign_encrypt_passphrase");
		gtk_widget_set_sensitive (widget, FALSE);
  		widget = get_widget (sign_encrypt, 
				"signencrypt_usernamecombo");
		gtk_widget_set_sensitive (widget, TRUE);
	}
}


void
on_encrypt_clipboard_btn_toggled       (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  	GtkWidget *widget;

	clipboard = GTK_TOGGLE_BUTTON (togglebutton)->active;
 	if (clipboard) {
		widget = get_widget (sign_encrypt, 
				"sign_encrypt_file");
		gtk_widget_set_sensitive (widget, FALSE);
		widget = get_widget (sign_encrypt, 
				"sign_encrypt_file_btn");
		gtk_widget_set_sensitive (widget, FALSE);
	} else {
		widget = get_widget (sign_encrypt, 
				"sign_encrypt_file");
		gtk_widget_set_sensitive (widget, TRUE);
		widget = get_widget (sign_encrypt, 
				"sign_encrypt_file_btn");
		gtk_widget_set_sensitive (widget, TRUE);
	}
}


void
on_outfile_btn_toggled                 (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
}


void
on_recvkey_file_radiobtn_toggled       (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	GtkWidget *widget;

	if (GTK_TOGGLE_BUTTON (togglebutton)->active) {
		recvkey_fromkeyserver = FALSE;
  		widget = get_widget (recv_key, "recvkey_keyserver");
		gtk_widget_set_sensitive (widget, FALSE);
  		widget = get_widget (recv_key, "recvkey_keyid");
		gtk_widget_set_sensitive (widget, FALSE);
  		widget = get_widget (recv_key, "recvkey_keyfile");
		gtk_widget_set_sensitive (widget, TRUE);
  		widget = get_widget (recv_key, "recvkey_selectfile");
		gtk_widget_set_sensitive (widget, TRUE);
	} else {
		recvkey_fromkeyserver = TRUE;
  		widget = get_widget (recv_key, "recvkey_keyserver");
		gtk_widget_set_sensitive (widget, TRUE);
  		widget = get_widget (recv_key, "recvkey_keyid");
		gtk_widget_set_sensitive (widget, TRUE);
  		widget = get_widget (recv_key, "recvkey_keyfile");
		gtk_widget_set_sensitive (widget, FALSE);
  		widget = get_widget (recv_key, "recvkey_selectfile");
		gtk_widget_set_sensitive (widget, FALSE);
	}
}


void
on_sendkey_file_radiobtn_toggled       (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	GtkWidget *widget;

	if (GTK_TOGGLE_BUTTON (togglebutton)->active) {
		sendkey_tokeyserver = FALSE;
  		widget = get_widget (send_key, "sendkey_keyserver");
		gtk_widget_set_sensitive (widget, FALSE);
  		widget = get_widget (send_key, "sendkey_keyid");
		gtk_widget_set_sensitive (widget, TRUE);
  		widget = get_widget (send_key, "sendkey_keyfile");
		gtk_widget_set_sensitive (widget, TRUE);
  		widget = get_widget (send_key, "sendkey_selectfile");
		gtk_widget_set_sensitive (widget, TRUE);
	} else {
		sendkey_tokeyserver = TRUE;
  		widget = get_widget (send_key, "sendkey_keyserver");
		gtk_widget_set_sensitive (widget, TRUE);
  		widget = get_widget (send_key, "sendkey_keyid");
		gtk_widget_set_sensitive (widget, TRUE);
  		widget = get_widget (send_key, "sendkey_keyfile");
		gtk_widget_set_sensitive (widget, FALSE);
  		widget = get_widget (send_key, "sendkey_selectfile");
		gtk_widget_set_sensitive (widget, FALSE);
	}
}


void
on_sign_btn_toggled                    (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  	GtkWidget *widget;

	signencrypt_active_btn = 1;
	if (GTK_TOGGLE_BUTTON (togglebutton)->active) {
		detachedsign = 0;
		encrypt = 0;
		sign = 1;
  		widget = get_widget (sign_encrypt, 
				"sign_encrypt_passphrase");
		gtk_widget_set_sensitive (widget, TRUE);
  		widget = get_widget (sign_encrypt, 
				"signencrypt_usernamecombo");
		gtk_widget_set_sensitive (widget, FALSE);
	}
}


void
on_signencrypt_btn_toggled             (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  	GtkWidget *widget;

	signencrypt_active_btn = 3;
	if (GTK_TOGGLE_BUTTON (togglebutton)->active) {
		detachedsign = 0;
		encrypt = 1;
		sign = 1;
  		widget = get_widget (sign_encrypt, 
				"sign_encrypt_passphrase");
		gtk_widget_set_sensitive (widget, TRUE);
  		widget = get_widget (sign_encrypt, 
				"signencrypt_usernamecombo");
		gtk_widget_set_sensitive (widget, TRUE);
	}
}


/********************* Sign & Encrypt *********************/

void
on_sign_encrypt_activate               (GtkEditable     *editable,
                                        gpointer         user_data)
{
	on_sign_encrypt_ok_clicked (NULL, NULL);
}


void
on_sign_encrypt_ok_clicked             (GtkButton       *button,
                                        gpointer         user_data)
{
	char *output;
	char *passphrase = "";
	char *signencrypt_username;
	int result;
  	GtkWidget *log_textbox;
  	GtkWidget *passphrase_entry;
  	GtkWidget *sign_encrypt_file;
  	GtkWidget *sign_encrypt_username;


  	log_textbox = get_widget (main_win, "log_textbox");

	if (clipboard) {
		get_selection (sign_encrypt);
	}


	if (sign || detachedsign) {
  		passphrase_entry = get_widget (sign_encrypt, 
					       "sign_encrypt_passphrase");
		if (check_entry (passphrase_entry, &passphrase))
			return;
	}

	if (encrypt) {
  		sign_encrypt_username = get_widget (sign_encrypt, 
					     "sign_encrypt_username");
		if (check_entry (sign_encrypt_username,
					&signencrypt_username))
			return;
	}

	if (clipboard) {
		while (!selection_received)
			gtk_main_iteration();
		if (clipboard_string == NULL) {
			gtk_text_insert(GTK_TEXT (log_textbox), NULL, NULL, NULL, "\nCan't encrypt clipboard, because no text is selected!\n", -1);
			return;
		}
	} else {
		sign_encrypt_file = get_widget (sign_encrypt,
				"sign_encrypt_file");
		if (check_free_entry (sign_encrypt_file,
					&signencrypt_filename))
			return;
	}

	result = gpg_sign_encrypt (sign, detachedsign, encrypt,
			ascii_arm, clipboard,
			signencrypt_filename, clipboard_string,
			passphrase, signencrypt_username,
			&output);

  	gtk_widget_destroy (sign_encrypt);

	if (result == 0) {
	}
	else ;

	gtk_text_insert(GTK_TEXT (log_textbox), NULL, NULL, NULL, output, -1);
	free (output);
}


void
on_sign_encrypt_cancel_clicked         (GtkButton       *button,
                                        gpointer         user_data)
{
	gtk_widget_destroy (sign_encrypt);
}


void
on_sign_encrypt_file_clicked           (GtkButton       *button,
                                        gpointer         user_data)
{
	encrypt_fileselection = create_encrypt_fileselection ();
	gtk_widget_show (encrypt_fileselection);
}


void
encrypt_button_clicked                 (GtkButton       *button,
                                        gpointer         user_data)
{
	on_encrypt_activate (NULL, NULL);
}


void
on_encrypt_activate                    (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	GList		*glist;
	GtkWidget	*ascii_arm_btn;
	GtkWidget	*encrypt_clipboard_btn;
	GtkWidget	*combo;
  	GtkWidget	*active_btn;
	GtkWidget	*sign_encrypt_file;
	GtkWidget	*sign_encrypt_file_btn;

  	sign_encrypt = create_sign_encrypt ();

	if (signencrypt_active_btn == 1) {
		active_btn = get_widget (sign_encrypt, "sign_btn");
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (active_btn), TRUE);
	} else if (signencrypt_active_btn == 2) {
		active_btn = get_widget (sign_encrypt, "encrypt_btn");
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (active_btn), TRUE);
	} else if (signencrypt_active_btn == 3) {
		active_btn = get_widget (sign_encrypt, "signencrypt_btn");
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (active_btn), TRUE);
	} else {
		active_btn = get_widget (sign_encrypt, "detachedsign_btn");
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (active_btn), TRUE);
	}


  	sign_encrypt_file = get_widget (sign_encrypt, "sign_encrypt_file");
	if (signencrypt_filename != NULL) {
		gtk_entry_set_text (GTK_ENTRY (sign_encrypt_file),
				signencrypt_filename);
	}
	
	if ((glist = make_name_list ()) != NULL) {
		combo = get_widget (sign_encrypt,
				"signencrypt_usernamecombo");
		gtk_combo_set_popdown_strings(GTK_COMBO(combo), glist);
	}

  	ascii_arm_btn = get_widget (sign_encrypt, "ascii_arm_btn");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ascii_arm_btn),
			ascii_arm);

  	encrypt_clipboard_btn = get_widget (sign_encrypt, 
			"encrypt_clipboard_btn");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON
			(encrypt_clipboard_btn), clipboard);
  	sign_encrypt_file_btn = get_widget (sign_encrypt, 
			"sign_encrypt_file_btn");
 	if (clipboard) {
		gtk_widget_set_sensitive (sign_encrypt_file, FALSE);
		gtk_widget_set_sensitive (sign_encrypt_file_btn, FALSE);
	} else {
		gtk_widget_set_sensitive (sign_encrypt_file, TRUE);
		gtk_widget_set_sensitive (sign_encrypt_file_btn, TRUE);
	}

  	gtk_widget_show (sign_encrypt);
}


/********************* Verify & Decrypt *********************/

void
on_verify_decrypt_activate             (GtkEditable     *editable,
                                        gpointer         user_data)
{
	on_verify_decrypt_ok_clicked (NULL, NULL);
}


void
on_verify_decrypt_ok_clicked           (GtkButton       *button,
                                        gpointer         user_data)
{
	char *output;
	char *passphrase = "";
	int outfile = 0;
	int result;
  	GtkWidget *log_textbox;
  	GtkWidget *decrypt_btn;
  	GtkWidget *outfile_btn;
  	GtkWidget *verify_decrypt_file;
  	GtkWidget *passphrase_entry;


  	log_textbox = get_widget (main_win, "log_textbox");
	
	if (clipboard) {
		get_selection (verify_decrypt);
		while (!selection_received)
			gtk_main_iteration();
		if (clipboard_string == NULL) {
			gtk_text_insert(GTK_TEXT (log_textbox), NULL, NULL, NULL, "\nCan't decrypt clipboard, because no text is selected!\n", -1);
			return;
		}
	} else {
  		verify_decrypt_file = get_widget (verify_decrypt,
				"verify_decrypt_file");
		if (check_free_entry (verify_decrypt_file,
					&verifydecrypt_filename))
			return;
	}


  	decrypt_btn = get_widget (verify_decrypt, "decrypt_btn");
	if (decrypt) {
  		passphrase_entry = get_widget (verify_decrypt, 
					       "verify_decrypt_passphrase");
		if (check_entry (passphrase_entry, &passphrase))
			return;
	}

  	outfile_btn = get_widget (verify_decrypt, "outfile_btn");
	if (GTK_TOGGLE_BUTTON (outfile_btn)->active)
		outfile = 1;

	result = gpg_decrypt (decrypt, clipboard, outfile,
			verifydecrypt_filename, clipboard_string,
			passphrase, &output);

  	gtk_widget_destroy (verify_decrypt);

	gtk_text_insert(GTK_TEXT (log_textbox), NULL, NULL, NULL, output, -1);
	free (output);

	if (result == 0) {
	}
	else ;
}


void
on_verify_decrypt_cancel_clicked       (GtkButton       *button,
                                        gpointer         user_data)
{
  gtk_widget_destroy (verify_decrypt);
}


void
on_verify_decrypt_file_clicked         (GtkButton       *button,
                                        gpointer         user_data)
{
  decrypt_fileselection = create_decrypt_fileselection ();
  gtk_widget_show (decrypt_fileselection);
}


void
decrypt_button_clicked                 (GtkButton       *button,
                                        gpointer         user_data)
{
	on_decrypt_activate (NULL, NULL);
}


void
on_decrypt_activate                    (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  	GtkWidget	*decrypt_btn;
  	GtkWidget	*decrypt_clipboard_btn;
	GtkWidget	*verify_decrypt_file;
	GtkWidget	*verify_decrypt_file_btn;

	verify_decrypt = create_verify_decrypt ();

  	decrypt_btn = get_widget (verify_decrypt, "decrypt_btn");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (decrypt_btn),
			decrypt);

  	verify_decrypt_file = get_widget (verify_decrypt,
			"verify_decrypt_file");
	if (verifydecrypt_filename != NULL) {
		gtk_entry_set_text (GTK_ENTRY (verify_decrypt_file),
				verifydecrypt_filename);
	}
	
  	decrypt_clipboard_btn = get_widget (verify_decrypt, 
			"decrypt_clipboard_btn");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON
			(decrypt_clipboard_btn), clipboard);
  	verify_decrypt_file_btn = get_widget (verify_decrypt,
			"verify_decrypt_file_btn");
 	if (clipboard) {
		gtk_widget_set_sensitive (verify_decrypt_file, FALSE);
		gtk_widget_set_sensitive (verify_decrypt_file_btn, FALSE);
	} else {
		gtk_widget_set_sensitive (verify_decrypt_file, TRUE);
		gtk_widget_set_sensitive (verify_decrypt_file_btn, TRUE);
	}

	gtk_widget_show (verify_decrypt);
}


/********************* Export Key *********************/

void
on_sendkey_activate                    (GtkEditable     *editable,
                                        gpointer         user_data)
{

	on_sendkey_ok_clicked (NULL, NULL);
}

void
on_sendkey_ok_clicked                  (GtkButton       *button,
                                        gpointer         user_data)
{
	char keyid[9];
	char *keyid_buf;
	char *keyserver;
	char *output;
	int result;
  	GtkWidget *key_file;
  	GtkWidget *key_id;
  	GtkWidget *key_server;
  	GtkWidget *log_textbox;


  	log_textbox = get_widget (main_win, "log_textbox");
	
  	key_id = get_widget (send_key, "sendkey_keyid");
	if (check_entry (key_id, &keyid_buf))
		return;
	strncpy (keyid,keyid_buf, 8);
	keyid[8] = '\0';
	
	if (sendkey_tokeyserver) {
		key_server = get_widget (send_key, "sendkey_keyserver");
		if (check_entry (key_server, &keyserver))
			return;

		result = gpg_send_key (keyserver, keyid, &output);
	} else {
		key_file = get_widget (send_key, "sendkey_keyfile");
		if (check_free_entry (key_file, &sendkey_keyfilename))
			return;
		result = gpg_export_key (sendkey_keyfilename, keyid,
				ascii_arm, &output);
	}
  	gtk_widget_destroy (send_key);

	gtk_text_insert(GTK_TEXT (log_textbox), NULL, NULL, NULL,
			output, -1);
	free (output);

	if (result == 0) {
	}
	else ;
}


void
on_sendkey_cancel_clicked              (GtkButton       *button,
                                        gpointer         user_data)
{
	gtk_widget_destroy (send_key);
}


void
on_sendkey_selectkeyfile_clicked       (GtkButton       *button,
                                        gpointer         user_data)
{
	sendkey_fs = create_sendkey_fs ();
	gtk_widget_show (sendkey_fs);
}


void
exportkey_button_clicked               (GtkButton       *button,
                                        gpointer         user_data)
{
	on_send_key_activate (NULL, NULL);
}


void
on_send_key_activate                   (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	GList		*glist;
	GtkWidget	*ascii_arm_btn;
	GtkWidget	*combo;
	GtkWidget	*key_file;
	GtkWidget	*radio_btn;

	glist = make_key_list ();
	send_key = create_send_key ();

	if (glist != NULL) {
		combo = get_widget (send_key, "sendkey_keyidcombo");
		gtk_combo_set_popdown_strings(GTK_COMBO(combo), glist);
	}

	key_file = get_widget (send_key, "sendkey_keyfile");
	if (sendkey_keyfilename != NULL) {
		gtk_entry_set_text (GTK_ENTRY (key_file),
				sendkey_keyfilename);
	}
	
  	ascii_arm_btn = get_widget (send_key, "ascii_arm_btn");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ascii_arm_btn),
			ascii_arm);

	if (sendkey_tokeyserver)
		radio_btn = get_widget (send_key,
				"sendkey_server_radiobtn");
	else radio_btn = get_widget (send_key, "sendkey_file_radiobtn");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio_btn), TRUE);


	gtk_widget_show (send_key);
}


/********************* Import Key *********************/

void
on_recvkey_activate                    (GtkEditable     *editable,
                                        gpointer         user_data)
{
	on_recvkey_ok_clicked (NULL, NULL);
}

void
on_recvkey_ok_clicked                  (GtkButton       *button,
                                        gpointer         user_data)
{
	char *keyserver;
	char *output;
	int result;
  	GtkWidget *key_id;
  	GtkWidget *key_file;
  	GtkWidget *key_server;
  	GtkWidget *log_textbox;


  	log_textbox = get_widget (main_win, "log_textbox");
	
	
	if (recvkey_fromkeyserver) {
		key_server = get_widget (recv_key, "recvkey_keyserver");
		if (check_entry (key_server, &keyserver))
			return;

  		key_id = get_widget (recv_key, "recvkey_keyid");
		if (check_free_entry (key_id, &recvkey_keyid))
			return;

		result = gpg_recv_key (keyserver, recvkey_keyid, &output);
	} else {
		key_file = get_widget (recv_key, "recvkey_keyfile");
		if (check_free_entry (key_file, &recvkey_keyfilename))
			return;

		result = gpg_import_key (recvkey_keyfilename, &output);
	}

  	gtk_widget_destroy (recv_key);

	gtk_text_insert(GTK_TEXT (log_textbox), NULL, NULL, NULL, output, -1);
	free (output);

	if (result == 0) {
	}
	else ;
}


void
on_recv_key_cancel_clicked             (GtkButton       *button,
                                        gpointer         user_data)
{
  gtk_widget_destroy (recv_key);
}


void
on_recvkey_selectkeyfile_clicked       (GtkButton       *button,
                                        gpointer         user_data)
{
  recvkey_fs = create_recvkey_fs ();
  gtk_widget_show (recvkey_fs);
}


void
importkey_button_clicked               (GtkButton       *button,
                                        gpointer         user_data)
{
	on_receive_key_activate (NULL, NULL);
}


void
on_receive_key_activate                (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	GtkWidget	*key_file;
	GtkWidget	*key_id;
	GtkWidget	*radio_btn;

  	recv_key = create_recv_key ();

	key_file = get_widget (recv_key, "recvkey_keyfile");
	if (recvkey_keyfilename != NULL) {
		gtk_entry_set_text (GTK_ENTRY (key_file),
				recvkey_keyfilename);
	}
	
	key_id = get_widget (recv_key, "recvkey_keyid");
	if (recvkey_keyid != NULL) {
		gtk_entry_set_text (GTK_ENTRY (key_id),
				recvkey_keyid);
	}
	
	if (recvkey_fromkeyserver)
		radio_btn = get_widget (recv_key,
				"recvkey_server_radiobtn");
	else radio_btn = get_widget (recv_key, "recvkey_file_radiobtn");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio_btn), TRUE);

  	gtk_widget_show (recv_key);
}


/********************* *********************/

void
on_keys_dialog_ok_clicked              (GtkButton       *button,
                                        gpointer         user_data)
{
  	gtk_widget_hide (keys_dialog);
}


void
listkeys_button_clicked                (GtkButton       *button,
                                        gpointer         user_data)
{
	on_key_list_activate (NULL, NULL);
}


void
on_key_list_activate                   (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	char *output;
  	GtkWidget *key_textbox;

  	key_textbox = get_widget (keys_dialog, "key_textbox");
	gpg_list_keys (&output);
	gtk_text_set_point (GTK_TEXT (key_textbox), 0);
	gtk_text_forward_delete (GTK_TEXT (key_textbox), gtk_text_get_length (GTK_TEXT (key_textbox)));
	gtk_text_insert(GTK_TEXT (key_textbox), NULL, NULL, NULL, output, -1);
	free (output);
	gtk_widget_show (keys_dialog);
}


void
on_about_activate                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget *about_dialog;

  about_dialog = create_about_dialog ();
  gtk_widget_show (about_dialog);
}


void
recvkeyfs_ok_clicked                   (GtkButton       *button,
                                        gpointer         user_data)
{
	char *filename;
  	GtkWidget *recvkey_keyfile;

  	recvkey_keyfile = get_widget (recv_key, "recvkey_keyfile");
	filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION (recvkey_fs));
        gtk_entry_set_text (GTK_ENTRY (recvkey_keyfile), filename);
	gtk_widget_destroy (recvkey_fs);
}


void
recvkeyfs_cancel_clicked               (GtkButton       *button,
                                        gpointer         user_data)
{
	gtk_widget_destroy (recvkey_fs);
}


void
on_gen_key_activate                    (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	gen_key = create_gen_key ();
	gtk_widget_show (gen_key);
}

void
on_genkey_activate                     (GtkEditable     *editable,
                                        gpointer         user_data)
{
	on_genkey_ok_clicked (NULL, NULL);
}

void
on_genkey_ok_clicked                  (GtkButton       *button,
                                        gpointer         user_data)
{
	char	*keytype;
	char	*realname;
	char	*email;
	char	*comment;
	char	*passphrase;
	int result;
  	GtkWidget *keytype_txtbx;
  	GtkWidget *realname_txtbx;
  	GtkWidget *email_txtbx;
  	GtkWidget *comment_txtbx;
  	GtkWidget *passphrase_txtbx;
  	GtkWidget *log_textbox;


  	log_textbox = get_widget (main_win, "log_textbox");

  	keytype_txtbx = get_widget (gen_key, "keytype_txtbx");
	if (check_entry (keytype_txtbx, &keytype))
		return;

  	realname_txtbx = get_widget (gen_key, "realname_txtbx");
	if (check_entry (realname_txtbx, &realname))
		return;
	
  	email_txtbx = get_widget (gen_key, "email_txtbx");
	if (check_entry (email_txtbx, &email))
		return;
	
  	comment_txtbx = get_widget (gen_key, "comment_txtbx");
	if (check_entry (comment_txtbx, &comment))
		return;
	
  	passphrase_txtbx = get_widget (gen_key, "passphrase_txtbx");
	if (check_entry (passphrase_txtbx, &passphrase))
		return;
	
  	gtk_widget_hide (gen_key);
	result = gpg_gen_key (keytype, 1024, 0, realname, email,
			      comment, passphrase, log_textbox);
  	gtk_widget_destroy (gen_key);

	if (result == 0) {
	}
	else ;
}


void
on_genkey_cancel_clicked              (GtkButton       *button,
                                        gpointer         user_data)
{
	gtk_widget_destroy (gen_key);
}


void
sendkeyfs_ok_clicked                   (GtkButton       *button,
                                        gpointer         user_data)
{
	char *filename;
  	GtkWidget *sendkey_keyfile;

  	sendkey_keyfile = get_widget (send_key, "sendkey_keyfile");
	filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION (sendkey_fs));
        gtk_entry_set_text (GTK_ENTRY (sendkey_keyfile), filename);
	gtk_widget_destroy (sendkey_fs);
}


void
sendkeyfs_cancel_clicked               (GtkButton       *button,
                                        gpointer         user_data)
{
	gtk_widget_destroy (sendkey_fs);
}


/*********************************************************************/

void
on_sign_encrypt_selection_received     (GtkWidget       *widget,
                                        GtkSelectionData *data,
                                        guint            time,
                                        gpointer         user_data)
{
	if (clipboard_string != NULL)
		free (clipboard_string);
	selection_received = TRUE;
	if (data->length < 0) {
		/*g_print ("Selection retrieval failed\n");*/
		return;
	}
	if (data->type != GDK_SELECTION_TYPE_STRING) {
		/*g_print ("Selection was not returned as string!\n");*/
		return;
	}
        clipboard_string = strdup ((char *)data->data);
	return;
}


void
on_verify_decrypt_selection_received   (GtkWidget       *widget,
                                        GtkSelectionData *data,
                                        guint            time,
                                        gpointer         user_data)
{
	if (clipboard_string != NULL)
		free (clipboard_string);
	selection_received = TRUE;
	if (data->length < 0) {
		/*g_print ("Selection retrieval failed\n");*/
		return;
	}
	if (data->type != GDK_SELECTION_TYPE_STRING) {
		/*g_print ("Selection was not returned as string!\n");*/
		return;
	}
        clipboard_string = strdup ((char *)data->data);
	return;
}


