/*  Zgrep - search compressed files for a regular expression
    Copyright (C) 2010, 2011, 2012, 2013 Antonio Diaz Diaz.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

void show_zgrep_help()
  {
  std::printf( "Zgrep is a front end to the grep program that allows transparent search\n"
               "on any combination of compressed and uncompressed files. If any given\n"
               "file is compressed, its decompressed content is used. If a given file\n"
               "does not exist, and its name does not end with one of the known\n"
               "extensions, zgrep tries the compressed file names corresponding to the\n"
               "supported formats. If no files are specified, data is read from\n"
               "standard input, decompressed if needed, and fed to grep. Data read from\n"
               "standard input must be of the same type; all uncompressed or all\n"
               "in the same compression format.\n"
               "\nThe supported formats are bzip2, gzip, lzip and xz.\n"
               "\nUsage: zgrep [options] <pattern> [files]\n"
               "\nExit status is 0 if match, 1 if no match, 2 if trouble.\n"
               "\nOptions:\n"
               "      --help                 display this help and exit\n"
               "  -V, --version              output version information and exit\n"
               "  -a, --text                 treat all files as text\n"
               "  -A, --after-context=<n>    print <n> lines of trailing context\n"
               "  -b, --byte-offset          print the byte offset of each line\n"
               "  -B, --before-context=<n>   print <n> lines of leading context\n"
               "  -c, --count                only print a count of matching lines per file\n"
               "  -C, --context=<n>          print <n> lines of output context\n"
               "  -e, --regexp=<pattern>     use <pattern> as the pattern to match\n"
               "  -E, --extended-regexp      <pattern> is an extended regular expression\n"
               "  -f, --file=<file>          obtain patterns from <file>\n"
               "  -F, --fixed-strings        <pattern> is a set of newline-separated strings\n"
               "      --format=<fmt>         force given format (bz2, gz, lz, xz)\n"
               "  -h, --no-filename          suppress the prefixing filename on output\n"
               "  -H, --with-filename        print the filename for each match\n"
               "  -i, --ignore-case          ignore case distinctions\n"
               "  -I                         ignore binary files\n"
               "  -l, --files-with-matches   only print names of files containing matches\n"
               "  -L, --files-without-match  only print names of files containing no matches\n"
               "  -m, --max-count=<n>        stop after <n> matches\n"
               "  -n, --line-number          print the line number of each line\n"
               "  -N, --no-rcfile            don't read runtime configuration file\n"
               "  -o, --only-matching        show only the part of a line matching <pattern>\n"
               "  -q, --quiet                suppress all messages\n"
               "  -r, --recursive            operate recursively on directories\n"
               "  -s, --no-messages          suppress error messages\n"
               "  -v, --invert-match         select non-matching lines\n"
               "      --verbose              verbose mode (show error messages)\n"
               "  -w, --word-regexp          match only whole words\n"
               "  -x, --line-regexp          match only whole lines\n"
               "      --bz2=<command>        set compressor and options for bzip2 format\n"
               "      --gz=<command>         set compressor and options for gzip format\n"
               "      --lz=<command>         set compressor and options for lzip format\n"
               "      --xz=<command>         set compressor and options for xz format\n" );
  show_help_addr();
  }


int zgrep_stdin( int infd, const int format_index,
                 const std::vector< const char * > & grep_args )
  {
  Children children;
  if( !set_data_feeder( &infd, children, format_index ) ) return 2;
  const pid_t grep_pid = fork();
  if( grep_pid == 0 )			// child (grep)
    {
    if( dup2( infd, STDIN_FILENO ) >= 0 && close( infd ) == 0 )
      {
      const char ** const argv = new const char *[grep_args.size()+2];
      argv[0] = GREP;
      for( unsigned i = 0; i < grep_args.size(); ++i )
        argv[i+1] = grep_args[i];
      argv[grep_args.size()+1] = 0;
      execvp( argv[0], (char **)argv );
      }
    show_exec_error( GREP );
    _exit( 2 );
    }
  if( grep_pid < 0 )			// parent
    { show_fork_error( GREP ); return 2; }

  int retval = wait_for_child( grep_pid, GREP );

  if( !good_status( children, retval == 1 ) ) retval = 2;

  if( close( infd ) != 0 )
    { show_close_error( "data feeder" ); return 2; }
  return retval;
  }


int zgrep_file( int infd, const int format_index,
                const std::string & input_filename,
                const std::vector< const char * > & grep_args,
                const int grep_list_mode, const bool grep_show_name )
  {
  Children children;
  if( !set_data_feeder( &infd, children, format_index ) ) return 2;
  int fda[2];				// pipe from grep
  if( pipe( fda ) < 0 )
    { show_error( "Can't create pipe", errno ); return 2; }
  const pid_t grep_pid = fork();
  if( grep_pid == 0 )			// child (grep)
    {
    if( dup2( infd, STDIN_FILENO ) >= 0 &&
        dup2( fda[1], STDOUT_FILENO ) >= 0 &&
        close( infd ) == 0 && close( fda[0] ) == 0 && close( fda[1] ) == 0 )
      {
      const char ** const argv = new const char *[grep_args.size()+2];
      argv[0] = GREP;
      for( unsigned i = 0; i < grep_args.size(); ++i )
        argv[i+1] = grep_args[i];
      argv[grep_args.size()+1] = 0;
      execvp( argv[0], (char **)argv );
      }
    show_exec_error( GREP );
    _exit( 2 );
    }
  if( grep_pid < 0 )			// parent
    { show_fork_error( GREP ); return 2; }

  close( fda[1] );
  enum { buffer_size = 256 };
  uint8_t buffer[buffer_size];
  bool line_begin = true;
  while( true )
    {
    const int size = readblock( fda[0], buffer, buffer_size );
    if( size != buffer_size && errno )
      { show_error( "Read error", errno ); return 2; }
    if( size > 0 && !grep_list_mode )
      {
      if( grep_show_name )
        for( int i = 0; i < size; ++i )
          {
          if( line_begin )
            { line_begin = false; std::printf( "%s:", input_filename.c_str() ); }
          if( buffer[i] == '\n' ) line_begin = true;
          putchar( buffer[i] );
          }
      else if( std::fwrite( buffer, 1, size, stdout ) != (unsigned)size )
        { show_error( "Write error", errno ); return 2; }
      }
    if( size < buffer_size ) break;
    }

  int retval = wait_for_child( grep_pid, GREP );

  if( !good_status( children, retval == 1 ) ) retval = 2;

  if( grep_list_mode && (retval == 0) == (grep_list_mode == 1) )
    std::printf( "%s\n", input_filename.c_str() );
  if( close( infd ) != 0 )
    { show_close_error( "data feeder" ); return 2; }
  if( close( fda[0] ) != 0 )
    { show_close_error( GREP ); return 2; }
  return retval;
  }
