/*  Ztest - verify integrity of compressed files
    Copyright (C) 2010 Antonio Diaz Diaz.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

void show_ztest_help() throw()
  {
  std::printf( "Ztest verifies the integrity of the specified compressed files.\n" );
  std::printf( "Non-compressed files are ignored. If no files are specified, the\n" );
  std::printf( "integrity of compressed data read from standard input is verified. Data\n" );
  std::printf( "read from standard input must be all compressed with the same compressor.\n" );
  std::printf( "The supported compressors are bzip2, gzip, lzip and xz.\n" );
  std::printf( "\nUsage: ztest [options] [files]\n" );
  std::printf( "\nExit status is 2 if any compressed file is corrupt, 0 otherwise.\n" );
  std::printf( "\nOptions:\n" );
  std::printf( "  -h, --help                 display this help and exit\n" );
  std::printf( "  -V, --version              output version information and exit\n" );
  std::printf( "  -q, --quiet                suppress all messages\n" );
  std::printf( "  -r, --recursive            operate recursively on directories\n" );
  std::printf( "  -v, --verbose              be verbose (a 2nd -v gives more)\n" );
  show_help_addr();
  }


int ztest_stdin( const int infd,
                 const std::vector< const char * > & ztest_args )
  {
  std::string file_type;
  const uint8_t * magic_data;
  int magic_size = 0;
  if( !test_format( infd, file_type, &magic_data, &magic_size ) )
    { show_error( "Unknown data format read from stdin." ); return 2; }
  int fda[2];
  if( pipe( fda ) < 0 )
    { show_error( "Can't create pipe", errno ); return 1; }
  const pid_t pid = fork();

  if( pid == 0 )			// child (decompressor)
    {
    if( dup2( fda[0], STDIN_FILENO ) >= 0 &&
        close( fda[0] ) == 0 && close( fda[1] ) == 0 )
      {
      const char ** const argv = new const char *[ztest_args.size()+3];
      argv[0] = file_type.c_str();
      for( unsigned int i = 0; i < ztest_args.size(); ++i )
        argv[i+1] = ztest_args[i];
      argv[ztest_args.size()+1] = "-t";
      argv[ztest_args.size()+2] = 0;
      execvp( argv[0], (char **)argv );
      }
    if( verbosity >= 0 )
      std::fprintf( stderr, "%s: Can't exec `%s': %s.\n",
                    util_name, file_type.c_str(), std::strerror( errno ) );
    _exit( 1 );
    }
					// parent
  if( pid < 0 )
    {
    if( verbosity >= 0 )
      std::fprintf( stderr, "%s: Can't fork `%s': %s.\n",
                    util_name, file_type.c_str(), std::strerror( errno ) );
    return 1;
    }
  close( fda[0] );
  if( !feed_data( infd, fda[1], magic_data, magic_size ) ) return 1;
  if( close( fda[1] ) != 0 )
    { show_error( "Can't close output of data feeder", errno ); return 1; }
  return wait_for_child( pid, file_type.c_str() );
  }


int ztest_file( const int infd, const std::string & input_filename,
                const std::vector< const char * > & ztest_args )
  {
  std::string file_type;
  const uint8_t * magic_data;
  int magic_size = 0;
  if( !test_format( infd, file_type, &magic_data, &magic_size ) )
    return 0;				// ignored
  const pid_t pid = fork();

  if( pid == 0 )			// child (decompressor)
    {
    const char ** const argv = new const char *[ztest_args.size()+5];
    argv[0] = file_type.c_str();
    for( unsigned int i = 0; i < ztest_args.size(); ++i )
      argv[i+1] = ztest_args[i];
    argv[ztest_args.size()+1] = "-t";
    argv[ztest_args.size()+2] = "--";
    argv[ztest_args.size()+3] = input_filename.c_str();
    argv[ztest_args.size()+4] = 0;
    execvp( argv[0], (char **)argv );
    if( verbosity >= 0 )
      std::fprintf( stderr, "%s: Can't exec `%s': %s.\n",
                    util_name, argv[0], std::strerror( errno ) );
    _exit( 1 );
    }
					// parent
  if( pid < 0 )
    {
    if( verbosity >= 0 )
      std::fprintf( stderr, "%s: Can't fork `%s': %s.\n",
                    util_name, file_type.c_str(), std::strerror( errno ) );
    return 1;
    }
  return wait_for_child( pid, file_type.c_str() );
  }
