/*
 * Copyright (C) 2018-2019 Alexandros Theodotou <alex at zrythm dot org>
 *
 * This file is part of Zrythm
 *
 * Zrythm is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Zrythm is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Zrythm.  If not, see <https://www.gnu.org/licenses/>.
 */

/**
 * \file
 *
 * Object to hold information for the chord track.
 *
 * Contains project scale, chord markers, etc.
 */

#ifndef __AUDIO_CHORD_TRACK_H__
#define __AUDIO_CHORD_TRACK_H__

#include <stdint.h>

#include "audio/track.h"

/**
 * @addtogroup audio
 *
 * @{
 */

#define P_CHORD_TRACK (TRACKLIST->chord_track)

typedef struct ChordObject ChordObject;
typedef struct _ChordTrackWidget ChordTrackWidget;
typedef struct MusicalScale MusicalScale;

typedef struct Track ChordTrack;

/**
 * Creates a new chord Track.
 */
ChordTrack *
chord_track_new ();

/**
 * Adds a ChordObject to the Track.
 *
 * @param gen_widget Create a widget for the chord.
 */
void
chord_track_add_chord (
  ChordTrack * track,
  ChordObject *     chord,
  int          gen_widget);

/**
 * Adds a ChordObject to the Track.
 *
 * @param gen_widget Create a widget for the chord.
 */
void
chord_track_add_scale (
  ChordTrack *  track,
  ScaleObject * chord,
  int           gen_widget);

/**
 * Removes a chord from the chord Track.
 */
void
chord_track_remove_chord (
  ChordTrack *  self,
  ChordObject * chord,
  int           free);

/**
 * Removes a scale from the chord Track.
 */
void
chord_track_remove_scale (
  ChordTrack *  self,
  ScaleObject * scale,
  int free);

/**
 * Returns the current chord.
 */
#define chord_track_get_chord_at_playhead(ct) \
  chord_track_get_chord_at_pos (ct, &PLAYHEAD)

/**
 * Returns the ChordObject at the given Position
 * in the TimelineArranger.
 */
ChordObject *
chord_track_get_chord_at_pos (
  const Track * ct,
  const Position * pos);

/**
 * Returns the current scale.
 */
#define chord_track_get_scale_at_playhead(ct) \
  chord_track_get_scale_at_pos (ct, &PLAYHEAD)

/**
 * Returns the ScaleObject at the given Position
 * in the TimelineArranger.
 */
ScaleObject *
chord_track_get_scale_at_pos (
  const Track * ct,
  const Position * pos);

/**
 * Frees the chord Track.
 */
void
chord_track_free (ChordTrack * self);

/**
 * @}
 */

#endif
