#!/usr/bin/perl -w
package zebot;
use strict;
##############################################################################
=pod

=head1 NAME

zebot a fun (barking) modular irc bot 

=head1 COPYRIGHT and LICENCE

  Copyright (c) 2002 Bruno Boettcher

  zebot is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; version 2
  of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

=head1 SYNOPSIS

  customize your zebot.conf file and run ./zebot  on any shell

=head1 DESCRIPTION

ircbot written in perl, extensible through self-writable perl - modules using
OO-techniques. Thought more as a toy, even if channel camping facilities and
user managment are included: the thing acts as a dog :D

The origin from this bot comes from a french linux journal, which showed how to
write a basic bot in perl.

This code was then heavily extended, adding especially the ability to
differentiate between owner,ops and normal users. A further addition was mainly
the addition of OO-build modules.

Actually the bot was/is rewritten to use POE::Component architecture. The core
part is allready completely converted, using the P:C:IRC module for the
communication with the irc server, the modules are in rewriting stage.

The actual bot is fun writing and playing with, since it sort of emulates a
dog: tease it and it bites you, be gentle with it and it rewards you ;D

=head2 Stable modules

=over

=item PAM

acknowledge the users to pertain to a certain user group on a
certain channel, used by many other modules, e.g. to decide if a user has
operator rights or not.

=item Operators 

a module that check join messages and gives joining users
operator rights, depending on the PAM module.

=item Mailbox 

leave messages for certain users, if the user is present the
message is delivered immediately, otherwise the user is presented the message
as soon as he joins.

=back

=head2 Modules in development:

=over

=item File Server: 

a module that allows users to browse a filesystem, request
some files and be served those files in a queued fashion.

=item FServer-spider: 

a module that looks for Fserver banners in the monitored
channels and connects to those fservers fetching regularly a listing of their
content and managing download lists, fetching the files requested by the user.

=back

=head2 Modules in project stage

=over

=item Retorsion: 

get on the nerves of the bot and fear its modulated and
increasing violence :D

=item Others

Port some fun modules from other well known irc bots.

=back

=head1 Methods of this class

=over

=cut

use Data::Dumper;

sub BEGIN
{
    #push (@INC,$ENV{HOME}.'/',$ENV{HOME}.'/ircbots')
    #      if(!( (join(' ',@INC)) =~ /$ENV{HOME}\/ircbots/));
    #push (@INC,$ENV{HOME}.'/',$ENV{HOME}.'/ircbots/zebot')
    #      if(!( (join(' ',@INC)) =~ /$ENV{HOME}\/ircbots\/zebot/));
    my $pwd = $ENV{PWD};
    my @fullpath = split('/',$pwd);
    pop @fullpath;
    my $pwdup = join('/',@fullpath);
    push (@INC,$pwdup)
          if(!( (join(' ',@INC)) =~ /$pwdup/));
    #we also need to include .., since our modules are searched in zebot/*
    #print "added $pwdup to path\n";
    push (@INC,$pwd)
          if(!( (join(' ',@INC)) =~ /$pwd/));
    #print "added also $pwd to path\n";
    #print "include path is : ".Dumper(\@INC)."\n";
};


# sub POE::Kernel::ASSERT_EVENTS () { 1 };
# sub POE::Kernel::ASSERT_STATES () { 1 };
# sub POE::Kernel::ASSERT_DEFAULT () { 1 };
#sub POE::Kernel::TRACE_REFCNT () { 1 };
 #sub POE::Kernel::TRACE_DEFAULT () { 1 };

use POE::Kernel;
use POE::Session;
use zebot::ircServer;
use zebot::configHandler;
use Symbol qw(delete_package);

#####################################################################
=pod

=item new, Constructor


The constructor of a zebot object, initializes a number of datastructures that will be used later, creates a blessed reference

=cut

#####################################################################
sub new
{
  my $o_FserverHandler = { };

  bless $o_FserverHandler, "zebot";

  return $o_FserverHandler;
}# sub new
#####################################################################
=pod

=item init

Here we really build up this object

=cut

#####################################################################
sub init
{
  my ($this) = @_;
  print("ZEBOT INIT!!\n");

  $this->{"dessus"} = 0;
  $this->{"running"} = 1;

    $this->{"settings"} = {} if(!($this->{"settings"}));

 $this->{"settings"} = {
   "configpath"	=> "$ENV{HOME}/.zebot/",	#path to configfiles
   "owner"	=> "Lasade",			#username as which the bot will appear if dummypam is used
   "streamName"	=> 'botlog',			#the name of the file that will be used as debugging stream
   "reconnect"	=> 1,				#reconnect on disconnect
   "ownerpass"	=> "anappropriatedpasswd",	#global password only known by the owners of this bot, will allow to register yourself as owner vs the bot
   "language"	=> "en",			#default language of the bot
   "rejoininterval"	=> 60,			#at which rythm the bot tries to rejoin a channel it was kicked from
   "reconnectinterval"	=> 60,			#at which rythm the bot tries to rejoin a server it was disconnected from
   "debug"	=> 1,				#print out debug informations
   "pingtmout"	=> 5,				#time we are waiting before the missing ping from the server bothers us
 };
  $this->parseCmdlineArgument();
  $this->{"settings"}->{"configpath"} = $this->{"tmpsettings"}->{"configpath"}
  if($this->{"tmpsettings"}->{"configpath"});

  $this->loadSettings();
  #print("dumping settings ".Dumper($this->{"settings"})."\n");
  #print("dumping actors ".Dumper($this->{"actornames"})."\n");

 #set the reference on the bot
    $this->{"zebot"} = $this;

  open (STREAM, ">".$this->setting("streamName")) || 
    die("couldn't open ".$this->setting("streamName"));
  #TODO see how to get this stream thing working...
  #$this->{"STREAM"} = STREAM;
  my $oldfh = select(STREAM);
  $| = 1;
  select($oldfh);
  print STREAM "starting log\n";

  #create an own session to handle incoming events

POE::Session->create( object_states => [ 
  
  $this =>
                   {  _start	=> "_start",
		      _stop	=> "_stop",
                     _default	=> "_default",
                      irc_socketerr	=> "irc_socketerr",
                      irc_kick	=> "irc_kick",
                      irc_public	=> "irc_public",
		      irc_quit	=> "irc_quit",
		      irc_part	=> "irc_part",
		      irc_mode	=> "irc_mode",
		      irc_join	=> "irc_join", 
		      irc_msg	=> "irc_msg",
		      irc_ping	=> "irc_ping",
		      irc_invite	=> "irc_invite",
		      irc_notice	=> "irc_notice",
		      irc_ctcp_ping	=> "irc_ctcp_ping",
		      irc_ctcpreply_ping	=> "irc_ctcpreply_ping",
		      irc_ctcp_action	=> "irc_ctcp_action",
		      irc_ctcp_version	=> "irc_ctcp_version",
		      channel_join_topic=> "channel_join_topic",
		      channel_namelist	=> "channel_namelist",
		      channel_join_success=> "channel_join_success",
		      irc_dcc_done	=> "irc_dcc_done",
		      irc_dcc_error	=> "irc_dcc_error",
		      irc_dcc_request	=> "irc_dcc_request",
		      irc_dcc_get	=> "irc_dcc_get",
		      loadActors	=> "loadActors",
		      dropActors	=> "dropActors",
		      closeActors	=> "closeActors",
		      reloadActors	=> "reloadActors",
		      setircserver	=> "setircserver",
		      loadSettings	=> "loadSettings",
                   }
  ]
                 );
}# sub init
#########################################################
=pod

=item _default, Default handler

for the stuff that couldn't be identifyed, or where we don't have specifyed a
handler

=cut

#########################################################
sub _default 
{
  my ($state, $event, $args,$kernel,$heap) = @_[STATE, ARG0, ARG1,KERNEL,HEAP];
  $args ||= [];
  my ($this) = @_;

  # Uncomment for noisy operation.
  #print "_default::$state -- $event @$args\n";

  my $subargs = {
    "kernel" => $kernel,
    "usernick" =>"",
    "username" =>"",
    "userhost" =>"",
    "line" =>$args,
    "event" =>$event,
    "context" =>$_[SENDER], #this one holds the link ot the actual server, can be used as post target
  };

  my $theactors = $this->{"actors"};
  my $success = 0;
  foreach my $actorname (keys(%$theactors))
  {
    my $actor = $theactors->{$actorname};
    #print "Connection with [$actorname]\n";
    #print "Connection with [$actorname]:".Dumper($actor)."\n";
    $success = $success || $actor->uncatched($subargs);
  }#foreach $actor ($this->setting("actors"))
  return 0;
}# sub _default 
#########################################################
=pod

=item _start, session initialisation

This gets executed as soon as the kernel sets up this session.

=cut

#########################################################
sub _start 
{
  my ($kernel, $session,$context) = @_[KERNEL, SESSION, SENDER];
  my ($this) = @_;

  # Uncomment this to turn on more verbose POE debugging information.
  # $session->option( trace => 1 );

  # Make an alias for our session, to keep it from getting GC'ed.
  $kernel->alias_set( 'mainframe' );

  print "zebot mainframe started\n";
  $this->loadActors();
  $this->setircserver($kernel);
}# sub _start 
#########################################################
=pod

=item irc_001

connection to a server achieved

After we successfully log into the IRC server, join a channel.

=cut

#########################################################
sub irc_001 
{
  print("zebot 001\n");
  my ($this,$subargs) = @_[OBJECT,ARG0];
  #
  $subargs->{"kernel"} = $_[KERNEL];
  my $success = 0;
  my $theactors = $this->{"actors"};
  foreach my $actorname (keys(%$theactors))
  {
    my $actor = $theactors->{$actorname};
    $success = $success || $actor->CONNaction($subargs);
  }#foreach $actor ($this->setting("actors"))
}
#########################################################
=pod

=item irc_socketerr 

failed to make the connection to the server

=cut

#########################################################
sub irc_socketerr 
{
  my ($this,$args) = @_[OBJECT,ARG0];
  print("zebot::SOCJETERROR method sending init to all\n");
  print "zebot::irc_socketerr()\n";
  #loadActors();
}
#########################################################
=pod

=item _stop

allow the garbage collector to remove this

=cut

#########################################################
sub _stop 
{
  my $kernel = $_[KERNEL];
  my  $context = $_[SENDER];
  my ($this) = @_;

  print "Control session stopped.\n";
  $kernel->post( $context, 'quit', 'Neenios on ice!' );
  $kernel->alias_remove( 'mainframe' );
}
#########################################################
=pod

=item irc_connected

 handle connects and hopefully reconnects

=cut

#########################################################
sub irc_connected 
{
  my ($this,$subargs) = @_[OBJECT,ARG0];
  $subargs->{"kernel"} = $_[KERNEL];

}
#########################################################
=pod

=item irc_disconnected 

Reconnect to the server when we die.

=cut

#########################################################
sub irc_disconnected 
{
  my ($this,$subargs) = @_[OBJECT,ARG0];
  #print "irc_discon".Dumper(@_);
  print "irc_discon";
  $subargs->{"kernel"} = $_[KERNEL];
}
#########################################################
=pod

=item irc_ctcp_ping

react to a ping request

=cut

#########################################################
sub irc_ctcp_ping 
{
  my ($this,$subargs) = @_[OBJECT,ARG0];
  $subargs->{"kernel"} = $_[KERNEL];

  my $success = 0;
  my $theactors = $this->{"actors"};

  ##foreach $actor (keys(%$theactors))
  foreach my $actorname (keys(%$theactors))
  {
    my $actor = $theactors->{$actorname};
    $success = $success || $actor->PONGaction($subargs);
  }#foreach $actor ($this->setting("actors"))
}# sub irc_ping 
#########################################################
=pod

=item irc_ctcpreply_ping 

Gives lag results for outgoing PINGs.

=cut

#########################################################
sub irc_ctcpreply_ping
{
  my ($this,$subargs) = @_[OBJECT,ARG0];
  $subargs->{"kernel"} = $_[KERNEL];
}
#########################################################
=pod

=item channel_join_topic

we received a channel topic

=cut

#########################################################
sub channel_join_topic
{
  my ($this,$subargs) = @_[OBJECT,ARG0];
  $subargs->{"kernel"} = $_[KERNEL];
  #print "detected topic: ".$subargs->{server}." for ".$subargs->{args}."\n";
  #print "detected topic: ".Dumper($subargs)."\n";

  
  #trying to fetch the list of people
  return 0;
}#sub channel_namelist
#########################################################
=pod

=item channel_join_success

we managed to join the channel

=cut

#########################################################
sub channel_join_success
{
  #trying to fetch the list of people
  my ($this,$subargs ) = @_[OBJECT,ARG0];
  $subargs->{"kernel"} = $_[KERNEL];
  #print("channel_join_success:: '".$subargs->{"usernick"}."','".$subargs->{"channel"}."','".$subargs->{"network"}."'\n");

  my $botname = $subargs->{"heap"}->{"botname"};
 
  my $success = 0;
  my $theactors = $this->{"actors"};
  #foreach $actor (keys(%$theactors))
  foreach my $actorname (keys(%$theactors))
  {
    my $actor = $theactors->{$actorname};
    my %tmpcopy = %$subargs;
    #print("ZEBOT::join_success sending to $actorname if $success\n");
    $success = $success || $actor->join_success(\%tmpcopy);
  }#foreach $actor ($this->setting("actors"))
}#sub channel_join_success
#########################################################
=pod

=item module

issue the list of installed modules
ir if an argument was given return the asked module

=cut

#########################################################
sub module
{
  my ($this,$arg,$module) = @_;
  my $actorRef = $this->{"actors"};
  if($arg)
  {
    if($module)
    {
      $actorRef->{$arg} = $module; 
    }# if($module)
   return($actorRef->{$arg}); 
  }# if($arg)

  #foreach my $i (keys(%$actorRef))
  #{
  #  print("debMainTest: $i ->".$actorRef->{$i}."\n");
  #}#foreach $i (keys(%$this->{"actors"}))
  return $actorRef;
}# sub module

#########################################################
=pod

=item irc_mode

react on mode change issued by someone

=cut

#########################################################
sub irc_mode
{
  my ($this,$subargs) = @_[OBJECT,ARG0];
  $subargs->{"kernel"} = $_[KERNEL];

  my $success = 0;
  my $theactors = $this->{"actors"};
  #foreach $actor (keys(%$theactors))
  foreach my $actorname (keys(%$theactors))
  {
    my $actor = $theactors->{$actorname};
    $success = $success || $actor->MODEaction($subargs);
  }#foreach $actor ($this->setting("actors"))
}
#########################################################
=pod

=item irc_notice

react on notice send to the actual chan

=cut

#########################################################
sub irc_notice
{
  my ($this,$subargs) = @_[OBJECT,ARG0];
  $subargs->{"kernel"} = $_[KERNEL];
  my $usernick = $subargs->{"nick"};
  if ($usernick)
  {
    my $success = 0;
    my $theactors = $this->{"actors"};
    ##foreach $actor (keys(%$theactors))
    foreach my $actorname (keys(%$theactors))
    {
      my $actor = $theactors->{$actorname};
      $success = $success || $actor->NOTICEaction($subargs);
    }#foreach $actor ($this->setting("actors"))
  }#if ( $usernick =~ /\S+!\S+@\S+/)
  else
  {
    #print "server irc_notice \n";
  }
}
#########################################################
=pod

=item irc_public

react on a line send to the actual chan

=cut

#########################################################
sub irc_public
{
  my ($this,$subargs ) = @_[OBJECT,ARG0];
  $subargs->{"kernel"} = $_[KERNEL];

  my $success = 0;
  my $theactors = $this->{"actors"};
  foreach my $actorname (keys(%$theactors))
  {
    my $actor = $theactors->{$actorname};
    $success = $success || $actor->PUBLICaction($subargs);
  }#foreach $actor ($this->setting("actors"))
}# sub irc_public
#########################################################
=pod

=item irc_msg

react on private msg send directly to the bot

=cut

#########################################################
sub irc_msg
{
  my ($this,$kernel,$subargs) = @_[OBJECT,KERNEL,ARG0];
  $subargs->{"kernel"} = $_[KERNEL];
  my $success = 0;
  my $theactors = $this->{"actors"};
  foreach my $actorname (keys(%$theactors))
  {
    my $actor = $theactors->{$actorname};
    my %tmpcopy = %$subargs;
    $success = $success || $actor->PRIVMSGaction(\%tmpcopy);
  }#foreach $actor ($this->setting("actors"))
  if($success == 0)
  {
    my $nick = $subargs->{"usernick"}; #kicker
      my $restLine = $subargs->{"line"}; #kicker
      print("posting $nick waff??? ($restLine)\n");
    my $context = $subargs->{"context"};
    $kernel->post( $context, 'privmsg', $nick,"waff??? ($restLine)");
  }#if($success == 0)
}# sub irc_msg
#########################################################
=pod

=item irc_ctcp_action

react on private msg send directly to the bet

=cut

#########################################################
sub irc_ctcp_action
{
  my ($this,$subargs) = @_[OBJECT,ARG0];
  $subargs->{"kernel"} = $_[KERNEL];
  #print "irc_ctcp_action: '$nick' $name\@$host '".Dumper($addressed)."' '$line'\n";

  my $botname = $subargs->{"heap"}->{"botname"};
  my $restLine = $subargs->{"line"};

  my $success = 0;
  my $theactors = $this->{"actors"};

  #foreach $actor (keys(%$theactors))
  foreach my $actorname (keys(%$theactors))
  {
    my $actor = $theactors->{$actorname};
    $success = $success || $actor->irc_ctcp_action($subargs);
  }#foreach $actor ($this->setting("actors"))
}#irc_ctcp_action
#########################################################
=pod

=item irc_ctcp_version

spit out the version num and info when asked through ctcp version

=cut

#########################################################
sub irc_ctcp_version
{
  my ($this,$subargs) = @_[OBJECT,ARG0];
  $subargs->{"kernel"} = $_[KERNEL];

  my $basereac = $this->{"actors"}->{"basereactions"};

  $subargs->{"command"} = "version";
  $basereac->processCmd($subargs);
}#irc_ctcp_version
#########################################################
=pod

=item irc_kick

react on some user or the bot itself beeing kicked

=cut

#########################################################
sub irc_kick
{
  my ($this,$args) = @_[OBJECT,ARG0];
  $args->{"kernel"} = $_[KERNEL];

  my $nick = $args->{"usernick"}; #kicker
  my $name = $args->{"username"};
  my $host = $args->{"userhost"};
  my $addressed = $args->{"addressed"};
  my $channel = $args->{"channel"};
  my $kickeduser = $args->{"kickeduser"};
  my $reason = $args->{"reason"};
  my $kernel = $args->{"kernel"};
  my $context = $args->{"context"};
  my $heap = $args->{"heap"};

  #print "onlkicki:".Dumper(@_);
  print "onkick: on chan $channel\n";
  #:Lasade!bboett@r234m235.cybercable.tm.fr KICK '.$channel.' irc :irc
  #print "restline= ".$line->{"line"}."\n";
  my $botname = $args->{"heap"}->{"botname"};
  my $msgHand = $this->module("messageHandler");
  #my $actLang = $msgHand->fetchChannelLang($channel);
  #$args->{"lang"} = $actLang;
  my $actLang = $args->{"lang"};

  #print "kicked user = '$2' and kicker='$1'\n";
  if( $kickeduser eq $botname ) 
  {
      #$kernel->post( $context, 'privmsg', $addressed,'Kay! Kay!' );
      $kernel->post( $context, 'privmsg', $channel, $this->getMesg("KAYKAY",$args));

  }#if( $1 eq $botname ) 
  else
  {
    $kernel->post( $context, 'privmsg', $channel,'fiii '.$kickeduser );
  }#else
  my $success = 0;
  my $theactors = $this->{"actors"};
  ##foreach $actor (keys(%$theactors))
  foreach my $actorname (keys(%$theactors))
  {
    my $actor = $theactors->{$actorname};
    my %subargs = %{$args};
    $success = $success || $actor->KICKaction(\%subargs);
  }#foreach $actor ($this->setting("actors"))
  #$kernel->post( 'mainframe','KICKaction', $subargs );
}# sub irc_kick
#########################################################
=pod

=item irc_part

react on some user leaving the channel

=cut

#########################################################
sub irc_part
{
  my ($this,$subargs) = @_[OBJECT,ARG0];
  $subargs->{"kernel"} = $_[KERNEL];
  #print "irc_part".Dumper(@_);
  print "zebot::irc_part ".$subargs->{"usernick"}."\n";
  my $success = 0;
  my $theactors = $this->{"actors"};
  foreach my $actorname (keys(%$theactors))
  {
    my $actor = $theactors->{$actorname};
    $success = $success || $actor->PARTaction($subargs);
  }#foreach $actor ($this->setting("actors"))
}
#########################################################
=pod

=item irc_quit

react on some user dropping out of the channel

=cut

#########################################################
sub irc_quit
{
  my ($this,$subargs) = @_[OBJECT,ARG0];
  $subargs->{"kernel"} = $_[KERNEL];
  print "zebot::irc_quit ".$subargs->{"usernick"}."\n";
  my $success = 0;
  my $theactors = $this->{"actors"};
  foreach my $actorname (keys(%$theactors))
  {
    my $actor = $theactors->{$actorname};
    $success = $success || $actor->PARTaction($subargs);
  }#foreach $actor ($this->setting("actors"))
}# sub irc_quit
#########################################################
=pod

=item irc_join

react on some user joining the channel

=cut

#########################################################
sub irc_join
{
  my ($this,$subargs) = @_[OBJECT,ARG0];
  $subargs->{"kernel"} = $_[KERNEL];
  my $success = 0;
  my $theactors = $this->{"actors"};
  foreach my $actorname (keys(%$theactors))
  {
    my $actor = $theactors->{$actorname};
    #print "trying to join for $actorname is a ".$actor->isa()."\n";
    $success = $success || $actor->JOINaction($subargs);
    #print "success = $success\n";
  }# foreach $actor (keys(%$theactors))
}# sub irc_join
#dcc_accept
# Accepts an incoming DCC connection from another host. First argu-
# ment: the magic cookie from an 'irc_dcc_request' event. In the case
# of a DCC GET, the second argument can optionally specify a new name
# for the destination file of the DCC transfer, instead of using the
# sender's name for it. (See the 'irc_dcc_request' section below for
# more details.)
#
#
#########################################################
=pod

=item irc_dcc_get

data was transmitted successfully

 Notifies you that another block of data has been successfully
 transferred from the client on the other end of your DCC GET con-
 nection.  ARG0 is the connection's magic cookie, ARG1 is the nick
 of the person on the other end, ARG2 is the port number, ARG3 is
 the filename, ARG4 is the total file size, and ARG5 is the number
 of bytes successfully transferred so far.

=cut

#########################################################
sub irc_dcc_get 
{
  my ($this,$subargs) = @_[OBJECT,ARG0];
  $subargs->{"kernel"} = $_[KERNEL];
  my $success = 0;
  my $theactors = $this->{"actors"};
  foreach my $actorname (keys(%$theactors))
  {
    my $actor = $theactors->{$actorname};
    $success = $success || $actor->DCCget($subargs);
  }#foreach $actor ($this->setting("actors"))
}# sub irc_dcc_get 
#########################################################
=pod

=item irc_dcc_chat

DCC chat stuff incoming

=cut

#########################################################
sub irc_dcc_chat 
{
  my ($this,$subargs) = @_[OBJECT,ARG0];
  $subargs->{"kernel"} = $_[KERNEL];
  my $success = 0;
  my $theactors = $this->{"actors"};
  foreach my $actorname (keys(%$theactors))
  {
    my $actor = $theactors->{$actorname};
    $success = $success || $actor->DCCchat($subargs);
  }#foreach $actor ($this->setting("actors"))
}# sub irc_dcc_chat 
#########################################################
=pod

=item irc_dcc_done

react on some dcc transfer success

=cut

#########################################################
sub irc_dcc_done 
{
  my ($this,$subargs) = @_[OBJECT,ARG0];
  $subargs->{"kernel"} = $_[KERNEL];
  print "irc_dcc_done: ";
  my $nick = $subargs->{"usernick"}; #kicker
  my $type = $subargs->{"type"}; #kicker
  my $file = $subargs->{"file"}; #kicker
  my $done = $subargs->{"done"}; #kicker

  print "DCC $type to $nick ($file) done: $done bytes transferred.\n";
  my $success = 0;
  my $theactors = $this->{"actors"};
  foreach my $actorname (keys(%$theactors))
  {
    my $actor = $theactors->{$actorname};
    $success = $success || $actor->DCCdone($subargs);
  }#foreach $actor ($this->setting("actors"))
}# sub irc_dcc_done 
#########################################################
=pod

=item irc_dcc_error

react on some dcc transfer fail

=cut

#########################################################
sub irc_dcc_error 
{
  my ($this,$subargs) = @_[OBJECT,ARG0];
  $subargs->{"kernel"} = $_[KERNEL];

  my $success = 0;
  my $theactors = $this->{"actors"};
  foreach my $actorname (keys(%$theactors))
  {
    my $actor = $theactors->{$actorname};
    $success = $success || $actor->DCCerror($subargs);
  }#foreach $actor ($this->setting("actors"))
}# sub irc_dcc_error 
#########################################################
=pod

=item irc_dcc_request

react on some dcc transfer/chat proposal

 You receive this event when another IRC client sends you a DCC SEND
 or CHAT request out of the blue. You can examine the request and
 decide whether or not to accept it here. ARG0 is the nick of the
 client on the other end. ARG1 is the type of DCC request (CHAT,
 SEND, etc.). ARG2 is the port number. ARG3 is a "magic cookie"
 argument, suitable for sending with 'dcc_accept' events to signify
 that you want to accept the connection (see the 'dcc_accept' docs).
 For DCC SEND and GET connections, ARG4 will be the filename, and
 ARG5 will be the file size.

=cut

#########################################################
sub irc_dcc_request 
{
  my ($this,$subargs) = @_[OBJECT,ARG0];
  $subargs->{"kernel"} = $_[KERNEL];

  my $success = 0;
  my $theactors = $this->{"actors"};
  foreach my $actorname (keys(%$theactors))
  {
    my $actor = $theactors->{$actorname};
    $success = $success || $actor->DCCrequest($subargs);
  }#foreach $actor ($this->setting("actors"))
}# sub irc_dcc_request 
#########################################################
=pod

=item setircserver

prepare a connection with the ircserver

=cut

#########################################################
sub setircserver
{
  my ($this,$kernel) = @_;
  #print "new connections\n";
  #sequential stuff
  #POE::Component::IRC->new( 'zebot' ) or
  #  print "Can't instantiate new IRC component!\n";
  #parallel server stuff
  my $serverData = $this->{"servers"};

  #instantiate the server links
  #print("firing up the server handelrs\n");
  print($this->getMesg("FIRESERVHAND")."\n");
  foreach my $name (keys(%$serverData))
  {
  print("firing up the server handler $name\n");
    print($this->connectServer($kernel,$name));
  }#foreach my $name (keys($this->{"servers"}))
  #print "started all irc server objects\n";
  print($this->getMesg("SERVINSTOK")."\n");

  #print "done add zebot handler\n";
}# sub setircserver
#########################################################
=pod

=item connectServer

use the data from the serverlist and the nickname for a given network to
connect to a server

=cut

#########################################################
sub connectServer
{
  my ($this,$kernel,$name,$mode) = @_;
  my $msg ;
  my $arg = { "target" => $name };
  if(!$name)
  {
    $msg .= $this->getMesg("You must specify a servername")."\n";
    return $msg; 
  }# if(!$name)

  if($this->{"handles"}->{$name})
  {
    #we are reconneecting to a server
    #just in case we were still connected, be nice and tell the server we want
    #to go away, will trigger a irc_disconnect event if we still were
    #connected...
    $kernel->post( $name, 'quit', 'Neenios on ice!' );
    #set the state of the irc handler to connect mode
    $this->{"handles"}->{$name}->{"connecting"} = 1;
    if($mode)
    {
      $this->{"handles"}->{$name}->{"autoconnect"} = 1;
    }# if($mode)
    #if we weren't connected, the the quit will yield nothing, here we force
    #the connection cycle, doesn't matter if we call this several times, since
    #it will be cancelled as soon as we are connected....
    $msg .= $this->getMesg("Reconnecting to target")."\n";
    $kernel->post( $name."-session", 'doConnect'); 
  }# if($this->{"handles"}->{$name}) 
  else 
  { 
    if($this->{"servers"}->{$name})
    {
      $msg .= $this->getMesg("INSTSERVHAND",$arg)."\n";
      $this->{"handles"}->{$name} = new zebot::ircServer(); 
      if($mode)
      {
	$this->{"handles"}->{$name}->{"autoconnect"} = 1;
      }# if($mode)
      $msg .= $this->getMesg("INITSERVHAND",$arg)."\n";
      $this->{"handles"}->{$name}->init($name, $this->{"servers"}->{$name},$this);
      $msg .= $this->getMesg("SERVINSTOK")."\n"; 
    }# if($this->{"servers"}->{$name})
    else
    {
      $msg .= $this->getMesg("There is no template for server",$arg)."\n"; 
    }# else
  }# else
  return $msg;
  #print "done add zebot handler\n";
}# sub connectServer

#########################################################
=pod

=item setstreamName

open the log file

=cut

#########################################################
sub setstreamName
{
  my ($this) = @_;
  open (STREAM, ">>".$this->setting("streamName")) || 
  die("couldn't open ".$this->setting("streamName"));
  my $oldfh = select(STREAM);
  $| = 1;
  select($oldfh);
  print STREAM "starting NEW log\n";
}
#########################################################
=pod

=item parseCmdlineArgument

parse in the arguments from the command line, and store them in a temporary
structure, we need to do this, since some of the settings could affect the way
the settings from files are read... and since we want command line args to
override file set parms, we need to split this in 2 stages....

TODO i made this before beeing aware of GetOpt this should be used instead of my code here.... especially since it could be added online explanation of the settings, on the other side i didn't find out on the fast way how to preparse 

=cut

#########################################################
sub parseCmdlineArgument 
{
  my ($this) = @_;
  $this->{"tmpsettings"} = {} if(!$this->{"tmpsettings"});

  return if(!@ARGV);
  my $switch = shift(@ARGV);
  #$switch = quotemeta($switch);
  #$switch =~ s/\\-/-/g;
  if($switch =~ /^--/)
  {
    $switch =~ s/^--//;
    $switch = lc($switch);
    #its a switch!
    my $value = shift(@ARGV);
    #$value = quotemeta($value);
    #$value =~ s/\\-/-/g;

    if($value =~ /^--/)
    {
      #hmmm a noargtype switch
      unshift(@ARGV,$value);
      $value = 1;
      #check if its a no-switch
      if($switch =~ /^no/)
      {
	$switch =~ s/^no//;
	$value = 0;
      }# if($switch =~ /^no/)
    }# if($value =~ /^--/)
    $this->{"tmpsettings"}->{$switch} = $value;
  }# if($switch =~ /^--/)
  else
  {
    print("having trouble parsing $switch\n");
  }# else
  #print("type of args = ".ref($args)."\nvalue:$args\n");
  $this->parseCmdlineArgument();
}#sub parseCmdlineArgument 
#########################################################
=pod

=item loadSettings

load form the conf file the different bot settings means using the setting
configpath all files ending with conf (and as soon as i figured out how to
setup the same stuff in XML, also ending with .xml) the config directory is
scanned for conf files and read in and added to the setings of the bot. 

TODO, leave the non loaded actors settings away...

=cut

#########################################################
sub loadSettings 
{
  my ($this) = @_;
  my $configs = new zebot::configHandler($this->{"settings"});
  $configs->init();
  #print("loading settings from ".$this->{"settings"}->{"configpath"}."\n");

  $configs->loadSettings($this->{"settings"}->{"configpath"});

  #print( "laodSettings : ".Dumper($this->{"settings"})."\n" );
  $this->{"servers"} = $this->{"settings"}->{"subsettings"}->{"servers"};
  delete $this->{"settings"}->{"subsettings"}->{"servers"};
  $this->{"actornames"} = $this->{"settings"}->{"subsettings"}->{"actors"};
  delete $this->{"settings"}->{"subsettings"}->{"actors"};
  #print("dumping1 actors ".Dumper($this->{"actornames"})."\n");

  if($_[KERNEL])
  {
    my $line = $_[ARG0];
    $_[KERNEL]->post( $line->{"context"}, 'privmsg',$line->{"usernick"},"loaded settings ok");
  }# if($_[KERNEL])

}# sub loadSettings 
#########################################################
=pod

=item dumpSettings 

issue the actual settings of the bot

=cut

#########################################################
sub dumpSettings 
{
  my ($this,$hash) = @_;
  my $i;
  my $listing = "";
  $hash = $this->{"settings"} if(!$hash);

  foreach $i (sort(keys(%{$hash})))
  {
    my $argl = $hash->{$i};
    $listing .= $i." ".$hash->{$i}."\n" if (ref(\$argl) eq "SCALAR") ;
  }# foreach $i (keys(%hash))
  #TODO and the server lists?
  return $listing;
}
#########################################################
=pod

=item saveSettings 

save actual bot settings to the conf file

=cut

#########################################################
sub saveSettings 
{
  my ($this) = @_;
  my $basepath = $this->{"settings"}->{"configpath"};
  $basepath .= "config/";

  my $path = $basepath."zebot.conf";
  open (SETTINGS, ">$path") || die("couldn't open $path");
  print SETTINGS $this->dumpSettings($this->{"settings"});
  close (SETTINGS);
  #TODO and the server lists?


  my @subgroups = keys(%{$this->{"settings"}->{"subsettings"}});
  #be sure not to override the servers conf as long as we have this system...
  my $tmp = join(' ',@subgroups);
  $tmp =~ s/servers//g;
  @subgroups = split(/ /,$tmp);
  foreach my $name (@subgroups)
  {
    $path = $basepath."$name.conf";

    open (SETTINGS, ">$path") || die("couldn't open $path");
    my $dump = $this->dumpSettings($this->{"settings"}->{"subsettings"}->{$name});
    print SETTINGS "$dump\n";
    close (SETTINGS);
  }# foreach my $name (@actornames)
}
#########################################################
=pod

=item loadActor

load the object definitions of eventual external modules

=cut

#########################################################
sub loadActors
{
  my($this,$line,$kernel) = @_[OBJECT,ARG0,KERNEL];
  #print("zebot::LOADACTORS method\n");

  #empty actors
  $this->{"actors"} = {};

  #some shortcuts to the array of actornames, settings and actor instances
  # eval("use zebot::baseactor");
  my $actors = $this->{"actors"};

  foreach my $file (keys(%{$this->{"actornames"}}))
  {
    #print"doing $file.pm\n";
    #do $file.".pm";
    #TODO don't know how to handle this correctly... i want to forcibly reload the class when loading/reloading the actors, but i don't want to load them twice either....
    #print"using $file.pm\n";
    eval("use zebot::$file");
    unless( exists $zebot::{$file.'::'})
    {
      print "couldn't find lib $file\n";
      print "trying do....\n";
      eval("do zebot::$file");
      print "no way to load lib $file\n" unless exists $zebot::{$file.'::'};
    }
    #print"instantiating $file\n";
    my $cmd = 'new zebot::'.$file.'()';
    #print "trying to eval $cmd\n";
    my $newObj = eval($cmd);
    my $subsettings = $this->{"settings"}->{"subsettings"}->{"actors"};
    if($subsettings->{$file})
    {
      $this->print("Su_BSETTING ARGS FOR $file with :".Dumper($subsettings->{$file}."\n"));
      $newObj->setArgument($subsettings->{$file});
    }# if($subsettings->{"actors"}->{$file})
    #my $newObj = new mailbox() if( $file eq "mailbox" );
    if($newObj)
    {
      #$newObj->init($this);
      $actors->{$newObj->isa()} = $newObj;
      #print "loaded ".$newObj->isa()."\n";
    }#if($newObj)
    else 
    {
      print "failed $file!!\n";
    }#else 

  }#foreach $file ($this->setting("actors"))

  foreach my $i (keys(%$actors))
  {
    $actors->{$i}->init($this);
  }#foreach $i (keys(%$this->{"actors"}))

  foreach my $i ("pam", "messageHandler")
  {
    if(!($actors->{$i}))
    {
      print("vital module $i missing!! ensure it is represented in actors.conf!\n");
      exit(0);
    }# if(!($actors->{$i}))
  }# foreach my $i (["pam", "messageHandler")
  if($kernel)
  {
    my $usernick = $line->{"usernick"};

    #in case we are reloading we need to simulate a new connection to the
    #channels we are connected to simulate the fact that we rejoined, otherwise
    #the modules won#t be aware we are connected.

    if($line && $line->{"reloading"})
    {
      #print("RELOADING :args=".Dumper(keys(%$line))."\n");
      print("RELOAD parsing now handlers \n");
      foreach my $net (keys(%{$this->{"handles"}}))
      {
	my $handler = $this->{"handles"}->{$net};

	if($handler->{"autoconnect"})
	{
	  $line->{"heap"} = $handler->{"heap"};
	  print("RELOAD simulating join for ".$line->{"heap"}->{"network"}." of ".$handler->{"name"}."\n");
	  foreach my $channel (@{$handler->{"channel"}})
	  {
	    print("sending join success $channel@".$line->{"heap"}->{"network"}."\n");
	    my %tmp = %$line;
	    $tmp{"channel"} = $channel;
	    $kernel->post('mainframe','channel_join_success',\%tmp);
	  }# foreach my $channels (keys(%{$handler->{"channels"}}))
	}# if($handler->{"autoconnect"})
	else
	{
	  print("RELOAD aborting for ".$line->{"heap"}->{"network"}." of ".$handler->{"name"}." : no autoconnect \n");
	}# else

      }# foreach my $net (keys(%{$this->{"handles"}->{$name}}))
    }# if($line->{"reloading"})

    $kernel->post( $line->{"context"}, 'privmsg',$usernick,"loaded actors ok");
    $kernel->post( $line->{"context"}, 'privmsg',$usernick,'reload complete');
    print "load actors done \n";
  }# if($_[KERNEL])
}# sub loadActors
#########################################################
=pod

=item dropActor

drop the object definitions of eventual external modules

=cut

#########################################################
sub dropActors
{
  my($this,$kernel) = @_[OBJECT,KERNEL];
  #In case there were allready some loaded actors, try to unload them
  #print("zebot::LOADACTORS method\n");
  #print("To unload : ".Dumper($this->{"actors"})."\n");
  foreach my $i (keys(%{$this->{"actors"}}))
  {
    my $file = $this->{"actors"}->{$i}->isa();
    delete $this->{"actors"}->{$i};

    my $package = "zebot::$file";
    print("unload zebot  $i : $file\n");
    eval("no zebot::$file");
    #eval("unimport zebot::$file");
    $package .= "::";

    delete_package($package);
    #eval("delete $package");
    print "deleted\n" unless exists $zebot::{$file.'::'};
  }#foreach $i (keys(%$this->{"actors"}))
  my $package = "zebot::baseactor";
  eval("no $package");
  $package .= "::";
  eval("delete $package");
  delete_package($package);
  print "deleted\n" unless exists $zebot::{$package};
  #print "deleted baseactor\n" unless exists $zebot::{'baseactor::'};
  ##eval("use $package");
  print("Done unloading \n");
}# sub dropActors
#########################################################
=pod

=item closeActor

let the different modules come to a gracious end

=cut

#########################################################
sub closeActors
{
  my($this,$kernel,$line) = @_[OBJECT,KERNEL,ARG0];
    foreach my $actorname (keys(%{$this->{"actors"}}))
    {
      my $actor = $this->{"actors"}->{$actorname};
      $kernel->post( $line->{"context"}, 'privmsg',$line->{"usernick"},"shutdown for $actorname");
      if($actor->session)
      {
        $kernel->post( $actor->session, 'shutdown',$line);
      print "post shutdown to $actorname to ".$actor->session."\n";
      }# if($botref->{"actors"}->{$actorname}->session)
      else
      {
	$actor->shutdown($line);
      print "shutdown for $actorname\n";
      }
    }#foreach $actor ($botref->setting("actors"))
    $kernel->post( $line->{"context"}, 'privmsg',$line->{"usernick"},"shutting down modules ok");
}# sub closeActors
#########################################################
=pod

=item reloadActors

conveninence method to tell the system to close down the modules, purge them
from the system, reload them and initialize them

=cut

#########################################################
sub reloadActors
{
  my($this,$kernel,$line) = @_[OBJECT,KERNEL,ARG0];

  $kernel->yield('closeActors',$line);
  $kernel->delay_add('dropActors',2,$line);
  $kernel->delay_add('loadSettings',2,$line);
  $line->{"reloading"} = 1;
  $kernel->delay_add('loadActors',2,$line);
}# sub reloadActors
#########################################################
=pod

=item getMesg

fetch the message handler and ask it for the given translation of a token into a localized string
 this method takes the arguments in the following order:  
     language,
     token to be translated,
     the mode of the actual message
     a default string in case the token couldn't be found

=cut

#########################################################
sub getMesg
{
  my($this,$token,$line) = @_;
  return "toto" if(!$token);
  $line->{"lang"} = $this->setting("language") if(!($line->{"lang"}));
  my $Msg = $this->module("messageHandler");

  my $msg = $token;
  if($Msg)
  {
  $msg = $Msg->getMesg($token,$line);
  }# if($Msg)
  else
  {
    $msg = "ZEBOT AYEEEEH no messagehandler : $msg\n";
  }# else
  return $msg;
}# sub getMesg
#########################################################
=pod

=item setting 

setter/getter for the settings, to be able to choose any format for those
settings without disturbing the rest of the program...

=cut

#########################################################
sub setting 
{
  my ($this,$key,$value) = @_;

  if(!$key)
  {
    #make a dump of the settings
    my $msg = "Settings of Mainframe\n";
    foreach $key (sort(keys(%{$this->{"settings"}})))
    {
      $msg .= "$key: ".$this->{"settings"}->{$key}."\n" 
	if(ref($this->{"settings"}->{$key}) eq ""); 
    }# foreach $key (keys(%$this))
    $msg .= "End of listing\n";
    return($msg);
  }# if(!$key)

  if(!$value)
  {
    return $this->{"settings"}->{$key};
  }# if ($key)
  $this->{"settings"}->{$key} = $value;
}#setting
#########################################################
=pod

=item irc_invite 

got invited by someone..

=cut

#########################################################
sub irc_invite 
{
  my ($this,$subargs) = @_[OBJECT,ARG0];
  $subargs->{"kernel"} = $_[KERNEL];
  print "irc_invite";
}
#########################################################
=pod

=item irc_ping 

got invited by someone..

=cut

#########################################################
sub irc_ping 
{
  my ($this,$subargs) = @_[OBJECT,ARG0];
  $subargs->{"kernel"} = $_[KERNEL];
  #print "irc_ping\n";
}
#########################################################
=pod

=item channel_namelist 

got invited by someone..

=cut

#########################################################
sub channel_namelist 
{
  my ($this,$subargs) = @_[OBJECT,ARG0];
  $subargs->{"kernel"} = $_[KERNEL];
  print ("irc_channel_namelist\n");
}
#########################################################
=pod

=item language

fetch the language for a given channel, otherwise return the list of installed
channels and languages

in otherterms setter and getter for the language...

=cut

#########################################################
sub language
{
  my ($this,$channel,$lang) = @_;
  my $languageRef = $this->{"channelLangs"};
  if($channel)
  {
    $channel = lc($channel);
    if($lang)
    {
      $languageRef->{$channel} = $lang;
    }# if($lang)
    return($languageRef->{$channel}); 
  }# if($arg)

  my $msg;
  foreach my $i (keys(%$languageRef))
  {
    $msg .= "$i ->".$languageRef->{$i}."\n";
  }#foreach $i (keys(%$this->{"actors"}))
  $msg .= "...X\n";
  return $msg;
}# sub language

1
__END__


=back

=head1 THANKS

the french linux journal and J-M Libs for the impulsion to write this thing

the guys from the perl usenet group for their steady help

the guys from the POE list and especially R. Caputo for their intensive help about all this POE stuff.

and last but not least Mark Fowler who helped me with his own bot to understand and isolate the parts of POE::IRC that needed to be spawned on their own to make this bot multi server aware.

=head1 AUTHOR

Actually i am the only developer and user of this bot, seems there are too much
projects out there with irc bots, so that mine doesn't look appealing enough
for people to come even visit its project page :D but who cares, in a not so
far future this bot will have gained a complete peer-to-peer filetransfer
ability, and will be IMHO an absolute killer app for people wanting to share
and download stuff from irc channels.

Bruno Bttcher <bboett at adlp.org>

=head1 SEE ALSO

  zebot home page  http://www.freesoftware.fsf.org/zebot/ 
  Net::IRC, 
  RFC 1459,
  http://www.irchelp.org/, 
  http://poe.perl.org/
  http://www.cs.cmu.edu/~lenzo/perl/, 
  http://www.infobot.org/,


=cut

