#!/usr/bin/perl
##############################################################################
=pod

=head1 NAME

 zebot::DB::DBspawner

=head1 DESCRIPTION

zebot::DB::DBspawner module, fires up a number of worker processes that connect to the database, and then forks off with the incoming tasks.


=head1 COPYRIGHT and LICENCE

  Copyright (c) 2002 Bruno Boettcher

  DB::DBspawner.pm is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; version 2
  of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

=head1 Methods of this class

=over

=cut

##############################################################################
package zebot::DB::DBspawner;
use strict;

use POSIX qw(strftime);
use Data::Dumper;

use POE::Session;
use POE::Wheel::Run;
use POE::Filter::Reference;

use zebot::DB::Helper;

sub MAX_CONCURRENT_TASKS () { 3 }


#our ($statements,$stindex);


##############################################################################
=pod

=item CTOR

instantiation of the emoteparser

=cut

##############################################################################
sub new {
  my ($class,$name) = @_; 
  my $this = bless {}, "zebot::DB::DBspawner";
  $this->{"name"} = $name;
  #print("SPAWNER CTOR incoming args: ".Dumper(\@_)."\n");
  return $this;
}
##############################################################################
=pod

=item init

Initiaize, means set up this module

=cut

######################################################################
sub init
{
  my ($this, $botref,$args) = @_;
  $this->sysdata($botref);
  $this->{"queries"} = {};
  $this->{"dbsettings"} = $args;

# Start the session that will manage all the children.  The _start and
# next_task events are handled by the same function.

  POE::Session->create
    ( object_states =>
      [
      $this =>
      { _start => "_start",
      _stop => "_stop",
      doQuery   => "doQuery",
      batch   => "batch",
      task_result => "handle_task_result",
      task_done   => "handle_task_done",
      task_debug  => "handle_task_debug",
      "shutdown"  => "shutdown",
      }
  ]
    );

}#sub init
######################################################################
=pod

=item _start

start up to  MAX_CONCURRENT_TASKS workers, make the connect to the database and store theyre references for later use. 

we use 3 stacks here:
the stack of DBI helpers
the stack of waiting requests
the stack of running forks

the resultAsReference is set automaticly for the DBI helpers, to signalize to
send the data as storable objects instead of plain references

=cut

######################################################################
sub _start 
{
  my ($this,$heap,$kernel) = @_[OBJECT,HEAP,KERNEL];
  $kernel->alias_set( $this->{"name"} );
  #$this->print("DBspawner set alias to ".$this->{"name"});
  #my $settings = $this->sysdata()->{"settings"};
  my $opts = $this->{"dbsettings"};
  $this->{"workers"} = [];
  $this->{"batch"} = [];
  $this->{"tasks"} = {};
  my $workers = $this->{"workers"};

  for ( my $i = 0; $i < MAX_CONCURRENT_TASKS; $i++ ) 
  {
    my $agent = new zebot::DB::Helper();
    #print("starting a spawner with ".$opts->setting("database")." ".$opts->setting("dbuser")." ".$opts->setting("dbpasswd")."\n");
    $agent->init($opts->setting("database"), $opts->setting("dbuser"), $opts->setting("dbpasswd"));
    #print("init ok\n");
    $this->{"agents"}->{"Helper$i"}  = $agent;
    $agent->{"name"} = "Helper$i";
    $agent->{"resultAsReference"} = 1;
    push(@$workers, $agent);
  }#for ( my $i = 0; $i < MAX_CONCURRENT_TASKS; $i++ ) 
}#sub _start 
######################################################################
=pod

=item doQuery

can be invoqued with a request or without.  If no request is given, a try is
made to fetch one from the stack of waiting ones....

otherwise this takes as arguments the query object, the session the results
should be posted back and the event that should be invoqued on that session.

if a helper is available a fork is called adn the data transmitted to the forked off code, otherwise the request is stored.

=cut

######################################################################
sub doQuery 
{
  my ($this,$heap, $query) = @_[OBJECT,HEAP, ARG0];

  my $workers = $this->{"workers"};
  my $tasks = $this->{"tasks"};
  my $batch = $this->{"batch"};
  if(ref($query) ne 'HASH')
  {
    $this->print("oyoyoy what did you send me?? ".Dumper($query)."\n");
  }# if(ref($query) ne 'HASH')
  #$this->print("doQuery!!!! in session: ".$_[KERNEL]->alias_list($_[SESSION]));
  if(scalar(@$workers) > 0)
  {
    #$this->print("enough workers!!!! !");
    #we have some workers avaiable for the given task
    my $actor = shift(@$workers);
    if(!$query)
    {
      my $stored = shift(@$batch);
      if($stored)
      {
	$query = $stored->{"query"};
      }# if($stored)
    }# if(!$query)

    if($query)
    {
      $actor->{"session"} = $query->{"session"};
      $actor->{"event"} = $query->{"event"};
      #$this->print("casting off thread!");
      #remove eventually strange stuff that won't pass the Reference
      delete $query->{"kernel"};
      my $task = POE::Wheel::Run->new
	( Program => sub { $actor->doQuery($query) },
	  StdoutFilter => POE::Filter::Reference->new(),
	  StdoutEvent  => "task_result",
	  StderrEvent  => "task_debug",
	  CloseEvent   => "task_done",
	);

      $tasks->{ $task->ID } = {"wheel" => $task, "actor" => $actor };
    }# if($query)
  }# if(scalar(@$workers) > 0)
  else
  {
    #store the request for upto when a query is ready
    my ($this,$heap, $query, $pbsession, $pbevent) = @_[OBJECT,HEAP, ARG0 .. ARG2];
    push(@$batch, { "type" => "query", "query" => $query, });
  }# else
  #$this->print("done doQuery");
}# sub doQuery 
######################################################################
=pod

=item batch

here a whole transaction is send in one go to the helper, since it is forked
off, no splitting of the job is required...

=cut

######################################################################
sub batch 
{
  my ($this,$heap, $job) = @_[OBJECT,HEAP, ARG0];
  my $query = $job->{"query"};
  my $pbsession = $job->{"session"};
  my $pbevent = $job->{"event"};

  my $workers = $this->{"workers"};
  my $tasks = $this->{"tasks"};
  my $batch = $this->{"batch"};
  if(scalar(@$workers) > 0)
  {
    #we have some workers avaiable for the given task
    my $actor = shift(@$workers);

    if(!$query)
    {
      my $stored = shift(@$batch);
      if($stored)
      {
	$query = $stored->{"query"};
	$pbsession = $stored->{"session"};
	$pbevent = $stored->{"event"};
      }# if($stored)
    }# if(!$query)

    if($query)
    {
      $actor->{"session"} = $pbsession;
      $actor->{"event"} = $pbevent;
      #remove binary stuff
      delete $job->{"kernel"};
      #print("sending out  batch: ".Dumper( $job)."\n");
      #print("sending out  batch \n");
      #$this->print("query valid sending out  batch".Dumper($job)." \n");
      my $task = POE::Wheel::Run->new
	( Program => sub { 
	  $actor->batch($job) 
	  },
	  StdoutFilter => POE::Filter::Reference->new(),
	  StdoutEvent  => "task_result",
	  StderrEvent  => "task_debug",
	  CloseEvent   => "task_done",
	  );
	  #print("sent out  batch: ".$job->{"debug"}." \n");

	  $tasks->{ $task->ID } = {"wheel" => $task, "actor" => $actor };
    }#if($query)
  }# if(scalar(@$workers) > 0)
  else
  {
    #store the request for upto when a query is ready
    my ($this,$heap, $query, $pbsession, $pbevent) = @_[OBJECT,HEAP, ARG0 .. ARG2];
    push(@$batch, { "type" => "batch", "query" => $query, 
	"session" => $pbsession, "event" => $pbevent});
  }# else
}# sub batch 
######################################################################
=pod

=item handle_task_result

Handle information returned from the task.  Since we're using
POE::Filter::Reference, the $result is however it was created in the
child process.  its a hash reference following the format requested in DB::Helper

=cut

######################################################################
sub handle_task_result 
{
  my ($this,$kernel,$result) = @_[OBJECT,KERNEL,ARG0];
  #$this->print("Result for $result->{session} with $result->{event} and debug: $result->{debug}\n");
  $kernel->post($result->{"session"}, $result->{"event"}, $result);
}# sub handle_task_result 
######################################################################
=pod

=item handle_task_debug

Catch and display information from the child's STDERR.  This was
useful for debugging since the child's warnings and errors were not
being displayed otherwise.

=cut

######################################################################
sub handle_task_debug 
{
  my $result = $_[ARG0];
  #print "Debug: $result\n";
}
######################################################################
=pod

=item handle_task_done

A job completed, push the returning helper onto the stack of available workers, check for waiting requests and eventuall fire off an doQuery or batch event to start their processing.

=cut

######################################################################
sub handle_task_done 
{
  my ($this, $kernel, $heap, $task_id ) = @_[OBJECT, KERNEL, HEAP, ARG0 ];

  my $workers = $this->{"workers"};
  my $tasks = $this->{"tasks"};
  my $batch = $this->{"batch"};

  my $actor = $tasks->{$task_id}->{"actor"};
  #$this->print("Result for ".Dumper($task_id)."\n");

  #$this->print( "taskdone for ".$actor->{"name"});
  delete   $tasks->{$task_id};
  push(@$workers,$actor);
  #my $actor = $tasks->{$task_id} = {"wheel" => $task, "actor" => $actor };
  #push back the worker into the pile of available ones
  if((scalar(@$workers) > 0) && (scalar(@$batch) > 0))
  {
    #seems we have some liberties...
    my $nextjob = $batch->[0];
    my $jobtype = $nextjob->{"type"};
    if($jobtype eq "query")
    {
      $kernel->yield("doQuery");
    }# if($jobtype eq "query")
    elsif($jobtype eq "batch")
    {
      $kernel->yield("batch");
    }# elsif($jobtype eq "batch")
  }# if((scalar(@$workers) > 0) && (scalar(@$batch) > 0))
}# sub handle_task_done 
######################################################################
=pod

=item sysdata

attribute getter previously provided by ObjecTemplate

=cut

######################################################################
sub sysdata
{
  my($this,$lsysdata) = @_;
  if($lsysdata)
  {
    $this->{"sysdata"} = $lsysdata;
  }
  return $this->{"sysdata"};
}# sub sysdata
#########################################################
=pod

=item print

delegator, to make a copy of the console stream to a file, for debugging
purposes only, since somehow the STDOUT gets swallowed up for some cases....

=cut

#########################################################
sub print
{
  my ($this,$msg) = @_;
  $msg = $this if(!$msg);
  if(open(FILE,">>logger"))
  {
    print FILE ("$msg\n");
    close(FILE);
  }# if(open(FILE,">logger"))
  print("$msg\n");
}# sub$this->print
######################################################################
=pod

=item shutdown

close down the activity for a safe shutdown

=cut

######################################################################
sub shutdown
{
  my ($this,$heap,$kernel) = @_[OBJECT,HEAP,KERNEL];

  foreach my $agent (keys(%{$this->{"agents"}}))
  {
    $this->{"agents"}->{$agent}->shutdown();;
  }# foreach my $agent (keys(@{$this->{"agents"}}))
   if($kernel)
   {
     my @aliases = $kernel->alias_list();
     #print("available aliases...:".Dumper(\@aliases)."\n");
     if(scalar(@aliases) >0)
     {
     $kernel->alias_remove( $this->{"name"} );
        #if($kernel->alias_resolve( $this->{"name"} ));
     }# if(scalar(@aliases) >0)
   }
   else
   {
    print("couldn't shut down the DBspawner.... no kernel...\n"); 
   }# else
}#sub shutdown
######################################################################
=pod

=item _stop

close down this thing

=cut

######################################################################
sub _stop
{
  my ($this,$dbh_id, $errtype, $errstr, $err) = @_[OBJECT,ARG0..ARG3];
  $this->print("DBspawner _STOP!!!!!");
  foreach my $agent (keys(%{$this->{"agents"}}))
  {
    $this->{"agents"}->{$agent}->shutdown();;
  }# foreach my $agent (keys(@{$this->{"agents"}}))
  #$_[KERNEL]->alias_remove( $this->{"name"} );
  #exit(0);
}#sub _stop
######################################################################
=pod

=item prepare

prepare a sql statement for further refinement
its a dirty hack at the moment, since if uses either a free Helper, or the last (still working) one from the list of helpers.... doesn't seem to be a problem at the moment, but htere's surely a more elegant way to do this ...

=cut

######################################################################
sub prepare
{
  my ($this, $sql_query) = @_;
  #$this->{"agents"}->{"Helper$i"}  = $agent;
  my $actor;
  my $workers = $this->{"workers"};
  if((scalar(@$workers) > 0))
  {
    $actor = $workers->[0];
  }# if((scalar(@$workers) > 0))
  else
  {
    #TODO: DIRTY HACK... use one of the allready working handlers...
    $actor = $this->{"agents"}->{"Helper".(MAX_CONCURRENT_TASKS()-1)};
  }# else
  #$this->print("using actor $actor\n");
  my $prepared =  $actor->prepare($sql_query);
  #$this->print("PREPARED ".Dumper($prepared)."\n");
  #$this->print("PREPARED $prepared\n");
  return $prepared;
}# sub prepare
1
__END__

=back

=head1 AUTHOR

Bruno Bttcher <bboett at adlp.org>

=head1 SEE ALSO

zebot home page  http://www.freesoftware.fsf.org/zebot/ 
POD documentation of zebot

=cut

