package zebot::ircServer;
use strict;
##############################################################################
=pod

=head1 NAME

ircServer

=head1 DESCRIPTION

This  module is a helper for the zebot project, it represents the link to exactly one server, which can in retrun handle several channel connections.

it has only minimal signal support, using objects are thought to register themselves as listeners for this object and process themselves the incoming data

=head1 COPYRIGHT and LICENCE

Copyright (c) 2002 Bruno Boettcher

  zebot::ircServer is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; version 2
  of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


=head1 Methods of this class

=over

=cut

##############################################################################

use IO::Socket;
use Data::Dumper;
#use ObjectTemplate;
#use Net::IRC;
# sub POE::Kernel::ASSERT_DEFAULT () { 1 }
# sub POE::Kernel::ASSERT_PONY () { 1 }
# sub POE::Kernel::ASSERT_EVENTS () { 1 }
# sub POE::Kernel::ASSERT_STATES () { 1 }

use POE::Kernel;
use POE::Session;
use POE::Component::IRC;

#@ISA = ("ObjectTemplate");
#attributes("sysdata");


######################################################################
=pod

=item new 

constructor for the new irc-server handler object

=cut

######################################################################
sub new
{
  my ($this) = @_;
  my $o_FserverHandler = {

    "Status" 		=> undef,		# Our current fin. aut. state for nick
  };

  bless $o_FserverHandler, "zebot::ircServer";
  return $o_FserverHandler;
}# sub new
#########################################################
=pod

=item init

prepare a connection with the ircserver

=cut

#########################################################
sub init
{
  my ($this,$name,$data,$botref) = @_;
  #print("initializing this with \n".Dumper($this)."\nname=$name\ndata=".Dumper($data)."\nbot:\n".Dumper($botref));
  #print("setting up irc server handler $name for channels ".@{$data->{"channel"}});
  $this->{"name"} = $name;
  $this->{"zebot"} = $botref if( $botref);
  $this->{"channelLangs"} = {};
  #print("setting up irc server handler $name for channels ".Dumper($data->{"channel"}));
  my $arg = { "target" => $name, "line" => join(',',@{$data->{"channel"}})};
  print($botref->getMesg("SETIRCSERV",$arg)."\n");

  foreach my $record (keys(%$data))
  {
    $this->{$record} = $data->{$record};
  }# foreach my $record (keys(%$data))

  #add the active chans for this server to the global list
  #$botref->{"activeChans"} = "" if(!($botref->{"activeChans"}));
  $botref->{"activeChans"} .= " ".$this->{"activeChans"};
  $botref->{"activeChans"} =~ s/  / /gi;

  my @Chans = split (/,/,$this->{"activeChans"});

  foreach my $canal (@Chans)
  {
    $canal =~ /(.*?)%(.*?)%/;
    $canal = $1;
    my $lang = $2;
    $botref->{"channelLangs"}->{"$canal"} = $lang;
    $this->{"channelLangs"}->{lc($canal)} = $lang;
    #$heap->{"channels"}->{"$canal"} = $lang;
    #print("SET lang $lang as default lang of $canal: dumping channel languages: ".Dumper($botref->{"channelLangs"}));

  }# foreach my $canal (@Chans)

  $this->{"zebot"} = $botref;
  my $server = $this->{"server"};

  POE::Component::IRC->new( $name ) or
    print($botref->getMesg("ERRNOIRCINST")."\n");
    #print "Can't instantiate new IRC component!\n";
  #print "created irc component for $server\n";
  $arg = {"target" => $server};
  print($botref->getMesg("CREATEIRCH",$arg)."\n");
  #########################################################
  # now add to this connection all sort of different handlers
  #########################################################

  #print("creating SESSION!!\n");
  #POE::Session->new( $name =>
POE::Session->create( object_states => [ 
  
  $this =>
                   {  _start => "_start",
		      _stop  => "_stop",
		      'irc_disconnected' => "irc_disconnected",
		      irc_error => "irc_disconnected",
		      irc_001 => "network_connected",
		      irc_connected => "irc_connected",
		      doConnect => "doConnect",
		      irc_socketerr => "irc_socketerr",
		      rejoin => "rejoin",
		      irc_353	=> "channel_namelist",
		      irc_433	=> "irc_nick_taken",
                      irc_kick => "irc_kick",
		      irc_366	=> "channel_join_success",
		      irc_ping	=> "irc_ping",
		      keep_connected	=> "keep_connected",
		      irc_invite	=> "irc_invite",
                      irc_public => "irc_public",
		      irc_quit => "irc_quit",
		      irc_part => "irc_part",
		      irc_mode => "irc_mode",
		      irc_join => "irc_join", 
		      irc_msg => "irc_msg",
		      irc_notice => "irc_notice",
		      irc_ctcp_ping => "irc_ctcp_ping",
		      irc_ctcpreply_ping => "irc_ctcpreply_ping",
		      irc_ctcp_action => "irc_ctcp_action",
		      irc_ctcp_version => "irc_ctcp_version",
		      irc_332	=> "channel_join_topic",
		      irc_dcc_done => "irc_dcc_done",
		      irc_dcc_error => "irc_dcc_error",
		      irc_dcc_request => "irc_dcc_request",
		      irc_dcc_get => "irc_dcc_get",
		      irc_dcc_chat => "irc_dcc_chat",
		      },
  $botref =>
                   {  _default => "_default",
                   }
  ]
                 );
  #print "done add handler for $server\n";
  print($botref->getMesg("OKADDIRCHAND",$arg)."\n");
}# sub init
#########################################################
=pod

=item _start

start up operation concerning this irc server

=cut

#########################################################
sub _start 
{
  my ($this, $kernel, $session,$context, $network, $heap) = 
    @_[OBJECT, KERNEL, SESSION, SENDER, ARG0, HEAP];
  #my ($this) = @_;

  #print $this->{"name"}." starting session for network $network\n";
  #print $this->{"name"}." starting session for network\n";
  #print("_start of ".$this->{"name"}." initializing heap: ".Dumper($heap)."\n");
  $heap->{"network"} = $this->{"name"};
  $heap->{"rejoins"} = {};
  #keep a reference on this session heap needed in reload
  $this->{"heap"} =  $heap;
  #print("initializing heap done: ".Dumper($heap)."\n");

  foreach my $canal (keys(%{$this->{"channelLangs"}}))
  {
    $heap->{"channels"}->{"$canal"} = $this->{"channelLangs"}->{"$canal"};
    #print("SET lang $lang as default lang of $canal: dumping channel languages: ".Dumper($botref->{"channelLangs"}));

  }# foreach my $canal (@Chans)
  # Uncomment this to turn on more verbose POE debugging information.
  # $session->option( trace => 1 );

  # Make an alias for our session, to keep it from getting GC'ed.
  #print $this->{"name"}." setting alias to ".$this->{"name"}."\n";
  $kernel->alias_set( $this->{"name"}."-session" );
  #$kernel->alias_set( ($this->{"name"}."_slave") );

  # Ask the IRC component to send us all IRC events it receives. This
  # is the easy, indiscriminate way to do it.
  #$kernel->post( $context, 'register', 'all');
  $kernel->post( $this->{"name"}, 'register', 'all');
  #print $this->{"name"}." registered all on ".$context."\n";
  #
  # # Setting Debug to 1 causes P::C::IRC to print all raw lines of text
  # # sent to and received from the IRC server. Very useful for debugging.
  # 
  # 
  #print "attempting connection\n";
  #$kernel->post( $context, 'connect',   Debug    => 1,
  $heap->{"nick"} = $this->{"nick"};
  $heap->{"botname"} = $this->{"nick"};

  if(!($this->{"autoconnect"}))
  {
    #print("Net ".$this->{"name"}." not autoconnect\n");
   return; 
  }# if(!($this->setting("autoconnect")))
      my $args = {
      "kernel" => $kernel,
      };
      $this->{"connecting"} = 1;
      $kernel->delay_add( "doConnect", 1, $args);
      $kernel->delay_add( "keep_connected", ($this->setting("pingtmout"))*60);

      my $msg = "Nick = ".$this->{"nick"};
      $msg .= " Server = ". $this->{"server"};
      $msg .= " Port = ". $this->{"port"};
      $msg .= " Username = ". $this->{"username"};
      $msg .= " Ircname = ". $this->{"ircname"};
      #print $this->{"name"}." attempted connection with $msg \n";
      #print $this->{"name"}." attempted connection with ".$this->{"server"}." \n";
      #$poe_kernel->run();
      #print "started kernel\n";
}# sub _start 
#########################################################
=pod

=item _stop

shut down operation concerning this irc server

=cut

#########################################################
sub _stop 
{
  my $kernel = $_[KERNEL];
  my  $context = $_[SENDER];
  my ($this) = @_;

  print $this->{"name"}." Control session stopped.\n";
  $kernel->post( $context, 'quit', 'Neenios on ice!' );
  $kernel->alias_remove( $this->{"name"} );
}
#########################################################
=pod

=item irc_disconnected 

Reconnect to the server when we die.

=cut

#########################################################
sub irc_disconnected
{
  #print "irc_discon".Dumper(@_);
  print "irc_discon\n";
    my $server = $_[ARG0];
    my $kernel = $_[KERNEL];
    my $context = $_[SENDER];
  my ($this) = @_;
    #my ($server,$kernel) = $_[ARG0,KERNEL];
  if($this->{"zebot"} && !$this->{"zebot"}->{"running"})
  {
    print("ircServer::irc_disconnected not attempting to reconect: bot stopped\n");
    return;
  }# if(!$this->{"zebot"}->{"running"})
  #$this->{"zebot"} = $botref;


  #print STREAM "Disconnected from ", $server, " (", $server, "). Attempting to reconnect...\n";
  print "attempting connection\n";
  my $args = {
       "kernel" => $kernel,
          };
  $this->{"connecting"} = 1;
  $kernel->delay_add( "doConnect",$this->setting("reconnectinterval") , $args);

  print "after attempting reconnection\n";
  my $subargs = {
    "heap"	=> $_[HEAP],
    "network"	=> $_[HEAP]->{"network"},
    "context" =>$this->{"name"}, 
  };
  $kernel->post('mainframe', "irc_disconnected",$subargs);
}
#########################################################
=pod

=item network_connected

connection to a server achieved

After we successfully log into the IRC server, join a channel.

=cut

#########################################################
sub network_connected 
{
  #print("ircServer 001\n");
  my $kernel = $_[KERNEL];
  my $context = $_[SENDER];
  my $heap = $_[HEAP];
  my $this = $_[OBJECT];
  #we have a link no need to further try to connect...
  $this->{"connecting"} = 0;
  delete $this->{"connecting"};
  $kernel->post( $context, 'mode', $this->{"nick"}, '+i' );
  $kernel->post( $context, 'away', $this->setting("away"));

  #TODO register with nickserv
  $this->{"authpass"} = "toto" if(!($this->{"authpass"}));
  $this->{"authcmd"} = "auth" if(!($this->{"authcmd"}));
  $this->{"authhandler"} = "nickserv" if(!($this->{"authhandler"}));
  my $authsequence = $this->{"authcmd"}." ".$this->{"authpass"};
  $kernel->post( $context, 'privmsg', $this->{"authhandler"},$authsequence);
  #TODO check autojoin list for channels to be asked to be invited by chanserv
  foreach my $i (  @{$this->{"autoinvite"}})
  {
    #print("ircServer 001 asking for invite $i\n");
    $kernel->post( $context, 'privmsg', 'chanserv', "invite $i");
  }# foreach $i ($this->{"channel"})
  my $seconds =  $this->setting("rejoininterval");
  foreach my $i (@{$this->{"channel"}})
  {
    #print("ircServer 001 joining $i\n");
    $kernel->post($context, 'join', $i );
    $kernel->post($context,'privmsg', $i,$this->{"zebot"}->getMesg("CHANJOIN"));
    #give them some time and then check if we maanged to join
      my $subargs = { 
	 "channel"=> $i, 
                    };
      $heap->{"rejoins"}->{$i} = $subargs;
      #$this->rejoin($kernel,$context,$addressed,$nick,$actLang);
    print("posting rejoin from network_connected\n");
      $kernel->delay_add( "rejoin", $seconds, $subargs);
  }# foreach $i ($this->{"channel"})
  #print("ircServer 001 joining done\n");

  my $args = {
    "heap"	=> $_[HEAP],
    "network"	=> $_[HEAP]->{"network"},
    "context" =>$this->{"name"}, 
  };
  $kernel->post('mainframe', "irc_001",$args);
}# sub network_connected 
#########################################################
=pod

=item version

return the version of this module

=cut

#########################################################
sub version
{
    return '$Revision: 1.17 $';
}#sub isa
#########################################################
=pod

=item doConnect

do the actual job of connecting to the server

=cut

#########################################################
sub doConnect
{
  my($this) = @_;
  #uhm soneone reminds me what were the args for???
  my ($kernel, $session,$context,$args) = @_[KERNEL, SESSION, SENDER,ARG0];
  if($this->{"connecting"})
  {
    if(!$kernel)
    {
      print("??? kernel vanished.... don't know how to handle this...\n");
      return;
    }# if(!$kernel)

    $kernel->post( $this->{"name"}, 'connect', { Debug    => $this->setting("debug"),
	Nick     => $this->{"nick"},
	Server   => $this->{"server"},
	Port     => $this->{"port"},
	Username => $this->{"username"},
	Ircname  => $this->{"ircname"},
	}
    );
    $kernel->delay_add( "doConnect",$this->setting("reconnectinterval") , $args);
  }#if($this->{"connecting"})
}# sub doConnect
#########################################################
=pod

=item irc_socketerr 

failed to make the connection to the server

=cut

#########################################################
sub irc_socketerr 
{
  my($this) = @_;
  my ($kernel, $session,$context,$args) = @_[KERNEL, SESSION, SENDER,ARG0];
  my $err = $_[ARG0];
  my $name = $_[HEAP]->{"network"};

  print "socket error from server[$name]: $err\n Attempting to reconnect...\n";
  my $subargs = {
    "kernel" => $kernel,
  };
  $this->{"connecting"} = 1;
  $kernel->delay_add( "doConnect",$this->setting("reconnectinterval") , $subargs);

  $args = {
    "heap"	=> $_[HEAP],
    "network"	=> $name,
    "error"	=> $err,
    "context" =>$this->{"name"}, 
  };
  $kernel->post('mainframe', "irc_socketerr",$args);
}# sub irc_socketerr 
#########################################################
=pod

=item irc_connected 

we got a socket open with the server! first success!

=cut

#########################################################
sub irc_connected 
{
  my($this) = @_;
  my ($kernel, $session,$context,$args) = @_[KERNEL, SESSION, SENDER,ARG0];
  my $err = $_[ARG0];

  $this->{"connecting"} = 0;
  if(!$this->{"zebot"})
  {
    print("uhm? no bot ref in irc_connected??? :".$this->{"zebot"}."\n");
    return;
  }# if(!$this->{"zebot"})
  print("we are connected with ".$_[HEAP]->{"network"}."\n");

  $args = {
    "heap"	=> $_[HEAP],
    "network"	=> $_[HEAP]->{"network"},
    "line"	=> $args,
    "context" =>$this->{"name"}, 
  };
  $kernel->post('mainframe', "irc_connected",$args);
}# sub irc_connected 
#########################################################
=pod

=item channel_namelist

we received a end of names list, means we are newly connected

=cut

#########################################################
sub channel_namelist
{
  my $server = $_[ARG0];
  my $args = $_[ARG1];
  my $kernel = $_[KERNEL];
  my $heap = $_[HEAP];
  my $context = $_[SENDER];
  my ($this) = @_;
  #print "detected name list:\n";
  
  #trying to fetch the list of people

  #my @actives = $kernel->post( $context, 'names', $this->setting("channel") );
  ##got 'thor.skroz.net' '= #toutouland :toutou Lasade ' ''
  my ($channel, $names) = ($args =~ /= (#\S+) :(.*)/);
  my @users = split(/\s+/,$names);
  #print "+++ names for channel $channel are @users\n";
  $heap->{$channel}->{"users"} = \@users;
  #if($this->{"rejoins"}->{$channel})
  if($heap->{"rejoins"}->{$channel})
  {
    my @chan = keys(%{$heap->{"rejoins"}});
    my $channame = shift(@chan);
    #print($heap->{"network"}." seems we are rejoining ".Dumper($heap->{"rejoins"})."\n");
    print($heap->{"network"}." seems we are rejoining ".$channame."\n");
    #just in case...
    #$this->{"rejoins"}->{$channel} = 0;
    #delete $this->{"rejoins"}->{$channel};
    delete $heap->{"rejoins"}->{$channel};
  }#if($this->{"rejoins"}->{$channel})

  my $actLang = $heap->{"channels"}->{"$channel"};
  $actLang = ($this->{"zebot"})->setting("language") if(!$actLang);

    my $subargs = {
      "heap" => $heap,
      "userlist" =>\@users,
      "channel" =>$channel, 
      "lang" => $actLang,
      "network" => $heap->{"network"},
      "context" =>$this->{"name"}, #this one holds the link ot the actual server, can be used as post target
    };

  $kernel->post('mainframe', "channel_namelist",$subargs);
  return 0;
}#sub channel_namelist
#########################################################
=pod

=item irc_kick

react on some user or the bot itself beeing kicked

=cut

#########################################################
sub irc_kick
{
  my ($this) = $_[OBJECT];

  my $usernick = $_[ARG0]; #kicker
  my ($nick,$name,$host) = ( $usernick =~ /(\S+)!(\S+)@(\S+)/);
  my $addressed = $_[ARG1];
  my $kickeduser = $_[ARG2];
  my $reason = $_[ARG3];
  my $kernel = $_[KERNEL];
  my $context = $_[SENDER];
  my $heap = $_[HEAP];
  print "irc_kick user $kickeduser by $usernick from ".Dumper($addressed)." with $reason\n";

  if( $kickeduser eq $this->{"nick"} ) 
  {
    if( $this->setting("reconnect") ) 
    {

      my $seconds =  $this->setting("rejoininterval");
      my $subargs = { 
	 "channel"=> $addressed, 
	 "nick" => $nick, 
                    };
      $heap->{"rejoins"}->{$addressed} = $subargs;
      #$this->rejoin($kernel,$context,$addressed,$nick,$actLang);
      print("posting rejoin from irc_kick\n");
      $kernel->delay_add( "rejoin", $seconds, $subargs);
    }
  }# if( $kickeduser eq $this->{"nick"} ) 
  print("Propagating to zebot num of args: ".scalar(@_)."\n");

  my $actLang = $heap->{"channels"}->{"$addressed"};
  $actLang = ($this->{"zebot"})->setting("language") if(!$actLang);


    my $subargs = {
      "heap" => $heap,
      "usernick" =>$nick,
      "username" =>$name,
      "userhost" =>$host,
      "lang" => $actLang,
      "kickeduser" =>$kickeduser,
      "reason" =>$reason,
      "network" => $heap->{"network"},
      "addressed" =>[$addressed],
      "channel" =>$addressed, #hope we have only one channel target...
      "context" =>$this->{"name"}, #this one holds the link ot the actual server, can be used as post target
    };
  $kernel->post('mainframe', "irc_kick",$subargs);
}
#########################################################
=pod

=item rejoin

bot was kicked probably.... try to rejoin the channel

=cut

#########################################################
sub rejoin
{
  #my($this,$kernel,$context,$addressed,$nick,$actLang) = @_;
  my ($kernel, $session,$context,$args) = @_[KERNEL, SESSION, SENDER,ARG0];
  my $heap = $_[HEAP];
  my ($this) = @_;
  #my ($state, $event, $args,$kernel) = @_[STATE, ARG0, ARG1,KERNEL];
  my $addressed = $args->{"channel"};
      #$this->{"rejoins"}->{$addressed} = $subargs;
      #$this->rejoin($kernel,$context,$addressed,$nick,$actLang);
  my $seconds =  $this->setting("rejoininterval");
      #$kernel->delay_add( "rejoin", $seconds, $args);
  #prepare reconnection
    #print("dumping it all... ".Dumper(\@_)."\n");

  #if($this->{"rejoins"}->{$addressed})
  if($heap->{"rejoins"}->{$addressed})
  {
    $kernel->post( $this->{"name"}, 'join', $addressed );
    #$kernel->post( 'zebot', 'join', $addressed );
    #print("trying to rejoin channel $addressed\n");
    #print($heap->{"network"}." trying to rejoin channel $addressed with  ".Dumper($heap->{"rejoins"})."\n");

    my $nick = $args->{"nick"};
    if($nick)
    {
    #my $actLang = $args->{"lang"};
      my $msgHand = $this->{"zebot"}->{"actors"}->{"messageHandler"};
      my $message = $msgHand->getMesg("MORDS",$args);
      $message =~ s/%target/$nick/g;
      $kernel->post( $context, 'privmsg', $addressed, $message);
    }#if($nick)
    $seconds = $this->setting("rejoininterval");
    #$seconds = 10;
    print("posting rejoin from rejoin\n");
    $kernel->delay_add( "rejoin", $seconds, $args);
  }#if($this->{"rejoins"}->{$addressed})
  else
  {
    #print("rejoin called but as it seems we are allready on the channel\n");
    print($heap->{"network"}." rejoin called for  $addressed but as it seems we are allready on the channel\n");
    #print("hases ".Dumper($this->{"rejoins"})."\n");
    #print("has ".Dumper($heap->{"rejoins"})."\n");
  }#else
}# sub rejoin
#########################################################
=pod

=item channel_join_success

we managed to join the channel

=cut

#########################################################
sub channel_join_success
{
  #trying to fetch the list of people
  my ($this,$usernick,$addressed,$line, $kernel,$heap,$context ) 
    = @_[OBJECT, ARG0 .. ARG2,KERNEL,HEAP,SENDER];
  my ($nick,$name,$host) = ( $usernick =~ /(\S+)!(\S+)@(\S+)/);
  #print "irc_msg to ".Dumper($addressed)." with $line\n";
  $addressed =~ /^(#\S+)\s+/;
  my $channel = $1;
  #print("channel_join_success:: '$usernick','$channel','$addressed'\n");

  if($heap->{"rejoins"}->{$channel})
  {
    print($heap->{"network"}." seems we have rejoined $channel but rejoins is still set...\n");
    #just in case...
    #$this->{"rejoins"}->{$channel} = 0;
    #delete $this->{"rejoins"}->{$channel};
    delete $heap->{"rejoins"}->{$channel};
  }#if($this->{"rejoins"}->{$channel})

  my $actLang = $heap->{"channels"}->{"$channel"};
  $actLang = $this->setting("language") if(!$actLang);

    my $subargs = {
      "heap" => $heap,
      "usernick" =>$nick,
      "username" =>$name,
      "userhost" =>$host,
      "line" =>$addressed,
      "lang" => $actLang,
      "network" => $heap->{"network"},
      "addressed" =>$channel,
      "channel" =>$channel, #hope we have only one channel target...
      "context" =>$this->{"name"}, #this one holds the link ot the actual server, can be used as post target
    };
 
  #print "irc_msg: '$nick' '".Dumper($addressed)."' '$line'\n";
 
  $kernel->post('mainframe', "channel_join_success",$subargs);
}#sub channel_join_success
#########################################################
=pod

=item irc_ping

monitor the ping request, if they come to miss our own session will catch it
and try to send somethgn to the server to force either a response our an
acknowledgment that the connection is down

=cut

#########################################################
sub irc_ping
{
  #trying to fetch the list of people
  my ($this,$usernick,$addressed,$line, $kernel,$heap,$context ) 
    = @_[OBJECT, ARG0 .. ARG2,KERNEL,HEAP,SENDER];
  if(!($heap->{"pingdiff"})&& $heap->{"lastping"})
  {
    $heap->{"pingdiff"} = time() - $heap->{"lastping"};
   print("irc_ping set timediff to ".$heap->{"pingdiff"}." for ".$heap->{"network"}."\n");
  }# if(!($heap->{"pingdiff"})&& $heap->{"lastping"})
  $heap->{"lastping"} = time();

    my $subargs = {
      "heap" => $heap,
      "usernick" =>$usernick,
      "line" =>$line,
      "network" => $heap->{"network"},
      "context" =>$this->{"name"},
    };
  $kernel->post('mainframe', "irc_ping",$subargs);
}#sub irc_ping
#########################################################
=pod

=item keep_connected

monitor the ping request, if they come to miss our own session will catch it
and try to send somethgn to the server to force either a response our an
acknowledgment that the connection is down, this supposes that at least once
the sampling of the working server was made

=cut

#########################################################
sub keep_connected
{
  #trying to fetch the list of people
  my ($this,$usernick,$addressed,$line, $kernel,$heap,$context ) 
    = @_[OBJECT, ARG0 .. ARG2,KERNEL,HEAP,SENDER];
  #use default if we haven't mesured the real thing....
  my $compareTo = ($this->setting("pingtmout"))*60;
  if($heap->{"pingdiff"})
  {
  my $timediff =   time() - $heap->{"lastping"};

    $compareTo = $heap->{"pingdiff"}; 

    if($timediff > 2*$compareTo)
    {
      #worst case.... we missed at least 2 real pings...
      $kernel->post( $this->{"name"}, 'ison', $this->{"nick"});
      $kernel->post( $this->{"name"}, 'privmsg', "bboett","disconnected detected send ison");
      print("DISCONnected detected send ison\n");
    }# if($timediff > 2*$compareTo)
  }# if($heap->{"pingdiff"})
  else
  {
      print("DISCONnected not configured yet for ".$heap->{"network"}."\n");
  }# else
  $compareTo *= 2;
  $kernel->delay_add( "keep_connected", $compareTo);
}#sub keep_connected
#########################################################
=pod

=item setting 

setter/getter for the settings, to be able to choose any format for those
settings without disturbing the rest of the program...

=cut

#########################################################
sub setting 
{
  my $this = shift(@_);
  my $botref =  $this->{"zebot"};
  return $botref->setting(@_);
}#setting
#########################################################
=pod

=item irc_invite

someone invited us, be so nice as to join their channel

=cut

#########################################################
sub irc_invite
{
  my ($this,$usernick,$channel, $kernel,$heap,$context ) 
    = @_[OBJECT, ARG0 .. ARG1,KERNEL,HEAP,SENDER];
    $kernel->post($context, 'join', $channel );
    $kernel->post($context,'privmsg', $channel,$this->{"zebot"}->getMesg("CHANJOIN"));
  my $actLang = $heap->{"channels"}->{"$channel"};
  $actLang = ($this->{"zebot"})->setting("language") if(!$actLang);

    my $subargs = {
      "heap" => $heap,
      "usernick" =>$usernick,
      "channel" =>$channel, 
      "lang" => $actLang,
      "network" => $heap->{"network"},
      "context" =>$this->{"name"}, 
    };

  $kernel->post('mainframe', "irc_invite",$subargs);
}#sub irc_invite
#########################################################
=pod

=item irc_public

react on a line send to the actual chan

=cut

#########################################################
sub irc_public
{
  my ($this) = @_;

  #print "irc_public\n";

  my $usernick = $_[ARG0];
  my ($nick,$name,$host) = ( $usernick =~ /(\S+)!(\S+)@(\S+)/);
  my $addressed = $_[ARG1];
  my $line = $_[ARG2];
  my $kernel = $_[KERNEL];
  my $heap = $_[HEAP];

  my $actLang = $heap->{"channels"}->{$addressed->[0]};
  $actLang = $this->setting("language") if(!$actLang);

    my $subargs = {
      "heap" => $heap,
      "usernick" =>$nick,
      "username" =>$name,
      "userhost" =>$host,
      "line" =>$line,
      "lang" => $actLang,
      "network" => $heap->{"network"},
      "addressed" =>\$addressed,
      "channel" =>$addressed->[0], #hope we have only one channel target...
      "context" =>$this->{"name"}, #this one holds the link to the actual server, can be used as post target
    };

  $kernel->post('mainframe', "irc_public",$subargs);
}# sub irc_public
#########################################################
=pod

=item irc_quit

react on some user dropping out of the channel

=cut

#########################################################
sub irc_quit
{
  my ($this) = @_;
  my $usernick = $_[ARG0];
  my ($nick,$name,$host) = ( $usernick =~ /(\S+)!(\S+)@(\S+)/);
  my $addressed = $_[ARG1];
  my $line = $_[ARG2];
  my $kernel = $_[KERNEL];
  my $heap = $_[HEAP];
  my $msg = "irc_quit ";
  $msg .= " from $usernick" if($usernick);
  $msg .= " to $addressed" if($addressed);
  $msg .= " with $line" if($line);
  $msg .= "\n";
  print $msg;

  my $channel;
  if(ref($addressed) eq 'ARRAY')
  {
    $channel = $addressed->[0];
  }# if(ref($addressed) eq 'ARRAY')
  else
  {
    $channel = $addressed;
  }# else

  #exit(1) if(!($channel =~ /^#/));
  if(!($channel =~ /^#/))
  {
    $channel = "";
    $line = $addressed;
  }# if(!($channel =~ /^#/))

  my $actLang = $heap->{"channels"}->{"$channel"};
  $actLang = $this->setting("language") if(!$actLang);

    my $subargs = {
      "heap" => $heap,
      "usernick" =>$nick,
      "username" =>$name,
      "userhost" =>$host,
      "line" => $line,
      "network" => $heap->{"network"},
      "lang" => $actLang,
      "addressed" =>\$addressed,
      "channel" =>$channel, #hope we have only one channel target...
      "context" =>$this->{"name"}, #this one holds the link ot the actual server, can be used as post target
    };

  $kernel->post('mainframe', "irc_quit",$subargs);
}# sub irc_quit
#########################################################
=pod

=item irc_part

react on some user leaving the channel

=cut

#########################################################
sub irc_part
{
  my ($this) = @_;
  my $usernick = $_[ARG0];
  my ($nick,$name,$host) = ( $usernick =~ /(\S+)!(\S+)@(\S+)/);
  my $addressed = $_[ARG1];
  my $line = $_[ARG2];
  my $kernel = $_[KERNEL];
  my $heap = $_[HEAP];

  my $actLang = $heap->{"channels"}->{$addressed};
  $actLang = $this->setting("language") if(!$actLang);

  #print "irc_part of $usernick from $addressed\n";
    my $subargs = {
      "heap" => $heap,
      "usernick" =>$nick,
      "username" =>$name,
      "userhost" =>$host,
      "line" =>$line,
      "lang" => $actLang,
      "network" => $heap->{"network"},
      "addressed" =>[$addressed],
      "channel" =>$addressed, #hope we have only one channel target...
      "context" =>$this->{"name"}, #this one holds the link ot the actual server, can be used as post target
    };

  $kernel->post('mainframe', "irc_part",$subargs);
}
#########################################################
=pod

=item irc_mode

react on mode change issued by someone

=cut

#########################################################
sub irc_mode
{
  my ($this,$usernick) = @_[OBJECT,ARG0];

  #print "irc_mode\n";
  my ($nick,$name,$host) = [ "","",""];
  if( $usernick =~ /(\S+)!(\S+)@(\S+)/)
  {
    $nick = $1; $name=$2,$host = $3;
  }#if( $usernick =~ /(\S+)!(\S+)@(\S+)/)
  else
  {$nick = $usernick;}

  my $userchan = $_[ARG1];
  my $modes = $_[ARG2];
  my $kernel = $_[KERNEL];
  my $heap = $_[HEAP];

  #print "irc_mode".Dumper(@_);
  #print "irc_mode\n";
  #$kernel->post( $context, 'privmsg', $this->setting("channel"),"user '$nick' ($usernick)changed modes to $modes in $userchan");

  my $actLang = $heap->{"channels"}->{"$userchan"};
  $actLang = $this->setting("language") if(!$actLang);

    my $subargs = {
      "heap" => $heap,
      "usernick" =>$nick,
      "username" =>$name,
      "userhost" =>$host,
      "lang" => $actLang,
      "channel" =>$userchan,
      "network" => $heap->{"network"},
      "line" =>$modes,
      "context" =>$this->{"name"}, #this one holds the link ot the actual server, can be used as post target
    };
  $kernel->post( 'mainframe','irc_mode', $subargs );
}# sub irc_mode
#########################################################
=pod

=item irc_join

react on some user joining the channel

=cut

#########################################################
sub irc_join
{
  #print "irc_join".Dumper(@_);
  my ($this,$usernick) = @_[OBJECT,ARG0];
  my ($nick,$name,$host) = ( $usernick =~ /(\S+)!(\S+)@(\S+)/);
  my $addressed = $_[ARG1];
  my $line = $_[ARG2];
  my $kernel = $_[KERNEL];
  my $heap = $_[HEAP];
  #print "irc_join of$usernick on $addressed\n";

  my $actLang = $heap->{"channels"}->{"$addressed"};
  $actLang = $this->setting("language") if(!$actLang);

    my $subargs = {
      "heap" => $heap,
      "usernick" =>$nick,
      "username" =>$name,
      "userhost" =>$host,
      "lang" => $actLang,
      "line" =>$line,
      "network" => $heap->{"network"},
      "addressed" => [ $addressed ],
      "channel" =>$addressed, #hope we have only one channel target...
      "context" =>$this->{"name"}, #this one holds the link ot the actual server, can be used as post target
    };

  $kernel->post( 'mainframe','irc_join', $subargs );
}# sub irc_join
#########################################################
=pod

=item irc_msg

react on private msg send directly to the bot

=cut

#########################################################
sub irc_msg
{
  my ($this,$usernick) = @_[OBJECT,ARG0];
  my ($nick,$name,$host) = ( $usernick =~ /(\S+)!(\S+)@(\S+)/);
  my $addressed = $_[ARG1];
  my $line = $_[ARG2];
  my $kernel = $_[KERNEL];
  my $heap = $_[HEAP];
  my $context = $_[SENDER];
  #print "irc_msg to ".Dumper($addressed)." with $line\n";

  my $actLang = $heap->{"channels"}->{$addressed->[0]};
  $actLang = $this->setting("language") if(!$actLang);

    my $subargs = {
      "heap" => $heap,
      "usernick" =>$nick,
      "username" =>$name,
      "userhost" =>$host,
      "lang" => $actLang,
      "network" => $heap->{"network"},
      "line" =>$line,
      "addressed" =>\$addressed,
      "channel" =>$addressed->[0], #hope we have only one channel target...
      "context" =>$this->{"name"}, #this one holds the link ot the actual server, can be used as post target
    };

  #print "irc_msg: '$nick' '".Dumper($addressed)."' '$line'\n";

  $kernel->post( 'mainframe','irc_msg', $subargs );
}# sub irc_msg
#########################################################
=pod

=item irc_notice

react on notice send to the actual chan

=cut

#########################################################
sub irc_notice
{
  my ($this,$usernick) = @_[OBJECT,ARG0];
  my ($nick,$name,$host) = ( $usernick =~ /(\S+)!(\S+)@(\S+)/);
  my $addressed = $_[ARG1];
  my $line = $_[ARG2];
  my $kernel = $_[KERNEL];
  my $heap = $_[HEAP];

  #print "irc_notice: type ".$addressed->[0]." with $line\n";

  if ( $usernick =~ /\S+!\S+@\S+/)
  {
    print "irc_notice \n";
    print "data cn:;'$usernick', n='$nick',na='$name',h='$host'\n";
    print "add='".Dumper($addressed)."' msg='$line'\n";
  }#if ( $usernick =~ /\S+!\S+@\S+/)
  else
  {
    #print "server irc_notice \n";
  }

  my $actLang = $this->setting("language");

    my $subargs = {
      "heap" => $heap,
      "usernick" =>$nick,
      "username" =>$name,
      "userhost" =>$host,
      "lang" => $actLang,
      "line" =>$line,
      "network" => $heap->{"network"},
      "addressed" =>\$addressed,
      "channel" =>$addressed->[0], #hope we have only one channel target...
      "context" =>$this->{"name"}, #this one holds the link ot the actual server, can be used as post target
    };
  $kernel->post( 'mainframe','irc_notice', $subargs );
}# sub irc_notice
#########################################################
=pod

=item irc_ctcp_ping

react to a ping request

=cut

#########################################################
sub irc_ctcp_ping 
{
  my ($this,$kernel,$heap,$context) = @_[OBJECT,KERNEL,HEAP,SENDER];
  #print "irc_ping".Dumper(@_);
  print "irc_ping\n";
  my $usernick = $_[ARG0];
  my $channel = $_[ARG1];
  my $restline = $_[ARG2];


  my ($nick,$name,$host) = [ "","",""];
  if( $usernick =~ /(\S+)!(\S+)@(\S+)/)
  {
    $nick = $1; $name=$2,$host = $3;
  }#if( $usernick =~ /(\S+)!(\S+)@(\S+)/)
  else
  {$nick = $usernick;}

  #my $username = $line->{"username"};
  #my $userhost = $line->{"userhost"};
  my $actLang = $heap->{"channels"}->{"$channel"};
  $actLang = $this->setting("language") if(!$actLang);

    my $line = {
      "heap" => $heap,
      "usernick" =>$nick,
      "username" =>$name,
      "userhost" =>$host,
      "line" =>$restline,
      "lang" => $actLang,
      "network" => $heap->{"network"},
      "addressed" =>$channel,
      "channel" =>$channel,
    };

  $kernel->post($context, 'ctcp_reply',$nick, join (' ', ($restline)));
  print STREAM "*** CTCP PING request from $nick received\n";

  $kernel->post( 'mainframe','irc_ping', $line );
}# sub irc_ping 
#########################################################
=pod

=item irc_ctcpreply_ping 

Gives lag results for outgoing PINGs.

=cut

#########################################################
sub irc_ctcpreply_ping
{
  my ($this,$usernick,$channel,$args,$kernel,$heap) = @_[OBJECT,ARG0..ARG2,KERNEL,HEAP];
  #print "irc_rping".Dumper(@_);
  print "irc_rping\n";

  my ($nick,$name,$host) = [ "","",""];
  if( $usernick =~ /(\S+)!(\S+)@(\S+)/)
  {
    $nick = $1; $name=$2,$host = $3;
  }#if( $usernick =~ /(\S+)!(\S+)@(\S+)/)
  else
  {$nick = $usernick;}

  #my $username = $line->{"username"};
  #my $userhost = $line->{"userhost"};
  my $actLang = $heap->{"channels"}->{"$channel"};
  $actLang = $this->setting("language") if(!$actLang);

    my $line = {
      "heap" => $heap,
      "usernick" =>$nick,
      "username" =>$name,
      "userhost" =>$host,
      "line" =>$args,
      "lang" => $actLang,
      "network" => $heap->{"network"},
      "addressed" =>$channel,
      "channel" =>$channel,
    };


  $args = time - $args;
  print STREAM "*** CTCP PING reply from $nick: $args sec.\n";
  $kernel->post( 'mainframe','irc_ctcpreply_ping', $line );
}# sub irc_ctcpreply_ping
#########################################################
=pod

=item irc_ctcp_action

react on private msg send directly to the bet

=cut

#########################################################
sub irc_ctcp_action
{
  my ($this,$usernick,$addressed,$line,$kernel,$heap) = @_[OBJECT,ARG0..ARG2,KERNEL,HEAP];
  my ($nick,$name,$host) = ( $usernick =~ /(\S+)!(\S+)@(\S+)/);
  print "irc_ctcp_action $usernick on ".Dumper($addressed)." with $line\n";

  my $actLang = $heap->{"channels"}->{$addressed->[0]};
  $actLang = $this->setting("language") if(!$actLang);

    my $subargs = {
      "heap" => $heap,
      "usernick" =>$nick,
      "lang" => $actLang,
      "username" =>$name,
      "userhost" =>$host,
      "line" =>$line,
      "network" => $heap->{"network"},
      "addressed" =>$addressed,
      "channel" =>$addressed->[0], #hope we have only one channel target...
      "context" =>$this->{"name"}, #this one holds the link ot the actual server, can be used as post target
    };

  $kernel->post( 'mainframe','irc_ctcp_action', $subargs );

}#irc_ctcp_action
#########################################################
=pod

=item irc_ctcp_version

spit out the version num and info when asked through ctcp version

=cut

#########################################################
sub irc_ctcp_version
{
  my ($this,$usernick,$addressed,$line,$kernel,$heap) = @_[OBJECT,ARG0..ARG2,KERNEL,HEAP];
  my ($nick,$name,$host) = ( $usernick =~ /(\S+)!(\S+)@(\S+)/);
  #print "irc_ctcp_version to ".Dumper($addressed)." with $line\n";

  my $actLang = $heap->{"channels"}->{$addressed->[0]};
  $actLang = $this->setting("language") if(!$actLang);

  my $subargs = {
      "heap" => $heap,
    "usernick" =>$nick,
    "username" =>$name,
    "userhost" =>$host,
    "lang" => $actLang,
    "line" =>$line,
      "network" => $heap->{"network"},
    "addressed" =>\$addressed,
    "channel" =>$addressed->[0], #hope we have only one channel target...
      "context" =>$this->{"name"}, #this one holds the link ot the actual server, can be used as post target
  };

  $kernel->post( 'mainframe','irc_ctcp_version', $subargs );
}#irc_ctcp_version
#########################################################
=pod

=item channel_join_topic

we received a channel topic

=cut

#########################################################
sub channel_join_topic
{
  my ($this,$server,$args,$kernel,$heap) = @_[OBJECT,ARG0..ARG1,KERNEL,HEAP];
  #my ($nick,$name,$host) = ( $usernick =~ /(\S+)!(\S+)@(\S+)/);
  my @arguments = @_[ARG0..ARG9];

  print("channel_join_topic unnkown args:".Dumper(\@arguments)."\n");

  #my $actLang = $heap->{"channels"}->{$addressed->[0]};
  my $actLang ;
  $actLang = $this->setting("language") if(!$actLang);

  my $subargs = {
      "heap" => $heap,
    "lang" => $actLang,
      "network" => $heap->{"network"},
      "context" =>$this->{"name"}, #this one holds the link ot the actual server, can be used as post target
  };

  #trying to fetch the list of people
  $kernel->post( 'mainframe','channel_join_topic', $subargs );
}#sub channel_namelist
#########################################################
=pod

=item irc_dcc_done

react on some dcc transfer success

=cut

#########################################################
sub irc_dcc_done 
{
  my ($this,$kernel,$heap) = @_[OBJECT,KERNEL,HEAP];
  my ($magic, $usernick, $type, $port, $file, $size, $done) = @_[ARG0 .. ARG6];
  my ($nick,$name,$host) = ( $usernick =~ /(\S+)!(\S+)@(\S+)/);

    my $subargs = {
      "heap" => $heap,
      "usernick" =>$nick,
      "username" =>$name,
      "userhost" =>$host,
      "magic" =>$magic,
      "type" =>$type,
      "file" =>$file,
      "port" =>$port,
      "network" => $heap->{"network"},
      "size" =>$size,
      "done" =>$done,
      "context" =>$this->{"name"}, #this one holds the link ot the actual server, can be used as post target
    };

  $kernel->post( 'mainframe','irc_dcc_done', $subargs );
}# sub irc_dcc_done 
#########################################################
=pod

=item irc_dcc_error

react on some dcc transfer fail

=cut

#########################################################
sub irc_dcc_error 
{
  my ($this,$kernel,$heap) = @_[OBJECT,KERNEL,HEAP];
  my ($err, $usernick, $type, $file) = @_[ARG0 .. ARG2, ARG4];
  my ($nick,$name,$host) = ( $usernick =~ /(\S+)!(\S+)@(\S+)/);

    my $subargs = {
      "heap" => $heap,
      "network" => $heap->{"network"},
      "usernick" =>$nick,
      "username" =>$name,
      "userhost" =>$host,
      "error" =>$err,
      "type" =>$type,
      "file" =>$file,
      "context" =>$this->{"name"}, #this one holds the link ot the actual server, can be used as post target
    };

  $kernel->post( 'mainframe','irc_dcc_error', $subargs );
}# sub irc_dcc_error 
#########################################################
=pod

=item irc_dcc_request

react on some dcc transfer/chat proposal

 You receive this event when another IRC client sends you a DCC SEND
 or CHAT request out of the blue. You can examine the request and
 decide whether or not to accept it here. ARG0 is the nick of the
 client on the other end. ARG1 is the type of DCC request (CHAT,
 SEND, etc.). ARG2 is the port number. ARG3 is a "magic cookie"
 argument, suitable for sending with 'dcc_accept' events to signify
 that you want to accept the connection (see the 'dcc_accept' docs).
 For DCC SEND and GET connections, ARG4 will be the filename, and
 ARG5 will be the file size.

=cut

#########################################################
sub irc_dcc_request 
{
  my ($this,$kernel,$heap, $usernick, $type, $port, $magic, $filename, $size) =
    @_[OBJECT,KERNEL,HEAP, ARG0 .. ARG5];
    if($type eq 'ACCEPT')
    {
       #the args are in wrong order and missing shift the args 1 up
       $magic->{port} = $magic->{addr};

       #print("Dumping alt cookie : ".Dumper($_[HEAP]->{cookies})."\n");
       my $altcookie = $_[HEAP]->{cookies}->{$filename};
       $magic->{addr} = $altcookie->{addr};
       delete $_[HEAP]->{cookies}->{$filename};
       #TODO beware a possible memory leak here... 
    }# if($type eq 'ACCEPT')
  my ($nick,$name,$host) = ( $usernick =~ /(\S+)!(\S+)@(\S+)/);
  my $tmpServer = $_[SENDER];
  if(!$tmpServer)
  {# try a random one
    my @names = keys(%{$this->{"servers"}});
    my $actServer =  $names[0];
    print("dcc_req using random server with name: $actServer\n");
    $tmpServer = $this->{"handles"}->{$actServer};
  }# if(!$tmpServer)
  #$this->{"handles"} = $serverList;

    my $subargs = {
      "heap" => $heap,
      "network" => $heap->{"network"},
      "usernick" =>$nick,
      "username" =>$name,
      "userhost" =>$host,
      "magic" =>$magic,
      "type" =>$type,
      "file" =>$filename,
      "port" =>$port,
      "size" =>$size,
      "context" =>$this->{"name"}, #this one holds the link to the actual server, can be used as post target
    };

  $kernel->post( 'mainframe','irc_dcc_request', $subargs );
}# sub irc_dcc_request 
#########################################################
=pod

=item irc_dcc_get

data was trasnmitted successfully

 Notifies you that another block of data has been successfully
 transferred from the client on the other end of your DCC GET con-
 nection.  ARG0 is the connection's magic cookie, ARG1 is the nick
 of the person on the other end, ARG2 is the port number, ARG3 is
 the filename, ARG4 is the total file size, and ARG5 is the number
 of bytes successfully transferred so far.

=cut

#########################################################
sub irc_dcc_get 
{
  my ($this,$kernel,$heap) = @_[OBJECT,KERNEL,HEAP];
  #-- irc_dcc_get 4 bboett!bboett@192.168.0.5 1024 psychic_academy_01_mh.avi 65507328 3454464
    
  my ($magic, $usernick, $port, $file, $size, $done) = @_[ARG0 .. ARG5];
  my ($nick,$name,$host) = ( $usernick =~ /(\S+)!(\S+)@(\S+)/);

    my $subargs = {
      "heap" => $heap,
      "network" => $heap->{"network"},
      "usernick" =>$nick,
      "username" =>$name,
      "userhost" =>$host,
      "magic" =>$magic,
      "file" =>$file,
      "port" =>$port,
      "size" =>$size,
      "done" =>$done,
      "context" =>$this->{"name"}, #this one holds the link ot the actual server, can be used as post target
    };

  $kernel->post( 'mainframe','irc_dcc_get', $subargs );
}# sub irc_dcc_get 
#########################################################
=pod

=item irc_dcc_chat

Notifies you that one line of text has been received from the client on the
other end of a DCC CHAT connection. ARG0 is the con- nection's magic cookie,
ARG1 is the nick of the person on the other end, ARG2 is the port number, and
ARG3 is the text they sent.

=cut

#########################################################
sub irc_dcc_chat 
{
  my ($this,$kernel,$heap) = @_[OBJECT,KERNEL,HEAP];
    
  my ($magic, $usernick, $port, $text) = @_[ARG0 .. ARG3];
  my ($nick,$name,$host) = ( $usernick =~ /(\S+)!(\S+)@(\S+)/);

    my $subargs = {
      "heap" => $heap,
      "network" => $heap->{"network"},
      "usernick" =>$nick,
      "username" =>$name,
      "userhost" =>$host,
      "magic" =>$magic,
      "line" =>$text,
      "port" =>$port,
      "context" =>$this->{"name"}, #this one holds the link ot the actual server, can be used as post target
    };

  $kernel->post( 'mainframe','irc_dcc_chat', $subargs );
}# sub irc_dcc_chat 
#########################################################
=pod

=item irc_nick_taken 

 nick allready taken, create a new one
 TODO add here the nickserv handler!!!

=cut

#########################################################
sub irc_nick_taken 
{
  my ($this,$heap) = @_[OBJECT,HEAP];

  if($this->{"nick"} =~ /^(.*?)(\d+)$/)
  {
    $this->{"nick"} = $1.($2+1);
  }# if($this->{"nick"} =~ /^(.*?)(\d+)$/)
  else
  {
    $this->{"nick"} .= "2";
  }# else
  $heap->{"nick"} = $this->{"nick"};
}
1
__END__

=back

=head1 AUTHOR

Bruno Bttcher <bboett at adlp.org>

=head1 SEE ALSO

zebot home page  http://www.freesoftware.fsf.org/zebot/ 
POD documentation of zebot

=cut

