##############################################################################
=pod

=head1 NAME

Channel module

=head1 DESCRIPTION

Handles all the channel's related graphical things

=head1 COPYRIGHT

Copyright (c) 2002 Bruno Boettcher

=head1 LICENCE
Channel.pm is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; version 2
of the License.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public
License along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

=head1 Methods of this class

=over

=cut

##############################################################################
package zebot::gui::Channel;
use strict;

use constant FALSE => 0;
use constant TRUE  => 1;

use zebot::gui::Commands;
use zebot::gui::Gui;


##############################################################################
=pod

=item new

Create a new Channel window

=cut

##############################################################################
sub new 
{
  my($classname,$data) = @_;

  my $self = bless {}, $classname;

  $self->{name}   = $data->{"channel"};
  $self->{server} = $data->{"network"};
  $self->{main} = $data->{"callback"};

  $self->{shell} = Gtk2::Entry->new;
  $self->{shell}->signal_connect("activate",
      sub { $self->input($self->{shell}->get_text) });

  $self->{text}  = Gtk2::TextView->new;
  $self->{text}->set_wrap_mode('char');
  $self->{text}->set_editable(0);
  $self->{text}->set_size_request(400, 200);
  my $st = Gtk2::ScrolledWindow->new(undef, undef);
  $st->set_policy("never", "automatic");
  $st->add($self->{text});
  $self->init_tags();


  $self->{list}  = Gtk2::ListStore->new("Glib::String");
  my $treeview   = Gtk2::TreeView->new_with_model($self->{list});
  my $column     = Gtk2::TreeViewColumn->new_with_attributes(
      "nicks", Gtk2::CellRendererText->new,
      text => 0);
  $treeview->append_column($column);
  $treeview->set_headers_visible(0);
  my $sw = Gtk2::ScrolledWindow->new(undef, undef);
  $sw->set_policy("never", "automatic");
  $sw->add($treeview);

  my $vb = Gtk2::VBox->new;
  my $hp = Gtk2::HPaned->new;
  $hp->pack1($st, 1, 0);
  $hp->pack2($sw, 0, 0);
  $vb->pack_start($hp, TRUE, TRUE, 0);
  $vb->pack_start($self->{shell}, FALSE, TRUE, 0);
  $vb->show_all;

  $self->{"main"}->{"notebook"}->append_page($vb, Gtk2::Label->new($self->{name}));
  $self->{"main"}->{"notebook"}->set_current_page($self->{"main"}->{"notebook"}->page_num($vb));
  $self->{shell}->grab_focus;


  return $self;
}

##############################################################################
=pod

=item init_tags

Initialise the text tags for each irc color

=cut

##############################################################################

sub init_tags {
		my $buf = shift->{text}->get_buffer;
		my($tg, $ground);
		my @colors = ("snow",     "black",     "navy",   "SeaGreen3", "red",
				      "brown",    "purple",    "orange", "yellow",    "green",
					  "SkyBlue3", "turquoise", "blue",   "magenta",   "gray55");
		
		($tg, $ground) = ("fg", "foreground");
		$buf->create_tag("$tg$_", $ground => $colors[$_]) for(0..14);
		($tg, $ground) = ("bg", "background");
		$buf->create_tag("$tg$_", $ground => $colors[$_]) for(0..14);
}


##############################################################################
=pod

=item input

Handles the user input

=cut

##############################################################################

sub input {
    my $self   = shift;
    my($input) = @_;

	$self->{shell}->set_text("");

	
    if($input =~ m|^/(\S+)(?: (.*))?|) {
        $Cmd{$1} ? $Cmd{$1}->($self, split(/ /,$2||"")) : 
			       $Cmd{unknown}->($self, $1);
		return;
	}
	
	$Cmd{noserver}->($self), return
		unless($self->{server} && $self->{server}{sock});
	
	$self->{server}->send("$input\r\n");
	$self->insert($input);
}

##############################################################################
=pod

=item insert

Inserts the text into the Channel window

=cut

##############################################################################

sub insert {
	my $self   = shift;
	my($input) = @_;
	$input = "\n".$input;

	my $buf = $self->{text}->get_buffer;
	my($start, $end) = $buf->get_bounds;

	while($input) {
		$input =~ s/^(?:\x03(\d{1,2})?)?(?:,(\d{0,2})?)?([^\x03]*)//;
		$buf->insert_with_tags_by_name($end, "$3",
				defined($1) ? $2 ? ("fg".int$1, "bg".int$2) : "fg".int$1 : ());
	}

	my $mark = $buf->create_mark("", $end, FALSE);
	$self->{text}->scroll_to_mark($mark, 0, TRUE, 0.0, 1.0);
	$buf->delete_mark($mark);
}
1;
