#!/usr/bin/perl -w
package zebot::configHandler;
use strict;
use warnings;
##############################################################################
=pod

=head1 NAME

zebot::configHandler

=head1 COPYRIGHT and LICENCE

  Copyright (c) 2003 Bruno Boettcher

  zebot::configHandler is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; version 2
  of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

=head1 DESCRIPTION

This class is an entity that is given a target and then loads or saves the settings accordingly

=back

=head1 Methods of this class

=over

=cut

use Data::Dumper;

use constant FALSE => 0;
use constant TRUE => 1;


#####################################################################
=pod

=item new, Constructor


The constructor of a zebot::configHandler object, initializes a number of datastructures that will be used later, creates a blessed reference

=cut

#####################################################################
sub new
{
  my ($classname, $settings) = @_;

  my $obj_ref = { };

  bless $obj_ref, $classname;

  $obj_ref->{"settings"} = $settings if ($settings);

  return $obj_ref;
}# sub new
#####################################################################
=pod

=item init

Here we really build up this object

=cut

#####################################################################
sub init
{
  my ($this) = @_;
  $this->{"settings"} = {} if(!$this->{"settings"});
}# sub init
#########################################################
=pod

=item loadSettings

load form the conf file the different bot settings
means using the setting configpath all files ending with conf (and as soon as i figured out how to setup the same stuff in XML, also ending with .xml) the config directory is scanned for conf files and read in and added to the setings of the bot. 

TODO, leave the non loaded actors settings away...

=cut

#########################################################
sub loadSettings 
{
  my ($this, $basepath) = @_;


  $basepath .= "/" if(!($basepath =~ /\/$/));
  #ensure the / at the end the other modules will trust that...
  $this->{"settings"}->{"configpath"} = $basepath;
  $basepath .= "config/";

  my $subsettings = {};

  if(opendir(DIRLIST,$basepath))
  {
    my @dirlisting = grep { /\.conf|\.xml$/ && -f "$basepath/$_" }readdir(DIRLIST);
    foreach my $filename (@dirlisting) 
    {
      if($filename =~ /(.*?)\.conf$/)
      {
	my $actor = $1;
	my $actorSets = {};
	$filename = $basepath .$filename;
	#print("calling loadhash on $filename\n");
	$this->loadHash($filename,$actorSets);
	#print("Loaded for $actor: ".Dumper($actorSets)."\n");
	$subsettings->{$actor} = $actorSets;
      }# if($filename =~ /\.conf$/)
      elsif($filename =~ /(.*?)\.xml$/)
      {
	print("XML configuration files ($filename) aren't parsed for the moment\n");
      #  my $actor = $1;
      #  my $actorSets = {};
      #  $actorSets = XMLin($basepath."$filename");
      #  #print "dumping config:\n".Dumper($actorSets)."\n";
      #  print( "DUMPING config keys:\n".join(' ',keys(%$actorSets))."\n");
      #  
      }# elsif($filename =~ /(.*?)\.xml$/)
    }#foreach $filename (@dirlisting) 

    closedir(DIRLIST);
  }# if(opendir(DIRLIST,$lpath))
  else
  {
    print("zebot::loadSettings couldn't open the configdir:'$basepath'!!!\n");
    die("You didn't read the documentation!!\n");
  }# else
  # throwing out the hashes of subsettings that aren't needed
  #print( "-----------------------------------------------------\n" );
  my $actorSettings = $subsettings->{"actors"};
  my @actornames = keys(%$actorSettings);
  $this->{"actornames"} = \@actornames;
  #cleanse the actor settings from the records that have no real data
  foreach my $name (@actornames)
  {
    if($actorSettings->{$name} != 1)
    {
      print("DOING something with actor $name\n");
      $this->{"settings"}->{"subsettings"}->{"actors"}->{$name} = $actorSettings->{$name};
    }# if($actorSettings->{$name} != 1)
  }# foreach my $name (@actornames)
  #throw this out of further processing
  #delete $subsettings->{"actors"};
  #merge the zebot stuff with the general settings
  foreach my $name (keys(%{$subsettings->{"zebot"}}))
  {
   $this->{"settings"}->{$name} = $subsettings->{"zebot"}->{$name};
  }# foreach my $name (keys(%{$subsettings->{"zebot"}}))
  #throw this out of further processing
  delete $subsettings->{"zebot"};
  #add the remaining stuff to the subsettings
  foreach my $subset (keys(%$subsettings))
  {
    if($subsettings->{$subset} && scalar(keys(%{$subsettings->{$subset}})) >0)
    {
      #print("added data for $subset\n");
    $this->{"settings"}->{"subsettings"}->{$subset} = $subsettings->{$subset};
    }# if($subsettings->{$subset})
  }# foreach my $subset (keys(%$subsettings))
  #remove temporary structures
  delete $this->{"tmpsettings"};
  # Load the irc server settings.... for the moment still a hash...
  #$this->{"servers"} = do($basepath.'servers.conf');
  #$this->{"servers"} = $this->{"settings"}->{"subsettings"}->{"servers"};
  #delete $this->{"settings"}->{"subsettings"}->{"servers"};
  return $this->{"settings"};
}# sub loadSettings 
#########################################################
=pod

=item loadHash

suppose a file organized line wise with
key value
open the file, read it in, aprse each line for those key value pairs and add
them to the hash of settings


if lines are empty or prepended with a # ignore them

=cut

#########################################################
sub loadHash
{
  my ($this,$filename,$params) = @_;
  my $debug = 0;
  $debug = 1 if( $filename =~ /server/ );
  my @stack;
  my $actParam = $params;

  if(open (SETTINGS, $filename))
  {
  #print "-- loadSettings over settings ".Dumper($this->{"settings"})."\n";
    while(<SETTINGS>)
    {
      chomp( $_ );
      my $tagname;
      my $value;
      #print("extracter $_\n") if($debug);
      #print("extracter $tagname = $value\n") if($debug);
      if(/^\s*"(.*?)"\s*=>\s*\{/)
	{
	  #start of sub-settings
	  my $subParms = {};
	  $actParam->{$1} = $subParms;
	  push(@stack,$actParam);
	  $actParam = $subParms;
	}# if(/ =>\s*\{/)
      elsif(/^\s*\}/)
      {
	if(scalar(@stack) > 0)
	{
	  $actParam = pop(@stack);
	}# if(scalar(@stack) > 0)
      }# elsif(/^\s*\}/)
      elsif(/^#/ || /^\s*$/ || /^\s*\{/)
      {
	#ignore....
      }
      elsif(/^(\S+)\s+(.*)$/)
      {
	$tagname = $1;
	$value = $2;
	#print "setting $tagname to $value\n";
	$value =~ s/\s*#.*$//;
	$actParam->{$tagname} = $value if(!($value =~ /^HASH/));
      }#if(/\w+:\w+:\w+.*/)
      elsif(/^\s*"(\S+)"\s*=>\s*(.*?)\s*,?\s*$/)
      {
	$tagname = $1;
	$value = $2;
	$value =~ s/^\s*|\s*$//g;
	$value =~ s/^'|'$//g;
	$value =~ s/^"|"$//g;
	if($value =~ /\[(.*?)\]/)
	{
	  $value = [];
	  @{$value} = split(/,/,$1);
	  foreach (@{$value}) { s/^\s*'//; s/\s*'\s*$//; }
	 #print("implicite array!! with $tagname = ".Dumper($value)."\n"); 
	}# if($value =~ /\[.*?\]/)
	else
	{
	  #print("line is hash el: $tagname = $value\n") if($debug);
	}
	#print "setting $tagname to $value\n";
	$actParam->{$tagname} = $value if(!($value =~ /^HASH/));
      }#if(/\w+:\w+:\w+.*/)
      elsif(/^(\S+)/)
      {
	$tagname = $1;
	$value = 1;
	#print "setting $tagname to $value\n";
	$actParam->{$tagname} = $value;
      }#if(/\w+:\w+:\w+.*/)
    }#while(<SETTINGS>)
    close (SETTINGS);
  }#if(open (SETTINGS, "zebot.conf"))
  else
  { print("zebot::loadHash couldn't open $filename\n");}
}# sub loadHash
#########################################################
=pod

=item dumpSettings 

issue the actual settings of the bot

=cut

#########################################################
sub dumpSettings 
{
  my ($this,$hash) = @_;
  my $i;
  my $listing = "";

  foreach $i (sort(keys(%{$hash})))
  {
    my $argl = $hash->{$i};
    $listing .= $i." ".$hash->{$i}."\n" if (ref(\$argl) eq "SCALAR") ;
  }# foreach $i (keys(%hash))
  #TODO and the server lists?
  return $listing;
}
#########################################################
=pod

=item saveSettings 

save actual bot settings to the conf file

=cut

#########################################################
sub saveSettings 
{
  my ($this,$settings) = @_;

  $this->{"settings"} = $settings;
  my $basepath = $this->{"settings"}->{"configpath"};
  $basepath .= "config/";

  my $path = $basepath."zebot.conf";
  open (SETTINGS, ">$path") || die("couldn't open $path");
  print SETTINGS $this->dumpSettings($this->{"settings"});
  close (SETTINGS);
  #TODO and the server lists?


  my @subgroups = keys(%{$this->{"settings"}->{"subsettings"}});
  #be sure not to override the servers conf as long as we have this system...
  my $tmp = join(' ',@subgroups);
  $tmp =~ s/servers//g;
  @subgroups = split(/ /,$tmp);
  foreach my $name (@subgroups)
  {
    $path = $basepath."$name.conf";

    open (SETTINGS, ">$path") || die("couldn't open $path");
    my $dump = $this->dumpSettings($this->{"settings"}->{"subsettings"}->{$name});
    print SETTINGS "$dump\n";
    close (SETTINGS);
  }# foreach my $name (@actornames)
}# sub saveSettings 
######################################################################
=pod

=item setValidKeys 

getter/setter for the hash of keys that should be accepted through shell
arguments, if this isn't present any argument will be accepted.

=cut

######################################################################
sub setValidKeys 
{
  my ($this,$hashref) = @_;
  if($hashref && ref($hashref) eq 'HASH')
  {
    $this->{"allowedkeys"} = $hashref;
  }# if($hashref)
  else
  {
    #print("returning available allowed keys:".Dumper($this->{"allowedkeys"})."\n");
    return $this->{"allowedkeys"};
  }# else
}# sub setValidKeys 
######################################################################
=pod

=item container 

getter/setter for the hash of settings we are manipulating.

=cut

######################################################################
sub container 
{
  my ($this,$hashref) = @_;
  if($hashref && ref($hashref) eq 'HASH')
  {
    $this->{"settings"} = $hashref;
  }# if($hashref)
  else
  {
    return $this->{"settings"};
  }# else
}# sub container 
######################################################################
=pod

=item parseArgs 

parse the incoming arguments, if allowedkeys hash is present, it is used to
translate the keys to the sttings used in the program, otherwise the keyname is
used as parameter name. Also if the allowedkeys hash is present only the keys
psecifiyed in there are allowed. 

using the hash implies: the keys are still with the prepended '-''s, if after
the '-'s a no is detected it is thought as a negation setting if there isn't a
valid value following.

=cut

######################################################################
sub parseArgs
{
  my($this,$arguments) = @_;

  while(scalar(@$arguments) >0)
  {
    my $key = shift(@$arguments);
    $key = lc($key);
    my $negate = FALSE;
    if($key =~ /(-+)no(.*)/)
    {
      $key = $1.$2;
      $negate = TRUE;
    }# if($key =~ /(-+)no(.*)/)
    
    if($this->{"allowedkeys"})
    {
      if($this->{"allowedkeys"}->{$key} || $this->{"allowedkeys"}->{"no".$key})  
      {
	my $value = $this->fetchNext($arguments,$negate);
	if($negate && $value ne TRUE)
	{
	  $key = "no".$key;
	}# if($negate && $value ne TRUE)
	$this->{"settings"}->{$this->{"allowedkeys"}->{$key}} = $value;
      }# if($this->{"allowedkeys"}->{$key})  
      else
      {
	print("invalid argument submitted, allowed are ".Dumper($this->{"allowedkeys"})."\n");
      }# else
    }# if($this->{"allowedkeys"})
    else
    {
      $key =~ s/^-+//;
	my $value = $this->fetchNext($arguments,$negate);
	if($negate && $value ne TRUE)
	{
	  $key = "no".$key;
	}# if($negate && $value ne TRUE)
	$this->{"settings"}->{$key} = $value;
    }# else
  }# while(scalar(@$arguments) >0)
}#sub parseArgs
######################################################################
=pod

=item fetchNext 

check out the next argument, if it isn't prepended with - it should be a value
that we return, otherwise we push back the thing and return the logical
attribute.

=cut

######################################################################
sub fetchNext
{
  my($this,$arguments,$bool) = @_;

    my $value = shift(@$arguments);
    if($value =~ /^-/)
    {
      unshift(@$arguments,$value);
      return $bool;
    }# if($key =~ /(-+)no(.*)/)

    return $value;
}#sub fetchNext
1
__END__


=back

=head1 AUTHOR

Bruno Bttcher <bboett at adlp.org>

=head1 SEE ALSO

  zebot home page  http://www.freesoftware.fsf.org/zebot/ 
  Net::IRC, 
  RFC 1459,
  http://www.irchelp.org/, 
  http://poe.perl.org/
  http://www.cs.cmu.edu/~lenzo/perl/, 
  http://www.infobot.org/,


=cut

