///////////////////////////////////////////////////////////////////////////////
// Copyright (C) 2008 Rene Jensen                                            //
//                                                                           //
// This file is part of YUV4MPEG Motion Tools (YUVMotionTools).              //
//                                                                           //
// Authors: Rene Jensen <centipede@takhis.net>                               //
//                                                                           //
// YUVMotionTools is free software: you can redistribute it and/or modify    //
// it under the terms of the GNU General Public License as published by      //
// the Free Software Foundation, either version 3 of the License, or         //
// (at your option) any later version.                                       //
//                                                                           //
// YUVMotionTools is distributed in the hope that it will be useful,         //
// but WITHOUT ANY WARRANTY; without even the implied warranty of            //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             //
// GNU General Public License for more details.                              //
//                                                                           //
// You should have received a copy of the GNU General Public License         //
// along with YUVMotionTools.  If not, see <http://www.gnu.org/licenses/>.   //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#include <map>
#include <list>
#include <vector>
#include <utility>
#include <iterator>
#include <iostream>
#include <fstream>
#include <cstdio>
#include <cstring>
#include <cmath>
#include <boost/format.hpp>
#include <opencv/cv.h>
#include <mjpegtools/mjpeg_types.h>
#include <mjpegtools/yuv4mpeg.h>
#include <mjpegtools/mjpeg_logging.h>

#include "featurepoint.h"
#include "trackingbuffers.h"

using namespace std;
using boost::format;

#define LOG_INFO 2    // Dunno about these values.
#define LOG_WARN 1
#define LOG_ERROR 0

TrackingBuffers* buffers = 0;
featurepoints_t  points;
featurepoints_t  savedPoints;


typedef std::multimap <int, pair<float,float> > velocity_statistics;
struct isFrameLess    { bool operator()(const velocity_statistics::value_type& a, const velocity_statistics::value_type& b) { return a.first < b.first; } };
struct isFrameGreater { bool operator()(const velocity_statistics::value_type& a, const velocity_statistics::value_type& b) { return a.first > b.first; } };

int main(int argc, char*argv[])
{
    //stdin = fopen ("./motiontest01.yuv", "rb");

    bool show_points = false;
    bool show_pointVelocities = false;
    bool show_velocityDistribution = false;
    bool do_fixMotion = true;

    for (int i=1; i<argc; ++i)
    {
        if (std::strcmp("--show-points", argv[i]) == 0) show_points = true;
        if (std::strcmp("--show-point-velocities", argv[i]) == 0) show_pointVelocities = true;
        if (std::strcmp("--show-velocity-distribution", argv[i]) == 0) show_velocityDistribution = true;
        if (std::strcmp("--dont-fix-motion", argv[i]) == 0) do_fixMotion = false;
        if (std::strcmp("--do-fix-motion", argv[i]) == 0) do_fixMotion = true;
    }

    y4m_frame_info_t frameinfo;
    y4m_stream_info_t streaminfo;

    // Read the header
    //
    int     errno = 0;
    int     frameWidth, frameHeight;
    uint8_t*frameBuffers[4] = {0,0,0,0};
    int     fpsNum, fpsDen;
    char    interlaceType;
    int     aspectNum;
    int     aspectDen;
    int     chromaMode;
    int     chromaSubSampleX;
    int     chromaSubSampleY;
    char*   colorSpace = new char[100];
    char*   token = new char[100];
    float   fixTranslateX = 0.0;
    float   fixTranslateY = 0.0;


    y4m_accept_extensions (1);
    y4m_init_stream_info (&streaminfo);
    y4m_init_frame_info (&frameinfo);

    if ((errno = y4m_read_stream_header (0, &streaminfo)) != Y4M_OK)
        mjpeg_error_exit1 ("Couldn't read YUV4MPEG header: %s!", y4m_strerr (errno));

    if (y4m_si_get_plane_count (&streaminfo) != 3)
        mjpeg_error_exit1 ("Only 3-plane formats supported.");

    frameWidth = y4m_si_get_width (&streaminfo);
    frameHeight = y4m_si_get_height (&streaminfo);
    chromaMode = y4m_si_get_chroma (&streaminfo);
    chromaSubSampleX = y4m_chroma_ss_x_ratio (chromaMode).d;
    chromaSubSampleY = y4m_chroma_ss_y_ratio (chromaMode).d;

    y4m_write_stream_header (1, &streaminfo);



    //scanf  ("YUV4MPEG2 W%d H%d F%d:%d I%c A%d:%d %s\n", &frameWidth, &frameHeight, &fpsNum, &fpsDen, &interlaceType, &aspectNum, &aspectDen, colorSpace);
    //scanf  ("YUV4MPEG2 W%d H%d F%d:%d I%c A%d:%d\n", &frameWidth, &frameHeight, &fpsNum, &fpsDen, &interlaceType, &aspectNum, &aspectDen);
    //printf ("YUV4MPEG2 W%d H%d F%d:%d I%c A%d:%d C420paldv\n", frameWidth, frameHeight, fpsNum, fpsDen, interlaceType, aspectNum, aspectDen);
    //printf ("YUV4MPEG2 W%d H%d F%d:%d I%c\n", frameWidth, frameHeight, fpsNum, fpsDen, interlaceType);

    buffers = new TrackingBuffers (frameWidth, frameHeight, chromaSubSampleX, chromaSubSampleY);

    while (errno = y4m_read_frame_header (0, &streaminfo, &frameinfo), errno == Y4M_OK)
    {
        // Read the frame
        //if (scanf ("%s\n", token) == EOF)
        //    break;
        //if (strcmp(token, "FRAME") != 0)
        //{
        //    cerr<< "Damn!: " << token << endl;
            //assert (strcmp(token, "FRAME") == 0);
        //}
        //printf ("FRAME\n");

        //char *Y, *U, *V;

        //buffers->getBuffers (Y,U,V);
        //fread (Y, frameWidth*frameHeight,   1, stdin);
        //fread (U, frameWidth*frameHeight/4, 1, stdin);
        //fread (V, frameWidth*frameHeight/4, 1, stdin);
        //buffers->prepareForRead ();
        buffers->getBuffers (frameBuffers[0], frameBuffers[1], frameBuffers[2]);
        y4m_read_frame_data (0, &streaminfo, &frameinfo, frameBuffers);
        buffers->prepareForRead ();

        // Update all (current) tracking points
        FeaturePoint::trackPoints (points, buffers->getBackLuma(), buffers->getFrontLuma(), buffers->getBackPyramid(), buffers->getFrontPyramid());


        // Kill bad points
        for (featurepoints_t::iterator I=points.begin(); I!=points.end();)
        {
            featurepoints_t::iterator J = I;
            ++I;
            if ( (*J)->getStatus() == DEAD)
            {
                savedPoints.push_back (*J);
                points.erase(J);
            }

        }

        // Paint all tracking points
        //

        // .. Paint the center just for reference
        // cvCircle (buffers->getImageRGB(), cvPoint(frameWidth/2,frameHeight/2), 6.0, CV_RGB(255,255,255), 1, CV_AA, 0 );

        // .. Paint all points
        for (featurepoints_t::iterator I=points.begin(); I!=points.end(); ++I)
        {
            // .. Paint circle where point is
            if (show_points)
            {
                cvCircle (buffers->getImageRGB(), (*I)->getPoint(), 1.0, CV_RGB(0,0,255), 1, CV_AA, 0 );
            }

            // .. Paint velocity vector
            if (show_pointVelocities)
            {
                CvPoint P1 = (*I)->getPoint();
                CvPoint P2 = P1;
                P2.x += (*I)->getVelocity().x;
                P2.y += (*I)->getVelocity().y;
                cvLine   (buffers->getImageRGB(),P1 , P2, (*I)->getColor(), 1, CV_AA, 0 );
            }

            // .. Paint a midscreen-based cluster of velocity vectors
            if (show_velocityDistribution)
            {
                CvPoint Pv = cvPoint (frameWidth/2,frameHeight/2);
                Pv.x += (*I)->getVelocity().x * 3;
                Pv.y += (*I)->getVelocity().y * 3;
                cvCircle (buffers->getImageRGB(), Pv, 1.0, CV_RGB(255,0,255), 1, CV_AA, 0 );
            }
        }


        // Paint a very odd picture: We want to assign a 2D (red+green) "color" to each pixel in the image.
        //  This color is a 2D vector (x,y) which is interpolated from all the velocity vectors from nearby feature points
        /*
        IplImage * Vimg = buffers->getImageRGB();
        unsigned char* Vbuf = (unsigned char*) Vimg->imageData;
        unsigned char* Vy   = Vbuf;
        unsigned char* Vx   = Vbuf;
        for (int y=0; y<576; ++y)
        {
            Vx = Vy;
            Vy += Vimg->widthStep;
            for (int x=0; x<720; ++x)
            {
                float dX = 0.0;
                float dY = 0.0;
                float w  = 0.0;
                for (featurepoints_t::iterator I=points.begin(); I!=points.end(); ++I)
                {
                    CvPoint2D32f Pp = (*I)->getLatestPos();
                    CvPoint2D32f Pv = (*I)->getVelocity();
                    float dist = sqrt ( (Pp.x-x)*(Pp.x-x) + (Pp.y-y)*(Pp.y-y) );
                    float wdist = 1.0 / (dist + 0.0001);
                    dX += Pv.x * wdist;
                    dY += Pv.y * wdist;
                    w  += wdist;
                }

                Vx[0] = (dX*255/w) + 128;
                Vx[1] = (dY*255/w) + 128;
                Vx[2] = 0;
                Vx += 3;
            }
        }
        */

        // Find average deviance for this frame
        float dX = 0.0;
        float dY = 0.0;
        float weight = 0.0;  // .. So we can weight the influence of each point

        for (featurepoints_t::iterator I=points.begin(); I!=points.end(); ++I)
            (*I)->updateDeltas (buffers->getCounter(), dX, dY, weight);

        if (weight > 0.0)
        {
            dX /= weight;
            dY /= weight;
            if (! (std::abs(dX)>50 || std::abs(dY)>50))
            {
                fixTranslateX -= dX;
                fixTranslateY -= dY;
            }
            fixTranslateX *= 0.95;  // Slowly drift back again
            fixTranslateY *= 0.95;  // - " -
        }

        // Translate our frame according to our tracked deviance
        if (do_fixMotion)
        {
            buffers->translateRGB (fixTranslateX, fixTranslateY);
        }

        mjpeg_log ((log_level_t)LOG_INFO, "X,Y motion in this frame: %f, %f. Current translation: %f, %f", dX, dY, fixTranslateX, fixTranslateY);

        // If we don't have enough feature points to track, then make some more (just through them into the pool on top of the existing ones)
        if (points.size() < 500)
        {
            featurepoints_t newPoints = FeaturePoint::findNew (1000, buffers->getFrontLuma(), buffers->getCounter(), buffers->getSize());
            copy (newPoints.begin(), newPoints.end(), back_inserter(points));
        }

        buffers->prepareForWrite ();
        buffers->getBuffers (frameBuffers[0], frameBuffers[1], frameBuffers[2]);
        y4m_write_frame (1, &streaminfo, &frameinfo, frameBuffers);

        //buffers->prepareForWrite ();
        //buffers->getBuffers (Y,U,V);
        //fwrite (Y, frameWidth*frameHeight,   1, stdout);
        //fwrite (U, frameWidth*frameHeight/4, 1, stdout);
        //fwrite (V, frameWidth*frameHeight/4, 1, stdout);
    }

    // Done now. Save history of all points. Start by copying the still living ones to the saved pool
    copy (points.begin(), points.end(), back_inserter(savedPoints));

    velocity_statistics velocityStat;

    ofstream f ("testclip.data");
    int ip = 0;
    for (featurepoints_t::iterator I=savedPoints.begin(); I!=savedPoints.end(); ++I)
    {
        FeaturePoint* P = *I;
        f << format("%d %d %d %d") % ip % P->getFirstFrame() % P->getLastFrame() % P->getPointCount() << endl;

        for (points_t::iterator J=P->getPointsBeginIt(); J!=P->getPointsEndIt(); ++J)
            f << format("  %f %f") % J->x % J->y << endl;

        int jp = 0;
        for (points_t::iterator J=P->getPointsBeginIt()+1; J!=P->getPointsEndIt(); ++J)
        {
            points_t::iterator JJ = J-1;
            velocityStat.insert (make_pair ( P->getFirstFrame() + jp , make_pair(J->x - JJ->x , J->y - JJ->y ) ));
            ++jp;
        }
        ++ip;
    }
    int firstFrame = min_element (velocityStat.begin(), velocityStat.end(), isFrameLess()) -> first;
    int lastFrame  = max_element (velocityStat.begin(), velocityStat.end(), isFrameLess()) -> first;
    ofstream fv ("testclip.veloc.data");
    for (int f=firstFrame; f<=lastFrame; ++f)
    {
        fv << format("%d") % f << endl;
        for (velocity_statistics::iterator V=velocityStat.lower_bound(f); V!=velocityStat.upper_bound(f); ++V)
        {
            fv << format("  %f, %f;") % V->second.first % V->second.second << endl;

        }
    }
}





