#!/usr/bin/python

# Handle http requests for XRT
# Copyright (C) 2006 by Tapsell-Ferrier Limited

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; see the file COPYING.  If not, write to the
# Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
# Boston, MA  02110-1301  USA

"""HTTP adaptor for XRT"""


import string
import re
import libxml2
import logging
import xslt
import StringIO
import sys
import os
import glob
import mimetypes
import urllib


HTTP_REQUEST_NS = "http://www.tapsellferrier.co.uk/namespace/xslt_http_ns"

def eval_http_xpath(context, query):
    """Eval the query within the http namespace"""

    xpath_ctx = None

    if isinstance(context, libxml2.xmlDoc):
        xpath_ctx = context.xpathNewContext()
        xpath_ctx.setContextNode(context.getRootElement())
    else:
        xpath_ctx = context.get_doc().xpathNewContext()
        xpath_ctx.setContextNode(context)

    node = None
    try:
        xpath_ctx.xpathRegisterNs("http", HTTP_REQUEST_NS)
    except:
        logger.debug("namespace error? ")
    else:
        try:
            node = xpath_ctx.xpathEval(query)
        except libxml2.xpathError, e:
            logger.debug("couldn't eval query")

    xpath_ctx.xpathFreeContext()
    return node


# Utility method
def iterfieldstorage(fs):
    keys = fs.keys()
    while keys:
        key = keys.pop()
        value = fs[key].value
        yield key,value
    return


import email.MIMENonMultipart

def http_handle(xrt_path, method, uri, headers, parameters):
    """handle the HTTP request with xslt.

    The elements of the request are packaged into a namespaced
    document and passed to the default xslt.

    The result document is collected, the http namespaced elements
    removed and returned with the result document.

    """
    # The DOM for the request
    request_dom = libxml2.newDoc("1.0")

    root = request_dom.newChild(None, "request", None)
    ns = root.newNs(HTTP_REQUEST_NS, "http")
    root.setNs(ns)

    # Add the method
    root.newChild(ns, "method", method)

    # Add a uri node
    node = root.newChild(ns, "uri", uri)
    # Convert the request path
    ### I'm not so sure about this...
    ### user code could just split the uri when it needed to match like this
    path_parts = re.split("/", uri)[1:]
    if path_parts:
        node = root.newChild(ns, "path", None)
        for part in path_parts:
            node.newChild(ns, "part", "/" + part)

    # Convert the headers
    if headers:
        node = root.newChild(ns, "headers", None)
        for hdr,val in headers.iteritems():
            header = node.newChild(ns, "header", None)
            header.newChild(ns, "name", hdr)
            header.newChild(ns, "value", val)

    # Convert the parameters
    if parameters:
        node = root.newChild(ns, "parameters", None)
        for param,val in iterfieldstorage(parameters):  ## FieldStorage doesn't support iteration
            parameter = node.newChild(ns, "parameter", None)
            parameter.newProp("name", param)
            parameter.newChild(ns, "name", param)
            parameter.newChild(ns, "value", val)

    # Call XSLT to handle the request
    output_dom = libxml2.newDoc("1.0")

    # Process the xslt
    xslt.xslt(xrt_path, output_dom, request_dom)

    # Setup defaults
    result = email.MIMENonMultipart.MIMENonMultipart("text", "html")
    status_code = 200

    # The XSLT could cause a file to be returned
    file_required = False
    file_to_return = None

    # Process any http response we can find
    http_response_nodes = eval_http_xpath(output_dom, "//http:response/*")
    for node in http_response_nodes:
        name = node.get_name()
        if name == "status":
            status_code = int(node.get_content())
        elif name == "header":
            try:
                hdr_name = eval_http_xpath(node, ".//http:name")[0].get_content()
                hdr_value = eval_http_xpath(node, ".//http:value")[0].get_content()
            except:
                logger.log("failure getting header name or value")
            else:
                if result[hdr_name]:
                    del result[hdr_name]
                result[hdr_name] = hdr_value
        elif name == "file":
            file_required = True
            sel = node.prop("select")
            if sel == "uri":
                file_to_return = glob.glob(os.getcwd() + uri)

                ### FIXME!!!
                ### 404 errors!!!
                ### <http:file> should be an extension element for this reason
                ### so that the stylesheet can specify a handler for 404
                ###
                ### For now this is all we can do
                if not file_to_return:
                    status_code = 404
                
                # Set the mime type accordingly
                try:
                    mt = mimetypes.guess_type(file_to_return[0])
                except:
                    # Unkown mimetype?
                    logger.error("unknown mimtype?")
                    del result["content-type"]
                    result["content-type"] = "text/plain"
                else:
                    charset_arg = ""
                    if mt[1]:
                        charset_arg = "; charset=%s" % (mt[1])
                    del result["content-type"]
                    result["content-type"] = "%s%s" % (mt[0], charset_arg)

    # Strip out any http ns stuff
    namespace_strip = 1
    if namespace_strip and not file_required:
        http_result = eval_http_xpath(output_dom, "//http:response")
        if http_result:
            # Take off the main result node
            http_result[0].unlinkNode()    ## FIXME:: should we remove all in the sequence
            # Take off the http namespace now we've removed the nodes
            output_dom.getRootElement().removeNsDef(HTTP_REQUEST_NS)

    if file_required:
        if status_code == 200:
            fd = None
            try:
                try:
                    fd = open(file_to_return[0])
                except:
                    logger.error("tried to return %s" % (file_to_return[0]))
                else:
                    result.set_payload(string.join([line for line in fd]))
            finally:
                if fd:
                    fd.close()
    else:
        # Add the dom to the result.
        str_out = StringIO.StringIO()
        xml_buf = libxml2.createOutputBuffer(str_out, 'UTF-8')
        output_dom.saveFormatFileTo(xml_buf, 'UTF-8', 1)
        result.set_payload(str_out.getvalue())
        str_out.close()
    
    return (status_code, result)


if __name__ == "__main__":
    import pdb

    def run():
        method = os.environ["HTTP_METHOD"]
        uri = os.environ["HTTP_URI"]

        headers = {}
        if os.environ.has_key("HTTP_HEADER"):
            for name,value in [ (hdr.split(":")) for hdr in os.environ["HTTP_HEADER"].split("\r") ]:
                headers[name] = value

        # We should read this from stdin really
        parameters = {}
        if os.environ.has_key("HTTP_PARAMS"):
            for name,value in [ (hdr.split(":")) for hdr in os.environ["HTTP_PARAMS"].split("\r") ]:
                parameters[name] = value

        status_code, result = http_handle(os.getcwd() + "/top.xslt",
                                          method,
                                          uri,
                                          headers,
                                          parameters)
        print "status: %d\n%s" % (status_code, result.as_string())
        return

    hdlr = logging.StreamHandler(sys.stderr)
    hdlr.setFormatter(logging.Formatter('%(asctime)s %(name)s %(lineno)d %(message)s'))
    logging.getLogger().addHandler(hdlr)
    logging.getLogger().setLevel(logging.ERROR)


    sys.path = sys.path + [ os.getcwd() ]
    run()

# End
