/*
  xpcomp_builtin.c - test builtin for xpcomp

  Copyright (C) 2001 Ingo K"ohne
  
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
*/

#include "defines.h"

#include <errno.h>
#include <stdio.h>
#include <signal.h>		/* kill() */
#include <stdlib.h>		/* rand() */
#include <ctype.h>		/* isprint() */
#include <stdarg.h>		/*  */
#include <time.h>		/* time() */

#include "bashansi.h"		/* malloc() */
#include "shell.h"
#include "builtins.h"
#include "builtins/bashgetopt.h"
#include "builtins/common.h"	/* builtin_error */
#include "posixstat.h"		/* stat () */

#include "xpcomp.h"


/* possible actions of this builtin */
typedef enum xpc_command
{
  XPC_UNKNOWN,
  XPC_TEST,
  XPC_SHOW
}
XPC_COMMAND;

typedef enum
{
  test_passed,
  test_resume,
  test_failed
}
test_result_t;


/* test function signature */
typedef test_result_t (*test_function_t) (void);

typedef struct
{
  char *name;
  test_function_t testfn;
}
test_t;



/* THIS time i try to have as much global variables as possible :)*/

/* log file for informational messegas of this builtin
   they will go also to stderr */
static char *logname = NULL;
static FILE *logfile = NULL;
/* script file that is on stdin of the bash in which this builtin runs */
static char *scriptname = NULL;
static FILE *scriptfile = NULL;

/* line buffer for composing bash input lines (mostly) */
static const int bufflen = 32000;
static char *buff = NULL;

/* array of tests named on the commandline
   (this is zero on first entry) */
static test_t **tests_to_do = NULL;
/* loop pointer */
static test_t **current_test = NULL;
/* how many times the loop has been run */
//static int loops = 0;

extern int install_rl_event_hook;



#define rl_set_mark "\x1b "
#define rl_beginning_of_line "\x01"
#define rl_backward_char "\x02"
#define rl_forward_char "\x06"
#define rl_complete "\x09"
#define rl_kill_line "\x0b"
#define rl_unix_line_discard "\x15"
#define rl_exchange_point_and_mark "\x18\x18"




static void
log_error (const char *template, ...)
{
  va_list ap;
  int log_failure = 0;

  assert (template);

  errno = 0;
  if (logfile)
    {
      va_start (ap, template);
      vfprintf (logfile, template, ap);
      va_end (ap);
      fprintf (logfile, "\n");
    }

  if (errno)
    {
      fprintf (stderr, "xpcomptest: failed write to logfile: %s",
	       strerror (errno));
      errno = 0;
      log_failure = 1;
    }

  fprintf (stderr, "xpcomptest: ");
  va_start (ap, template);
  vfprintf (stderr, template, ap);
  va_end (ap);
  fprintf (stderr, "\n");

  fflush (NULL);

  if (errno && log_failure)
    exit (1);
}



static int
bash_in_printf (const char *template, ...)
{
  va_list ap;
  assert (template);

  errno = 0;
  if (scriptfile)
    {
      va_start (ap, template);
      vfprintf (scriptfile, template, ap);
      va_end (ap);
    }
  else
    {
      log_error ("Ooops, script FILE* went NULL");
      return EXECUTION_FAILURE;
    }

  if (errno)
    {
      log_error ("could not write to script file: %s", strerror (errno));
      return EXECUTION_FAILURE;
    }
  return EXECUTION_SUCCESS;
}



static int
bash_in (const char *str)
{

  errno = 0;
  assert (str);

  if (scriptfile)
    {
      fprintf (scriptfile, "%s", str);
    }
  else
    {
      log_error ("Ooops, script FILE* went NULL");
      //kill(0,SIGKILL);
      return EXECUTION_FAILURE;
    }

  if (errno)
    {
      log_error ("could not write to script file: %s", strerror (errno));
      fprintf (stderr, "%s\n", str);
      //kill(0,SIGKILL);
      return EXECUTION_FAILURE;
    }
  return EXECUTION_SUCCESS;
}



static int
buff_sprintf (char **pointer, const char *template, ...)
{
  size_t formatted_len;
  size_t max_len;

  va_list ap;

  assert (pointer);
  assert (buff <= *pointer);
  assert (*pointer < buff + bufflen);

  max_len = buff + bufflen - *pointer;

  va_start (ap, template);
  formatted_len = vsnprintf (*pointer, max_len, template, ap);
  va_end (ap);

  if (formatted_len > max_len)
    return EXECUTION_FAILURE;


  *pointer += formatted_len;

  return EXECUTION_SUCCESS;
}



static int
send_tab_on_all_chars (char *str)
{
  int i;
  int len = strlen (str);

  if (len < 2)
    {
      bash_in (rl_beginning_of_line rl_kill_line);
      return EXECUTION_SUCCESS;
    }

  /* skip first two characters at the beginning of the line */
  bash_in (rl_beginning_of_line rl_forward_char rl_forward_char rl_set_mark);

  for (i = 2; i < len; i++)
    {
      bash_in (rl_kill_line);

      bash_in (str + i);

      bash_in (rl_exchange_point_and_mark rl_forward_char
	       rl_set_mark rl_complete);
    }

  bash_in (rl_beginning_of_line rl_kill_line);

  return EXECUTION_SUCCESS;
}



static int
buff_send ()
{
  int len;

  /* make sure there is a terminating zero. */
  for (len = 0; len < bufflen; len++)
    if (*(buff + len) == 0)
      break;
  if (len == bufflen)
    return EXECUTION_FAILURE;

  bash_in (buff);
  return EXECUTION_SUCCESS;
}



static inline XPC_COUNT
rand_between (XPC_COUNT min, XPC_COUNT max)
{
  assert (min <= max);
  if (min == max)
    return min;
  else
    return (XPC_COUNT) min + rand () / (RAND_MAX / (max - min) - 1);
}



static inline XPC_CMD *
xpc_cmd_random ()
{
  return GETCMD (rand_between (1, cache->idx->cmd_count));
}



static inline XPC_OPT *
xpc_cmd_opt_random (XPC_CMD * c)
{
  if (c->opt_count == 0)
    return NULL;

  return GETOPT (c->opts + rand_between (0, c->opt_count - 1));
}



static void
random_word (char **pointer)
{
  buff_sprintf (pointer, " ");
  do
    {
      char c;
      do
	{
	  c = (char) (rand () / (RAND_MAX / 128));
	}
      while (c == '\n' || !isprint (c));

      buff_sprintf (pointer, "%c", c);
    }
  while (rand () > (RAND_MAX / 2));

}



static void
compose_random_cmdline ()
{
  char *cp;
  char *pointer = buff;
  XPC_CMD *c = xpc_cmd_random ();

  if (c->opt_count < 10)
    c = xpc_cmd_random ();

  buff_sprintf (&pointer, GETCHR (c->name));
  cp = strchr (buff, 0x1);
  if (cp)
    *cp = ' ';


  while (rand () > RAND_MAX / 4)
    {
      XPC_OPT *o = xpc_cmd_opt_random (c);
      char *optname = GETCHR (o->name);

      if (optname[0] == 0x1)
	continue;

      if (optname[1] == 0)
	buff_sprintf (&pointer, " -%s", optname);
      else if (c->mode.longopt1dash
	       && !(c->mode.longopt2dash && rand () > RAND_MAX / 2))
	buff_sprintf (&pointer, " -%s", optname);
      else if (c->mode.longopt2dash)
	buff_sprintf (&pointer, " --%s", optname);

      if (o->mode.arg == arg_required
	  || (o->mode.arg == arg_optional && rand () > RAND_MAX / 2))
	random_word (&pointer);
    }
  while (rand () > RAND_MAX / 3)
    random_word (&pointer);

}


/* These are the various test functions:

   (1) Return values: test_failed, test_resume, test_passed

   (2) If you have a static variable that holds the state of the test remeber
       to reset it to its initial value when the test is finished.  Otherwise
       the test can not run twice.

   (3) It is propably necessary to distinguish between initialization that is
       done once when the builtin has been loaded, and the initialization per
       execution of the test

*/

static test_result_t
test_random ()
{
  static enum
  {
    init,
    start,
    test,
    exit
  }
  state = init;

  static int lines = 0;

  switch (state)
    {
    case init:
      bash_in (rl_beginning_of_line rl_kill_line
	       "source scripts/loadxpcomp.sh\n");
      state = start;
      break;

    case start:

      bash_in ("exithandler () "
	       "{ rm -f /tmp/cache$$; touch /tmp/unexpected_exit$$; }\n"
	       "trap exithandler EXIT\n" "set -e\ncd $test_dir\n");
      state = test;
      break;

    case test:

      cache = heap.next;
      if (!cache)
	{
	  /* somebody stole the cache (xpcompcore probably unloaded) */
	  state = exit;
	  break;
	}

      compose_random_cmdline ();
      buff_send (buff);
      send_tab_on_all_chars (buff);

      lines++;

      if (lines >= 100)
	state = exit;

      break;

    case exit:

      state = start;
      lines = 0;
      bash_in (rl_beginning_of_line rl_kill_line "trap - EXIT\n");
      return test_passed;
    }

  return test_resume;
}



static test_result_t
test_bla ()
{
  fprintf (stderr, "executing bla\n");
  return test_passed;
}



static void
finish ()
{
  fflush (NULL);
  bash_in ("\nxpcomptest\n");
  fflush (NULL);
}



test_t test[] = {
  {
   "random",
   test_random},
  {
   "bla",
   test_bla},
  /* end mark, do not remove: */
  {
   NULL,
   NULL}
};



static void
exit_bash (int exitval)
{
  if (buff)
    free (buff);
  if (scriptname)
    free (scriptname);
  if (tests_to_do)
    free (tests_to_do);

  bash_in_printf ("\nxpcomp -X; rm -f /tmp/cache$$; exit %i\n", exitval);
  fflush (NULL);
}

static int
xpcomptest_builtin (args)
     WORD_LIST *args;
{
  int c;
  int last_result;
  WORD_LIST *wl;
  XPC_COMMAND command;
  struct stat st;

  /* loop counter; for easy detecting wrong syntax */
  int getopt_count;

  if (xpcomp_state)
    switch (xpcomp_state)
      {
      case XPC_UNINITIALIZED:

	errstream = stderr;
	if (xpc_libinit ())
	  return EXECUTION_FAILURE;
	break;

      default:

	log_error ("xpcomptest: xpcomp_state is %i", xpcomp_state);
	return EXECUTION_FAILURE;

	break;
      }

  errno = 0;

  reset_internal_getopt ();

  c = 0;
  last_result = EXECUTION_SUCCESS;
  command = XPC_UNKNOWN;
  getopt_count = 0;

  if (current_test)
    {
      fflush (NULL);
      if (*current_test)
	switch (((*current_test)->testfn) ())
	  {
	  case test_failed:
	    exit_bash (1);
	    break;

	  case test_passed:
	    current_test++;

	  case test_resume:
	    finish ();
	  }
      else
	exit_bash (0);

      return EXECUTION_SUCCESS;
    }


  if (!buff)
    {
      srand (time (0));
      buff = malloc (bufflen);
      xpc_disable_if_not (buff);
      install_rl_event_hook = 0;
    }

  while (!last_result && (c = internal_getopt (args, "f:l:m:n:")) != -1)
    {
      switch (c)
	{
	case 'f':
	  scriptname = list_optarg;
	  break;

	case 'm':
	  break;

	case 'n':
	  break;

	case 'l':
	  logname = list_optarg;
	  break;

	default:
	  builtin_usage ();
	  last_result = EX_USAGE;
	}
    }

  if (!scriptname)
    {
      int len;
      errno = 0;

      len = readlink ("/proc/self/fd/0", buff, bufflen);

      if (len > 0)
	{
	  buff[len] = 0;
	  scriptname = strdup (buff);
	  xpc_disable_if_not (scriptname);
	}
      else
	{
	  log_error ("Could not read /proc/self/fd/0 "
		     "to determine name of input fifo: %s", strerror (errno));
	  return EXECUTION_FAILURE;
	}
    }

  errno = 0;
  if (stat (scriptname, &st))
    {
      log_error ("Could not stat %s, "
		 "standard input probably not a regular file: %s", scriptname,
		 strerror (errno));
      return EXECUTION_FAILURE;
    }

  if (!S_ISREG (st.st_mode))
    {
      builtin_error ("%s is not a regular file.", scriptname);
      return EXECUTION_FAILURE;
    }

  errno = 0;
  scriptfile = fopen (scriptname, "a+");

  if (!scriptfile)
    {
      builtin_error ("unable to open standard input: %s", strerror (errno));
      return EXECUTION_FAILURE;
    }

  if (logname && !*logname)
    {
      builtin_error ("Empty log file name.");
      last_result = EX_USAGE;
    }

  if (loptend)
    {
      command = XPC_TEST;
    }

  if (!last_result)
    {
      int i;
      int k;

      switch (command)
	{
	case XPC_TEST:

	  for (i = 0, wl = loptend; wl; i++, wl = wl->next);

	  tests_to_do = (test_t **) malloc ((i + 1) * sizeof (test_t *));

	  for (i = 0, wl = loptend; wl; i++, wl = wl->next)
	    {
	      for (k = 0; test[k].name; k++)
		if (strcmp (wl->word->word, test[k].name) == 0)
		  tests_to_do[i] = &test[k];
	    }
	  tests_to_do[i] = NULL;

	  current_test = &tests_to_do[0];
	  finish ();
	  break;

	case XPC_UNKNOWN:

	  builtin_usage ();
	  last_result = EX_USAGE;
	  break;

	default:

	  builtin_error ("Option -%c not yet implemented", c);
	  last_result = EXECUTION_SUCCESS;
	}
    }

  if (last_result)
    {
      builtin_error ("xpcomptest failure");
      fflush (NULL);
    }

  return last_result;
}



static char *xpcomptest_doc[] = {
  "internal builtin used for stress testing the xpcomp builtins.",
  (char *) NULL
};

struct builtin xpcomptest_struct = {
  "xpcomptest",
  xpcomptest_builtin,
  BUILTIN_ENABLED,
  xpcomptest_doc,
  "xpcomptest [-m N] [-n N] [-f filename] [-l filename] [-p]",
  (char *) 0
};
