/*
  xpcomp_builtin.c - perform global configuration of xpcomp

  Copyright (C) 2001 Ingo K"ohne
  
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
*/

#include "defines.h"

#if defined (HAVE_UNISTD_H)
#  include <sys/unistd.h>	/* getuid() */
#endif
#if defined(HAVE_MMAP)
#  include <sys/mman.h>
#  include <fcntl.h>
#endif
#include <errno.h>
#include <stdio.h>

#include "bashansi.h"		/* malloc() */
#include "shell.h"
#include "builtins.h"
#include "builtins/bashgetopt.h"
#include "builtins/common.h"	/* builtin_error */
#include "posixstat.h"		/* stat () */

#include "xpcomp.h"

typedef enum xpc_command
{
  XPC_UNKNOWN = 0,
  XPC_RESTRICT,
  XPC_DEBUG,
  XPC_REMOVE,
  XPC_WRITE,
  XPC_LOAD,
  XPC_UNLOAD,
  XPC_VERSION,
  XPC_SHOW,
  XPC_CLEAN
}
XPC_COMMAND;

typedef struct xpc_mmap_data
{
  void *loadaddr;
  FILE *file;
  struct stat st;
}
XPC_MMAP_DATA;

static const char *MSG_EXCLUSIVE =
  "The options -r -l -u -w -D -R -X are mutually exclusive.";



static int
cache_write (const char *filename)
{
#if defined(HAVE_MMAP)

  FILE *file;
  unsigned long int checksum;

  xpc_cache_set (&heap);

  assert (cache && cache->idx);

  if (!filename)
    return EXECUTION_FAILURE;

  xpc_cache_gc ();

  if (!strncpy (cache->idx->magic, XPC_MAGIC, XPC_MAGIC_LEN))
    return EXECUTION_FAILURE;

  cache->idx->major = XPC_MAJOR;
  cache->idx->minor = XPC_MINOR;

  file = fopen (filename, "w");
  if (!file)
    return EXECUTION_FAILURE;

  if (1 != fwrite (cache->idx, sizeof (XPC_INDEX), 1, file))
    return EXECUTION_FAILURE;

  if (cache->idx->cmd_count
      != fwrite (GETCMD (0), sizeof (XPC_CMD), cache->idx->cmd_count, file))
    return EXECUTION_FAILURE;

  if (cache->idx->opt_count
      != fwrite (GETOPT (0), sizeof (XPC_OPT), cache->idx->opt_count, file))
    return EXECUTION_FAILURE;

  if (cache->idx->compspec_count
      != fwrite (GETCOMPSPEC (0), sizeof (COMPSPEC),
		 cache->idx->compspec_count, file))
    return EXECUTION_FAILURE;

  if (cache->idx->chr_count
      != fwrite (GETCHR (0), sizeof (XPC_CHR), cache->idx->chr_count, file))
    return EXECUTION_FAILURE;

  checksum = xpc_cache_checksum ();
  if (1 != fwrite (&checksum, sizeof (unsigned long int), 1, file))
    return EXECUTION_FAILURE;

  if (fclose (file))
    return EXECUTION_FAILURE;
  return EXECUTION_SUCCESS;

#else /* ! defined(HAVE_MMAP) */
  builtin_error ("Missing mmap(2). Cache files not supported.");
  return EXECUTION_FAILURE;
#endif
}



static void
mmap_cleanup (XPC_MMAP_DATA * md)
{
  if (md->loadaddr)
    munmap (md->loadaddr, md->st.st_size);
  free (cache);
  cache = NULL;
  fclose (md->file);
}



static int
cache_load (const char *filename)
{
#if defined(HAVE_MMAP)

  XPC_MMAP_DATA md;

  memset (&md, 0, sizeof (XPC_MMAP_DATA));

  if (stat (filename, &md.st))
    return EXECUTION_FAILURE;

  if (!S_ISREG (md.st.st_mode))
    {
      builtin_error ("%s is not a regular file.", filename);
      return EXECUTION_FAILURE;
    }

  if (md.st.st_uid && (getuid () != md.st.st_uid))
    {
      builtin_error ("%s must be owned by you or by root.", filename);
      return EXECUTION_FAILURE;
    }

  if (md.st.st_mode & (S_IWGRP | S_IWOTH))
    {
      builtin_error ("%s must not be writeable by group or other.", filename);
      return EXECUTION_FAILURE;
    }

  if ((unsigned int) md.st.st_size < sizeof (XPC_INDEX))
    {
      builtin_error ("%s corrupted (too small).", filename);
      return EXECUTION_FAILURE;
    }

  md.file = fopen (filename, "r");
  if (!md.file)
    {
      builtin_error ("Could not open file %s.", filename);
      return EXECUTION_FAILURE;
    }

  cache = (XPC_CACHE *) malloc (sizeof (XPC_CACHE));
  xpc_disable_if_not (cache);

  begin_unwind_frame ("xpcomp_builtin_cache_load");
  add_unwind_protect (mmap_cleanup, &md);

  md.loadaddr =
    mmap (NULL, md.st.st_size, PROT_READ, MAP_PRIVATE, fileno (md.file), 0);

  if (!md.loadaddr)
    {
      builtin_error ("Could not map file to virtual memory.");
      run_unwind_frame ("xpcomp_builtin_cache_load");
      return EXECUTION_FAILURE;
    }

  if (-1 == fcntl (fileno (md.file), F_SETFD, FD_CLOEXEC))
    builtin_error ("Error setting close-on-exec flag for mmaped file.");

  cache->idx = (XPC_INDEX *) md.loadaddr;

  if (strcmp (cache->idx->magic, XPC_MAGIC))
    {
      builtin_error ("File '%s' has not the right magic "
		     "(propably not an xpcomp cache file)", filename);
      run_unwind_frame ("xpcomp_builtin_cache_load");
      return EXECUTION_FAILURE;
    }

  if (cache->idx->major != XPC_MAJOR)
    {
      builtin_error ("Incompatible cache file '%s'"
		     "(major version is %i and should be %i).",
		     filename, cache->idx->major, XPC_MAJOR);
      run_unwind_frame ("xpcomp_builtin_cache_load");
      return EXECUTION_FAILURE;
    }

  if (cache->idx->minor > XPC_MINOR)
    {
      builtin_error ("Incompatible cache file '%s' "
		     "(minor version is %i and should be <=%i).",
		     filename, cache->idx->minor, XPC_MINOR);
      run_unwind_frame ("xpcomp_builtin_cache_load");
      return EXECUTION_FAILURE;
    }


  if ((unsigned int) md.st.st_size != sizeof (XPC_INDEX)
      + sizeof (XPC_CMD) * cache->idx->cmd_count
      + sizeof (XPC_OPT) * cache->idx->opt_count
      + sizeof (COMPSPEC) * cache->idx->compspec_count
      + sizeof (XPC_CHR) * cache->idx->chr_count + sizeof (unsigned long int))
    {
      builtin_error
	("Cache file '%s' corrupted "
	 "(description in index does not match length).", filename);
      run_unwind_frame ("xpcomp_builtin_cache_load");
      return EXECUTION_FAILURE;
    }

  if (cache->idx->cmd_count == 0)
    return EXECUTION_SUCCESS;

  cache->cmds = (XPC_CMD *) (cache->idx + 1);

  cache->opts = (XPC_OPT *) (GETCMD (cache->idx->cmd_count));

  cache->compspecs = (COMPSPEC *) (GETOPT (cache->idx->opt_count));

  cache->chrs = (XPC_CHR *) (GETCOMPSPEC (cache->idx->compspec_count));

  cache->checksum = *(unsigned long int *) (GETCHR (cache->idx->chr_count));

  if (xpc_cache_validate ())
    {
      builtin_error ("Cache file '%s' corrupted (inconsistent pointers).",
		     filename);
      run_unwind_frame ("xpcomp_builtin_cache_load");
      return EXECUTION_FAILURE;
    }


  if (xpc_cache_checksum () != cache->checksum)
    {
      builtin_error ("Cache file '%s' corrupted (invalid checksum).",
		     filename);
      run_unwind_frame ("xpcomp_builtin_cache_load");
      return EXECUTION_FAILURE;
    }


  cache->filename = strdup (filename);
  xpc_disable_if_not (cache->filename);

  discard_unwind_frame ("xpcomp_builtin_cache_load");

  cache->next = 0;

  /* link new cache */
  {
    XPC_CACHE *c;
    c = &heap;
    while (c->next)
      c = c->next;
    c->next = cache;
  }
  xpc_cache_set (NULL);

  return EXECUTION_SUCCESS;

#else /* ! defined(HAVE_MMAP) */
  builtin_error ("Missing mmap(2). Cache files not supported.");
  return EXECUTION_FAILURE;
#endif
}



static int
cache_unload (const char *filename)
{
#if defined(HAVE_MMAP)

  XPC_CACHE *ch;

  assert (filename && *filename);

  for (ch = &heap; ch->next || (ch = NULL); ch = ch->next)
    if (!strcmp (ch->next->filename, filename))
      {
	XPC_CACHE *old = ch->next;
	ch->next = ch->next->next;
	ch = old;
	break;
      }

  if (!ch)
    return EXECUTION_SUCCESS;

  munmap (ch->idx, sizeof (XPC_INDEX)
	  + sizeof (XPC_CMD) * ch->idx->cmd_count
	  + sizeof (XPC_OPT) * ch->idx->opt_count
	  + sizeof (COMPSPEC) * ch->idx->compspec_count
	  + sizeof (XPC_CHR) * ch->idx->chr_count
	  + sizeof (unsigned long int));

  free (ch->filename);
  free (ch);

  xpc_cache_set (NULL);
  return EXECUTION_SUCCESS;

#else /* ! defined(HAVE_MMAP) */
  builtin_error ("Missing mmap(2). Cache files not supported.");
  return EXECUTION_FAILURE;
#endif
}



static void
printcmds (void)
{
  xpc_cache_set (NULL);

  for (cache = &heap; cache; cache = cache->next)
    {
      XPC_COUNT n;

      assert (cache->idx);

      if (cache->idx->cmd_count < 2)
	continue;

      if (cache->filename)
	printf ("%s:", cache->filename);
      else
	printf ("session:");

      for (n = 1; n < cache->idx->cmd_count; n++)
	{
	  XPC_COUNT name = GETCMD (n)->name;
	  if (!strchr (GETCHR (name), 0x1))
	    printf (" %s", GETCHR (name));
	}

      printf ("\n");
    }
}


static int
xpcomp_builtin (args)
     WORD_LIST *args;
{
  int c;
  int res;
  char *filename;
  XPC_COMMAND command;

  /* loop counter; for easy detecting wrong syntax */
  int getopt_count;

  if (setjmp (self_destruct))
    {
      xpcomp_state = XPC_DISABLED;
      fprintf (stderr,
	       "The xpcomp builtins have been disabled.  "
	       "Further calls will silently fail.\n");
      return EXECUTION_FAILURE;
    }

  if (xpcomp_state)
    switch (xpcomp_state)
      {
      case XPC_UNINITIALIZED:

	errstream = stderr;
	if (xpc_libinit ())
	  return EXECUTION_FAILURE;
	break;

      case XPC_EXPANDING:

	return EXECUTION_FAILURE;

      case XPC_DISABLED:

	return EXECUTION_FAILURE;

      default:

	break;
      }

  errno = 0;

  reset_internal_getopt ();

  c = 0;
  res = EXECUTION_SUCCESS;
  filename = NULL;
  command = XPC_UNKNOWN;
  getopt_count = 0;

  while (!res && (c = internal_getopt (args, "RD;l:u;vw:Xrp")) != -1)
    {
      getopt_count++;

      if (command)
	{
	  builtin_error (MSG_EXCLUSIVE);
	  res = EX_USAGE;
	  break;
	}

      switch (c)
	{
	case 'D':

	  filename = list_optarg;
	  command = XPC_DEBUG;
	  break;

	case 'R':

	  command = XPC_RESTRICT;
	  break;

	case 'l':

	  filename = list_optarg;
	  command = XPC_LOAD;
	  break;

	case 'u':

	  filename = list_optarg;
	  command = XPC_UNLOAD;
	  break;

	case 'r':

	  command = XPC_REMOVE;
	  break;

	case 'p':

	  command = XPC_SHOW;
	  break;

	case 'v':

	  command = XPC_VERSION;
	  break;

	case 'w':

	  filename = list_optarg;
	  command = XPC_WRITE;
	  break;

	case 'X':

	  command = XPC_CLEAN;
	  break;

	default:

	  builtin_usage ();
	  res = EX_USAGE;
	}
    }

  if (filename && !*filename)
    {
      builtin_error ("Empty filename.");
      res = EX_USAGE;
    }

  if (getopt_count != 1 || loptend)
    {
      builtin_usage ();
      res = EX_USAGE;
    }

  if (!res)
    {
      switch (command)
	{
	case XPC_DEBUG:

	  if (errstream && errstream != stderr)
	    fclose (errstream);

	  if (filename)
	    {
	      errstream = fopen (filename, "w");
	      if (!errstream)
		res = EXECUTION_FAILURE;
	      else
		{
		  xpc_debug = 1;
		  printf ("Debug messages to file `%s'\n", filename);
		}
	    }
	  else
	    {
	      xpc_debug = !xpc_debug;
	      printf ("Debug messages %s\n", xpc_debug ? "ON" : "OFF");
	      res = EXECUTION_SUCCESS;
	    }
	  break;

	case XPC_RESTRICT:
#if defined(RESTRICTED_SHELL)
	  xpc_restrict = !xpc_restrict;
	  printf ("Restrictive mode %s\n", xpc_restrict ? "ON" : "OFF");
	  res = EXECUTION_SUCCESS;
#else
	  printf ("Restrictive mode not availible.\n");
	  res = EXECUTION_FAILURE;
#endif
	  break;

	case XPC_WRITE:

	  res = cache_write (filename);
	  break;

	case XPC_LOAD:

	  res = cache_load (filename);
	  break;

	case XPC_UNLOAD:

	  if (filename)
	    res = cache_unload (filename);
	  else
	    while (!res && heap.next)
	      res = cache_unload (heap.next->filename);

	  break;

	case XPC_SHOW:

	  printcmds ();
	  break;

	case XPC_REMOVE:

	  xpc_heap_cache_delete ();
	  res = xpc_libinit ();
	  break;

	case XPC_CLEAN:

	  while (!res && heap.next)
	    res = cache_unload (heap.next->filename);

	  xpc_heap_cache_delete ();
	  break;

	case XPC_VERSION:

	  printf ("xpcomp " __VALUESTRING (VERSION)
#if defined(DEBUG)
		  " debug"
#endif
		  "\n");
	  break;

	case XPC_UNKNOWN:

	  builtin_usage ();
	  res = EX_USAGE;
	  break;

	default:

	  builtin_error ("Option -%c not yet implemented", c);
	  res = EXECUTION_SUCCESS;
	}
    }

  if (res && errno)
    perror (NULL);

  return res;
}



static char *xpcomp_doc[] = {
  "The documentation is in the man page.",
  (char *) NULL
};

struct builtin xpcomp_struct = {
  "xpcomp",
  xpcomp_builtin,
  BUILTIN_ENABLED,
  xpcomp_doc,
  "xpcomp [-lw file] [-u [file]] [-D [file]] [-R] [-X] [-r] [-p]",
  (char *) 0
};
