/*
  xpcomp_lib.c - library functions for xpcomp

  Copyright (C) 2001 Ingo K"ohne
  exept for parts taken from the bash sources (search for 'complete.def') 
  
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
*/

#include "defines.h"

#include <string.h>
#include <stdlib.h>		/* bsearch () */
#include <stdio.h>

#include "shell.h"

#include "xpcomp.h"

/* at least this much space is added when a buffer runs empty */
const XPC_COUNT realloc_size = 256;

/* at most this much empty space is left in a buffer */
const XPC_COUNT max_alloc_size = 4096;


typedef struct xpc_strref
{
  XPC_COUNT value;
  unsigned int count;
}
XPC_STRREF;

typedef struct xpc_leak
{
  XPC_COUNT start;
  XPC_COUNT count;
  XPC_COUNT sum;

  struct xpc_leak *next;
}
XPC_LEAK;

/* The current cache */
XPC_CACHE *cache;

/* if the user wants debug output */
int xpc_debug = 0;

/* current state of the xpcomp builtins */
XPC_STATE xpcomp_state = XPC_UNINITIALIZED;

/* jump target for disableing the builtins */
jmp_buf self_destruct;

/* where to put debug output */
FILE *errstream;
/* more detailed description of an error (mainly for debugging) */
char const *xpc_errmsg;

/* unused areas in the allocated memory */
static XPC_LEAK *leak_opt = NULL;
static XPC_LEAK *leak_compspec = NULL;
static XPC_LEAK *leak_chr = NULL;

/* array of object for reference counting strings in heap.chrs */
static XPC_STRREF *strref_buff = NULL;
static XPC_COUNT strref_buff_size = 0;
static XPC_COUNT strref_count = 0;

/* the 'cuurent' command (for performance only)  */
static XPC_CMD *cmd = NULL;

/* The first cache is on the heap and can be manipulated by the user,
 all others are added to the list and are read-only mmap(2)ed cache files. */
XPC_CACHE heap = { NULL, NULL, NULL, NULL, NULL, NULL, NULL, 0 };
/* This index contains the sizes of the arrays allocated for HEAP */
static XPC_INDEX buffs;
/* This index contains used parts of the arrays in BUFFS */
static XPC_INDEX heapidx;

/* All comparison functions compare in the end strings.
   this one is used if a key object pointer is NULL.
   This saves me from creating the real objects as keys.*/
static const char *xpc_keystr = NULL;



static int
xpc_strref_compare (const void *a, const void *b)
{
  XPC_STRREF *strref_a = (XPC_STRREF *) a;
  XPC_STRREF *strref_b = (XPC_STRREF *) b;

  assert ((strref_a && strref_b) || xpc_keystr);

  return strcmp (strref_a ? GETCHR (strref_a->value) : xpc_keystr,
		 strref_b ? GETCHR (strref_b->value) : xpc_keystr);
}


/* the xpc_*_compare functions compare objects for btree() */

static int
xpc_cmd_compare (const void *a, const void *b)
{
  XPC_CMD *cmd_a = (XPC_CMD *) a;
  XPC_CMD *cmd_b = (XPC_CMD *) b;

  assert ((cmd_a && cmd_b) || xpc_keystr);

  return strcmp (cmd_a ? GETCHR (cmd_a->name) : xpc_keystr,
		 cmd_b ? GETCHR (cmd_b->name) : xpc_keystr);
}



static int
xpc_opt_compare (const void *a, const void *b)
{
  XPC_OPT *opt_a = (XPC_OPT *) a;
  XPC_OPT *opt_b = (XPC_OPT *) b;

  const char *str_a;
  const char *str_b;

  assert ((opt_a && opt_b) || xpc_keystr);

  str_a = opt_a ? GETCHR (opt_a->name) : xpc_keystr;
  str_b = opt_b ? GETCHR (opt_b->name) : xpc_keystr;

  assert (str_a);
  assert (str_b);
  assert (*str_a);
  assert (*str_b);

  /* short option is always 'smaller' than long */
  if (!*(str_a + 1) ^ !*(str_b + 1))
    return *(str_a + 1) ? 1 : -1;

  return strcmp (str_a, str_b);
}



static XPC_STRREF *
xpc_strref_find (const char *str)
{
  assert (str);
  assert (*str);

  xpc_keystr = str;

  if (!strref_count)
    return NULL;

  assert (strref_buff);

  return (XPC_STRREF *) bsearch (NULL, strref_buff, strref_count,
				 sizeof (XPC_STRREF), xpc_strref_compare);
}


/* the xpc_check_realloc_* make sure that at least MIN objects are availible
 at the end of the array (or 1 if no min argument) */

static void
xpc_check_realloc_strref ()
{
  XPC_COUNT newsize = 0;

  if (strref_buff_size - strref_count == 0
      || strref_buff_size - strref_count > max_alloc_size)
    newsize = strref_count + realloc_size;

  if (newsize)
    {
      strref_buff = (XPC_STRREF *) realloc (strref_buff, sizeof (XPC_STRREF)
					    * newsize);
      xpc_disable_if_not (strref_buff);

      strref_buff_size = newsize;
    }
}



static void
xpc_check_realloc_cmds ()
{
  XPC_COUNT newsize = 0;

  if (buffs.cmd_count - heap.idx->cmd_count == 0
      || buffs.cmd_count - heap.idx->cmd_count > max_alloc_size)
    newsize = heap.idx->cmd_count + realloc_size;

  if (newsize)
    {
      heap.cmds = (XPC_CMD *) realloc (heap.cmds, sizeof (XPC_CMD) * newsize);

      xpc_disable_if_not (heap.cmds);

      buffs.cmd_count = newsize;
    }
}



static void
xpc_check_realloc_opts (XPC_COUNT min)
{
  XPC_COUNT newsize = 0;

  if (buffs.opt_count - heap.idx->opt_count - min <= 0
      || buffs.opt_count - heap.idx->opt_count - min > max_alloc_size)
    newsize = heap.idx->opt_count + MAX (min, realloc_size);

  if (newsize)
    {
      heap.opts = (XPC_OPT *) realloc (heap.opts, sizeof (XPC_OPT) * newsize);

      xpc_disable_if_not (heap.opts);

      buffs.opt_count = newsize;
    }
}



static void
xpc_check_realloc_compspecs (XPC_COUNT min)
{
  XPC_COUNT newsize = 0;

  if (buffs.compspec_count - heap.idx->compspec_count - min <= 0
      || buffs.compspec_count - heap.idx->compspec_count - min >
      max_alloc_size)
    newsize = heap.idx->compspec_count + MAX (min, realloc_size);

  if (newsize)
    {
      heap.compspecs =
	(COMPSPEC *) realloc (heap.compspecs, sizeof (COMPSPEC) * newsize);

      xpc_disable_if_not (heap.compspecs);

      buffs.compspec_count = newsize;
    }
}



static int
xpc_check_realloc_chrs (XPC_COUNT min)
{
  XPC_COUNT newsize = 0;

  if (buffs.chr_count - heap.idx->chr_count - min <= 0
      || buffs.chr_count - heap.idx->chr_count - min > max_alloc_size)
    newsize = heap.idx->chr_count + MAX (min, realloc_size);

  if (newsize)
    {
      heap.chrs = (XPC_CHR *) realloc (heap.chrs, sizeof (XPC_CHR) * newsize);

      xpc_disable_if_not (heap.chrs);

      buffs.chr_count = newsize;
    }
  return EXECUTION_SUCCESS;
}



/* mark COUNT objects beginning from START as unused in *LEAKP */
static void
xpc_leak_add (XPC_LEAK ** leakp, XPC_COUNT start, XPC_COUNT count)
{
  XPC_LEAK *pred;
  XPC_LEAK *succ;

  XPC_LEAK *new;

  assert (leakp);
  assert (start > 0);
  assert (count > 0);

  /* search for predecessor and successor */
  pred = NULL;
  succ = *leakp;
  while (succ && succ->start < start)
    {
      pred = succ;
      succ = succ->next;
    }

  /* areas should not overlap */
  assert (!pred || pred->start + pred->count <= start);
  assert (!succ || start + count <= succ->start);

  /* merge if continous area accrues */
  if (pred && pred->start + pred->count == start)
    {
      pred->count += count;
      if (succ && pred->start + pred->count == succ->start)
	{
	  pred->count += succ->count;
	  pred->next = succ->next;
	  free (succ);
	}
      return;
    }

  if (succ && start + count == succ->start)
    {
      succ->start = start;
      succ->count += count;
      return;
    }

  new = (XPC_LEAK *) malloc (sizeof (XPC_LEAK));

  xpc_disable_if_not (new);

  new->start = start;
  new->count = count;
  new->sum = 0;
  new->next = succ;

  if (pred)
    pred->next = new;
  else
    *leakp = new;

  return;
}


/* find command by its name CMDNAME in the current cache */
XPC_CMD *
xpc_cmd_find (const char *cmdname)
{
  assert (cmdname);
  assert (cache);
  assert (cache->idx);

  /* TODO: shell hash */

  xpc_keystr = cmdname;

  if (!cache->idx->cmd_count)
    return NULL;

  assert (cache->cmds);

  return (XPC_CMD *) bsearch (NULL, GETCMD (1), cache->idx->cmd_count - 1,
			      sizeof (XPC_CMD), xpc_cmd_compare);
}



/* find command by its name CMDNAME in all caches */
XPC_CMD *
xpc_cmd_find_all (const char *cmdname)
{
  assert (cmdname);

  if (cmd && !strcmp (GETCHR (cmd->name), cmdname))
    return cmd;

  cmd = NULL;
  for (cache = &heap; cache; cache = cache->next)
    if (cache->idx && (cmd = xpc_cmd_find (cmdname)))
      break;

  return cmd;
}


/* find option of CMD by its name OPTNAME */
XPC_OPT *
xpc_opt_find (XPC_CMD * cmd, char *optname)
{
  assert (cmd);
  assert (optname);

  if (!cmd->opts)
    return NULL;

  xpc_keystr = optname;

  return bsearch (NULL,
		  GETOPT (cmd->opts),
		  cmd->opt_count, sizeof (XPC_OPT), xpc_opt_compare);
}



/* find option of CMD matching PATTERN or next option matching previous pattern
if CMD is NULL.  Return value is 0 if no option was found, 1 if the last option
was found and 2 if another call with CMD==NULL will succeed */
int
xpc_opt_match (XPC_CMD * cmd, const char *pattern, XPC_OPT ** opt)
{
  static XPC_OPT *next;

  static XPC_CMD *prev_cmd;
  static const char *prev_pattern;
  static int prev_pattern_length;

  int pattern_length;

  XPC_COUNT n;

  assert (opt);

  if (cmd)
    {
      assert (pattern);
      pattern_length = strlen (pattern);
      next = NULL;

      *opt = NULL;
      for (n = 0; n < cmd->opt_count; n++)
	{
	  XPC_OPT *o = GETOPT (cmd->opts + n);

	  if (*(GETCHR (o->name) + 1) < 2)
	    continue;
	  if (!strncmp (GETCHR (o->name), pattern, pattern_length))
	    {
	      *opt = o;
	      break;
	    }
	}

      if (!*opt)
	return 0;

      prev_cmd = cmd;
      prev_pattern = pattern;
      prev_pattern_length = pattern_length;
    }
  else
    {
      *opt = next;

      if (!*opt)
	return 0;

      cmd = prev_cmd;
      pattern = prev_pattern;
      pattern_length = prev_pattern_length;
    }

  next = *opt + 1;
  if (next - GETOPT (cmd->opts) == (int) cmd->opt_count
      || strncmp (GETCHR (next->name), pattern, pattern_length))
    next = NULL;

  return next ? 2 : 1;
}



/* store a string and give back its location */
XPC_COUNT
xpc_str_add (char *str)
{
  XPC_STRREF *ref;

  XPC_COUNT new;
  XPC_COUNT len;

  assert (cache == &heap);

  if (!str || !*str)
    return 0;

  /* look if we already have that string */

  ref = xpc_strref_find (str);

  if (ref)
    {
      ref->count++;
      return ref->value;
    }


  len = strlen (str);

  /* first alloc all we need */

  xpc_check_realloc_chrs (len + 1);
  xpc_check_realloc_strref (1);

  /* add a new string */

  new = cache->idx->chr_count;
  cache->idx->chr_count += len + 1;

  strcpy (GETCHR (new), str);

  /* add the reference counter */

  strref_buff[strref_count].value = new;
  strref_buff[strref_count].count = 1;
  strref_count++;

  qsort (strref_buff, strref_count, sizeof (XPC_STRREF), xpc_strref_compare);

  return new;
}



void
xpc_str_del (XPC_COUNT str)
{
  XPC_STRREF *ref;

  assert (cache == &heap);

  if (!str)
    return;

  ref = xpc_strref_find (GETCHR (str));

  assert (ref);

  if (--ref->count)
    return;

  xpc_leak_add (&leak_chr, str, strlen (GETCHR (str)) + 1);

  strref_count--;

  memmove (ref, ref + 1,
	   sizeof (XPC_STRREF) * (&strref_buff[strref_count] - ref));

  return;
}



XPC_CMD *
xpc_cmd_add (char *cmdname)
{
  XPC_CMD *new;

  assert (cache == &heap);

  xpc_check_realloc_cmds ();

  new = GETCMD (heap.idx->cmd_count);
  memset (new, 0, sizeof (XPC_CMD));

  new->name = xpc_str_add (cmdname);
  if (!new->name)
    return NULL;

  cache->idx->cmd_count++;

  if (cache->idx->cmd_count > 2)
    qsort (GETCMD (1),
	   cache->idx->cmd_count - 1, sizeof (XPC_CMD), xpc_cmd_compare);

  return xpc_cmd_find (cmdname);
}



void
xpc_cmd_del (XPC_CMD * c)
{
  XPC_COUNT n;

  assert (cache == &heap);
  assert (c);
  assert (GETCMD (0) < c);
  assert (c <= GETCMD (cache->idx->cmd_count));

  cmd = NULL;

  if (c->opt_count)
    xpc_leak_add (&leak_opt, c->opts, c->opt_count);

  for (n = c->opts; n < c->opt_count; n++)
    {
      xpc_str_del (GETOPT (n)->name);
      xpc_str_del (GETOPT (n)->test);
      xpc_str_del (GETOPT (n)->varname);
      if (GETOPT (n)->mode.description_only)
	xpc_str_del (GETOPT (n)->compspec);
      else
	xpc_compspec_del (GETOPT (n)->compspec);
    }

  memmove (c, c + 1, sizeof (XPC_CMD) *
	   (cache->idx->cmd_count - (c - cache->cmds) - 1));

  cache->idx->cmd_count--;

  return;
}



/* initialization routine for the library, called once if xpcomp_state is
 XPC_UNINITIALIZED */
int
xpc_libinit ()
{
  XPC_COUNT n;

  xpc_cache_set (&heap);

  cache->idx = &heapidx;

  xpc_check_realloc_cmds ();
  memset (GETCMD (0), 0, sizeof (XPC_CMD));
  cache->idx->cmd_count = 1;

  xpc_check_realloc_opts (1);
  memset (GETOPT (0), 0, sizeof (XPC_OPT));
  cache->idx->opt_count = 1;

  xpc_check_realloc_compspecs (COMPSPEC_RESERVED);
  memset (GETCOMPSPEC (0), 0, sizeof (COMPSPEC) * COMPSPEC_RESERVED);
  cache->idx->compspec_count = COMPSPEC_RESERVED;

  for (n = 1; n < COMPSPEC_RESERVED; n++)
    GETCOMPSPEC (n)->refcount = 1;

  GETCOMPSPEC (COMPSPEC_FILE)->actions |= CA_FILE;
  GETCOMPSPEC (COMPSPEC_DIRECTORY)->actions |= CA_DIRECTORY;
  GETCOMPSPEC (COMPSPEC_COMMAND)->actions |= CA_COMMAND;

  xpc_check_realloc_chrs (1);
  *GETCHR (0) = 0;
  cache->idx->chr_count = 1;

  xpc_check_realloc_strref (1);	/* necessary ? */

  xpcomp_state = XPC_OK;

  return EXECUTION_SUCCESS;
}



/* delete all entries from the heap cache, leaving it XPC_UNINITIALIZED */
void
xpc_heap_cache_delete ()
{
  XPC_CACHE *next;

  xpc_cache_set (&heap);

  xpcomp_state = XPC_UNINITIALIZED;

  xpc_cmd_del_all ();

  free (strref_buff);
  strref_buff = NULL;
  strref_buff_size = 0;

  free (cache->cmds);
  free (cache->opts);
  free (cache->compspecs);
  free (cache->chrs);
  memset (cache->idx, 0, sizeof (XPC_INDEX));
  memset (&buffs, 0, sizeof (XPC_INDEX));

  next = cache->next;
  memset (cache, 0, sizeof (XPC_CACHE));
  cache->next = next;

  xpc_cache_set (NULL);
}



XPC_OPT *
xpc_cmd_addopt (XPC_CMD * cmd, char *optname)
{
  XPC_OPT *new;

  assert (cache == &heap);

  assert (cmd);
  assert (optname);
  assert (*optname);

  new = xpc_opt_find (cmd, optname);

  if (new)
    return new;

  if (cmd->opts && (cmd->opts + cmd->opt_count != cache->idx->opt_count))
    {
      xpc_check_realloc_opts (cmd->opt_count + 1);

      if (cmd->opt_count)
	xpc_leak_add (&leak_opt, cmd->opts, cmd->opt_count);

      memmove (GETOPT (cache->idx->opt_count),
	       GETOPT (cmd->opts), cmd->opt_count * sizeof (XPC_OPT));

      /*  cache->idx->opt_count points to first unused */
      cmd->opts = cache->idx->opt_count;
      cache->idx->opt_count += cmd->opt_count;
    }
  else
    xpc_check_realloc_opts (1);

  new = GETOPT (cache->idx->opt_count);

  memset (new, 0, sizeof (XPC_OPT));

  new->name = xpc_str_add (optname);
  if (!new->name)
    return NULL;

  if (!cmd->opts)
    cmd->opts = cache->idx->opt_count;

  cmd->opt_count += 1;
  cmd->opt_len += strlen (optname);

  cache->idx->opt_count++;

  if (cmd->opt_count > 1)
    qsort (GETOPT (cmd->opts),
	   cmd->opt_count, sizeof (XPC_OPT), xpc_opt_compare);

  return xpc_opt_find (cmd, optname);
}



void
xpc_cmd_delopt (XPC_CMD * cmd, XPC_OPT * opt)
{
  assert (cache == &heap);
  assert (cmd && opt);
  assert (cache->cmds < cmd);
  assert (cmd < cache->cmds + cache->idx->cmd_count);
  assert (cache->opts + cmd->opts <= opt);
  assert (opt < cache->opts + cmd->opts + cmd->opt_count);

  cmd->opt_len -= strlen (GETCHR (opt->name));

  xpc_str_del (opt->name);
  xpc_str_del (opt->test);
  xpc_str_del (opt->varname);
  if (opt->mode.description_only)
    xpc_str_del (opt->compspec);
  else
    xpc_compspec_del (opt->compspec);

  if (!cmd->opts)
    return;

  memmove (opt, opt + 1, sizeof (XPC_OPT) *
	   (GETOPT (cmd->opts + cmd->opt_count) - opt - 1));

  cmd->opt_count--;

  xpc_leak_add (&leak_opt, cmd->opts + cmd->opt_count, 1);

  if (!cmd->opt_count)
    cmd->opts = 0;
}



/* integrates over the leak structure to compute the shifts needed to eliminate
 the unused areas */
static void
xpc_leak_integrate (XPC_LEAK * l)
{
  XPC_COUNT sum = 0;

  for (; l; l = l->next)
    l->sum = (sum += l->count);
}



/* returns the shift for an object needed to eliminate the unused areas */
static XPC_COUNT
xpc_leak_sum (XPC_LEAK * l, XPC_COUNT n)
{
  if (!l || n < l->start)
    return 0;

  for (; l; l = l->next)
    {
      if (l->start + l->count <= n && (!l->next || n < l->next->start))
	return l->sum;
    }

  assert (!"Should not be called for an area inside the leak");

  return 0;
}



void
xpc_cmd_del_all ()
{
  XPC_LEAK *l;
  assert (cache = &heap);

  cache->idx->cmd_count = 1;
  cache->idx->opt_count = 1;
  cache->idx->chr_count = 1;

  strref_count = 0;

  l = leak_chr;
  while (l)
    {
      XPC_LEAK *del = l;
      l = l->next;
      free (del);
    }
  leak_chr = NULL;

  l = leak_opt;
  while (l)
    {
      XPC_LEAK *del = l;
      l = l->next;
      free (del);
    }
  leak_opt = NULL;

  xpc_keystr = NULL;
  cmd = NULL;
}



/* Do garbage collection for the current cache. Remove all the leaks from
   leak_opt, leak_chr and leak_compspec. */
void
xpc_cache_gc ()
{
  XPC_LEAK *l;
  XPC_COUNT n;

  assert (cache == &heap);

  /* gc for XPC_OPT array: */

  xpc_leak_integrate (leak_opt);

  for (l = leak_opt; l; l = l->next)
    {
      XPC_COUNT block_start = l->start + l->count;

      memmove (GETOPT (block_start - l->sum),
	       GETOPT (block_start),
	       ((l->next ? l->next->start : cache->idx->opt_count)
		- block_start) * sizeof (XPC_OPT));

      if (!l->next)
	cache->idx->opt_count -= l->sum;
    }

  for (n = 1; n < cache->idx->cmd_count; n++)
    {
      XPC_CMD *c = GETCMD (n);
      if (c->opt_count)
	c->opts -= xpc_leak_sum (leak_opt, c->opts);
    }

  while (leak_opt)
    {
      XPC_LEAK *del = leak_opt;
      leak_opt = leak_opt->next;
      free (del);
    }
  leak_opt = NULL;

  /* gc for COMPSPEC array: */

  xpc_leak_integrate (leak_compspec);

  for (l = leak_compspec; l; l = l->next)
    {
      XPC_COUNT block_start = l->start + l->count;

      memmove (GETCOMPSPEC (block_start - l->sum),
	       GETCOMPSPEC (block_start),
	       ((l->next ? l->next->start : cache->idx->compspec_count)
		- block_start) * sizeof (COMPSPEC));

      if (!l->next)
	cache->idx->compspec_count -= l->sum;
    }

  for (n = 1; n < cache->idx->opt_count; n++)
    {
      XPC_OPT *o = GETOPT (n);
      if (!o->mode.description_only)
	o->compspec -= xpc_leak_sum (leak_compspec, o->compspec);
    }

  while (leak_compspec)
    {
      XPC_LEAK *del = leak_compspec;
      leak_compspec = leak_compspec->next;
      free (del);
    }
  leak_compspec = NULL;

  /* gc for XPC_CHR array: */

  xpc_leak_integrate (leak_chr);

  for (l = leak_chr; l; l = l->next)
    {
      XPC_COUNT block_start = l->start + l->count;

      memmove (GETCHR (block_start - l->sum),
	       GETCHR (block_start),
	       ((l->next ? l->next->start : cache->idx->chr_count)
		- block_start) * sizeof (XPC_CHR));

      if (!l->next)
	cache->idx->chr_count -= l->sum;
    }

  for (n = 1; n < cache->idx->cmd_count; n++)
    {
      XPC_CMD *c = GETCMD (n);

      c->name -= xpc_leak_sum (leak_chr, c->name);
    }

  for (n = 1; n < cache->idx->opt_count; n++)
    {
      XPC_OPT *o = GETOPT (n);

      o->name -= xpc_leak_sum (leak_chr, o->name);
      if (o->mode.description_only)
	o->compspec -= xpc_leak_sum (leak_chr, o->compspec);
      o->test -= xpc_leak_sum (leak_chr, o->test);
      o->varname -= xpc_leak_sum (leak_chr, o->varname);
    }

  for (n = COMPSPEC_RESERVED; n < cache->idx->compspec_count; n++)
    {
      COMPSPEC *cs = GETCOMPSPEC (n);

      cs->globpat -= xpc_leak_sum (leak_chr, (XPC_COUNT) cs->globpat);
      cs->words -= xpc_leak_sum (leak_chr, (XPC_COUNT) cs->words);
      cs->prefix -= xpc_leak_sum (leak_chr, (XPC_COUNT) cs->prefix);
      cs->suffix -= xpc_leak_sum (leak_chr, (XPC_COUNT) cs->suffix);
      cs->funcname -= xpc_leak_sum (leak_chr, (XPC_COUNT) cs->funcname);
      cs->command -= xpc_leak_sum (leak_chr, (XPC_COUNT) cs->command);
      cs->filterpat -= xpc_leak_sum (leak_chr, (XPC_COUNT) cs->filterpat);
    }

  for (n = 0; n < strref_count; n++)
    strref_buff[n].value -= xpc_leak_sum (leak_chr, strref_buff[n].value);

  while (leak_chr)
    {
      XPC_LEAK *del = leak_chr;
      leak_chr = leak_chr->next;
      free (del);
    }
  leak_chr = NULL;

  cmd = NULL;

  if (xpc_cache_validate ())
    {
      fprintf (stderr, "%s\n", xpc_errmsg);
      assert (!"validation failure");
    }
}



/* compute a checksum needed for writing the cache to a file */
unsigned long int
xpc_cache_checksum ()
{
  unsigned long int checksum = 0;
  unsigned long int checksum_mask = (unsigned long int) -1 >> 1;
  char *cp;

  for (cp = (char *) cache->idx; cp < (char *) (cache->idx + 1); cp++)
    checksum = (checksum + *cp) & checksum_mask;

  for (cp = (char *) GETCMD (0);
       cp < (char *) GETCMD (cache->idx->cmd_count); cp++)
    checksum = (checksum + *cp) & checksum_mask;

  for (cp = (char *) GETOPT (0);
       cp < (char *) GETOPT (cache->idx->opt_count); cp++)
    checksum = (checksum + *cp) & checksum_mask;

  for (cp = (char *) GETCOMPSPEC (0);
       cp < (char *) GETCOMPSPEC (cache->idx->compspec_count); cp++)
    checksum = (checksum + *cp) & checksum_mask;

  for (cp = (char *) GETCHR (0);
       cp < (char *) GETCHR (cache->idx->chr_count); cp++)
    checksum = (checksum + *cp) & checksum_mask;

  return checksum;
}



/* validates all the XPC_COUNT pointers in the cache */
int
xpc_cache_validate ()
{
  XPC_COUNT n;

  assert (cache && cache->idx);

  cmd = NULL;

  for (n = 1; n < cache->idx->cmd_count; n++)
    {
      XPC_CMD *c = GETCMD (n);

      xpc_errmsg = "command name not found";
      if (c->name >= cache->idx->chr_count
	  || (c->name && *(GETCHR (c->name) - 1)))
	return EXECUTION_FAILURE;
      xpc_errmsg = "command option array too large";
      if (c->opts + c->opt_count > cache->idx->opt_count)
	return EXECUTION_FAILURE;
    }

  for (n = 1; n < cache->idx->opt_count; n++)
    {
      XPC_OPT *o = GETOPT (n);

      if (o->mode.description_only)
	{
	  xpc_errmsg = "option description not found";
	  if (o->compspec >= cache->idx->chr_count)
	    return EXECUTION_FAILURE;
	}
      else
	{
	  xpc_errmsg = "option compspec not found";
	  if (o->compspec >= cache->idx->compspec_count)
	    return EXECUTION_FAILURE;
	}
      xpc_errmsg = "option name corrupt";
      if (o->name >= cache->idx->chr_count
	  || (o->name && *(GETCHR (o->name) - 1)))
	return EXECUTION_FAILURE;
      xpc_errmsg = "option test corrupt";
      if (o->test >= cache->idx->chr_count
	  || (o->test && *(GETCHR (o->test) - 1)))
	return EXECUTION_FAILURE;
      xpc_errmsg = "option varname corrupt";
      if (o->varname >= cache->idx->chr_count
	  || (o->varname && *(GETCHR (o->varname) - 1)))
	return EXECUTION_FAILURE;
    }

  for (n = 1; n < cache->idx->compspec_count; n++)
    {
      COMPSPEC *cs = GETCOMPSPEC (n);

      xpc_errmsg = "compspec globpat corrupt";
      if ((XPC_COUNT) cs->globpat >= cache->idx->chr_count
	  || (cs->globpat && *(GETCHR ((XPC_COUNT) cs->globpat) - 1)))
	return EXECUTION_FAILURE;
      xpc_errmsg = "compspec words corrupt";
      if ((XPC_COUNT) cs->words >= cache->idx->chr_count
	  || (cs->words && *(GETCHR ((XPC_COUNT) cs->words) - 1)))
	return EXECUTION_FAILURE;
      xpc_errmsg = "compspec prefix corrupt";
      if ((XPC_COUNT) cs->prefix >= cache->idx->chr_count
	  || (cs->prefix && *(GETCHR ((XPC_COUNT) cs->prefix) - 1)))
	return EXECUTION_FAILURE;
      xpc_errmsg = "compspec suffix corrupt";
      if ((XPC_COUNT) cs->suffix >= cache->idx->chr_count
	  || (cs->suffix && *(GETCHR ((XPC_COUNT) cs->suffix) - 1)))
	return EXECUTION_FAILURE;
      xpc_errmsg = "compspec funcname corrupt";
      if ((XPC_COUNT) cs->funcname >= cache->idx->chr_count
	  || (cs->funcname && *(GETCHR ((XPC_COUNT) cs->funcname) - 1)))
	return EXECUTION_FAILURE;
      xpc_errmsg = "compspec command corrupt";
      if ((XPC_COUNT) cs->command >= cache->idx->chr_count
	  || (cs->command && *(GETCHR ((XPC_COUNT) cs->command) - 1)))
	return EXECUTION_FAILURE;
      xpc_errmsg = "compspec filterpat corrupt";
      if ((XPC_COUNT) cs->filterpat >= cache->idx->chr_count
	  || (cs->filterpat && *(GETCHR ((XPC_COUNT) cs->filterpat) - 1)))
	return EXECUTION_FAILURE;
    }

  return EXECUTION_SUCCESS;
}



/* set the current cache */
void
xpc_cache_set (XPC_CACHE * c)
{
  if (cache != c || c == NULL)
    {
      cache = c;
      cmd = NULL;
    }
}



#define STR_EQUAL(n, ptr) ( \
   ( (n)==0 && (ptr)==NULL ) \
|| ( (n)!=0 && (ptr)!=NULL && strcmp(GETCHR((XPC_COUNT) n), ptr)==0 )   )

static int
xpc_compspec_equal (XPC_COUNT n, COMPSPEC * cs2)
{
  COMPSPEC *cs1 = GETCOMPSPEC (n);

  if (cs1->actions != cs2->actions)
    return 0;

  if (STR_EQUAL (cs1->globpat, cs2->globpat)
      && STR_EQUAL (cs1->words, cs2->words)
      && STR_EQUAL (cs1->suffix, cs2->suffix)
      && STR_EQUAL (cs1->funcname, cs2->funcname)
      && STR_EQUAL (cs1->command, cs2->command)
      && STR_EQUAL (cs1->filterpat, cs2->filterpat))
    return 1;

  return 0;
}



XPC_COUNT
xpc_compspec_add (COMPSPEC * values)
{
  XPC_COUNT n;
  COMPSPEC *cs;

  assert (cache == &heap);

  if (!values || xpc_compspec_equal (COMPSPEC_EMPTY, values))
    {
      GETCOMPSPEC (COMPSPEC_EMPTY)->refcount++;
      return COMPSPEC_EMPTY;
    }
  else if (xpc_compspec_equal (COMPSPEC_FILE, values))
    {
      GETCOMPSPEC (COMPSPEC_FILE)->refcount++;
      return COMPSPEC_FILE;
    }
  else if (xpc_compspec_equal (COMPSPEC_DIRECTORY, values))
    {
      GETCOMPSPEC (COMPSPEC_DIRECTORY)->refcount++;
      return COMPSPEC_DIRECTORY;
    }
  else if (xpc_compspec_equal (COMPSPEC_COMMAND, values))
    {
      GETCOMPSPEC (COMPSPEC_COMMAND)->refcount++;
      return COMPSPEC_COMMAND;
    }

  xpc_check_realloc_compspecs (1);

  assert (heap.idx->compspec_count >= COMPSPEC_RESERVED);

  n = heap.idx->compspec_count;
  cs = GETCOMPSPEC (n);

  memset (cs, 0, sizeof (COMPSPEC));

  cs->refcount = 1;
  cs->actions = values->actions;
  cs->globpat = (char *) xpc_str_add (values->globpat);
  cs->words = (char *) xpc_str_add (values->words);
  cs->prefix = (char *) xpc_str_add (values->prefix);
  cs->suffix = (char *) xpc_str_add (values->suffix);
  cs->funcname = (char *) xpc_str_add (values->funcname);
  cs->command = (char *) xpc_str_add (values->command);
  cs->filterpat = (char *) xpc_str_add (values->filterpat);

  cache->idx->compspec_count++;

  return n;
}



void
xpc_compspec_del (XPC_COUNT n)
{
  COMPSPEC *cs;

  assert (cache == &heap);

  cs = GETCOMPSPEC (n);

  if (n == 0)
    return;

  if (--cs->refcount > 0)
    return;

  assert (cs->refcount == 0);
  assert (n >= 8);

  xpc_str_del ((XPC_COUNT) cs->globpat);
  xpc_str_del ((XPC_COUNT) cs->words);
  xpc_str_del ((XPC_COUNT) cs->prefix);
  xpc_str_del ((XPC_COUNT) cs->suffix);
  xpc_str_del ((XPC_COUNT) cs->funcname);
  xpc_str_del ((XPC_COUNT) cs->command);
  xpc_str_del ((XPC_COUNT) cs->filterpat);

  memset (cs, 0, sizeof (COMPSPEC));

  xpc_leak_add (&leak_compspec, n, 1);

  return;
}



void
xpc_compspec_get (XPC_COUNT n, COMPSPEC * ret)
{
  COMPSPEC *cs = GETCOMPSPEC (n);

  ret->actions = cs->actions;
  ret->globpat = cs->globpat ? GETCHR ((XPC_COUNT) cs->globpat) : NULL;
  ret->words = cs->words ? GETCHR ((XPC_COUNT) cs->words) : NULL;
  ret->prefix = cs->globpat ? GETCHR ((XPC_COUNT) cs->prefix) : NULL;
  ret->suffix = cs->suffix ? GETCHR ((XPC_COUNT) cs->suffix) : NULL;
  ret->funcname = cs->funcname ? GETCHR ((XPC_COUNT) cs->funcname) : NULL;
  ret->command = cs->command ? GETCHR ((XPC_COUNT) cs->command) : NULL;
  ret->filterpat = cs->filterpat ? GETCHR ((XPC_COUNT) cs->filterpat) : NULL;
}



/* this part is copied from complete.def and modified */

#define SQPRINTARG(a, f) \
  do { \
    if (a) \
      { \
        x = sh_single_quote (local? GETCHR((XPC_COUNT) a) : a); \
        fprintf (stream, " %s %s", f, x); \
        free (x); \
      } \
  } while (0)

#define PRINTARG(a, f) \
  do { \
    if (a) \
      fprintf (stream, " %s %s", f, local? GETCHR((XPC_COUNT) a ): a); \
  } while (0)

#define PRINTOPT(a, f) \
  do { \
    if (acts & a) \
      fprintf (stream, " %s", f); \
  } while (0)

#define PRINTACT(a, f) \
  do { \
    if (acts & a) \
      fprintf (stream, " -A %s", f); \
  } while (0)



void
xpc_compspec_print (FILE * stream, COMPSPEC * cs, unsigned int local)
{
  unsigned long acts;
  char *x;

  acts = cs->actions;

  /* simple flags first */
  PRINTOPT (CA_ALIAS, "-a");
  PRINTOPT (CA_BUILTIN, "-b");
  PRINTOPT (CA_COMMAND, "-c");
  PRINTOPT (CA_DIRECTORY, "-d");
  PRINTOPT (CA_EXPORT, "-e");
  PRINTOPT (CA_FILE, "-f");
  PRINTOPT (CA_KEYWORD, "-k");
  PRINTOPT (CA_JOB, "-j");
  PRINTOPT (CA_USER, "-u");
  PRINTOPT (CA_VARIABLE, "-v");

  /* now the rest of the actions */
  PRINTACT (CA_ARRAYVAR, "arrayvar");
  PRINTACT (CA_BINDING, "binding");
  PRINTACT (CA_DISABLED, "disabled");
  PRINTACT (CA_ENABLED, "enabled");
  PRINTACT (CA_FUNCTION, "function");
  PRINTACT (CA_HELPTOPIC, "helptopic");
  PRINTACT (CA_HOSTNAME, "hostname");
  PRINTACT (CA_RUNNING, "running");
  PRINTACT (CA_SETOPT, "setopt");
  PRINTACT (CA_SHOPT, "shopt");
  PRINTACT (CA_SIGNAL, "signal");
  PRINTACT (CA_STOPPED, "stopped");

  /* now the rest of the arguments */

  /* arguments that require quoting */
  SQPRINTARG (cs->globpat, "-G");
  SQPRINTARG (cs->words, "-W");
  SQPRINTARG (cs->prefix, "-P");
  SQPRINTARG (cs->suffix, "-S");
  SQPRINTARG (cs->filterpat, "-X");
  SQPRINTARG (cs->command, "-C");

  /* simple arguments that don't require quoting */
  PRINTARG (cs->funcname, "-F");
}

/* end of code copied from bash */
