/*
  xpcompcore_builtin.c - match generator for xpcomp

  Copyright (C) 2001 Ingo K"ohne
  (exept for parts from query_display_matches() which were copied from
  readline/complete.c)
  
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.  */

#include "defines.h"

#include <getopt.h>
#include <stdio.h>
#include <ctype.h>		/* isspace() */
#include <errno.h>

#include <sys/types.h>		/* stat() */
#include <sys/stat.h>		/* stat() */
#include <unistd.h>		/* stat() */

#include "bashansi.h"		/* malloc () */
#include "shell.h"
#include "builtins.h"
#include "config.h"
#if defined(RESTRICTED_SHELL)
#  include "flags.h"		/* restricted */
#endif
#include "readline/readline.h"

#include "xpcomp.h"

extern int test_builtin (WORD_LIST * args);
extern int echo_builtin (WORD_LIST * args);

/* from rlprivate.h */
extern int rl_display_fixed;
extern int _rl_abort_internal (void);

/* a list of all shell variables we create so we can clean up at end
   (There is a VARLIST struct in the bash header files now (bash-2.05b)
   (hence the renaming to LVARLIST) will use it in the future if there
   appear some list handling procedures to replace my hand written ones.)
*/
typedef struct varlist
{
  char *varname;
  struct varlist *next;
}
LVARLIST;
static LVARLIST *shellvar_first = NULL;

/* backup of rl variables that we might change */
static char old_rl_completion_append_character = 0;
static VFunction *old_rl_completion_display_matches_hook = NULL;
static Function *old_rl_event_hook = NULL;
static char *old_rl_line_buffer;
static int old_rl_point;

/* readline changes its input reading method when rl_event_hook is installed.
 xpcomptest doesnt work then because readline looses characters
 sometimes, so i need a way to skip setting of the hook from xpcomptest. */
int install_rl_event_hook = 1;

/* bash will be checked for the behaviour of split_at_delims() */
int split_at_delims_bug = 0;

/* this is the difference between what we think is the completed word
 and what rl thinks ( due to rl_basic_word_break_characters ) */
static int signed_prefix_len = 0;
static char *prefix = NULL;

/* if the completion runs in restricted mode */
#if defined(RESTRICTED_SHELL)
int xpc_restrict = 1;
#endif
typedef enum
{
  COMPL_SHELLVAR,
  COMPL_REDIRECTIONS,
  COMPL_FIND_COMMAND,
  COMPL_GETOPT_PARSE,
  COMPL_CURRENT_WORD_OPTION,
  COMPL_PREVIOUS_WORD_OPTION,
  COMPL_NUMBERED_NONOPTION,
  COMPL_NONOPTION,
  COMPL_DEFAULT
}
COMPLETION_STEPS;



static char *xpcompcore_doc[] = {
  "The documentation is in the man page.",
  (char *) NULL
};


static void
free_argv (char **argv)
{
  char **word;

  if (!argv)
    return;

  for (word = argv; *word; word++)
    {
      free (*word);
      *word = NULL;
    }

  free (argv);
  argv = NULL;
}



static void
Xfree (void **ptr)
{
  if (ptr && *ptr)
    free (*ptr);
}



static void
print_array (ARRAY * a)
{
  ARRAY_ELEMENT *e;

  if (!a || !array_head (a) || array_empty (a))
    return;

  for (e = element_forw (a->head); e != a->head; e = element_forw (e))
    if (element_value (e))
      fprintf (errstream, "\"%s\" ", element_value (e));
}



/* insert a word in an argv-style array
  (if ever used outside get_line think carefully about the unwind_protect)
*/
static char *
insert_word (int *argcp, char ***argvp, const char *word, int pos)
{
  if (pos > *argcp)
    pos = *argcp;
  (*argcp)++;

  remove_unwind_protect ();
  *argvp = (char **) realloc (*argvp, sizeof (char *) * (*argcp + 1));
  add_unwind_protect ((Function *) free_argv, *argvp);

  if (pos < *argcp)
    memmove (*argvp + pos + 1, *argvp + pos,
	     sizeof (char *) * (*argcp - pos));
  (*argvp)[pos] = strdup (word);
  xpc_disable_if_not ((*argvp)[pos]);
  return (*argvp)[pos];
}



static void
print_argv (int argc, char *const *argv)
{
  int n;
  if (!argv)
    return;
  for (n = 0; n < argc; n++)
    fprintf (errstream, "\"%s\" ", argv[n]);
  fprintf (errstream, "(%i)", argc);
}



static int
is_redir_operator (const char *str, const char **endp, int *argp)
{
  const char *end = NULL;
  int arg = 0;
  int have_fd = 0;
  const char *cp = str;

  assert (str);

  if (*str >= '0' && *str <= '9')
    {
      have_fd = 1;

      while (*(++cp))
	if (!isdigit (*cp))
	  break;
    }

  if (cp[0] == '<')
    {
      switch (cp[1])
	{
	case '<':
	  if (have_fd)
	    break;
	  if (cp[2] == '-')
	    end = cp + 3;
	  else
	    end = cp + 2;
	  arg = 3;
	  break;
	case '>':
	  end = cp + 2;
	  arg = 1;
	  break;
	case '&':
	  end = cp + 2;
	  arg = 2;
	  break;
	default:
	  end = cp + 1;
	  arg = 1;
	  break;
	}
    }
  else if (cp[0] == '>')
    {
      switch (cp[1])
	{
	case '>':
	case '|':
	  end = cp + 2;
	  arg = 1;
	  break;
	case '&':
	  end = cp + 2;
	  if (have_fd)
	    arg = 2;
	  else
	    arg = 1;
	  break;
	default:
	  end = cp + 1;
	  arg = 1;
	  break;
	}
    }
  else if (cp[0] == '&')
    {
      if (!have_fd && cp[1] == '>')
	{
	  end = cp + 2;
	  arg = 1;
	}
    }

  if (endp)
    *endp = end;
  if (argp)
    *argp = arg;

  return end ? 1 : 0;
}



static char *
shell_var_start (char *str)
{
  char *cp;

  assert (str);
  cp = str + strlen (str);

  while (--cp >= str)
    {
      if ((unsigned int) *cp < 128 && (isalnum (*cp) || *cp == '_'))
	continue;

      if (isdigit (*(cp + 1)))
	return NULL;

      if (*cp == '$' || (cp > str && *cp == '{' && *(cp - 1) == '$'))
	return cp + 1;

    }

  return NULL;
}



static int
get_line (WORD_LIST * args, int *linecp, char ***linevp, int *curword_nop,
	  char **pattern_rlp)
{
  /* some shell variables provided by pcomplete.c */
  SHELL_VAR *COMP_WORDS = find_variable ("COMP_WORDS");
  SHELL_VAR *COMP_CWORD = find_variable ("COMP_CWORD");
  SHELL_VAR *COMP_LINE = find_variable ("COMP_LINE");
  SHELL_VAR *COMP_POINT = find_variable ("COMP_POINT");
  SHELL_VAR *COMP_CPOINT = find_variable ("COMP_CPOINT");

  /* current line */
  char *line;

  /* where is the cursor in the current line */
  size_t point_in_line;

  /* a flag for the algorith to find the point in the current word */
  int found_point_in_word = 0;

  /* where is the point in the current word */
  size_t point_in_word = 0;

  /* the values of shell variables (just for checking) */
  int COMP_CWORD_val;
  int COMP_CPOINT_val;

  /* how many words we have added (not counted in COMP_CWORD) */
  int added_words = 0;
  /* how many redirection words we have removed (not counted in *curword_nop) */
  int removed_redir_words = 0;
  /*  */
  int switch_curword = 0;

  /* variables used when looking for redirection operators. */
  size_t len_sum = 0;
  int redir_start = 0;
  int redir_end = 0;
  /* the character after the end of the redirection operator */
  const char *redir_op_end;
  /* the type of argument this operator takes. */
  /* ( 0 - none; 1 - file; 2 - file descriptor; 3 - here-document tag ) */
  int redir_arg_type;
  /* if we are in a redirection oprator */
  int in_redir = 0;

  /* loop counter */
  char *cp;
  int n;

  assert (linecp);
  assert (linevp);
  assert (curword_nop);
  assert (pattern_rlp);

  /* return values */
  *linecp = 0;
  *linevp = NULL;
  *curword_nop = 0;

  if (!COMP_WORDS || !COMP_CWORD || !COMP_LINE || !COMP_POINT)
    {
      fprintf (stderr, "\n");
      if (!COMP_WORDS)
	fprintf (stderr, "xpcompcore: Shell variable COMP_WORDS not found\n");
      if (!COMP_CWORD)
	fprintf (stderr, "xpcompcore: Shell variable COMP_CWORD not found\n");
      if (!COMP_LINE)
	fprintf (stderr, "xpcompcore: Shell variable COMP_LINE not found\n");
      if (!COMP_POINT)
	fprintf (stderr, "xpcompcore: Shell variable COMP_POINT not found\n");
      rl_forced_update_display ();
      return EXECUTION_FAILURE;
    }

  assert (value_cell (COMP_WORDS));
  assert (value_cell (COMP_CWORD));
  assert (value_cell (COMP_LINE));
  assert (value_cell (COMP_POINT));


  line = value_cell (COMP_LINE);
  point_in_line = atoi (value_cell (COMP_POINT));
  COMP_CWORD_val = atoi (value_cell (COMP_CWORD));
  if (COMP_CPOINT)
    COMP_CPOINT_val = atoi (value_cell (COMP_CPOINT));

  if (args && args->next && args->next->word)
    {
      size_t len;

      *pattern_rlp = args->next->word->word;
      len = strlen (*pattern_rlp);

      if (point_in_line < len)
	{
	  if (strcmp (*pattern_rlp, "{") != 0)
	    {
	      fprintf (stderr,
		       "\nxpcompcore: Pattern \"%s\" exceeds line \"%s\".\n",
		       *pattern_rlp, line);
	      return EXECUTION_FAILURE;
	    }
	  (*pattern_rlp)++;
	}
      if (strncmp (line + point_in_line - len, *pattern_rlp, len))
	{
	  fprintf (stderr,
		   "\nxpcompcore: Pattern \"%s\" not fount in line \"%s\" at point \"%i\".\n",
		   *pattern_rlp, line, point_in_line);
	  return EXECUTION_FAILURE;
	}
    }
  else
    {
      fprintf (stderr, "\nxpcompcore: Missing argument to xpcompcore\n");
      rl_forced_update_display ();
      return EXECUTION_FAILURE;
    }



  *linecp = array_num_elements (array_cell (COMP_WORDS));
  *linevp = array_to_argv (array_cell (COMP_WORDS));
  add_unwind_protect ((Function *) free_argv, *linevp);

  if (xpc_debug)
    {
      print_argv (*linecp, *linevp);
      fprintf (errstream, " current is %s\n", value_cell (COMP_CWORD));
    }

  cp = line;
  for (n = 0;; n++)
    {
      size_t len;
      static const char non_whitespace_delims[] = "()<>;&|";

      /* bash looses '<' and '>' when first character of a word with no
         preceeding whitespace.  Either reinsert it here either as a new word
         or prepend it to the next word if that word starts with a redirection
         character. */
      if (split_at_delims_bug && member (cp[0], non_whitespace_delims)
	  && (cp == line || !isspace (cp[-1])
	      || (cp - 1 != line && cp[-2] == '\\')))
	{
	  if (member (cp[1], non_whitespace_delims))
	    {
	      char *new;
	      char *old;

	      assert ((*linevp)[n] && member ((*linevp)[n][0],
					      non_whitespace_delims));

	      old = (*linevp)[n];
	      new = (char *) malloc (strlen ((*linevp)[n]) + 2);
	      xpc_disable_if_not (new);
	      new[0] = *cp;
	      strcpy (new + 1, (*linevp)[n]);
	      (*linevp)[n] = new;
	      free (old);
	    }
	  else
	    {
	      char str[] = { 'x', 0x0 };

	      str[0] = cp[0];

	      insert_word (linecp, linevp, str, n);

	      if (!*curword_nop && !((*linevp)[n] && !(*linevp)[n][0]))
		added_words++;
	    }

	}

      /* this is the single loop end condition. */
      if (!(*linevp)[n])
	{
	  if (!found_point_in_word && !redir_start)
	    {
	      insert_word (linecp, linevp, "", n);
	      *curword_nop = n;
	      point_in_word = 0;
	    }
	  break;
	}

      len = strlen ((*linevp)[n]);

      /* an unescaped closing brace seems to be a word breaking character */
      while (*cp && (isspace (*cp)))
	cp++;

      if (!found_point_in_word && line + point_in_line <= cp + len)
	{
	  found_point_in_word = 1;
	  *curword_nop = n;

	  if (line + point_in_line == cp + len
	      && member (*(cp + len), non_whitespace_delims))
	    switch_curword = -1;

	  if (line + point_in_line == cp + len
	      && member (*cp, non_whitespace_delims))
	    {
	      found_point_in_word = 0;
	      if (*(cp + len) == 0 || isspace (*(cp + len)))
		switch_curword = 1;
	    }
	  else if (line + point_in_line < cp)
	    {
	      /* the point is between words. an extra word is inserted.
	         for some reason this not done by the bash completion
	         algorithm if we are on a whitespace character
	         directly before the word or after the last word. */

	      point_in_word = 0;

	      if (switch_curword != 1 &&
		  (line + point_in_line < cp - 1 ||
		   (line + point_in_line == cp
		    && point_in_line == strlen (line))))
		{
		  /* check if empty word exists */
		  if (len)
		    {
		      fprintf (stderr, "\nxpcompcore: '%s',%i\n", line,
			       point_in_word);
		      fprintf (stderr,
			       "\nxpcompcore: Missing empty word around point.\n");
		    }
		}
	      else
		{
		  insert_word (linecp, linevp, "", n);
		  n++;
		}
	    }
	  else
	    point_in_word = point_in_line - (cp - line);
	}

      if (!redir_start && len && is_redir_operator (cp, &redir_op_end,
						    &redir_arg_type))
	redir_start = n;

      if (redir_start && !redir_end)
	{
	  len_sum += len;
	  if ((int) len_sum >= redir_op_end - cp)
	    {
	      len_sum = 0;
	      redir_end = n;
	      if (redir_arg_type)
		redir_end++;
	    }
	}

      if (strncmp ((*linevp)[n], cp, len))
	{
	  fprintf (stderr, "\nxpcompcore: '%s',%i\n", line, point_in_line);
	  fprintf (stderr, "\nxpcompcore: Recognized different word #%i:"
		   " \"%s\" \"%s\"\n", n, (*linevp)[n], cp);

	  rl_forced_update_display ();
	  return EXECUTION_FAILURE;
	}

      if (redir_start)
	{
	  void *old;
	  old = (*linevp)[n];
	  (*linevp)[n] = NULL;
	  free (old);

	  if (!(*linevp)[n + 1])
	    redir_end = n;
	  if (redir_end == n)
	    {
	      memmove ((*linevp) + redir_start,
		       (*linevp) + redir_end + 1,
		       sizeof (char **) * (*linecp - redir_end + 1));
	      n -= redir_end - redir_start + 1;
	      *linecp -= redir_end - redir_start + 1;

	      if (!found_point_in_word)
		{
		  removed_redir_words += redir_end - redir_start + 1;
		}
	      else if (*curword_nop >= redir_start
		       && *curword_nop <= redir_end)
		{
		  in_redir = 1;
		  removed_redir_words += *curword_nop - redir_start + 1;
		}

	      redir_start = 0;
	      redir_end = 0;
	    }
	}

      cp += len;
    }

  if (xpc_debug)
    {
      print_argv (*linecp, *linevp);
      fprintf (errstream, " current is %i\n", *curword_nop);
    }

  /* cut current word at point */
  if (!in_redir)
    {
      assert (strlen ((*linevp)[*curword_nop]) >= point_in_word);
      *((*linevp)[*curword_nop] + point_in_word) = 0;
    }

  /* pcomp code returns -2 if point is at space before end of line */
  if ((COMP_CWORD_val == -2 && *linecp - 1 != *curword_nop)
      || (COMP_CWORD_val != -2 && COMP_CWORD_val + switch_curword !=
	  *curword_nop + removed_redir_words - added_words))
    if (!in_redir)
      {
	fprintf (stderr, "\nxpcompcore: '%s',%i\n", line, point_in_line);
	fprintf (stderr,
		 "\nxpcompcore: COMP_CWORD(=%i) plus our correction(=%i) "
		 "differs from our estimate=(%i) of the current word.\n",
		 COMP_CWORD_val, switch_curword,
		 *curword_nop + removed_redir_words - added_words);
      }

  if (COMP_CPOINT)
    if (!in_redir && !switch_curword && point_in_word != COMP_CPOINT_val
	&& !(point_in_word == 0 && COMP_CPOINT_val < 0))
      {
	fprintf (stderr, "\nxpcompcore: '%s',%i\n", line, point_in_line);
	fprintf (stderr, "xpcompcore: Recognized different "
		 "point in current word #%i: %d %d\n", COMP_CWORD_val,
		 point_in_word, COMP_CPOINT_val);
	rl_forced_update_display ();
      }

  if (in_redir)
    *curword_nop = 0;

  return EXECUTION_SUCCESS;
}



/* Set shell variable WANT_VARNAME to value VALUE, creating it if it not
   already exists.  Keep an internal list all created variables so they can be
   cleaned up when we leave the xpcompcore_builtin.  If WANT_ARRAY is true,
   turn WANT_VARNAME into an array and append VALUE.  NOTE: Invalid characters
   in WANT_VARNAME are silently changed to '_' and "XPC_" is prepended.
*/
static int
make_shellvar (char *want_varname, char *value, int want_array)
{
  /* create variable & listitem and appen to list */

  SHELL_VAR *new;
  LVARLIST *newvarlist;
  LVARLIST *oldvarlist;
  char *varname;

  char *cp;

  assert (want_varname);
  if (!*want_varname)
    return EXECUTION_FAILURE;

  varname = malloc (strlen (want_varname) + 5);
  xpc_disable_if_not (varname);
  strcpy (varname, "XPC_");
  strcat (varname, want_varname);

  /*make proper variable name (underscoreify) */
  for (cp = varname + 4; *cp; cp++)
    if (!isalnum (*cp) || (unsigned char) *cp > 0x7f)
      *cp = '_';


  if (want_array)
    {
      new = find_variable (varname);
      if (!new || !array_p (new))
	new = make_local_array_variable (varname);
      if (!new)
	{
	  free (varname);
	  return EXECUTION_FAILURE;
	}

      array_insert (array_cell (new),
		    array_num_elements (array_cell (new)),
		    value ? value : "");
      if (xpc_debug)
	{
	  fprintf (errstream, "%s=(", varname);
	  print_array (array_cell (new));
	  fprintf (errstream, ")\n");
	}
    }
  else
    {
      new = make_local_variable (varname);
      if (!new)
	{
	  free (varname);
	  return EXECUTION_FAILURE;
	}

      bind_variable_value (new, value ? value : "");
      if (xpc_debug)
	fprintf (errstream, "%s=\"%s\"\n", varname, value ? value : "");
    }

  /* look if already in list */
  for (oldvarlist = shellvar_first; oldvarlist; oldvarlist = oldvarlist->next)
    {
      if (!strcmp (oldvarlist->varname, varname))
	{
	  free (varname);
	  return EXECUTION_SUCCESS;
	}
      if (!oldvarlist->next)
	break;
    }

  newvarlist = (LVARLIST *) malloc (sizeof (LVARLIST));
  if (!newvarlist)
    {
      makunbound (varname, shell_variables);
      free (varname);
      return EXECUTION_FAILURE;
    }

  newvarlist->varname = varname;
  newvarlist->next = NULL;

  if (oldvarlist)
    oldvarlist->next = newvarlist;
  else
    shellvar_first = newvarlist;

  return EXECUTION_SUCCESS;
}



static int
getopt_parse (XPC_CMD * c, int linec, char **linev, int curword_no,
	      int *skip_wordsp, char **new_cmdnamep, int *current_nonopt)
{
  XPC_COUNT n;
  XPC_OPT *o;

  /* Array of short options indexed by option character */
  XPC_OPT *opts_short[256] = { NULL };

  /* Array of long options */
  XPC_OPT **opts_long = NULL;

  /* when plus options are enabled, all words in linev that start
     with '+' are switched to '-'; this way we can do the parsing with getopt.
     The array word_start_plus helds which words were changed 
   */
  char **word_start_plus = NULL;

  /* in glued mode all words are examined if they start with '-' and end
     on and glued short option (an extra empty word is inserted then).
     this character array holds all glued short option characters for a
     command. */
  char *glued_opt_chars = NULL;

  /* getopt arguments */
  char *optstring = NULL;
  struct option *long_options = NULL;

  /* loop counter */
  char *cp;

  assert (c);
  assert (skip_wordsp);
  assert (new_cmdnamep);
  if (*new_cmdnamep)
    {
      free (*new_cmdnamep);
      *new_cmdnamep = NULL;
    }


  /* create optstring for short option */
  optstring = malloc (c->opt_count * 3 + 1);
  xpc_disable_if_not (optstring);
  add_unwind_protect (free, optstring);

  cp = optstring;
  *(cp++) = '-';
  for (n = 0, o = GETOPT (c->opts + n); n < c->opt_count; n++, o++)
    {
      char *optname = GETCHR (o->name);

      if (*(optname + 1))	/* long option */
	break;

      *(cp++) = *optname;
      switch (o->mode.arg)
	{
	case arg_optional:
	  *(cp++) = ':';
	case arg_required:
	  *(cp++) = ':';
	}

      opts_short[(unsigned int) *optname] = o;

      if (o->mode.glued)
	{
	  if (!glued_opt_chars)
	    {
	      /* string size chosen so, that if all remaining options were
	         'glued' the string would be large enough to hold them
	       */
	      glued_opt_chars = malloc (c->opt_count - n + 2);
	      xpc_disable_if_not (glued_opt_chars);
	      add_unwind_protect (free, glued_opt_chars);
	      *glued_opt_chars = 0;
	    }
	  strcat (glued_opt_chars, optname);
	}
    }				/* for */
  *cp = 0;

  /* create struct option array for long options */
  if (c->mode.longopt1dash || c->mode.longopt2dash)
    {
      int short_opt_count = n;

      long_options =
	(struct option *) malloc (sizeof (struct option) *
				  (c->opt_count - short_opt_count + 1));
      xpc_disable_if_not (long_options);
      add_unwind_protect (free, long_options);

      opts_long =
	(XPC_OPT **) malloc (sizeof (XPC_OPT *) *
			     (c->opt_count - short_opt_count + 1));
      xpc_disable_if_not (opts_long);
      add_unwind_protect (free, opts_long);

      for (; n < c->opt_count; n++, o++)
	{
	  struct option *lo = long_options + n - short_opt_count;

	  assert (o->name);
	  lo->name = GETCHR (o->name);
	  lo->flag = (int *) &n;
	  lo->val = n - short_opt_count;
	  switch (o->mode.arg)
	    {
	    case arg_none:
	      lo->has_arg = no_argument;
	      break;
	    case arg_optional:
	      lo->has_arg = optional_argument;
	      break;
	    case arg_required:
	      lo->has_arg = required_argument;
	    }
	  opts_long[n - short_opt_count] = o;
	}
      memset (&long_options[n - short_opt_count], 0, sizeof (struct option));
    }
  o = NULL;

  /* switch '+' options to '-' so they can be handled by getopt */
  if (c->mode.plusopt)
    {
      int i;
      word_start_plus =
	(char **) malloc (sizeof (char *) * (linec - *skip_wordsp));
      xpc_disable_if_not (word_start_plus);
      add_unwind_protect (free, word_start_plus);

      for (i = 1; i != linec - *skip_wordsp; i++)
	if (*(linev + *skip_wordsp)[i] == '+')
	  {
	    word_start_plus[i] = (linev + *skip_wordsp)[i];
	    *(word_start_plus[i]) = '-';
	  }
	else
	  {
	    word_start_plus[i] = NULL;
	  }
    }

  /* Look for short options that only accept arguments glued to themselves.
     Insert an empty arg if it is last in word. */
  if (glued_opt_chars)
    {
      char **srcp = linev;
      linev = (char **) malloc (sizeof (char *) * linec * 2);
      xpc_disable_if_not (linev);
      add_unwind_protect (free, linev);
      for (linec = 0; *srcp; srcp++)
	{
	  linev[linec++] = *srcp;
	  if (linec < *skip_wordsp)
	    continue;
	  if (*srcp[0] == '-'
	      && strchr (glued_opt_chars, (*srcp)[strlen (*srcp) - 1]))
	    {
	      if (curword_no >= linec)
		curword_no++;
	      linev[linec++] = "";
	    }
	}
      linev[linec] = NULL;
    }

  if (xpc_debug)
    fprintf (errstream, "Assigned shell variables: ");

  /* do the parsing; create & set shell variables */
  {
    int nonoption_count = 0;
    int is_curword;

    opterr = 0;
    optind = 0;

    while (1)
      {
	int ch;
	int found_nonoption = 0;

	char nonopt_name[4] = { 1, 1, 0, 0 };

	n = 0;


	if (c->mode.longopt1dash)
	  ch =
	    getopt_long_only (linec - *skip_wordsp,
			      linev + *skip_wordsp,
			      optstring, long_options, (int *) &n);
	else if (c->mode.longopt2dash)
	  ch =
	    getopt_long (linec - *skip_wordsp,
			 linev + *skip_wordsp,
			 optstring, long_options, (int *) &n);
	else
	  ch = getopt (linec - *skip_wordsp, linev + *skip_wordsp, optstring);

	is_curword = (curword_no == *skip_wordsp + optind - 1);

	if (ch == -1)		/* at end */
	  break;

	switch (ch)
	  {
	  case '?':
	  case ':':
	    /* Error */

	    if (!xpc_debug)
	      break;

	    fprintf (errstream, "ERR%c\"", ch);
	    if (optopt)
	      fprintf (errstream, "%c\" ", optopt);
	    else
	      fprintf (errstream, "%s\" ", linev[*skip_wordsp + optind - 1]);

	    continue;

	  case 1:
	    /* non-option word found */

	    nonoption_count++;

	    if (is_curword)
	      {
		char NONOPT_NO_value[2] = { '0', 0 };
		NONOPT_NO_value[0] += nonoption_count;
		make_shellvar ("NONOPT_NO", NONOPT_NO_value, 0);

		if (current_nonopt)
		  *current_nonopt = nonoption_count;
	      }

	    if (nonoption_count > 0 && nonoption_count < 10)
	      {
		nonopt_name[2] = '0' + nonoption_count;
		o = xpc_opt_find (c, nonopt_name);
	      }
	    if (!o)
	      {
		nonopt_name[2] = 0;
		o = xpc_opt_find (c, nonopt_name);
	      }

	    found_nonoption = 1;

	    break;

	  case 0:
	    /* long option found */

	    /* ignore if we do not want long options with two dashes */
	    if (!c->mode.longopt2dash
		&& linev[*skip_wordsp + optind - 1][1] == '-')
	      continue;

	    o = opts_long[n];

	    break;

	  default:
	    /* short option found */

	    o = opts_short[ch];

	    break;
	  }

	/* stop parsing if -m 'getoptstop' */
	if (curword_no < *skip_wordsp + optind - 1 && c->mode.getoptstop)
	  break;

	if (o && o->mode.subcommand)
	  {
	    char *subcmdname;

	    if (curword_no <= *skip_wordsp + optind - 1)
	      break;

	    subcmdname = optarg ? optarg : GETCHR (o->name);
	    *new_cmdnamep = (char *) malloc (strlen (GETCHR (c->name))
					     + strlen (subcmdname) + 2);
	    xpc_disable_if_not (*new_cmdnamep);
	    strcpy (*new_cmdnamep, GETCHR (c->name));
	    strcat (*new_cmdnamep, "\x1");
	    strcat (*new_cmdnamep, subcmdname);
	  }

	if (o && o->mode.cut)
	  {
	    if (curword_no <= *skip_wordsp + optind - 1)
	      break;

	    if (optarg)
	      {
		if (!*new_cmdnamep)
		  {
		    *new_cmdnamep = strdup (optarg);
		    xpc_disable_if_not (*new_cmdnamep);
		  }
		*skip_wordsp = optind - 1;
	      }

	    break;
	  }

	if (o && (o->mode.varname || found_nonoption))
	  {
	    char *varname;
	    char subopt_end_back = 0;
	    char *subopt_end = NULL;

	    varname = GETCHR (o->varname);

	    assert (varname);

	    if (is_curword)
	      {
		if (o->mode.suboptions && optarg)
		  {
		    if (o->subsep)
		      {
			/*FIXME: is it ok to write optarg? */
			if (strchr (optarg, o->subsep))
			  subopt_end = strrchr (optarg, o->subsep);
			else
			  subopt_end = optarg;

			subopt_end_back = *subopt_end;
			*subopt_end = 0;
		      }
		  }
		else
		  continue;
	      }



	    /* prepend 'PLUS_' to varname if we have a plus option */
	    if (o->mode.plusopt && word_start_plus[optind])
	      {
		char *newvarname;

		newvarname = (char *) malloc (strlen (varname) + 6);
		xpc_disable_if_not (newvarname);
		strcpy (newvarname, "PLUS_");
		newvarname[5] = 0;
		strcat (newvarname, varname);
		varname = newvarname;
	      }

	    if (o->mode.arg == 1 && !found_nonoption)
	      make_shellvar (varname, GETCHR (o->name), o->mode.array);
	    else
	      make_shellvar (varname, optarg, o->mode.array);

	    if (subopt_end)
	      *subopt_end = subopt_end_back;

	    if (o->mode.plusopt && word_start_plus[optind])
	      free (varname);
	  }
      }
  }

  /* switch options back to '+' */
  if (c->mode.plusopt)
    {
      int i;

      for (i = 1; i != linec - *skip_wordsp; i++)
	if (word_start_plus[i])
	  *(word_start_plus[i]) = '+';
    }

  if (xpc_debug)
    {
      fprintf (errstream, "\n");
      if (*current_nonopt)
	fprintf (errstream, "Current word is non-option word #%i\n",
		 *current_nonopt);
    }

  /*if (word_start_plus)
     free (word_start_plus);
     if (optstring)
     free (optstring);
     if (long_options)
     free (long_options);
     if (opts_long)
     free (opts_long);
   */
  return EXECUTION_SUCCESS;
}



static int
test (char *str)
{
  int res;
  WORD_LIST *list = NULL;
  WORD_LIST *expanded = NULL;

  if (!str || !*str)
    return EXECUTION_FAILURE;

  begin_unwind_frame ("xpcompcore_test");
  add_unwind_protect (dispose_words, list);
  add_unwind_protect (dispose_words, expanded);

  list = parse_string_to_word_list (str, "xpcomp test");

  expanded = expand_words_shellexp (list);

  res = test_builtin (expanded);

  run_unwind_frame ("xpcompcore_test");

  return res;
}



static void
get_compspec (XPC_OPT * o, char **patternp, COMPSPEC ** csp)
{
  assert (o);
  assert (patternp);
  assert (csp);

  if (o->mode.description_only)
    {
      *csp = (COMPSPEC *) GETCHR (o->compspec);
      return;
    }

  *csp = compspec_create ();
  xpc_disable_if_not (*csp);

  xpc_compspec_get (o->compspec, *csp);
  (*csp)->refcount = 2;

  if (o->mode.suboptions)
    {
      if (o->subsep)
	{
	  if (strchr (*patternp, o->subsep))
	    *patternp = strrchr (*patternp, o->subsep) + 1;
	}
      else if (*patternp)
	*patternp += strlen (*patternp);
    }

  if (o->mode.noappend)
    rl_completion_append_character = 0;
  else
    {
      if (o->mode.suboptions)
	rl_completion_append_character = o->subsep;
      else
	rl_completion_append_character = ' ';
    }
}



static int
get_option_compspec (XPC_CMD * c, char *word,
		     char **patternp, COMPSPEC ** csp, XPC_OPT_MODE * modep)
{
  XPC_OPT *o = NULL;
  XPC_OPT *o_long = NULL;
  XPC_OPT *o_short = NULL;

  int look_for_long = 0;
  int look_for_short = 0;
  int found_matches = 0;

  int plusopt = 0;
  char *cp;
  char *equ_sign = NULL;
  char *optarg_found = NULL;

  int do_find_pattern;

  XPC_COUNT n;

  assert (c);
  assert (word);
  assert (patternp);

  do_find_pattern = !(*patternp);

  cp = word;

  if (*cp != '-')
    {
      if (*cp == '+' && c->mode.plusopt)
	plusopt = 1;
      else
	{
	  if (do_find_pattern)
	    *patternp = cp;
	  return EXECUTION_SUCCESS;
	}
    }
  cp++;

  if (*cp == '-' && !plusopt && c->mode.longopt2dash)
    {
      cp++;
      look_for_long = 1;
      look_for_short = 0;
    }
  else
    {
      look_for_long = c->mode.longopt1dash;
      look_for_short = 1;
    }

  if (look_for_long)
    {
      /* find '=' and replace by 0 */
      equ_sign = strchr (cp, '=');

      if (equ_sign)
	*equ_sign = 0;
    }

  /* find the ruling option; if found, point cp to option argument */
  while (*cp)
    {
      if (look_for_long)
	{
	  /* look for matching long option */
	  found_matches = xpc_opt_match (c, cp, &o_long);

	  while (plusopt && o_long && !o_long->mode.plusopt)
	    found_matches = xpc_opt_match (c, cp, &o_long);
	}

      if (look_for_short)
	{
	  /* look if *cp is a valid short option; assign to o */
	  for (n = 0, o_short = GETOPT (c->opts + n);
	       n < c->opt_count; n++, o_short++)
	    {
	      char *optname = GETCHR (o_short->name);

	      if (*(optname + 1))
		{
		  o_short = NULL;
		  break;
		}
	      if (*cp == *optname)
		{
		  if (plusopt && !o_short->mode.plusopt)
		    o_short = NULL;
		  break;
		}
	    }			/* for */
	}


      if (!o_short && !o_long)
	{
	  /* return if no valid option found (wrong syntax) */
	  break;
	}
      else if (o_short && !o_long)
	{
	  o = o_short;
	  /* break if o may have an argument or is the last */
	  if (!*(cp + 1) || o->mode.arg != arg_none)
	    {
	      optarg_found = cp + 1;

	      if (!*optarg_found && !o->mode.glued)
		optarg_found = NULL;
	      break;
	    }
	}
      else if (!o_short && o_long)
	{
	  o = o_long;

	  if (equ_sign)
	    optarg_found = equ_sign + 1;
	  break;
	}
      else			/* if (o_short && o_long) */
	{
	  if (equ_sign)
	    optarg_found = equ_sign + 1;

	  o = (equ_sign || *(cp + 1) || *(cp - 1) != '-') ? o_long : o_short;

	  break;
	}

      cp++;
    }				/* while */

  /* restore the original word if it was changed (s.a.) */
  if (equ_sign)
    *equ_sign = '=';

  /* set completion */

  /* if we are called for the previous word (not interested in pattern)
     and no option argument was found return the completion */
  if (!do_find_pattern)
    {
      if (found_matches == 2)
	o = xpc_opt_find (c, cp);

      if (plusopt && o && !o->mode.plusopt)
	o = NULL;

      if (o && o->mode.arg != 1 && !optarg_found)
	{
	  get_compspec (o, patternp, csp);
	  *modep = o->mode;
	}


      return EXECUTION_SUCCESS;
    }

  /* if an argument is present return the completion
     if the option pattern is unique or an option of that name exists */
  if (optarg_found)
    {
      *patternp = optarg_found;

      if (found_matches == 2)
	o = xpc_opt_find (c, cp);

      if (plusopt && o && !o->mode.plusopt)
	o = NULL;

      if (o && o->mode.arg != arg_none)
	{
	  get_compspec (o, patternp, csp);
	  *modep = o->mode;
	}

      return EXECUTION_SUCCESS;
    }


  /* complete all matching option (no option argument was found) */

  *csp = compspec_create ();
  xpc_disable_if_not (*csp);

  (*csp)->refcount = 1;
  (*csp)->words = NULL;

  /* EXTRAOPTS feature */
  o = xpc_opt_find (c, "\x1\x1X");
  if (o)
    {
      STRINGLIST *sl;
      COMPSPEC *cs;

      cs = compspec_create ();
      xpc_disable_if_not (cs);
      add_unwind_protect (Xfree, cs);
      xpc_compspec_get (o->compspec, cs);

      sl = gen_compspec_completions (cs, NULL, cp, 0, 0);

      if (sl)
	{
	  int i;
	  int len = 0;

	  for (i = 0; i < sl->list_len; i++)
	    if (sl->list[i])
	      len += strlen (sl->list[i]) + 1;



	  (*csp)->words = malloc (len + c->opt_len + 2 * c->opt_count + 20);
	  xpc_disable_if_not ((*csp)->words);
	  *((*csp)->words) = 0;

	  for (i = 0; i < sl->list_len; i++)
	    {
	      assert (sl->list[i]);
	      if (!*sl->list[i])
		continue;
	      if (look_for_short && !*(sl->list[i] + 1))
		strcat ((*csp)->words, sl->list[i]);
	      else if (look_for_long && *(sl->list[i] + 1))
		strcat ((*csp)->words, sl->list[i]);

	      strcat ((*csp)->words, " ");
	    }
	}
    }

  if (!(*csp)->words)
    {
      (*csp)->words = malloc (c->opt_len + 2 * c->opt_count + 20);
      xpc_disable_if_not ((*csp)->words);
      *((*csp)->words) = 0;
    }

  if (o_short)
    {
      strcat ((*csp)->words, " ");
      strcat ((*csp)->words, GETCHR (o_short->name));
    }

  if (!o_short && !o_long)
    {
      if (look_for_short)
	{
	  for (n = 0, o_short = GETOPT (c->opts);; o_short++, n++)
	    {
	      char *optname;

	      if (n == c->opt_count)
		{
		  o_short = NULL;
		  break;
		}

	      optname = GETCHR (o_short->name);

	      if (*(optname + 1))
		{
		  o_short = NULL;
		  break;
		}

	      if (o_short->test && test (GETCHR (o_short->test)))
		continue;

	      if (plusopt && !o_short->mode.plusopt)
		continue;

	      strcat ((*csp)->words, " ");
	      strcat ((*csp)->words, optname);
	    }			/* for ( o_short ... */
	}
      if (look_for_long)
	xpc_opt_match (c, "", &o_long);

    }

  for (; o_long; xpc_opt_match (NULL, NULL, &o_long))
    {
      char *optname = GETCHR (o_long->name);

      if (o_long->test && test (GETCHR (o_long->test)))
	continue;

      if (plusopt && !o_long->mode.plusopt)
	continue;

      strcat ((*csp)->words, " ");
      strcat ((*csp)->words, optname);
    }

  rl_completion_append_character = ' ';

  *patternp = cp;

  return EXECUTION_SUCCESS;
}



static int
get_nonopt_compspec (XPC_CMD * c, int nonopt_no,
		     char **patternp, COMPSPEC ** csp, XPC_OPT_MODE * modep)
{
  XPC_OPT *o;
  char nonopt_name[4] = { 1, 1, 0, 0 };

  assert (csp);
  assert (modep);

  if (nonopt_no < 0 || nonopt_no > 9)
    return EXECUTION_SUCCESS;

  nonopt_name[2] = nonopt_no ? '0' + nonopt_no : 0;

  o = xpc_opt_find (c, nonopt_name);

  if (!o || (o->test && test (GETCHR (o->test))))
    return EXECUTION_SUCCESS;

  get_compspec (o, patternp, csp);
  *modep = o->mode;

  return EXECUTION_SUCCESS;
}



static void
gen_COMPREPLY (COMPSPEC * cs, char *pattern, SHELL_VAR ** COMPREPLYP)
{
  STRINGLIST *sl = NULL;
  ARRAY *a;
  register int i = 0;

  assert (COMPREPLYP);
  *COMPREPLYP = NULL;

  sl = gen_compspec_completions (cs, NULL, pattern, 0, 0);
  if (!sl || sl->list_len == 0)
    return;

  add_unwind_protect (strlist_dispose, sl);

  dispose_variable (find_variable ("COMPREPLY"));
  *COMPREPLYP = make_new_array_variable ("COMPREPLY");
  if (!*COMPREPLYP)
    return;

  /* create the array elements from the stringlist; the values are moved and
     NOT copied. the entries in the stringlist are zeroed to prevent them to
     be freed twice. */

  a = array_cell (*COMPREPLYP);
  i = sl->list_len - 1;
  a->max_index = i;

  while (i >= 0)
    {
      ARRAY_ELEMENT *new;
      char *val = (sl->list)[i];

      new = array_create_element (i, NULL);

      (sl->list)[i] = NULL;
      new->value = val;

      new->prev = a->head;
      new->next = a->head->next;

      a->head->next->prev = new;
      a->head->next = new;

      a->num_elements++;

      i--;
    }
}



/* finds the least common directory part in bytes length */
int
find_lcd (const char *a, const char *b)
{
  const char *x = a;
  const char *f = a;

  for (; *a && *b && *a == *b; a++, b++)
    if (*a == '/')
      f = a + 1;

  return (f - x);
}



/* append directory slashes to COMPREPLY */
static void
COMPREPLY_mark_directories (SHELL_VAR * COMPREPLY, int *dir_prefix_lenp)
{
  ARRAY *COMPREPLY_array = COMPREPLY ? array_cell (COMPREPLY) : NULL;
  ARRAY_ELEMENT *e;
  char *last_match = NULL;

  if (!COMPREPLY_array
      || !array_head (COMPREPLY_array) || array_empty (COMPREPLY_array))
    return;

  e = element_forw (COMPREPLY_array->head);
  while (e != COMPREPLY_array->head)
    {
      struct stat finfo;

      if (stat (element_value (e), &finfo) == 0 && S_ISDIR (finfo.st_mode))
	{
	  char *dirname = element_value (e);
	  if (dirname[strlen (dirname)] != '/')
	    {
	      element_value (e) =
		(char *) malloc (strlen (element_value (e)) + 2);
	      strcpy (element_value (e), dirname);
	      strcat (element_value (e), "/");
	      if (element_forw (e) == COMPREPLY_array->head)
		rl_completion_append_character = 0;
	      free (dirname);
	    }
	}
      if (!last_match)
	{
	  last_match = element_value (e);
	  *dir_prefix_lenp = strlen (element_value (e));
	}

      *dir_prefix_lenp = MIN (*dir_prefix_lenp,
			      find_lcd (element_value (e), last_match));
      last_match = element_value (e);

      e = element_forw (e);
    }
}



/* delete all items from COMPREPLY that do not match PATTERN */
static void
COMPREPLY_match (SHELL_VAR * COMPREPLY, const char *pattern)
{
  ARRAY *COMPREPLY_array = COMPREPLY ? array_cell (COMPREPLY) : NULL;
  ARRAY_ELEMENT *e;
  ARRAY_ELEMENT *d = NULL;

  /* do not match COMPREPLY against the pattern if filename comletion is done
     because it does funny things with quotes and does the checking anyway. */
  if (rl_filename_completion_desired)
    return;

  if (!COMPREPLY_array
      || !array_head (COMPREPLY_array) || array_empty (COMPREPLY_array))
    return;

  e = element_forw (COMPREPLY_array->head);

  while (e != COMPREPLY_array->head)
    {
      d = NULL;

      if (!element_value (e)
	  || strncmp (element_value (e), pattern, strlen (pattern)))
	d = e;

      if (d)
	{
	  e = element_forw (e);
	  d = array_remove (COMPREPLY_array, element_index (d));
	  if (d)
	    array_dispose_element (d);
	  continue;
	}
      e = element_forw (e);
    }
}



static void
print_description (const char *str)
{
  fprintf (stderr, "\n<%s>\n", str);
  rl_forced_update_display ();
}


static void
COMPREPLY_rlhint (SHELL_VAR * COMPREPLY, char **pattern)
{
  ARRAY *COMPREPLY_array = COMPREPLY ? array_cell (COMPREPLY) : NULL;
  ARRAY_ELEMENT *e;
  ARRAY_ELEMENT *d = NULL;

  if (!COMPREPLY_array
      || !array_head (COMPREPLY_array) || array_empty (COMPREPLY_array))
    return;

  e = element_forw (COMPREPLY_array->head);

  while (e != COMPREPLY_array->head)
    {
      d = NULL;

      if (!element_value (e) || !*element_value (e))
	d = e;

      if (!d && *element_value (e) == '\e')
	{
	  d = e;
	  if (!strcmp (element_value (e) + 1, "filename"))
	    rl_filename_completion_desired = 1;
	  else if (!strncmp (element_value (e) + 1, "append=", 7))
	    {
	      rl_completion_append_character = element_value (e)[8];
	      if (rl_completion_append_character == '0')
		rl_completion_append_character = 0;
	    }
	  else if (!strncmp (element_value (e) + 1, "prefix=", 7))
	    {
	      int len = strlen (element_value (e) + 8);
	      if (len && !strncmp (*pattern, element_value (e) + 8, len))
		*pattern += len;
	    }
	  else if (!strncmp (element_value (e) + 1, "description_only=", 17))
	    {
	      print_description (element_value (e) + 18);
	    }
	  else
	    {
	      if (xpc_debug)
		{
		  fprintf (errstream, "Unknown 'rlhint' tag: '%s'\n",
			   element_value (e) + 1);
		}
	    }
	}

      if (d)
	{
	  e = element_forw (e);
	  d = array_remove (COMPREPLY_array, element_index (d));
	  if (d)
	    array_dispose_element (d);
	  continue;
	}
      e = element_forw (e);
    }
}



static int
COMPREPLY_skip_or_prepend (SHELL_VAR * COMPREPLY, const char *prefix,
			   const int signed_prefix_len)
{
  ARRAY *COMPREPLY_array = COMPREPLY ? array_cell (COMPREPLY) : NULL;
  ARRAY_ELEMENT *e;
  ARRAY_ELEMENT *head = array_head (COMPREPLY_array);
  char *t;
  char *prev_match = NULL;
  int found_different_match = 0;

  if (!COMPREPLY_array || !head || array_empty (COMPREPLY_array))
    return EXECUTION_SUCCESS;

  if (signed_prefix_len < 0)
    {
      for (e = element_forw (head); e != head; e = element_forw (e))
	{
	  if (strncmp (element_value (e), prefix, -signed_prefix_len) != 0)
	    return EXECUTION_FAILURE;
	}
    }

  e = element_forw (head);

  while (e != head)
    {
      int newlen = strlen (element_value (e)) + signed_prefix_len;

      t = malloc (newlen + 1);
      xpc_disable_if_not (t);

      if (signed_prefix_len > 0)
	{
	  assert (prefix && *prefix);
	  strcpy (t, prefix);
	  strcat (t, element_value (e));
	}
      else
	strcpy (t, element_value (e) - signed_prefix_len);

      free (element_value (e));

      e->value = t;

      /* check if all completions are equal */
      if (!found_different_match)
	{
	  if (prev_match && strcmp (element_value (e), prev_match) != 0)
	    found_different_match = 1;
	  prev_match = element_value (e);
	}

      e = element_forw (e);
    }

  if (!found_different_match)
    return EXECUTION_FAILURE;

  return EXECUTION_SUCCESS;
}



/* copied from readline/complete.c */
static int
query_display_matches (int len)
{
  int c;
  int answer;

  rl_crlf ();
  fprintf (rl_outstream, "Display all %d possibilities? (y or n)", len);
  fflush (rl_outstream);

  for (;;)
    {
      RL_SETSTATE (RL_STATE_MOREINPUT);
      c = rl_read_key ();
      RL_UNSETSTATE (RL_STATE_MOREINPUT);

      answer = 1;
      if (c == 'y' || c == 'Y' || c == ' ')
	break;
      answer = 0;
      if (c == 'n' || c == 'N' || c == RUBOUT)
	break;
      if (c == ABORT_CHAR)
	_rl_abort_internal ();
      rl_ding ();
    }

  if (answer == 0)
    {
      rl_crlf ();

      rl_forced_update_display ();
      rl_display_fixed = 1;
    }

  return answer;
}



static void
display_match_list (char **matches, int len, int max)
{
  int new_max = 0;
  int new_len = 0;
  char **new_matches = NULL;
  int signed_prefix_len_local = signed_prefix_len;
  char *prefix_local = prefix;
  register int i;
  int called_after_xpcompcore = 1;

  if (signed_prefix_len == 0)
    {
      fprintf (errstream,
	       "\nxpcompcore: signed_prefix_len == 0 but called anyway\n");
      called_after_xpcompcore = 0;
    }

  if (signed_prefix_len > 0 && prefix == NULL)
    {
      fprintf (errstream, "\nxpcompcore: prefix missing\n");
      called_after_xpcompcore = 0;
    }

  /* cleanup first */
  signed_prefix_len = 0;
  prefix = NULL;		/* gets freed at end  */
  rl_completion_display_matches_hook = old_rl_completion_display_matches_hook;

  /* the line must be the same exept for some insertions at point,
     else we are not called for the same completion. */
  if (strncmp (old_rl_line_buffer, rl_line_buffer, old_rl_point)
      || strcmp (old_rl_line_buffer + old_rl_point,
		 rl_line_buffer + rl_point))
    called_after_xpcompcore = 0;

  if (called_after_xpcompcore)
    {
      new_matches = (char **) malloc (sizeof (char *) * (len + 2));
      if (!new_matches)
	{
	  free (prefix_local);
	  return;
	}

      if (signed_prefix_len_local > 0)
	{
	  for (i = 0; i <= len; i++)
	    if (strlen (matches[i]) >= (unsigned int) signed_prefix_len_local)
	      new_matches[new_len++] = matches[i] + signed_prefix_len_local;
	}
      else
	{
	  for (i = 0; i <= len; i++)
	    {
	      assert (prefix_local && *prefix_local);

	      new_matches[new_len] = malloc (strlen (matches[i])
					     - signed_prefix_len_local + 1);
	      if (!new_matches[new_len])
		{
		  free_argv (new_matches);
		  return;
		}
	      strcpy (new_matches[new_len], prefix_local);
	      strcat (new_matches[new_len], matches[i]);
	      if (rl_filename_completion_desired)
		{
		  char *chp;
		  int displen;
		  chp = strrchr (new_matches[new_len], '/');
		  displen = strlen (chp ? chp + 1 : new_matches[new_len]);
		  if (displen > new_max)
		    new_max = displen;
		}

	      new_len++;
	    }
	}

      new_matches[new_len] = NULL;

      matches = new_matches;
      len = new_len - 1;
      if (rl_filename_completion_desired)
	max = new_max;
      else
	max -= signed_prefix_len_local;
      /*
         rl_display_match_list (new_matches, new_len - 1,
         max - signed_prefix_len_local); */
    }
  else				/* ! called_after_xpcompcore */
    {
      /* filename completion changes the readline buffer
         so the prefix may not match, this is ok and we do
         the plain display in this case without error message */
      if (!(rl_filename_completion_desired && rl_filename_quoting_desired))
	fprintf (errstream, "\nxpcompcore: Leftover hook\n");

    }

  if (len < rl_completion_query_items || query_display_matches (len))
    {
      rl_display_match_list (matches, len, max);
      rl_forced_update_display ();
      rl_display_fixed = 1;
    }

  free (prefix_local);
  if (old_rl_line_buffer)
    {
      free (old_rl_line_buffer);
      old_rl_line_buffer = NULL;
    }

  if (called_after_xpcompcore)
    {
      if (signed_prefix_len_local > 0)
	free (new_matches);
#if 0
      /* FIXME: now here is a memory leak but leaving
       * free_argv in place causes bug #7515 */ 
      else
	free_argv (new_matches);
#endif
    }
}



static int
cleanup_rl_hook (void)
{
  rl_completion_append_character = old_rl_completion_append_character;

  if (old_rl_event_hook)
    old_rl_event_hook ();

  return 1;			/* not used */
}



/* Dispose of all used shell variables and free internal list. */
static void
delete_shellvars ()
{
  LVARLIST *vl;
  vl = shellvar_first;
  while (vl)
    {
      LVARLIST *old = vl;
      makunbound (vl->varname, shell_variables);
      vl = vl->next;
      free (old->varname);
      free (old);
    }
  shellvar_first = NULL;
}



static COMPSPEC *
get_default_compspec ()
{
  COMPSPEC *cs = compspec_create ();
  xpc_disable_if_not (cs);
  cs->refcount = 1;
  cs->actions = CA_FILE;
  return cs;
}



static int
strdiff (const char *str1, const char *str2, char **diffp, int *lenp)
{
  assert (str1);
  assert (str2);
  assert (diffp);
  assert (lenp);

  *lenp = strlen (str1) - strlen (str2);

  if (!*lenp)
    return strcmp (str1, str2) ? EXECUTION_FAILURE : EXECUTION_SUCCESS;

  assert (!*diffp);

  *diffp = malloc (ABS (*lenp) + 1);

  xpc_disable_if_not (*diffp);

  if (*lenp > 0)
    {
      if (strcmp (str1 + *lenp, str2))
	return EXECUTION_FAILURE;
      strncpy (*diffp, str1, (unsigned int) *lenp);
      *(*diffp + *lenp) = 0;
    }
  else
    {
      if (strcmp (str1, str2 - *lenp))
	return EXECUTION_FAILURE;
      strncpy (*diffp, str2, (unsigned int) (-*lenp));
      *(*diffp - *lenp) = 0;
    }
  return EXECUTION_SUCCESS;
}



static void
set_state_ok ()
{
#if defined(RESTRICTED_SHELL)
  if (xpc_restrict)
    restricted--;
#endif
  xpcomp_state = XPC_OK;
}



static void
xpcompcore_init ()
{
  WORD_LIST *wl;

  /* i do not want command expansion */
  rl_directory_completion_hook = NULL;

  /* check for the split_at_delims() bug */
#if defined (SHELL_VAR_COMP_CPOINT)
  wl = split_at_delims ("a>b", 3, ">", 0, 0, 0, 0);
#else
  wl = split_at_delims ("a>b", 3, ">", 0, 0, 0);
#endif
  if (!wl || !wl->next || !wl->next->word || !wl->next->word->word)
    return;

  switch (wl->next->word->word[0])
    {
    case '>':
      split_at_delims_bug = 0;
      break;
    case 'b':
      split_at_delims_bug = 1;
      break;
    default:
      return;
    }
}



static int
xpcompcore_builtin (args)
     WORD_LIST *args;
{
  /* the commmand whose line is completed */
  XPC_CMD *c;

  /* state of the completion finding process */
  COMPLETION_STEPS step;

  /* the name of the commmand whose line is completed */
  char *cmdname;

  /* new command name that was found during line parsing */
  char *new_cmdname = NULL;

  /* the word where the pointer is on */
  char *curword = NULL;

  /* the word before the one the pointer is on */
  char *prevword;

  /* the matches are passed back to the completion algorithm via COMPLREPLY */
  SHELL_VAR *COMPREPLY;

  /* to be completed pattern as found by readline */
  char *pattern_rl = NULL;

  /* to be completed pattern as found by xpc_get_option_compspec */
  char *pattern = NULL;

  /* common directory prefix length of all filname matches */
  int dir_prefix_len = 0;

  /* return value of the latest called function */
  int res;

  /* completion specifier if found */
  COMPSPEC *cs = NULL;

  /* option mode */
  XPC_OPT_MODE mode = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

  /* non-zero if current word is a non-option argument;
     value gives the ordinal number */
  int nonopt_no = 0;

  /* the command line in ARGC/ARGV style */
  char **linev = NULL;
  int linec = 0;
  int curword_no = 0;

  /* how many words to skip from the commandline (when parsing with getopt
     leading word may be skipped and the command name changed) */
  int skip_words = 0;

  /* A flag to execute some initialization code once after the builtin has been
     loaded */
  static int xpcompcore_initialize = 1;

  if (xpcompcore_initialize)
    {
      xpcompcore_init ();
      xpcompcore_initialize = 0;
    }

  if (xpcomp_state)
    {
      fprintf (stderr, "\nxpcompcore: builtin disabled.\n");
      rl_forced_update_display ();
      return EXECUTION_FAILURE;
    }

  begin_unwind_frame ("xpcompcore");
#if defined(RESTRICTED_SHELL)
  if (xpc_restrict)
    restricted++;
#endif
  xpcomp_state = XPC_EXPANDING;
  add_unwind_protect ((Function *) set_state_ok, NULL);
  add_unwind_protect ((Function *) Xfree, &new_cmdname);
  add_unwind_protect ((Function *) delete_shellvars, NULL);

  if (setjmp (self_destruct))
    {
      /* FIXME: I do neither know if this is necessary nor if its
         problematic. */
      discard_unwind_frame ("xpcompcore");

      xpcomp_state = XPC_DISABLED;
      fprintf (stderr,
	       "The xpcomp builtins have been disabled."
	       "  Further calls will silently fail.\n");
      return EXECUTION_FAILURE;
    }

  if (xpc_debug)
    fprintf (errstream, "\n");

  if (install_rl_event_hook)
    {
      if (rl_event_hook != cleanup_rl_hook)
	old_rl_event_hook = rl_event_hook;
      rl_event_hook = cleanup_rl_hook;
      old_rl_completion_append_character = rl_completion_append_character;
    }

  if (get_line (args, &linec, &linev, &curword_no, &pattern_rl))
    {
      run_unwind_frame ("xpcompcore");
      return EXECUTION_FAILURE;
    }

  cmdname = linev[0];
  c = 0;
  res = EXECUTION_SUCCESS;
  step = COMPL_SHELLVAR;
  while (!res && !cs)
    {
      errno = 0;
      switch (step++)
	{
	case COMPL_SHELLVAR:
	  if ((pattern = shell_var_start (pattern_rl)))
	    {
	      cs = compspec_create ();
	      xpc_disable_if_not (cs);
	      cs->refcount = 1;
	      cs->actions = CA_VARIABLE;
	    }
	  break;

	case COMPL_REDIRECTIONS:
	  if (!curword_no)
	    {
	      cs = get_default_compspec ();
	      pattern = pattern_rl;
	    }
	  break;

	case COMPL_FIND_COMMAND:

	  c = xpc_cmd_find_all (cmdname);
	  if (!c)
	    {
	      /* find basename of command */
	      if (cmdname && strchr (cmdname, '/'))
		cmdname = strrchr (cmdname, '/') + 1;
	      c = xpc_cmd_find_all (cmdname);
	    }

	  if (!c)
	    {
	      pattern = pattern_rl;
	      step = COMPL_DEFAULT;
	    }

	  break;

	case COMPL_GETOPT_PARSE:

	  if (!c->mode.getoptparse)
	    break;

	  res = getopt_parse (c, linec, linev, curword_no,
			      &skip_words, &new_cmdname, &nonopt_no);
	  if (new_cmdname)
	    {
	      cmdname = new_cmdname;
	      step = COMPL_FIND_COMMAND;
	    }

	  break;

	case COMPL_CURRENT_WORD_OPTION:
	  curword = linev[curword_no];
	  xpc_disable_if_not (curword);
	  /* PATTERN is NULL and will be set by get_option_compspec.  CS and
	     MODE will only be set if an option is found in the current word. */
	  res = get_option_compspec (c, curword, &pattern, &cs, &mode);
	  break;

	case COMPL_PREVIOUS_WORD_OPTION:
	  if (curword_no <= 1 + skip_words)
	    break;
	  prevword = linev[curword_no - 1];
	  /* PATTERN was already set by the COMPL_CURRENT_WORD_OPTION step.  CS
	     and MODE will only be set if an option is found in the previous
	     word that does not have its argument in the same word. */
	  res = get_option_compspec (c, prevword, &pattern, &cs, &mode);
	  break;

	case COMPL_NUMBERED_NONOPTION:
	  if (nonopt_no)
	    res = get_nonopt_compspec (c, nonopt_no, &pattern, &cs, &mode);
	  break;

	case COMPL_NONOPTION:
	  res = get_nonopt_compspec (c, 0, &pattern, &cs, &mode);
	  break;

	case COMPL_DEFAULT:
	  cs = get_default_compspec ();
	  break;

	default:
	  assert (!"Unexpected completion step.");
	}
    }

  if (!cs)
    goto cleanup;

  if (mode.description_only)
    {
      print_description ((char *) cs);
      goto cleanup;
    }

  assert (!res);

  if (cs->refcount == 1)
    add_unwind_protect (compspec_dispose, cs);
  else
    add_unwind_protect (free, cs);

  if (xpc_debug)
    {
      fprintf (errstream, "Compspec: ");
      xpc_compspec_print (errstream, cs, 0);
      fprintf (errstream, "\n");
    }

  if ((cs->actions & CA_FILE) || mode.filename)
    {
      size_t pattern_len = strlen (pattern);
      size_t pattern_rl_len = strlen (pattern_rl);

      rl_filename_completion_desired = 1;

      if (pattern_len > pattern_rl_len)
	while (strlen (pattern) > pattern_rl_len
	       && (pattern[0] == '"' || pattern[0] == '\''))
	  pattern++;
      else if (pattern_len < pattern_rl_len)
	while (strlen (pattern) < pattern_rl_len
	       && (pattern[-1] == '"' || pattern[-1] == '\''))
	  pattern--;
    }

  if (prefix)
    {
      fprintf (errstream, "\nxpcompcore: Leftover prefix.\n");

      rl_completion_display_matches_hook =
	old_rl_completion_display_matches_hook;
      free (prefix);
      prefix = NULL;
    }

  make_shellvar ("PATTERN", pattern, 0);
  gen_COMPREPLY (cs, pattern, &COMPREPLY);

  if (!COMPREPLY || !array_cell (COMPREPLY))
    goto cleanup;

  if (mode.rlhint)
    COMPREPLY_rlhint (COMPREPLY, &pattern);

  if (mode.description_only)
    {
      /* print (via echo) COMPREPLY as the description and delete it */
      WORD_LIST *wl;

      array_rshift (array_cell (COMPREPLY), 1, "<");
      array_rshift (array_cell (COMPREPLY), 1, "-e");
      array_insert (array_cell (COMPREPLY),
		    array_max_index (array_cell (COMPREPLY)) + 1, ">");
      wl = array_to_word_list (array_cell (COMPREPLY));
      add_unwind_protect ((Function *) dispose_words, wl);
      array_dispose (array_cell (COMPREPLY));
      array_cell (COMPREPLY) = NULL;
      echo_builtin (NULL);
      echo_builtin (wl);
      rl_forced_update_display ();
      goto cleanup;
    }

  COMPREPLY_match (COMPREPLY, pattern);

  res = strdiff (pattern_rl, pattern, &prefix, &signed_prefix_len);

  if (!prefix)
    goto cleanup;

  if (rl_filename_completion_desired)
    {
      rl_filename_completion_desired = 0;
      COMPREPLY_match (COMPREPLY, pattern);
      COMPREPLY_mark_directories (COMPREPLY, &dir_prefix_len);
    }

  if (COMPREPLY_skip_or_prepend (COMPREPLY, prefix, signed_prefix_len)
      || array_num_elements (array_cell (COMPREPLY)) < 2
      || (rl_completion_type != '!' && rl_completion_type != '?'))
    {
      if (prefix)
	{
	  free (prefix);
	  prefix = NULL;
	}
      goto cleanup;
    }

  /* this this modifies PREFIX and SIGNED_PREFIX_LEN so that
     the common directory part of the completed filename wont be displayed */
  if (dir_prefix_len)
    {
      int new_signed_prefix_len = signed_prefix_len + dir_prefix_len;
      char *new_prefix = (char *) malloc (ABS (new_signed_prefix_len) + 1);

      if (signed_prefix_len > 0)
	{
	  strcpy (new_prefix, prefix);
	  strncat (new_prefix,
		   element_value (element_forw
				  ((array_cell (COMPREPLY))->head)),
		   dir_prefix_len);
	}
      else
	{
	  if (new_signed_prefix_len > 0)
	    {
	      strncpy (new_prefix,
		       element_value (element_forw
				      ((array_cell (COMPREPLY))->head)) -
		       signed_prefix_len, new_signed_prefix_len);
	    }
	  else
	    {
	      strncpy (new_prefix, prefix + dir_prefix_len,
		       -new_signed_prefix_len);
	    }
	}
      free (prefix);

      prefix = new_prefix;
      signed_prefix_len = new_signed_prefix_len;
    }

  if (rl_completion_display_matches_hook != display_match_list)
    {
      old_rl_completion_display_matches_hook =
	rl_completion_display_matches_hook;
      rl_completion_display_matches_hook = (VFunction *) display_match_list;
    }

  if (old_rl_line_buffer)
    free (old_rl_line_buffer);
  old_rl_line_buffer = strdup (rl_line_buffer);
  xpc_disable_if_not (old_rl_line_buffer);

  old_rl_point = rl_point;
cleanup:

  run_unwind_frame ("xpcompcore");

  if (xpc_debug)
    {
      if (res)
	fprintf (errstream, "EXECUTION_FAILURE\n");

      if (errstream == stderr)
	rl_forced_update_display ();
    }

  if (errno && res)
    perror ("");

  return res;
}



struct builtin xpcompcore_struct = {
  "xpcompcore",
  xpcompcore_builtin,
  BUILTIN_ENABLED,
  xpcompcore_doc,
  "xpcompcore",
  (char *) 0
};
