/*
  optcomplete_builtin.c - option completion for xpcomp

  Copyright (C) 2001 Ingo K"ohne
  
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
*/

#include "defines.h"

#include <errno.h>
#include <stdio.h>
#include <ctype.h>		/* isalnum() */

#include "bashansi.h"		/* malloc() */
#include "shell.h"
#include "builtins.h"
#include "builtins/bashgetopt.h"
#include "builtins/common.h"	/* builtin_error */

#include "xpcomp.h"

extern int complete_builtin (WORD_LIST * args);

/*  Main actions for the optcomplete builtin.  */
typedef enum xpc_command
{
  XPC_UNKNOWN = 0,
  XPC_ADD,
  XPC_REMOVE,
  XPC_SETMODE,
  XPC_HELPMODE,
  XPC_SHOW
}
XPC_COMMAND;

static const char *MSG_EXCLUSIVE =
  "The options -p -r are mutually exclusive.";



static char *optcomplete_doc[] = {
  "The documentation is in the man page.",
  (char *) NULL
};



static int
is_valid_varname (const char *name)
{
  const unsigned char *cp;

  for (cp = name; *cp; cp++)
    if ((!isalnum (*cp) && *cp != '_') || *cp > 0x7f)
      break;

  return *cp ? 0 : 1;
}



/*  Return internal representation of external option name OPTNAME.  */
static char *
get_internal_optname (char *optname)
{
  char *internal_optname;

  if (!optname || !*optname)
    {
      builtin_error ("Empty option name");
      return NULL;
    }

  if (!strncmp (optname, "NONOPT", 6))
    {
      char nonopt_name[4] = { 1, 1, 0, 0 };

      if (strlen (optname) > 7 || (optname[6] && !isdigit (optname[6])))
	{
	  builtin_error ("Invalid non-option word specifier: %s", optname);
	  return NULL;
	}
      nonopt_name[2] = optname[6];

      internal_optname = strdup (nonopt_name);
    }
  else if (!strcmp (optname, "EXTRAOPTS"))
    internal_optname = strdup ("\x1\x1X");
  else
    internal_optname = strdup (optname);

  xpc_disable_if_not (internal_optname);

  return internal_optname;
}



/*  Print external representation of internal option name NAME.  */
static void
print_optname (char *name)
{
  if (name[0] == 0x1)
    {
      char c = name[2];

      if (c == 0)
	printf ("NONOPT");
      else if (c >= '0' && c <= '9')
	printf ("NONOPT%c", c);
      else if (c == 'X')
	printf ("EXTRAOPTS");
      else
	printf ("<unknown>");
    }
  else
    printf ("'%s'", name);
}



/*  Substitute all colons in STR by 0x1.  */
static void
subst_colon (char *str)
{
  while (str)
    {
      str = strchr (str, ':');
      if (str)
	*(str++) = 0x1;
    }
}



/*  Set command mode of C from keywords in MODESTR.  */
static int
xpc_cmd_setmode (XPC_CMD * c, const char *modestr)
{
  const char *cp;
  size_t len;
  XPC_CMD_MODE mode;

  assert (c);

  memset (&mode, 0, sizeof (XPC_CMD_MODE));

  for (cp = modestr; cp && *cp; cp += len)
    {
      char *end;

      if (*cp == ',')
	cp++;

      end = strchr (cp, ',');
      len = end ? end - cp : strlen (cp);

      if (len == 0)
	continue;

      if (len == 5 && strncmp ("clear", cp, len) == 0)
	memset (&mode, 0, sizeof (XPC_CMD_MODE));
      else if (len == 5 && strncmp ("short", cp, len) == 0)
	{
	  mode.longopt1dash = 0;
	  mode.longopt2dash = 0;
	}
      else if (len == 10 && strncmp ("singledash", cp, len) == 0)
	{
	  mode.longopt1dash = 1;
	  mode.longopt2dash = 0;
	}
      else if (len == 4 && strncmp ("long", cp, len) == 0)
	{
	  mode.longopt1dash = 0;
	  mode.longopt2dash = 1;
	}
      else if (len == 8 && strncmp ("longonly", cp, len) == 0)
	{
	  mode.longopt1dash = 1;
	  mode.longopt2dash = 1;
	}
      else if (len == 10 && strncmp ("getoptstop", cp, len) == 0)
	mode.getoptstop = 1;
      else if (len == 11 && strncmp ("getoptparse", cp, len) == 0)
	mode.getoptparse = 1;
      else if (len == 12 && strncmp ("prefinvisible", cp, len) == 0)
	mode.prefinvisible = 1;
      else if (len == 7 && strncmp ("plusopt", cp, len) == 0)
	mode.plusopt = 1;
      else
	{
	  builtin_error
	    ("Error parsing command mode string '%s' at character %i.",
	     modestr, cp - modestr);
	  return EX_USAGE;
	}
    }
  c->mode = mode;
  return EXECUTION_SUCCESS;
}



/*  Set option mode of O from keywords in MODESTR.  */
static int
xpc_opt_setmode (XPC_OPT_MODE * mode, const char *modestr)
{
  const char *cp;
  size_t len;
  XPC_OPT_MODE new_mode;

  assert (mode);

  memset (&new_mode, 0, sizeof (XPC_OPT_MODE));

  for (cp = modestr; cp && *cp; cp += len)
    {
      char *end;

      if (*cp == ',')
	cp++;

      end = strchr (cp, ',');
      len = end ? end - cp : strlen (cp);

      if (len == 0)
	continue;

      if (len == 5 && strncmp ("clear", cp, len) == 0)
	memset (&new_mode, 0, sizeof (XPC_OPT_MODE));
      else if (len == 8 && strncmp ("arg_none", cp, len) == 0)
	new_mode.arg = arg_none;
      else if (len == 12 && strncmp ("arg_optional", cp, len) == 0)
	new_mode.arg = arg_optional;
      else if (len == 12 && strncmp ("arg_required", cp, len) == 0)
	new_mode.arg = arg_required;
      else if (len == 8 && strncmp ("noappend", cp, len) == 0)
	new_mode.noappend = 1;
      else if (len == 10 && strncmp ("suboptions", cp, len) == 0)
	new_mode.suboptions = 1;
      else if (len == 8 && strncmp ("filename", cp, len) == 0)
	new_mode.filename = 1;
      else if (len == 3 && strncmp ("cut", cp, len) == 0)
	new_mode.cut = 1;
      else if (len == 10 && strncmp ("subcommand", cp, len) == 0)
	new_mode.subcommand = 1;
      else if (len == 5 && strncmp ("array", cp, len) == 0)
	new_mode.array = 1;
      else if (len == 7 && strncmp ("plusopt", cp, len) == 0)
	new_mode.plusopt = 1;
      else if (len == 6 && strncmp ("rlhint", cp, len) == 0)
	new_mode.rlhint = 1;
      else if (len == 5 && strncmp ("glued", cp, len) == 0)
	new_mode.glued = 1;
      else
	{
	  builtin_error
	    ("Error parsing option mode string '%s' at character %i.",
	     modestr, cp - modestr);
	  return EX_USAGE;
	}
    }
  *mode = new_mode;
  return EXECUTION_SUCCESS;
}


/*  Print command mode specifier (for use as arg of -m) for command C.  */
static void
xpc_cmd_printmode (XPC_CMD * c)
{
  if (c->mode.longopt1dash && c->mode.longopt2dash)
    fprintf (stdout, "longonly,");
  else if (c->mode.longopt2dash)
    fprintf (stdout, "long,");
  else if (c->mode.longopt1dash)
    fprintf (stdout, "singledash,");
  else
    fprintf (stdout, "short,");

  if (c->mode.getoptparse)
    fprintf (stdout, "getoptparse,");
  if (c->mode.getoptstop)
    fprintf (stdout, "getoptstop,");
  if (c->mode.prefinvisible)
    fprintf (stdout, "prefinvisible,");
}



/*  Print opion mode specifier (for use as arg of -m) for option O.  */
static void
xpc_opt_printmode (XPC_OPT * o)
{
  char buf[256];
  char *cp = buf;

  switch (o->mode.arg)
    {
    case arg_none:
      if (o->compspec != COMPSPEC_EMPTY)
	cp += sprintf (cp, "arg_none,");
      break;
    case arg_optional:
      cp += sprintf (cp, "arg_optional,");
      break;
    case arg_required:
      if (o->compspec == COMPSPEC_EMPTY)
	cp += sprintf (cp, "arg_required,");
    default:
      break;
    }

  if (o->mode.noappend)
    cp += sprintf (cp, "noappend,");
  if (o->mode.suboptions)
    cp += sprintf (cp, "suboptions,");
  if (o->mode.filename)
    cp += sprintf (cp, "filename,");
  if (o->mode.cut)
    cp += sprintf (cp, "cut,");
  if (o->mode.subcommand)
    cp += sprintf (cp, "subcommand,");
  if (o->mode.array)
    cp += sprintf (cp, "array,");
  if (o->mode.plusopt)
    cp += sprintf (cp, "plusopt,");
  if (o->mode.rlhint)
    cp += sprintf (cp, "rlhint,");
  if (o->mode.glued)
    fprintf (stdout, "glued,");

  if (cp != buf)
    {
      assert (buf < cp);
      assert (cp < buf + 256);
      *(--cp) = 0;
      printf (" -m %s", buf);
    }
}



/* Print optcomplete command line for option O. */
static void
xpc_opt_print (XPC_OPT * o)
{
  printf ("optcomplete ");

  if (o->mode.suboptions)
    printf (" -O %c", o->subsep ? o->subsep : '0');

  if (o->mode.varname)
    printf (" -V %s", sh_single_quote (GETCHR (o->varname)));

  if (o->test)
    printf (" -T %s", sh_single_quote (GETCHR (o->test)));

  if (o->mode.added_info)
    printf (" -I");

  xpc_opt_printmode (o);

  if (o->mode.description_only)
    printf (" -D '%s'", GETCHR (o->compspec));
  else
    xpc_compspec_print (stdout, GETCOMPSPEC (o->compspec), 1);

  printf (" $cmd  ");
  print_optname (GETCHR (o->name));
  printf ("\n");
}



/*  Print shell script for all settings of CMDNAME.  PATTERN, if exists,
    restricts which command options are printed.  */
static void
printopts (char *cmdname, const char *pattern)
{
  XPC_CMD *c;
  XPC_OPT *o;
  XPC_COUNT n;

  if (!cmdname || !*cmdname)
    return;

  c = xpc_cmd_find_all (cmdname);
  if (!c)
    return;

  while (c)
    {
      printf ("\ncmd=");
      {
	char *cp;
	for (cp = GETCHR (c->name); *cp; cp++)
	  if (*cp == 0x1)
	    putchar (':');
	  else
	    putchar (*cp);
	putchar ('\n');
      }

      if (pattern)
	{
	  xpc_opt_match (c, pattern, &o);
	  while (o)
	    {
	      xpc_opt_print (o);
	      xpc_opt_match (NULL, NULL, &o);
	    }
	}
      else
	for (n = c->opts; n < c->opts + c->opt_count; n++)
	  {
	    o = GETOPT (n);
	    xpc_opt_print (o);
	  }

      printf ("optcomplete -m ");
      xpc_cmd_printmode (c);
      printf ("  $cmd\n");

      if ((XPC_COUNT) (c - cache->cmds) < cache->idx->cmd_count - 1
	  && !strncmp (cmdname, GETCHR ((c + 1)->name), strlen (cmdname))
	  && GETCHR ((c + 1)->name)[strlen (cmdname)] == 1)
	c++;
      else
	c = NULL;
    }
}



static void
force_cleanup ()
{
  fprintf (stderr,
	   "The optcomplete builtin has been interrupted. Due to currently missing protection mechanisms the new cache will be deleted, sorry.\n");
  xpc_heap_cache_delete ();
}



static int
optcomplete_builtin (args)
     WORD_LIST *args;
{
  int c;
  int last_result;

  XPC_COMMAND command;

  int have_mode;

  char *cmdname;
  COMPSPEC *cs;
  char *test;
  char subsep;
  char *varname;
  char *optname;
  char *modestr;
  char *description;
  int added_info;

  WORD_LIST *complete_args;
  WORD_LIST *options;

  /* loop counter; for easy detecting wrong syntax */
  int getopt_count;


  if (setjmp (self_destruct))
    {
      xpcomp_state = XPC_DISABLED;
      fprintf (stderr,
	       "The xpcomp builtins have been disabled.  "
	       "Further calls will silently fail.\n");
      return EXECUTION_FAILURE;
    }

  if (xpcomp_state)
    switch (xpcomp_state)
      {
      case XPC_UNINITIALIZED:

	errstream = stderr;
	if (xpc_libinit ())
	  return EXECUTION_FAILURE;
	break;

      case XPC_EXPANDING:

	return EXECUTION_FAILURE;

      case XPC_DISABLED:

	return EXECUTION_FAILURE;

      case XPC_OK:

	break;
      }

  errno = 0;

  /* FIXME: I really have to look if the builtin can be interrupted without
     evaluating script input. If not, the unwind protection is not really
     necessary here. */
  begin_unwind_frame ("optcomplete");
  add_unwind_protect ((Function *) force_cleanup, NULL);

  reset_internal_getopt ();

  /* initialize all dynamic variables */
  last_result = EXECUTION_SUCCESS;
  command = XPC_UNKNOWN;
  have_mode = 0;
  cmdname = NULL;
  cs = NULL;
  test = NULL;
  subsep = -1;
  varname = NULL;
  optname = NULL;
  modestr = NULL;
  complete_args = NULL;
  options = NULL;
  description = NULL;
  added_info = 0;
  getopt_count = 0;

  while (!last_result
	 && (c =
	     internal_getopt (args,
			      "m:O:T:V:abcdefjkpruvA:D:G:IW:P:S:X:F:C:")) !=
	 -1)
    {
      char tmpword[3] = { '-', 'X', 0 };

      getopt_count++;

      switch (c)
	{

	case 'A':
	case 'G':
	case 'W':
	case 'P':
	case 'S':
	case 'X':
	case 'F':
	case 'C':

	  complete_args = make_word_list (make_bare_word (list_optarg),
					  complete_args);

	case 'a':
	case 'b':
	case 'c':
	case 'd':
	case 'e':
	case 'f':
	case 'j':
	case 'k':
	case 'u':
	case 'v':

	  tmpword[1] = c;
	  complete_args = make_word_list (make_bare_word (tmpword),
					  complete_args);
	  break;

	case 'D':

	  if (!*list_optarg)
	    builtin_error ("Missing description.");
	  description = list_optarg;
	  break;

	case 'I':

	  added_info = 1;
	  break;

	case 'm':

	  if (!*list_optarg)
	    {
	      builtin_error ("Empty mode, use -m clear if you wanted this.");
	      last_result = EX_USAGE;
	      break;
	    }

	  have_mode = 1;
	  modestr = list_optarg;
	  break;

	case 'p':

	  if (command)
	    {
	      builtin_error (MSG_EXCLUSIVE);
	      last_result = EX_USAGE;
	      break;
	    }
	  command = XPC_SHOW;
	  break;

	case 'r':

	  if (command)
	    {
	      builtin_error (MSG_EXCLUSIVE);
	      last_result = EX_USAGE;
	      break;
	    }
	  command = XPC_REMOVE;
	  break;

	case 'O':

	  if (strlen (list_optarg) != 1)
	    {
	      builtin_error ("Suboption separator must be one character.");
	      last_result = EX_USAGE;
	      break;
	    }
	  subsep = *list_optarg;
	  if (subsep == '0')
	    subsep = 0;

	  break;

	case 'T':

	  if (!*list_optarg)
	    {
	      builtin_error ("Empty test argument.");
	      last_result = EX_USAGE;
	    }
	  test = list_optarg;
	  break;

	case 'V':

	  if (!*list_optarg)
	    {
	      builtin_error ("Empty variable name.");
	      last_result = EX_USAGE;
	      break;
	    }

	  if (!is_valid_varname (list_optarg))
	    {
	      builtin_error ("Not a valid variable name: %s\n", list_optarg);
	      last_result = EX_USAGE;
	      break;
	    }

	  varname = list_optarg;
	  break;

	default:

	  builtin_usage ();
	  last_result = EX_USAGE;
	}
    }

  if (loptend)
    {
      cmdname = loptend->word->word;
      subst_colon (cmdname);
      options = copy_word_list (loptend->next);
    }
  else
    {
      builtin_error ("missing command name");
      last_result = EX_USAGE;
    }

  if (!last_result && complete_args)
    {
      if (command)
	{
	  builtin_error ("complete options only when adding an option.");
	  last_result = EX_USAGE;
	}
      else
	{
	  WORD_DESC tmpkey = { "__xpctmp", 0 };
	  WORD_LIST tmpkeyword = { NULL, &tmpkey };

	  WORD_LIST *l;

	  for (l = complete_args; l->next; l = l->next);
	  l->next = &tmpkeyword;

	  if (complete_builtin (complete_args))
	    last_result = EX_USAGE;
	  else
	    {
	      cs = progcomp_search (tmpkey.word);
	      if (!cs)
		last_result = EXECUTION_FAILURE;
	      else if ((cs->globpat && !*cs->globpat)
		       || (cs->words && !*cs->words)
		       || (cs->prefix && !*cs->prefix)
		       || (cs->suffix && !*cs->suffix)
		       || (cs->funcname && !*cs->funcname)
		       || (cs->command && !*cs->command)
		       || (cs->filterpat && !*cs->filterpat))
		{
		  builtin_error ("Empty arg in compspec");
		  last_result = EXECUTION_FAILURE;
		}
	    }
	}
      command = XPC_ADD;
    }

  if (!last_result && !command)
    {
      if (options)
	command = XPC_ADD;
      else
	command = have_mode ? XPC_SETMODE : XPC_SHOW;
    }

  if (!last_result)
    {
      XPC_CMD *c = NULL;
      XPC_OPT *o = NULL;
      XPC_OPT_MODE mode;

      xpc_opt_setmode (&mode, "clear");

      switch (command)
	{
	case XPC_ADD:

	  last_result = EXECUTION_FAILURE;	/* the default */

	  if (cache != &heap)
	    xpc_cache_set (&heap);
	  assert (cache->idx);

	  c = xpc_cmd_find (cmdname);
	  if (!c)
	    c = xpc_cmd_add (cmdname);

	  if (!c)
	    break;


	  if (xpc_opt_setmode (&mode, modestr))
	    {
	      last_result = EX_USAGE;
	      break;
	    }

	  if (description)
	    mode.description_only = 1;

	  /* by default the to be completed option
	     argument mode is 'arg_required' */
	  if (mode.arg == 0)
	    mode.arg = (cs
			|| mode.description_only) ? arg_required : arg_none;

	  if (subsep != -1)
	    mode.suboptions = 1;

	  if (varname && *varname)
	    mode.varname = 1;

	  if (added_info)
	    mode.added_info = 1;

	  last_result = EXECUTION_SUCCESS;
	  for (args = options; args && !last_result; args = args->next)
	    {
	      XPC_OPT *newopt;

	      last_result = EXECUTION_FAILURE;	/* default */

	      optname = get_internal_optname (args->word->word);
	      if (!optname)
		break;

	      newopt = xpc_cmd_addopt (c, optname);
	      free (optname);
	      if (!newopt)
		break;

	      if (newopt->mode.description_only)
		xpc_str_del (newopt->compspec);
	      else
		xpc_compspec_del (newopt->compspec);

	      if (mode.description_only)
		newopt->compspec = xpc_str_add (description);
	      else
		newopt->compspec = xpc_compspec_add (cs);

	      if (newopt->test)
		{
		  xpc_str_del (newopt->test);
		  newopt->test = 0;
		}
	      if (test && *test)
		{
		  newopt->test = xpc_str_add (test);
		  if (!newopt->test)
		    break;
		}

	      if (newopt->varname)
		{
		  xpc_str_del (newopt->varname);
		  newopt->varname = 0;
		}
	      if (varname && *varname)
		{
		  newopt->varname = xpc_str_add (varname);
		  if (!newopt->varname)
		    break;
		}

	      newopt->subsep = subsep;

	      newopt->mode = mode;

	      last_result = EXECUTION_SUCCESS;
	    }
	  break;

	case XPC_SETMODE:
	  if (getopt_count > 1)
	    {
	      builtin_usage ();
	      last_result = EX_USAGE;
	      break;
	    }

	  if (!cmdname)
	    {
	      builtin_error ("Missing command name.");
	      last_result = EX_USAGE;
	      break;
	    }

	  xpc_cache_set (&heap);
	  c = xpc_cmd_find (cmdname);

	  if (!c)
	    {
	      builtin_error ("Command not found: %s", cmdname);
	      last_result = EXECUTION_FAILURE;
	    }
	  else
	    last_result = xpc_cmd_setmode (c, modestr);

	  break;

	case XPC_REMOVE:

	  if (getopt_count > 1)
	    {
	      builtin_usage ();
	      last_result = EX_USAGE;
	      break;
	    }

	  xpc_cache_set (&heap);
	  c = xpc_cmd_find (cmdname);
	  if (!c)
	    {
	      if (xpc_cmd_find_all (cmdname))
		{
		  builtin_error ("Can not remove from loaded cache file.");
		  last_result = EXECUTION_FAILURE;
		}
	      break;
	    }

	  if (!options)
	    {
	      xpc_cmd_del (c);
	      last_result = EXECUTION_SUCCESS;
	    }
	  else
	    for (args = options; !last_result && args; args = args->next)
	      {
		char *optname = get_internal_optname (args->word->word);

		if (!optname)
		  {
		    last_result = EXECUTION_FAILURE;
		    break;
		  }

		o = xpc_opt_find (c, optname);
		if (o)
		  xpc_cmd_delopt (c, o);

		last_result = EXECUTION_SUCCESS;

		free (optname);

		if (last_result)
		  break;
	      }

	  break;

	case XPC_SHOW:

	  if (getopt_count > 1)
	    {
	      builtin_usage ();
	      last_result = EX_USAGE;
	      break;
	    }

	  printopts (cmdname, NULL);

	  last_result = EXECUTION_SUCCESS;
	  break;

	case XPC_UNKNOWN:

	  builtin_usage ();
	  last_result = EX_USAGE;
	  break;

	default:

	  assert (!"Command not imlemented.");
	  last_result = EXECUTION_SUCCESS;
	}
    }

  discard_unwind_frame ("optcomplete");

  if (last_result && errno)
    perror (NULL);

  return last_result;
}



const struct builtin optcomplete_struct = {
  "optcomplete",
  optcomplete_builtin,
  BUILTIN_ENABLED,
  optcomplete_doc,
  "optcomplete <complete options> [-V varname] [-D description] [-I] [-T test] [-O char] [-m mode]  cmdname optname ...\n"
    "optcomplete [-p|-r]  cmdname [optname] ...\n"
    "optcomplete -m mode  cmdname\n",
  (char *) 0
};
