/* XLanguage - the eXtensible Language
 * Copyright (C) 2001 Patrick Deschenes
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* These files are distributed at http://www.freesoftware.fsf.org/xlang/
 */

#include <xlexpr.h>
#include <xlmain.h>

XLExpr*
xl_expr_new ()
{
  XLExpr *self;

  self = x_new (XLExpr);
  x_object_init_object (X_OBJECT (self), xl_expr_destroy);
  
  self->expr_type = XL_EXPR_TYPE_NONE;
  self->data = NULL;
  self->fnct = NULL;
  self->type = NULL;
  self->identifier = NULL;
  self->source_file = NULL;
  self->source_line = 0;
  self->check = TRUE;
  
  self->list_expr = x_list_new ();
 
  return self;
}

XLExpr*
xl_expr_new_fnct (xstr l_fnct)
{
  XLExpr*	self;
  XString*	l_name;
  
  self = xl_expr_new ();
  l_name = x_string_new ();
  x_string_set_str (l_name, l_fnct);
  xl_expr_set_fnct (self, l_name);
  x_unref (l_name);
  
  return self;
}

XLExpr*
xl_expr_new_identifier (xstr p_identifier)
{
  XLExpr*	self;
  XString*	l_name;
  
  self = xl_expr_new ();
  l_name = x_string_new ();
  x_string_set_str (l_name, p_identifier);
  xl_expr_set_identifier (self, l_name);
  x_unref (l_name);
  
  return self;
}

xvoid
xl_expr_destroy (XObject* self)
{
  x_unref (XL_EXPR (self)->data);
  x_unref (XL_EXPR (self)->fnct);
  x_unref (XL_EXPR (self)->type);
  x_unref (XL_EXPR (self)->source_file);
  x_unref (XL_EXPR (self)->identifier);
  x_unref (XL_EXPR (self)->list_expr);

  x_destroy (self);
}

xvoid
xl_expr_add (XLExpr* self, XLExpr* p_expr)
{
  x_list_add (self->list_expr, X_OBJECT (p_expr));
}

xvoid
xl_expr_evaluate (XLExpr* self)
{
  XString* l_name;
  XLExpr*  l_subexpr;
  XLFnct*  l_fnct;
  XLData*  l_data;
  XLVar*   l_var;
  xuint    i, l_size;

  if (!self) return;
  if (g_xl_fnct_halt) return;

  if (g_xl_main_global->debugger)
    xl_debugger_break (self);

  switch (self->expr_type)
    {
    case XL_EXPR_TYPE_NONE:
      break;
    case XL_EXPR_TYPE_BLOCK:
      l_size = x_list_get_size (self->list_expr);
      for (i = 0; i < l_size; i++)
	{
	  l_subexpr = XL_EXPR (x_list_get (self->list_expr, i));
	  xl_expr_evaluate (l_subexpr);
	  x_unref (l_subexpr);
	}
      break;
    case XL_EXPR_TYPE_DATA:
       break;
    case XL_EXPR_TYPE_IDENTIFIER:
      l_name = x_string_new ();
      x_string_set_str (l_name, "self");

      /* If 'self' is defined, we look for an 'attribute' before
       * looking for a variable.
       */

      if (x_string_cmp_str (self->identifier, "self") != 0)
	{
	  l_var = xl_main_get_var (g_xl_main_global, l_name);
	  if (l_var)
	    {
	      l_data = xl_data_get_field (l_var->data, self->identifier, FALSE);
	      x_unref (l_var);
	      if (l_data)
		{
		  x_unref (l_name);
		  x_unref (self->data);
		  self->data = l_data;
		  return;
		}
	    }
	}
      x_unref (l_name);

      l_var = xl_main_get_var (g_xl_main_global, self->identifier);
      if (l_var)
	{
	  x_unref (self->data);
	  self->data = x_addref (XLData, l_var->data);
	  x_unref (l_var);
	}
      else
	{
	  x_unref (self->data);
	  self->data = NULL;

	  l_fnct = xl_main_get_fnct (g_xl_main_global, self->identifier);
	  if (l_fnct)
	    {
	      XLType* l_type;
	      XLData* l_data;
	      
	      l_type = xl_type_new (XL_TYPE_TYPE_POINTER, XL_TYPE_ARRAY_TYPE_NONE, 0, NULL);
	      l_data = xl_data_new (l_type, NULL);
	      x_access (xl_data_get_raw (l_data), 0, xptr) = (xptr) &l_fnct->stub[0];
	      x_unref (l_type);
	      
	      self->data = l_data;
	    }
	  else
	    {
	      if (self->check)
		{
		  printf ("ERROR: variable '%s' not founded\n", x_string_get_str (self->identifier));
		  exit (0);
		}
	    }
	}
      break;
    case XL_EXPR_TYPE_FNCT:

      if (!(self->fnct))
	self->fnct = xl_main_get_fnct (g_xl_main_global, self->identifier);

      if (self->fnct)
	{
	  x_unref (self->data);
	  self->data = NULL;

	  l_data = xl_fnct_call (self->fnct, self);
	  if (l_data)
	    self->data = x_addref (XLData, l_data);
	  else
	    x_unref (self->data);

	  x_unref (l_data);
	}
      else
	{
	  printf ("ERROR: function '%s' not founded\n", x_string_get_str (self->identifier));
	  exit (-1);
	}

      break;
    default:
      break;
    }
}

xbool
xl_expr_check (XLExpr* self, XLExprType p_type)
{
  if (self->expr_type == p_type)
    return TRUE;
  else
    return FALSE;
}

XLExpr*
xl_expr_get (XLExpr* self, xuint p_index)
{
  return XL_EXPR (x_list_get (self->list_expr, p_index));
}

xuint
xl_expr_get_size (XLExpr* self)
{
  return x_list_get_size (self->list_expr);
}

xvoid
xl_expr_set_block (XLExpr* self)
{
  self->expr_type = XL_EXPR_TYPE_BLOCK;
}

xvoid
xl_expr_set_data (XLExpr* self, XLData* p_data)
{
  self->expr_type = XL_EXPR_TYPE_DATA;

  x_unref (self->data);
  self->data = x_addref (XLData, p_data);
}

xvoid
xl_expr_set_fnct (XLExpr* self, XString* p_name)
{
  self->expr_type = XL_EXPR_TYPE_FNCT;

  x_unref (self->identifier);
  self->identifier = x_addref (XString, p_name);
}

xvoid
xl_expr_set_type (XLExpr* self, XLType* p_type)
{
  self->expr_type = XL_EXPR_TYPE_TYPE;

  x_unref (self->type);
  self->type = x_addref (XLType, p_type);
}

xvoid
xl_expr_set_identifier (XLExpr* self, XString* p_identifier)
{
  self->expr_type = XL_EXPR_TYPE_IDENTIFIER;

  x_unref (self->identifier);
  self->identifier = x_addref (XString, p_identifier);
}

xuint nb_tab = 0;

xvoid
xl_expr_debug (XLExpr* self)
{
  xuint i;
  xuint l_size;

  if (!self)
    return;

  if (self->expr_type == XL_EXPR_TYPE_FNCT)
    {
      for (i=0; i<nb_tab; i++)
	printf (" ");
      
      printf ("FNCT '%s'\n", x_string_get_str (self->identifier));
      nb_tab++;

      l_size = x_list_get_size (self->list_expr);      
      for (i=0; i<l_size; i++)
	xl_expr_debug (xl_expr_get (self, i));

      nb_tab--;
    }

  if (self->expr_type == XL_EXPR_TYPE_BLOCK)
    {
      for (i=0; i<nb_tab; i++)
	printf (" ");
      
      printf ("BLOCK\n");
      nb_tab++;

      l_size = x_list_get_size (self->list_expr);      
      for (i=0; i<l_size; i++)
	xl_expr_debug (xl_expr_get (self, i));

      nb_tab--;
    }
}


