/*
    Copyright (C) 2012 Oleksiy Chernyavskyy

    This file is part of XDClient.

    XDClient is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    XDClient is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XDClient.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include "utf8.h"


#define MBS_BUF_SIZE (1024*100)
char global_temp_tombs_buffer[MBS_BUF_SIZE];
char global_temp_pfmbs_buffer[MBS_BUF_SIZE];
#define WCS_BUF_SIZE (1024*100*4)
wchar_t global_temp_towcs_buffer[WCS_BUF_SIZE];
wchar_t global_temp_pfwcs_buffer[WCS_BUF_SIZE];

int wc_mb_len(wchar_t wc)
{
  if (wc <= 0x7f)
	return 1;
  else if (wc <= 0x7ff)
	return 2;
  else if (wc <= 0xffff)
	return 3;
  else if (wc <= 0x1fffff)
	return 4;
  else if (wc <= 0x3ffffff)
	return 5;
  else if (wc <= 0x7fffffff)
	return 6;
}

size_t wcslen_fast(const wchar_t *s)
{
  unsigned long j;

  if (!s)
	return 0;

  for (j=0; s[j] != L'\0'; j++);
  return j;
}

int wcsncmp_fast(const wchar_t *s1, const wchar_t *s2, size_t n)
{
  unsigned long j;

  if (!s1) {
	if (!s2) {
	  return 0;
	} else {
	  if (wcslen_fast(s2))
		return -1;
	  else
		return 0;
	}
  } else if (!s2) {
	if (wcslen_fast(s1))
	  return 1;
	else
	  return 0;
  }

  if (n == 0)
	return 0;

  for (j=0; j<n && s1[j] == s2[j] && s1[j] != L'\0' && s2[j] != L'\0'; j++);
  if (j == n) {
	return 0;
  } else if (s2[j] == L'\0') {
	if (j>0) {
	  return 0;
	} else if (s1[j] == L'\0') {
	  return 0;
	} else {
	  return 1;
	}
  } else if (s1[j] == L'\0') {
	return -1;
  } else if (s1[j] > s2[j]) {
	return 1;
  } else {
	return -1;
  }
}

wchar_t* wcsstr_fast(wchar_t *s1, const wchar_t *s2)
{
  unsigned long j;
  size_t l2;

  l2 = wcslen_fast(s2);

  for(j=0; s1[j] != L'\0'; j++) {
	if (wcsncmp_fast(&s1[j], s2, l2) == 0)
	  return &s1[j];
  }
  return NULL;
}

wchar_t* mbs2wcs(char *mbs)
{
  wchar_t *wcs;
  int mb_len;

  if (!mbs) {
	fprintf(stderr, "xdclient: %s: error: NULL string received\n", __FUNCTION__);
	return NULL;
  }

  mb_len = mbs_nchars(mbs);
  if (mb_len == 0)
	return NULL;

  wcs = (wchar_t*) malloc(sizeof(wchar_t) * (mb_len+1));
  if (mbstowcs(wcs, (const char*) mbs, mb_len) != -1) {
	wcs[mb_len] = L'\0';
  } else {
	fprintf(stderr, "xdclient: %s: error: convert mulibyte string to wide string\n", __FUNCTION__);
	free(wcs);
	return NULL;
  }
  return wcs;
}

char *wcs2mbs(wchar_t *wcs)
{
  int mb_len;
  char *mbs;

  if (!wcs) {
	fprintf(stderr, "xdclient: %s: error: NULL string pointer\n", __FUNCTION__);
	return NULL;
  }

  mb_len = 4 * (wcslen(wcs)+1);
  mbs = (char*) malloc(sizeof(char) * mb_len);

  if (wcstombs(mbs, wcs, mb_len) == -1) {
	fprintf(stderr, "xdc: %s: error: wcstombs\n", __FUNCTION__);
	free(mbs);
	return NULL;
  }
  return mbs;
}

size_t mbs_nchars(const char *s)
{
  size_t charlen, chars;
  mbstate_t mbs;

  if (!s)
	return 0;

  chars = 0;
  memset(&mbs, 0, sizeof(mbs));
  while ((charlen = mbrlen(s, MB_CUR_MAX, &mbs)) != 0 &&
	  charlen != (size_t)-1 && charlen != (size_t)-2) {
	s += charlen;
	chars++;
  }

  return (chars);
}

char* mbsdup(char *mbs)
{
  char *copy;
  wchar_t *wcs;

  if (!mbs)
	return NULL;

  wcs = mbs2wcs(mbs);
  copy = wcs2mbs(wcs);
  free(wcs);
  return copy;
}

char *tombs(wchar_t *wcs)
{
  char *mbs;
  char nc;

  nc = wctob(L'\0');

  mbs = global_temp_tombs_buffer;

  if (!wcs) {
	mbs[0] = nc;
	return mbs;
  }

  if (wcstombs(mbs, wcs, MBS_BUF_SIZE) == -1)
	mbs[0] = nc;
  mbs[MBS_BUF_SIZE-1] = nc;

  return mbs;
}

wchar_t* towcs(char *mbs)
{
  wchar_t *wcs;
  int mb_len;

  wcs = global_temp_towcs_buffer;

  mb_len = mbs_nchars(mbs);
  if (!mbs || mb_len == 0) {
	wcs[0] = L'\0';
	return wcs;
  }

  if (mbstowcs(wcs, (const char*) mbs, WCS_BUF_SIZE) == -1)
	wcs[0] = L'\0';
  wcs[WCS_BUF_SIZE-1] = L'\0';

  return wcs;
}

char* pfmbs(char *format, ...)
{
	char *mbs;
	va_list ap;
	int ret;

	va_start(ap, format);
	mbs = global_temp_pfmbs_buffer;

	if (format)
	  ret = vsnprintf(mbs, MBS_BUF_SIZE, format, ap);
	else
	  ret = vsnprintf(mbs, MBS_BUF_SIZE, "%s", ap);

	if (ret >= MBS_BUF_SIZE)
	  mbs[0] = '\0';
	else
	  mbs[MBS_BUF_SIZE-1] = '\0';

	va_end(ap);
	return mbs;
}

wchar_t* pfwcs(wchar_t *format, ...)
{
	wchar_t *wcs;
	va_list ap;
	int ret;

	va_start(ap, format);
	wcs = global_temp_pfwcs_buffer;

	if (format)
	  ret = vswprintf(wcs, WCS_BUF_SIZE, format, ap);
	else
	  ret = vswprintf(wcs, WCS_BUF_SIZE, L"%S", ap);

	if (ret >= WCS_BUF_SIZE)
	  wcs[0] = L'\0';
	else
	  wcs[WCS_BUF_SIZE-1] = L'\0';

	va_end(ap);
	return wcs;
}

wchar_t* wcsndup(wchar_t *wcs, int len)
{
  wchar_t *wstr;
  int _len;

  if (!wcs)
	return NULL;

  _len = wcslen(wcs);
  if (_len < len)
	len = _len;

  wstr = (wchar_t*) malloc(sizeof(wchar_t) * (len + 1));
  wcsncpy(wstr, wcs, len);
  wstr[len] = L'\0';

  return wstr;
}

mbs_ll_t* mbs_split(char *mbs, char sc)
{
  char *p, *p1, *p2;
  mbs_ll_t *mbsl_start, *mbsl, *mbsl_new;
  int plen;


  if (!mbs || !mbs[0] || !sc)
	return NULL;

  p1 = mbs;
  p2 = strchr(p1, sc);
  mbsl_start = mbsl = NULL;
  while (p1 && p1[0]) {
	if (!p2)
	  plen = strlen(p1);
	else
	  plen = p2-p1;

	if (plen > 0) {
	  for(; isblank(p1[plen-1]) && plen>0; plen--);
	  p = strndup(p1, plen);
	  if (!p)
		goto MS_INEXT;
	  mbsl_new = mbsl_create(strdup(p));
	  free(p);
	  if (mbsl)
		mbsl->next = mbsl_new;;
	  mbsl = mbsl_new;
	  if (!mbsl_start)
		mbsl_start = mbsl;
	}

MS_INEXT:
	if (p2) {
	  p1 = p2+1;
	  p2 = strchr(p1, sc);
	  while(isblank(p1[0]))
		p1++;
	} else {
	  p1 = NULL;
	}
  }

  return mbsl_start;
}

wcs_ll_t* wcs_split(wchar_t *wcs, wchar_t sc)
{
  wchar_t *p, *p1, *p2;
  wcs_ll_t *wcsl_start, *wcsl, *wcsl_new;
  int plen;


  if (!wcs || !wcs[0] || !sc)
	return NULL;

  p1 = wcs;
  p2 = wcschr(p1, sc);
  wcsl_start = wcsl = NULL;
  while (p1 && p1[0]) {
	if (!p2)
	  plen = wcslen(p1);
	else
	  plen = p2-p1;

	if (plen > 0) {
	  for(; iswblank(p1[plen-1]) && plen>0; plen--);
	  p = wcsndup(p1, plen);
	  if (!p)
		goto WS_INEXT;
	  wcsl_new = (wcs_ll_t*) malloc(sizeof(wcs_ll_t));
	  wcsl_new->wcs = wcsdup(p);
	  wcsl_new->next = NULL;
	  free(p);
	  if (wcsl)
		wcsl->next = wcsl_new;;
	  wcsl = wcsl_new;
	  if (!wcsl_start)
		wcsl_start = wcsl;
	}

WS_INEXT:
	if (p2) {
	  p1 = p2+1;
	  p2 = wcschr(p1, sc);
	  while(iswblank(p1[0]))
		p1++;
	} else {
	  p1 = NULL;
	}
  }

  return wcsl_start;
}

void mbsl_free(mbs_ll_t *mbsl)
{
  mbs_ll_t *mbsl_next;

  while(mbsl) {
	if (mbsl->mbs)
	  free(mbsl->mbs);
	mbsl_next = mbsl->next;
	free(mbsl);
	mbsl = mbsl_next;
  }
}

mbs_ll_t* mbsl_create(char *mbs)
{
  mbs_ll_t *mbsl;

  mbsl = (mbs_ll_t*) malloc(sizeof(mbs_ll_t));
  mbsl->mbs = mbs;
  mbsl->next = NULL;
  return mbsl;
}

void wcsl_free(wcs_ll_t *wcsl)
{
  wcs_ll_t *wcsl_next;

  while(wcsl) {
	if (wcsl->wcs)
	  free(wcsl->wcs);
	wcsl_next = wcsl->next;
	free(wcsl);
	wcsl = wcsl_next;
  }
}

void ppad(int shift)
{
  int i;
  for (i=0; i<shift; i++)
	fprintf(stdout, " ");
}

void print_mbsll(mbs_ll_t *mbsl, int shift)
{
  int i;

  if (!mbsl)
	return;

  i=0;
  while(mbsl) {
	ppad(shift);
	fprintf(stdout, "%d: %s\n", i, mbsl->mbs);
	i++;
	mbsl = mbsl->next;
  }
}

void print_wcsll(wcs_ll_t *wcsl, int shift)
{
  int i;

  if (!wcsl)
	return;

  i=0;
  while(wcsl) {
	ppad(shift);
	fwprintf(stdout, L"%d: %S\n", i, wcsl->wcs);
	i++;
	wcsl = wcsl->next;
  }
}

char* mbs_pref_crop(char *mbs, char cc)
{
  int i, h;
  
  if (!mbs || !cc)
	return;

  for (h=0; mbs[h] == cc; h++);
  for (i=0; mbs[h] != '\0'; i++, h++)
	mbs[i] = mbs[h];
  mbs[i] = '\0';

  return mbs;
}

wchar_t* wcs_pref_crop(wchar_t *wcs, wchar_t cc)
{
  int i, h;
  
  if (!wcs || !cc)
	return;

  for (h=0; wcs[h] == cc; h++);
  for (i=0; wcs[h] != L'\0'; i++, h++)
	wcs[i] = wcs[h];
  wcs[i] = L'\0';

  return wcs;
}

char* mbs_sufx_crop(char *mbs, char cc)
{
  int h;

  if (!mbs || !cc)
	return;

  for (h = strlen(mbs)-1; h >= 0 && mbs[h] == cc; h--);
  mbs[h+1] = '\0';

  return mbs;
}

wchar_t* wcs_sufx_crop(wchar_t *wcs, wchar_t cc)
{
  int h;

  if (!wcs || !cc)
	return;

  for (h = wcslen(wcs)-1; h >= 0 && wcs[h] == cc; h--);
  wcs[h+1] = L'\0';

  return wcs;
}

