/*
    Copyright (C) 2012 Oleksiy Chernyavskyy

    This file is part of XDClient.

    XDClient is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    XDClient is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XDClient.  If not, see <http://www.gnu.org/licenses/>.
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef TARGET_FREEBSD
#include <sys/types.h>
#else
#include <stdint.h>
#include <wctype.h>
#endif
#include <dirent.h>
#include <sys/stat.h>
#include <unistd.h>
#include "file.h"
#include "utf8.h"
#include "common.h"
#include "regexp_utf8.h"
#include "xdc.h"

int check_cfg(char *cfg_dir)
{
  if (is_dir(cfg_dir) && can_rw(cfg_dir))
	return 1;

  return 0;
}

int can_rw(char *path)
{
  struct stat statbuf;

  if (stat(path, &statbuf) == -1)
	return 0;

  if (statbuf.st_mode & S_IRUSR && \
	  statbuf.st_mode & S_IWUSR && \
	  statbuf.st_mode & S_IXUSR) {
	if (statbuf.st_uid == getuid())
	  return 1;
  }

  if (statbuf.st_mode & S_IRGRP && \
	  statbuf.st_mode & S_IWGRP && \
	  statbuf.st_mode & S_IXGRP) {
	if (statbuf.st_gid == getgid())
	  return 1;
  }

  if (statbuf.st_mode & S_IROTH && \
	  statbuf.st_mode & S_IWOTH && \
	  statbuf.st_mode & S_IXOTH) {
	  return 1;
  }

}

long file_size(char* path)
{
  struct stat statbuf;

  if (stat(path, &statbuf) == -1)
	return -1;

  return statbuf.st_size;
}

long file_mtime(char *path)
{
  struct stat statbuf;

  if (stat(path, &statbuf) == -1)
	return -1;

  return statbuf.st_mtime;
}

long file_ctime(char *path)
{
  struct stat statbuf;

  if (stat(path, &statbuf) == -1)
	return -1;

  return statbuf.st_ctime;
}

int file_exist(char *path)
{
  struct stat statbuf;

  if (stat(path, &statbuf) == -1)
	return 0;
  return 1;
}

int is_rfile(char *path)
{
  struct stat statbuf;

  if (stat(path, &statbuf) == -1)
	return 0;

  if (S_ISREG(statbuf.st_mode))
	return 1;

  return 0;
}

int is_dir(char *path)
{
  struct stat statbuf;

  if (stat(path, &statbuf) == -1)
	return 0;

  if (S_ISDIR(statbuf.st_mode))
	return 1;

  return 0;
}


wchar_t* abs_path(wchar_t *path_in)
{
  wchar_t *path;
  wchar_t *path_ret;

  if (!path_in || !path_in[0])
	return NULL;

  path = wcsdup(path_in);
  wcs_pref_crop(path, ' ');
  wcs_sufx_crop(path, ' ');
  wcs_pref_crop(path, '\t');
  wcs_sufx_crop(path, '\t');

  path_ret = _abs_path(path);
  free(path);

  return path_ret;
}

wchar_t* _abs_path(wchar_t *path)
{
  wchar_t *buf;
  int buf_len;
  int i, j;
  wchar_t wc, wc2, wc3, wc4;
  wchar_t *w_sdir = NULL;
  char *path_ret = NULL;
  int len;
  wchar_t *buf_ret;

  if (!path || path[0] == L'\0') {
	fprintf(stderr, "xdclient: %s: error: empty path string\n", __FUNCTION__);
	return NULL;
  }

  len = wcslen(path);
  buf_len = MAXPATH > len*2 ? MAXPATH : len*2;
  buf = (wchar_t*) malloc(sizeof(wchar_t) * buf_len);

  i = 0;
  if (path[0] != L'/') {
	if (path[0] == L'~')
	  w_sdir = mbs2wcs(getenv("HOME"));
	else
	  w_sdir = mbs2wcs(getenv("PWD"));

	if (!w_sdir) {
	  fprintf(stderr, "xdclient: %s: couldn't get PWD env\n", __FUNCTION__);
	  free(buf);
	  return NULL;
	}
	len = wcslen(w_sdir);
	if (len >= buf_len/2) {
	  free(buf);
	  buf_len = len*2;
	  buf = (wchar_t*) malloc(sizeof(wchar_t) * buf_len);
	}
	wcsncpy(buf, w_sdir, buf_len);
	free(w_sdir);

	i = len-1;
	while (i>=0 && buf[i] == L'\0')
	  i--;
	while (i>=0 && buf[i] == L'/')
	  i--;
  }

  j = 0;
  wc = path[j];
  wc2 = wc3 = wc4 = path[j+1];
  if (wc3 != L'\0')
	wc3 = wc4 = path[j+2];
  if (wc4 != L'\0')
	wc4 = path[j+3];
	  
  while(wc != L'\0' && i < buf_len-3) {
	if (wc == L'~' && j == 0) {
	  j++;
	} else if (wc == L'/' && wc2 == L'/') {
	  j++;
	} else if (wc == L'/' && wc2 == L'.' && wc3 == L'/') {
	  j+= 2;
	} else if (wc == L'/' && wc2 == L'.' && wc3 == L'.' && (wc4 == L'/' || wc4 == L'\0')) {
	  while (i>=0 && buf[i] == L'/')
		i--;
	  while (i>=0 && buf[i] != L'/')
		i--;
	  while (i>=0 && buf[i] == L'/')
		i--;
	  j+=3;
	} else if (wc == L'.' && wc2 == L'.' && (wc3 == L'/' || wc3 == L'\0')) {
	  while (i>=0 && buf[i] == L'/')
		i--;
	  while (i>=0 && buf[i] != L'/')
		i--;
	  while (i>=0 && buf[i] == L'/')
		i--;
	  j+=2;
	} else if (wc == L'.' && wc2 == L'/') {
	  j++;
	} else {
	  while (i>=0 && buf[i] == L'/')
		i--;
	  if (i<0)
		i = 0;
	  else if (i > 0)
		i++;
	  buf[i] = L'/';

	  if (wc != L'/')
		buf[++i] = wc;
	  wc = path[++j];
	  while (wc != L'/' && wc != L'\0' && i < buf_len-1) {
		buf[++i] = wc;
		wc = path[++j];
	  }
	}
	wc = wc2 = wc3 = wc4 = path[j];
	if (wc2 != '\0')
	  wc2 = wc3 = wc4 = path[j+1];
	if (wc3 != L'\0')
	  wc3 = wc4 = path[j+2];
	if (wc4 != L'\0')
	  wc4 = path[j+3];
  }

  if (wc != L'\0') {
	free(buf);
	fprintf(stderr, "xdclient: %s: error: buffer overflow\n", __FUNCTION__);
	return NULL;
  }

  if (i<0) {
	i=0;
	buf[i] = L'/';
  }
  buf[++i] = L'\0';

  buf_ret = wcsdup(buf);
  free(buf);
  return buf_ret;
}

char* abs_path_8(wchar_t *path)
{
  wchar_t *wcs_path = NULL;
  char *path_ret;

  if (!path || path[0] == L'\0') {
	fprintf(stderr, "xdclient: %s: error: empty path string\n", __FUNCTION__);
	return NULL;
  }

  wcs_path = abs_path(path);
  path_ret = wcs2mbs(wcs_path);
  if (wcs_path)
	free(wcs_path);
  return path_ret;
}

char* get_prog_dir(char *argv0)
{
  char *dir;
  char *mbs;
  int i;

  if (!argv0)
	return NULL;

  i = strlen(argv0) - 1;
  while(i>=0 && argv0[i] != '/')
	i--;
  if (i>=0) {
	if (! (dir = abs_path_8(towcs(argv0))))
	  return NULL;

	i = strlen(dir) - 1;
	while(i>=0 && dir[i] != '/')
	  i--;
	if (i>=0) {
	  dir[i] = '\0';
	} else {
	  free(dir);
	  return NULL;
	}
	return dir;
  } else {
	mbs = prog_locate(argv0);
	if (!mbs)
	  return NULL;
	dir = abs_path_8(towcs(mbs));
	free(mbs);

	i = strlen(dir) - 1;
	while(i>=0 && dir[i] != '/')
	  i--;
	if (i>=0) {
	  dir[i] = '\0';
	} else {
	  free(dir);
	  return NULL;
	}
	return dir;
  }
}

char* prog_locate(char *name)
{
  char *dir;
  char *p_env;
  char *path, *path_save;
  char buf[MAXPATH];

  if (! (path = path_save = mbsdup(getenv("PATH"))))
	return NULL;

  while ((dir = strsep(&path, ":")) != NULL) {
	if (*dir == '\0')
	  dir = ".";
	if (snprintf(buf, sizeof(buf), "%s/%s", dir, name) >= (int) sizeof(buf))
	  continue;
	if (can_exec(buf)) {
	  free(path_save);
	  dir = mbsdup(buf);
	  return dir;
	}
  }
  free(path_save);

  return NULL;
}

int can_exec(char *path)
{
  struct stat s_stat;

  if (access(path, X_OK) == 0 &&
	  stat(path, &s_stat) == 0 &&
	  S_ISREG(s_stat.st_mode) &&
	  (getuid() != 0 ||
	   (s_stat.st_mode & (S_IXUSR | S_IXGRP | S_IXOTH)) != 0)) {
	return 1;
  }
  return 0;
}

char* chop_path(char *path)
{
  int i;

  if (!path)
	return NULL;

  i = strlen(path) - 1;
  while(i>=0 && path[i] == '/')
	i--;
  path[i+1] = '\0';

  return path;
}

void flist_print(fstat_lt *flist, int shift)
{
  if (!flist)
	return;

  while (flist) {
	ppad(shift);
	fprintf(stdout, "path: %s\n", flist->path);
	ppad(shift);
	fprintf(stdout, "size: %ld\n", flist->size);
	ppad(shift);
	fprintf(stdout, "mtime: %ld\n", flist->mtime);
	ppad(shift);
	fprintf(stdout, "ctime: %ld\n", flist->ctime);
	ppad(shift);
	fprintf(stdout, "md5_full: %s\n", flist->md5_full ? flist->md5_full : "");
	ppad(shift);
	fprintf(stdout, "md5_part: %s\n", flist->md5_part ? flist->md5_part : "");
	fprintf(stdout, "\n");
	flist = flist->next;
  }
}

fstat_lt* fs_find(char *path_in, char *pat, unsigned flags)
{
  char *path_abs;
  fstat_lt *flist;

  if (!path_in || !pat)
	return NULL;

  path_abs = abs_path_8(towcs(path_in));
  flist = find_file(path_abs, pat, flags);
  free(path_abs);
  return flist;
}

fstat_lt* find_file(char *path_in, char *pat, unsigned flags)
{
  DIR *dirp;
  struct dirent *de;
  char path[MAXPATH];
  char _path_in[MAXPATH];
  char fname[MAXPATH];
  fstat_lt *file_st, *file_root, *file_ret;
  unsigned re_flags, stat_flags;
  int i, h;
  int len;

  re_flags = stat_flags = 0;
  file_st = NULL;

  if (!path_in || !path_in[0] || !pat)
	return NULL;

  if (flags & FF_CASE_INS)
	re_flags |= RE_IGN_CASE;

  if (flags & FF_MD5_FULL)
	stat_flags |= FSTAT_MD5_FULL;

  if (flags & FF_MD5_PART)
	stat_flags |= FSTAT_MD5_PART;

  strncpy(_path_in, path_in, MAXPATH-1);
  _path_in[MAXPATH-1] = '\0';
  mbs_sufx_crop(_path_in, '/');

  if (!_path_in[0]) {
	_path_in[0] = '/';
	_path_in[1] = '\0';
  }

  len = strlen(_path_in);
  for (i=len-1; i>=0 && _path_in[i] && _path_in[i] != '/'; i--);

  if (i >= 0 && len > 1) { 
	i++;
	for (h=0; _path_in[i]; h++, i++)
	  fname[h] = _path_in[i];
	fname[h] = '\0';

	if (reg_test_8(fname, towcs(pat), re_flags | RE_PREFIX_DASH | RE_SUFFIX_DOLLAR, NULL)) {
	  file_st = fstat_create();
	  file_st->path = strdup(_path_in);
	  fstat_update(file_st, stat_flags);
	}
  }

  if (flags & FF_NO_DIR_ENTER)
	return file_st;

  if (! is_dir(_path_in))
	return file_st;

  if (flags & FF_NO_DEEP)
	flags |= FF_NO_DIR_ENTER;

  dirp = opendir(_path_in);
  if (! dirp) {
	fprintf(stderr, "xdc: warning: could not open %s\n", _path_in);
	return file_st;
  }

  file_root = file_st;

  while((de = readdir(dirp)) != NULL) {
	if (de->d_name[0] == '.') {
#ifdef _DIRENT_HAVE_D_NAMLEN
	  if (de->d_namlen == 1)
		continue;
	  if (de->d_name[1] == '.' && de->d_namlen == 2)
		continue;
#else
	  if (strlen(de->d_name) == 1)
		continue;
	  if (de->d_name[1] == '.' && strlen(de->d_name) == 2)
		continue;
#endif
	}
#ifdef _DIRENT_HAVE_D_NAMLEN
	strncpy(fname, de->d_name, de->d_namlen);
	fname[de->d_namlen] = '\0';
#else
	strcpy(fname, de->d_name);
#endif
	snprintf(path, MAXPATH, "%s/%s", _path_in, fname);
	if (file_ret = find_file(path, pat, flags)) {
	  if (!file_root)
		file_root = file_ret;

	  if (!file_st)
		file_st = file_ret;
	  else
		file_st->next = file_ret;

	  while(file_st->next)
		file_st = file_st->next;
	}
  }

  closedir(dirp);
  return file_root;
}

int fstat_update(fstat_lt *file, unsigned flags)
{
  if (!file)
	return 0;

  if (!file_exist(file->path))
	return 0;

  file->size = file_size(file->path);
  file->mtime = file_mtime(file->path);
  file->ctime = file_ctime(file->path);

  if (file->md5_full)
	free(file->md5_full);
  if (flags & FSTAT_MD5_FULL)
	file->md5_full = gen_md5sum_full(file->path);
  else
	file->md5_full = NULL;

  if (file->md5_part)
	free(file->md5_part);
  if (flags & FSTAT_MD5_PART)
	file->md5_part = gen_md5sum_part2(file->path, MD5_PART_SIZE, MD5_NPARTS);
  else
	file->md5_part = NULL;

  return 1;
}

fstat_lt* fstat_create(void)
{
  fstat_lt *file;

  file = (fstat_lt*) malloc(sizeof(fstat_lt));
  file->path = NULL;
  file->size = 0;
  file->mtime = 0;
  file->ctime = 0;
  file->md5_full = NULL;
  file->md5_part = NULL;
  file->udata = NULL;
  file->udata_type = 0;
  file->udata_free = NULL;
  file->next = NULL;

  return file;
}

void flist_free(fstat_lt *file, unsigned flags)
{
  if (!file)
	return;

  if (file->path)
	free(file->path);
  if (file->md5_full)
	free(file->md5_full);
  if (file->md5_part)
	free(file->md5_part);

  if (file->udata && file->udata_free)
	(*file->udata_free)(file->udata);
  else if (file->udata && flags & FR_FORCE)
	free(file->udata);

  flist_free(file->next, flags);
  free(file);
}

void flist_dup_clean(xdc_conf_t *xdc_conf, fstat_lt *flist_start)
{
  fstat_lt *flist, *flist_next, *flist_prev;
  nameval_t *nv;

  flist = flist_start;
  flist_next = flist_prev = NULL;
  while (flist) {
	nv = hlookup(xdc_conf->symtab, pfmbs("flist_dup_clean:%s", flist->path), 1);
	if (! nv->ival) {
	  nv->ival = 1;
	  flist_prev = flist;
	  flist = flist->next;
	} else {
	  flist_next = flist->next;
	  flist->next = NULL;
	  flist_free(flist, 0);
	  flist = flist_next;
	  if (flist_prev)
		flist_prev->next = flist;
	}
  }
  htable_regex_free(xdc_conf->symtab, "flist_dup_clean:.*", 0);
}

char* mbs_dirname(char *mbs)
{
  char *mbs_ret;
  int i;

  if (!mbs || !mbs[0])
	return NULL;

  mbs_ret = strdup(mbs);
  mbs_sufx_crop(mbs_ret, '/');
  i = strlen(mbs_ret) - 1;
  while(i >= 0 && mbs_ret[i] != '/')
	i--;
  if (i == 0)
	i++;
  mbs_ret[i] = '\0';

  return mbs_ret;
}

