/* Copyright (C) 2002 Asfand Yar Qazi.

 This file is part of XBobble.

    XBobble is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    XBobble is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XBobble; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/** @file util.hh Misc things needed by all classes. */

#ifndef XBOBBLE_UTIL_HH
#define XBOBBLE_UTIL_HH

#include <string>
#include <stdexcept>
#include <set>
#include <cmath>

namespace XBobble
{

/// Exception thrown if error encountered during initialisation (eg
/// option parsing, getting video mode, etc.)
class Init_Error : public std::runtime_error
{
public:
	Init_Error(const std::string& s)
	 : std::runtime_error(s)
	{
	}

}; // Init_Error

/// Exception thrown when a singleton object is attempted to be
/// initialised twice.
class Singleton_Object_Error : public std::logic_error
{
public:
	Singleton_Object_Error(const std::string& arg)
	 : std::logic_error(arg)
	{
	}

}; // Singleton_Object_Error

/// Exception thrown when we a paranoid about something!  ie assertion
/// exception.
class Paranoia_Confirmed_Error : public std::logic_error
{
public:
	Paranoia_Confirmed_Error(const std::string& arg)
	 : std::logic_error(arg)
	{
	}

}; // class Paranoia_Confirmed_Error

/// Derive from this FIRST to provide a singleton class.  It will
/// throw a Singleton_Object_Error before any of the members of your
/// own class have been initialised, therefore saving you lots of
/// trouble.  Nice, eh?
class Singleton_Class
{
protected:
	/// Throws an exception if this class is instansiatied twice or
	/// more (pass name of child class to identify it uniquely)
	Singleton_Class(const std::string& arg);

	/// When this is called, removes the 'my_name' from the
	/// class_names container, thereby enabling an object of this
	/// type to be created afterwards.
	virtual
	~Singleton_Class();

	/// Returns whether or not the given class exists
	static bool
	exists(const std::string& arg);

	/// Returns my_name
	const std::string&
	get_my_name() const
	{
		return my_name;
	}

private:
	/// Holds this object's name (so it can be removed later.)
	std::string my_name;

	/// Holds names of classes already instansiated.
	static std::set<std::string> class_names;

}; // class Singleton_Class



/** A simple 2D mathematical vector. */
class V2D
{
public:
	float x, y;

	V2D()
	{
	}

	V2D(float ax, float ay)
	 : x(ax), y(ay)
	{
	}

	float
	magnitude() const
	{
		return std::sqrt(std::pow(x, 2) + std::pow(y, 2));
	}

 	V2D
	operator+(const V2D& arg) const
	{
		return V2D(x + arg.x, y + arg.y);
	}

 	V2D
	operator-(const V2D& arg) const
	{
		return V2D(x - arg.x, y - arg.y);
	}

	V2D
	operator*(float arg) const
	{
		return V2D(x*arg, y*arg);
	}

	V2D
	operator/(float arg) const
	{
		return V2D(x/arg, y/arg);
	}

	const V2D&
	operator+=(const V2D& arg)
	{
		x += arg.x;
		y += arg.y;
		return *this;
	}

	const V2D&
	operator-=(const V2D& arg)
	{
		x -= arg.x;
		y -= arg.y;
		return *this;
	}

	const V2D&
	operator*=(float arg)
	{
		x *= arg;
		y *= arg;
		return *this;
	}

	const V2D&
	operator/=(float arg)
	{
		x /= arg;
		y /= arg;
		return *this;
	}

}; // class V2D


#if 0
/// A colour.  Stored in floating-point values
class Colour
{
public:
	/// Init with given values
	Colour(float ar, float ag, float ab, float aa)
	 : r(ar), g(ag), b(ab), a(aa)
	{
	}

	/// The colour values - 0=red, 1=green, 2=blue, 3=alpha
	float colour[4];

	//@{
	/// Convenience and safety
	float&
	r() {return colour[0];}

	float&
	g() {return colour[1];}

	float&
	b() {return colour[2];}

	float&
	a() {return colour[3];}
	//@}

}; // class Colour
#endif


} // namespace XBobble

#endif // #define XBOBBLE_UTIL_HH

