/* Copyright (C) 2002 Asfand Yar Qazi.

 This file is part of XBobble.

    XBobble is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    XBobble is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XBobble; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */
#include <iostream>
#include <iomanip>
#include <typeinfo>

#include "OliorGrid.hh"

using namespace XBobble;

template<typename T>
std::ostream&
operator<<(std::ostream& os, const OliorGrid<T>& g)
{
	for(typename OliorGrid<T>::size_type_single rc = 0;
	    rc != g.num_rows(); ++rc) {
		os << 'R' << static_cast<unsigned long>(rc) << '\t';
		if(g.odd_row(rc))
			os << '\t';
		for(typename OliorGrid<T>::size_type_single cc = 0;
		    cc != g.elements_in_row(rc); ++cc) {
			os << std::setw(8) << g.at(rc, cc) << '\t';
		}
		os << '\n';
	}

	return os;
}

template<typename T>
void
grid_info(const OliorGrid<T>& g)
{
	std::cout << "Grid of type " << typeid(g).name() << '\n'
		  << "Has " << g.num_rows() << " rows and "
		  << g.num_cols() << " cols. " << '\n';
}

template<typename T>
void
try_at(const OliorGrid<T>& g,
       typename OliorGrid<T>::size_type_single row,
       typename OliorGrid<T>::size_type_single col)
{
	try
	{
		g.at(row, col);
	}
	catch(std::exception& e)
	{
		std::cout << "Exception " << typeid(e).name()
			  << " occured: " << e.what() << std::endl;
	}
}

template<typename T>
void
test_indexing(const OliorGrid<T>& g)
{
	std::cout << "You shouldn't see any exceptions here....\n";
	try
	{
		for(typename OliorGrid<T>::size_type_single rc = 0;
		    rc != g.num_rows(); ++rc) {
			for(typename OliorGrid<T>::size_type_single cc = 0;
			    cc != g.elements_in_row(rc); ++cc) {
				g.at(rc, cc);
			}
		}
	}
	catch(std::exception& e)
	{
		std::cout << "BUG\n";
		throw;
	}

	std::cout << "You should see 4 exceptions here....\n";

	std::cout << "1: ";
	try_at(g, g.num_rows(), 0);

	std::cout << "2: ";
	try_at(g, 0, g.num_cols());

	std::cout << "3: ";
	try_at(g, g.num_rows()-1,
	     g.elements_in_row(g.num_rows()-1) + 1 );

	std::cout << "4: ";
	try_at(g, 0, g.num_cols());

	std::cout << "Testing iterator interface.....\n";
	typename OliorGrid<T>::const_iterator it = g.begin();
	typename OliorGrid<T>::const_iterator itend = g.end();
	unsigned i = 0;
	for(; it != itend; ++i)
		++it;
	if(i != g.size()) {
		std::cout << "ERROR! - (i == " << i << ") and (g.size() == "
			  << g.size() << ")\n";
		throw std::domain_error("test_indexing: i != g.size() !");
	}

	OliorGrid<T> g2(g);

	std::transform(g2.begin(), g2.end(),
		       g2.begin(),
		       std::bind2nd(std::multiplies<T>(), 2));

	std::cout << "All values multiplied by 2 gives: \n" << g2 << "\n";

	std::cout << "OK\n";
}

int
prog(int argc, char** argv)
{
	OliorGrid<double> grid1(20, 5, 200.112), grid2(3, 3, 50.2213);

	grid1.swap(grid2);

	std::cout << "Grid1:\n";
	grid_info(grid1);
	std::cout << '\n' << grid1 << std::endl;

	std::cout << "Grid2:\n";
	grid_info(grid2);
	std::cout << '\n' << grid2 << std::endl;

	std::cout << "Testing grid1\n";
	test_indexing(grid1);

	std::cout << "Testing grid2\n";
	test_indexing(grid2);

	return 0;

} // prog()

int
main(int argc, char** argv)
{
	int retval;

	try
	{
		retval = prog(argc, argv);
	}
	catch(std::exception& e)
	{
		std::cout << "Exception " << typeid(e).name()
			  << " occured: " << e.what() << std::endl;
		throw;
	}

	return retval;
}

