/* Copyright (C) 2002 Asfand Yar Qazi.

 This file is part of XBobble.

    XBobble is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    XBobble is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XBobble; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/** @file User_Input_Manager.hh

    @see class User_Input_Manager */

#ifndef XBOBBLE_USER_INPUT_MANAGER_HH
#define XBOBBLE_USER_INPUT_MANAGER_HH

#include <ayq/stdint.h>
#include <string>
#include <SDL.h>

#include "util.hh"

namespace XBobble
{

class System;

/// Class that provide the ability to process and store the results of
/// SDL events related to user input (joystick, key, etc).
///
/// For keys, a 'push' is when the button is pressed down and then
/// released all while the focus is still on the program window.
class User_Input_Manager : public Singleton_Class
{
public:
	/// Default constructor - inits joystick according to config file
	User_Input_Manager(System& arg_sys);

	/// Process the given structure for user input events, and
	/// store the results.  Note that an inner dead-zone and an
	/// outer dead-zone of 10% imposed, and is taken into account
	/// during conversion.
	void
	process_events(SDL_Event& ev);

	/// Tell us that we should unpress all keyboard keys, since
	/// the app has lost focus
	void
	reset_kbd_keys();

	/// Resets the joy values.  This is so that, for example
	/// during state transitions, we can stop user input
	/// 'carrying' over from the previous state (button down for
	/// example)
	void
	reset_joy();

	/// Gets the value of the axis when last processed - it will
	/// be in-between -100 and 100.
	int32_t
	get_axis() const;

	/// Return spacebar state
	bool
	get_spacebar() const;

	/// Get escape state
	bool
	get_escape() const;

	/// Get pseudo axis for left/right keys
	int32_t
	get_kbdaxis() const;

	/// Gets state of button (whether or not a button was pushed)
	bool
	get_button() const;

private:
	/// Owning system object
	System& system;

	/// The index of the selected joystick
	Uint8 my_joystick;

	/// The selected axis number for left/right movement
	Uint8 my_axis;

	/// The value of the axis as of the last read, scaled to
	/// between -100 and 100.
	int32_t axisval;

	/// The inner deadzone of the axis
	int deadzone;

	/// @internal Used for misc calculations for axis
	int32_t jmax, jmin;

	/// Was a button pushed?
	mutable bool button;

	/// Spacebar key down(1)/up(2)
	mutable int_fast8_t spacebar;

	/// Escape key (same as spacebar)
	///
	/// @see spacebar;
	int_fast8_t escape;

	/// @internal: Left/right keys.  Used to update 'kbdaxisval'.
	int_fast8_t left, right;

	/// pseudo-axis from keyboard keys left/right (-1 = left, 0 =
	/// no change, 1 = right)
	int32_t kbdaxisval;

}; // class User_Input_Manager



} // namespace XBobble


#endif // #define XBOBBLE_USER_INPUT_MANAGER_HH

