/* Copyright (C) 2002 Asfand Yar Qazi.

 This file is part of XBobble.

    XBobble is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    XBobble is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XBobble; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/** @file Out_Of_Game_State.cc @see Out_Of_Game_State.hh */

#include <ayq/stdint.h>
#include <memory>
#include <iostream>
#include <stdexcept>
#include <SDL_opengl.h>
#include <ayq/utils/maths/Trig_Lookup.hh>
#include "Out_Of_Game_State.hh"
#include "Game.hh"
#include "Game_Manager.hh"
#include "Grid.hh"
#include "Launcher.hh"
#include "Chooser.hh"
#include "Ambient_Effects_Manager.hh"
#include "Popped_Balls_Manager.hh"
#include "Video_Output_Manager.hh"
#include "User_Input_Manager.hh"
#include "Sound_Output_Manager.hh"
#include "Sound_Data.hh"
#include "glstuff.hh"

namespace XBobble
{


namespace
{

typedef AYQ::Trig_Tables<float, 256> Trig_Tables_T_;
const Trig_Tables_T_ tables_;
const Trig_Tables_T_::BiLookup trig(tables_);

} // namespace

class Out_Of_Game_State::Pimpl
{
	friend class Out_Of_Game_State;

	Pimpl()
	 : light_x(0), light_y(64)
	{
	}

	/// Texture name
	uint32_t texname;

	/// Intro display list
	uint32_t dlist;

	/// Sound data
	Sound_Data sound_data;

	/// The 'angle' measuring the x/y-displacement of the light
	int32_t light_x, light_y;

	/// Delta values are accumulated, and light_x is updated
	/// according to how many values have been collected so far
	uint32_t light_x_delta_accum, light_y_delta_accum;
};

Out_Of_Game_State::Out_Of_Game_State(Game::Impl& arg)
 : State(arg), impl(new Pimpl())
{
	try
	{
		uint32_t& texname = impl->texname;
		uint32_t& dlist = impl->dlist;
		Sound_Data& sound_data = impl->sound_data;

		Video_Output_Manager& vo
			= game.game_manager.system.video_output_manager;
		vo.register_data(this);
		texname = vo.get_texture("images/intro.png");
		dlist = glGenLists(1);
		glNewList(dlist, GL_COMPILE); {
			const int32_t tw = 360, th = 240; // pleez make these
			// multiples of 4!!!!
			glTranslatef(-tw/2, -th/2, 0);
			glBindTexture(GL_TEXTURE_2D, texname);
			glMaterialf(GL_FRONT, GL_SHININESS, 100);
			call_glMaterialfv(GL_FRONT, GL_AMBIENT,
					  0.8f, 0.8f, 0.8f, 1);
			call_glMaterialfv(GL_FRONT, GL_DIFFUSE,
					  0.8f, 0.8f, 0.8f, 1);
			call_glMaterialfv(GL_FRONT, GL_SPECULAR,
					  1.0f, 1.0f, 1.0f, 1);
			glNormal3f(0, 0, -1);
			create_tess_rect_of_4by4(tw, th);
		}; glEndList();

		Sound_Output_Manager& so
			= game.game_manager.system.sound_output_manager;
		so.load_sound("sounds/goat.voc", sound_data);
	}
	catch(...)
	{
		delete impl;
		throw;
	}
}

Out_Of_Game_State::~Out_Of_Game_State()
{
	delete impl;

}

void
Out_Of_Game_State::entry()
{
	enable_drawing(true);

	game.game_manager.system.user_input_manager.reset_joy();
	game.game_manager.system.user_input_manager.reset_kbd_keys();

	get_grid().enable_drawing(false);
	get_launcher().enable_tock_funcs(false);
	get_chooser().enable_drawing(false);
	get_ambient_effects_manager().enable_tock_funcs(false);
	get_popped_balls_manager().enable_tock_funcs(false);
	get_popped_balls_manager().reset();

	impl->sound_data.play_sound(-1);

	const GLfloat ambient0[] = {0, 0, 0, 1};
	const GLfloat diffuse0[] = {0.7, 0.7, 0.7, 1};
	const GLfloat specular0[] = {0.9, 0.9, 0.9, 1};

	glLightfv(GL_LIGHT0, GL_AMBIENT, ambient0);
	glLightfv(GL_LIGHT0, GL_DIFFUSE, diffuse0);
	glLightfv(GL_LIGHT0, GL_SPECULAR, specular0);

	glEnable(GL_BLEND);
	glEnable(GL_TEXTURE_2D);
	glEnable(GL_LIGHTING);

	glClearColor(0.3, 0.0, 0.0, 0);
}

void
Out_Of_Game_State::exit()
{
	enable_drawing(false);

	impl->sound_data.stop_sound();
}

void
Out_Of_Game_State::tock(uint32_t)
{
	// Yeah, I know these'll both wrap around sooner or later, but
	// wha' the hey...
	++impl->light_x;
	++impl->light_y;

	User_Input_Manager& ui = game.game_manager.system.user_input_manager;
	if(ui.get_button() || ui.get_spacebar()) {
		std::cout << "here\n"; std::cout.flush();
		game.set_state(Game::GAMEPLAY);
	}
}

void
Out_Of_Game_State::tick(uint32_t delta)
{
}

void
Out_Of_Game_State::sync_tick(uint32_t)
{
}

void
Out_Of_Game_State::draw_handler() const
{
	GLfloat lightpos[4] = {0, 0, -50, 1};
	lightpos[0] = trig.get_cos(impl->light_x)*100.0f;
	lightpos[1] = trig.get_cos(impl->light_y)*75.0f;
	glLightfv(GL_LIGHT0, GL_POSITION, lightpos);
	glCallList(impl->dlist);
}

} // namespace XBobble


