/* Copyright (C) 2002 Asfand Yar Qazi.

 This file is part of XBobble.

    XBobble is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    XBobble is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XBobble; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/** @file Launcher.cc @see Launcher.hh */

#include <ayq/stdint.h>
#include <ayq/utils/maths/Trig_Lookup.hh>
#include <memory>
#include <SDL_opengl.h>
#include "Launcher.hh"
#include "Game.hh"
#include "Game_Manager.hh"
#include "Grid.hh"
#include "Chooser.hh"
#include "Launched_Ball_Manager.hh"
#include "Video_Output_Manager.hh"
#include "Sound_Data.hh"
#include "Sound_Output_Manager.hh"
#include "gameplay_variables.hh"
#include "Ball.hh"
#include "glstuff.hh"
#include <iostream>
#include <iomanip>

namespace XBobble
{

class Launcher::Impl
{
	friend class Launcher;

	/// Default init
	Impl()
	 : angle(0), ball(255)
	{
	}

	/// Angle of the launcher.  Initially 0
	float angle;

	/// Display list
	uint32_t dlist;

	/// The current ball
	Ball ball;

	/// Light position (used to control where it points)
	float light_position[4];

	/// 'Click while turning' sound
	Sound_Data click_snd;

}; // class Launcher::Impl

namespace
{

typedef AYQ::Trig_Tables<float, 360<<2 > Trig_Tables_T;
Trig_Tables_T tables__;
Trig_Tables_T::Lookup trig(tables__);

} // namespace

Launcher::Launcher(Game::Impl& arg)
 : Game_Element(arg), impl(new Impl())
{
	std::auto_ptr<Impl> tmp_impl(impl);
	Video_Output_Manager& vo
		= game.game_manager.system.video_output_manager;
	vo.register_data(this);

	const float r = get_grid().get_r();
	const float h = get_grid().get_h();
	impl->ball.trait = get_chooser().get_next_ball();
	impl->ball.pos.x = 0;
	impl->ball.pos.y = h/2 + r*2.0f;

	impl->light_position[0] = impl->ball.pos.x;
	// a little behind
	impl->light_position[1] = impl->ball.pos.y + 50.0f;
	impl->light_position[2] = 0;
	impl->light_position[3] = 1;

	impl->dlist = glGenLists(1);
	glNewList(impl->dlist, GL_COMPILE); {
		call_glMaterialfv(GL_FRONT, GL_AMBIENT,
				  0.2, 0.2, 0.2, 1);
		call_glMaterialfv(GL_FRONT, GL_DIFFUSE,
				  0.7, 0.7, 0.9, 1);
		call_glMaterialfv(GL_FRONT, GL_SPECULAR,
				  0.0, 0.0, 0.0, 1);
		glMaterialf(GL_FRONT, GL_SHININESS, 0);
		glBegin(GL_TRIANGLES); {
			glNormal3f(0, 0, -1);
			glVertex3f(-10, 0, 0);
			glVertex3f(10, 0, 0);
			glVertex3f(0, -35, 0);
		}; glEnd();
	}; glEndList();

	Sound_Output_Manager& so
		= game.game_manager.system.sound_output_manager;
	so.load_sound("sounds/click.voc", impl->click_snd);

	tmp_impl.release();
}

Launcher::~Launcher()
{
	delete impl;
}

void
Launcher::tock_handler(uint32_t tocks_passed)
{
	using namespace Vars;
	User_Input_Manager& ui = game.game_manager.system.user_input_manager;

	float oldangle = impl->angle;

	float maxangle = max_launch_angle;
	impl->angle += static_cast<float>(ui.get_axis())/100
		* max_degrees_per_tock;

	impl->angle += static_cast<float>(ui.get_kbdaxis())
		* max_degrees_per_tock;

	float angle_change = impl->angle - oldangle;

	if(std::abs(angle_change) >= 1e-06)
		impl->click_snd.play_sound(0);

	if(impl->angle < -maxangle) impl->angle = -maxangle;
	else if(impl->angle > maxangle) impl->angle = maxangle;
}

float
Launcher::get_angle() const
{
	return impl->angle;
}

void
Launcher::launch()
{
	using namespace std;
	// convert angle to a fixed point q=2 number for lookup
	int32_t angle = static_cast<int32_t>(impl->angle*(1<<2));
	impl->ball.vel.x = Vars::launch_speed*trig.get_sin(angle);
	impl->ball.vel.y = -Vars::launch_speed*trig.get_cos(angle);
	get_launched_ball_manager().add_ball(impl->ball);
	impl->ball.trait = get_chooser().get_next_ball();
}

void
Launcher::reset()
{
	impl->angle = 0;
	impl->ball.trait = get_chooser().get_next_ball();
}

void
Launcher::enable_drawing_handler(bool arg)
{
	switch(arg) {
	case false:
		glDisable(GL_LIGHT1);
		break;

	case true: {
		if(game.game_manager.get_lod() != 1)
			break;
		const float ambient1[] =  {0.0f, 0.0f, 0.0f, 1.0f,};
		const float diffuse1[] =  {0.4f, 0.4f, 0.4f, 1.0f,};
		const float specular1[] = {1.0f, 1.0f, 1.0f, 1.0f,};
		glEnable(GL_LIGHT1);
		glLightfv(GL_LIGHT1, GL_AMBIENT, ambient1);
		glLightfv(GL_LIGHT1, GL_DIFFUSE, diffuse1);
		glLightfv(GL_LIGHT1, GL_SPECULAR, specular1);
		impl->light_position[0] = impl->angle;
		glLightfv(GL_LIGHT1, GL_POSITION, impl->light_position);
		break;
	}

	} // switch(arg)
}

void
Launcher::draw_handler() const
{
	const float r = get_grid().get_r();

	glEnable(GL_LIGHTING);
	glDisable(GL_TEXTURE_2D);

	glPushMatrix();
	    get_grid().draw_ball_commands(impl->ball);
	glPopMatrix();

	impl->light_position[0] = impl->angle;
	glLightfv(GL_LIGHT1, GL_POSITION, impl->light_position);

	glTranslatef(0, impl->ball.pos.y, -r);
	glRotatef(impl->angle, 0, 0, 1);
	glCallList(impl->dlist);
}

} // namespace XBobble


