/* Copyright (C) 2002 Asfand Yar Qazi.

 This file is part of XBobble.

    XBobble is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    XBobble is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XBobble; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/** @file Grid.hh
    The main grid. */

#ifndef XBOBBLE_GRID_HH
#define XBOBBLE_GRID_HH

#include <cstddef>
#include <algorithm>
#include <cmath>
#include <limits>
#include <vector>
#include <sstream>
#include <stdexcept>
#include <ayq/stdint.h>
#include <memory>

#include "OliorGrid.hh"
#include "util.hh"
#include "Ball_Trait.hh"
#include "Video_Output_Manager.hh"
#include "Sound_Data.hh"
#include "Ball.hh"
#include "Game_Elements_Accessor.hh"


namespace std { class bad_alloc; }

namespace XBobble
{

class Grid_Space;
class Grid;

/** A space on the grid containing an object. */
class Grid_Space
{
public:
	/// Grid needs to modify x/y/r values
	friend class Grid;

	/** Default constructor - default values. */
	Grid_Space()
	 : pos(0.0, 0.0), adj(6), ball(0), coord()
	{
	}

	/** Initialise it with arguments. */
	Grid_Space(float arg_x, float arg_y,
		   std::size_t arg_cx, std::size_t arg_cy)
	 : pos(arg_x, arg_y), adj(6), ball(0), visited(false),
	   coord(arg_cx, arg_cy)
	{
	}

	/** Create copies for everything (including ball deep copy)
	    but don't connect up adj pointers.  Let somebody else do
	    it.  Also, visited = false. */
	Grid_Space(const Grid_Space& arg)
	 : pos(arg.pos), adj(6), ball(0), visited(false), coord(arg.coord)
	{
		if(Ball* tmp = arg.ball.get())
			ball.reset(new Ball(*tmp));
	}

	/// As copy constructor
	const Grid_Space&
	operator=(const Grid_Space& arg)
	{
		pos = arg.pos;
		if(Ball* tmp = arg.ball.get())
			ball.reset(new Ball(*tmp));
		else
			ball.reset();
		coord = arg.coord;
		visited = false;
		return *this;
	}

	/// Gets position vector of centre of gridspace relative to
	/// grid.  @see pos
	V2D
	get_pos() const {return pos;}

	/// Gets grid coords
	std::pair<std::size_t, std::size_t>
	get_coord() const {return coord;}

	/** Gives names to the six adjacent spaces. */
	enum {
		R  = 0, ///< right
		DR = 1, ///< down-right
		DL = 2, ///< down-left
		L  = 3, ///< left
		UL = 4, ///< up-left
		UR = 5, ///< up-right
	}; // enum {

	/// Get the given adjacent space
	Grid_Space*
	get_adj(std::size_t space)
	{
		return adj.at(space);
	}

	/// Get the given adjacent space
	const Grid_Space*
	get_adj(std::size_t space) const
	{
		return adj.at(space);
	}

	/// Returns the ball
	const Ball*
	get_ball() const
	{
		return ball.get();
	}

private:
	/// centre position __relative to grids top-left corner__.
	V2D pos;

	/// Adjacent spaces are pointed to by this.  If there isn't an
	/// adjacent space, contains NULL.
	std::vector<Grid_Space*> adj;

	/// The optional ball occupying this space, or NULL.
	std::auto_ptr<Ball> ball;

	/// The visited flag - used by Grid for graph traversal
	mutable bool visited;

	/// The grid co-ordinate (rows, cols) to this space.
	std::pair<std::size_t, std::size_t> coord;

}; // class Grid_Space

class Game;
namespace { class Level_Parser; }

//D: XBobble::Grid
/**
<pre><code>
A grid of spaces (each of which can store an item).

Arranged like so:

A * * * * * ....
 B * * * *  ....
* * C * * * ....
 * * * * *  ....

(see header file Grid.hh for above ascii art diagram - doxygen doesn't
render it properly.)

NOTE: A above is grid space (0, 0).  B is (1, 0).  C is (2, 2).

As you can see, in the grid space co-ordinate (r, c), r is the row
from the top, and c is the column (or space in the row, regardless of
whether it is an odd row or an even row.)

The grid spaces are circles, arranged as shown in diagram
grid_pos.png in the doc directory.
</pre></code>
*/
class Grid : public OliorGrid<Grid_Space>,
	     public Video_Data, public Game_Elements_Accessor
{
public:
	typedef OliorGrid<Grid_Space> Parent;

private:
// constructor/loader helpers

	/// Construct a display list for the current grid
	void
	make_dlist(uint32_t lod);

public:
	/** Initialises with row/column sizes. */
	Grid(Game::Impl& arg_game, size_type rows, size_type cols, float arg_r,
	     const std::string& filename);

	/// Load from the given file
	void
	load_from_file(const std::string& arg);

	/// Radius of balls in grid
	float
	get_r() const {return r;}

	/// Width of grid in local space
	float
	get_w() const {return w;}

	/// Height of grid in local space
	float
	get_h() const {return h;}

	/// Return the given trait, or throw out_of_range
	const Ball_Trait&
	get_trait(std::size_t arg) const
	{
		return ball_traits.at(arg);
	}

	/// Adds the given trait to the traits list and returns its
	/// index
	std::size_t
	add_trait(const Ball_Trait& arg);

private:
	/// Return the given trait, or throw out_of_range
	Ball_Trait&
	get_trait(std::size_t arg)
	{
		return ball_traits.at(arg);
	}

public:
	/// Return a random trait from the traits that are being used.
	/// Throws if no traits left in grid.
	std::size_t
	get_random_trait() const;

	/// Executes commands to draw given ball (using appropriate
	/// trait) at its location (b.pos)
	void
	draw_ball_commands(const Ball& b) const;

private:
	/// Draws the grid and its balls
	void
	draw_handler() const;

public:
	/// Converts the given grid coords to world coords
	V2D
	to_world(const V2D& arg) const;

	/// Converts the given world coords to grid coords
	V2D
	to_grid(const V2D& arg) const;

	/// Sets all the 'visited' flags in all grid spaces to false
	void
	reset_visited_flags();

private:
	/// Sets all the 'visited' flags in all grid spaces to false
	void
	reset_visited_flags() const;

public:
	/// The return types for the add_ball() function
	enum Condition
	{
		OK,
		FINISHED_LEVEL,
		GAME_OVER,
	}; // enum Condition

	/// Adds given ball to grid, gives popped balls to
	/// popped_balls_manager, and throws 'Game_Over' if the grid
	/// check revealed that the lowest ball on the grid is below
	/// the 'game_over_row'.
	Condition
	add_ball(const Ball& b);

	/// Get the grid condition
	Condition
	get_condition() const;

	/// Starting here, traverse grid and label all touching grid
	/// spaces with balls in them.
	void
	Grid::flag_touchings(Grid_Space& arg);

	/// Returns value of 'balls_count'
	uint32_t
	get_balls_count() const
	{
		return balls_count;
	}

private:
	/// Drop the ball at the given Grid_Space (throws
	/// std::invalid_argument if there isn't a ball there)
	void
	drop_ball(Grid_Space&);

	/// Help Grid::count_same_trait_balls()
	void
	count_same_trait_balls_helper(const Grid_Space& arg,
				      uint32_t& count) const;

	/// Counts the balls of the same trait index as the starting
	/// grid ball (including the starting grid ball) that are
	/// touching in a consecutive order.
	uint32_t
	count_same_trait_balls(Grid_Space&) const;

	/// Starting from the given grid space, traverse the grid
	/// spaces and send all balls of the same trait to the
	/// popped_balls_manager.
	void
	drop_same_trait_balls(Grid_Space&);

	/// Get grid space nearest to given coords (given in grid
	/// coordinates)
	Grid_Space&
	get_nearest(const V2D& pos);

	/// @internal Clears out all balls from all spaces
	void
	reset();

	//@{

	/// Finds the lowest space in the grid with a ball in it.  If
	/// there are several, returns the first one it finds.  Throws
	/// 'std::invalid_argument' if there aren't any balls in the
	/// grid.  Doesn't use 'reset_visited_flags()'
	Grid_Space&
	get_lowest_space();

	const Grid_Space&
	get_lowest_space() const;

	//@}

	/** @internal: sets up the co-ordinate values for all the grid
	    spaces.

	    @see doc/grid_layout.png. */
	void
	setup_spaces();

	/** @internal: read 'get grid space pointer'.

	    Takes two arguments, row and column that indicate a grid
	    space.  If they are out of range, returns NULL, else
	    returns pointer to the required grid space.  Used by
	    'setup_touchings().

	    Note how arguments can be negative. */
	Grid_Space*
	getgsptr(int32_t row, int32_t col);

	/** @internal: sets up touching pointers for grid spaces so
	    that they know who the adjacent grid spaces are. */
	void
	setup_touchings();

private:
	friend class Level_Parser;

	/** Radius of one space */
	float r;

	/** Width/height of grid in world coordinates */
	float w, h;

	/// The grid texture
	uint32_t
	texname;

	//@{

	/// Used to keep track of grid move down.
	uint32_t max_move_down_count;
	uint32_t move_down_count;
	uint32_t game_over_row;
	//@}

	/// Ball trait list
	std::vector<Ball_Trait> ball_traits;

	/// Display list for the grid
	uint32_t dlist;

	/// Keeps track of number of balls in the grid
	uint32_t balls_count;

	// Sound data for 'explosion' on balls being popped.
	Sound_Data expl_snd;

}; // class Grid


} // namespace XBobble {

#endif// #define XBOBBLE_GRID_HH


