/* Copyright (C) 2002 Asfand Yar Qazi.

 This file is part of XBobble.

    XBobble is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    XBobble is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XBobble; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/** @file Config_Parser.hh Classes to parse a list of configuration
    statements, from a file or other std::istream source.

    Pseudo syntax-directed translator of configuration statements is
    as follows:

      config_stmt_list: stmt_list

      stmt_list: stmt stmt_list | <empty>

      stmt: string '=' string ';' { var_hash[$1] = $3; }

      string: \".+\"

*/

#ifndef XBOBBLE_CONFIG_PARSER_HH
#define XBOBBLE_CONFIG_PARSER_HH


#include <iostream>
#include <iomanip>
#include <string>
#include <map>
#include <stdexcept>
#include "Lexer_Base.hh"

namespace XBobble
{

/// Token names for all terminal symbols
enum Token
{
	UNKNOWN_TOKEN = 0,
	ASSIGNMENT_TOKEN = static_cast<int>('='),
	END_STATEMENT_TOKEN = static_cast<int>(';'),
	STRING_TOKEN = 256,
	EOF_TOKEN = 257,
	INCOMPLETE_STRING_TOKEN = 258,

}; // enum Token

/// Lexical analyser for config statements; note that that this class
/// takes no responsibility for the construction/deconstruction of its
/// current istream class.
class Config_Lexer : public Basic_Lexer
{
public:
	typedef Basic_Lexer Parent;

	/// Init with no input stream.  The input stream has to be set
	/// later before lexical analysis can commence.
	Config_Lexer();

	/// Init with an istream object to read characters from.  If
	/// arg_istream is not ok to read from, the internal istream
	/// is set to NULL.
	Config_Lexer(std::istream& arg_istream);

	/// Read the next token from the istream, and returns it.  If
	/// the token is a STRING_TOKEN, its value is returned by
	/// calling 'get_last_string()'.  EOF_TOKEN is always returned
	/// if the stream can't be read from any more.  If an
	/// incomplete string (starts with ", but doesn't end with
	/// one) is encountered, returns INCOMPLETE_STRING_TOKEN.  If
	/// EOF was encountered after the INCOMPLETE_STRING, returns
	/// EOF_TOKEN on the next call.
	Token
	next();

	/// Returns the lexeme for the last STRING_TOKEN.  If there
	/// wasn't a last STRING_TOKEN, returns a blank string.
	std::string
	get_last_string() const;

	/// Calls parent's set_istream, and sets stmt_num_ to 1 as well.
	void
	set_istream(std::istream& arg_istream);

	/// Returns the current statement number.  Note that the
	/// current statement number is reset to 1 after loading a new
	/// istream.
	int
	get_stmt_num() const;

protected:
	/// Last STRING_TOKEN lexeme (if there was one.)
	std::string last_string;

	/// The current statement number being processed; reset to 1
	/// every time istream is reset.
	int stmt_num;

}; // class Config_Lexer

/// A simple predictive parser for config statements with rudimentary
/// error handling.
class Config_Parser
{
public:
	/// The type used for the symbol table
	typedef std::map<std::string, std::string> Symtab;
	typedef const std::map<std::string, std::string> constSymtab;

	/// Init this object with default values.
	Config_Parser();

	/// Returns the lookahead token.
	Token
	get_lookahead() const;

	/// Returns a const reference to the lexer being used.  Note
	/// that since the reference is const, only the const
	/// functions of the lexer may be called.
	const Config_Lexer&
	get_lexer() const;

	/// Exception thrown by Config_Parser::process() if any of the
	/// input or output streams are invalid before being used.
	class Bad_Stream : public std::logic_error
	{
	public:
		Bad_Stream(const std::string& s)
		 : std::logic_error(s)
		{
		}

	}; // class Bad_Stream

	/// Processes the input from the given istream.  Writes error
	/// messages to the given ostream.  Variables are stored in
	/// the given std::map<string, string>.  error_prefix is
	/// prefixed onto each error message
	void
	process(std::istream& input,
		std::ostream& error_output,
		Symtab& symtab,
		const std::string& error_prefix = "")
		throw(Bad_Stream);

protected:
	/// Exception thrown by match() doesn't match the lookahead
	/// with the argument.
	class Unmatched : public std::runtime_error
	{
	public:
		Unmatched()
		 : std::runtime_error("")
		{
		}

	}; // class Unmatched

	/// If the lookahead matches the given token, get the next
	/// lookahead.  If the lookahead doesn't match the given
	/// token, throw Unmatched and don't get the next lookahead.
	void
	match(Token arg) throw(Unmatched);

	/// Procedure for the stmt_list nonterminal.
	void
	stmt_list();

	/// Procedure for the stmt nonterminal.
	void
	stmt();

private:
	/// The current lookahead token.
	Token lookahead;

	/// The lexer this parser uses to get tokens.
	Config_Lexer lexer;

	/// The current map<string, string> to use for variable
	/// entries.
	Symtab* symtab;

	/// The ostream to output error messages to.
	std::ostream* streamout;

	/// The error prefix to give it
	std::string error_prefix;

}; // class Config_Parser


} // namespace XBobble


#endif // #define XBOBBLE_CONFIG_PARSER_HH


