{-
Tests.hs

Xadrez - A p2p chess written in Haskell
Copyright (C) 2004-2005 Marco Tulio Gontijo e Silva

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-}
module Tests (correct, valid) where

import Char

import Board

correct :: String -> String -> Bool
correct p1 p2
    = (length p1 == 2) && (length p2 == 2) && (isLetter $ head p1) &&
      (isLetter $ head p2) && (isNum $ head $ tail p1) &&
      (isNum $ head $ tail p2)

isLetter, isNum :: Char -> Bool
isLetter c = (c >= 'A') && (c <= 'H')
isNum c = (c >= '1') && (c <= '8')

valid :: Board -> Bool -> Place -> Place -> Bool
valid board player from to
    = (sameColor from' player) &&
      ((isEmpty to') || (not $ sameColor to' player)) &&
      (pieceTest board from to) 
      -- && (flip notCheck player $ move (from, to) board)
      where to' = pieceIn board to
	    from' = pieceIn board from

pieceTest :: Board -> Place -> Place -> Bool
pieceTest board from to
    | isKing from' = next from to || isCastling board from to
    | isQueen from'
	= ((isV from to) && (freeWay from' $ inV board from to)) ||
	  ((isH from to) && (freeWay from' $ inH board from to)) ||
          ((isD from to) && (freeWay from' $ inD board from to))
    | isRook from'
	= ((isV from to) && (freeWay from' $ inV board from to)) ||
	  (isH from to) && (freeWay from' $ inH board from to)
    | isKnight from' = knightTest from to
    | isBishop from'
        = ((isD from to) && (freeWay from' $ inD board from to))
    | isPawn from' = pawnTest board from to || isPassant board from to
    | otherwise = error "Tests.pieceTest called with an unexpected parameter"
    where from' = pieceIn board from

next :: Place -> Place -> Bool
next (i1, j1) (i2, j2) = (abs (i1 - i2) <= 1) && (abs (j1 - j2) <= 1)

isV :: Place -> Place -> Bool
isV (_, j1) (_, j2) = j1 - j2 == 0

isH :: Place -> Place -> Bool
isH (i1, _) (i2, _) = i1 - i2 == 0

isD :: Place -> Place -> Bool
isD (i1, j1) (i2, j2) = (abs (i1 - i2)) == (abs (j1 - j2))

inV :: Board -> Place -> Place -> [Piece]
inV board (i1, j1) (i2, _)
    | big == small = []
    | otherwise = pieceIn board (big, j1) : inV board (big - 1, j1) (small, j1)
    where big = max i1 i2
	  small = min i1 i2

inH :: Board -> Place -> Place -> [Piece]
inH board (i1, j1) (_, j2)
    | big == small = []
    | otherwise = pieceIn board (i1, big) : inV board (i1, big - 1) (i1, small)
    where big = max j1 j2
	  small = min j1 j2

inHPlace :: Place -> Place -> [Place]
inHPlace (i1, j1) (_, j2)
    | big == small = [(i1, j1)]
    | otherwise = (i1, big) : inHPlace (i1, big - 1) (i1, small)
    where big = max j1 j2
	  small = min j1 j2

inD :: Board -> Place -> Place -> [Piece]
inD board (i1, j1) (i2, j2)
    | i1 == i2 = []
    | otherwise = pieceIn board (i1, j1) :
		  inD board (i1 + opri, j1 + oprj) (i2, j2)
    where opri = if i1 > i2 then (-1) else 1
	  oprj = if j1 > j2 then (-1) else 1

freeWay :: Piece -> [Piece] -> Bool
freeWay _ [] = True
freeWay x (p : ps)
    | isEmpty p = freeWay x ps
    | otherwise = (x == p) && freeWay x ps

knightTest :: Place -> Place -> Bool
knightTest (i1, j1) (i2, j2)
    = ((abs (i1 - i2) == 2) && (abs (j1 - j2) == 1)) ||
      ((abs (j1 - j2) == 2) && (abs (i1 - i2) == 1))

pawnTest :: Board -> Place -> Place -> Bool
pawnTest board (i1, j1) to@(i2, j2)
    | isEmpty $ pieceIn board to = (m (i2 - i1)) && (j1 - j2 == 0)
    | otherwise = (i2 - i1 == 1) && (abs (j1 - j2) == 1)
    where m = if i1 == 1 then (`elem` [1, 2]) else (== 1)

isCastling :: Board -> Place -> Place -> Bool
isCastling board from@(_, j1) to@(_, j2)
    = (isH from to) && (abs(j1 - j2) == 2) && (firstMove from') && 
      (freeWay from' $ inH board from to) &&
      (clean board (sameColor from' white) $ inHPlace from to) &&
      (((j1 > j2) && (firstMove $ pieceIn board (0, 0))) || 
       ((j2 > j1) && (firstMove $ pieceIn board (0, 7))))
    where from' = pieceIn board from

clean :: Board -> Bool -> [Place] -> Bool
clean _ _ [] = True
clean board c (p : _)
    = (hookAttack board c p) && (bishopAttack board c p) &&
      (knightAttack board c p) && (kingAttack board c p)

hookAttack :: Board -> Bool -> Place -> Bool
hookAttack board c (i, j)
    = (up board c (i + 1, j)) && (right board c (i, j + 1))
      && (left board c (i, j - 1))

up :: Board -> Bool -> Place -> Bool
up board c p@(i, j)
    | i > 7 = True
    | isEmpty p' = up board c (i + 1, j)
    | sameColor p' c = True
    | (isQueen p') || (isRook p') = False
    | otherwise = error "Tests.up called with an unexpected parameter"
    where p' = pieceIn board p

right :: Board -> Bool -> Place -> Bool
right board c p@(i, j)
    | j > 7 = True
    | isEmpty p' = right board c (i, j + 1)
    | sameColor p' c = True
    | (isQueen p') || (isRook p') = False
    | otherwise = error "Tests.right called with an unexpected parameter"
    where p' = pieceIn board p

left :: Board -> Bool -> Place -> Bool
left board c p@(i, j)
    | j < 0 = True
    | isEmpty p' = left board c (i, j - 1)
    | sameColor p' c = True
    | (isQueen p') || (isRook p') = False
    | otherwise = error "Tests.left called with an unexpected parameter"
    where p' = pieceIn board p

bishopAttack :: Board -> Bool -> Place -> Bool
bishopAttack board c (i, j)
    = (upRight board c (i + 1, j + 1)) && (upLeft board c (i + 1, j - 1))

upRight :: Board -> Bool -> Place -> Bool
upRight board c p@(i, j)
    | i > 7 || j > 7 = True
    | isEmpty p' = upRight board c (i + 1, j + 1)
    | sameColor p' c = True
    | (isQueen p') || (isBishop p') = False
    | otherwise = error "Tests.upRight called with an unexpected parameter"
    where p' = pieceIn board p

upLeft :: Board -> Bool -> Place -> Bool
upLeft board c p@(i, j)
    | i > 7 || j < 0 = True
    | isEmpty p' = upLeft board c (i + 1, j - 1)
    | sameColor p' c = True
    | (isQueen p') || (isBishop p') = False
    | otherwise = error "Tests.upLeft called with an unexpected parameter"
    where p' = pieceIn board p

knightAttack :: Board -> Bool -> Place -> Bool
knightAttack board c (_, j)
    = isOther board c [(1, j - 2), (2, j - 1), (2, j + 1), (1, j + 2)] isKnight

isOther :: Board -> Bool -> [Place] -> (Piece -> Bool) -> Bool
isOther _ _ [] _ = True
isOther board c (p : ps) f
    | fromKing p = (not ((f p') && (sameColor p' (not c)))) &&
	       isOther board c ps f
    | otherwise = isOther board c ps f
    where p' = pieceIn board p

fromKing :: Place -> Bool
fromKing (i, j) = (i >= 0) && (j >= 0) && (i <= 7) && (j <= 7)

kingAttack :: Board -> Bool -> Place -> Bool
kingAttack board c (_, j) 
    = (isOther board c
       [(0, j - 1), (1, j - 1), (1, j), (1, j + 1), (0, j + 1)] isKing) ||
      (isOther board c [(1, j - 1), (0, j + 1)] isPawn)

isPassant :: Board -> Place -> Place -> Bool
isPassant board (i1, j1) to@(i2, j2)
    = (i2 - i1 == 1) && (abs(j2 - j1) == 1) && (isEmpty to') &&
      (isPassantPawn $ pieceIn board (i2 - 1, j2))
    where to' = pieceIn board to
{-
notCheck :: Board -> Bool -> Bool
notCheck board player
    = clean [kingPlace board player]

kingPlace :: Board -> Bool -> Place
kingPlace board player -}