#!/usr/bin/env python

""" Generates HTML from wiki pages input.
    The wiki text syntax and translation rules to HTML are
    encapsulated in the class WikiTranslator.
"""
copyright = """
Copyright (C) 2003 Jaime Villate <villate@gnu.org>
 """
license = """
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""

import re, sys, traceback, pagemodel
from cgi import escape
from wikiup import PrintPage, Config
try:
    from cStringIO import StringIO
except:    
    from StringIO import StringIO
BOGUS_NAME = Config('static','bogus_name')

class PageConfiguration:
    """ Defines an object to hold configuration information for a
        specific wiki.
    """
    def __init__(self, ScriptName, ArchiveDir, CgiUrl, DefaultPageName):
        self.scriptName = ScriptName
        self.archiveDir = ArchiveDir
        self.cgiUrl = CgiUrl
        self.defaultPageName = DefaultPageName
        return

def IsLegalPageName(PageName):
    """ Return 1 if PageName is legal, 0 otherwise.
    """
    return re.match('[A-Za-z\d][A-Za-z-_\d\/]+$', PageName)

def DefaultBogusPageName():
    
    """ Return a default page name for illegal pages.
    """
    return BOGUS_NAME

def DefaultPageText(PageName):
    
    """ Return a list of text strings to initialize a new page
        to default.
    
        PageName is the name of the new page.
    """
    return "The page " + escape(PageName) + " does not exist yet. " \
           + "You can create it using the Edit link below"


def SearchResults(Configuration, SearchString, SearchResults,
                      ArchivesFound, ArchivesSearched, template, contributor):
    """ Send the results of a text search.
    
        SearchString is the text that was searched for.
        SearchResults is a list of tuples (page, inword)
          where page is the name of the page where the text was found
          and inword is the entire word in which the text fragment was found.
        ArchivesFound is the number of pages in which matches were found.
        ArchivesSearched is the number of page archives which were searched.
    """
          
    esc_searchString = escape(SearchString)
    title = 'Search for ' + esc_searchString
    text = ''
    for page, inword in SearchResults:
        text += '<a href="' + Configuration.scriptName + '?' + page + '">' \
              + page + '</a> . . . . . .  ' + inword + '<br>\n<hr>\n'
    text += '%s pages found out of %s pages searched.' % \
          (ArchivesFound, ArchivesSearched)
    PrintPage(title, text, template, username=contributor)
    return


def Page(Configuration, PageName, PageText, Date, SearchArg, template,
         contributor):
    """ Given raw page data, a page is formatted and sent.

        PageName is the name of the page.
        PageText is a wiki markup text string for the page.
        Date is the latest revision date for PageName
        SearchArg is a string to be pre-place in any search boxes on
            the page.
    """
    esc_name = escape(PageName)
    title = esc_name
    text = contentPage(Configuration, PageText, SearchArg)
    text += '<hr>\n'
    text += '<a href="' + Configuration.scriptName + '?edit=' \
          + esc_name + '">Edit</a> this page\n'
    if Date:
        text += '(last edited %s)\n' % Date
    text +=  '<br>\n<a href="' + Configuration.scriptName +'?FindPage&value='\
          + esc_name + '">FindPage</a> by browsing or searching<br>\n'
    PrintPage(title, text, template, username=contributor)
    return

    
def TextEditor(Configuration, PageName, PageText, PageRevision,
           TipDateTimeAtCheckout, template, contributor, Rows=18, Cols=80):

    """ Send an edit form with Wiki format page text.
    
        PageName is the name of the page to be edited.
        PageText is a wiki markup text string for the page.
        PageRevision is the revision of the page that PageText came from.

        If the optional Rows and Cols args are given, these will set the
        size of the edit window. If they are not given, a default size edit
        window is sent.
    """
    esc_name = escape(PageName)
    esc_rev = escape(PageRevision)
    title = 'Edit: ' + esc_name
    text = '<h1>Edition of: ' + esc_name + '</h1>\n'
    text += _contentTextForm(Configuration, esc_name, PageText, esc_rev,
                     TipDateTimeAtCheckout, Rows, Cols)
    PrintPage(title, text, template, username=contributor)
    return

def PreviewEditor(Configuration, PageName, PageText, PageRevision,
             TipDateTimeAtCheckout, template, contributor, Rows=18, Cols=80):

    """ Send an edit form with Wiki format page text.
    
        PageName is the name of the page to be edited.
        PageText is a wiki markup text string for the page.
        PageRevision is the revision of the page that PageText came from.

        If the optional Rows and Cols args are given, these will set the
        size of the edit window. If they are not given, a default size edit
        window is sent.
    """
    esc_name = escape(PageName)
    esc_rev = escape(PageRevision)
    title = esc_name
    text = '<h1>Preview of: ' + esc_name + '</h1>\n<hr>\n'
    text += contentPage(Configuration, PageText, '')
    text += """<hr>
    <p>You can still edit this page in the editor at the bottom.<br>
    <font color="red"><strong>Do not forget to save your changes
    once you are satisfied with the results!</strong></font></p>
    <hr>
    """
    text += _contentTextForm(Configuration, esc_name, PageText, esc_rev,
                       TipDateTimeAtCheckout, Rows, Cols)
    PrintPage(title, text, template, username=contributor)
    return

def ContributionAcknowledge(Configuration, PageName, template, contributor):

    """ Send acknowledgement of a change contribution.
    
        PageName is the name of the page changed.
        Contributor identifies the entity responsible for the change.
    """    
    esc_name = escape(PageName)
    esc_contributor = escape(contributor)
    title = 'Thanks for editing ' + esc_name
    text = '<h1>Thanks for editing ' + esc_name + '.</h1>\n'
    text += '<p>Thank you for editing <a href="' + Configuration.scriptName +\
          '?' + esc_name + '">' + esc_name + '</a>.</p>\n'
    text += '<p>p.s. Be sure to <em>Reload</em> your old pages.</p>\n'
    PrintPage(title, text, template, username=contributor)
    return
 

def Log(Configuration, PageName, LogEntries, template, contributor):

    """ Sends a revision log page with edit links for each revision.

        PageName is the name of a page.
        LogEntries is a list of tuples (Revision, RevisionDate, Contributor)
        where Revision is the revision of the file, RevisionDate is the date
        of that revision, and Contributor identifies the entity
        responsible for the revision.
    """
    esc_name = escape(PageName)
    title = esc_name + ' Revisions'
    text = '<h1>' + esc_name + ' Revision Log</h1>\n'
    rev, revDate, contributor = LogEntries[0]
    text += '<dl>\n<dt>' + rev + ' - Current Revision</dt>\n<dd><ul>\n'
    text += '<li> revised: ' + revDate + '</li>\n<li> by: ' + contributor
    text += '</li>\n</ul></dd></dl>\n'
    for rev, revDate, contributor in LogEntries[1:]:
        text += '<dl>\n<dt> <a href="' + Configuration.scriptName + '?edit=' \
                + PageName + '&rev=' + rev + '">' + rev + '</a></dt>\n'
        text += '<dd><ul>\n<li> revised: ' + revDate + '</li>\n<li> by: ' \
                + contributor
        text += '</li></ul></dd></dl>\n<hr>'
    PrintPage(title, text, template, username=contributor)
    return    


def Error(PageName, Title, ErrorText, template, contributor):

    """ Sends an error page with arbitrary text.
    
        PageName is the name of a page.
        Title is the title for the error page.
        ErrorText is the text of the desired error message.
    """
    esc_title = escape(Title)
    title = esc_title
    text = '<h1>' + esc_title + '</h1>\n' + escape(ErrorText) + '\n'
    PrintPage(title, text, template, username=contributor)
    return

def ErrorTraceback(PageName, Title, ErrorText, template, contributor):

    """ Sends an error page with arbitrary text and stack traceback.
    
        PageName is the name of a page.
        Title is the title for the error page.
        ErrorText is the text of the desired error message.
    """
    esc_title = escape(Title)
    title = esc_title
    text = '<h1>' + esc_title + '</h1>\n' + escape(ErrorText) + '\n'
    text += ''.join(traceback.format_exception(sys.exc_type, sys.exc_value, \
               sys.exc_traceback))
    PrintPage(title, text, template, username=contributor)
    return

def Info(PageName, Title, InfoText, template, contributor):

    """ Sends an information page with arbitrary text.
    
        PageName is the name of a page.
        Title is the title for the information page.
        InfoText is the text of the desired error message.
    """
    esc_title = escape(Title)
    title =  esc_title
    text = '<h1>' + esc_title + '</h1>\n' + escape(InfoText)
    PrintPage(title, text, template, username=contributor)
    return

def Report(PageTitle, ReportText, template, contributor):

    """ Sends an report page with arbitrary text.
    
        PageTitle is the title of the page.
        ReportText is the text of the desired report.
    """
    esc_title = escape(PageTitle)
    title = esc_title
    text = '<h1>' + esc_title + '</h1>\n' + ReportText
    PrintPage(title, text, template, username=contributor)
    return

def contentPage(Configuration, PageText, SearchArg):
    """Prints the contents of a page."""
    wt = WikiTranslator(Configuration, PageText, SearchArg)
    return ''.join(wt.TranslateToHtmlLines())

def _contentTextForm(Configuration, esc_name, PageText, esc_rev,
                       TipDateTimeAtCheckout, Rows, Cols):

    """ Create the form for the Text Editor. Internal use only.
    """
    form = '<form method="POST" action="' + Configuration.scriptName + '">'
    form += """
    <input type="submit" name="action" value=" Preview ">
    <input type="submit" name="action" value=" Save ">
    <input type="reset" value=" Reset "></br>
    Log message:
    <input type="text" name="log" size="60" maxlength="360">
    """
    form += '<textarea name="text" ROWS="%s" COLS="%s" wrap="virtual">' % \
              (Rows, Cols)
    form += '\n' + escape(PageText) + '\n</textarea><br>\n'
    form += '<a href="' + Configuration.scriptName + \
            '?GoodStyle">GoodStyle</a> tips for editing.<br>'
    if esc_rev != '':
        form += '<a href="' + Configuration.scriptName + \
                 '?rows=12&cols=60&edit=' + esc_name + '&rev=' + esc_rev + \
                 '">EditPage</a> using a smaller text area.<br>\n'
    else:
        form += '<a href="' + Configuration.scriptName + \
              '?rows=12&cols=60&edit=' + esc_name + \
              '">EditPage</a> using a smaller text area.<br>\n'
    form += '<a href="' + Configuration.scriptName + '?copy=' \
            + esc_name + '">EditCopy</a> from earlier revision.<br>\n'
    if esc_rev != '':
         form += '<p><b>Editing copy. This is not the current '
         form += 'revision!</b><br>\n'
    form += '<input type="hidden" size=1 name="text_edit" value="yes">\n'
    form += '<input type="hidden" size=1 name="page_name" value="' \
          + esc_name + '">\n'
    form += '<input type="hidden" size=1 name="rev" value="' + esc_rev + \
          '">\n'
    form += '<input type="hidden" size=1 name="tip_date_time" value="' + \
          TipDateTimeAtCheckout + '">\n</form>\n'
    return form

#--- Wiki Translation ---------------------------------------------------------

TranslationToken = '\0263'

class WikiTranslator:
    """ Instantiated with page data, the TranslateToHtmlLines() method
        of a WikiTranslator object returns a list of HTML formatted text
        lines.
    """

    def __init__(self, Configuration, PageText, SearchArg):
        self.configuration = Configuration
        self.pageText = PageText
        self.searchArg = SearchArg
        self.ampExpr = re.compile('&')
        self.ltExpr = re.compile('<')
        self.gtExpr = re.compile('>')
        self.urlExpr = re.compile( r'([^\[]|^)\b((http|ftp|mailto|news|file|gopher)[^\s\<\>\[\]"\'\(\)]*[^\s\<\>\[\]"\'\(\)\,\.\?])')
        self.blankExpr = re.compile(r'^\s*$')
        self.listExpr = re.compile(r'\t+')
        self.dlExpr = re.compile(r'^\t+(.+):\t')
        self.ulExpr = re.compile(r'^\t+\*')
        self.olExpr = re.compile(r'^\t+\d+\.?')
        self.preExpr = re.compile(r'^\s')
        self.strongExpr = re.compile(r'\*(.*?)\*')
        self.ttExpr = re.compile(r'\`(.*?)\`')
        self.emphExpr = re.compile(r'\_(.*?)\_')
        self.hExpr = re.compile(r'^=+')
        self.headerExpr = re.compile(r'^=+\s*([^=]+)\s*=+')
        self.hrExpr = re.compile(r'^-----*')
        self.internExpr = re.compile(r'\[([A-Za-z\d][A-Za-z-_\d\/]+)\]')
        self.internExpr2 = re.compile(r'\[([A-Za-z\d][A-Za-z-_\d\/]+)\s+([^\]]+)\]')
        self.externExpr = re.compile(r'\[(http:[^\s\<\>\[\]"\'\(\)]+)\s+([^\]]+)\]')
        self.inplaceurlExpr = re.compile( \
                TranslationToken + r'(\d+)' + TranslationToken)
        self.searchExpr = re.compile(r'\[Search\]')
        self.imageExpr = re.compile(r'.*\.((gif)|(jpg)|(png))')
        return


    def TranslateToHtmlLines(self):
        """
        """
        self.htmlPageText = []
        estk = self.emitterStack(self.htmlPageText)
        self.inPlaceUrls = 0
        self.inPlaceUrlList = []
        self.inPlaceTextList = []
        for line in StringIO(self.pageText).readlines():
            line = self.urlExpr.sub(self.urlPlaceHolder, line)
            line = self.ampExpr.sub('&amp;', line)
            line = self.ltExpr.sub('&lt;', line)
            line = self.gtExpr.sub('&gt;', line)
            Code = ""
            (line, subsMade) = self.blankExpr.subn('<p>', line)
            if subsMade > 0:
                Code = '...'
            m = self.listExpr.match(line)
            if m:
                depth = len(m.group(0))
                (line, subsMade) = self.dlExpr.subn(r'<dt>\g<1><dd>', line)
                if subsMade > 0:
                    Code = 'DL'
                    estk.emitCode(Code, depth)
                (line, subsMade) = self.ulExpr.subn('<li> ', line)
                if subsMade > 0:
                    Code = 'UL'
                    estk.emitCode(Code, depth)
                (line, subsMade) = self.olExpr.subn('<li> ', line)
                if subsMade > 0:
                    Code = 'OL'
                    estk.emitCode(Code, depth)
            if self.preExpr.match(line):
                Code = 'PRE'
                estk.emitCode(Code, 1)
            if Code == "":
                estk.emitCode("", 0)

            line = self.strongExpr.sub(self.substituteStrongText, line)
            line = self.emphExpr.sub(self.substituteEmphasizedText, line)
            line = self.ttExpr.sub(self.substituteTypewriterText, line)
            m = self.hExpr.match(line)
            if m:
                depth = len(m.group(0))
                (line, subsMade) = self.headerExpr.subn('<h'+`depth`+ \
                                           r'>\g<1></h'+`depth`+'>', line)
            line = self.hrExpr.sub('<hr>', line)
            line = self.internExpr.sub(self.substituteInternalLink, line)
            line = self.internExpr2.sub(self.substituteInternalLink2, line)
            line = self.externExpr.sub(self.substituteExternalLink, line)
            line = self.inplaceurlExpr.sub(self.substituteInPlaceUrls, line)
            line = self.searchExpr.sub(self.substituteSearchField, line)
            self.htmlPageText.append(line)
        estk.emitCode("", 0)
        return self.htmlPageText

    def urlPlaceHolder(self, MatchObject):
        pre = MatchObject.group(1)
        self.inPlaceUrlList.append(MatchObject.group(2))
        self.inPlaceTextList.append(MatchObject.group(2))
        placeNum = str(self.inPlaceUrls)
        self.inPlaceUrls = self.inPlaceUrls + 1
        return pre + TranslationToken + placeNum + TranslationToken

    def substituteInternalLink(self, MatchObject):
        fullName = MatchObject.group(1)
        text = fullName
        if pagemodel.PageExists(self.configuration.archiveDir, fullName):
            return '<a href="' + self.configuration.scriptName + '?' + \
                   fullName + '">' + text + '</a>'
        else:
            return text + \
                   '<a href="' + self.configuration.scriptName + '?' + \
                   fullName + '">?</a>'

    def substituteInternalLink2(self, MatchObject):
        fullName = MatchObject.group(1)
        text = MatchObject.group(2)
        if pagemodel.PageExists(self.configuration.archiveDir, fullName):
            return '<a href="' + self.configuration.scriptName + '?' + \
                   fullName + '">' + text + '</a>'
        else:
            return text + \
                   '<a href="' + self.configuration.scriptName + '?' + \
                   fullName + '">?</a>'

    def substituteExternalLink(self, MatchObject):
        self.inPlaceUrlList.append(MatchObject.group(1))
        self.inPlaceTextList.append(MatchObject.group(2))
        placeNum = str(self.inPlaceUrls)
        self.inPlaceUrls = self.inPlaceUrls + 1
        return TranslationToken + placeNum + TranslationToken

    def substituteInPlaceUrls(self, MatchObject):
        url = self.inPlaceUrlList[int(MatchObject.group(1))]
        text = self.inPlaceTextList[int(MatchObject.group(1))]
        if self.imageExpr.match(url):
            return ('<img src="%s">' % (url))
        else:
            return ('<a href="%s">%s</a>' % (url,text))


    class emitterStack:
        """ class used by translateToHtmlLines() above. """

        def __init__(self, HtmlPageText):
            self.htmlPageText = HtmlPageText
            self.stack = []
        def push(self, Code):
            self.stack = [Code] + self.stack
        def pop(self):
            top, self.stack = self.stack[0], self.stack[1:]
            return top

        def emitCode(self, Code, Depth):
            while (len(self.stack) > Depth):
                self.htmlPageText.append("</%s>\n" % self.pop())
            while (len(self.stack) < Depth):
                self.push(Code)
                self.htmlPageText.append("<%s>\n" % Code)
            if len(self.stack) != 0 and self.stack[0] != Code:
                self.htmlPageText.append("</%s><%s>\n" % (self.stack[0], Code))
                self.stack[0] = Code
            return

    def substituteEmphasizedText(self, MatchObject):
        return '<em>' + MatchObject.group(1) + '</em>'

    def substituteStrongText(self, MatchObject):
        return '<strong>' + MatchObject.group(1) + '</strong>'

    def substituteTypewriterText(self, MatchObject):
        return '<tt>' + MatchObject.group(1) + '</tt>'

    def substituteSearchField(self, MatchObject):
        return '<p><form><input type="text" size="40" name="search" value="' \
               + self.searchArg + '"></form><p>'


