#!/usr/bin/env python

""" Parses CGI input and executes the appropriate events.
    The events invoke functions in the pagemodel and pageview modules to
    provide specific wiki functionalities.
"""
copyright = """
Copyright (C) 2003 Jaime Villate <villate@gnu.org>
 """
license = """
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
"""

import os, sys, re, string, cgi
import pagemodel, pageview
from wikiup import ReadCookie, GetUsername, Config
try:
    from cStringIO import StringIO
except:    
    from StringIO import StringIO
CHANGE_LOG = Config('cvs','change_log')
DYN_TEMPLATE = Config('dynamic','template')
STA_TEMPLATE = Config('static','template')

class CgiEvents:
    """ Initialized with a wiki Configuration object. The Process()
        method interprets a CGI query and invokes appropriate functions
        in the pagemodel and pageview modules.
    """

    def __init__(self, Configuration):
        """
            The Configuration class is defined in pageview.py
        """
        self.configuration = Configuration
        self.archiveDir = self.configuration.archiveDir
        return

    def Process(self, Environment, Input):
        """
            To be called by the base CGI script. All the environment and
            standard input processing is handled from here.
        """
	self.dyntemplate = DYN_TEMPLATE
	self.statemplate = STA_TEMPLATE
        self.contributor = ''
        hash = ReadCookie('session')
        if hash: self.contributor = GetUsername(hash)
	if not self.contributor: self.contributor = 'Unknown'
        self.queries = {}
        if Environment['REQUEST_METHOD'] == 'GET':
            try:
                queryString = Environment['QUERY_STRING']
            except:
                queryString = ""
            if queryString == "":
                queryString = 'browse=' + self.configuration.defaultPageName
            queryString = re.sub(r"^([^=&]*(?:$|&.*))",
                                  "browse=\g<0>",
                                  queryString) 
            self.queries = cgi.parse_qs(queryString)
            self.GET_request()
        else:
            queryString = Input.read(string.atoi(Environment['CONTENT_LENGTH']))
            self.queries = cgi.parse_qs(queryString)
            if Environment.has_key('REMOTE_HOST'):
                self.contributor = Environment['REMOTE_HOST']
            if self.contributor == '' and Environment.has_key('REMOTE_ADDR'):
                self.contributor = Environment['REMOTE_ADDR']
            self.POST_request()


    def POST_request(self):
        """ Handle a CGI POST query.
        """
        if self.queries.has_key("page_name"):
            self.pageName = self._validatePageName(self.queries["page_name"][0])

            if self.queries.has_key("text_edit"):
                try:
                    pageText = self.queries['text'][0]
                except:
                    pageText = ''

                try:
                    logMsg = self.queries['log'][0]
                except:
                    logMsg = ''

                try: 
                    revision = self.queries["rev"][0]
                except:
                    revision = ''    

                if  self.queries.has_key("rows"):
                    rows = self.queries["rows"][0]
                else:
                    rows = 0    
                if  self.queries.has_key("cols"):
                    cols = self.queries["cols"][0]
                else:
                    cols = 0

                try: 
                    tipDateTime = self.queries["tip_date_time"][0]
                except:
                    tipDateTime = ""
                try:
                    action = self.queries['action'][0]
                except:
                    action = " Preview "
                if action == " Preview ":
                    self._eventTextPreview(pageText, tipDateTime,
                                           revision, rows, cols)
                else:
                    self._eventTextSave(pageText, tipDateTime, logMsg,
                                        self.statemplate)

            else:
                pageview.Error(self.pageName, 'Incomplete Form',
                               "Illegal edit form. No edit type.",
                               self.dyntemplate, self.contributor)
        
        else:
            pageview.Error('', 'Incomplete Form',
                           "Illegal edit form. No page name.",
                               self.dyntemplate, self.contributor)
        return    


    def GET_request(self):
        """ Handle a CGI GET query.
        """
        try: 
            revision = self.queries["rev"][0]
        except:
            revision = ''    

        if self.queries.has_key("edit"):
            self.pageName = self._validatePageName(self.queries["edit"][0])
            if  self.queries.has_key("rows"):
                rows = self.queries["rows"][0]
            else:
                rows = 0    
            if  self.queries.has_key("cols"):
                cols = self.queries["cols"][0]
            else:
                cols = 0
            self._eventTextFormRequest(rows, cols, revision)

        elif self.queries.has_key("copy"):
            self.pageName = self._validatePageName(self.queries["copy"][0])
            self._eventLogRequest()

        elif self.queries.has_key("search"):
            self._eventSearchRequest(self.queries["search"][0])

        else:
            try:
                self.pageName = self._validatePageName(self.queries["browse"][0])
            except:
                pageview.Error('', 'Illegal Request',
                               'Request not understood.',
                               self.dyntemplate, self.contributor)

            if self.queries.has_key("value"):
                searchString = self.queries["value"][0]
            else:
                searchString = ''
            self._eventPageRequest(searchString)
        return

    def _eventTextPreview(self, PageText, CheckoutTipDateTime, Revision,
                          Rows, Cols):
        # Convert spaces into tabs
        newText = ''
        for line in StringIO(PageText).readlines():
            line = re.sub(' {3,8}', self.substituteTabsForSpaces, line)
            newText += line
        PageText = newText
        if Rows == 0 and Cols == 0:
            pageview.PreviewEditor(self.configuration, self.pageName, PageText,
               Revision, CheckoutTipDateTime, self.dyntemplate,
               self.contributor)
        else:
            pageview.PreviewEditor(self.configuration, self.pageName, PageText,
              Revision, CheckoutTipDateTime, self.dyntemplate,
              self.contributor, Rows, Cols)
        return

    def _eventTextSave(self, PageText, CheckoutTipDateTime, logMsg, Template):
        # Convert spaces to tabs
        newText = ''
        for line in StringIO(PageText).readlines():
            line = re.sub(' {3,8}', self.substituteTabsForSpaces, line)
            newText += line
        PageText = newText
	title = self.pageName
        home = Config('static','home_path')
        text = pageview.contentPage(self.configuration, newText, '')
        html = open(Template)
        HTMLText = html.read() % locals()
        html.close()
        try:
            pagemodel.PutPage(self.archiveDir, self.pageName, PageText,
                    CheckoutTipDateTime, self.contributor, logMsg, HTMLText)
        except:
            pageview.Error(self.pageName, 'Editing Conflict',
                           "Another author has modified this page " +
                           "since you checked it out. So that you " +
                           "won't interfere with the other author's " +
                           "work, you need to reload, re-edit, and " +
                           "submit again. Sorry!",
                            self.dyntemplate, self.contributor)
            return

        pageview.ContributionAcknowledge(self.configuration, self.pageName,
                                         self.dyntemplate, self.contributor)
        if pageview.IsLegalPageName(self.pageName):
            # don't show changes to 'bogus' page
            pagemodel.UpdateChangesPage(self.archiveDir, CHANGE_LOG,
                          self.pageName, self.contributor, logMsg)
        return

    def substituteTabsForSpaces(self, MatchObject):
        """ utility function used by _eventTextForm above."""
        return '\t'

    def _eventTextFormRequest(self, Rows, Cols, Revision):
        if pagemodel.PageExists(self.archiveDir, self.pageName):
            try:
                (pageText, revision, nowTipDateTime) = \
                   pagemodel.GetPage(self.archiveDir, self.pageName, Revision)
            except:
                pageview.Error(self.pageName, 'Page Access Error',
                               " Couldn't get page for editing." +
                               " Try again later.",
                               self.dyntemplate, self.contributor)
                return            
        else:
            pageText = pageview.DefaultPageText(self.pageName)
            nowTipDateTime = ''
        if Rows == 0 and Cols == 0:
            pageview.TextEditor(self.configuration, self.pageName, pageText,
             Revision, nowTipDateTime, self.dyntemplate, self.contributor)
        else:
            pageview.TextEditor(self.configuration, self.pageName, pageText,
               Revision, nowTipDateTime, self.dyntemplate, self.contributor,
               Rows, Cols)
        return

    def _eventLogRequest(self):
        if pagemodel.PageExists(self.archiveDir, self.pageName):
            try:
                pageview.Log(self.configuration, self.pageName,
                   pagemodel.GetLog(self.archiveDir, self.pageName),
                   self.dyntemplate, self.contributor)
            except:
                pageview.Error(self.pageName, 'Page Log Access Error',
                               " Couldn't get log." +
                               " Try again later.",
                               self.dyntemplate, self.contributor)
        else:
            pageview.Info(self.pageName, 'No Page Log',
                               " Page does not yet exist." +
                               " Log not available.", self.dyntemplate,
                               self.contributor)
        return

    def _eventSearchRequest(self, SearchArg):
        try:
            (searchResults, archivesFound, archivesSearched) = \
               pagemodel.Search(self.archiveDir, SearchArg)
            pageview.SearchResults(self.configuration, SearchArg,
                         searchResults, archivesFound, archivesSearched,
			 self.templae, self.contributor)
        except:
            pageview.Error(self.pageName, 'Search Error',
                           " Couldn't perform search." +
                           " Try again later.",
                           self.dyntemplate, self.contributor)
        
        return

    def _eventPageRequest(self, SearchArg):
        if pagemodel.PageExists(self.archiveDir, self.pageName):
            try:
                (pageText, revision, datetime) = \
                   pagemodel.GetPage(self.archiveDir, self.pageName, '')
                date = pagemodel.PageRevisionDate(self.archiveDir,
                                                      self.pageName)
            except:
                pageview.Error(self.pageName, 'Page Access Error',
                               " Couldn't get page." +
                               " Try again later.",
                               self.dyntemplate, self.contributor)
                return
        else:
            pageText = pageview.DefaultPageText(self.pageName)
            date = ''
        pageview.Page(self.configuration, self.pageName, pageText, date, \
                     SearchArg, self.dyntemplate, self.contributor)
        return

    def _validatePageName(self, PageName):
        if PageName == '':
            PageName = self.configuration.defaultPageName
        elif not pageview.IsLegalPageName(PageName):
            PageName = pageview.DefaultBogusPageName()
        return PageName
