#!/usr/bin/env python
# $Id: wikiup.py,v 1.38 2005/09/02 14:39:36 villate Exp $

""" Wikiup utility functions.
"""

copyright = """
Copyright (C) 2003, 2004, 2005 Jaime Villate <villate@gnu.org>
 """
license = """
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifht Floor, Boston,
MA 02110-1301, USA.
"""

import os

def removeFile(fileName):
    """Quietly remove a file.
       taken from pywiki 0.4.2, Copyright (C) 1998  Tim Voght
    """
    try:
        os.remove(fileName)
    except:
        pass
    return

def devNull():
    """Return the appropriate file name for an infinite sink file.
        taken from pywiki 0.4.2, Copyright (C) 1998  Tim Voght
    """
    if os.name == 'nt':
        return 'nul'
    return '/dev/null'

def quotate(fileName):
    """taken from pywiki 0.4.2, Copyright (C) 1998  Tim Voght """
    if os.name == 'nt':
        return '\"' + fileName + '\"'
    return fileName

def MD5sum(str):
    """Generates an MD5 checksum for a string (a 32-digit hexadecimal number).
       It does for a string the same thing that the GNU/Linux command md5sum
       does for a file's content. It uses the md5 module, which implements
       the algorithm in RFC 1321.
    """
    import md5
    from string import hexdigits
    md5sum = ''
    for char in md5.new(str).digest():
	asciival = ord(char)
	md5sum += hexdigits[(asciival >> 4) & 0xF] + hexdigits[asciival & 0xF]
    return md5sum

def MD5sumCheck(str,md5sum):
    """Checks the MD5 checksum for a string.
       It retuns 1 if the given md5sum string (a 32-digit hexadecimal number)
       matches the MD5 check sum of the string, and zero otherwise.
    """
    return (md5sum == MD5sum(str))

def Config(section,*options):
    """Reads options from a configuration file
       Given the name of a section, followed by a list of options, the
       value of the options in the configuration file is returned. A
       blank string is returned for each unknown option.
    """
    from ConfigParser import ConfigParser
    conf = ConfigParser()
    conf.read('wikiup.conf')
    unknown = 0
    values = ()
    if not conf.has_section(section): unknown = 1
    for option in options:
	if unknown:
	   values += '',
	else:
	   if conf.has_option(section,option):
	      values += conf.get(section,option),
	   else:
	      values += '',
    if len(values) == 1:
        return values[0]
    else:
        return values

def PrintPage(title, text, template, username='', cookie=''):
    """Creates an HTML page with header, from a template
       The template is an HTML file which can should contain a string
       %(text)s in the place where the text of the page should go.
       The variables %(title)s and %(username)s can also appear in the HTML
       template, and their values will be substituted.
    """
    html = open(template)
    page = html.read() % locals()
    html.close()
    print "Content-type: text/html"
    if cookie: print cookie
    print '\n' + page
    return

def LatexToPNG(equation, file , resolution=72):
    import tempfile, os.path
    texfile = tempfile.NamedTemporaryFile('w+b',1,'.tex')
    texfile.write('\\documentclass[12pt]{article}\n')
    texfile.write('\\nonstopmode\\nofiles\\pagestyle{empty}\n')
    texfile.write('\\begin{document}\n')
    texfile.write('$$' + equation + '$$')
    texfile.write('\\end{document}\n')
    texname = os.path.basename(texfile.name)
    os.system('latex %s 1>%s' % (texfile.name, devNull()))
    texfile.close()
    dviname = texname.replace('.tex','.dvi')
    logname = texname.replace('.tex','.log')
    os.system('dvipng -o %s -D%s -z9 --truecolor -T tight %s 1>%s' % (file, resolution, dviname, devNull()))
    removeFile(dviname)
    removeFile(logname)
    return

def ApplyHtmlTemplate(page):
    """Creates an HTML from a template
       page is a dictionary which must contain an entry 'text' with
       page contents in HTML.
       The template is an HTML file which should contain a string
       %(text)s in the place where the contens of the page should go.
       Other keywords contained in the dictionary "page" will be
       included in the place where the template has symbols of the
       form %(variable)s. If any of the variables in the field %()s
       is not in the 'page' dictionary, the page obtained will be
       identical to the template.
    """
    # Prepare the HTML template
    try:
        htmlfile = open(page['template'])
        htmltemplate = htmlfile.read()
        htmlfile.close()
    except:
        fontsize = int(round(12*float(page['zoom'])))
        htmltemplate = """<!DOCTYPE html PUBLIC "-//W3C//DTD HTML 4.01//EN">
<html>
<head>
<meta name="generator" content="wikiup 0.2, http://villate.org/wikiup/">
<title>%(html-title)s</title>
<style type="text/css">
div.figure {margin: 0; padding: 1.5em; text-align: center;}
div.equation {margin: 0; padding: 0.5em; text-align: center;}
table {margin-left: auto; margin-right: auto;}
"""
        htmltemplate += 'body {font-size: %spx;}\n</style>\n' % fontsize
        if 'stylesheet' in page:
            htmltemplate += '<link type="text/css" rel="stylesheet" href="' \
                            + page['stylesheet'] + '">\n'
        htmltemplate += '</head>\n<body>\n%(text)s\n</body>\n</html>'

    # Parse page dictionary into the HTML template
    try:
        htmlpage = htmltemplate % page
    except:
        htmlpage = htmltemplate
    return htmlpage

def GetUsername(hash):
    """Finds the username, given the login hash"""
    username = ''
    login = open(Config('account','login_file'))
    for line in login.read().splitlines():
       field = line.split(':')
       if field[1] == hash:
	  username = field[0]
	  break
    login.close()
    return username

def UserInfo(username,*fields):
    """Reads information from a file with users information
       Given a username and a list of fields, a list is returned with the
       value of those fields for the given user. If a field or the username
       do not exist, blank values are returned.
    """
    userfields = {'password':1, 'user':2, 'group':3, 'name':4}
    userinfo = ['','','','','']
    users_file = open(Config('account','passw_file'))
    for line in users_file.read().splitlines():
	field = line.split(':')
	if field[0] == username:
	   userinfo = field
	   break
    users_file.close()
    result = ()
    for field in fields:
	result += userinfo[userfields.get(field,1)],
    if len(result) == 1:
        return result[0]
    else:
        return result

def MakeCookie(path='',expires='',**fields):
    """Creates Cookies
       For each variable=value pair given, a Cookie is created. Two optional
       fields "path" and "expires" can be included in the Cookies.
    """
    import Cookie
    cookie = Cookie.SmartCookie()
    for key, value in fields.items():
	cookie[key] = value
	if path: cookie[key]['path'] = path
	if expires: cookie[key]['expires'] = expires
    return cookie.output()

def ReadCookie(name):
    """Reads the value of a Cookie
       The name of the Cookie to be read is passed as input argument.
    """
    import Cookie
    cookie = Cookie.SmartCookie(os.environ.get('HTTP_COOKIE',''))
    try:
	value = cookie[name].value
    except:
        value = ''
    return value

class WikiTranslator:
    """ Instantiated with page data, the TranslateToHtmlLines() method
        of a WikiTranslator object returns a list of HTML formatted text
        lines.
    """
    def __init__(self, PageText):
        import re

        self.pageText = PageText

        # basic syntax expressions
        
        self.Token = '\0263'
        self.LineBreak = '\0263\0263'
        self.urlExpr = re.compile( r'([^\[]|^)\b((http|https|ftp|mailto|news|file|gopher):[^\s\<\>\[\]"\'\(\)]*[^\s\<\>\[\]"\'\`\(\)\,\.\?])', re.I)
        self.urlExpr2 = re.compile( r'\[((http|https|ftp|mailto|news|file|gopher):[^\s\<\>\[\]"\'\(\)]*[^\s\<\>\[\]"\'\`\(\)\,\.\?])\s+([^\]]+)\]', re.I)
        self.keywdExpr = re.compile(r'^##\s*([a-zA-Z\-\d\_]+):\s*(.*)$')
        self.listExpr = re.compile(r'^\t+')
        self.tableExpr = re.compile(r'^\s*|.+|\s*$')
        self.dlExpr = re.compile(r'^\t+(\S[^\t]*):\t+')
        self.ulExpr = re.compile(r'^\t+\* ')
        self.olExpr = re.compile(r'^\t+\d+[\.\-] ')
        self.strongExpr = re.compile(r'\*([^\s\*][^\*]*)\*')
        self.mathExpr = re.compile(r'\$([^\$][^\$]*)\$')
        self.displaymathExpr = re.compile(r'^\s*\$[^\$][^\$]*\$$')
        self.thExpr = re.compile(r'^\*([^\s\*][^\*]*)\*$')
        self.ttExpr = re.compile(r'\`([^\s\`][^\`]*)\`')
        self.emphExpr = re.compile(r'\_([^\s\_][^\_]*)\_')
        self.escapeExpr = re.compile(r'\\([^\n\t])')
        self.escdollarExpr = re.compile(r'\\(\$)')
        self.hExpr = re.compile(r'^=+')
        self.headerExpr = re.compile(r'^=+\s*([^=]+)\s*=+')
        self.hrExpr = re.compile(r'^-----*\s*$')
        self.verbatimExpr = re.compile(r'^---\s*$')
        self.internExpr = re.compile(r'\[([^\[\]\s"\'><][^\[\]\s"\'><]*)\]')
        self.figureExpr = re.compile(r'^\s*\[[^\[\]\s"\'><][^\[\]\s"\'><]*\.((gif)|(jpg)|(jpeg)|(png))\]', re.I)
        self.internExpr2 = re.compile(r'\[([^\[\]\s"\'><][^\[\]\s"\'><]*)\s+([^\]]+)\]')
        self.externExpr = re.compile(r'\[(https?:[^\s\<\>\[\]"\'\(\)]+)\s+([^\]]+)\]')
        self.inplaceurlExpr = re.compile(self.Token + r'(\d+)' + self.Token)
        self.encodedExpr = re.compile(self.Token+'chr'+r'(\d+)'+self.Token)
        self.imageExpr = re.compile(r'.*\.((gif)|(jpg)|(jpeg)|(png))', re.I)
        self.wikiPage = re.compile(r'^(\.?\.?[^\.\'\"\`\s;:,\?\$@&\=\+\*#]+)(#?[^\.\'\"\`\s;:,\?\$@&\=\+\*]*)$')
        self.eqprefix = 'eq-'
        return

    def SetEqPrefix(self, prefix):
        self.eqprefix = prefix
        return

    def TranslateToHtmlLines(self):
        """
        """
        try:
            from cStringIO import StringIO
        except:    
            from StringIO import StringIO
        self.page = {}                    # page contents dictionary
        self.htmlPageText = []            # array for HTML parsed lines
        self.Estk = self.emitterStack()   # stack of all tags
        self.listLevel = 0                # current list level
        self.inPlaceUrls = 0              # count of hyperlinks parsed
        self.inPlaceUrlList = []          # list of URL's parsed
        self.inPlaceTextList = []         # text corresponding to URL's
        self.Code = ''
        self.Depth = 1
        self.TblStyle = ''
        self.TblColcount = 0

        # processes the text line by line
        previous = ''
        for line in StringIO(self.pageText).readlines():
            line = line.rstrip()

            # protects line breaks and parse line continuations
            if (previous.endswith('\\\\\\') and self.Code != 'pre'):
                previous = previous[:-3] + self.LineBreak + line.lstrip()
                continue
            if (previous.endswith('\\\\') and self.Code != 'pre'):
                previous = previous[:-2] + self.LineBreak
            if (previous.endswith('\\') and self.Code != 'pre'):
                previous = previous[:-1] + line.lstrip()
                continue

            # sets up html class attributes for equations and figures
            if (self.displaymathExpr.match(line)):
                self.Estk.html['div'] = 'class="equation"'
            if (self.figureExpr.match(line)):
                self.Estk.html['div'] = 'class="figure"'

            # processes the previous line
            text = self.parseLine(previous)
            if (text or self.Code == 'pre'):
                text += '\n'
            self.htmlPageText.append(text)
            previous = line

        # pocesses the last line
        if line:
            text = self.parseLine(line)
            if text: text += '\n'
            self.htmlPageText.append(text)
        self.htmlPageText.append(self.Estk.emitCode('...', 0))
        self.page['text'] = ''.join(self.htmlPageText)
        return

    def parseLine(self, line):

        if self.Code != 'pre':
            
            # blank lines
            if (not line or line.isspace()):
                self.Code = '...'
                self.TblStyle = ''
                self.TblColcount = 0
                if self.Depth > 0:
                    return self.Estk.emitCode(self.Code, self.Depth-1)
                else:
                    return ''
            
            # protect escaped dollar signs
            line = self.escdollarExpr.sub(self.encodeEscapedText, line)
            line.replace('        ','\t')
        
            # equations
            line = self.mathExpr.sub(self.substituteMath, line)

            # protect escaped characters
            line = self.escapeExpr.sub(self.encodeEscapedText, line)
        
        # parse special HTML characters
        line = line.replace('&','&amp;')
        line = line.replace('<','&lt;')
        line = line.replace('>','&gt;')

        # Processing of verbatim blocks
        # if the line starts with ---, it is the end of the block
        # otherwise, the line is saved
        if (self.Code == 'pre'):
            if (self.verbatimExpr.match(line)):
                self.Code = '...'
                if self.Depth > 0:
                    return self.Estk.emitCode(self.Code, self.Depth-1)
                else:
                    return ''
            else:
                return line
            
        # Parsing of comments
        if (line.startswith('#')):
            m = self.keywdExpr.match(line)
            # a 'Keyword: value' pair
            if m:
                keywd = m.group(1).strip()
                value = m.group(2).strip()
                if keywd == 'html':
                    tag = value.split(':')[0]
                    attr = value.split(':')[1]
                    self.Estk.html[tag.strip()] = attr.strip()
                else:
                    if self.page.has_key(keywd):
                        self.page[keywd] += ' ' + value
                    else:
                        self.page[keywd] = value
            return ''

        # Tables
        if (line.startswith('|')):
            tbline = ''
            fields = line.split('|')
            if not self.TblStyle:
                self.Code = 'table'
                if 'table' in self.Estk.html:
                    if self.Estk.html['table'].find('summary') == -1:
                        self.Estk.html['table'] += ' summary=""'
                else:
                    self.Estk.html['table'] = 'summary=""'
                if line.endswith('|'):
                    self.TblStyle = 'lines'
                    if self.Estk.html['table'].find('rules="all"') == -1:
                        self.Estk.html['table'] += ' frame="box" rules="all"'
                    self.TblColcount = len(fields) - 2
                else:
                    self.TblStyle = 'nolines'
                    self.Estk.html['table'] = self.Estk.html['table'].replace(' frame="box" rules="all"','')
                    self.TblColcount = len(fields) - 1
                tbline = self.Estk.emitCode(self.Code, self.Depth)
            if tbline:
                tbline += '\n<tr>'
            else:
                tbline += '<tr>'
            span = 1
            for i in range(self.TblColcount):
                tag = 'td'
                align = ''
                if i+1 < len(fields):
                    field = fields[i+1]
                    if not field:
                        span += 1
                        continue
                    field = field.lstrip()
                    if fields[i+1] == field:
                        f = field.rstrip()
                        if f != field:
                            align = 'left'
                            field = f
                    else:
                        f = field.rstrip()
                        if f == field:
                            align = 'right'
                        else:
                            align = 'center'
                            field = f
                    if self.thExpr.match(field):
                        tag = 'th'
                        field = self.thExpr.sub(r'\g<1>', field)
                    tbline += '<' + tag
                    if align: tbline += ' align="' + align + '"'
                    if span > 1:
                        tbline += ' colspan="' + str(span) + '"'
                        span = 1
                    # Parse hyperlinks
                    field =self.urlExpr2.sub(self.urlPlaceHolder2, field)
                    field =self.urlExpr.sub(self.urlPlaceHolder, field)
                    field=self.externExpr.sub(self.substituteExternalLink,field)
                    # parsing of bold, italics and monospace font
                    field=self.strongExpr.sub(self.substituteStrongText,field)
                    field=self.emphExpr.sub(self.substituteEmphasizedText,field)
                    field=self.ttExpr.sub(self.substituteTypewriterText,field)

                    # replace links
                    field=self.internExpr.sub(self.substituteInternalLink,field)
                    field=self.internExpr2.sub(self.substituteInternalLink2,field)
                    field=self.inplaceurlExpr.sub(self.substituteInPlaceUrls,field)
                    # replace encode characters
                    field=self.encodedExpr.sub(self.substituteEncodedChar,field)
                    # replace line breaks
                    field = field.replace(self.LineBreak,'<br>')

                    tbline += '>' + field + '</' + tag + '>'
                else:
                    tbline += '<td>&nbsp;</td>'
            tbline += '</tr>'
            return tbline
        else:
            self.TblStyle = ''
            self.TblColcount = 0

        # Start of a verbatim block
        if (self.verbatimExpr.match(line)):
            self.Code = 'pre'
            return self.Estk.emitCode(self.Code, self.Depth)

        # Parse hyperlinks
        line = self.urlExpr2.sub(self.urlPlaceHolder2, line)
        line = self.urlExpr.sub(self.urlPlaceHolder, line)
        line = self.externExpr.sub(self.substituteExternalLink, line)

        # parsing of bold, italics and monospace font
        line = self.strongExpr.sub(self.substituteStrongText, line)
        line = self.emphExpr.sub(self.substituteEmphasizedText, line)
        line = self.ttExpr.sub(self.substituteTypewriterText, line)

        if 'div' in self.Estk.html:
            self.Code = 'div'
        else:
            self.Code = 'p'
        self.Depth = 1
#        self.listLevel = 0
        pre = ''

        # parse lists
        m = self.listExpr.match(line)
        if m:
            listlevel = len(m.group(0))
            if listlevel > self.listLevel:
                self.Code = 'blockquote'
                self.listLevel += 1
            else:
                self.listLevel = listlevel
            self.Depth = 2*self.listLevel + 1
            d = self.dlExpr.match(line)
            # description lists
            if d:
                line = self.dlExpr.sub('', line)
                self.Code = 'dl'
                pre += self.Estk.emitCode(self.Code, self.Depth-2)
                pre += self.Estk.emitCode('dt', self.Depth-1)
                text = d.group(1)
                text=self.internExpr.sub(self.substituteInternalLink,text)
                text=self.internExpr2.sub(self.substituteInternalLink2,text)
                text = self.inplaceurlExpr.sub(self.substituteInPlaceUrls,text)
                text = self.encodedExpr.sub(self.substituteEncodedChar,text)
                pre += text
                pre += self.Estk.emitCode('dd', self.Depth-1)
                pre += self.Estk.emitCode('p', self.Depth)
            (line, subsMade) = self.ulExpr.subn('', line)
            # itemized lists
            if subsMade > 0:
                self.Code = 'ul'
                pre += self.Estk.emitCode(self.Code, self.Depth-2)
                pre += self.Estk.emitCode('li', self.Depth-1)
                pre += self.Estk.emitCode('p', self.Depth)
            (line, subsMade) = self.olExpr.subn('', line)
            # ordered lists
            if subsMade > 0:
                self.Code = 'ol'
                pre += self.Estk.emitCode(self.Code, self.Depth-2)
                pre += self.Estk.emitCode('li', self.Depth-1)
                pre += self.Estk.emitCode('p', self.Depth)
            # other indented lines are either in a paragraph or a blockquote
            if self.Code == 'blockquote':
                self.listLevel -= 1
                self.Depth = 2*self.listLevel + 1
            if len(self.Code) != 2:
                pre += self.Estk.emitCode(self.Code, self.Depth)
        else:
            m = self.hExpr.match(line)
            # section headings
            if m:
                level = len(m.group(0))
                (line, subsMade) = self.headerExpr.subn('<h'+`level`+'>'+ \
                                    r'\g<1>'+'</h'+`level`+'>', line)
                self.Code = '...'
                self.Depth = 1
                pre += self.Estk.emitCode(self.Code, 0)
            else:
                # anything else should be a paragraph
                pre += self.Estk.emitCode(self.Code, self.Depth)

        # replace links
        line = self.internExpr.sub(self.substituteInternalLink, line)
        line = self.internExpr2.sub(self.substituteInternalLink2, line)
        line = self.inplaceurlExpr.sub(self.substituteInPlaceUrls, line)

        #replace encode characters
        line = self.encodedExpr.sub(self.substituteEncodedChar, line)
        
        # replace line breaks
        line = line.replace(self.LineBreak,'<br>')

        return (pre + line.lstrip())
        
    class emitterStack:
        """ Class that implements a stack of code tags """

        def __init__(self):
            self.stack = []
            self.html = {}     # dictionary of html tag's attributes
        def push(self, Code):
            self.stack = [Code] + self.stack
        def pop(self):
            top, self.stack = self.stack[0], self.stack[1:]
            return top
        def count(self):
            return len(self.stack)
        def emitCode(self, Code, Depth):
            line = ''
            # close open tags down to Depth (trigered by a new paragraph)
            if Code=='...':
                if self.count() > 0:
                    while (self.count() > Depth):
                        line += ("</%s>" % self.pop())
            else:
                tag = Code
                # bring the number of tags down to the value of Depth
                while (self.count() > Depth):
                    line += ("</%s>\n" % self.pop())
                if (self.count() < Depth):
                    self.push(Code)
                    if tag in self.html:
                        if tag == 'div':
                            tag += ' ' + self.html['div']
                            del self.html['div']
                        else:
                            tag = tag + ' ' + self.html[tag]
                    line += ("<%s>" % tag)

                # if the current tag does not match the top of the stack
                if self.count() > 0 and self.stack[0] != Code:
                    if tag in self.html:
                        if tag == 'div':
                            tag += ' ' + self.html['div']
                            del self.html['div']
                        else:
                            tag = tag + ' ' + self.html[tag]
                    line += ("</%s>\n<%s>" % (self.stack[0], tag))
                    self.stack[0] = Code
            return line

    def urlPlaceHolder(self, MatchObject):
        pre = MatchObject.group(1)
        self.inPlaceUrlList.append(MatchObject.group(2))
        self.inPlaceTextList.append(MatchObject.group(2))
        placeNum = str(self.inPlaceUrls)
        self.inPlaceUrls = self.inPlaceUrls + 1
        return pre + self.Token + placeNum + self.Token

    def urlPlaceHolder2(self, MatchObject):
        self.inPlaceUrlList.append(MatchObject.group(1))
        text = MatchObject.group(3)
        text = self.strongExpr.sub(self.substituteStrongText, text)
        text = self.emphExpr.sub(self.substituteEmphasizedText, text)
        text = self.ttExpr.sub(self.substituteTypewriterText, text)
        self.inPlaceTextList.append(text)
        placeNum = str(self.inPlaceUrls)
        self.inPlaceUrls = self.inPlaceUrls + 1
        return self.Token + placeNum + self.Token

    def substituteMath(self, MatchObject):
        equation = MatchObject.group(1).strip()

        # Initiate the equations list, if necessary
        if not self.page.has_key('equations'):
            self.page['equations']=[]

        # If the same equation has appeared befored, reuse it
        if equation in self.page['equations']:
            num = self.page['equations'].index(equation) + 1
            
        # Otherwise, add the equation to the list
        else:
            self.page['equations'].append(equation)
            num = len(self.page['equations'])

        # Return a pinter to the PNG file where the equation will be
        return ('[%s%s.png]' % (self.eqprefix, num))

    def substituteInternalLink(self, MatchObject):
        fullName = MatchObject.group(1)
        text = fullName
        m = self.wikiPage.match(fullName)
        if m:
            fullName = m.group(1) + '.html' + m.group(2)
        if self.imageExpr.match(fullName):
            return ('<img alt="%s" src="%s">' % (fullName, fullName))
        else:
            return ('<a href="%s">%s</a>' % (fullName,text))

    def substituteInternalLink2(self, MatchObject):
        fullName = MatchObject.group(1)
        text = MatchObject.group(2)
        if fullName.startswith('%'):
            return ('<a name="%s">%s</a>' % (fullName[1:],text))
        m = self.wikiPage.match(fullName)
        if m:
            fullName = m.group(1) + '.html' + m.group(2)
        return ('<a href="%s">%s</a>' % (fullName,text))

    def substituteExternalLink(self, MatchObject):
        self.inPlaceUrlList.append(MatchObject.group(1))
        text = MatchObject.group(2)
        text = self.strongExpr.sub(self.substituteStrongText, text)
        text = self.emphExpr.sub(self.substituteEmphasizedText, text)
        text = self.ttExpr.sub(self.substituteTypewriterText, text)
        self.inPlaceTextList.append(text)
        placeNum = str(self.inPlaceUrls)
        self.inPlaceUrls = self.inPlaceUrls + 1
        return self.Token + placeNum + self.Token

    def substituteInPlaceUrls(self, MatchObject):
        url = self.inPlaceUrlList[int(MatchObject.group(1))]
        text = self.inPlaceTextList[int(MatchObject.group(1))]
        if (self.imageExpr.match(url) and url == text):
            return ('<img alt="%s" src="%s">' % (url, url))
        else:
            return ('<a href="%s">%s</a>' % (url,text))

    def substituteEmphasizedText(self, MatchObject):
        return '<em>' + MatchObject.group(1) + '</em>'

    def substituteStrongText(self, MatchObject):
        return '<strong>' + MatchObject.group(1) + '</strong>'

    def substituteTypewriterText(self, MatchObject):
        return '<tt>' + MatchObject.group(1) + '</tt>'

    def encodeEscapedText(self, MatchObject):
        char = MatchObject.group(1)
        if char in ('&', '<', '>'):
            return char
        else:
            numchar = 'chr%s' % ord(char)
            return self.Token + numchar + self.Token

    def substituteEncodedChar(self, MatchObject):
        return chr(int(MatchObject.group(1)))


